/*
  A test driver for formula parsing and pretty printing.
  Copyright (C) 2005 The MITRE Corporation

  Author: John D. Ramsdell

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
*/

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <obstack.h>
#include <slat/xmalloc.h>
#include <slat/pprint.h>
#include <slat/symbol.h>
#include <slat/intern.h>
#include <slat/formula.h>
#include <slat/action.h>

extern int yyparse(void);

#ifdef PACKAGE
const char package[] = PACKAGE;
#else
const char *package = (const char *)0;
#endif

#ifdef VERSION
const char version[] = VERSION;
#else
const char version[] = "Version information not available";
#endif

static void
print_version(const char *program)
{
  if (package)
    program = package;
  printf("%s %s\n", program, version);
}

static int
usage(const char *prog)
{
  printf("Usage: %s [file name]\n", prog);
  print_version(prog);
  return 0;
}

/* An implementation of a symbol */

/* Allocation using obstack */

#define obstack_chunk_alloc xmalloc
#define obstack_chunk_free free

static struct obstack stack[1];

void
symbol_init(void)
{
  obstack_init(stack);
}

struct symbol
{
  const char sym[1];
};

symbol_t
make_symbol(const char *name)
{
  return (symbol_t)obstack_copy0(stack, name, strlen(name));
}

const char *
symbol_name(symbol_t symbol)
{
  return symbol->sym;
}

/* Actions */

static lts_t lts;

void
parsed_lts(lts_t x)
{
  lts = x;
}

void
parsed_diagrams(diagram_list_t diagrams)
{
}

void
parsed_transition(tran_t tran)
{
}

int
main(int argc, char **argv)
{
  int i;
  switch (argc) {
  case 1:
    break;
  case 2:
    if (argv[1][0] == '-') {
      if (argv[1][1])
	return usage(argv[0]);
      else
	break;
    }
    else if (setfile(argv[1]) == -1) {
      fprintf(stderr, "Cannot open %s\n", argv[1]);
      return 1;
    }
    break;
  default:
    fprintf(stderr, "Bad arg count\n");
    return 1;
  }
  pprint_init();
  intern_init();
  symbol_init();
  formula_stack(stack);
  i = yyparse();
  if (i)
    return i;
  if (!lts) {
    fprintf(stderr, "Internal error\n");
    return 1;
  }
  intern_free();
  print_lts(stdout, lts);
  return 0;
}
