#include <sys/ioctl.h>
#include <linux/soundcard.h>

#include "sjog.h"
#include "sjog_volume.h"
#include "sjog_volume_callbacks.h"

/* -------------------------------------------------------------------------- */

GtkWidget *volumeWindow;     /* volume dialog box */
GtkWidget *progressBar;      /* widget to show volume level */
guint volumeTimer;           /* window hide timer  */
guint currentVolume;         /* startup volume */
guint mixerFD = 0;           /* mixer file descriptor */

/* -------------------------------------------------------------------------- */

/* 
 * Called by sjog_jog_position_changed when we are in SJOG_MODE_VOLUME.
 * Checks the jog and increase/decrease the volume level according to
 * its position. Also polls the actual volume to update the progress bar
 * in case it has been modified by another program.
 */
void
sjog_volume_check_jog_position(int jog)
{

#ifdef DEBUG
   fprintf(stdout, "sjog_volume_check_jog_position: currentVolume=%d\n",
           currentVolume);
#endif

   /* update currentVolume just in case it has been modified externally */
   currentVolume = sjog_get_volume();

   switch (jog)
   {

        /* decrease volume */
     case JOG_BOTTOM:
        sjog_reset_volume_timer();
        if (currentVolume > 0)
        {
           /* round the volume before decrease */
           currentVolume = (int) (((float) currentVolume / 10) + 0.5) * 10;
           currentVolume -= 10;
           sjog_set_volume(currentVolume);
        }
        break;

        /* hide volume window */
     case JOG_PUSHED:
        gtk_timeout_remove(volumeTimer);
        sjog_volume_hide();
        break;

        /* increase volume */
     case JOG_TOP:
        sjog_reset_volume_timer();
        if (currentVolume < 100)
        {
           /* round the volume before increase */
           currentVolume = (int) (((float) currentVolume / 10) + 0.5) * 10;
           currentVolume += 10;
           sjog_set_volume(currentVolume);
        }
        break;

        /* update the progress bar in case the volume is externally modified */
     default:
        gtk_progress_set_value(GTK_PROGRESS(progressBar), currentVolume);
        break;

   }

}

/* -------------------------------------------------------------------------- */

void
sjog_set_volume(gint volume)
{

   gint vol = 0;

#ifdef DEBUG
   fprintf(stdout, "sjog_set_volume: volume=%d\n", volume);
#endif

   vol = (volume << 8) + volume;
   ioctl(mixerFD, MIXER_WRITE(SOUND_MIXER_VOLUME), &vol);

   /* finally sets the progress bar to the new volume value */
   gtk_progress_set_value(GTK_PROGRESS(progressBar), currentVolume);

}

/* -------------------------------------------------------------------------- */

gint
sjog_get_volume()
{

   gint r, l, vol;

#ifdef DEBUG
   fprintf(stdout, "sjog_get_volume\n");
#endif

   ioctl(mixerFD, MIXER_READ(SOUND_MIXER_VOLUME), &vol);

   l = vol & 0xff;
   r = (vol & 0xff00) >> 8;

   return (r + l) / 2;

}

/* -------------------------------------------------------------------------- */

void
sjog_init_mixer_fd()
{
   mixerFD = open(MIXER_DEVICE, O_RDWR);
}

/* -------------------------------------------------------------------------- */

/* Shows the screen volume level setting dialog.
 */
void
sjog_volume_show()
{

   /* first, try to open the mixer device */
   if (mixerFD == 0)
   {
      sjog_init_mixer_fd();

      /* do nothing if we can't open it */
      if (mixerFD == 0)
      {
         fprintf(stderr, "mixer initialization failed\n");
         sjog_set_mode(SJOG_MODE_LIST);
         return;
      }
   }

   /* first launch needs gui initialization */
   if (volumeWindow == NULL)
      init_volume_gui();

   /* read the volume value from mixer and update the progress bar */
   currentVolume = sjog_get_volume();
   gtk_progress_set_value(GTK_PROGRESS(progressBar), currentVolume);

   /* popup the volume window */
   gtk_widget_show_all(volumeWindow);

   /* initializes the hide timer */
   sjog_reset_volume_timer();
}

/* -------------------------------------------------------------------------- */

/* 
 * Hides the volume level setting dialog.
 */
void
sjog_volume_hide()
{
   if (mixerFD != 0)
   {
      close(mixerFD);
      mixerFD = 0;
   }
   gtk_timeout_remove(volumeTimer);
   gtk_widget_hide_all(volumeWindow);
   sjog_set_mode(SJOG_MODE_SCROLL);
}

/* -------------------------------------------------------------------------- */

/* 
 * Remove existing volume timer if exists and set a new one.
 */
void
sjog_reset_volume_timer()
{
   if (volumeTimer != 0)
      gtk_timeout_remove(volumeTimer);
   volumeTimer =
      gtk_timeout_add(SJOG_TIMEOUT, (GtkFunction) sjog_volume_timer_callback,
                      NULL);
}

/* -------------------------------------------------------------------------- */

/* 
 * Builds the widgets of the volume dialog window
 * Widgets hierarchy:
 *      volumeWindow -> vbox -> volume_pixmap
 *                               -> progressBar
 */
void
init_volume_gui()
{

   GtkAdjustment *adjust;
   GtkWidget *vbox;
   GdkPixmap *gd_pixmap;
   GtkWidget *volume_pixmap;

   /* vertical box */
   vbox = gtk_vbox_new(FALSE, 10);

   /* volume window */
   volumeWindow = gtk_window_new(GTK_WINDOW_DIALOG);
   gtk_container_set_border_width(GTK_CONTAINER(volumeWindow), 10);
   sjog_window_set_position(GTK_WINDOW(volumeWindow));
   gtk_window_set_title(GTK_WINDOW(volumeWindow), "S-Jog: Volume");
   gtk_signal_connect(GTK_OBJECT(volumeWindow), "delete-event",
                      GTK_SIGNAL_FUNC
                      (sjog_volume_signal_delete_event_callback), NULL);
   gtk_container_add(GTK_CONTAINER(volumeWindow), vbox);

   /* volume image */
   gtk_widget_realize(volumeWindow);
   gd_pixmap =
      gdk_pixmap_create_from_xpm(gtk_widget_get_parent_window(vbox), NULL,
                                 NULL,
                                 strdup(PKGDATADIR
                                        "/pixmaps/sjog-volume.xpm"));

   volume_pixmap = gtk_pixmap_new(gd_pixmap, NULL);
   gtk_container_add(GTK_CONTAINER(vbox), volume_pixmap);

   /* volume progress bar */
   adjust =
      (GtkAdjustment *) gtk_adjustment_new(DEFAULT_VALUE, MIN_VALUE,
                                           MAX_VALUE, STEP_INC, PAGE_INC,
                                           PAGE_SIZE);
   progressBar = gtk_progress_bar_new_with_adjustment(adjust);
   gtk_progress_bar_set_orientation(GTK_PROGRESS_BAR(progressBar),
                                    GTK_PROGRESS_LEFT_TO_RIGHT);
   gtk_progress_bar_set_bar_style(GTK_PROGRESS_BAR(progressBar),
                                  GTK_PROGRESS_CONTINUOUS);
   gtk_progress_set_format_string(GTK_PROGRESS(progressBar), "%v%%");
   gtk_progress_set_show_text(GTK_PROGRESS(progressBar), TRUE);

   gtk_container_add(GTK_CONTAINER(vbox), progressBar);

}
