/*
 * Copyright (c) 1998 - 2001 Phil Thompson <phil@river-bank.demon.co.uk>
 *
 * The main module for SIP.
 */


#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

#include "sip.h"


#ifndef PACKAGE
#define	PACKAGE	"sip"
#endif

#ifndef VERSION
#define	VERSION	"unknown"
#endif


static char *sipPackage = PACKAGE;


static void appendString(stringList **,char *);
static void help(void);
static void version(void);
static void usage(void);
static char parseopt(int,char **,char *,int *,char **);


int main(int argc,char **argv)
{
	char *filename, *docFile, *codeDir, *cppSuffix, *cppMName;
	char arg, *optarg;
	int optnr;
	FILE *file;
	sipSpec spec;
	stringList *makefiles, *timeslots;

	/* Initialise. */

	sipVersion = VERSION;
	includeDirList = NULL;
	timeslots = NULL;
	makefiles = NULL;
	codeDir = NULL;
	docFile = NULL;
	cppSuffix = ".cpp";
	cppMName = NULL;

	/* Parse the command line. */

	optnr = 1;

	while ((arg = parseopt(argc,argv,"hVm:c:d:I:s:p:t:",&optnr,&optarg)) != '\0')
		switch (arg)
		{
		case 'c':
			/* Where to generate the code. */

			codeDir = optarg;
			break;

		case 'm':
			/* Where to generate a Makefile. */

			appendString(&makefiles,optarg);
			break;

		case 'd':
			/* Where to generate the documentation. */

			docFile = optarg;
			break;

		case 't':
			/* Which version to generate code for. */

			appendString(&timeslots,optarg);
			break;

		case 'I':
			/* Where to get included files from. */

			appendString(&includeDirList,optarg);
			break;

		case 's':
			/* The suffix to use for C++ source files. */

			cppSuffix = optarg;
			break;

		case 'p':
			/* The name of the generated C++ module. */

			cppMName = optarg;
			break;

		case 'h':
			/* Help message. */

			help();
			break;

		case 'V':
			/* Display the version number. */

			version();
			break;

		default:
			usage();
		}

	if (optnr < argc)
	{
		file = NULL;
		filename = argv[optnr++];

		if (optnr < argc)
			usage();
	}
	else
	{
		file = stdin;
		filename = "stdin";
	}

	/* Parse the input file. */

	parse(&spec,file,filename,cppMName,timeslots);

	/* Verify and transform the parse tree. */

	transform(&spec);

	/* Generate code. */

	generateCode(&spec,codeDir,makefiles,docFile,cppSuffix);

	/* All done. */

	exit(0);
}


/*
 * Parse the next command line argument - similar to UNIX getopts().
 */

static char parseopt(int argc,char **argv,char *opts,int *optnrp,char **optargp)
{
	char arg, *op;
	int optnr = *optnrp;

	/* Check there is an argument and it is a switch. */

	if (optnr >= argc || argv[optnr] == NULL || argv[optnr][0] != '-')
		return '\0';

	/* Check it is a valid switch. */

	arg = argv[optnr][1];

	if (arg == '\0' || (op = strchr(opts,arg)) == NULL)
		usage();

	/* Check for the switch parameter, if any. */

	if (op[1] == ':')
	{
		if (argv[optnr][2] != '\0')
		{
			*optargp = &argv[optnr][2];
			++optnr;
		}
		else if (optnr + 1 >= argc || argv[optnr + 1] == NULL)
			usage();
		else
		{
			*optargp = argv[optnr + 1];
			optnr += 2;
		}
	}
	else if (argv[optnr][2] != '\0')
		usage();
	else
	{
		*optargp = NULL;
		++optnr;
	}

	*optnrp = optnr;

	return arg;
}


/*
 * Append a string to a list of them.
 */

static void appendString(stringList **headp,char *s)
{
	stringList *sl;

	/* Create the new entry. */

	sl = sipMalloc(sizeof (stringList));

	sl -> s = s;
	sl -> next = NULL;

	/* Append it to the list. */

	while (*headp != NULL)
		headp = &(*headp) -> next;

	*headp = sl;
}


/*
 * Display all or part of a one line error message describing a fatal error.
 * If the message is complete (it has a newline) then the program exits.
 */

void fatal(char *fmt,...)
{
	static int start = TRUE;

	va_list ap;

	if (start)
	{
		fprintf(stderr,"%s: ",sipPackage);
		start = FALSE;
	}

	va_start(ap,fmt);
	vfprintf(stderr,fmt,ap);
	va_end(ap);

	if (strchr(fmt,'\n') != NULL)
		exit(1);
}


/*
 * Display the SIP version number on stdout and exit with zero exit status.
 */

static void version(void)
{
	printf("%s\n",sipVersion);
	exit(0);
}


/*
 * Display the help message on stdout and exit with zero exit status.
 */

static void help(void)
{
	printf(
"Usage:\n"
"    %s [-h] [-V] [-c dir] [-d file] [-m file] [-I dir] [-s suffix] [-p module] [-t version] [file]\n"
"where:\n"
"    -h          display this help message\n"
"    -V          display the %s version number\n"
"    -c dir      the name of the code directory [default not generated]\n"
"    -d file     the name of the documentation file [default not generated]\n"
"    -m file     the name of the Makefile [default none generated]\n"
"    -I dir      look in this directory when including files\n"
"    -s suffix   the suffix to use for C++ source files [default \".cpp\"]\n"
"    -p module   the name of the generated C++ module [default %%Module]\n"
"    -t version  the primary version to generate code for [default latest]\n"
"    file        the name of the specification file [default stdin]\n"
		,sipPackage,sipPackage);

	exit(0);
}


/*
 * Display the usage message.
 */

static void usage(void)
{
	fatal("Usage: %s [-h] [-V] [-c dir] [-d file] [-m file] [-I dir] [-s suffix] [-p module] [-t version] [file]\n",sipPackage);
}
