#include <asio.hpp>
#include <getopt.h>
#include <sys/wait.h>
#include <boost/bind.hpp>
#include <iostream>
#include "sinfoserver.h"
#include "udpmessageserver.h"
#include "tcpmessageserver.h"
#include "cleanuplist.h"
#include "measureserver.h"
#include "broadcastreceiver.h"
#include "demoninit.h"
#include "sinfo.h"
using namespace std;


class MainClass
{
private:
  bool broadcastonlymode;
  bool watchdogmode;
  string bcast_address;
  string networkcard;
  string marker;

  void mainLoopNoWatchdog();
  void mainLoopWithWatchdog();

  void printHelpAndExit(const char * plaint = "Usage:-");

public:
  MainClass(int argc, char * argv[]);

  int mainLoop();
};


MainClass::MainClass(int argc, char * argv[])
{
  bool demon = true;
  bool quiet = false;

  broadcastonlymode = false;
  watchdogmode = false;
  bcast_address=string("255.255.255.255");

  int nicevalue = 10; // start with nice 10 to ensure niceability in accidental cases.


#ifdef HAVE_GETOPT_LONG
  static const struct option long_options[] =
  {
    { "bcastaddress", required_argument, 0, 'b'},
    { "networkcard", required_argument, 0, 'n'},
    { "marker", required_argument, 0, 'm'},
    { "nice", required_argument, 0, 'N' },
    { "foreground", no_argument, 0, 'F' },
    { "version", no_argument, 0, 'V' },
    { "quiet", no_argument, 0, 'q' },

    { "spymode", no_argument, 0, 's' },
    { "watchdog", no_argument, 0, 'W' },

    { "help", no_argument, 0, '?' },

    { 0, 0, 0, 0 }
  };
#endif


  int c;
#ifdef HAVE_GETOPT_LONG
  while ((c = getopt_long(argc, argv, "b:B:VFn:N:qa:A:lsWh?", long_options, NULL)) != -1)
#else
  while ((c = getopt (argc, argv, "b:B:VFn:N:qa:A:lsWh?")) != -1)
#endif
  {
    switch (c)
    {
    case 'b':
      bcast_address = string(optarg);
      break;
    case 'n':
      networkcard = string(optarg);
      break;
    case 'm':
      marker = string(optarg);
      break;
    case 'N':
      nicevalue = atoi(optarg);
      break;
    case 'F':
      demon = false;
      break;
    case 'q':
      quiet = true;
      break;
    case 'V':
      cout << argv[0] << " " << VERSION
           << "  (compiled " __DATE__ " " __TIME__ ")"
           << endl;
      exit(0);
      break;
    case 's':
      broadcastonlymode = true;
      break;
    case 'W':
      watchdogmode = true;
      break;
    case '?':
    case 'h':
      printHelpAndExit();
      break;

    default:
      printHelpAndExit("No such option");
    }
  }

  if (false == quiet)
    cout << argv[0] << " " VERSION "  (compiled " __DATE__ " " __TIME__ ")" << endl;


  nice(nicevalue);

  if (demon)
    demoninit();
}


void MainClass::printHelpAndExit(const char * plaint)
{
  cout << plaint << endl
       << endl
       << "sinfod [-F][-q][-V] [-b #.#.#.#] [-n card] [-N nicevalue ] [-s] [-W]" << endl
       << endl
       << "OPTIONS (for full descriptions see \"man sinfod\")" << endl
       << "-F/--foreground" << endl
       << "  Do not detach from the terminal." << endl
       << "-q/--quiet" << endl
       << " be quiet - don't display startup informations." << endl
       << "-V/--version" << endl
       << "  Print the version number and exit." << endl
       << "-b #.#.#.#/--bcastaddress=#.#.#.#" << endl
       << "  Set  broadcast address of sinfod." << endl
       << "-n card/--networkcard=card" << endl
       << "  Determine network load fron card." << endl
       << "-N nicevalue/--nice=nicevalue" << endl
       << "  Set the priority of this demon to nicevalue." << endl
       << "-s/--spymode" << endl
       << "  Deactivate TCP interface, broadcast only mode." << endl
       << "-W/--watchdog" << endl
       << "  start with simple watchdog." << endl
       << endl;
  exit(1);
}


void MainClass::mainLoopNoWatchdog()
{
  try
  {
    asio::io_service ioservice;
    list < Wsinfo > wsinfoList;

    if (true==broadcastonlymode)
    {
      MeasureServer measureServer(ioservice, bcast_address, networkcard, marker);
      ioservice.run();
    }
    else
    {
      bool restartCounterEventFlag=true;

      SinfoServer sinfoServer(wsinfoList);
      UDPMessageServer udpMessageServerIPv4(ioservice,asio::ip::udp::endpoint(asio::ip::udp::v4(), SINFO_REQUEST_PORT));
      udpMessageServerIPv4.receiveMessageSignal.connect(boost::bind(&SinfoServer::receiveMessageSlot, &sinfoServer, _1, _2));
      UDPMessageServer udpMessageServerIPv6(ioservice,asio::ip::udp::endpoint(asio::ip::udp::v6(), SINFO_REQUEST_PORT));
      udpMessageServerIPv6.receiveMessageSignal.connect(boost::bind(&SinfoServer::receiveMessageSlot, &sinfoServer, _1, _2));

      TCPMessageServer tcpMessageServerIPv4(ioservice, asio::ip::tcp::endpoint(asio::ip::tcp::v4(), SINFO_REQUEST_PORT));
      tcpMessageServerIPv4.receiveMessageSignal.connect(boost::bind(&SinfoServer::receiveMessageSlot, &sinfoServer, _1, _2));
      TCPMessageServer tcpMessageServerIPv6(ioservice, asio::ip::tcp::endpoint(asio::ip::tcp::v6(), SINFO_REQUEST_PORT));
      tcpMessageServerIPv6.receiveMessageSignal.connect(boost::bind(&SinfoServer::receiveMessageSlot, &sinfoServer, _1, _2));

      MeasureServer measureServer(ioservice, bcast_address, networkcard, marker);
      CleanupList cleanupList(ioservice, wsinfoList);
      BroadcastReceiver broadcastReceiver(ioservice, restartCounterEventFlag, wsinfoList);
      while (ioservice.run_one())
      {
        // while (messagepassingEventQueue.size()>0)
        // {
        //   deliverEvent(messagepassingEventQueue.pop());
        // }
        // FIXME : restartCounterEventFlag is much better than a global variable,
        // but the Event passing should be more generic and the classes shoud register a callback by themselve
        // e.g. by using an observer design pattern
        if (restartCounterEventFlag)
        {
          measureServer.restartCounterEvent();
          restartCounterEventFlag=false;
        }
      }
    }

    cout << "ioservice run exit" << endl;
  }
  catch (exception& e)
  {
    cerr << "Exception: " << e.what() << "\n";
  }
}


void MainClass::mainLoopWithWatchdog()
{
  while (1)
  {

    pid_t childpid = fork();
    cout << childpid << endl;
    if ( -1 == childpid)
    {
      cerr << "error forking child process" << endl;
      exit(1);
    }

    if (0 == childpid)
    {
      mainLoopNoWatchdog();
    }

    if (childpid > 0)
    {
      int status;
      waitpid(childpid, &status, 0);
    }

    sleep(1);
  }
}


int MainClass::mainLoop()
{
  // writing to a bad socket will abort the porgram if we dont ignore SIGPIPE
  signal(SIGPIPE, SIG_IGN);

  if (true==watchdogmode)
  {
    mainLoopWithWatchdog();
  }
  else
  {
    mainLoopNoWatchdog();
  }
  return 0;
}


int main(int argc, char * argv[])
{
  MainClass mainClass(argc,argv);

  return mainClass.mainLoop();
}
