/*
**  Sinek (Video Player)
**  Copyright (c) 2001 - 2002 the Sinek Team, see the AUTHORS file.
**
**  This code is free software; you can redistribute it and/or
**  modify it under the terms of the GNU General Public License.
**
**  utility functions
*/

#include "common.h"
#include <stdarg.h>

struct window
{
	GtkWidget *win;
	int flags;
};

static GList *windows = NULL;

static void wm_map_cb(GtkWidget *win, struct window *w);
static void wm_unmap_cb(GtkWidget *win, struct window *w);
static gint wm_key_cb(GtkWidget *win, GdkEventKey *ev, struct window *w);
static gint wm_button_cb(GtkWidget *win, GdkEventButton *ev, struct window *w);
static gint wm_delete_cb(GtkWidget *win, GdkEvent *ev, struct window *w);
static gint wm_destroy_cb(GtkWidget *win, struct window *w);


struct fsel_struct
{
	GtkWidget *win;
	void (*callback)(char *filename);
};

static void fsel_ok(GtkWidget *w, struct fsel_struct *fsel);
static void fsel_cancel(GtkWidget *w, struct fsel_struct *fsel);
static gint fsel_delete(GtkWidget *w, GdkEvent *ev, struct fsel_struct *fsel);
static void freq_ok(GtkWidget *w, GtkFileSelection *fs);
static void freq_add_selected(GtkWidget *w, GtkFileSelection *fs);
static void freq_add_all(GtkWidget *w, GtkFileSelection *fs);
static void freq_add_files(GtkFileSelection *fs);


void add_tooltip(GtkWidget *w, char *tip)
{
	static GtkTooltips *tips = NULL;

	if(NULL == tips) tips = gtk_tooltips_new();
	gtk_tooltips_set_tip(tips, w, tip, NULL);
}


static void cb_url(GtkButton *but, const char *url)
{
	int pid, argc;
	char *argv[5], buf3[1024];

	sprintf(buf3, "openURL(%s)", url);
	argv[0] = "mozilla";
	argv[1] = "-remote";
	argv[2] = buf3;
	argc = 3;

	pid = fork();
	if(pid == -1)
		return;
	if(pid == 0)
	{
		execvp(argv[0], argv);
		_exit(0);
	}
	else
	{
		return;
	}
}


GtkWidget *create_url(const char *url)
{
	static char buf[512];
	GdkColor uri_color[2] = {{0, 0, 0, 0xffff}, {0, 0xffff, 0, 0}};
	gboolean success[2];
	GtkStyle *style;
	GdkColormap *cmap;
	GtkWidget *box, *but;
	int i;

	box = gtk_hbox_new(FALSE, 0);
	gtk_widget_show(box);
	sprintf(buf, " %s ", url);
	but = gtk_button_new_with_label(buf);
	gtk_widget_show(but);
	gtk_box_pack_start(GTK_BOX(box), but, TRUE, FALSE, 0);
	gtk_button_set_relief(GTK_BUTTON(but), GTK_RELIEF_NONE);
	buf[0] = ' ';
	for(i = 1; i <= strlen(url); i++) buf[i] = '_';
	buf[i] = ' ';
	gtk_label_set_pattern(GTK_LABEL(GTK_BIN(but)->child), buf);
//
	cmap = gdk_window_get_colormap(sinek.control_win->window);
	gdk_colormap_alloc_colors(cmap, uri_color, 2, FALSE, TRUE, success);
	if (success[0] == TRUE && success[1] == TRUE)
	{
		gtk_widget_ensure_style(GTK_BIN(but)->child);
		style = gtk_style_copy(gtk_widget_get_style(GTK_BIN(but)->child));
		style->fg[GTK_STATE_NORMAL] = uri_color[0];
		style->fg[GTK_STATE_ACTIVE] = uri_color[1];
		style->fg[GTK_STATE_PRELIGHT] = uri_color[0];
		gtk_widget_set_style(GTK_BIN(but)->child, style);
	} else
		g_warning("about_create(): color allocation failed.\n");
//
	gtk_signal_connect(GTK_OBJECT(but), "clicked", GTK_SIGNAL_FUNC(cb_url), (gpointer)url);

	return box;
}


static void button_cb(GtkWidget *w, cmd_type data)
{
	execute_cmd(data);
}


GtkWidget *create_pix_button(char *xpm[], cmd_type cmd, char *tip)
{
	GtkWidget *but, *icon;
	GdkPixmap *image;
	GdkBitmap *transp;

	but = gtk_button_new();
	gtk_widget_show(but);
	image = gdk_pixmap_colormap_create_from_xpm_d(NULL, gdk_colormap_get_system(), &transp, NULL, xpm);
	icon = gtk_pixmap_new(image, transp);
	gtk_widget_show(icon);
	gtk_container_add(GTK_CONTAINER(but), icon);
	gtk_signal_connect(GTK_OBJECT(but), "clicked", GTK_SIGNAL_FUNC(button_cb), (gpointer)cmd);
	if(tip) add_tooltip(but, tip);
	return but;
}


void update_pix_toggle(GtkWidget *w)
{
	GdkPixmap *pm;
	GdkBitmap *t;

	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)))
	{
		pm = gtk_object_get_data(GTK_OBJECT(w), "on");
		t = gtk_object_get_data(GTK_OBJECT(w), "ton");
	}
	else
	{
		pm = gtk_object_get_data(GTK_OBJECT(w), "off");
		t = gtk_object_get_data(GTK_OBJECT(w), "toff");
	}
	gtk_pixmap_set(GTK_PIXMAP(GTK_BIN(w)->child), pm, t);
	gtk_widget_show(GTK_BIN(w)->child);   
}


static void toggle_cb(GtkWidget *w, cmd_type cmd)
{
	execute_cmd(cmd);
	update_pix_toggle(w);
}


GtkWidget *create_pix_toggle(char *on_xpm[], char *off_xpm[], cmd_type cmd, char *tip)
{
	GtkWidget *tog, *pix;
	GdkPixmap *on, *off;
	GdkBitmap *ton, *toff;

	tog = gtk_toggle_button_new();
	gtk_widget_show(tog);
	off = gdk_pixmap_colormap_create_from_xpm_d(NULL, gdk_colormap_get_system(), &toff, NULL, off_xpm);
	on = gdk_pixmap_colormap_create_from_xpm_d(NULL, gdk_colormap_get_system(), &ton, NULL, on_xpm);
	gtk_object_set_data(GTK_OBJECT(tog), "on", (gpointer)on);
	gtk_object_set_data(GTK_OBJECT(tog), "off", (gpointer)off);
	gtk_object_set_data(GTK_OBJECT(tog), "ton", (gpointer)ton);
	gtk_object_set_data(GTK_OBJECT(tog), "toff", (gpointer)toff);
	pix = gtk_pixmap_new(off, toff);
	gtk_widget_show(pix);
	gtk_container_add(GTK_CONTAINER(tog), pix);
	gtk_signal_connect(GTK_OBJECT(tog), "toggled", GTK_SIGNAL_FUNC(toggle_cb), (gpointer)cmd);
	add_tooltip(tog, tip);
	return tog;
}


void wm_manage(GtkWidget *win, int flags)
{
	struct window *w;

	w = malloc(sizeof(struct window));
	w->win = win;
	w->flags = flags;
	windows = g_list_append(windows, w);

	gtk_signal_connect(GTK_OBJECT(win), "map", GTK_SIGNAL_FUNC(wm_map_cb), (gpointer)w);
	gtk_signal_connect(GTK_OBJECT(win), "unmap", GTK_SIGNAL_FUNC(wm_unmap_cb), (gpointer)w);
	if(flags & WM_KEYS)
		gtk_signal_connect(GTK_OBJECT(win), "key_press_event", GTK_SIGNAL_FUNC(wm_key_cb), (gpointer)w);
	if(flags & WM_MENU || flags & WM_VOLUME)
		gtk_signal_connect(GTK_OBJECT(win), "button_press_event", GTK_SIGNAL_FUNC(wm_button_cb), (gpointer)w);
	if(flags & WM_AUTOHIDE)
		gtk_signal_connect(GTK_OBJECT(win), "delete_event", GTK_SIGNAL_FUNC(wm_delete_cb), (gpointer)w);
	gtk_signal_connect(GTK_OBJECT(win), "destroy", GTK_SIGNAL_FUNC(wm_destroy_cb), (gpointer)w);
}


static void wm_map_cb(GtkWidget *win, struct window *w)
{
	GdkEventMask mask;

	sinek.nr_mapped++;
	if(w->flags & WM_MENU || w->flags & WM_VOLUME)
	{
		mask = gdk_window_get_events(w->win->window);
		gdk_window_set_events(w->win->window, GDK_BUTTON_PRESS_MASK | mask);
	}
	video_lift(w->win);
}


static void wm_unmap_cb(GtkWidget *win, struct window *w)
{
	sinek.nr_mapped--;
	if(0 == sinek.nr_mapped) gtk_main_quit();
}


static gint wm_key_cb(GtkWidget *win, GdkEventKey *ev, struct window *w)
{
	if(GTK_IS_ENTRY(GTK_WINDOW(w->win)->focus_widget))
		return FALSE;
	key_handle(ev->keyval, ev->state);
	return TRUE;
}


static gint wm_button_cb(GtkWidget *win, GdkEventButton *ev, struct window *w)
{
	if(ev->button == 3 && w->flags & WM_MENU)
		popup_pop(ev->time);
	else if(ev->button == 4 && w->flags & WM_VOLUME)
		audio_slide_volume(5);
	else if(ev->button == 5 && w->flags & WM_VOLUME)
		audio_slide_volume(-5);
	else
		return FALSE;
	return TRUE;
}


static gint wm_delete_cb(GtkWidget *win, GdkEvent *ev, struct window *w)
{
	gtk_widget_hide(win);
	return TRUE;
}

	
static gint wm_destroy_cb(GtkWidget *win, struct window *w)
{
	windows = g_list_remove(windows, w);
	free(w);
	sinek.nr_mapped--;
	if(0 == sinek.nr_mapped) gtk_main_quit();
	return FALSE;
}


void wm_relayer_all(void)
{
	GList *item;
	GtkWidget *win;

	for(item = g_list_first(windows); item; item = g_list_next(item))
	{
		win = ((struct window *)item->data)->win;
		if(GTK_WIDGET_REALIZED(win)) video_lift(win);
	}
}


void wm_show_all(void)
{
	GList *item;
	struct window *w;

	for(item = g_list_first(windows); item; item = g_list_next(item))
	{
		w = (struct window *)item->data;
		if(w->flags & 4096)
		{
			gtk_widget_show(w->win);
			w->flags &= ~4096;
		}
	}
}


void wm_hide_all(void)
{
	GList *item;
	struct window *w;

	for(item = g_list_first(windows); item; item = g_list_next(item))
	{
		w = (struct window *)item->data;
		if(GTK_WIDGET_MAPPED(w->win))
		{
			gtk_widget_hide(w->win);
			w->flags |= 4096;
		}
	}
}


void select_file(void (*func)(char *filename), char *title)
{
	struct fsel_struct *fsel;
	GtkWidget *win, *b;

	fsel = malloc(sizeof(struct fsel_struct));
	fsel->callback = func;
	win = gtk_file_selection_new(title);
	fsel->win = win;
	gtk_container_set_border_width(GTK_CONTAINER(win), 10);
	b = GTK_FILE_SELECTION(win)->ok_button;
	gtk_signal_connect(GTK_OBJECT(b), "clicked", GTK_SIGNAL_FUNC(fsel_ok), fsel);
	b = GTK_FILE_SELECTION(win)->cancel_button;
	gtk_signal_connect(GTK_OBJECT(b), "clicked", GTK_SIGNAL_FUNC(fsel_cancel), fsel);
	gtk_signal_connect(GTK_OBJECT(win), "delete_event", GTK_SIGNAL_FUNC(fsel_delete), fsel);
	wm_manage(win, 0);
	gtk_widget_show(win);
}


static void fsel_ok(GtkWidget *w, struct fsel_struct *fsel)
{
	char *file;

	file = gtk_file_selection_get_filename(GTK_FILE_SELECTION(fsel->win));
	fsel->callback(file);
	gtk_widget_destroy(fsel->win);
	free(fsel);
}


static void fsel_cancel(GtkWidget *w, struct fsel_struct *fsel)
{
	gtk_widget_destroy(fsel->win);
	free(fsel);
}


static gint fsel_delete(GtkWidget *w, GdkEvent *ev, struct fsel_struct *fsel)
{
	free(fsel);
	return FALSE;
}


static void warning_ok_cb(GtkWidget *w, GtkWidget *dia)
{
	char *msg;

	msg = gtk_object_get_data(GTK_OBJECT(dia), "msg");
	g_free(msg);
	gtk_widget_destroy(dia);
}


static gint warning_delete_cb(GtkWidget *w, GdkEvent *ev, GtkWidget *dia)
{
	char *msg;

	msg = gtk_object_get_data(GTK_OBJECT(dia), "msg");
	g_free(msg);
	return FALSE;
}


void warning(char *fmt, ...)
{
	GtkWidget *w, *vb, *b;
	va_list ap;
	char *msg;

	va_start(ap, fmt);
	msg = g_strdup_vprintf(fmt, ap);
	va_end(ap);

	w = gtk_dialog_new();
	gtk_window_set_title(GTK_WINDOW(w), _("Sinek Message"));
	gtk_window_set_position(GTK_WINDOW(w), GTK_WIN_POS_CENTER);
	gtk_object_set_data(GTK_OBJECT(w), "msg", msg);
	gtk_signal_connect(GTK_OBJECT(w), "delete_event", GTK_SIGNAL_FUNC(warning_delete_cb), (gpointer)w);

	vb = GTK_DIALOG(w)->vbox;
	gtk_widget_show(vb);
	b = gtk_label_new(msg);
	gtk_widget_show(b);
	gtk_container_add(GTK_CONTAINER(vb), b);

	vb = GTK_DIALOG(w)->action_area;
	gtk_widget_show(vb);
	b = gtk_button_new_with_label(_("OK"));
	gtk_widget_show(b);
	GTK_WIDGET_SET_FLAGS(b, GTK_CAN_DEFAULT);
	gtk_container_add(GTK_CONTAINER(vb), b);
	gtk_widget_grab_default(b);
	gtk_signal_connect(GTK_OBJECT(b), "clicked", GTK_SIGNAL_FUNC(warning_ok_cb), (gpointer)w);

	wm_manage(w, WM_MENU);
	gtk_widget_show(w);
}


GtkWidget *file_request(int mode)
{
	GtkWidget *freq, *box, *b;

	freq = gtk_file_selection_new(_("Select Video File"));
	gtk_container_set_border_width(GTK_CONTAINER(freq), 10);
	gtk_window_set_wmclass(GTK_WINDOW(freq), "Sinek File Add Request", "Sinek");
	gtk_clist_set_selection_mode(GTK_CLIST(GTK_FILE_SELECTION(freq)->file_list), GTK_SELECTION_EXTENDED);
	gtk_object_set_data(GTK_OBJECT(freq), "mode", (gpointer)mode);

	b = GTK_FILE_SELECTION(freq)->ok_button;
	gtk_widget_show(b);
	GTK_WIDGET_SET_FLAGS(b, GTK_CAN_DEFAULT);
	gtk_signal_connect(GTK_OBJECT(b), "clicked", GTK_SIGNAL_FUNC(freq_ok), freq);

	b = GTK_FILE_SELECTION(freq)->cancel_button;
	gtk_widget_show(b);
	GTK_WIDGET_SET_FLAGS(b, GTK_CAN_DEFAULT);
	gtk_signal_connect_object(GTK_OBJECT(b), "clicked", gtk_widget_hide, GTK_OBJECT(freq));

	box = gtk_hbutton_box_new();
	gtk_widget_show(box);
	gtk_button_box_set_layout(GTK_BUTTON_BOX(box), GTK_BUTTONBOX_END);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(box), 0);
	gtk_box_pack_end(GTK_BOX(GTK_FILE_SELECTION(freq)->action_area), box, TRUE, TRUE, 0);

	b = gtk_button_new_with_label(_("Add selected files"));
	gtk_widget_show(b);
	gtk_box_pack_start(GTK_BOX(box), b, FALSE, FALSE, 0);
	gtk_signal_connect(GTK_OBJECT(b), "clicked", GTK_SIGNAL_FUNC(freq_add_selected), freq);

	b = gtk_button_new_with_label(_("Add all files in directory"));
	gtk_widget_show(b);
	gtk_box_pack_start(GTK_BOX(box), b, FALSE, FALSE, 0);
	gtk_signal_connect(GTK_OBJECT(b), "clicked", GTK_SIGNAL_FUNC(freq_add_all), freq);

	gtk_signal_connect_object(GTK_OBJECT(freq), "delete_event", gtk_widget_hide, GTK_OBJECT(freq));

	return freq;
}


static void freq_ok(GtkWidget *w, GtkFileSelection *fs)
{
	GList *node;
	gchar *mrl, *mrl2, *path;
	int tmp;

	tmp = (int)gtk_object_get_data(GTK_OBJECT(fs), "mode");
	gtk_label_get(GTK_LABEL(GTK_BIN(fs->history_pulldown)->child), &path);

	node = GTK_CLIST(fs->file_list)->selection;
	if(node)
	{
		if(tmp != 0 || g_list_next(node)) freq_add_selected(w, fs);
		if(tmp == 0)
		{
			gtk_clist_get_text(GTK_CLIST(fs->file_list), GPOINTER_TO_INT(node->data), 0, &mrl);
			mrl2 = g_strconcat(path, "/", mrl, NULL);
			video_play(mrl2);
			g_free(mrl2);
		}
	}
	else
	{
		mrl = gtk_file_selection_get_filename(fs);
		if(mrl && mrl[0] != 0 && mrl[strlen(mrl) - 1] != '/')
		{
			if(tmp == 0)
				video_play(mrl);
			else
				pl_append(mrl);
		}
	}
	gtk_widget_hide(GTK_WIDGET(fs));
}


static void freq_add_selected(GtkWidget *w, GtkFileSelection *fs)
{
	freq_add_files(fs);
	gtk_entry_set_text(GTK_ENTRY(fs->selection_entry), "");
}


static void freq_add_all(GtkWidget *w, GtkFileSelection *fs)
{
	gtk_clist_freeze(GTK_CLIST(fs->file_list));
	gtk_clist_select_all(GTK_CLIST(fs->file_list));
	freq_add_files(fs);
	gtk_clist_unselect_all(GTK_CLIST(fs->file_list));
	gtk_clist_thaw(GTK_CLIST(fs->file_list));
	gtk_entry_set_text(GTK_ENTRY(fs->selection_entry), "");
}


static void freq_add_files(GtkFileSelection *fs)
{
	GList *node;
	gchar *text, *text2, *tmp, *path = NULL;

	gtk_label_get(GTK_LABEL(GTK_BIN(fs->history_pulldown)->child), &tmp);
	path = g_strconcat(tmp, "/", NULL);

	node = GTK_CLIST(fs->file_list)->selection;
	if(node)
	{
		while(node)
		{
			gtk_clist_get_text(GTK_CLIST(fs->file_list), GPOINTER_TO_INT(node->data), 0, &text);
			text2 = g_strconcat(path, text, NULL);
			pl_append(text2);
			g_free(text2);
			node = g_list_next(node);
		}
	}
	else
	{
		text = gtk_file_selection_get_filename(fs);
		if(text[strlen(text) - 1] != '/')
		{
			pl_append(text);
			gtk_file_selection_set_filename(fs, "");
		}
	}

	if(path) g_free(path);
}
