/*
    si - Version 1.0 - By Joe Turgeon (jturgeon@isd.net)
    loadinfo.c

    si - System Information Viewer
    Copyright (C) 1999  Joe Turgeon (jturgeon@isd.net)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include "main.h"


// Structure Sizes
const size_t STRING_ARRAY_SIZE		= 128;
const size_t DEV_STRUCT_SIZE		= 130;
const size_t FILESYS_STRUCT_SIZE	= 256;
const size_t CPU_STRUCT_SIZE		= 1024;
const size_t PROC_STRUCT_SIZE		= 17434;
const size_t SCSIDEV_STRUCT_SIZE	= 774;
const size_t SCSIDRV_STRUCT_SIZE	= 256;
const size_t TTYDRIVERS_STRUCT_SIZE	= 640;
const size_t TTYLDISCS_STRUCT_SIZE	= 130;
const size_t MTRR_STRUCT_SIZE		= 264;
const size_t IDEDRV_STRUCT_SIZE		= 256;
const size_t IDEDEV_STRUCT_SIZE		= 646;
const size_t PARPORT_STRUCT_SIZE	= 896;
const size_t MOUNTS_STRUCT_SIZE		= 512;
// Global Variables
time_t BOOTTIME;
long int UPTIME;
// PROCFS Filename Defines
char   PROCINFO  [PROCINFO_COUNT_MAX]  [STORAGE_STRING];
int PROC_DISK		= 0;
int PROC_DISKRIO	= 0;
int PROC_DISKWIO	= 0;
int PROC_DISKRBLK	= 0;
int PROC_DISKWBLK	= 0;
string  *MODULES   [MODULES_COUNT_MAX];
struct DEV_STRUCT  *CHARDEV   [CHARDEV_COUNT_MAX]; 
struct DEV_STRUCT  *BLOCKDEV  [BLOCKDEV_COUNT_MAX];
struct DEV_STRUCT  *ISADMA    [ISADMA_COUNT_MAX];
struct FILESYS_STRUCT  *FILESYS   [FILESYS_COUNT_MAX];
string  *IRQINT    [IRQINT_COUNT_MAX];
string  *IOPORTS   [IOPORTS_COUNT_MAX];
string  *PARTS     [PARTS_COUNT_MAX];
struct MOUNTS_STRUCT  *MOUNTS    [MOUNTS_COUNT_MAX];
string  *SWAPS     [SWAPS_COUNT_MAX];
struct CPU_STRUCT   *CPU         [CPU_COUNT_MAX];
struct PROC_STRUCT  *PROC        [PROC_COUNT_MAX];
struct SCSIDEV_STRUCT  *SCSIDEV   [SCSIDEV_COUNT_MAX];
struct SCSIDRV_STRUCT  *SCSIDRV   [SCSIDRV_COUNT_MAX];
struct TTYDRIVERS_STRUCT  *TTYDRIVERS  [TTYDRIVERS_COUNT_MAX];
struct TTYLDISCS_STRUCT   *TTYLDISCS   [TTYLDISCS_COUNT_MAX];
string  *SNDDRV     [SNDDRV_COUNT_MAX];
string  *SNDCONF    [SNDCONF_COUNT_MAX];
string  *SNDAUDIO   [SNDAUDIO_COUNT_MAX];
string  *SNDSYNTH   [SNDAUDIO_COUNT_MAX];
string  *SNDMIDI    [SNDMIDI_COUNT_MAX];
string  *SNDTIMERS  [SNDTIMERS_COUNT_MAX];
string  *SNDMIXERS  [SNDMIXERS_COUNT_MAX];
struct MTRR_STRUCT  *MTRR  [MTRR_COUNT_MAX];
struct IDEDRV_STRUCT  *IDEDRV  [IDEDRV_COUNT_MAX];
struct IDEDEV_STRUCT  *IDEDEV  [IDEDEV_COUNT_MAX];
struct PARPORT_STRUCT  *PARPORT  [PARPORT_COUNT_MAX];
struct DEV_STRUCT  *MISC       [MISC_COUNT_MAX];
int NUM_MODULES=-1;
int NUM_CHARDEV=-1;
int NUM_BLOCKDEV=-1;
int NUM_ISADMA=-1;
int NUM_FILESYS=-1;
int NUM_IRQINT=-1;
int NUM_IOPORTS=-1;
int NUM_PARTS=-1;
int NUM_MOUNTS=-1;
int NUM_SWAPS=-1;
int NUM_CPU=-1;
int NUM_PROC=-1;
int NUM_SCSIDEV=-1;
int NUM_SCSIDRV=-1;
int NUM_TTYDRIVERS=-1;
int NUM_TTYLDISCS=-1; 
int NUM_SNDDRV=-1;
int NUM_SNDCONF=-1;
int NUM_SNDAUDIO=-1;
int NUM_SNDSYNTH=-1;
int NUM_SNDMIDI=-1;
int NUM_SNDTIMERS=-1;
int NUM_SNDMIXERS=-1;
int NUM_MTRR=-1;
int NUM_IDEDRV=-1;
int NUM_IDEDEV=-1;
int NUM_PARPORT=-1;
int NUM_MISC=-1;


// loadinfo() Function
int loadinfo () {
	// Display Status
	fprintf (fstdout, "si:  Loading Information\n");
	// If OPT_DISPLAY_MODE==DISPLAY_MODE_PS then only 
	//   load process information
	//   requires cpu and stat information
	if (OPT_DISPLAY_MODE==DISPLAY_MODE_PS) {
		handlecpu ();
		handlestat ();
		handleprocesses ();
		return 0;
	}
	// Load All proc Filesystem Information
	// si
		sprintf (PROCINFO[PROC_EXINFO], "si - Version 1.0 - By Joe Turgeon (jturgeon@isd.net)");
	// cmdline
		handleproc(PROC_CMDLINE, "cmdline");
	// domainname
		handleproc(PROC_DOMAINNAME, "sys/kernel/domainname");
	// hostname
		handleproc(PROC_HOSTNAME, "sys/kernel/hostname");
	// modprobe
		if (KERNEL != VER20)
			handleproc(PROC_MODPROBE, "sys/kernel/modprobe");
	// osrelease
		handleproc(PROC_OSRELEASE, "sys/kernel/osrelease");
	// ostype
		handleproc(PROC_OSTYPE, "sys/kernel/ostype");
	// version
		handleproc(PROC_VERSION, "sys/kernel/version");
	// modules
		handlemodules ();
	// loadavg and uptime
		handletime ();
	// cpu statistics
		handlecpu ();
	// block and character devices
		handledevices ();
	// isa dma channels
		handleisadma ();
	// filesystems
		handlefilesys ();
	// interrupts
		handleinterrupts ();
	// ioports
		NUM_IOPORTS=readprocline("ioports", IOPORTS);
	// partitions
		if (KERNEL != VER20) {
			NUM_PARTS=readprocline("partitions", PARTS);
			if (NUM_PARTS>=0) sprintf (*PARTS[0], "  Maj   Min    Bytes  Name");
		}
	// stat
		handlestat ();
	// meminfo
		handlememinfo ();
	// mounts
		handlemounts ();
	// swaps
		handleswaps ();
	// processes
		handleprocesses ();
	// scsi devices and drivers
		handlescsi ();
	// tty drivers and ldiscs
		handletty ();
	// sound
		handlesound ();
	// ide
		handleide ();
	// parport
		handleparport ();
	// misc
		handlemisc ();
	// mtrr
		handlemtrr ();
	// Default Return Value
	return 0;
}

// handleproc() Function
void handleproc(int procnum, char *filename) {
	char tmpstring [INPUT_STRING];
	int tmplength;

	tmplength=readproc(filename, tmpstring);
	tmplength=killcrlf(tmpstring, tmplength);
	strncpy (PROCINFO[procnum], tmpstring, STORAGE_STRING-1);
	PROCINFO[procnum][STORAGE_STRING-1]=0;
	return;
}

// readprocline() Function
int readprocline (char *shortname, string *dataarray[READPROCLINE_COUNT]) {
	char filename [1024];
	FILE *filep;
	char tmpstring [INPUT_STRING];
	int c=0;

	sprintf (filename, "%s%s", PROCLOC, shortname);
	filep=fopen(filename, "r");
	if (filep==NULL) return -1;
	do {
		fgets (tmpstring, INPUT_STRING, filep);
		if (feof(filep)!=0) break;
		if (strlen(tmpstring)==1) continue;
		if (strlen(tmpstring) > STORAGE_STRING)
			tmpstring[STORAGE_STRING]=0;
		killcrlf (tmpstring, strlen(tmpstring));
		dataarray[c]=malloc(STRING_ARRAY_SIZE);
		if (dataarray[c]==NULL) error_handler(ERROR_MALLOC);
		strncpy (*dataarray[c], tmpstring, STORAGE_STRING-1);
		(*dataarray[c])[STORAGE_STRING-1]=0;
		if (c==READPROCLINE_COUNT) break;
		c++;
	} while (0==0);
	c--;
	fclose (filep);
	return c;
}

// readproc() Function
int readproc (char *shortname, char *string) {
	char filename[1024];
	FILE *filep;
	char tmpstring [INPUT_STRING];
	int tmplength;
	int sizeread;

	sprintf (filename, "%s%s", PROCLOC, shortname);
	filep=fopen(filename, "r");
	if (filep==NULL) {
		tmpstring[0]=0;
		tmplength=strlen(tmpstring);
	}
	else {
		sizeread=fgetstr(filep, tmpstring, INPUT_STRING-1);
		tmplength=sizeread;
		fclose (filep);
	}
	strncpy(string, tmpstring, STORAGE_STRING-1);
	string[STORAGE_STRING-1]=0;
	return tmplength;
}


/* VERSION 1.0 MODIFICATION:
	ALL SPECIFIC CODE MOVED INTO SEPARATE SOURCE FILES:

// handlemodules() Function
void handlemodules () {
	FILE *filep;
	char filename [INPUT_STRING];
	char tmpstring [INPUT_STRING];
	int c=0;
	int tmplength;
	
	// Open /proc/modules
	sprintf (filename, "%smodules", PROCLOC);
	filep=fopen(filename, "r");
	if (filep==NULL) return;
	// Fill Modules Structure
	do {
		fgets (tmpstring, INPUT_STRING-1, filep);
		if (feof(filep)!=0) break;
		tmplength=killcrlf(tmpstring, strlen(tmpstring));
		if (strlen(tmpstring)==1) continue;
		MODULES[c]=malloc(STRING_ARRAY_SIZE);
		if (MODULES[c]==NULL) error_handler(ERROR_MALLOC);
		strncpy (*MODULES[c], tmpstring, STORAGE_STRING-1);
		(*MODULES[c])[STORAGE_STRING-1]=0;
		c++;
		if (c==MODULES_COUNT_MAX) break;
	} while (0==0);
	NUM_MODULES=c-1;
	// Close and Cleanup
	fclose (filep);
	return;
}

// handletime() Function
void handletime () {
	FILE *filep;
	char filename [INPUT_STRING];
	char strsec [128];
	long int numsec;
	long int nummin;
	long int numhour;
	long int numday;

	// Open and Handle /proc/loadavg
	sprintf (filename, "%sloadavg", PROCLOC);
	filep=fopen(filename, "r");
	if (filep!=NULL) {
		fscanf (filep, "%s %s %s", PROCINFO[PROC_LOADAVG_1],
			PROCINFO[PROC_LOADAVG_5],
			PROCINFO[PROC_LOADAVG_15]);
		fclose (filep);
	}
	// Open and Handle /proc/uptime
	sprintf (filename, "%suptime", PROCLOC);
	filep=fopen(filename, "r");
	if (filep!=NULL) {
		fscanf (filep, "%s", strsec);
		UPTIME=atol(strsec);
		numsec=UPTIME;
		numday=numsec/86400;
		numhour=(numsec-(numday*86400))/3600;
		nummin=(numsec-(numday*86400)-(numhour*3600))/60;
		numsec=(numsec-(numday*86400)-(numhour*3600)-(nummin*60));
		sprintf (PROCINFO[PROC_UPTIME], "%ld Days  %ld Hours  %ld Minutes  %ld Seconds (%s Seconds)", numday, numhour, nummin, numsec, strsec);
		fclose (filep);
	}
	return;
}

// handlecpuv20 ()
// 	code specific to the Linux Kernel 2.0
void handlecpuv20 () {
	FILE *filep;
	char filename [INPUT_STRING];
	char tmpstring [INPUT_STRING];
	int colpos;
	int c=-1;

	sprintf (filename, "%scpuinfo", PROCLOC);
	filep=fopen(filename, "r");
	if (filep==NULL) return;
	do {
		fgets (tmpstring, INPUT_STRING, filep);
		if (feof(filep)!=0) break;
		if (strlen(tmpstring) > STORAGE_STRING)
			tmpstring[STORAGE_STRING]=0;
		if (strncmp(tmpstring, "processor", 9)==0) {
			c++;
			if (c==CPU_COUNT_MAX) break;
			CPU[c]=malloc(CPU_STRUCT_SIZE);
			if (CPU[c]==NULL) error_handler(ERROR_MALLOC);
			CPU[c]->SPEED[0]=0;
			CPU[c]->CACHE[0]=0;
		}
		if (strncmp(tmpstring, "vendor_id", 9)==0) {
			killcrlf(tmpstring, strlen(tmpstring));
			colpos=cinstr(tmpstring, ':');
			rstring(tmpstring, colpos+2, CPU[c]->VENDOR);
		}
		if (strncmp(tmpstring, "model", 5)==0) {
			killcrlf(tmpstring, strlen(tmpstring));
			colpos=cinstr(tmpstring, ':');
			rstring(tmpstring, colpos+2, CPU[c]->MODEL);
		}
	} while (0==0);
	NUM_CPU=c;
	fclose (filep);
	// Check All Values
	for (c=0; c<=NUM_CPU; c++) {
		if (strlen(CPU[c]->VENDOR)==0) sprintf(CPU[c]->VENDOR, "unknown");
		if (strlen(CPU[c]->MODEL)==0)  sprintf(CPU[c]->MODEL, "unknown");
		if (strlen(CPU[c]->SPEED)==0)  sprintf(CPU[c]->SPEED, "unknown");
		if (strlen(CPU[c]->CACHE)==0)  sprintf(CPU[c]->CACHE, "unknown");
	}
	// Return
	return;
}

// handlecpuv22 ()
//	code specific to the Linux Kernel 2.2
void handlecpuv22 () {
	FILE *filep;
	char filename [INPUT_STRING];
	char tmpstring [INPUT_STRING];
	int colpos;
	int c=-1;

	sprintf (filename, "%scpuinfo", PROCLOC);
	filep=fopen(filename, "r");
	if (filep==NULL) return;
	do {
		fgets (tmpstring, INPUT_STRING, filep);
		if (feof(filep)!=0) break;
		if (strlen(tmpstring) > STORAGE_STRING)
			tmpstring[STORAGE_STRING]=0;
		killcrlf(tmpstring, strlen(tmpstring));
		colpos=cinstr(tmpstring, ':');
		// on "processor" increment CPU counter
		if (strncmp(tmpstring, "processor", 9)==0) {
			c++;
			if (c==CPU_COUNT_MAX) break;
			CPU[c]=malloc(CPU_STRUCT_SIZE);
			if (CPU[c]==NULL) error_handler(ERROR_MALLOC);
		}
		// "vendor_id" information
		if (strncmp(tmpstring, "vendor_id", 9)==0)
			rstring(tmpstring, colpos+2, CPU[c]->VENDOR);
		// "model name" information
		if (strncmp(tmpstring, "model name", 10)==0)
			rstring(tmpstring, colpos+2, CPU[c]->MODEL);
		// "cpu MHz" information
		if (strncmp(tmpstring, "cpu MHz", 7)==0)
			rstring(tmpstring, colpos+2, CPU[c]->SPEED);
		// "cache size" information
		if (strncmp(tmpstring, "cache size", 10)==0)
			rstring(tmpstring, colpos+2, CPU[c]->CACHE);
	} while (0==0);
	NUM_CPU=c;
	fclose (filep);
	// Check All Values
	for (c=0; c<=NUM_CPU; c++) {
		if (strlen(CPU[c]->VENDOR)==0) sprintf(CPU[c]->VENDOR, "unknown");
		if (strlen(CPU[c]->MODEL)==0)  sprintf(CPU[c]->MODEL, "unknown");
		if (strlen(CPU[c]->SPEED)==0)  sprintf(CPU[c]->SPEED, "unknown");
		if (strlen(CPU[c]->CACHE)==0)  sprintf(CPU[c]->CACHE, "unknown");
	}
	// Return
	return;
}

// handledevices() Function
void handledevices () {
	FILE *filep;
	char filename [INPUT_STRING];
	char tmpstring [INPUT_STRING];
	int c;

	sprintf (filename, "%sdevices", PROCLOC);
	filep=fopen(filename, "r");
	if (filep==NULL) return;
	// Character Devices
	c=0;
	do {
		fgets (tmpstring, INPUT_STRING, filep);
		if (strlen(tmpstring)==1) continue;
		if (strncmp(tmpstring, "Character devices", 17)==0) continue;
		if (strncmp(tmpstring, "Block devices", 13)==0) break;
		killcrlf (tmpstring, strlen(tmpstring));
		CHARDEV[c]=malloc(DEV_STRUCT_SIZE);
		if (CHARDEV[c]==NULL) error_handler(ERROR_MALLOC);
		sscanf (tmpstring, "%d %s", &CHARDEV[c]->NUMBER, CHARDEV[c]->NAME);
		c++;
		if (c==CHARDEV_COUNT_MAX) break;
	} while (0==0);
	NUM_CHARDEV=c-1;
	// Block Devices
	c=0;
	do {
		fgets (tmpstring, INPUT_STRING, filep);
		if (feof(filep)!=0) break;
		if (strncmp(tmpstring, "Block devices", 13)==0) continue;
		killcrlf (tmpstring, strlen(tmpstring));
		BLOCKDEV[c]=malloc(DEV_STRUCT_SIZE);
		if (BLOCKDEV[c]==NULL) error_handler(ERROR_MALLOC);
		sscanf (tmpstring, "%d %s", &BLOCKDEV[c]->NUMBER, BLOCKDEV[c]->NAME);
		c++;
		if (c==BLOCKDEV_COUNT_MAX) break;
	} while (0==0);
	NUM_BLOCKDEV=c-1;
	// Cleanup
	fclose (filep);
	return;
}

// handleisadma() Function
void handleisadma () {
	char filename [INPUT_STRING];
	char tmpstring [INPUT_STRING];
	FILE *filep;
	int c=0;

	sprintf (filename, "%sdma", PROCLOC);
	filep=fopen(filename, "r");
	if (filep==NULL) return;
	do {
		fgets (tmpstring, INPUT_STRING, filep);
		if (feof(filep)!=0) break;
		if (strlen(tmpstring)==1) continue;
		killcrlf (tmpstring, strlen(tmpstring));
		ISADMA[c]=malloc(DEV_STRUCT_SIZE);
		if (ISADMA[c]==NULL) error_handler(ERROR_MALLOC);
		sscanf (tmpstring, "%2d%*1s %s", &ISADMA[c]->NUMBER, ISADMA[c]->NAME);
		c++;
		if (c==ISADMA_COUNT_MAX) break;
	} while (0==0);
	NUM_ISADMA=c-1;
	fclose (filep);
	return;
}

// handlefilesys() Function
void handlefilesys () {
	FILE *filep;
	char filename [INPUT_STRING];
	char tmpstring [INPUT_STRING];
	int c=0;

	sprintf (filename, "%sfilesystems", PROCLOC);
	filep=fopen(filename, "r");
	if (filep==NULL) return;
	do {
		fgets (tmpstring, INPUT_STRING, filep);
		if (feof(filep)!=0) break;
		if (strlen(tmpstring) > STORAGE_STRING)
			tmpstring[STORAGE_STRING]=0;
		FILESYS[c]=malloc(FILESYS_STRUCT_SIZE);
		if (FILESYS[c]==NULL) error_handler(ERROR_MALLOC);
		killcrlf (tmpstring, strlen(tmpstring));
		if (cinstr(tmpstring, 9)==0)
			rstring (tmpstring, 1, FILESYS[c]->NAME);
		else
			sscanf (tmpstring, "%s\t%s", FILESYS[c]->DEV, FILESYS[c]->NAME);
		c++;
		if (c==FILESYS_COUNT_MAX) break;
	} while (0==0);
	NUM_FILESYS=c-1;
	fclose (filep);
	return;
}

// handleinterruptsv20() Function
// 	code specific to the Linux Kernel 2.0
void handleinterruptsv20 () {
	FILE *filep;
	char filename [INPUT_STRING];
	char tmpstring [INPUT_STRING];
	int intnum;
	char intname [INPUT_STRING];
	char intcalls [INPUT_STRING];
	int c=0;

	sprintf (filename, "%sinterrupts", PROCLOC);
	filep=fopen(filename, "r");
	if (filep==NULL) return;
	do {
		fgets (tmpstring, INPUT_STRING, filep);
		if (feof(filep)!=0) break;
		if (strlen(tmpstring)==1) continue;
		sscanf (tmpstring, "%i %*1s %11s", &intnum, intcalls);
		rstring (tmpstring, 17, intname);
		killcrlf (intname, strlen(intname));
		IRQINT[c]=malloc(STRING_ARRAY_SIZE);
		if (IRQINT[c]==NULL) error_handler(ERROR_MALLOC);
		sprintf (*IRQINT[c], "%3i  %12s  %s", intnum, intcalls, intname);
		c++;
		if (c==IRQINT_COUNT_MAX) break;
	} while (0==0);
	NUM_IRQINT=c-1;
	fclose (filep);
	return;
}

// handleinterruptsv22() Function
// 	code specific to the Linux Kernel 2.2
void handleinterruptsv22 () {
	FILE *filep;
	char filename [INPUT_STRING];
	char tmpstring [INPUT_STRING];
	int intnum;
	char intname [INPUT_STRING];
	char intcalls [INPUT_STRING];
	char intdesc [INPUT_STRING];
	int c=0;
	int i=0;

	sprintf (filename, "%sinterrupts", PROCLOC);
	filep=fopen(filename, "r");
	if (filep==NULL) return;
	do {
		fgets (tmpstring, INPUT_STRING, filep);
		if (feof(filep)!=0) break;
		if (strlen(tmpstring)==1) continue;
		if (strncmp(tmpstring, "           CPU0", 15)==0) continue;
		if (strncmp(tmpstring, "NMI", 3)==0) continue;
		if (strncmp(tmpstring, "ERR", 3)==0) continue;
		sscanf (tmpstring, "%i %*1s", &intnum);
		IRQINT[c]=malloc(STRING_ARRAY_SIZE);
		if (IRQINT[c]==NULL) error_handler(ERROR_MALLOC);
		sprintf (*IRQINT[c], "%3i", intnum);
		rstring (tmpstring, 5, tmpstring);
		for (i=0; i<=NUM_CPU; i++) {
			sscanf (tmpstring, "%11s", intcalls);
			sprintf (*IRQINT[c], "%s  %12s", *IRQINT[c], intcalls);
			rstring (tmpstring, 12, tmpstring);
		}
		killcrlf (tmpstring, strlen(tmpstring));
		sscanf (tmpstring, "%s %s", intdesc, intname);
		sprintf (*IRQINT[c], "%s  %15s  %s", *IRQINT[c], intdesc, intname);
		c++;
		if (c==IRQINT_COUNT_MAX) break;
	} while (0==0);
	NUM_IRQINT=c-1;
	fclose (filep);
	return;
}

// handlestatv20() Function
// 	code specific to the Linux Kernel 2.0
void handlestatv20 () {
	FILE *filep;
	char filename [INPUT_STRING];
	char tmpstring [INPUT_STRING];
	char field [INPUT_STRING];
	char strsec [INPUT_STRING];
	long int totalsec;
	long int numsec;
	long int nummin;
	long int numhour;
	long int cpuuser, cpunice, cpusys, cpuidle;

	sprintf (filename, "%sstat", PROCLOC);
	filep=fopen(filename, "r");
	if (filep==NULL) return;
	do {
		fgets (tmpstring, INPUT_STRING, filep);
		if (feof(filep)!=0) break;
		sscanf (tmpstring, "%s", field);
		if (strcmp(field, "cpu")==0) {
			sscanf (tmpstring, "%s %s %s %s %s", field, PROCINFO[PROC_CPUUSER], PROCINFO[PROC_CPUNICE], PROCINFO[PROC_CPUSYS], PROCINFO[PROC_CPUIDLE]);
			// PROC_CPUUSER
			strncpy (strsec, PROCINFO[PROC_CPUUSER], INPUT_STRING-1);
			PROCINFO[PROC_CPUUSER][INPUT_STRING-1]=0;
			cpuuser=atol(strsec);
			totalsec=atol(strsec)/100;
			numhour=totalsec/3600;
			nummin=(totalsec-(numhour*3600))/60;
			numsec=(totalsec-(numhour*3600)-(nummin*60));
			sprintf (PROCINFO[PROC_CPUUSER], "%7ld Seconds (%3ld:%.2ld:%.2ld)", totalsec, numhour, nummin, numsec);
			// PROC_CPUNICE
			strncpy (strsec, PROCINFO[PROC_CPUNICE], INPUT_STRING-1);
			PROCINFO[PROC_CPUNICE][INPUT_STRING-1]=0;
			cpunice=atol(strsec);
			totalsec=atol(strsec)/100;
			numhour=totalsec/3600;
			nummin=(totalsec-(numhour*3600))/60;
			numsec=(totalsec-(numhour*3600)-(nummin*60));
			sprintf (PROCINFO[PROC_CPUNICE], "%7ld Seconds (%3ld:%.2ld:%.2ld)", totalsec, numhour, nummin, numsec);
			// PROC_CPUSYS
			strncpy (strsec, PROCINFO[PROC_CPUSYS], INPUT_STRING-1);
			PROCINFO[PROC_CPUSYS][INPUT_STRING-1]=0;
			cpusys=atol(strsec);
			totalsec=atol(strsec)/100;
			numhour=totalsec/3600;
			nummin=(totalsec-(numhour*3600))/60;
			numsec=(totalsec-(numhour*3600)-(nummin*60));
			sprintf (PROCINFO[PROC_CPUSYS], "%7ld Seconds (%3ld:%.2ld:%.2ld)", totalsec, numhour, nummin, numsec);
			// PROC_CPUIDLE
			cpuidle = (((UPTIME*100)*(NUM_CPU+1)) - (cpuuser + cpunice + cpusys));
			totalsec = cpuidle / 100;
			numhour=totalsec/3600;
			nummin=(totalsec-(numhour*3600))/60;
			numsec=(totalsec-(numhour*3600)-(nummin*60));
			sprintf (PROCINFO[PROC_CPUIDLE], "%7ld Seconds (%3ld:%.2ld:%.2ld)", totalsec, numhour, nummin, numsec);
		}
		if (strcmp(field, "disk")==0) {
			sscanf (tmpstring, "%s %s", field, PROCINFO[PROC_DISK]);
		}
		if (strcmp(field, "disk_rio")==0) {
			sscanf (tmpstring, "%s %s", field, PROCINFO[PROC_DISKRIO]);
		}
		if (strcmp(field, "disk_wio")==0) {
			sscanf (tmpstring, "%s %s", field, PROCINFO[PROC_DISKWIO]);
		}
		if (strcmp(field, "disk_rblk")==0) {
			sscanf (tmpstring, "%s %s", field, PROCINFO[PROC_DISKRBLK]);
		}
		if (strcmp(field, "disk_wblk")==0) {
			sscanf (tmpstring, "%s %s", field, PROCINFO[PROC_DISKWBLK]);
		}
		if (strcmp(field, "page")==0) {
			sscanf (tmpstring, "%s %s %s", field, PROCINFO[PROC_PAGE_IN], PROCINFO[PROC_PAGE_OUT]);
		}
		if (strcmp(field, "swap")==0) {
			sscanf (tmpstring, "%s %s %s", field, PROCINFO[PROC_SWAP_IN], PROCINFO[PROC_SWAP_OUT]);
		}
		if (strcmp(field, "intr")==0) {
		}
		if (strcmp(field, "ctxt")==0) {
			sscanf (tmpstring, "%s %s", field, PROCINFO[PROC_CTXT]);
		}
		if (strcmp(field, "btime")==0) {
			sscanf (tmpstring, "%s %ld", field, &BOOTTIME);
			strncpy (tmpstring, ctime(&BOOTTIME), INPUT_STRING-1);
			tmpstring[INPUT_STRING-1]=0;
			killcrlf (tmpstring, strlen(tmpstring));
			strncpy (PROCINFO[PROC_BTIME], tmpstring, STORAGE_STRING-1);
			PROCINFO[PROC_BTIME][STORAGE_STRING-1]=0;
		}
		if (strcmp(field, "processes")==0) {
			sscanf (tmpstring, "%s %s", field, PROCINFO[PROC_PROCESS]);
		}
	} while (0==0);
	// Cleanup
	fclose (filep);
	return;
}

// handlestatv22() Function
// 	code specific to the Linux Kernel 2.2
void handlestatv22 () {
	FILE *filep;
	char filename [INPUT_STRING];
	char tmpstring [INPUT_STRING];
	char field [INPUT_STRING];
	char strsec [INPUT_STRING];
	long int totalsec;
	long int numsec;
	long int nummin;
	long int numhour;
	int i=0;

	sprintf (filename, "%sstat", PROCLOC);
	filep=fopen(filename, "r");
	if (filep==NULL) return;
	do {
		fgets (tmpstring, INPUT_STRING, filep);
		if (feof(filep)!=0) break;
		sscanf (tmpstring, "%s", field);
		if (strcmp(field, "cpu")==0) {
			sscanf (tmpstring, "%s %s %s %s %s", field, PROCINFO[PROC_CPUUSER], PROCINFO[PROC_CPUNICE], PROCINFO[PROC_CPUSYS], PROCINFO[PROC_CPUIDLE]);
			// PROC_CPUUSER
			strncpy (strsec, PROCINFO[PROC_CPUUSER], INPUT_STRING-1);
			strsec[INPUT_STRING-1]=0;
			totalsec=atol(strsec)/100;
			numhour=totalsec/3600;
			nummin=(totalsec-(numhour*3600))/60;
			numsec=(totalsec-(numhour*3600)-(nummin*60));
			sprintf (PROCINFO[PROC_CPUUSER], "%7ld Seconds (%3ld:%.2ld:%.2ld)", totalsec, numhour, nummin, numsec);
			// PROC_CPUNICE
			strncpy (strsec, PROCINFO[PROC_CPUNICE], INPUT_STRING-1);
			strsec[INPUT_STRING-1]=0;
			totalsec=atol(strsec)/100;
			numhour=totalsec/3600;
			nummin=(totalsec-(numhour*3600))/60;
			numsec=(totalsec-(numhour*3600)-(nummin*60));
			sprintf (PROCINFO[PROC_CPUNICE], "%7ld Seconds (%3ld:%.2ld:%.2ld)", totalsec, numhour, nummin, numsec);
			// PROC_CPUSYS
			strncpy (strsec, PROCINFO[PROC_CPUSYS], INPUT_STRING-1);
			strsec[INPUT_STRING-1]=0;
			totalsec=atol(strsec)/100;
			numhour=totalsec/3600;
			nummin=(totalsec-(numhour*3600))/60;
			numsec=(totalsec-(numhour*3600)-(nummin*60));
			sprintf (PROCINFO[PROC_CPUSYS], "%7ld Seconds (%3ld:%.2ld:%.2ld)", totalsec, numhour, nummin, numsec);
			// PROC_CPUIDLE
			strncpy (strsec, PROCINFO[PROC_CPUIDLE], INPUT_STRING-1);
			strsec[INPUT_STRING-1]=0;
			totalsec=atol(strsec)/100;
			numhour=totalsec/3600;
			nummin=(totalsec-(numhour*3600))/60;
			numsec=(totalsec-(numhour*3600)-(nummin*60));
			sprintf (PROCINFO[PROC_CPUIDLE], "%7ld Seconds (%3ld:%.2ld:%.2ld)", totalsec, numhour, nummin, numsec);
			// Single-CPU Computers
			if (NUM_CPU==0) {
				strncpy (CPU[0]->USER, PROCINFO[PROC_CPUUSER], STORAGE_STRING-1);
				CPU[0]->USER[STORAGE_STRING-1]=0;
				strncpy (CPU[0]->NICE, PROCINFO[PROC_CPUNICE], STORAGE_STRING-1);
				CPU[0]->NICE[STORAGE_STRING-1]=0;
				strncpy (CPU[0]->SYS,  PROCINFO[PROC_CPUSYS],  STORAGE_STRING-1);
				CPU[0]->SYS[STORAGE_STRING-1]=0;
				strncpy (CPU[0]->IDLE, PROCINFO[PROC_CPUIDLE], STORAGE_STRING-1);
				CPU[0]->IDLE[STORAGE_STRING-1]=0;
			}
		}
		if ((strncmp(field, "cpu", 3)==0) & (isdigit(field[3])!=0)) {
			sscanf (field, "%*3s%d", &i);
			sscanf (tmpstring, "%s %s %s %s %s", field, CPU[i]->USER, CPU[i]->NICE, CPU[i]->SYS, CPU[i]->IDLE);
			// CPU.USER
			strncpy (strsec, CPU[i]->USER, INPUT_STRING-1);
			strsec[INPUT_STRING-1]=0;
			totalsec=atol(strsec)/100;
			numhour=totalsec/3600;
			nummin=(totalsec-(numhour*3600))/60;
			numsec=(totalsec-(numhour*3600)-(nummin*60));
			sprintf (CPU[i]->USER, "%7ld Seconds (%3ld:%.2ld:%.2ld)", totalsec, numhour, nummin, numsec);
			// CPU.NICE
			strncpy (strsec, CPU[i]->NICE, INPUT_STRING-1);
			strsec[INPUT_STRING-1]=0;
			totalsec=atol(strsec)/100;
			numhour=totalsec/3600;
			nummin=(totalsec-(numhour*3600))/60;
			numsec=(totalsec-(numhour*3600)-(nummin*60));
			sprintf (CPU[i]->NICE, "%7ld Seconds (%3ld:%.2ld:%.2ld)", totalsec, numhour, nummin, numsec);
			// CPU.SYS
			strncpy (strsec, CPU[i]->SYS, INPUT_STRING-1);
			strsec[INPUT_STRING-1]=0;
			totalsec=atol(strsec)/100;
			numhour=totalsec/3600;
			nummin=(totalsec-(numhour*3600))/60;
			numsec=(totalsec-(numhour*3600)-(nummin*60));
			sprintf (CPU[i]->SYS, "%7ld Seconds (%3ld:%.2ld:%.2ld)", totalsec, numhour, nummin, numsec);
			// CPU.IDLE
			strncpy (strsec, CPU[i]->IDLE, INPUT_STRING-1);
			strsec[INPUT_STRING-1]=0;
			totalsec=atol(strsec)/100;
			numhour=totalsec/3600;
			nummin=(totalsec-(numhour*3600))/60;
			numsec=(totalsec-(numhour*3600)-(nummin*60));
			sprintf (CPU[i]->IDLE, "%7ld Seconds (%3ld:%.2ld:%.2ld)", totalsec, numhour, nummin, numsec);
		}
		if (strcmp(field, "disk")==0) {
			sscanf (tmpstring, "%s %s", field, PROCINFO[PROC_DISK]);
		}
		if (strcmp(field, "disk_rio")==0) {
			sscanf (tmpstring, "%s %s", field, PROCINFO[PROC_DISKRIO]);
		}
		if (strcmp(field, "disk_wio")==0) {
			sscanf (tmpstring, "%s %s", field, PROCINFO[PROC_DISKWIO]);
		}
		if (strcmp(field, "disk_rblk")==0) {
			sscanf (tmpstring, "%s %s", field, PROCINFO[PROC_DISKRBLK]);
		}
		if (strcmp(field, "disk_wblk")==0) {
			sscanf (tmpstring, "%s %s", field, PROCINFO[PROC_DISKWBLK]);
		}
		if (strcmp(field, "page")==0) {
			sscanf (tmpstring, "%s %s %s", field, PROCINFO[PROC_PAGE_IN], PROCINFO[PROC_PAGE_OUT]);
		}
		if (strcmp(field, "swap")==0) {
			sscanf (tmpstring, "%s %s %s", field, PROCINFO[PROC_SWAP_IN], PROCINFO[PROC_SWAP_OUT]);
		}
		if (strcmp(field, "intr")==0) {
		}
		if (strcmp(field, "ctxt")==0) {
			sscanf (tmpstring, "%s %s", field, PROCINFO[PROC_CTXT]);
		}
		if (strcmp(field, "btime")==0) {
			sscanf (tmpstring, "%s %ld", field, &BOOTTIME);
			strncpy (tmpstring, ctime(&BOOTTIME), INPUT_STRING-1);
			tmpstring[INPUT_STRING-1]=0;
			killcrlf (tmpstring, strlen(tmpstring));
			strncpy (PROCINFO[PROC_BTIME], tmpstring, STORAGE_STRING-1);
			PROCINFO[PROC_BTIME][STORAGE_STRING-1]=0;
		}
		if (strcmp(field, "processes")==0) {
			sscanf (tmpstring, "%s %s", field, PROCINFO[PROC_PROCESS]);
		}
	} while (0==0);
	// Cleanup
	fclose (filep);
	return;
}

// handlememinfo() Function
void handlememinfo () {
	FILE *filep;
	char filename [INPUT_STRING];
	char tmpstring [INPUT_STRING];
	char field [INPUT_STRING];

	sprintf (filename, "%smeminfo", PROCLOC);
	filep=fopen(filename, "r");
	if (filep==NULL) return;
	do {
		fgets (tmpstring, INPUT_STRING, filep);
		if (feof(filep)!=0) break;
		sscanf (tmpstring, "%s", field);
		if (strcmp(field, "MemTotal:")==0)
			sscanf (tmpstring, "%s %s", field, PROCINFO[PROC_MEMTOTAL]);
		if (strcmp(field, "MemFree:")==0)
			sscanf (tmpstring, "%s %s", field, PROCINFO[PROC_MEMFREE]);
		if (strcmp(field, "MemShared:")==0)
			sscanf (tmpstring, "%s %s", field, PROCINFO[PROC_MEMSHARED]);
		if (strcmp(field, "Buffers:")==0)
			sscanf (tmpstring, "%s %s", field, PROCINFO[PROC_BUFFERS]);
		if (strcmp(field, "Cached:")==0)
			sscanf (tmpstring, "%s %s", field, PROCINFO[PROC_CACHED]);
		if (strcmp(field, "SwapTotal:")==0)
			sscanf (tmpstring, "%s %s", field, PROCINFO[PROC_SWAPTOTAL]);
		if (strcmp(field, "SwapFree:")==0)
			sscanf (tmpstring, "%s %s", field, PROCINFO[PROC_SWAPFREE]);
	} while (0==0);
	// Cleanup
	fclose (filep);
	return;
}

// handlemounts() Function
void handlemounts () {
	FILE *filep;
	char filename [INPUT_STRING];
	char tmpstring [INPUT_STRING];
	int c=0;

	sprintf (filename, "%smounts", PROCLOC);
	filep=fopen(filename, "r");
	if (filep==NULL) return;
	do {
		fgets (tmpstring, INPUT_STRING, filep);
		if (feof(filep)!=0) break;
		MOUNTS[c]=malloc(MOUNTS_STRUCT_SIZE);
		if (MOUNTS[c]==NULL) error_handler(ERROR_MALLOC);
		sscanf (tmpstring, "%s %s %s %s", MOUNTS[c]->DEVICE, MOUNTS[c]->MOUNTPOINT, MOUNTS[c]->FILESYS, MOUNTS[c]->MODE);
		c++;
		if (c==MOUNTS_COUNT_MAX) break;
	} while (0==0);
	NUM_MOUNTS=c-1;
	// Cleanup
	fclose (filep);
	return;
}

// handleswaps() Function
void handleswaps () {
	FILE *filep;
	char tmpstring [INPUT_STRING];
	char filename [INPUT_STRING];
	char size [INPUT_STRING];
	char used [INPUT_STRING];
	int c=0;

	if (KERNEL==VER20) return;
	sprintf (filename, "%sswaps", PROCLOC);
	filep=fopen(filename, "r");
	if (filep==NULL) return;
	do {
		fgets (tmpstring, INPUT_STRING, filep);
		if (feof(filep)!=0) break;
		sscanf (tmpstring, "%s %*s %s %s %*s", filename, size, used);
		if (strcmp(filename, "Filename")==0) continue;
		SWAPS[c]=malloc(STRING_ARRAY_SIZE);
		if (SWAPS[c]==NULL) error_handler(ERROR_MALLOC);
		sprintf (*SWAPS[c], "%-20s%-10s%-10s", filename, size, used);
		c++;
		if (c==SWAPS_COUNT_MAX) break;
	} while (0==0);
	NUM_SWAPS=c-1;
	// Cleanup
	fclose (filep);
	return;
}

// handleprocesses() Function
void handleprocesses () {
	DIR *procdir;
	struct dirent *dirname;
	char tmpstring [INPUT_STRING];
	int i=0;
	int c=0;
	char filename [INPUT_STRING];
	char dirstr [INPUT_STRING];
	FILE *filep;
	time_t starttime;
	time_t curtime;
	long int totaltime;
	long int numday;
	long int numhour;
	long int nummin;
	long int numsec;
	char tty [INPUT_STRING];

	// Open Directory
	procdir=opendir(PROCLOC);
	if (procdir==NULL) return;
	// Read Entries
	do {
		dirname=readdir(procdir);
		if (dirname==NULL) break;
		strncpy (tmpstring, dirname->d_name, INPUT_STRING-1);
		tmpstring[INPUT_STRING-1]=0;
		if (isdigit(tmpstring[0])) {
			PROC[c]=malloc(PROC_STRUCT_SIZE);
			if (PROC[c]==NULL) error_handler(ERROR_MALLOC);
			PROC[c]->PID=atoi(tmpstring);
			c++;
			if (c==PROC_COUNT_MAX) break;
		}
	} while (0==0);
	NUM_PROC=c-1;
	// Close Directory
	closedir (procdir);
	// Retrieve Basic Information on Each Process
	for (i=0; i<=NUM_PROC; i++) {
		// cmdline
		sprintf (filename, "%s%ld/cmdline", PROCLOC, PROC[i]->PID);
		filep=fopen(filename, "r");
		if (filep!=NULL) {
			fgets (PROC[i]->CMDLINE, STORAGE_STRING, filep);
			fclose (filep);
		}
		// cwd
		sprintf (filename, "%s%ld/cwd", PROCLOC, PROC[i]->PID);
		readlink (filename, PROC[i]->CWD, STORAGE_STRING);
		// exe
		sprintf (filename, "%s%ld/exe", PROCLOC, PROC[i]->PID);
		readlink (filename, PROC[i]->EXE, STORAGE_STRING);
		// stat
		sprintf (filename, "%s%ld/stat", PROCLOC, PROC[i]->PID);
		filep=fopen(filename, "r");
		if (filep!=NULL) {
			fscanf (filep, "%*d %s %s %d %d %d %s %d %*u %*u", PROC[i]->NAME, PROC[i]->STATE, &PROC[i]->PPID, &PROC[i]->PGRP, &PROC[i]->SESSION, tty, &PROC[i]->TPGID);
			fscanf (filep, "%*u %*u %*u %d %d %*d %*d %*d %d %*u %*u %ld %u %*u %*u %u %u", &PROC[i]->UTIME, &PROC[i]->STIME, &PROC[i]->PRIORITY, &starttime, &PROC[i]->VSIZE, &PROC[i]->STARTCODE, &PROC[i]->ENDCODE);
			fscanf (filep, "%*u %*u %*u %*u %*u %*u %*u %*u %*u %*u");
			fclose (filep);
			// Drop Parentheses on Name
			rstring (PROC[i]->NAME, 1, PROC[i]->NAME);
			PROC[i]->NAME[strlen(PROC[i]->NAME)-1]=0;
			// Convert State
			if (strcmp(PROC[i]->STATE, "R")) sprintf(PROC[i]->STATE, "Running");
			if (strcmp(PROC[i]->STATE, "S")) sprintf(PROC[i]->STATE, "Uninterruptible");
			if (strcmp(PROC[i]->STATE, "D")) sprintf(PROC[i]->STATE, "Swapping");
			if (strcmp(PROC[i]->STATE, "Z")) sprintf(PROC[i]->STATE, "Zombie");
			if (strcmp(PROC[i]->STATE, "T")) sprintf(PROC[i]->STATE, "Stopped");
			// Fix Priority Level
			PROC[i]->PRIORITY = PROC[i]->PRIORITY + 1;
			// Fix Start Time
			starttime = (starttime / 100) + BOOTTIME;
			strncpy (PROC[i]->STARTTIME, ctime(&starttime), STORAGE_STRING-1);
			PROC[i]->STARTTIME[STORAGE_STRING-1]=0;
			killcrlf (PROC[i]->STARTTIME, strlen(PROC[i]->STARTTIME));
			// Fix TTY
			if (strcmp(tty, "0")==0)
				sprintf (PROC[i]->TTY, "unknown (%s)", tty);
			else
				sprintf (PROC[i]->TTY, "tty%d (%s)", (atoi(tty)-1024), tty);
			// Create Total Time
			time (&curtime);
			totaltime = curtime - starttime;
			numday=totaltime/86400;
			numhour=(totaltime-(numday*86400))/3600;
			nummin=(totaltime-(numday*86400)-(numhour*3600))/60;
			numsec=(totaltime-(numday*86400)-(numhour*3600)-(nummin*60));
			sprintf (PROC[i]->TOTALTIME, "%ld Days  %ld Hours  %ld Minutes  %ld Seconds", numday, numhour, nummin, numsec);
		}
		// files in use
		c=0;
		sprintf (filename, "%s%ld/fd", PROCLOC, PROC[i]->PID);
		procdir=opendir(filename);
		if (procdir!=NULL) {
			do {
				fillstring (tmpstring, 0, INPUT_STRING);
				dirname=readdir(procdir);
				if (dirname==NULL) break;
				strncpy (tmpstring, dirname->d_name, INPUT_STRING-1);
				tmpstring[INPUT_STRING-1]=0;
				if (strcmp(tmpstring, ".")==0) continue;
				if (strcmp(tmpstring, "..")==0) continue;
				if (strcmp(tmpstring, "0")==0) strcpy(tmpstring, "(stdin)");
				if (strcmp(tmpstring, "1")==0) strcpy(tmpstring, "(stdout)");
				if (strcmp(tmpstring, "2")==0) strcpy(tmpstring, "(stderr)");
				sprintf (PROC[i]->FILES[c], "%8s : ", tmpstring);
				sprintf (dirstr, "%s/%s", filename, dirname->d_name);
				fillstring (tmpstring, 0, INPUT_STRING);
				readlink (dirstr, tmpstring, INPUT_STRING);
				strncat (PROC[i]->FILES[c], tmpstring, STORAGE_STRING);
				c++;
				if (c==128) break;
			} while (0==0);
			closedir (procdir);
		}
		PROC[i]->NUM_FILES=c-1;
	}
	// Return
	return;
}

// handlescsi() Function
void handlescsi () {
	char filename [INPUT_STRING];
	char tmpstring [INPUT_STRING];
	FILE *filep;
	DIR *scsidir;
	struct dirent *dirname;
	int c=-1;

	// Open /proc/scsi
	sprintf (filename, "%sscsi/scsi", PROCLOC);
	filep=fopen(filename, "r");
	if (filep!=NULL) {
		do {
			fgets (tmpstring, INPUT_STRING, filep);
			if (feof(filep)!=0) break;
			killcrlf (tmpstring, strlen(tmpstring));
			if (strncmp(tmpstring, "Attached devices:", 17)==0) continue;
			if (strncmp(tmpstring, "Host:", 5)==0) {
				c++;
				if (c==SCSIDEV_COUNT_MAX) break;
				SCSIDEV[c]=malloc(SCSIDEV_STRUCT_SIZE);
				if (SCSIDEV[c]==NULL) error_handler(ERROR_MALLOC);
				sscanf (tmpstring, "%*5s %5s %*8s %2d %*3s %2d %*4s %2d", SCSIDEV[c]->HOST, &SCSIDEV[c]->CHANNEL, &SCSIDEV[c]->ID, &SCSIDEV[c]->LUN);
			}
			if (strncmp(tmpstring, "  Vendor:", 9)==0) {
				strncpy (SCSIDEV[c]->VENDOR, tmpstring + 10, 9);
				SCSIDEV[c]->VENDOR[9]=0;
				trim (SCSIDEV[c]->VENDOR);
				strncpy (SCSIDEV[c]->MODEL, tmpstring + 26, 17);
				SCSIDEV[c]->MODEL[17]=0;
				trim (SCSIDEV[c]->MODEL);
				strncpy (SCSIDEV[c]->REV, tmpstring + 48, STORAGE_STRING);
				SCSIDEV[c]->REV[STORAGE_STRING]=0;
				trim (SCSIDEV[c]->REV);
			}
			if (strncmp(tmpstring, "  Type:", 7)==0) {
				strncpy (SCSIDEV[c]->TYPE, tmpstring + 7, 33);
				SCSIDEV[c]->TYPE[33]=0;
				trim (SCSIDEV[c]->TYPE);
				strncpy (SCSIDEV[c]->ANSIREV, tmpstring + 62, STORAGE_STRING-1);
				SCSIDEV[c]->ANSIREV[STORAGE_STRING-1]=0;
				trim (SCSIDEV[c]->ANSIREV);
			}
		} while (0==0);
		NUM_SCSIDEV=c;
		fclose (filep);
	}
	// Find SCSI Drivers
	c=0;
	sprintf (filename, "%sscsi", PROCLOC);
	scsidir=opendir(filename);
	if (scsidir==NULL) return;
	// Read Entries
	do {
		dirname=readdir(scsidir);
		if (dirname==NULL) break;
		strncpy (tmpstring, dirname->d_name, INPUT_STRING-1);
		tmpstring[INPUT_STRING-1]=0;
		if (strcmp(tmpstring, "scsi")==0) continue;
		if (tmpstring[0]=='.') continue;
		SCSIDRV[c]=malloc(SCSIDRV_STRUCT_SIZE);
		if (SCSIDRV[c]==NULL) error_handler(ERROR_MALLOC);
		strncpy (SCSIDRV[c]->NAME, tmpstring, STORAGE_STRING-1);
		SCSIDRV[c]->NAME[STORAGE_STRING-1]=0;
		c++;
		if (c==SCSIDRV_COUNT_MAX) break;
	} while (0==0);
	NUM_SCSIDRV=c-1;
	// Read Descriptions
	for (c=0; c<=NUM_SCSIDRV; c++) {
		sprintf (filename, "%sscsi/%s/0", PROCLOC, SCSIDRV[c]->NAME);
		filep=fopen(filename, "r");
		if (filep!=NULL) {
			fgets (SCSIDRV[c]->DESC, STORAGE_STRING, filep);
			killcrlf (SCSIDRV[c]->DESC, strlen(SCSIDRV[c]->DESC));
			fclose (filep);
		}
// BUG:
// cannot assume /proc/scsi/scsi_driver/0 exists and is information file
// only true with ide-scsi driver
// this is temp fix - to clear string, so no bogus characters:
		else
			SCSIDRV[c]->DESC[0]=0;
// end bug
	}
	// Return
	return;
}

// handletty() Function
void handletty () {
	char filename [INPUT_STRING];
	FILE *filep;
	char tmpstring [INPUT_STRING];
	int c=0;

	// Open and Process /proc/tty/drivers
	sprintf (filename, "%stty/drivers", PROCLOC);
	filep=fopen(filename, "r");
	if (filep!=NULL) {
		do {
			fgets (tmpstring, INPUT_STRING, filep);
			if (feof(filep)!=0) break;
			TTYDRIVERS[c]=malloc(TTYDRIVERS_STRUCT_SIZE);
			if (TTYDRIVERS[c]==NULL) error_handler(ERROR_MALLOC);
			sscanf (tmpstring, "%20s %13s %3s %7s %s", TTYDRIVERS[c]->DRIVER, TTYDRIVERS[c]->NAME, TTYDRIVERS[c]->MAJOR, TTYDRIVERS[c]->RANGE, TTYDRIVERS[c]->TYPE);
			c++;
			if (c==TTYDRIVERS_COUNT_MAX) break;
		} while (0==0);
		NUM_TTYDRIVERS=c-1;
		fclose (filep);
	}
	// Open and Process /proc/tty/ldiscs
	c=0;
	sprintf (filename, "%stty/ldiscs", PROCLOC);
	filep=fopen(filename, "r");
	if (filep==NULL) return;
	do {
		fgets (tmpstring, INPUT_STRING, filep);
		if (feof(filep)!=0) break;
		TTYLDISCS[c]=malloc(TTYLDISCS_STRUCT_SIZE);
		if (TTYLDISCS[c]==NULL) error_handler(ERROR_MALLOC);
		sscanf (tmpstring, "%10s %2d", TTYLDISCS[c]->NAME, &TTYLDISCS[c]->COUNT);
		c++;
		if (c==TTYLDISCS_COUNT_MAX) break;
	} while (0==0);
	NUM_TTYLDISCS=c-1;
	fclose (filep);
	// Return
	return;
}

// handlesound() Function
void handlesound () {
	char filename [INPUT_STRING];
	FILE *filep;
	char tmpstring [INPUT_STRING];
	int c=0;
	int col=0;
	char type [INPUT_STRING];
	char desc [INPUT_STRING];

	// Open /proc/sound
	sprintf (filename, "%ssound", PROCLOC);
	filep=fopen(filename, "r");
	if (filep==NULL) return;
	// Find PROC_SOUNDDRV
	fgets (tmpstring, INPUT_STRING, filep);
	killcrlf (tmpstring, strlen(tmpstring));
	strncpy (PROCINFO[PROC_SOUNDDRV], tmpstring, STORAGE_STRING-1);
	PROCINFO[PROC_SOUNDDRV][STORAGE_STRING-1]=0;
	// Enter Loop
	do {
		fgets (tmpstring, INPUT_STRING, filep);
		if (feof(filep)!=0) break;
		if (strncmp(tmpstring, "Installed drivers:", 18)==0) {
			c=0;
			do {
				fgets (tmpstring, INPUT_STRING, filep);
				killcrlf (tmpstring, strlen(tmpstring));
				col=cinstr(tmpstring, ':');
				if (col==-1) break;
				fillstring (type, 0, INPUT_STRING);
				strncpy (type, tmpstring, col);
				type[col]=0;
				rstring (tmpstring, col+1, desc);
				SNDDRV[c]=malloc(STRING_ARRAY_SIZE);
				if (SNDDRV[c]==NULL) error_handler(ERROR_MALLOC);
				sprintf (*SNDDRV[c], "%-10s %s", type, desc);
				c++;
				if (c==SNDDRV_COUNT_MAX) break;
			} while (0==0);
			NUM_SNDDRV=c-1;
			continue;
		}
		if (strncmp(tmpstring, "Card config:", 12)==0) {
			c=0;
			do {
				fgets (tmpstring, INPUT_STRING, filep);
				if (strlen(tmpstring)==1) break;
				killcrlf (tmpstring, strlen(tmpstring));
				SNDCONF[c]=malloc(STRING_ARRAY_SIZE);
				if (SNDCONF[c]==NULL) error_handler(ERROR_MALLOC);
				strncpy (*SNDCONF[c], tmpstring, STORAGE_STRING-1);
				(*SNDCONF[c])[STORAGE_STRING-1]=0;
				c++;
				if (c==SNDCONF_COUNT_MAX) break;
			} while (0==0);
			NUM_SNDCONF=c-1;
			continue;
		}
		if (strncmp(tmpstring, "Audio devices:", 14)==0) {
			c=0;
			do {
				fgets (tmpstring, INPUT_STRING, filep);
				if (strlen(tmpstring)==1) break;
				killcrlf (tmpstring, strlen(tmpstring));
				SNDAUDIO[c]=malloc(STRING_ARRAY_SIZE);
				if (SNDAUDIO[c]==NULL) error_handler(ERROR_MALLOC);
				strncpy (*SNDAUDIO[c], tmpstring, STORAGE_STRING-1);
				(*SNDAUDIO[c])[STORAGE_STRING-1]=0;
				c++;
				if (c==SNDAUDIO_COUNT_MAX) break;
			} while (0==0);
			NUM_SNDAUDIO=c-1;
			continue;
		}
		if (strncmp(tmpstring, "Synth devices:", 14)==0) {
			c=0;
			do {
				fgets (tmpstring, INPUT_STRING, filep);
				if (strlen(tmpstring)==1) break;
				killcrlf (tmpstring, strlen(tmpstring));
				SNDSYNTH[c]=malloc(STRING_ARRAY_SIZE);
				if (SNDSYNTH[c]==NULL) error_handler(ERROR_MALLOC);
				strncpy (*SNDSYNTH[c], tmpstring, STORAGE_STRING-1);
				(*SNDSYNTH[c])[STORAGE_STRING-1]=0;
				c++;
				if (c==SNDSYNTH_COUNT_MAX) break;
			} while (0==0);
			NUM_SNDSYNTH=c-1;
			continue;
		}
		if (strncmp(tmpstring, "Midi devices:", 13)==0) {
			c=0;
			do {
				fgets (tmpstring, INPUT_STRING, filep);
				if (strlen(tmpstring)==1) break;
				killcrlf (tmpstring, strlen(tmpstring));
				SNDMIDI[c]=malloc(STRING_ARRAY_SIZE);
				if (SNDMIDI[c]==NULL) error_handler(ERROR_MALLOC);
				strncpy (*SNDMIDI[c], tmpstring, STORAGE_STRING-1);
				(*SNDMIDI[c])[STORAGE_STRING-1]=0;
				c++;
				if (c==SNDMIDI_COUNT_MAX) break;
			} while (0==0);
			NUM_SNDMIDI=c-1;
			continue;
		}
		if (strncmp(tmpstring, "Timers:", 7)==0) {
			c=0;
			do {
				fgets (tmpstring, INPUT_STRING, filep);
				if (strlen(tmpstring)==1) break;
				killcrlf (tmpstring, strlen(tmpstring));
				SNDTIMERS[c]=malloc(STRING_ARRAY_SIZE);
				if (SNDTIMERS[c]==NULL) error_handler(ERROR_MALLOC);
				strncpy (*SNDTIMERS[c], tmpstring, STORAGE_STRING-1);
				(*SNDTIMERS[c])[STORAGE_STRING-1]=0;
				c++;
				if (c==SNDTIMERS_COUNT_MAX) break;
			} while (0==0);
			NUM_SNDTIMERS=c-1;
			continue;
		}
		if (strncmp(tmpstring, "Mixers:", 7)==0) {
			c=0;
			do {
				fgets (tmpstring, INPUT_STRING, filep);
				if (feof(filep)!=0) break;
				if (strlen(tmpstring)==1) break;
				killcrlf (tmpstring, strlen(tmpstring));
				SNDMIXERS[c]=malloc(STRING_ARRAY_SIZE);
				if (SNDMIXERS[c]==NULL) error_handler(ERROR_MALLOC);
				strncpy (*SNDMIXERS[c], tmpstring, STORAGE_STRING-1);
				(*SNDMIXERS[c])[STORAGE_STRING-1]=0;
				c++;
				if (c==SNDMIXERS_COUNT_MAX) break;
			} while (0==0);
			NUM_SNDMIXERS=c-1;
			continue;
		}
	} while (0==0);
	fclose (filep);
	// Return
	return;
}

// handlemtrr() Function
void handlemtrr () {
	char filename [INPUT_STRING];
	FILE *filep;
	char tmpstring [INPUT_STRING];
	int c=0;
	int comma=0;

	NUM_MTRR=-1;
	// Open /proc/mtrr
	sprintf (filename, "%smtrr", PROCLOC);
	filep=fopen(filename, "r");
	if (filep==NULL) return;
	do {
		fgetstr (filep, tmpstring, INPUT_STRING);
		if (strlen(tmpstring)==0) break;
		MTRR[c]=malloc(MTRR_STRUCT_SIZE);
		if (MTRR[c]==NULL) error_handler(ERROR_MALLOC);
		sscanf (tmpstring, "%*3s%2d%*1s %*5s%10s %*1s%4d%*4s %*5s%4d%*3s %s %*6s%d", &MTRR[c]->REG, MTRR[c]->OFFHEX, &MTRR[c]->OFFMB, &MTRR[c]->SIZE, tmpstring, &MTRR[c]->COUNT);
		comma=cinstr(tmpstring, ',');
		strncpy (MTRR[c]->OPTION, tmpstring, comma);
		MTRR[c]->OPTION[comma]=0;
		c++;
		if (c==MTRR_COUNT_MAX) break;
	} while (0==0);
	fclose (filep);
	NUM_MTRR=c-1;
	// Return;
	return;
}

// handleide() Function
void handleide () {
	FILE *filep;
	DIR *dirp;
	DIR *dirq;
	DIR *dirr;
	struct dirent *dir;
	char filename [INPUT_STRING];
	char tmpstring [INPUT_STRING];
	char idename [INPUT_STRING];
	char hdname [INPUT_STRING];
	int c=0;
	int space=0;

	// IDE Drivers
	sprintf (filename, "%side/drivers", PROCLOC);
	filep=fopen(filename, "r");
	if (filep!=NULL) {
		do {
			fgetstr (filep, tmpstring, INPUT_STRING);
			if (strlen(tmpstring)==0) break;
			space=cinstr(tmpstring, ' ');
// BUG:
// this malloc statement causes problems for mtrr count
			IDEDRV[c]=malloc(IDEDRV_STRUCT_SIZE);
// please fix this!!!
			if (IDEDRV[c]==NULL) error_handler(ERROR_MALLOC);
			strncpy (IDEDRV[c]->NAME, tmpstring, space);
			IDEDRV[c]->NAME[space]=0;
			rstring (tmpstring, space+8, IDEDRV[c]->VERSION);
			c++;
			if (c==IDEDRV_COUNT_MAX) break;
		} while (0==0);
		NUM_IDEDRV=c-1;
		fclose (filep);
	}
	// IDE Devices
	c=0;
	sprintf (filename, "%side/", PROCLOC);
	dirp=opendir(filename);
	if (dirp==NULL) return;
	do {
		dir=readdir(dirp);
		if (dir==NULL) break;
		strncpy (idename, dir->d_name, INPUT_STRING-1);
		idename[INPUT_STRING-1]=0;
		// Test for Directory
		if (strncmp(idename, ".", 1)==0) continue;
		sprintf (filename, "%side/%s/", PROCLOC, idename);
		dirq=opendir(filename);
		if (dirq==NULL) continue;
		do {
			dir=readdir(dirq);
			if (dir==NULL) break;
			strncpy (hdname, dir->d_name, INPUT_STRING-1);
			hdname[INPUT_STRING-1]=0;
			if (strncmp(hdname, ".", 1)==0) continue;
			sprintf (filename, "%side/%s/%s", PROCLOC, idename, hdname);
			dirr=opendir(filename);
			if (dirr==NULL) continue;
			IDEDEV[c]=malloc(IDEDEV_STRUCT_SIZE);
			if (IDEDEV[c]==NULL) error_handler(ERROR_MALLOC);
			strncpy (IDEDEV[c]->NAME, hdname, STORAGE_STRING-1);
			IDEDEV[c]->NAME[STORAGE_STRING-1]=0;
			strncpy (IDEDEV[c]->PARENT, idename, STORAGE_STRING-1);
			IDEDEV[c]->PARENT[STORAGE_STRING-1]=0;
			c++;
			if (c==IDEDEV_COUNT_MAX) break;
			closedir (dirr);
		} while (0==0);
		closedir (dirq);
	} while (0==0);
	closedir (dirp);
	NUM_IDEDEV=c-1;
	// Information on Each Device
	for (c=0; c<=NUM_IDEDEV; c++) {
		// driver
		sprintf (filename, "%side/%s/%s/driver", PROCLOC, IDEDEV[c]->PARENT, IDEDEV[c]->NAME);
		filep=fopen(filename, "r");
		if (filep!=NULL) {
			fgets (IDEDEV[c]->DRIVER, STORAGE_STRING, filep);
			killcrlf (IDEDEV[c]->DRIVER, strlen(IDEDEV[c]->DRIVER));
			fclose (filep);
		}
		// media
		sprintf (filename, "%side/%s/%s/media", PROCLOC, IDEDEV[c]->PARENT, IDEDEV[c]->NAME);
		filep=fopen(filename, "r");
		if (filep!=NULL) {
			fgets (IDEDEV[c]->MEDIA, STORAGE_STRING, filep);
			killcrlf (IDEDEV[c]->MEDIA, strlen(IDEDEV[c]->MEDIA));
			fclose (filep);
		}
		// model
		sprintf (filename, "%side/%s/%s/model", PROCLOC, IDEDEV[c]->PARENT, IDEDEV[c]->NAME);
		filep=fopen(filename, "r");
		if (filep!=NULL) {
			fgets (IDEDEV[c]->MODEL, STORAGE_STRING, filep);
			killcrlf (IDEDEV[c]->MODEL, strlen(IDEDEV[c]->MODEL));
			fclose (filep);
		}
		// settings
		sprintf (filename, "%side/%s/%s/settings", PROCLOC, IDEDEV[c]->PARENT, IDEDEV[c]->NAME);
		filep=fopen(filename, "r");
		if (filep!=NULL) {
			do {
				fgets (tmpstring, INPUT_STRING, filep);
				if (feof(filep)!=0) break;
				if (strncmp(tmpstring, "name", 4)==0) continue;
				if (strncmp(tmpstring, "----", 4)==0) continue;
				if (strncmp(tmpstring, "bios_cyl", 7)==0)
					sscanf (tmpstring, "%*24s %d", &IDEDEV[c]->CYL);
				if (strncmp(tmpstring, "bios_head", 8)==0)
					sscanf (tmpstring, "%*24s %d", &IDEDEV[c]->HEAD);
				if (strncmp(tmpstring, "bios_sect", 8)==0)
					sscanf (tmpstring, "%*24s %d", &IDEDEV[c]->SECT);
			} while (0==0);
		}	
	}
	// Return
	return;
}

// handleparport() Function
void handleparport () {
	DIR *dirp;
	struct dirent *dirlist;
	FILE *filep;
	char filename [INPUT_STRING];
	char tmpstring [INPUT_STRING];
	int c=0;
	int len;

	// Open and Read /proc/parport
	sprintf (filename, "%sparport", PROCLOC);
	dirp=opendir(filename);
	if (dirp==NULL) return;
	do {
		dirlist=readdir(dirp);
		if (dirlist==NULL) break;
		strncpy (tmpstring, dirlist->d_name, INPUT_STRING-1);
		tmpstring[INPUT_STRING-1]=0;
		if (strncmp(tmpstring, ".", 1)==0) continue;
		PARPORT[c]=malloc(PARPORT_STRUCT_SIZE);
		if (PARPORT[c]==NULL) error_handler(ERROR_MALLOC);
		strncpy (PARPORT[c]->NAME, tmpstring, STORAGE_STRING-1);
		PARPORT[c]->NAME[STORAGE_STRING-1]=0;
		// autoprobe
		sprintf (filename, "%sparport/%s/autoprobe", PROCLOC, PARPORT[c]->NAME);
		filep=fopen(filename, "r");
		if (filep!=NULL) {
			fgets(tmpstring, INPUT_STRING, filep);
			len=killcrlf(tmpstring, strlen(tmpstring));
			strncpy (PARPORT[c]->AUTOPROBE, tmpstring, len);
			PARPORT[c]->AUTOPROBE[len]=0;
			fclose (filep);
		}
		// devices
		sprintf (filename, "%sparport/%s/devices", PROCLOC, PARPORT[c]->NAME);
		filep=fopen(filename, "r");
		if (filep!=NULL) {
			fgets (tmpstring, INPUT_STRING, filep);
			killcrlf (tmpstring, strlen(tmpstring));
			strncpy (PARPORT[c]->DEVICES, tmpstring+1, STORAGE_STRING-1);
			PARPORT[c]->DEVICES[STORAGE_STRING-1]=0;
			fclose (filep);
		}
		// hardware
		sprintf (filename, "%sparport/%s/hardware", PROCLOC, PARPORT[c]->NAME);
		filep=fopen(filename, "r");
		if (filep!=NULL) {
			do {
				fgets (tmpstring, INPUT_STRING, filep);
				if (feof(filep)!=0) break;
				killcrlf (tmpstring, strlen(tmpstring));
				if (strncmp(tmpstring, "base:", 5)==0)
					sscanf (tmpstring, "base:\t%s", PARPORT[c]->IOPORT);
				if (strncmp(tmpstring, "irq:", 4)==0)
					sscanf (tmpstring, "irq:\t%s", PARPORT[c]->IRQ);
				if (strncmp(tmpstring, "dma:", 4)==0)
					sscanf (tmpstring, "dma:\t%s", PARPORT[c]->DMA);
				if (strncmp(tmpstring, "modes:", 6)==0)
					sscanf (tmpstring, "modes:\t%s", PARPORT[c]->PORTTYPE);
			} while (0==0);
			fclose (filep);
		}
		c++;
		if (c==PARPORT_COUNT_MAX) break;
	} while (0==0);
	NUM_PARPORT=c-1;
	closedir (dirp);
	// Return
	return;
}

// handlemisc() Function
void handlemisc () {
	FILE *filep;
	char filename [INPUT_STRING];
	char tmpstring [INPUT_STRING];
	int c=0;

	// Open and Handle /proc/misc
	sprintf (filename, "%smisc", PROCLOC);
	filep=fopen(filename, "r");
	if (filep==NULL) return;
	do {
		fgets (tmpstring, INPUT_STRING, filep);
		if (feof(filep)!=0) break;
		killcrlf (tmpstring, strlen(tmpstring));
		MISC[c]=malloc(DEV_STRUCT_SIZE);
		if (MISC[c]==NULL) error_handler(ERROR_MALLOC);
		sscanf (tmpstring, "%d %s", &MISC[c]->NUMBER, MISC[c]->NAME);
		c++;
		if (c==MISC_COUNT_MAX) break;
	} while (0==0);
	fclose (filep);
	NUM_MISC=c-1;
	// Return
	return;
}

END OF MOVED CODE
*/

