/*
    si - Version 1.0 - By Joe Turgeon (jturgeon@isd.net)
    main.c

    si - System Information Viewer
    Copyright (C) 1999  Joe Turgeon (jturgeon@isd.net)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include "main.h"


// Global Variables
// File Pointers
FILE *fstdout;
FILE *pipes;
// Kernel Variables
int KERNEL;
// Command-Line Argument Variables
int OPT_SHOW_SYSTEM	= 1;
int OPT_SHOW_HARDWARE   = 1;
int OPT_SHOW_MEMORY	= 1;
int OPT_SHOW_KERNEL	= 1;
int OPT_SHOW_FILESYS	= 1;
int OPT_SHOW_NETWORK	= 1;
int OPT_SHOW_PROCESS	= 1;
int OPT_OUTPUT_PAGER	= 0;
int OPT_OUTPUT_FILE	= 0;
int OPT_DISABLE_PAGER	= 0;
int OPT_DISPLAY_MODE	= DISPLAY_MODE_DUMP;
int OPT_DISABLE_CMDLINE	= 0;
char OUTPUT_FILE [INPUT_STRING];


// main() Function
int main (int argc, char **argv, char **argp) {
	int retval;

	// Set Default Output
	fstdout = stderr;
	// Setup Signal Handlers
	if (SIG_ERR==signal(SIGINT, signal_handler)) error_handler(ERROR_SIGNAL);
	if (SIG_ERR==signal(SIGSEGV, signal_handler)) error_handler(ERROR_SIGNAL);
	if (SIG_ERR==signal(SIGPIPE, signal_handler)) error_handler(ERROR_SIGNAL);
	if (SIG_ERR==signal(SIGTERM, signal_handler)) error_handler(ERROR_SIGNAL);
	// initiate mcheck memory consistency checking
	mcheck (signal_handler);
	// Banner
	fprintf (fstdout, "si - Version 1.0 - By Joe Turgeon (jturgeon@isd.net)\n");
	// Process Configuration File
	configuration_file ();
	// Handle Command-Line Arguments
	if (OPT_DISABLE_CMDLINE==0) {
		if (command_line(argc, argv, argp)!=0) {
			fprintf (fstdout, "\n");
			fprintf (fstdout, "Usage:  %s  [-h | -? | --help]  [-o smkfnp]  [-f <file>]  [-p | --no-pager]\n", argv[0]);
			fprintf (fstdout, "            [--dump]  [--color]  [--ps]  [--html]\n");
			fprintf (fstdout, "\n");
			fprintf (fstdout, "Command Line Arguments:\n");
			fprintf (fstdout, "  -h, --help          Display This Help Screen\n");
			fprintf (fstdout, "  -?                        (same as -h)      \n");
			fprintf (fstdout, "  -f <file>           Send Output To File\n");
			fprintf (fstdout, "  -o                  Display Specified Sections  (default: all)\n");
			fprintf (fstdout, "                      s   System Section\n");
			fprintf (fstdout, "                      h   Hardware Section\n");
			fprintf (fstdout, "                      m   Memory Section\n");
			fprintf (fstdout, "                      k   Kernel Section\n");
			fprintf (fstdout, "                      f   File-System Section\n");
			fprintf (fstdout, "                      n   Network Section\n");
			fprintf (fstdout, "                      p   Process Section\n");
			fprintf (fstdout, "  -p, --no-pager      Disable Sending Output To Pager\n");
			fprintf (fstdout, "      --dump          Dump Output Mode (default)\n");
			fprintf (fstdout, "      --data          Data Dump Output Mode\n");
			fprintf (fstdout, "      --color         Color Output Mode\n");
			fprintf (fstdout, "      --ps            Process View Mode\n");
			fprintf (fstdout, "      --html          HTML Output Mode\n");
			fprintf (fstdout, "\n");
			exit_program (0);
		}
	}
	// Run Through Pager or Send Output to File
	if ((OPT_OUTPUT_FILE==1) && ((OPT_DISPLAY_MODE==DISPLAY_MODE_DUMP) || (OPT_DISPLAY_MODE==DISPLAY_MODE_DATA))) {
		fprintf (fstdout, "si:  Sending Output to:  %s\n", OUTPUT_FILE);
		fstdout=fopen(OUTPUT_FILE, "w");
		if (fstdout==NULL) {
			fprintf (stderr, "si:  File Specified Does Not Exist (%s)\n", OUTPUT_FILE);
			exit (1);
		}
		fprintf (fstdout, "si - Version 1.0 - By Joe Turgeon (jturgeon@isd.net)\n");
	}
	if ((isatty(1)) && (OPT_OUTPUT_FILE==0) && (OPT_DISABLE_PAGER==0) && ((OPT_DISPLAY_MODE==DISPLAY_MODE_DUMP) || (OPT_DISPLAY_MODE==DISPLAY_MODE_DATA))) {
		pipes=popen("less", "w");
		if (pipes!=NULL) {
			fprintf (fstdout, "si:  Using Pager...\n");
			fstdout=pipes;
			OPT_OUTPUT_PAGER=1;
			fprintf (fstdout, "si - Version 1.0 - By Joe Turgeon (jturgeon@isd.net)\n");
		}
		else {
			fprintf (fstdout, "si:  Could Not Start Pager...\n");
			fstdout=stdout;
		}
	}
	else {
		if ((OPT_OUTPUT_FILE==0) && ((OPT_DISPLAY_MODE==DISPLAY_MODE_DUMP) || (OPT_DISPLAY_MODE==DISPLAY_MODE_DATA))) {
			fstdout=stdout;
			fprintf (fstdout, "si - Version 1.0 - By Joe Turgeon (jturgeon@isd.net)\n");
		}
	}
	// Find Linux Kernel Version
	kernelversion ();
	// Load Information
	retval=loadinfo();
	if (retval!=0) exit_program (1);
	// Dump Information
	if (OPT_DISPLAY_MODE==DISPLAY_MODE_DUMP)   display_dump();
	if (OPT_DISPLAY_MODE==DISPLAY_MODE_DATA)   display_data();
#ifdef HAVE_LIBNCURSES
	if (OPT_DISPLAY_MODE==DISPLAY_MODE_COLOR)  display_color();
	if (OPT_DISPLAY_MODE==DISPLAY_MODE_PS)     display_ps();
#else
	if ((OPT_DISPLAY_MODE==DISPLAY_MODE_COLOR) || (OPT_DISPLAY_MODE==DISPLAY_MODE_PS)) {
		printf ("\nncurses Not Enabled During Installation.\n");
		printf ("color and ps display mode not enabled due to the absence of\n");
		printf ("ncurses during installation.\n\n");
	}
#endif
	if (OPT_DISPLAY_MODE==DISPLAY_MODE_HTML)   display_html();
	// Exit Program / Return
	exit_program (0);
	return 0;
}

// signal_handler() Function
void signal_handler (int signum) {
	switch (signum) {
		case SIGINT:
			fprintf (stderr, "si:  Interrupt Signal\n");
			break;
		case SIGSEGV:
			fprintf (stderr, "si:  Segmentation Fault\n");
			exit (signum);
			break;
		case SIGPIPE:
			// Ignore "Broken Pipe" Signal.
			// Let Execution Terminate.
			return;
			break;
		case SIGTERM:
			fprintf (stderr, "si:  Termination Signal\n");
			break;
		default:
			fprintf (stderr, "si:  Signal Caught:  Unknown Signal\n");
			break;
	}
	exit_program (signum);
	return;
}

// exit_program() Function
void exit_program (int retval) {
	// Clean-Up File Handlers
	if (OPT_OUTPUT_FILE==1) fclose(fstdout);
	if (OPT_OUTPUT_PAGER==1) pclose (pipes);
	fstdout=stderr;
	// Exit
	fprintf (fstdout, "si:  Clean Exit\n");
	exit (retval);
	// Return
	return;
}

// kernelversion() Function
void kernelversion () {
	int major, minor, rev;

	// use proc filesys code to read /proc/sys/kernel/os_release
	handleproc (PROC_OSRELEASE, "sys/kernel/osrelease");
	// determine version
	sscanf (PROCINFO[PROC_OSRELEASE], "%d.%d.%d", &major, &minor, &rev);
	if (major<2) KERNEL=VER20;
	if ((major==2) && (minor==0)) KERNEL=VER20;
	if ((major==2) && (minor==1)) KERNEL=VER22;
	if ((major==2) && (minor==2)) KERNEL=VER22;
	if ((major==2) && (minor==3)) KERNEL=VER24;
	if ((major==2) && (minor==4)) KERNEL=VER24;
	if ((major==2) && (minor>4)) KERNEL=VER24;
	if (major>2) KERNEL=VER24;
	// display detected version
	switch (KERNEL) {
		case VER24:
			fprintf (fstdout, "si:  Using Linux Kernel 2.4 Code\n");
			break;
		case VER22:
			fprintf (fstdout, "si:  Using Linux Kernel 2.2 Code\n");
			break;
		case VER20:
			fprintf (fstdout, "si:  Using Linux Kernel 2.0 Code\n");
			break;
		default:
			error_handler (ERROR_KERNEL);
			break;
	}
	// return
	return;
}

int command_line (int argc, char **argv, char **argp) {
	int i;
	struct option gopt [7];

	fprintf (fstdout, "si:  Checking Command-Line\n");
	// Build Command Line Options
	gopt[0].name="help";
	gopt[0].has_arg=no_argument;
	gopt[0].flag=NULL;
	gopt[0].val=512;

	gopt[1].name="dump";
	gopt[1].has_arg=no_argument;
	gopt[1].flag=NULL;
	gopt[1].val=250;

	gopt[2].name="color";
	gopt[2].has_arg=no_argument;
	gopt[2].flag=NULL;
	gopt[2].val=251;

	gopt[3].name="ps";
	gopt[3].has_arg=no_argument;
	gopt[3].flag=NULL;
	gopt[3].val=252;

	gopt[4].name="html";
	gopt[4].has_arg=no_argument;
	gopt[4].flag=NULL;
	gopt[4].val=253;
	
	gopt[5].name="data";
	gopt[5].has_arg=no_argument;
	gopt[5].flag=NULL;
	gopt[5].val=254;
	// Test Command Line
	while (0==0) {
		i=getopt_long(argc, argv, "f:o:h?p", gopt, NULL);
		if (i==EOF) break;
		switch (i) {
			case 'h':
			case '?':
				return 1;
				break;
			case 'f':
				OPT_OUTPUT_FILE=1;
				strncpy (OUTPUT_FILE, optarg, INPUT_STRING);
				if (strlen(optarg)>INPUT_STRING) OUTPUT_FILE[INPUT_STRING-1]=0;
				break;
			case 'o':
				if (cinstr(optarg, 's')==-1) OPT_SHOW_SYSTEM=0;
				if (cinstr(optarg, 'h')==-1) OPT_SHOW_HARDWARE=0;
				if (cinstr(optarg, 'm')==-1) OPT_SHOW_MEMORY=0;
				if (cinstr(optarg, 'k')==-1) OPT_SHOW_KERNEL=0;
				if (cinstr(optarg, 'f')==-1) OPT_SHOW_FILESYS=0;
				if (cinstr(optarg, 'n')==-1) OPT_SHOW_NETWORK=0;
				if (cinstr(optarg, 'p')==-1) OPT_SHOW_PROCESS=0;
				break;
			case 'p':
				OPT_DISABLE_PAGER=1;
				break;
			case 250:
				OPT_DISPLAY_MODE=DISPLAY_MODE_DUMP;
				break;
			case 251:
				OPT_DISPLAY_MODE=DISPLAY_MODE_COLOR;
				break;
			case 252:
				OPT_DISPLAY_MODE=DISPLAY_MODE_PS;
				break;
			case 253:
				OPT_DISPLAY_MODE=DISPLAY_MODE_HTML;
				break;
			case 254:
				OPT_DISPLAY_MODE=DISPLAY_MODE_DATA;
				break;
			default:
				return 1;
				break;
		}
	}
	return 0;
}

void configuration_file () {
	FILE *filep;
	char filename [FILENAME_MAX];
	char instring [INPUT_STRING];
	char tmpstring [INPUT_STRING];

	// Open "~/.si.conf" or "/etc/si.conf"
	sprintf (filename, "%s/.si.conf", getenv("HOME"));
	filep=fopen(filename, "r");
	if (filep==NULL) {
		filep=fopen("/etc/si.conf", "r");
		if (filep==NULL) return;
	}
	// Reset Default Values
	OPT_SHOW_SYSTEM=0;
	OPT_SHOW_HARDWARE=0;
	OPT_SHOW_MEMORY=0;
	OPT_SHOW_KERNEL=0;
	OPT_SHOW_FILESYS=0;
	OPT_SHOW_NETWORK=0;
	OPT_SHOW_PROCESS=0;
	// Loop for Input
	do {
		fgets (instring, INPUT_STRING, filep);
		if (feof(filep)!=0) break;
		if (strncmp(instring, ";", 1)==0) continue;
		sscanf (instring, "%s", tmpstring);
		killcrlf (tmpstring, strlen(tmpstring));
		if (strcmp(tmpstring, "show_system")==0)   OPT_SHOW_SYSTEM=1;
		if (strcmp(tmpstring, "show_hardware")==0) OPT_SHOW_HARDWARE=1;
		if (strcmp(tmpstring, "show_memory")==0)   OPT_SHOW_MEMORY=1;
		if (strcmp(tmpstring, "show_kernel")==0)   OPT_SHOW_KERNEL=1;
		if (strcmp(tmpstring, "show_filesys")==0)  OPT_SHOW_FILESYS=1;
		if (strcmp(tmpstring, "show_network")==0)  OPT_SHOW_NETWORK=1;
		if (strcmp(tmpstring, "show_process")==0)  OPT_SHOW_PROCESS=1;
		if (strcmp(tmpstring, "output_file")==0) {
			OPT_OUTPUT_FILE=1;
			sscanf (instring, "%*s %s", OUTPUT_FILE);
		}
		if (strcmp(tmpstring, "disable_pager")==0) OPT_DISABLE_PAGER=1;
		if (strcmp(tmpstring, "display_mode")==0) {
			sscanf (instring, "%*s %s", tmpstring);
			if (strcmp(tmpstring, "dump")==0) OPT_DISPLAY_MODE=DISPLAY_MODE_DUMP;
			if (strcmp(tmpstring, "data")==0) OPT_DISPLAY_MODE=DISPLAY_MODE_DATA;
			if (strcmp(tmpstring, "color")==0) OPT_DISPLAY_MODE=DISPLAY_MODE_COLOR;
			if (strcmp(tmpstring, "ps")==0) OPT_DISPLAY_MODE=DISPLAY_MODE_PS;
			if (strcmp(tmpstring, "html")==0) OPT_DISPLAY_MODE=DISPLAY_MODE_HTML;
		}
		if (strcmp(tmpstring, "disable_cmdline")==0) OPT_DISABLE_CMDLINE=1;
	} while (0==0);
	fclose (filep);
	// Return
	return;
}


