/* This file is part of the KDE project
   Copyright (C) 2003-2005 Jaroslaw Staniek <js@iidea.pl>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#ifndef KEXIDB_CONNECTION_H
#define KEXIDB_CONNECTION_H

#include <qobject.h>
#include <qstringlist.h>
#include <qintdict.h>
#include <qdict.h>
#include <qptrdict.h>
#include <qvaluevector.h>
#include <qvaluelist.h>
#include <qvariant.h>
#include <qguardedptr.h>

#include <kexidb/object.h>
#include <kexidb/connectiondata.h>
#include <kexidb/tableschema.h>
#include <kexidb/queryschema.h>
#include <kexidb/transaction.h>
#include <kexidb/driver.h>

#include <kexiutils/tristate.h>

namespace KexiDB {

//! structure for storing single record with type information
typedef QValueVector<QVariant> RowData; 

class Cursor;
class ConnectionPrivate;
class RowEditBuffer;

//! Database connection API, allowing queries and updates.
/*! This class represents a database connection established with given database source.
*/
class KEXI_DB_EXPORT Connection : public QObject, public KexiDB::Object
{
	Q_OBJECT

	public:

		/*! Opened connection is automatically disconnected and removed 
		 from driver's connections list. 
		 Note for driver developers: you should call destroy() 
		 from you Connection's subclass destructor. */
		virtual ~Connection();

		/*! \return parameters that were used to create this connection. */
		ConnectionData* data() { return m_data; }

		/*! \return the driver used for this connection. */
		Driver* driver() const { return m_driver; }

		/*! 
		\brief Connects to driver with given parameters. 
		\return true if successfull. */
		bool connect();

		/*! \return true, if connection is properly established. */
		bool isConnected() { return m_is_connected; }

		/*! \return true, both if connection is properly established
		 and any database within this connection is properly used
		 with useDatabase(). */
		bool isDatabaseUsed();

		/*! Reimplemented from Object: also clears sql string.
		 @sa recentSQLString() */
		virtual void clearError();

		/*! \brief Disconnects from driver with given parameters.
		 
		 The database (if used) is closed, and any active transactions 
		 (if supported) are rolled back, so commit these before disconnecting,
		 if you'd like to save your changes. */
		bool disconnect();

		/*! \return list of database names for opened connection.
		 If \a also_system_db is true, the system database names are also returned. */
		QStringList databaseNames(bool also_system_db = false);

		/*! \return true if database \a dbName exists.
		 If \a ignoreErrors is true, error flag of connection 
		  won't be modified for any errors (it will quietly return),
		  else (ignoreErrors == false) we can check why the database does 
		  not exist using error(), errorNum() and/or errorMsg(). */
		bool databaseExists( const QString &dbName, bool ignoreErrors = true );

		/*! \brief Creates new database with name \a dbName, using this connection.
		
		 If database with \a dbName already exists, or other error occured,
		 false is returned. 
		 For file-based drivers, \a dbName should be equal to the database
		 filename (the same as specified for ConnectionData).
		 \sa useDatabase() */
		bool createDatabase( const QString &dbName );

		/*! 
		\brief Opens an existing database specified by \a dbName.
		
		 If \a kexiCompatible is true (the default) initial checks will be performed
		 to recognize database Kexi-specific format. Set \a kexiCompatible to false
		 if you're using native database (one that have no Kexi System tables).
		 For file-based drivers, \a dbName should be equal to filename
		 (the same as specified for ConnectionData). */
		bool useDatabase( const QString &dbName, bool kexiCompatible = true );

		/*! 
		\brief Closes currently used database for this connection.
		
		 Any active transactions (if supported) are rolled back,
		 so commit these before closing, if you'd like to save your changes. */
		bool closeDatabase();

		/*! \brief Get the name of the current database
		
		\return name of currently used database for this connection or empty string
			if there is no used database */
		QString currentDatabase() { return m_usedDatabase; }

		/*! \brief Drops database with name \a dbName.
		
		 if dbName is not specified, currently used database name is used
		 (it is closed before dropping).
		*/
		bool dropDatabase( const QString &dbName = QString::null );

		/*! \return names of all table schemas stored in currently 
		 used database. If \a also_system_tables is true, 
		 internal KexiDB system table names (kexi__*) are also returned.
		 \sa kexiDBSystemTableNames() */
		QStringList tableNames(bool also_system_tables = false);
		
		/*! \return list of internal KexiDB system table names 
		 (kexi__*). This does not mean that these tables can be found
		 in currently opened database. Just static list of table 
		 names is returned. 
		 
		 The list contents may depend on KexiDB library version;
		 opened database can contain fewer 'system' tables than in current
		 KexiDB implementation, if the current one is newer than the one used 
		 to build the database. */
		static const QStringList& kexiDBSystemTableNames();

		//! Version information for this connection. Only valid when database is used.
		//! It's usually compared to drivers' and KexiDB library version.
		int versionMajor() const;
		int versionMinor() const;

		/*! \return ids of all table schema names stored in currently 
		 used database. These ids can be later used as argument for tableSchema().
		 This is a shortcut for objectIds(TableObjectType).
		 If \a also_system_tables is true, 
		 Internal KexiDB system tables (kexi__*) are not available here 
		 because these have no identifiers assigned (more formally: id=-1). */
		QValueList<int> tableIds();

		/*! \return ids of all database query schemas stored in currently 
		 used database. These ids can be later used as argument for querySchema().
		 This is a shortcut for objectIds(TableObjectType). */
		QValueList<int> queryIds();

		/*! \return names of all schemas of object with \a objType type 
		 that are stored in currently used database. */
		QValueList<int> objectIds(int objType);

		/*! \brief Creates new transaction handle and starts a new transaction.
		 \return KexiDB::Transaction object if transaction has been started 
		 successfully, otherwise null transaction. 
		 For drivers that allow single transaction per connection
		 (Driver::features() && SingleTransactions) this method can be called one time,
		 and then this single transaction will be default ( setDefaultTransaction() will 
		 be called).
		 For drivers that allow multiple transactions per connection, no default transaction is
		 set automatically in beginTransaction() method, you could do this by hand.
		 \sa setDefaultTransaction(), defaultTransaction().
		*/
		Transaction beginTransaction();

/*! \todo for nested transactions:
		Tansaction* beginTransaction(transaction *parent_transaction);
*/
		/*! Commits transaction \a trans.
		 If there is not \a trans argument passed, and there is default transaction 
		 (obtained from defaultTransaction()) defined, this one will be commited.
		 If default is not present, false is returned (when ignore_inactive is 
		 false, the default), or true is returned (when ignore_inactive is true).
		 
		 On successfull commit, \a trans object will be destroyed.
		 If this was default transaction, there is no default transaction for now.
		*/
		bool commitTransaction( Transaction trans = Transaction::null,
			bool ignore_inactive = false );
		
		/*! Rollbacks transaction \a trans.
		 If there is not \a trans argument passed, and there is default transaction 
		 (obtained from defaultTransaction()) defined, this one will be rolled back.
		 If default is not present, false is returned (when ignore_inactive is 
		 false, the default), or true is returned (when ignore_inactive is true).
		 
		 or any error occured, false is returned.
			
		 On successfull rollback, \a trans object will be destroyed.
		 If this was default transaction, there is no default transaction for now.
		*/
		bool rollbackTransaction( Transaction trans = Transaction::null,
			bool ignore_inactive = false );
		
		/*! \return handle for default transaction for this connection
		 or null transaction if there is no such a transaction defined. 
		 If transactions are supported: Any operation on database (e.g. inserts)
		 that is started without specifing transaction context, will be performed
		 in the context of this transaction.
		 
		 Returned null transaction doesn't mean that there is no transactions 
		 started at all.
		 Default transaction can be defined automatically for some drivers --
		 see beginTransaction(). 
		 \sa KexiDB::Driver::transactionsSupported()
		*/
		Transaction& defaultTransaction() const;

		/*! Sets default transaction that will be used as context for operations
		 on data in opened database for this connection. */
		void setDefaultTransaction(const Transaction& trans);
		
		/*! \return set of handles of currently active transactions.
		 Note that in multithreading environment some of these 
		 transactions can be already inactive after calling this method.
		 Use Transaction::active() to check that. Inactive transaction 
		 handle is useless and can be safely dropped.
		*/
		const QValueList<Transaction>& transactions();

		/*! \return true if "auto commit" option is on. 

		 When auto commit is on (the default on for any new Connection object),
		 every sql functional statement (statement that changes 
		 data in the database implicitly starts a new transaction. 
		 This transaction is automatically commited 
		 after successfull statement execution or rolled back on error.
		 
		 For drivers that do not support transactions (see Driver::features())
		 this method shouldn't be called because it does nothing ans always returns false.
		 
		 No internal KexiDB object should changes this option, although auto commit's
		 behaviour depends on database engine's specifics. Engines that support only single
		 transaction per connection (see Driver::SingleTransactions),
		 use this single connection for autocommiting, so if there is already transaction 
		 started by the KexiDB user program (with beginTransaction()), this transaction 
		 is commited before any sql functional statement execution. In this situation
		 default transaction is also affected (see defaultTransaction()).
		 
		 Only for drivers that support nested transactions (Driver::NestedTransactions),
		 autocommiting works independently from previously started transaction,
		 
		 For other drivers set this option off if you need use transaction 
		 for grouping more statements together.
		  
		 NOTE: nested transactions are not yet implemented in KexiDB API.
		*/
		bool autoCommit() const;

		/*! Changes auto commit option. This does not affect currently started transactions.
		 This option can be changed even when connection is not established.
		 \sa autoCommit() */
		bool setAutoCommit(bool on);

		/*! driver-specific string escaping */
//js: MOVED TO Driver		virtual QString escapeString(const QString& str) const = 0;
//		virtual QCString escapeString(const QCString& str) const = 0;
		
		/*! Prepares SELECT query described by raw \a statement. 
		 \return opened cursor created for results of this query 
		 or NULL if there was any error. Cursor can have optionally applied \a cursor_options
		 (one of more selected from KexiDB::Cursor::Options).
		 Preparation means that returned cursor is created but not opened.
		 Open this when you would like to do it with Cursor::open().

		 Note for driver developers: you should initialize cursor engine-specific 
		 resources and return Cursor subclass' object 
		 (passing \a statement and \a cursor_options to it's constructor).
		*/
		virtual Cursor* prepareQuery( const QString& statement, uint cursor_options = 0) = 0;

		/*! \overload prepareQuery( const QString& statement = QString::null, uint cursor_options = 0)
		 Prepares query described by \a query schema. 

		 Note for driver developers: you should initialize cursor engine-specific 
		 resources and return Cursor subclass' object 
		 (passing \a query and \a cursor_options to it's constructor).
		 Kexi SQL and driver-specific escaping is performed on table names.
		*/
		virtual Cursor* prepareQuery( QuerySchema& query, uint cursor_options = 0 ) = 0;

		/*! \overload prepareQuery( const QString& statement = QString::null, uint cursor_options = 0)
		 Statement is build from data provided by \a query schema.
		*/
//		Cursor* prepareQuery( QuerySchema& query, uint cursor_options = 0);

		/*! \overload prepareQuery( const QString& statement = QString::null, uint cursor_options = 0)
		 Statement is build from data provided by \a table schema, 
		 it is like "select * from table_name".*/
		Cursor* prepareQuery( TableSchema& table, uint cursor_options = 0);

		/*! Executes SELECT query described by \a statement.
		 \return opened cursor created for results of this query
		 or NULL if there was any error on the cursor creation or opening.
		 Cursor can have optionally applied \a cursor_options 
		 (one of more selected from KexiDB::Cursor::Options).
		 Identifiers in \a statement that are the same as keywords in Kexi
		 SQL or the backend's SQL need to have been escaped.
		 */
		Cursor* executeQuery( const QString& statement, uint cursor_options = 0 );

		/*! \overload executeQuery( const QString& statement, uint cursor_options = 0 )
		 Statement is build from data provided by \a query schema. 
		 Kexi SQL and driver-specific escaping is performed on table names. */
		Cursor* executeQuery( QuerySchema& query, uint cursor_options = 0 );

		/*! \overload executeQuery( const QString& statement, uint cursor_options = 0 )
		 Statement is build from data provided by \a table schema, 
		 it is like "select * from table_name".*/
		Cursor* executeQuery( TableSchema& table, uint cursor_options = 0 );

		/*! Deletes cursor \a cursor previously created by functions like executeQuery() 
		 for this connection.
		 There is an attempt to close the cursor with Cursor::close() if it was opened. 
		 Anyway, at last cursor is deleted. 
		 \return true if cursor is properly closed before deletion. */
		bool deleteCursor(Cursor *cursor);

		/*! \return schema of a table pointed by \a tableId, retrieved from currently 
		 used database. The schema is cached inside connection, 
		 so retrieval is performed only once, on demand. */
		TableSchema* tableSchema( int tableId );
		
		/*! \return schema of a table pointed by \a tableName, retrieved from currently 
		 used database. KexiDB system table schema can be also retrieved.
		 \sa tableSchema( int tableId ) */
		TableSchema* tableSchema( const QString& tableName );
		
		/*! \return schema of a query pointed by \a queryId, retrieved from currently 
		 used database. The schema is cached inside connection, 
		 so retrieval is performed only once, on demand. */
		QuerySchema* querySchema( int queryId );

		/*! \return schema of a query pointed by \a queryName, retrieved from currently 
		 used database.  \sa querySchema( int queryId ) */
		QuerySchema* querySchema( const QString& queryName );

//js: MOVED TO Driver		QString valueToSQL( const Field::Type ftype, const QVariant& v ) const;
//		QString valueToSQL( const Field *field, const QVariant& v ) const;

		/*! Executes \a sql query and stores first record's data inside \a data.
		 This is convenient method when we need only first record from query result,
		 or when we know that query result has only one record.
		 Adds a LIMIT clause to the query, \a sql should not include one already.
		 \return true if query was successfully executed and first record has been found. */
		bool querySingleRecord(const QString& sql, RowData &data);

		/*! Executes \a sql query and stores first record's field's (number \a column) string value 
		 inside \a value. For efficiency it's recommended that a query defined by \a sql
		 should have just one field (SELECT one_field FROM ....). 
		 Adds a LIMIT clause to the query, so \a sql should not include one already.
		 \return true if query was successfully executed and first record has been found.
		 \sa queryStringList() */
		bool querySingleString(const QString& sql, QString &value, uint column = 0);

		/*! Convenience function: executes \a sql query and stores first 
		 record's field's (number \a column) value inside \a number. \sa querySingleString(). 
		 Note: "LIMIT 1" is appended to \a sql statement */
		bool querySingleNumber(const QString& sql, int &number, uint column = 0);

		/*! Executes \a sql query and stores first record's first field's string value 
		 inside \a list. The list is initially cleared.
		 For efficiency it's recommended that a query defined by \a sql
		 should have just one field (SELECT one_field FROM ....). 
		 \return true if all values were fetched successfuly
		 On errors, the list is not cleared, it may contain few retrieved values. */
		bool queryStringList(const QString& sql, QStringList& list, uint column = 0);

		/*! \return true if there is at least one record returned in \a sql query.
		 Does not fetch any records. \a success will be set to false 
		 on query execution errors (true otherwise), so you can see a difference between 
		 "no results" and "query execution error" states. 
		 Note: real executed query is: "SELECT 1 FROM (\a sql) LIMIT 1" */
		bool resultExists(const QString& sql, bool &success);

		/*! \return true if there is at least one record in \a table. */
		bool isEmpty( TableSchema& table, bool &success );

//TODO perhaps use Q_ULLONG here?
		/*! \return number of records in \a sql query.
		 Does not fetch any records. -1 is returned on query execution errors (>0 otherwise).
		 Note: real executed query is: "SELECT COUNT() FROM (\a sql) LIMIT 1" 
		 (using querySingleNumber()) */
		int resultCount(const QString& sql);

		//PROTOTYPE:
		#define A , const QVariant&
		#define H_INS_REC(args) bool insertRecord(TableSchema &tableSchema args)
		#define H_INS_REC_ALL \
		H_INS_REC(A); \
		H_INS_REC(A A); \
		H_INS_REC(A A A); \
		H_INS_REC(A A A A); \
		H_INS_REC(A A A A A); \
		H_INS_REC(A A A A A A); \
		H_INS_REC(A A A A A A A); \
		H_INS_REC(A A A A A A A A)
		H_INS_REC_ALL;
		
		#undef H_INS_REC
		#define H_INS_REC(args) bool insertRecord(FieldList& fields args)

		H_INS_REC_ALL;
		#undef H_INS_REC_ALL
		#undef H_INS_REC
		#undef A
		
		bool insertRecord(TableSchema &tableSchema, QValueList<QVariant>& values);
		
		bool insertRecord(FieldList& fields, QValueList<QVariant>& values);

		/*! Creates table defined by \a tableSchema.
		 Schema information is also added into kexi system tables, for later reuse.
		 \return true on success - \a tableSchema object is then 
		 inserted to Connection structures - it is owned by Connection object now, 
		 so you shouldn't destroy the tableSchema object by hand 
		 (or declare it as local-scope variable). 

		 If \a replaceExisting is false (the default) and table with the same name 
		 (as tableSchema->name()) exists, false is returned. 
		 If \a replaceExisting is true, a table schema with the same name (if exists) 
		 is overwritten, then a new table schema gets the same identifier 
		 as existing table schema's identifier.

		 Note that on error:
		 - \a tableSchema is not inserted into Connection's structures,
		   so you are still owner of this object
		 - existing table schema object is not destroyed (i.e. it is still available 
		   e.g. using Connection::tableSchema(const QString& ), even if the table 
			 was physically dropped.
		*/
		bool createTable( TableSchema* tableSchema, bool replaceExisting = false );

		/*! Drops a table defined by \a tableSchema (both table object as well as physically).
		 If true is returned, schema information \a tableSchema is destoyed
		 (because it's owned), so don't keep this anymore!
		 No error is raised if the table does not exist physically 
		 - its schema is removed even in this case.
		*/
//TODO(js): update any structure (e.g. query) that depend on this table!
		tristate dropTable( TableSchema* tableSchema );

		/*! It is a convenience function, does exactly the same as 
		 bool dropTable( KexiDB::TableSchema* tableSchema ) */
		tristate dropTable( const QString& table );

		/*! Alters \a tableSchema using \a newTableSchema in memory and on the db backend. 
		 \return true on success, cancelled if altering was cancelled. */
//TODO(js): implement real altering
//TODO(js): update any structure (e.g. query) that depend on this table!
		tristate alterTable( TableSchema& tableSchema, TableSchema& newTableSchema);

		/*! Alters table's described \a tableSchema name to \a newName. 
		 If \a replace is true, destination table is replaced, if present.
		 If \a replace is false (the default) and destination table is present 
		 -- false is returned and ERR_OBJECT_EXISTS error is set.
		 \a tableSchema is updated on success.
		 \return true on success. */
		bool alterTableName(TableSchema& tableSchema, const QString& newName, bool replace = false);

		/*! Drops a query defined by \a querySchema. 
		 If true is returned, schema information \a querySchema is destoyed 
		 (because it's owned), so don't keep this anymore!
		*/
		bool dropQuery( QuerySchema* querySchema );

		/*! It is a convenience function, does exactly the same as 
		 bool dropQuery( KexiDB::QuerySchema* querySchema ) */
		bool dropQuery( const QString& query );

		/*! Removes information about object with \a objId 
		 from internal "kexi__object" and "kexi__objectdata" tables.
		 \return true on success. */
		bool removeObject( uint objId );

		/*! \return first field from \a fieldlist that has system name, 
		 null if there are no such field.
		 For checking Driver::isSystemFieldName() is used, so this check can 
		 be driver-dependent. */
		Field* findSystemFieldName(FieldList *fieldlist);

		/*! \return name of any (e.g. first found) database for this connection.
		 This method does not close or open this connection. The method can be used
		 (it is also internally used, e.g. for database dropping) when we need 
		 a database name before we can connect and execute any SQL statement 
		 (e.g. DROP DATABASE).
		 
		 The method can return nul lstring, but in this situation no automatic (implicit)
		 connections could be made, what is useful by e.g. dropDatabase().
		 
		 Note for driver developers: return here a name of database which you are sure 
		 is existing. 
		 Default implementation returns:
		 - value that previously had been set using setAvailableDatabaseName() for 
		   this connection, if it is not empty
		 - else (2nd priority): value of DriverBehaviour::ALWAYS_AVAILABLE_DATABASE_NAME 
		 if it is not empty.
		 
		 See decription of DriverBehaviour::ALWAYS_AVAILABLE_DATABASE_NAME member.
		 You may want to reimplement this method only when you need to depend on
		 this connection specifics 
		 (e.g. you need to check something remotely).
		*/
		virtual QString anyAvailableDatabaseName();
		
		/*! Sets \a dbName as name of a database that can be accessible. 
		 This is option that e.g. application that make use of KexiDB library can set
		 to tune connection's behaviour when it needs to temporary connect to any database
		 in the server to do some work.
		 You can pass empty dbName - then anyAvailableDatabaseName() will try return
		 DriverBehaviour::ALWAYS_AVAILABLE_DATABASE_NAME (the default) value 
		 instead of the one previously set with setAvailableDatabaseName().
		 
		 \sa anyAvailableDatabaseName()
		*/
		void setAvailableDatabaseName(const QString& dbName);

		/*! Because some engines need to have opened any database before
		 executing administrative sql statements like "create database" or "drop database",
		 this method is used to use appropriate, existing database for this connection.
		 For file-based db drivers this always return true and does not set tmpdbName
		 to any value. For other db drivers: this sets tmpdbName to db name computed 
		 using anyAvailableDatabaseName(), and if the name computed is empty, false 
		 is returned; if it is not empty, useDatabase() is called. 
		 False is returned also when useDatabase() fails.
		 You can call this method from your application's level if you really want to perform 
		 tasks that require any used database. In such a case don't forget 
		 to closeDatabase() if returned tmpdbName is not empty.
		 
		 Note: This method has nothing to do with creating or using temporary databases
		 in such meaning that these database are not persistent
		*/
		bool useTemporaryDatabaseIfNeeded(QString &tmpdbName);

		/*! \return autoincrement field's \a aiFieldName value 
		 of last inserted record. This refers \a tableName table.
		 
		 Simply, method internally fetches last inserted record and returns selected 
		 field's value. Requirements: field must be of integer type, there must be a
		 record inserted in current database session (whatever this means).
		 On error -1ULL is returned.
		 Last inserted record is identified by magical row identifier, usually called 
		 ROWID (PostgreSQL has it as well as SQLite; 
		 see DriverBehaviour::ROW_ID_FIELD_RETURNS_LAST_AUTOINCREMENTED_VALUE). 
		 ROWID's value will be assigned back to \a ROWID if this pointer is not null.
		*/
		Q_ULLONG lastInsertedAutoIncValue(const QString& aiFieldName, const QString& tableName, 
			Q_ULLONG* ROWID = 0);
		
		/*! \overload int lastInsertedAutoIncValue(const QString&, const QString&, Q_ULLONG*)
		*/
		Q_ULLONG lastInsertedAutoIncValue(const QString& aiFieldName, const TableSchema& table, Q_ULLONG* ROWID = 0);

		/*! Executes query \a statement, but without returning resulting 
		 rows (used mostly for functional queries). 
		 Only use this method if you really need. */
		bool executeSQL( const QString& statement );

		/*! \return "SELECT ..." statement's string needed for executing query 
		 defined by \a querySchema.

		 Note: The statement string can be specific for this connection's driver database, 
		 and thus not reusable in general.
		*/
		inline QString selectStatement( QuerySchema& querySchema, int idEscaping = Driver::EscapeDriver|Driver::EscapeAsNecessary ) const
		{
			return selectStatement(querySchema, false, idEscaping);
		}

		/*! \return sql string of actually executed SQL statement,
		 usually using drv_executeSQL(). If there was error during executing SQL statement, 
		 before, that string is returned instead. */
		const QString recentSQLString() const { return m_errorSql.isEmpty() ? m_sql : m_errorSql; }

		/*! Stores object's schema data (id, name, caption, help text)
		 described by \a sdata on the backend. 
		 If \a newObject is true, new entry is created, 
		 and (when sdata.id() was <=0), new, unique object identifier 
		 is obtained and assigned to \a sdata (see SchemaData::id()).
		 
		 If \a newObject is false, it's expected that entry on the 
		 backend already exists, so it's updated (changes to identifier are not allowed).
		 \return true on success. */
		bool storeObjectSchemaData( SchemaData &sdata, bool newObject );

		/*! Added for convenience. 
		 \sa setupObjectSchemaData( const KexiDB::RowData &data, SchemaData &sdata ) */
		bool loadObjectSchemaData( int objectID, SchemaData &sdata );

		/*! Finds object schema data for object of type \a objectType and name \a objectName.
		 If the object is found, resulted schema is stored in \a sdata and true is returned,
		 otherwise false is returned. */
		bool loadObjectSchemaData( int objectType, const QString& objectName, SchemaData &sdata );

		/*! Loads (potentially large) data block (e.g. xml form's representation), referenced by objectID
		 and puts it to \a dataString. The can be block indexed with optional \a dataID.
		 \return true on success
		 \sa storeDataBlock(). */
		bool loadDataBlock( int objectID, QString &dataString, const QString& dataID );

		/*! Stores (potentially large) data block \a dataString (e.g. xml form's representation), 
		 referenced by objectID. Block will be stored in "kexi__objectdata" table and 
		 an optional \a dataID identifier. 
		 If there is already such record in the table, it's simply overwritten.
		 \return true on success
		 \sa loadDataBlock(). */
		bool storeDataBlock( int objectID, const QString &dataString, const QString& dataID = QString::null );

		/*! Removes (potentially large) string data (e.g. xml form's representation), 
		 referenced by objectID, and pointed by optional \a dataID.
		 \return true on success. Does not fail if the block does not exist.
		 Note that if \a dataID is not specified, all data blocks for this dialog will be removed.
		 \sa loadDataBlock() storeDataBlock(). */
		bool removeDataBlock( int objectID, const QString& dataID = QString::null);

		class KEXI_DB_EXPORT TableSchemaChangeListenerInterface
		{
			public:
				TableSchemaChangeListenerInterface() {}
				virtual ~TableSchemaChangeListenerInterface() {}
				/*! Closes listening object so it will be deleted and thus no longer use 
				 a conflicting table schema. */
				virtual tristate closeListener() = 0;

				/*! i18n'd string that can be displayed for user to inform about 
				 e.g. conflicting listeners. */
				QString listenerInfoString;
		};
//TMP// TODO: will be more generic
		/** Register \a listener for receiving (listening) informations about changes 
		 in TableSchema object. Changes could be: altering and removing. */
		void registerForTableSchemaChanges(TableSchemaChangeListenerInterface& listener, 
			TableSchema& schema);

		void unregisterForTableSchemaChanges(TableSchemaChangeListenerInterface& listener, 
			TableSchema &schema);

		void unregisterForTablesSchemaChanges(TableSchemaChangeListenerInterface& listener);

		QPtrList<Connection::TableSchemaChangeListenerInterface>*
			tableSchemaChangeListeners(TableSchema& tableSchema) const;

		tristate closeAllTableSchemaChangeListeners(TableSchema& tableSchema);

		/*! @internal Removes \a tableSchema from internal structures and 
		 destroys it. Does not make any change at the backend. */
		void removeTableSchemaInternal(KexiDB::TableSchema *tableSchema);

	protected:
		/*! Used by Driver */
		Connection( Driver *driver, ConnectionData &conn_data );

		/*! Method to be called form Connection's subclass destructor.
		 \sa ~Connection() */
		void destroy();

		/*! @internal drops table \a tableSchema physically, but destroys 
		 \a tableSchema object only if \a alsoRemoveSchema is true.
		 Used (alsoRemoveSchema==false) on table altering: 
		 if recreating table can failed we're giving up and keeping 
		 the original table schema (even if it is no longer points to any real data). */
		tristate dropTable( KexiDB::TableSchema* tableSchema, bool alsoRemoveSchema);

		/*! For reimplemenation: connects to database
			\return true on success. */
		virtual bool drv_connect() = 0;

		/*! For reimplemenation: disconnects database
			\return true on success. */
		virtual bool drv_disconnect() = 0;

		/*! Executes query \a statement, but without returning resulting 
		 rows (used mostly for functional queries). 
		 Only use this method if you really need. */
		virtual bool drv_executeSQL( const QString& statement ) = 0;

		/*! For reimplemenation: loads list of databases' names available for this connection
		 and adds these names to \a list. If your server is not able to offer such a list,
		 consider reimplementing drv_databaseExists() instead. 
		 The method should return true only if there was no error on getting database names 
		 list from the server.
		 Default implementation puts empty list into \a list and returns true. */
		virtual bool drv_getDatabasesList( QStringList &list );

//TODO: move this somewhere to low level class (MIGRATION?)
		/*! LOW LEVEL METHOD. For reimplemenation: loads low-level list of table names
		 available for this connection. The names are in lower case.
		 The method should return true only if there was no error on getting database names 
		 list from the server. */
		virtual bool drv_getTablesList( QStringList &list ) = 0;

//TODO: move this somewhere to low level class (MIGRATION?)
		/*! LOW LEVEL METHOD. For reimplemenation: returns true if table 
		 with name \a tableName exists in the database.
		 \return false if it does not exist or error occured.
		 The lookup is case insensitive. */
		virtual bool drv_containsTable( const QString &tableName ) = 0;

		/*! For optional reimplemenation: asks server if database \a dbName exists.
		 This method is used internally in databaseExists(). The default  implementation
		 calls databaseNames and checks if that list contains \a dbName. If you need to
		 ask the server specifically if a database exists, eg. if you can't retrieve a list
		 of all available database names, please reimplement this method and do all 
		 needed checks.
			
		 See databaseExists() description for details about ignoreErrors argument. 
		 You should use this appropriately in your implementation.

		 Note: This method should also work if there is already database used (with useDatabase());
		 in this situation no changes should be made in current database selection. */
		virtual bool drv_databaseExists( const QString &dbName, bool ignoreErrors = true );

		/*! For reimplemenation: creates new database using connection */
		virtual bool drv_createDatabase( const QString &dbName = QString::null ) = 0;

		/*! For reimplemenation: opens existing database using connection */
		virtual bool drv_useDatabase( const QString &dbName = QString::null ) = 0;

		/*! For reimplemenation: closes previously opened database 
			using connection. */
		virtual bool drv_closeDatabase() = 0;
		
		/*! \return true if internal driver's structure is still in opened/connected 
		 state and database is used. 
		 Note for driver developers: Put here every test that you can do using your 
		 internal engine's database API, 
		 eg (a bit schematic):  my_connection_struct->isConnected()==true. 
		 Do not check things like Connection::isDatabaseUsed() here or other things 
		 that "KexiDB already knows" at its level. 
		 If you cannot test anything, just leave default implementation (that returns true).
		 
		 Result of this method is used as an addtional chance to check for isDatabaseUsed().
		 Do not call this method from your driver's code, it should be used at KexiDB 
		 level only.
		*/
		virtual bool drv_isDatabaseUsed() const { return true; }

		/*! For reimplemenation: drops database from the server
			using connection. After drop, database shouldn't be accessible 
			anymore. */
		virtual bool drv_dropDatabase( const QString &dbName = QString::null ) = 0;

		/*! \return "CREATE TABLE ..." statement string needed for \a tableSchema
		 creation in the database. 
		 
		 Note: The statement string can be specific for this connection's driver database, 
		 and thus not reusable in general.
		*/
		QString createTableStatement( const TableSchema& tableSchema ) const;


		/*! \return "SELECT ..." statement's string needed for executing query 
		 defined by "select * from table_name" where <i>table_name</i> is \a tableSchema's name.
		 This method's variant can be useful when there is no appropriate QuerySchema defined.

		 Note: The statement string can be specific for this connection's driver database, 
		 and thus not reusable in general.
		*/
		QString selectStatement( TableSchema& tableSchema ) const;

		/*! Like selectStatement( QuerySchema& querySchema, int idEscaping = Driver::EscapeDriver|Driver::EscapeAsNecessary ) const
		 but also retrieves ROWID information, if \a alsoRetrieveROWID is true.
		 Used by cursors. */
		QString selectStatement( QuerySchema& querySchema, bool alsoRetrieveROWID, int idEscaping = Driver::EscapeDriver|Driver::EscapeAsNecessary ) const;

		/*! Creates table using \a tableSchema information.
		 \return true on success. Default implementation 
		 builds a statement using createTableStatement() and calls drv_executeSQL()
		 Note for driver developers: reimplement this only if you want do to 
		 this in other way.
		 */
		virtual bool drv_createTable( const TableSchema& tableSchema );

		/*! 
		 Creates table named by \a tableSchemaName. Schema object must be on
		 schema tables' list before calling this method (otherwise false if returned).
		 Just uses drv_createTable( const KexiDB::TableSchema& tableSchema ).
		 Used internally, e.g. in createDatabase().
		 \return true on success
		*/
		virtual bool drv_createTable( const QString& tableSchemaName );

//		/*! Executes query \a statement and returns resulting rows 
//			(used mostly for SELECT query). */
//		virtual bool drv_executeQuery( const QString& statement ) = 0;

		/*! \return unique identifier of last inserted row. 
		 Typically this is just primary key value. 
		 This identifier could be reused when we want to reference
		 just inserted row.
		 Note for driver developers: contact js (at) iidea.pl 
		 if your engine do not offers this information. */
		virtual Q_ULLONG drv_lastInsertRowID() = 0;

		/*! Note for driver developers: begins new transaction
		 and returns handle to it. Default implementation just
		 executes "BEGIN" sql statement and returns just empty data (TransactionData object).
		 
		 Drivers that do not support transactions (see Driver::features())
		 do never call this method.
		 Reimplement this method if you need to do something more 
		 (e.g. if you driver will support multiple transactions per connection).
		 Make subclass of TransactionData (declared in transaction.h)
		 and return object of this subclass.
		 You should return NULL if any error occured.
		 Do not check anything in connection (isConnected(), etc.) - all is already done.
		*/
		virtual TransactionData* drv_beginTransaction();
		
		/*! Note for driver developers: begins new transaction
		 and returns handle to it. Default implementation just
		 executes "COMMIT" sql statement and returns true on success.
		 
		 \sa drv_beginTransaction()
		*/
		virtual bool drv_commitTransaction(TransactionData* trans);
		
		/*! Note for driver developers: begins new transaction
		 and returns handle to it. Default implementation just
		 executes "ROLLBACK" sql statement and returns true on success.
		 
		 \sa drv_beginTransaction()
		*/
		virtual bool drv_rollbackTransaction(TransactionData* trans);

		/*! Changes autocommiting option for established connection.
		 \return true on success.
		 
		 Note for driver developers: reimplement this only if your engine
		 allows to set special auto commit option (like "SET AUTOCOMMIT=.." in MySQL).
		 If not, auto commit behaviour will be simulated if at least single 
		 transactions per connection are supported by the engine.
		 Do not set any internal flags for autocommiting -- it is already done inside
		 setAutoCommit().
		 
		 Default implementation does nothing with connection, just returns true.
		 
		 \sa drv_beginTransaction(), autoCommit(), setAutoCommit()
		*/
		virtual bool drv_setAutoCommit(bool on);

		/*! Physically drops table named with \a name. 
		 Default impelmentation executes "DROP TABLE.." command,
		 so you rarely want to change this. */
		virtual bool drv_dropTable( const QString& name );

		/*! Alters table's described \a tableSchema name to \a newName. 
		 Default implementation is ineffective: 
		 - creates a copy of the table
		 - copies all rows
		 - drops old table.
		 All this is performed within single transaction. This is how SQLite driver work.
		 \return true on success.
		 More advanced server backends should reinplement this using "ALTER TABLE". 
		*/
		virtual bool drv_alterTableName(TableSchema& tableSchema, const QString& newName, bool replace = false);

		/*! Internal, for handling autocommited transactions:
		 begins transaction is one is supported.
		 \return true if new transaction started
		 successfully or no transactions are supported at all by the driver
		 or if autocommit option is turned off.
		 A handle to a newly created transaction (or null on error) is passed 
		 to \a tg parameter.

		 Special case when used database driver has only single transaction support 
		 (Driver::SingleTransactions): 
		 and there is already transaction started, it is commited before
		 starting a new one, but only if this transaction has been started inside Connection object.
		 (i.e. by beginAutoCommitTransaction()). Otherwise, a new transaction will not be started, 
		 but true will be returned immediately.
		*/
		bool beginAutoCommitTransaction(TransactionGuard& tg);
		
		/*! Internal, for handling autocommited transactions:
		 Commits transaction prevoiusly started with beginAutoCommitTransaction().
		 \return true on success or when no transactions are supported 
		 at all by the driver.

		 Special case when used database driver has only single transaction support 
		 (Driver::SingleTransactions): if \a trans has been started outside Connection object 
		 (i.e. not by beginAutoCommitTransaction()), the transaction will not be commited.
		*/
		bool commitAutoCommitTransaction(const Transaction& trans);
		
		/*! Internal, for handling autocommited transactions:
		 Rollbacks transaction prevoiusly started with beginAutoCommitTransaction().
		 \return true on success or when no transactions are supported 
		 at all by the driver.

		 Special case when used database driver has only single transaction support 
		 (Driver::SingleTransactions): \a trans will not be rolled back 
		 if it has been started outside this Connection object.
		*/
		bool rollbackAutoCommitTransaction(const Transaction& trans);

		/*! Creates cursor data and initializes cursor 
			using \a statement for later data retrieval. */
//		virtual CursorData* drv_createCursor( const QString& statement ) = 0;
		/*! Closes and deletes cursor data. */
//		virtual bool drv_deleteCursor( CursorData *data ) = 0;
		
		/*! Helper: checks if connection is established; 
			if not: error message is set up and false returned */
		bool checkConnected();

		/*! Helper: checks both if connection is established and database any is used; 
			if not: error message is set up and false returned */
		bool checkIsDatabaseUsed();

		/*! Setups schema data for object that owns sdata (e.g. table, query)
			using \a cursor opened on 'kexi__objects' table, pointing to a record
			corresponding to given object. */
		bool setupObjectSchemaData( const RowData &data, SchemaData &sdata );

		/*! \return a full table schema for a table using 'kexi__*' system tables. 
		 Used internally by tableSchema() methods. */
		TableSchema* setupTableSchema( const RowData &data );

		/*! \return a full query schema for a query using 'kexi__*' system tables. 
		 Used internally by querySchema() methods. */
		QuerySchema* setupQuerySchema( const RowData &data );

		bool updateRow(QuerySchema &query, RowData& data, RowEditBuffer& buf, bool useROWID = false);

		bool insertRow(QuerySchema &query, RowData& data, RowEditBuffer& buf, bool getROWID = false);

		bool deleteRow(QuerySchema &query, RowData& data, bool useROWID = false);

		bool deleteAllRows(QuerySchema &query);

		/*! Allocates all needed table KexiDB system objects for kexi__* KexiDB liblary's
		 system tables schema.
		 These objects are used internally in this connection
		 and are added to list of tables (by name, 
		 not by id because these have no ids).
		*/
		bool setupKexiDBSystemSchema();

		/*! used internally by setupKexiDBSystemSchema():
		 Allocates single table KexiDB system object named \a tsname 
		 and adds this to list of such objects (for later removal on closeDatabase()). 
		*/
		TableSchema* newKexiDBSystemTableSchema(const QString& tsname);

		//! Identifier escaping function in the associated Driver.
		/*! Calls the identifier escaping function in the associated Driver to
		 escape table and column names.  This should be used when explicitly
		 constructing SQL strings (e.g. "FROM " + escapeIdentifier(tablename)).
		 It should not be used for other functions (e.g. don't do
		 useDatabase(escapeIdentifier(database))), because the identifier will
		 be escaped when the called function generates, for example, "USE " +
		 escapeIdentifier(database).
		 
		 For efficiency, kexi__* system tables and columns therein are not escaped
		 - we assume these are valid identifiers for all drivers.
		*/
		inline QString escapeIdentifier(const QString& id, 
		    int drvEscaping = Driver::EscapeDriver|Driver::EscapeAsNecessary ) const {
			return m_driver->escapeIdentifier(id, drvEscaping);
		}
		
		/*! Called by TableSchema -- signals destruction to Connection object
		 To avoid having deleted table object on its list. */
		void removeMe(TableSchema *ts);

		/*! @internal 
		 \return true if the cursor \a cursor contains column \a column,
		 else, sets appropriate error with a message and returns false. */
		bool checkIfColumnExists(Cursor *cursor, uint column);

		QGuardedPtr<ConnectionData> m_data;
		QString m_name;
		QString m_usedDatabase; //!< database name that is opened now

		//! Table schemas retrieved on demand with tableSchema()
		QIntDict<TableSchema> m_tables;
		QDict<TableSchema> m_tables_byname;
		QIntDict<QuerySchema> m_queries;
		QDict<QuerySchema> m_queries_byname;

		//! used just for removing system TableSchema objects on db close.
		QPtrList<TableSchema> m_kexiDBSystemtables;

		//! cursors created for this connection
		QPtrDict<KexiDB::Cursor> m_cursors;
//		ConnectionInternal* m_internal;

		//! used to store of actually executed SQL statement
		QString m_sql, m_errorSql;

	friend class KexiDB::Driver;
	friend class KexiDB::Cursor;
	friend class KexiDB::TableSchema; //!< for removeMe()
	friend class ConnectionPrivate;

		ConnectionPrivate *d;
	private:
		Driver *m_driver;
		bool m_is_connected : 1;
		bool m_autoCommit : 1;
		bool m_destructor_started : 1; //!< helper: true if destructor is started
		
		QString m_availableDatabaseName; //! used by anyAvailableDatabaseName()
};

} //namespace KexiDB

#endif

