/*
 * Copyright (C) 2000 Richard Groult <rgroult@jalix.org>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include "rename.h"

#include <qbuttongroup.h>
#include <qcheckbox.h>
#include <qgroupbox.h>
#include <qheader.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qlistview.h>
#include <qpushbutton.h>
#include <qradiobutton.h>
#include <qspinbox.h>
#include <qlayout.h>
#include <qvariant.h>
#include <qtooltip.h>
#include <qwhatsthis.h>
#include <qimage.h>
#include <qpixmap.h>
#include <qfiledialog.h>


#include <klocale.h>
#include <kapp.h>
#include <kmessagebox.h>
#include <kiconloader.h>

/* 
 *  Constructs a RenameSeries which is a child of 'parent', with the 
 *  name 'name' and widget flags set to 'f' 
 *
 *  The dialog will by default be modeless, unless you set 'modal' to
 *  TRUE to construct a modal dialog.
 */
RenameSeries::RenameSeries( QWidget* parent,  const char* name, bool modal, WFlags fl )
    : KDialog( parent, name, modal, fl )
{
	if ( !name )
	    setName( "RenameSeries" );
	resize( 641, 552 ); 
	setMinimumSize( QSize( 0, 0 ) );
	setMaximumSize( QSize( 1024, 768 ) );
	setCaption( i18n( "Rename Series" ) );
	QWhatsThis::add(  this, i18n( "<u>Description:</u><br>\n\nAdds an expression describing the final file name. Valid tokens are:<br>\n<br>\n<b>$</b> old file name<br><b>%</b> old file name converted to lower case<br><b>Both old file names don't include the file extension!</b><br>\n<br>\n<b>#</b> Adds a number to the file name starting with StartIndex.<br><b>Example:</b>pic_$_#.jpg ." ) );

	GroupBox1 = new QGroupBox( this, "GroupBox1" );
	GroupBox1->setGeometry( QRect( 10, 0, 630, 500 ) ); 
	GroupBox1->setTitle( i18n( "Options" ) );

	overwrite = new QCheckBox( GroupBox1, "overwrite" );
	overwrite->setGeometry( QRect( 10, 80, 290, 20 ) );
	overwrite->setText( i18n( "Overwrite &existing file" ) );

	paternLineEdit = new QLineEdit( GroupBox1, "paternLineEdit" );
	paternLineEdit->setGeometry( QRect( 10, 40, 110, 20 ) );
	paternLineEdit->setText( i18n( "Pattern_#" ) );

	checkExtension = new QCheckBox( GroupBox1, "checkExtension" );
	checkExtension->setGeometry( QRect( 130, 20, 171, 60 ) );
	checkExtension->setText( i18n( "&Use original file extension" ) );
	checkExtension->setChecked( TRUE );
	QToolTip::add(  checkExtension, i18n( "This option should be enabled" ) );
	QWhatsThis::add(  checkExtension, i18n( "<u>Description:</u><br>\nThe original file extension will be added to the final file name. Useful if you want to rename many files of different types." ) );

	moveDownPushButton = new QPushButton( GroupBox1, "moveDownPushButton" );
	moveDownPushButton->setGeometry( QRect( 570, 250, 50, 30 ) );
	moveDownPushButton->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)0, (QSizePolicy::SizeType)0, moveDownPushButton->sizePolicy().hasHeightForWidth() ) );
	moveDownPushButton->setText( i18n( "&down" ) );
	moveDownPushButton->setDefault( FALSE );
	moveDownPushButton->setFlat( FALSE );
	QToolTip::add(  moveDownPushButton, i18n( "Move item down" ) );
	QWhatsThis::add(  moveDownPushButton, i18n( "Move the selected item down" ) );

	moveUpPushButton = new QPushButton( GroupBox1, "moveUpPushButton" );
	moveUpPushButton->setGeometry( QRect( 570, 220, 50, 30 ) ); 
	moveUpPushButton->setText( i18n( "&up" ) );

	GroupBox9 = new QGroupBox( GroupBox1, "GroupBox9" );
	GroupBox9->setGeometry( QRect( 10, 110, 270, 60 ) );
	GroupBox9->setTitle( i18n( "Numbers" ) );
	QWhatsThis::add(  GroupBox9, i18n( "<u>Describtion:</u><br>\n<b>This field is only necessary if you added a \"#\" to the file name!</b><br>\nThe first file will get the number start index, the second file start index + 1 and so on. E.g. if your file name is picture#, the files will be named picture1, picture2, ... .<br>\n<br>\n\"Fill with 0's\" adds 0 before the number if you have more than 9 files to rename. E.g. picture01,picture02, ... picture10." ) );

	QWidget* privateLayoutWidget = new QWidget( GroupBox9, "Layout1" );
	privateLayoutWidget->setGeometry( QRect( 10, 20, 180, 25 ) );
	Layout1 = new QHBoxLayout( privateLayoutWidget );
	Layout1->setSpacing( 6 );
	Layout1->setMargin( 0 );

	TextLabel2_2 = new QLabel( privateLayoutWidget, "TextLabel2_2" );
	TextLabel2_2->setText( i18n( "Start Index" ) );
	Layout1->addWidget( TextLabel2_2 );
	QSpacerItem* spacer = new QSpacerItem( 20, 20, QSizePolicy::Expanding, QSizePolicy::Minimum );
	Layout1->addItem( spacer );

	spinIndex = new QSpinBox( privateLayoutWidget, "spinIndex" );
	QToolTip::add(  spinIndex, i18n( "Start index, if numbers (#) are used in the file name" ) );
	Layout1->addWidget( spinIndex );

	GroupBox5 = new QGroupBox( GroupBox1, "GroupBox5" );
	GroupBox5->setGeometry( QRect( 10, 290, 270, 201 ) );
	GroupBox5->setTitle( i18n( "Date" ) );

	CheckBox7 = new QCheckBox( GroupBox5, "CheckBox7" );
	CheckBox7->setGeometry( QRect( 10, 20, 251, 20 ) );
	CheckBox7->setText( i18n( "&Change date " ) );
	QToolTip::add(  CheckBox7, i18n( "Changes the creation date of files" ) );

	kDate = new KDatePicker( GroupBox5 );
	kDate->setGeometry( QRect( 10, 50, 250, 140 ) ); 
	kDate->setEnabled(false);
	QToolTip::add( kDate, i18n( "Pick a date") );


	previewCheckBox = new QCheckBox( GroupBox1, "previewCheckBox" );
	previewCheckBox->setGeometry( QRect( 340, 290, 190, 20 ) ); 
	previewCheckBox->setText( i18n( "Show Image &Preview" ) );

	PixmapLabel1 = new QLabel( GroupBox1, "PixmapLabel1" );
	PixmapLabel1->setGeometry( QRect( 340, 310, 180, 160 ) ); 
	PixmapLabel1->setFrameShape( QLabel::Box );
	PixmapLabel1->setFrameShadow( QLabel::Raised );
	PixmapLabel1->setPixmap( QPixmap() );
	PixmapLabel1->setScaledContents( TRUE );

	Line2 = new QFrame( GroupBox1, "Line2" );
	Line2->setGeometry( QRect( 300, 30, 20, 430 ) ); 
	Line2->setFrameStyle( QFrame::VLine | QFrame::Sunken );

	TextLabel1 = new QLabel( GroupBox1, "TextLabel1" );
	TextLabel1->setGeometry( QRect( 10, 20, 140, 20 ) );
	TextLabel1->setText( i18n( "Rename pattern :" ) );

	TextLabel2 = new QLabel( GroupBox1, "TextLabel2" );
	TextLabel2->setGeometry( QRect( 340, 30, 131, 20 ) );
	TextLabel2->setText( i18n( "Renamed preview :" ) );

	remanedPreviewListView = new QListView( GroupBox1, "remanedPreviewListView" );
	remanedPreviewListView->addColumn( i18n( "origin" ) );
	remanedPreviewListView->header()->setClickEnabled( FALSE, remanedPreviewListView->header()->count() - 1 );
	remanedPreviewListView->addColumn( i18n( "renamed" ) );
	remanedPreviewListView->header()->setClickEnabled( FALSE, remanedPreviewListView->header()->count() - 1 );
	remanedPreviewListView->header()->setResizeEnabled( FALSE, remanedPreviewListView->header()->count() - 1 );

	remanedPreviewListView->setGeometry( QRect( 340, 50, 230, 230 ) ); 
	remanedPreviewListView->setMouseTracking( FALSE );
	remanedPreviewListView->setFocusPolicy( QListView::NoFocus );
	remanedPreviewListView->setFrameShape( QListView::Box );
	remanedPreviewListView->setFrameShadow( QListView::Plain );
	remanedPreviewListView->setLineWidth( 1 );
	remanedPreviewListView->setMargin( 0 );
	remanedPreviewListView->setMidLineWidth( 0 );
	remanedPreviewListView->setAllColumnsShowFocus( TRUE );
	remanedPreviewListView->setShowSortIndicator( TRUE );
	remanedPreviewListView->setItemMargin( 1 );
	remanedPreviewListView->setRootIsDecorated( FALSE );

	ButtonGroup1 = new QButtonGroup( GroupBox1, "ButtonGroup1" );
	ButtonGroup1->setGeometry( QRect( 10, 170, 270, 120 ) );
	ButtonGroup1->setTitle( i18n( "Type" ) );
	QWhatsThis::add(  ButtonGroup1, i18n( "<u>Description:</u><br>This option should be self explaining.<br>  <br><b>Warning!</b> Moving files does not work between different partitions.<br> File copy works OK though different partitions!" ) );

	optionCopy = new QRadioButton( ButtonGroup1, "optionCopy" );
	optionCopy->setGeometry( QRect( 10, 20, 251, 20 ) );
	optionCopy->setText( i18n( "&Copy files to..." ) );
	QToolTip::add(  optionCopy, i18n( "Your files will be copied<br>to the destination directory and then renamed" ) );
	QWhatsThis::add(  optionCopy, "" );

	optionMove = new QRadioButton( ButtonGroup1, "optionMove" );
	optionMove->setGeometry( QRect( 10, 40, 251, 20 ) );
	optionMove->setText( i18n( "&Move files to..." ) );
	QToolTip::add(  optionMove, i18n( "Your files will be moved<br>to the destination directory and then renamed" ) );

	optionRename = new QRadioButton( ButtonGroup1, "optionRename" );
	optionRename->setGeometry( QRect( 10, 60, 241, 20 ) ); 
	optionRename->setText( i18n( "&Rename input files" ) );
	optionRename->setChecked( TRUE );
	QToolTip::add(  optionRename, i18n( "Your files will be copied<br>to the destination directory and then renamed" ) );

	dirname = new QLineEdit( ButtonGroup1, "dirname" );
	dirname->setEnabled( FALSE );
	dirname->setGeometry( QRect( 10, 90, 211, 20 ) );

	buttonChooseDir = new QPushButton( ButtonGroup1, "buttonChooseDir" );
	buttonChooseDir->setEnabled( FALSE );
	buttonChooseDir->setGeometry( QRect( 230, 90, 30, 20 ) );
	buttonChooseDir->setText( "" );
	buttonChooseDir->setPixmap( QPixmap(BarIcon("folder_open",16 )));
	QToolTip::add(  buttonChooseDir, i18n( "Select a directory where<br>your files should be moved or copied" ) );

	cancelPushButton = new QPushButton( this, "cancelPushButton" );
	cancelPushButton->setGeometry( QRect( 560, 510, 80, 30 ) ); 
	cancelPushButton->setText( i18n( "&Cancel" ) );

	renamePushButton = new QPushButton( this, "renamePushButton" );
	renamePushButton->setGeometry( QRect( 470, 510, 80, 30 ) ); 
	renamePushButton->setText( i18n( "&Rename..." ) );

	// signals and slots connections
	connect( optionMove, SIGNAL( toggled(bool) ), dirname, SLOT( setEnabled(bool) ) );
	connect( optionMove, SIGNAL( toggled(bool) ), buttonChooseDir, SLOT( setEnabled(bool) ) );
	connect( optionCopy, SIGNAL( toggled(bool) ), dirname, SLOT( setEnabled(bool) ) );
	connect( optionCopy, SIGNAL( toggled(bool) ), buttonChooseDir, SLOT( setEnabled(bool) ) );
	connect( optionRename, SIGNAL( toggled(bool) ), dirname, SLOT( setDisabled(bool) ) );
	connect( optionRename, SIGNAL( toggled(bool) ), buttonChooseDir, SLOT( setDisabled(bool) ) );

	connect( buttonChooseDir, SIGNAL( clicked()), this, SLOT( chooseDir() ));

	connect( CheckBox7, SIGNAL( toggled(bool) ), kDate, SLOT( setEnabled(bool) ) );

	// signals and slots connections
	connect( renamePushButton, SIGNAL( clicked() ), this, SLOT( accept() ) );
	connect( cancelPushButton, SIGNAL( clicked() ), this, SLOT( reject() ) );
	connect( previewCheckBox, SIGNAL( stateChanged(int) ), this, SLOT( slotSetImagePreview(int) ) );
	connect( moveDownPushButton, SIGNAL( clicked() ), this, SLOT( slotMoveDown() ) );
	connect( remanedPreviewListView, SIGNAL( selectionChanged(QListViewItem*) ), this, SLOT( slotUpdatePreview(QListViewItem*) ) );
	connect( moveUpPushButton, SIGNAL( clicked() ), this, SLOT( slotMoveUp() ) );
	//
	connect( spinIndex,   SIGNAL( valueChanged( int ) ), this, SLOT( slotUpdateRenamed() ) );
	connect( checkExtension,   SIGNAL( toggled ( bool ) ), this, SLOT( slotUpdateRenamed() ) );
	//
	connect( paternLineEdit, SIGNAL(  textChanged ( const QString & ) ), this, SLOT( slotUpdateRenamed(const QString &)) );
	connect( paternLineEdit, SIGNAL( returnPressed() ), this, SLOT( accept() ) );

	paternLineEdit->setFocus();

	// tab order
	setTabOrder( paternLineEdit, checkExtension );
	setTabOrder( checkExtension, spinIndex );
	setTabOrder( spinIndex, previewCheckBox );
	setTabOrder( previewCheckBox, moveUpPushButton );
	setTabOrder( moveUpPushButton, moveDownPushButton );
	setTabOrder( moveDownPushButton, renamePushButton );
	setTabOrder( renamePushButton, cancelPushButton );
	setTabOrder( cancelPushButton, remanedPreviewListView );

	//
	arrayNames = QArray<QString*>();
	taille=0;
	last=NULL;
	currentItem=NULL;

	newNames = new BatchRenamer();
}

/*
 *  Destroys the object and frees any allocated resources
 */
RenameSeries::~RenameSeries()
{
	// no need to delete child widgets, Qt does it all for us
}

void RenameSeries::chooseDir()
{
 	 QString s (QFileDialog::getExistingDirectory ( QString::null, this ));
 	 if(!s.isEmpty())
 		dirname->setText( s );
}


void RenameSeries::addFile(QString filename)
{
	int pos = filename.findRev ("/");
	QString name = filename.right (filename.length () - pos-1);

	QListViewItem *item = new QListViewItem( remanedPreviewListView, last);
	last=item;
	item->setText( 0,  name );
	item->setText( 1,  paternLineEdit->text()+QString().setNum(taille)); 
	taille++;
	QString order;
	order.sprintf ("%100d",taille);
	item->setText( 2,  order);

	arrayNames.resize(taille);
	arrayNames[taille-1]=new QString(filename);
}

void RenameSeries::slotMoveDown()
{
	remanedPreviewListView->disconnect(this);

	if(currentItem)
	{
		QListViewItem* apres=currentItem->itemBelow();
		if(currentItem->itemBelow ())
		{
			QString sBelow=apres->text(0);
			apres->setText(0, currentItem->text(0));
			currentItem->setText(0, sBelow);	

			remanedPreviewListView->setCurrentItem(apres);
			remanedPreviewListView->setSelected(apres, true);
			currentItem = apres;

			////////
			int indice =(int)((float)remanedPreviewListView->itemPos(currentItem)/currentItem->height())-1;

			QString* tmp = arrayNames[indice+1];
			arrayNames[indice+1] = arrayNames[indice];
			arrayNames[indice]=tmp;
		}
	}
	connect( remanedPreviewListView, SIGNAL( selectionChanged(QListViewItem*) ), this, SLOT( slotUpdatePreview(QListViewItem*) ) );
}

void RenameSeries::slotMoveUp()
{
	remanedPreviewListView->disconnect(this);
	if(currentItem)
		if(currentItem->itemAbove())
		{
			QListViewItem* avant=currentItem->itemAbove();
			if(avant) 
			{
				QString sBelow=avant->text(0);
				avant->setText(0, currentItem->text(0));
				currentItem->setText(0, sBelow);

				remanedPreviewListView->setCurrentItem(avant);
				remanedPreviewListView->setSelected(avant, true);
				currentItem=avant;
			
				//////  
				int indice =(int)((float)remanedPreviewListView->itemPos(currentItem)/currentItem->height()+1);

				QString* tmp = arrayNames[indice-1];
				arrayNames[indice-1] = arrayNames[indice];
				arrayNames[indice]=tmp;
			}
		}      
	connect( remanedPreviewListView, SIGNAL( selectionChanged(QListViewItem*) ), this, SLOT( slotUpdatePreview(QListViewItem*) ) );
}

void RenameSeries::slotSetImagePreview(int)
{
	if(! previewCheckBox->isChecked())
	{   
		PixmapLabel1->setPixmap(QPixmap());     
	}
	else
	{
		if(!currentItem)
			return;

		QApplication::setOverrideCursor (waitCursor);	  // this might take time

		int indice =(int)((float)remanedPreviewListView->itemPos(currentItem)/currentItem->height());

		QImage im = QImage(*arrayNames[indice]);
		im = im.smoothScale(PixmapLabel1->width(), PixmapLabel1->height());
		QPixmap pix;
		pix.convertFromImage(im);
		PixmapLabel1->setPixmap(pix);

		QApplication::restoreOverrideCursor ();   // restore original cursor
	} 
}

void RenameSeries::slotUpdatePreview(QListViewItem* item)
{
	currentItem = item;

	if(previewCheckBox->isChecked())
	{
		QApplication::setOverrideCursor (waitCursor);	    // this might take time

		int indice =(int)((float)remanedPreviewListView->itemPos(item)/item->height());
		QImage im = QImage(*arrayNames[indice]);
		im = im.smoothScale(PixmapLabel1->width(), PixmapLabel1->height());
		QPixmap pix;
		pix.convertFromImage(im);
		PixmapLabel1->setPixmap(pix);

		QApplication::restoreOverrideCursor ();     // restore original cursor
	}
}

int RenameSeries::exec ()
{
	remanedPreviewListView->setSorting(2, true);
	remanedPreviewListView->sort();
	slotUpdateRenamed();
	return QDialog::exec();
}

void RenameSeries::slotUpdateRenamed()
{
	slotUpdateRenamed(NULL);
}

void RenameSeries::slotUpdateRenamed(const QString &)
{
	QString fileName, tmp;
	QListViewItem * myChild = remanedPreviewListView->firstChild();
	int pos=0;
	QFileInfo fi;
	while( myChild )
    	{
		 fileName = QFileInfo(myChild->text(0)).baseName();

		 tmp = newNames->findOldName(fileName , paternLineEdit->text() );
		 tmp = newNames->findOldNameLower( fileName, tmp );
		 tmp = newNames->findNumbers( tmp, pos, arrayNames.size(), spinIndex->value());
		
		 if(checkExtension->isChecked())
		 {
			fi.setFile( *arrayNames[pos] );
			if( !fi.extension().isEmpty())
			{
				tmp+="."+fi.extension();
			}
		 }

		 myChild->setText(1,tmp);

		 pos++;
		 myChild = myChild->nextSibling();
	}
}

bool RenameSeries::checkErrors(bool checkDir)
{
	// Error checking:
	QDir d;
	if( paternLineEdit->text().stripWhiteSpace().isEmpty() ) {
		KMessageBox::sorry( this, i18n("Please add an expression to rename the files!") );
		return false;
	}
	else 
	if( (checkDir) && (dirname->text().isEmpty() && !optionRename->isChecked()) )
	{
		KMessageBox::sorry( this, i18n("Please give a destination directory !") );
		return false;
	}
	else
	if((3+paternLineEdit->text().find('#')+paternLineEdit->text().find('$')+paternLineEdit->text().find('%'))==0)
	{
	 	 KMessageBox::sorry( this, i18n("The expression must contain <tt>'#'</tt>, <tt>'$'</tt>, or <tt>'\%'</tt>!") );
	 	 return false;
	}
	else
	if( (checkDir) && (!optionRename->isChecked()) )
	{
		if( dirname->text().left(1).compare("/") )
		{
			dirname->setText(QString("%1/").arg(dirname->text()));
		}
	}
	else
	if( (checkDir) && (!optionRename->isChecked()) )
	{
		if( d.exists( dirname->text(), FALSE ))
		{
			KMessageBox::sorry( this, "The given destination directory does not exist!" );
			return false;
		}
	}
	return true;

}

QString RenameSeries::getPath( QString fn )
{
	int pos = fn.findRev( "/" );
	return fn.mid( 0, pos + 1 );
}

void RenameSeries::accept()
{    
	if(!checkErrors())
		return;

	QFileInfo fi;
	data* files = new data[arrayNames.size()];
	values* val = new values;

	ProgressDialog *p = new ProgressDialog( parentWidget(), true, "renamed ProgressDialog", false );
	p->init(arrayNames.size());
	p->print(QString("Renaming %1 files....").arg(arrayNames.size()));

	for(unsigned int i=0; i<arrayNames.size();i++)
	{
		fi.setFile( *arrayNames[i] );
		files[i].source = fi.baseName();
		files[i].extension = fi.extension();
		files[i].count = arrayNames.size();
		if(! files[i].extension.isEmpty() )
			files[i].extension.insert( 0, '.' );

	    files[i].source_path = getPath(fi.filePath());
	}
	BatchRenamer *b = new BatchRenamer(p);

	//
	enum mode m = RENAME;
	if( optionCopy->isChecked() )
	{
	    m = COPY;	     
	}
	else 
	if( optionMove->isChecked() ) 
	{
	    m = MOVE;	     
	} 
	else 
	if( optionRename->isChecked() ) 
	{
	    m = RENAME;
	}

	//
	val->text = paternLineEdit->text();
	val->dirname = dirname->text();
	val->dvals.date = kDate->getDate();
	val->index = spinIndex->value();
	val->extension = checkExtension->isChecked();
	val->overwrite = overwrite->isChecked();
	val->dvals.bDate = CheckBox7->isChecked();
	if( val->dvals.bDate )
	{
		val->dvals.changeModification = true;
		val->dvals.changeAccess = true;
		val->dvals.hour = 0;
		val->dvals.minute = 0;
		val->dvals.second = 0;
	}

	hide();
	p->show();
	b->processFiles( files, m, val/*, true */);

	delete(b);
	delete(p);

	QDialog::accept();
}

