/*
 * Copyright (C) 2000 Richard Groult <rgroult@jalix.org>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
 /* directoryview.cpp */

#include "directoryview.h"

#include "describeAlbum.h"
#include "directory.h"
#include "replace.h"

#include <qtooltip.h>
#include <qdatastream.h>
#include <qmessagebox.h>
#include <qprogressdialog.h>
#include <qapp.h>
#include <qdir.h>
#include <qclipboard.h>

#include <kurlrequesterdlg.h>
#include <kio/job.h>
#include <kio/jobclasses.h>
#include <kmessagebox.h>
#include <klineeditdlg.h>
#include <kglobalsettings.h>
#include <kpropsdlg.h>
#include <kfiledialog.h>

class Directory;


DirectoryView::DirectoryView (QWidget * parent, ImageViewer * iv, ImageListeView* imageList, MainWindow * mw)
:KListView (parent, "DirectoryView")
{
	viewport ()->setAcceptDrops (TRUE);

	this->iv = iv;
	this->imageList = imageList;
	this->mw = mw;

	addColumn (i18n("Name"));
	addColumn (i18n("Type"));
	addColumn (i18n("Size"));

	setColumnAlignment( 1, Qt::AlignLeft );
	setColumnAlignment( 2, Qt::AlignRight );

	setAllColumnsShowFocus( TRUE );
	setShowSortIndicator( TRUE );

	setSorting (1);
	sort();

	setSelectionMode (QListView::Extended);
	setShowHiddenDir(false);
	dropping = false;

	currentOperation=NONE;
	autoSkip=false;
	replaceAll=false;
	currentOperation=NONE;
	fileToProcess=NULL;
	destDir=NULL;
	clickedItem=NULL;
	
	
	autoopenTime = 750;
	autoopen_timer = new QTimer (this);
	connect (autoopen_timer, SIGNAL (timeout ()),
			   this, SLOT (openFolder ()));

	copyProgress=new QProgressDialog (mw, "copyProgress", true);
	
	
	dirWatch = new KDirWatch();
	connect(dirWatch, SIGNAL(dirty (const QString&)),    mw, SLOT(slotDirChange(const QString&)));
	connect(dirWatch, SIGNAL(fileDirty (const QString&)),mw, SLOT(slotFileChange(const QString&)));
	startWatchDir();
};


void
DirectoryView::initMenu(KActionCollection *actionCollection)
{
	popup = new QPopupMenu();
	actionCollection->action("editdirnew")->plug(popup);
	popup->insertSeparator();
	popup->insertSeparator();
	actionCollection->action("editdircopy")->plug(popup);
	actionCollection->action("editdirmove")->plug(popup);
	actionCollection->action("editdirrename")->plug(popup);
	actionCollection->action("editdirtrash")->plug(popup);
	actionCollection->action("editdirdelete")->plug(popup);
	popup->insertSeparator();
	actionCollection->action("editdirpaste files")->plug(popup);
	popup->insertSeparator();
	actionCollection->action("Dir Info")->plug(popup);
	actionCollection->action("Dir Properties")->plug(popup);
}


Directory*
DirectoryView::getDir(QString dirfullname)
{
	if(!QFileInfo(dirfullname).isDir())
		return NULL;
	QCString *res = new QCString (dirfullname);
	if((*res)[res->length()-1]=='/')
		res=new QCString(res->left(res->length()-1));
	int pos = res->find ("/");
	Directory *ssrep;
	res->setStr (res->right (res->length () - pos - 1));
	pos = res->find ("/");
	ssrep = (Directory*)firstChild();
	while (pos != -1 && ssrep!=NULL)
	{
		ssrep = ssrep->find (res->left (pos));
		res->setStr (res->right (res->length () - pos - 1));
		if (ssrep!=NULL)		
			pos = res->find ("/");
	 }
	if(!res->isEmpty() && ssrep!=NULL)
		ssrep = ssrep->find(*res);
	return ssrep;
}

void
DirectoryView::triggerUpdate ()
{
}


void
DirectoryView::updateContents ()
{
}

void 
DirectoryView::slotDirInfo()
{
	clickedItem=currentItem();
	if(clickedItem)
	{
		   DescribeAlbum(mw, ((Directory*)clickedItem)->fullName(), "IDirInfo", true).exec();
	}
}

void 
DirectoryView::slotDirProperty()
{
	clickedItem=currentItem();
	if(clickedItem)
	{
		QApplication::setOverrideCursor (waitCursor);	
		KFileItem *item = new KFileItem(KURL(((Directory*)clickedItem)->fullName()), "inode/directory", -1);
        	KPropertiesDialog *prop	= new KPropertiesDialog( item,
			mw->iv, "KPropertiesDialog",
			true, false);
			
		QApplication::restoreOverrideCursor ();
		prop->exec();
	}
}

bool
DirectoryView::isDropping()
{
	return dropping;
}

void
DirectoryView::slotShowItem (QListViewItem * item)
{
	ensureItemVisible(item);
	item->sortChildItems (1, true);
}

void
DirectoryView::startWatchDir(QString dir)
{
	mw->changeDirectory(dir);
	
	dirWatch->stopScan();
	dirWatch->addDir(dir);
	dirWatch->startScan();
	
}

void DirectoryView::stopWatchDir(QString dir)
{
	dirWatch->removeDir(dir);
}

void
DirectoryView::startWatchDir()
{	
	  dirWatch->startScan();
}


void DirectoryView::stopWatchDir()
{
	dirWatch->stopScan();
}


void
DirectoryView::contentsDragEnterEvent (QDragEnterEvent * event)
{
	setSelectionMode (QListView::Single);
	
	oldCurrent = currentItem ();
	dropping = true;
	if (QTextDrag::canDecode (event))
	{
		event->acceptAction ();
		QListViewItem *i = itemAt (contentsToViewport (event->pos ()));
		if (i)
		{
			dropItem = i;
			autoopen_timer->start (autoopenTime);
		}
	}
}

void
DirectoryView::contentsDragLeaveEvent (QDragLeaveEvent *)
{
	autoopen_timer->stop ();
	dropItem = NULL;	
	setMultiSelection (false);
	setCurrentItem (oldCurrent);
	setSelected (oldCurrent, TRUE);
	setSelectionMode (QListView::Extended);
	dropping = false;
}

void
DirectoryView::contentsDragMoveEvent (QDragMoveEvent * e)
{
	if(!dropping)
		return;

	if (!QTextDrag::canDecode (e))
	{
		e->ignore ();
		return;
	}

	QListViewItem *i = itemAt (contentsToViewport ( e->pos ()));
	if (i)
	{
		setSelected (i, TRUE);
		e->acceptAction ();
		if (i != dropItem)
		{
			autoopen_timer->stop ();			
			dropItem = i;			
			autoopen_timer->start (autoopenTime);
		}
	}
	else
	{
		e->ignore ();
		autoopen_timer->stop ();
		dropItem = NULL;
	}
}


void
DirectoryView::contentsDropEvent (QDropEvent * event)
{
	autoopen_timer->stop ();
	setSelected (dropItem, false);
	
	if ( !QUriDrag::canDecode(event) )
	{
		event->ignore();
	}
	else
	{
		event->acceptAction(true);
		imageList->stopLoading();
		qApp->processEvents();
		
		stopWatchDir();

		QImage image;
		QStrList lst;

		if (QUriDrag::decode (event, lst))
		{
			if (((Directory *) dropItem)->text (1) != i18n("Directory"))
			{
				dropItem = NULL;
				KMessageBox::error (this, i18n("Adding file in an archive is not yet implemented"), i18n("File(s) copy/move"));
			}
			else
			if(!QFileInfo(((Directory*)dropItem)->fullName()).isWritable ())
			{
				dropItem = NULL;
				KMessageBox::error (this, i18n("The destination directory is not writable"), i18n("File(s) copy/move"));
			}
			else
			switch (event->action ())
			{
				case QDropEvent::Copy:
				case QDropEvent::Move:
				{
					if(!dropItem)
					{
						qWarning("%s %d ERROR dropIten est null !!!", __FILE__,__LINE__);
		    				return;
					}
					destDir=new QString( ((Directory*)dropItem)->fullName());
	
					event->acceptAction();
					contentsDragLeaveEvent(NULL);
	
					fileToProcess = new QList<QString>;
	
					QString *fdep = NULL;
					for (uint i = 0; i < lst.count (); i++)
					{
						fdep = new QString (QUriDrag::uriToLocalFile (lst.at (i)));
						fileToProcess->append(new QString (QUriDrag::uriToLocalFile (lst.at (i))));
					}
					copyProgress = new QProgressDialog (mw, "copyProgress", true);
					copyProgress->setTotalSteps(fileToProcess->count());
					copyProgress->setProgress(0);
					totalFiles=fileToProcess->count();
	
					if (event->action () == QDropEvent::Copy)
					{
							currentOperation=COPY;
							copyProgress->setLabelText(i18n("File(s) Copy"));
							copyNext();
					}
					else
					{
						currentItemName=imageList->currentItemName();
							
						currentOperation=MOVE;
						copyProgress->setLabelText(i18n("File(s) Move"));
						moveNext();
					}
				}
		 		break;
				
		 		default:break;
			}
		}
	}
	setCurrentItem (oldCurrent);
	setSelected (oldCurrent, TRUE);
	dropping = false;
	setSelectionMode (QListView::Extended);
}


void
DirectoryView::contentsMousePressEvent(QMouseEvent * e)
{
	if (e->button () == RightButton)
	{
		clickedItem = itemAt(contentsToViewport(e->pos()));
		if ( ((Directory*) clickedItem)->text (1) == i18n("Directory"))
		{
			setCurrentItem(clickedItem);
			popup->exec(e->globalPos());			
		}
	}
	else
	if (e->button () == LeftButton)
	{
		QListView::contentsMousePressEvent (e);
	}
	else
	if (e->button () == MidButton)
	{
	}
}


QString*
DirectoryView:: getDestName(QString *orgFile)
{
	QString *fdest = new QString (*destDir);         		    
	int pos = orgFile->findRev ("/");
	fdest->append(orgFile->right(orgFile->length() - pos));
	if (QDir(*orgFile) == QDir(*fdest))
	{
		QString rep;
		rep.sprintf(i18n("Can't copy <b>%s</b> because the destination file is the same"),
				            (const char *) *orgFile);
		copyProgress->show();
		KMessageBox::error (copyProgress, rep, i18n("File(s) copy"));
		return NULL;
	}
	if (QFile(QFileInfo(*fdest).dirPath(true) + "/" + QFileInfo(*fdest).fileName() ).exists())
	{  		
		if(replaceAll)
		{
			return  fdest;
		}
		else
		{
		copyProgress->show();
		ConfirmReplace cr(copyProgress, "ConfirmReplace", *orgFile, *fdest);
		switch (cr.exec())
		{
			case 1 : /*Replace()*/
				return fdest;
            	   	        break;
				
			case 2 : /*slotSkip*/
				autoSkip=true;
				return NULL;
				break;
				
			case 3 :/*slotReplaceAll*/
				replaceAll=true;
				return fdest;
				break;
			
			case 4 : /*slotCancel*/
				setMultiSelection (false);
				clearSelection ();

				setCurrentItem (oldCurrent);
				setSelected (oldCurrent, TRUE);

				setSelectionMode (QListView::Extended);
				dropping = false;
				
				currentOperation=NONE;

			        return NULL;
				break;

			case 5 : /*slotRename*/
				return new QString(cr.getRename());
				break;
			}
		}
	}	
	return fdest;
}

void
DirectoryView::copyNext()
{
	if(!fileToProcess->isEmpty())
	{	
		copyProgress->setProgress(totalFiles-fileToProcess->count());
		QString *dest=getDestName(fileToProcess->first());
		
		if(dest)
		{
			copyProgress->setProgress(totalFiles-fileToProcess->count());
			copyProgress->setLabelText(i18n(QString().sprintf("Copy <b>`%s'</b>",(const char *)shrinkdn(*dest))));
		}
		
		while( (dest==NULL) && (!fileToProcess->isEmpty()) && (currentOperation!=NONE))
		{
			fileToProcess->removeFirst();
			if(!fileToProcess->isEmpty()) dest=getDestName(fileToProcess->first());
		}
		if(!fileToProcess->isEmpty())
		{
			if(currentOperation!=NONE)
			{
				QString *org=fileToProcess->first();
				copy(org, dest);
			}
		}
		else
		{
			currentOperation=NONE;	
		}
	}
	else
	{
		currentOperation=NONE;
	}
	
	if(currentOperation==NONE)
	{
		currentOperation=NONE;
		delete(fileToProcess); fileToProcess=NULL;
		
		autoSkip=false;
  		replaceAll=false;	
			
		copyProgress->setProgress(copyProgress->totalSteps());
		copyProgress->hide();
		delete(copyProgress);
		
		currentItemName=QString();
		
		Directory* dd=mw->findDir((const char*)*destDir);
		if(dd) if (dd->isSelected())dd->refresh();
		delete(destDir); destDir=NULL;
		
		startWatchDir();
		imageList->slotLoadFirst();
	}
}

void
DirectoryView:: moveNext()
{
	if(!fileToProcess->isEmpty())
	{		
		QString *dest=getDestName(fileToProcess->first());
		if(dest)
		{
			copyProgress->setProgress(totalFiles-fileToProcess->count());
			copyProgress->setLabelText(i18n(QString().sprintf("Move <b>`%s'</b>",(const char *)shrinkdn(*dest))));
		}
		while( (dest==NULL) && (!fileToProcess->isEmpty()) && (currentOperation!=NONE))
		{
			fileToProcess->removeFirst();
			if(!fileToProcess->isEmpty()) dest=getDestName(fileToProcess->first());
		}
		if(!fileToProcess->isEmpty())
		{
			if(currentOperation!=NONE)
			{
				QString *org=fileToProcess->first();
				int pos = org->findRev ("/");
				const QString itfi(org->right (org->length () - (pos + 1)));
				FileItem *fi = imageList->findItem (itfi);  
				if(fi)
				{
					FileItem *next=fi->nextItem();
					if(next)
						currentItemName=next->text();
					else
					{
						next=fi->prevItem();
						if(next)
							currentItemName=next->text();	
						else
							currentItemName=QString();
					}
				}
				else
					currentItemName=QString();
				if(fi && fi->name()=="CompressedImageFileItem")
				{
					copy(org, dest);
				}
				else
				{
					move(org, dest);	
				}
			}
		}
		else
		{
			currentOperation=NONE;	
		}
	}
	else
	{
		currentOperation=NONE;
	}
	
	if(currentOperation==NONE)
	{
		currentOperation=NONE;
		delete(fileToProcess); fileToProcess=NULL;
		
		autoSkip=false;
  		replaceAll=false;
		
		copyProgress->setProgress(copyProgress->totalSteps());
		copyProgress->hide();
		delete(copyProgress);
		
		imageList->setCurrentItemName(currentItemName);
		
		currentItemName=QString();
				
		Directory* dd=mw->findDir(*destDir);
		if(dd) if (dd->isSelected()) dd->refresh();
		delete(destDir); destDir=NULL;
		
		startWatchDir();
		imageList->slotLoadFirst();
	}
}

void
DirectoryView::copyingDirDone( KIO::Job *job)
{
	if(job->error()==0)
	{
		//pas d'erreurs
		Directory* dest=getDir(dirDest);
		if(dest)
		{
			if(!dest->isOpen())
			{
				//rien a faire
			}
			else
			{
				QString name=QDir(dirOrg).dirName();
				if(!getDir(dirDest+name))
				{
					//ajout du rep
					(void)new Directory( dest, name, this, iv, imageList, mw);
					dest->setExpandable(true);
				}
			}
		}
		else
		{
		}
	}
	else
	{
		//erreur
		//copyProgress->show();
		job->showErrorDialog(copyProgress);
	}

}

void
DirectoryView::movingDirDone( KIO::Job *job)
{
	if(job->error()==0)
	{
		//pas d'erreurs
		Directory* dest=getDir(dirDest);
		if(dest)
		{
			if(!dest->isOpen())
			{
				//rien a faire
			}
			else
			{
				QString name=QDir(dirOrg).dirName();
				if(!getDir(dirDest+name))
				{
					//ajout du rep
					(void)new Directory( dest, name, this, iv, imageList, mw);
					dest->setExpandable(true);
				}
			}
		}
		else
		{
		}
		getDir(dirOrg)->setSelected(false);
		delete(getDir(dirOrg));
	}
	else
	{
		//erreur
		//copyProgress->show();
		job->showErrorDialog(copyProgress);
	}
}

void
DirectoryView::copyingDone( KIO::Job *job)
{
	if(job->error()==0)
	{
		//pas d'erreurs
	}
	else
	{
		//erreur
		job->showErrorDialog();
	}
	fileToProcess->removeFirst();		
	
	if(currentOperation==COPY)
		copyNext();
	else
		moveNext();
}

void
DirectoryView::movingDone( KIO::Job *job )
{
	if(job->error()==0)
	{
		//pas d'erreurs
		//on supprimme l'icone de l'affichage
		QString *fdep=fileToProcess->first();
		int pos = fdep->findRev ("/");
		const QString itfi (fdep->right (fdep->length () - (pos + 1)));
		FileItem *fi = imageList->findItem (itfi);
		if(fi) fi->suppression(false);		
	}
	else
	{
		//erreur
		job->showErrorDialog();
	}
	fileToProcess->removeFirst();		

	if(currentOperation==MOVE)
		moveNext();
	else
		copyNext();
}

bool
DirectoryView::copy (QString *dep, QString *dest)
{
	KURL urldep,urldest;
	urldep.setPath(*dep);
	urldest.setPath(*dest);	
	KIO::FileCopyJob *fcj=KIO::file_copy(urldep, urldest, -1,
										true, false,
										false);

	connect(fcj, SIGNAL(result( KIO::Job *)),
	      this, SLOT(copyingDone( KIO::Job *)));

	return true;
}


bool
DirectoryView::move (QString *dep, QString *dest)
{
	KURL urldep,urldest;
	urldep.setPath(*dep);
	urldest.setPath(*dest);	
	KIO::FileCopyJob *fcj=KIO::file_move(urldep, urldest, -1,
								true, false,
								false);
	connect(fcj, SIGNAL(result( KIO::Job *)),
		this, SLOT(movingDone( KIO::Job *)));
	return true;
}



void
DirectoryView::copy(QStrList uris, QString dest)
{
	if(!QFileInfo(QFileInfo(dest).dirPath()).isWritable())
	{
		 KMessageBox::error(mw->iv,QString().sprintf( i18n("The directory '<b>%s</b>' is not writable"),(const char*)shrinkdn(dest)));
		 return;
	}

	this->destDir=new QString(dest);
	fileToProcess = new QList<QString>;
	
	QString *fdep = NULL;
	for (uint i = 0; i < uris.count (); i++)
	{
		fdep = new QString (QUriDrag::uriToLocalFile(uris.at (i)));
		if(QFileInfo(*fdep).isFile())
		{
			fileToProcess->append(fdep);
		}
	}
	copyProgress = new QProgressDialog (mw, "copyProgress", true);
	copyProgress->setTotalSteps(fileToProcess->count());
	copyProgress->setProgress(0);
	totalFiles=fileToProcess->count();

	currentOperation=COPY;
	copyProgress->setLabelText(i18n("File(s) Copy"));
	copyNext();
}


void
DirectoryView::move(QStrList uris, QString dest)
{
	if(!QFileInfo(QFileInfo(dest).dirPath()).isWritable())
	{
		 KMessageBox::error(mw->iv,QString().sprintf( i18n("The directory '<b>%s</b>' is not writable"),(const char*)shrinkdn(dest)));
		 return;
	}
	
	this->destDir=new QString(dest);
	fileToProcess = new QList<QString>;
	
	QString *fdep = NULL;
	for (uint i = 0; i < uris.count (); i++)
	{
		fdep = new QString (QUriDrag::uriToLocalFile(uris.at (i)));
		if(QFileInfo(*fdep).isFile())
		{
			fileToProcess->append(fdep);
		}
	}
	copyProgress = new QProgressDialog (mw, "copyProgress", true);
	copyProgress->setTotalSteps(fileToProcess->count());
	copyProgress->setProgress(0);
	totalFiles=fileToProcess->count();

	currentOperation=MOVE;
	copyProgress->setLabelText(i18n("File(s) Move"));
	moveNext();
}



void
DirectoryView::openFolder ()
{
	autoopen_timer->stop ();
	if (dropItem && !dropItem->isOpen ())
	{
		setOpen(dropItem, true);
	}
}


void
DirectoryView::slotDirPasteFiles()
{
	clickedItem=currentItem();
	
	QStrList uris;
	if(QUriDrag::decode(QApplication::clipboard()->data(), uris))
	{
		if(!uris.isEmpty())
			copy(uris, ((Directory*)clickedItem)->fullName());
	}

}



void
DirectoryView::slotRename ()
{
	clickedItem=currentItem();
	slotRename(currentItem());
}

void
DirectoryView::slotRename (QListViewItem *item)
{
	Directory *dir=(Directory*)item;
	QString fullname=dir->fullName();
	QString name=dir->name();	
	bool ok;
	const QString newName(KLineEditDlg::getText(QString().sprintf(i18n("Rename directory '<b>%s</b>' :"),(const char*)shrinkdn(fullname)),
				name,
				&ok, mw).stripWhiteSpace());
	if(ok && newName!="")
	{
		QString newDirName=dir->path()+"/"+newName;
		if(QFileInfo(newDirName).exists())
		{
			KMessageBox::error(mw->iv, QString().sprintf(i18n("The directory '<b>%s</b>' already exists"), (const char*)shrinkdn(newDirName)));
			return;	
		}
		dir->rename(newName);		
	}
}
void
DirectoryView::renameDone( KIO::Job *job)
{
	if(job->error()==0)
	{
		//pas d'erreurs
		((Directory*)clickedItem)->rename();
	}
	else
	{
		//erreur
		job->showErrorDialog(mw);
	}

}

void
DirectoryView::slotNew ()
{
	slotNew(currentItem());
}


void
DirectoryView::slotNew (QListViewItem *item)
{
	bool ok;
	const QString newName(KLineEditDlg::getText((const QString&)QString().sprintf(i18n("Create new directory in <b>%s</b>"),(const char*)shrinkdn(((Directory*)item)->fullName())),
				i18n("dirname"),
				&ok, mw).stripWhiteSpace());
	if(!ok || newName=="")
		return;
		
	KURL url("file:"+((Directory*)item)->fullName()+"/"+newName);
	if(url.fileName()!=NULL)
	{
		if(QFileInfo(url.path()).exists())
		{
		 	 KMessageBox::error(mw->iv, QString().sprintf(i18n("The directory '<b>%s</b>' already exists"), (const char*)url.fileName()));
		 	 return;
		}
		((Directory*)item)->createDir(url.fileName());
	}
}

void
DirectoryView:: slotTrash()
{
	slotTrash(currentItem());
}

void
DirectoryView::slotTrash (QListViewItem *item)
{    
	if(checkTrash())
	if(KMessageBox::questionYesNo(mw->iv,
								QString().sprintf(i18n("Do you really want to move to trash '<b>%s</b>'?"),(const char*)shrinkdn(((Directory*)item)->fullName())),
								i18n("Confirm deletion")) == KMessageBox::No)
		return;
	KURL
		org("file:/"+((Directory*)item)->fullName()),
		dest("file:/"+KGlobalSettings::trashPath()+((Directory*)item)->text(0));
	KIO::file_move(org, dest, -1, true, false, false);
	delete(item);
}


void
DirectoryView:: slotSuppr()
{
	slotSuppr(currentItem());
}

void
DirectoryView::slotSuppr (QListViewItem *item)
{    
	if(checkDelete())
	if(KMessageBox::questionYesNo(mw->iv, QString().sprintf(i18n("Do you really want to delete '<b>%s</b>'?"),(const char*)shrinkdn(((Directory*)item)->fullName())), i18n("Confirm deletion")) == KMessageBox::No)
		return;

	QDir thisDir( ((Directory*)item)->fullName());
	const QFileInfoList * files = thisDir.entryInfoList(QDir::All);
	if(files->count()>2)
	{
		KMessageBox::error(mw->iv, QString().sprintf(i18n("The directory '<b>%s</b>' is not empty"),(const char*)shrinkdn(((Directory*)item)->fullName())));
		return;
	}
	
	KIO::rmdir(KURL("file:/"+((Directory*)item)->fullName()));
	delete(item);    
}


void 
DirectoryView::slotDirCopy()
{
	Directory* item=(Directory*)currentItem();
	if(!item)
		return;
	
		QString destDir=KFileDialog::getExistingDirectory(((Directory*)item)->fullName(),
									mw,
									QString().sprintf(i18n("Copy directory %s to..."), (const char*)shrinkdn(((Directory*)item)->fullName())));
	
	if(!destDir.isEmpty())
	{
		QString dest=destDir+"/";
		
		KURL
			urlorg("file:"+((Directory*)item)->fullName()),
			urldest("file:"+dest);		
		dirOrg=((Directory*)item)->fullName();
		dirDest=dest;
		
		KIO::CopyJob *job = KIO::copy(urlorg, urldest, true);
		connect(job, SIGNAL(result( KIO::Job *)), this, SLOT(copyingDirDone( KIO::Job *)));
	}   	
}

void 
DirectoryView::slotDirMove()
{
	Directory* item=(Directory*)currentItem();
	if(!item)
		return;
	
	QString destDir=KFileDialog::getExistingDirectory(((Directory*)item)->fullName(),
								mw,
								QString().sprintf(i18n("Move directory %s to..."), (const char*)shrinkdn(((Directory*)item)->fullName())));
	
	if(!destDir.isEmpty())
	{
		QString dest=destDir+"/"+((Directory*)item)->text(0);		
		if(QFileInfo(dest).exists())
		{
		 	 KMessageBox::error(mw->iv, QString().sprintf(i18n("The directory '<b>%s</b>' already exists"),(const char*)shrinkdn(dest)));
		 	 return;
		}
		
		if(!QFileInfo(QFileInfo(dest).dirPath()).isWritable())
		{
		 	 KMessageBox::error(mw->iv,QString().sprintf( i18n("The directory '<b>%s</b>' is not writable"),(const char*)shrinkdn(dest)));
		 	 return;
		}
		
		KURL
			urlorg("file:"+((Directory*)item)->fullName()),
			urldest("file:"+dest);
			
		dirOrg=((Directory*)item)->fullName();
		dirDest=destDir+"/";
		
		KIO::FileCopyJob *job = KIO::file_move(urlorg, urldest, -1, true, false, true);
		connect(job, SIGNAL(result( KIO::Job *)), this, SLOT(movingDirDone( KIO::Job *)));
	}
}

bool
DirectoryView::checkTrash()
{
	return trash;
}

void
DirectoryView::setCheckTrash(bool trash)
{
	this->trash=trash;
}

bool
DirectoryView::checkDelete()
{
	return del;
}

void
DirectoryView::setCheckDelete(bool del)
{
	this->del=del;
}

bool
DirectoryView::checkShred()
{
	return shred;
}

void
DirectoryView::setCheckShred(bool shred)
{
	this->shred=shred;
}


void
DirectoryView::setShowHiddenDir(bool show)
{
	__showHiddenDir__=show;
}

bool
DirectoryView::showHiddenDir()
{
	return __showHiddenDir__;
}

void
DirectoryView::setShowHiddenFile(bool show)
{
	__showHiddenFile__=show;
}

bool
DirectoryView::showHiddenFile()
{
	return __showHiddenFile__;
}

QString
DirectoryView::shrinkdn(QString str)
{
	const int len=20;
	if(str.length()<=len)
		return str;
	else
		return QString(str.left(len/2) + "..." + str.right(len/2));
}
