/*
 * unix_stream_client
 * 
 * Create a client socket, connect to a specified address,
 * send a message and receive the echo.
 *
 * Permit testing of socket_secure, connect_secure, 
 * sendto_secure, getsockname_secure and getpeername_secure.
 * 
 * Calls socket_secure if '-s socket_context' is specified.
 * Calls connect_secure if '-l listener_context' is specified.
 * Calls sendto_secure if '-p peer_context' is specified.
 * Uses the abstract name space if '-a' is specified.
 * Sends a file descriptor if '-f' is specified.
 *
 * Usage:  unix_stream_client [-a] [-f] [-s socket_context] [-l listener_context] [-p peer_context] pathname
 *
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/uio.h>
#include <sys/un.h>
#include <unistd.h>
#include <getopt.h>
#include <stdio.h>

#include <socket_secure.h>
#include <ss.h>

void usage(char *progname) 
{
	printf("usage:  %s [-a] [-f] [-s socket_context] [-l listener_context] [-p peer_context] pathname\n", progname);
	exit(1);
}

int main(int argc, char **argv)
{
	struct sockaddr_un sun;
	char buf[1024], context[255];
	int s, sunlen, ret, buflen, c, contextlen, abstract = 0;
	security_id_t so_sid = 0, listener_sid = 0, peer_sid = 0, tmp_sid = 0;
	struct msghdr msg = { 0 };
	struct iovec iov;
	struct cmsghdr *cmsg;
	int myfd = 0;
	char cmsgbuf[CMSG_SPACE(sizeof myfd)];
	int *fdptr;

	while ( (c = getopt(argc, argv, "s:l:p:af")) != EOF) {
		switch (c) {
		case 's':
		case 'l':
		case 'p':
			ret = security_context_to_sid(optarg,strlen(optarg)+1,&tmp_sid);
			if (ret) {
				perror(optarg);
				exit(ret);
			}
			if (c == 's')
				so_sid = tmp_sid;
			else if (c == 'l')
				listener_sid = tmp_sid;
			else if (c == 'p')
				peer_sid = tmp_sid;
			break;
		case 'a':
			abstract = 1;
			break;
		case 'f':
			myfd = open("/dev/null", 0);
			if (myfd < 0) {
				perror("/dev/null");
				exit(1);
			}
			break;
		default:
			usage(argv[0]);
		}
	}

	if (optind != (argc - 1)) {
		usage(argv[0]);
	}

	/* Create the client socket */
	if (so_sid) 
		s = socket_secure(AF_UNIX, SOCK_STREAM, 0, so_sid);
	else
		s = socket(AF_UNIX, SOCK_STREAM, 0);
	if (s == -1) {
		perror("socket");
		exit(1);
	}

	/* Obtain the security context of the client socket */
	sun.sun_family = AF_UNIX;
	sunlen = sizeof(struct sockaddr_un);
	ret = getsockname_secure(s, &sun, &sunlen, &tmp_sid);
	if (ret) {
		perror("getsockname_secure");
		exit(1);
	}

	contextlen = sizeof(context);
	ret = security_sid_to_context(tmp_sid, context, &contextlen);
	if (ret) {
		perror("security_sid_to_context");
		exit(1);
	}
	
	printf("Created socket with context %s\n", context);

	/* Connect to the server */
	sun.sun_family = AF_UNIX;
	if (abstract) {
		sun.sun_path[0] = 0;
		strcpy(sun.sun_path+1, argv[optind++]);
		sunlen = strlen(sun.sun_path+1) + 1 + sizeof(short);
	} else {
		strcpy(sun.sun_path, argv[optind++]);
		sunlen = strlen(sun.sun_path) + 1 + sizeof(short);
	}
	if (listener_sid) 
		ret = connect_secure(s, (struct sockaddr *)&sun, sunlen, listener_sid);
	else
		ret = connect(s, (struct sockaddr *)&sun, sunlen);
	if (ret == -1) {
		perror("connect");
		exit(1);
	}

	sunlen = sizeof(struct sockaddr_un);
	ret = getpeername_secure(s, (struct sockaddr*)&sun, &sunlen, &tmp_sid);
	if (ret) {
		perror("getpeername_secure");
		exit(1);
	}

	contextlen = sizeof(context);
	ret = security_sid_to_context(tmp_sid, context, &contextlen);
	if (ret) {
		perror("security_sid_to_context");
		exit(1);
	}

	if (sun.sun_path[0] == 0) {
		sun.sun_path[0] = '@';
		sun.sun_path[sunlen - sizeof(short)] = 0;
	}

	printf("Connected to peer (%s) with context %s\n",
	       sun.sun_path, context);

	/* Send and receive echo of message */
	strcpy(buf, "hello world");
	buflen = strlen(buf)+1;

	if (myfd) {
		iov.iov_base = buf;
		iov.iov_len = buflen;
		msg.msg_name = 0;
		msg.msg_namelen = 0;
		msg.msg_iov = &iov;
		msg.msg_iovlen = 1;
		msg.msg_control = cmsgbuf;
		msg.msg_controllen = sizeof cmsgbuf;
		cmsg = CMSG_FIRSTHDR(&msg);
		cmsg->cmsg_level = SOL_SOCKET;
		cmsg->cmsg_type = SCM_RIGHTS;
		cmsg->cmsg_len = CMSG_LEN(sizeof(int));
		fdptr = (int *)CMSG_DATA(cmsg);
		memcpy(fdptr, &myfd, sizeof(int));
		msg.msg_controllen = cmsg->cmsg_len;

		if (peer_sid) 
			ret = sendmsg_secure(s, &msg, 0, peer_sid, 0);
		else
			ret = sendmsg(s, &msg, 0);
	}
	else {
		if (peer_sid) 
			ret = sendto_secure(s, buf, buflen, 0, 0, 0, peer_sid, 0);
		else
			ret = send(s, buf, buflen, 0);
	}
	if (ret != buflen) {
		perror("send");
		exit(1);
	}

	buf[0] = 0;
	ret = recv(s, buf, sizeof(buf), 0);
	if (ret == -1) {
		perror("recv");
		exit(1);
	}

	printf("Received ##%s##\n", buf);
}
