/*
 * unix_dgram_client
 * 
 * Create a client socket, send a message and receive the echo.
 *
 * Permit testing of socket_secure, connect_secure, getsockname_secure,  
 * recvfrom_secure and sendto_secure.
 * 
 * Calls socket_secure if '-s socket_context' is specified.
 * Calls connect prior to sending if '-c' is specified.
 * Calls connect_secure if '-p peer_context' and '-c' is specified.
 * Calls sendto_secure if '-p peer_context' and '-c' is not specified.
 * Calls sendto_secure if '-m message_context' is specified.
 * Uses the abstract name space if '-a' is specified.
 * Sends a file descriptor if '-f' is specified.
 *
 * Usage:  unix_dgram_client [-a] [-f] [-c] [-s socket_context] [-p peer_context] [-m message_context] my_pathname peer_pathname
 *
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/uio.h>
#include <sys/un.h>
#include <unistd.h>
#include <getopt.h>
#include <signal.h>
#include <stdio.h>

#include <socket_secure.h>
#include <ss.h>

int abstract = 0;
char my_path[1024];
#define CLEANUP_AND_EXIT do { if (!abstract) unlink(my_path); exit(1); } while (0)

void handler(int sig)
{
	printf("Received signal %d, exiting...\n", sig);
	CLEANUP_AND_EXIT;
}

void usage(char *progname) 
{
	printf("usage:  %s [-a] [-f] [-c] [-s socket_context] [-p peer_context] [-m message_context] my_pathname peer_pathname\n", progname);
	exit(1);
}

int main(int argc, char **argv)
{
	struct sockaddr_un sun;
	char buf[1024], context[255], context2[255];
	int i, s, sunlen, ret, buflen, c, contextlen, useconnect = 0;
	security_id_t so_sid = 0, peer_sid = 0, msg_sid = 0, tmp_sid = 0; 
	struct msghdr msg = { 0 };
	struct iovec iov;
	struct cmsghdr *cmsg;
	int myfd = 0;
	char cmsgbuf[CMSG_SPACE(sizeof myfd)];
	int *fdptr;


	while ( (c = getopt(argc, argv, "s:p:m:caf")) != EOF) {
		switch (c) {
		case 's':
		case 'p':
		case 'm':
			ret = security_context_to_sid(optarg,strlen(optarg)+1,&tmp_sid);
			if (ret) {
				perror(optarg);
				exit(ret);
			}
			if (c == 's')
				so_sid = tmp_sid;
			else if (c == 'p')
				peer_sid = tmp_sid;
			else if (c == 'm')
				msg_sid = tmp_sid;
			break;
		case 'c':
			useconnect = 1;
			break;
		case 'a':
			abstract = 1;
			break;
		case 'f':
			myfd = open("/dev/null", 0);
			if (myfd < 0) {
				perror("/dev/null");
				exit(1);
			}
			break;
		default:
			usage(argv[0]);
		}
	}

	if (optind != (argc - 2)) 
		usage(argv[0]);

	for (i = 0; i < 32; i++) {
		signal(i, handler);
	}

	/* Create the client socket */
	if (so_sid) 
		s = socket_secure(AF_UNIX, SOCK_DGRAM, 0, so_sid);
	else
		s = socket(AF_UNIX, SOCK_DGRAM, 0);
	if (s == -1) {
		perror("socket");
		exit(1);
	}
	/* Obtain the security context of the server socket */
	sun.sun_family = AF_UNIX;
	sunlen = sizeof(struct sockaddr_un);
	ret = getsockname_secure(s, &sun, &sunlen, &tmp_sid);
	if (ret) {
		perror("getsockname_secure");
		exit(1);
	}

	contextlen = sizeof(context);
	ret = security_sid_to_context(tmp_sid, context, &contextlen);
	if (ret) {
		perror("security_sid_to_context");
		exit(1);
	}
	
	printf("Created socket with context %s\n", context);

	/* Bind to the specified port */
	sun.sun_family = AF_UNIX;
	if (abstract) {
		sun.sun_path[0] = 0;
		strcpy(sun.sun_path+1, argv[optind++]);
		sunlen = strlen(sun.sun_path+1) + 1 + sizeof(short);
	} else {
		strcpy(sun.sun_path, argv[optind++]);
		sunlen = strlen(sun.sun_path) + 1 + sizeof(short);
		strcpy(my_path, sun.sun_path);
	}
	ret = bind(s, (struct sockaddr *)&sun, sunlen);
	if (ret == -1) {
		perror("bind");
		exit(1);
	}

	sun.sun_family = AF_UNIX;
	if (abstract) {
		sun.sun_path[0] = 0;
		strcpy(sun.sun_path+1, argv[optind++]);
		sunlen = strlen(sun.sun_path+1) + 1 + sizeof(short);
	} else {
		strcpy(sun.sun_path, argv[optind++]);
		sunlen = strlen(sun.sun_path) + 1 + sizeof(short);
	}

	if (useconnect) {
		/* Set the peer */
		if (peer_sid) 
			/* Specify the peer SID */
			ret = connect_secure(s, (struct sockaddr *)&sun, sunlen, peer_sid);
		else
			ret = connect(s, (struct sockaddr *)&sun, sunlen);
	}

	/* Set up the message */
	strcpy(buf, "hello world");
	buflen = strlen(buf)+1;

	/* Send the message */
	if (myfd) {
		iov.iov_base = buf;
		iov.iov_len = buflen;
		if (useconnect) {
			msg.msg_name = 0;
			msg.msg_namelen = 0;
		} else {
			msg.msg_name = (void*)&sun;
			msg.msg_namelen = sunlen;
		}
		msg.msg_iov = &iov;
		msg.msg_iovlen = 1;
		msg.msg_control = cmsgbuf;
		msg.msg_controllen = sizeof cmsgbuf;
		cmsg = CMSG_FIRSTHDR(&msg);
		cmsg->cmsg_level = SOL_SOCKET;
		cmsg->cmsg_type = SCM_RIGHTS;
		cmsg->cmsg_len = CMSG_LEN(sizeof(int));
		fdptr = (int *)CMSG_DATA(cmsg);
		memcpy(fdptr, &myfd, sizeof(int));
		msg.msg_controllen = cmsg->cmsg_len;

		if (peer_sid || msg_sid) 
			ret = sendmsg_secure(s, &msg, 0, peer_sid, msg_sid);
		else
			ret = sendmsg(s, &msg, 0);
	} else if (useconnect) {
		if (msg_sid)
			/* Specify the message SID */
			ret = sendto_secure(s, buf, buflen, 0, 0, 0, 0,
					    msg_sid);
		else
			ret = send(s, buf, buflen, 0);
	} else {
		if (peer_sid || msg_sid)
			/* Specify the peer SID and/or message SID */
			ret = sendto_secure(s, buf, buflen, 0, 
					    (struct sockaddr*)&sun, sunlen, 
					    peer_sid, msg_sid);
		else
			ret = sendto(s, buf, buflen, 0, 
				     (struct sockaddr *)&sun, sunlen);
	}
	if (ret != buflen) {
		perror("sendto");
		CLEANUP_AND_EXIT;
	}

	/* Receive the echo from the server and obtain the peer socket context and message context */
	buf[0] = 0;
	sunlen = sizeof(struct sockaddr_un);
	ret = recvfrom_secure(s, buf, sizeof(buf), 0, 
			      (struct sockaddr*)&sun, &sunlen,
			      &peer_sid, &msg_sid);
	if (ret == -1) {
		perror("recvfrom");
		CLEANUP_AND_EXIT;
	}

	contextlen = sizeof(context);
	ret = security_sid_to_context(peer_sid, context, &contextlen);
	if (ret) {
		perror("security_sid_to_context");
		CLEANUP_AND_EXIT;
	}

	contextlen = sizeof(context2);
	ret = security_sid_to_context(msg_sid, context2, &contextlen);
	if (ret) {
		perror("security_sid_to_context");
		CLEANUP_AND_EXIT;
	}

	if (sun.sun_path[0] == 0) {
		sun.sun_path[0] = '@';
		sun.sun_path[sunlen - sizeof(short)] = 0;
	}

	printf("Received %s from peer (%s) with socket context %s, message context %s\n", 
	       buf, sun.sun_path, context, context2);

	CLEANUP_AND_EXIT;
}
