;*=====================================================================*/
;*    serrano/prgm/project/scribe/scribeinfo/parser.scm                */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Thu Oct 11 09:30:45 2001                          */
;*    Last change :  Wed Jan  9 16:26:09 2002 (serrano)                */
;*    Copyright   :  2001-02 Manuel Serrano                            */
;*    -------------------------------------------------------------    */
;*    The parser for texinfo format.                                   */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(module __scribeinfo_parser
   (import __scribeinfo_lexer
	   __scribeinfo_engine
	   __scribeinfo_param)
   (export (parse-file ::bstring)
	   (parse-string ::bstring . coord)
	   (parse-port ::input-port)))

;*---------------------------------------------------------------------*/
;*    parse-file ...                                                   */
;*---------------------------------------------------------------------*/
(define (parse-file file)
   (let ((p (open-input-file file)))
      (if (not (input-port? p))
	  (error "parse-file" "Can't open file for input" file)
	  (unwind-protect (parse-port p)
			  (close-input-port p)))))
 
;*---------------------------------------------------------------------*/
;*    parse-string ...                                                 */
;*---------------------------------------------------------------------*/
(define (parse-string string . coord)
   (with-input-from-string string
      (lambda ()
	 (if (pair? coord)
	     (try (read/lalrp parser lexer (current-input-port))
		  (lambda (escape proc mes obj)
		     (match-case obj
			((?token (?fname . ?pos) . ?-)
			 (error/location proc "parse error" token
					 (caar coord) (+fx (cdar coord) pos)))
			(else
			 (error/location proc mes obj
					 (caar coord) (cdar coord))))))
	     (read/lalrp parser lexer (current-input-port))))))
   
;*---------------------------------------------------------------------*/
;*    parse-port ...                                                   */
;*---------------------------------------------------------------------*/
(define (parse-port port)
   (try (read/lalrp parser lexer port)
	(lambda (escape proc mes obj)
	   (match-case obj
	      ((?token (?fname . ?pos) . ?-)
	       (error/location proc "parse error" token fname pos))
	      (else
	       (notify-error proc mes obj)
	       (error proc mes obj))))))

;*---------------------------------------------------------------------*/
;*    *node-mark* ...                                                  */
;*---------------------------------------------------------------------*/
(define *node-mark* #f)

;*---------------------------------------------------------------------*/
;*    parser ...                                                       */
;*---------------------------------------------------------------------*/
(define parser
   
   (lalr-grammar
      
      ;; tokens
      (STRING
       LINEBREAK
       BRACLO
       BRAOPEN
       NODE
       B I R W T EMPH DFN SAMP SC STRONG
       KBD KEY
       CODE VAR
       REF XREF PXREF INFOREF
       COPYRIGHT EXPANSION EQUIV ERROR RESULT PRINT DOTS FILE FOOTNOTE MATH TEX
       EMAIL URL
       IGNORE |END IGNORE|
       TABLE |TABLE CODE| |TABLE BULLET| |TABLE EMPH| |TABLE ASIS| 
       |TABLE T| |TABLE R| |TABLE MINUS|
       |END TABLE|
       ITEMIZE |ITEMIZE BULLET| |ITEMIZE MINUS| |END ITEMIZE|
       ENUMERATE |END ENUMERATE|
       ITEM
       ITEMX
       IFSET |END IFSET|
       MENU |END MENU|
       SETFILENAME SETTITLE
       PARAGRAPHINDENT
       FOOTNOTESTYLE
       FINALOUT
       SET VALUE
       INCLUDE
       SETCHAPTERNEWPAGE
       SHORTTITLEPAGE
       TITLEPAGE |END TITLEPAGE|
       TITLE SUBTITLE
       ATITLE ASUBTITLE
       AUTHOR AAUTHOR
       PAGE
       VSKIP
       DIRCATEGORY
       DIRENTRY |END DIRENTRY|
       SMALLLISP |END SMALLLISP|
       LISP |END LISP|
       FLUSHLEFT |END FLUSHLEFT|
       FLUSHRIGHT |END FLUSHRIGHT|
       SMALLEXAMPLE |END SMALLEXAMPLE|
       EXAMPLE |END EXAMPLE|
       FORMAT |END FORMAT|
       DISPLAY |END DISPLAY|
       QUOTATION |END QUOTATION|
       CHAPTER SECTION SUBSECTION SUBSUBSECTION
       UNNUMBERED UNNUMBEREDSEC UNNUMBEREDSUBSEC UNNUMBEREDSUBSUBSEC
       CINDEX VINDEX PINDEX PRINTINDEX SYNCODEINDEX SYNINDEX
       REFILL NOINDENT
       CONTENTS SUMMARYCONTENTS
       BYE
       DEFN |END DEFN|
       DEFFN |END DEFFN|
       DEFFNX
       DEFTP |END DEFTP|
       DEFMETHOD |END DEFMETHOD|
       SP
       CENTER)
      
      ;; commands
      (commands
       (() '())
       ((command commands) (cons command commands)))
      
      ;; command
      (command
       ((ignore) '())
       ((text) text)
       ((argument-command) argument-command)
       ((line-command) line-command)
       ((BRAOPEN commands BRACLO) `(list ,@commands)))
      
      ;; ignore
      (ignore
       ((MENU commands |END MENU|) '())
       ((SETFILENAME) '())
       ((FOOTNOTESTYLE) '())
       ((PARAGRAPHINDENT) '())
       ((FINALOUT) '())
       ((SETCHAPTERNEWPAGE) '())
       ((SHORTTITLEPAGE) '())
       ((PAGE) '())
       ((DIRCATEGORY) '())
       ((DIRENTRY commands |END DIRENTRY|) '())
       ((VSKIP) '())
       ((REFILL) '())
       ((CONTENTS) '())
       ((BYE) '()))
      
      ;; text
      (text
       ((STRING)
	(cadr STRING))
       ((LINEBREAK)
	`(linebreak ,(cadr LINEBREAK))))
      
      ;; argument-command
      (argument-command
       ((acommand BRACLO)
	acommand))
      
      (acommand
       ((VALUE STRING)
	`(value ,(cadr STRING)))
       ((COPYRIGHT)
	'(copyright))
       ((TEX)
	"TeX")
       ((DOTS)
	"...")
       ((B commands)
	`(bold ,@commands))
       ((ATITLE commands)
	`(title ,@commands))
       ((ASUBTITLE commands)
	`(subtitle ,@commands))
       ((AAUTHOR commands)
	`(author ,@commands))
       ((I commands)
	`(it ,@commands))
       ((R commands)
	`(list ,@commands))
       ((EMAIL commands)
	(mailto commands))
       ((URL commands)
	`(ref :url (list ,@commands)))
       ((W commands)
	`(bold ,@commands))
       ((T commands)
	`(tt ,@commands))
       ((CODE commands)
	`(code ,@commands))
       ((VAR commands)
	`(code ,@commands))
       ((EMPH commands)
	`(emph ,@commands))
       ((DFN text)
	(string-append "\"" text "\""))
       ((SAMP commands)
	`(samp ,@commands))
       ((SC commands)
	`(sc ,@commands))
       ((STRONG commands)
	`(bold ,@commands))
       ((KBD commands)
	`(kbd ,@commands))
       ((KEY commands)
	`(kbd ,@commands))
       ((REF commands)
	(ref '@ref REF commands))
       ((XREF commands)
	`(list "See " ,(ref '@xref XREF commands)))
       ((INFOREF commands)
	(inforef commands))
       ((PXREF commands)
	`(list "see " ,(ref '@pxref PXREF commands)))
       ((EXPANSION)
	"==>")
       ((EQUIV)
	"<=>")
       ((ERROR)
	"error-->")
       ((FOOTNOTE commands)
	`(footnote :note ,@commands))
       ((MATH commands)
	(math commands))
       ((RESULT)
	"=>")
       ((RESULT command)
	`(list "=>" ,command))
       ((PRINT)
	"-|")
       ((FILE text)
	(string-append "`" text "'")))
      
      ;; line-command
      (line-command
       ((NODE)
	(set! *node-mark*
	      `(mark ,(string-skip (nthword (car (cadr NODE)) 0 #\,)
				    #\space #\tab))))
       ((INCLUDE)
	`(list ,@(parse-file (car (cadr INCLUDE)))))
       ((itemize)
	itemize)
       ((table)
	table)
       ((ITEM)
	(if (pair? (cadr ITEM))
	    `(item ,@(cadr ITEM))
	    `(item)))
       ((ITEMX)
	`(itemx ,@(cadr ITEMX)))
       ((SET)
	(parse-set (cadr SET)))
       ((CHAPTER)
	(if *node-mark*
	    (let ((lbl *node-mark*))
	       (set! *node-mark* #f)
	       `(chapter :subtitle ,@(cadr CHAPTER) ,lbl))
	    `(chapter :subtitle ,@(cadr CHAPTER))))
       ((SECTION)
	(if *node-mark*
	    (let ((lbl *node-mark*))
	       (set! *node-mark* #f)
	       `(section :title (list ,@(cadr SECTION)) ,lbl))
	    `(section :title (list ,@(cadr SECTION)))))
       ((SUBSECTION)
	(if *node-mark*
	    (let ((lbl *node-mark*))
	       (set! *node-mark* #f)
	       `(subsection :title ,@(cadr SUBSECTION) ,lbl))
	    `(subsection :title ,@(cadr SUBSECTION))))
       ((SUBSUBSECTION)
	(if *node-mark*
	    (let ((lbl *node-mark*))
	       (set! *node-mark* #f)
	       `(subsubsection :title ,@(cadr SUBSUBSECTION) ,lbl))
	    `(subsubsection :title ,@(cadr SUBSUBSECTION))))
       ((UNNUMBERED)
	(if *node-mark*
	    (let ((lbl *node-mark*))
	       (set! *node-mark* #f)
	       `(chapter :number #f :toc #t :file #f :title ,@(cadr UNNUMBERED) ,lbl))
	    `(chapter :number #f :toc #t :file #f :title ,@(cadr UNNUMBERED))))
       ((UNNUMBEREDSEC)
	(if *node-mark*
	    (let ((lbl *node-mark*))
	       (set! *node-mark* #f)
	       `(section :number #f :title ,@(cadr UNNUMBEREDSEC) ,lbl))
	    `(section :number #f :title ,@(cadr UNNUMBEREDSEC))))
       ((UNNUMBEREDSUBSEC)
	(if *node-mark*
	    (let ((lbl *node-mark*))
	       (set! *node-mark* #f)
	       `(subsection :number #f :title ,@(cadr UNNUMBEREDSUBSEC) ,lbl))
	    `(subsection :number #f :title ,@(cadr UNNUMBEREDSUBSEC))))
       ((UNNUMBEREDSUBSUBSEC)
	(if *node-mark*
	    (let ((lbl *node-mark*))
	       (set! *node-mark* #f)
	       `(subsubsection :number #f :title ,@(cadr UNNUMBEREDSUBSUBSEC) ,lbl))
	    `(subsubsection :number #f :title ,@(cadr UNNUMBEREDSUBSUBSEC))))
       ((SETTITLE)
	`(header ,@(cadr SETTITLE)))
       ((TITLEPAGE commands |END TITLEPAGE|)
	`(title ,@commands))
       ((TITLE)
	`(title ,@(cadr TITLE)))
       ((SUBTITLE) 
	`(subtitle ,@(cadr SUBTITLE)))
       ((AUTHOR)
	`(author ,@(cadr AUTHOR)))
       ((IGNORE commands |END IGNORE|)
	(marks-only commands))
       ((IFSET commands |END IFSET|)
	(if (variable-bound? (car (cadr IFSET)))
	    `(list ,@commands)
	    (marks-only commands)))
       ((SMALLLISP commands |END SMALLLISP|)
	`(prgm :language bigloo :bg *prgm-color* ,@commands))
       ((LISP commands |END LISP|)
	`(prgm :language bigloo :bg *prgm-color* ,@commands))
       ((SMALLEXAMPLE commands |END SMALLEXAMPLE|)
	`(prgm :bg *example-color* ,@commands))
       ((EXAMPLE commands |END EXAMPLE|)
	`(prgm :bg *example-color* ,@commands))
       ((FORMAT commands |END FORMAT|)
	`(prgm :bg *example-color* ,@commands))
       ((DISPLAY commands |END DISPLAY|)
	`(prgm :bg *display-color* ,@commands))
       ((QUOTATION commands |END QUOTATION|)
	`(font :size "=-1" ,@commands))
       ((DEFN commands |END DEFN|)
	(deffn (cadr DEFN) commands))
       ((DEFFN commands |END DEFFN|)
	(deffn (cadr DEFFN) commands))
       ((DEFFNX)
	`(deffnx ,@(cadr DEFFNX)))
       ((DEFTP commands |END DEFTP|)
	(deffn (cadr DEFTP) commands))
       ((DEFMETHOD commands |END DEFMETHOD|)
	(defmethod (cadr DEFMETHOD) commands))
       ((SYNCODEINDEX)
	(let ((w1 (nthword (car (cadr SYNCODEINDEX)) 0))
	      (w2 (nthword (car (cadr SYNCODEINDEX)) 1)))
	   (syncodeindex (string->symbol w1) (string->symbol w2))))
       ((SYNINDEX)
	(let ((w1 (nthword (car (cadr SYNINDEX)) 0))
	      (w2 (nthword (car (cadr SYNINDEX)) 1)))
	   (synindex (string->symbol w1) (string->symbol w2))))
       ((FLUSHLEFT commands |END FLUSHLEFT|)
	`(flush :side 'left ,@commands))
       ((FLUSHRIGHT commands |END FLUSHRIGHT|)
	`(flush :side 'left ,@commands))
       ((CINDEX)
	`(mark ,(index 'cp (list `(it ,@(cadr CINDEX))))))
       ((PINDEX)
	`(mark ,(index 'pg (list `(it ,@(cadr PINDEX))))))
       ((VINDEX) 
	`(mark ,(index 'vr (list `(it ,@(cadr VINDEX))))))
       ((PRINTINDEX)
	(printindex (car (cadr PRINTINDEX))))
       ((SUMMARYCONTENTS)
	`(table-of-contents))
       ((NOINDENT)
	(if (pair? (cadr NOINDENT))
	    `(list ,@(cadr NOINDENT))
	    '()))
       ((SP)
	`(linebreak ,@(cadr SP)))
       ((CENTER)
	(if (pair? (cadr CENTER))
	    `(center ,@(cadr CENTER))
	    '())))
      
      ;; itemize
      (itemize
       ((ITEMIZE commands |END ITEMIZE|)
	(itemize 'itemize commands))
       ((|ITEMIZE BULLET| commands |END ITEMIZE|)
	(itemize 'itemize commands))
       ((|ITEMIZE MINUS| commands |END ITEMIZE|)
	(itemize 'itemize commands))
       ((ENUMERATE commands |END ENUMERATE|)
	(itemize 'enumerate commands)))
      
      ;; table
      (table
       ((TABLE commands |END TABLE|)
	(table commands (lambda (x) x)))
       ((|TABLE CODE| commands |END TABLE|)
	(table commands (lambda (x) `(code ,x))))
       ((|TABLE BULLET| commands |END TABLE|)
	(table commands (lambda (x) x)))
       ((|TABLE EMPH| commands |END TABLE|) 
	(table commands (lambda (x) `(emph ,x))))
       ((|TABLE ASIS| commands |END TABLE|) 
	(table commands (lambda (x) x)))
       ((|TABLE T| commands |END TABLE|) 
	(table commands (lambda (x) `(tt ,x))))
       ((|TABLE R| commands |END TABLE|) 
	(table commands (lambda (x) x)))
       ((|TABLE MINUS| commands |END TABLE|) 
	(table commands (lambda (x) x))))))

;*---------------------------------------------------------------------*/
;*    stringify ...                                                    */
;*---------------------------------------------------------------------*/
(define (stringify o)
   (cond
      ((string? o)
       o)
      ((pair? o)
       (match-case o
	  ((value . ?-)
	   o)
	  (else
	   (if (null? (cdr o))
	       (stringify (car o))
	       (apply string-append (map stringify o))))))
      (else
       (error "stringify" "Illegal value" o))))

;*---------------------------------------------------------------------*/
;*    marks-only ...                                                   */
;*---------------------------------------------------------------------*/
(define (marks-only lst)
   (define (find-marks expr res)
      (if (not (pair? expr))
	  res
	  (match-case expr
	     ((mark . ?-)
	      (cons expr res))
	     (else
	      (find-marks (car expr) (find-marks (cdr expr) res))))))
   (let loop ((lst lst)
	      (res '()))
      (if (null? lst)
	  (if (pair? res)
	      `(list ,@res)
	      '())
	  (loop (cdr lst)
		(find-marks (car lst) res)))))

;*---------------------------------------------------------------------*/
;*    parse-set ...                                                    */
;*---------------------------------------------------------------------*/
(define (parse-set token)
   (let* ((str (car token))
	  (var (nthword str 0))
	  (val (if (<fx (string-length var) (-fx (string-length str) 1))
		   (let ((val0 (substring str
					  (+fx (string-length var) 1)
					  (string-length str))))
		      (cons val0 (cdr token)))
		   (cdr token))))
      (bind-variable var val)
      '()))

;*---------------------------------------------------------------------*/
;*    mailto ...                                                       */
;*---------------------------------------------------------------------*/
(define (mailto commands)
   (let ((email (apply string-append (map (lambda (x)
					     (if (string? x)
						 x
						 ""))
					  commands))))
      (if (>fx (string-length email) 0)
	  `(mailto :email ,email ,@commands)
	  `(mailto ,@commands))))

;*---------------------------------------------------------------------*/
;*    table ...                                                        */
;*---------------------------------------------------------------------*/
(define (table commands deco)
   (define (table-lignes commands)
      (define (table-first-column commands lines)
	 (let loop ((commands commands)
		    (items '()))
	    (if (null? commands)
		(reverse! lines)
		(match-case (car commands)
		   ((item . ?item)
		    (if (not (null? items))
			;; this should be an error but it turns out that
			;; texinfo is laxist
			(loop (cdr commands) (append (map deco item) items))
			(loop (cdr commands) (map deco item))))
		   ((itemx . ?item)
		    (if (null? items)
			(error "table" "Illegal itemx" (car commands))
			(loop (cdr commands) (append (map deco item) items))))
		   (else
		    (table-second-column commands (reverse! items) lines))))))
      (define (table-second-column commands dd lines)
	 (let loop ((commands commands)
		    (vals '()))
	    (if (null? commands)
		(table-first-column '()
				    (cons `(item :key (list ,@dd)
						 ,@(reverse! vals))
					  lines))
		(match-case (car commands)
		   ((item . ?item)
		    (table-first-column commands
					(cons `(item :key (list ,@dd)
						     ,@(reverse! vals))
					      lines)))
		   (else
		    (loop (cdr commands) (cons (car commands) vals)))))))
      (table-first-column commands '()))
   `(description ,@(table-lignes commands)))
     
;*---------------------------------------------------------------------*/
;*    itemize ...                                                      */
;*---------------------------------------------------------------------*/
(define (itemize markup commands)
   (define (itemize-lignes commands)
      (define (itemize-item commands lines)
	 (let loop ((commands commands)
		    (items '()))
	    (if (null? commands)
		(reverse! lines)
		(match-case (car commands)
		   ((item . ?item)
		    (itemize-entry (cdr commands) (reverse! item) lines))
		   (else
		    (loop (cdr commands) items))))))
      (define (itemize-entry commands vals lines)
	 (let loop ((commands commands)
		    (vals vals))
	    (if (null? commands)
		(itemize-item '() (cons `(item ,@(reverse! vals)) lines))
		(match-case (car commands)
		   ((item . ?item)
		    (itemize-item commands
				  (cons `(item ,@(reverse! vals)) lines)))
		   ((itemx . ?item)
		    (error "itemize" "Illegal itemx" (car commands)))
		   (else
		    (loop (cdr commands) (cons (car commands) vals)))))))
      (itemize-item commands '()))
   `(,markup ,@(itemize-lignes commands)))

;*---------------------------------------------------------------------*/
;*    *deffn-marks* ...                                                */
;*---------------------------------------------------------------------*/
(define *deffn-marks* '())

;*---------------------------------------------------------------------*/
;*    deffn ...                                                        */
;*---------------------------------------------------------------------*/
(define (deffn def commands)
   (define (make-kind name rest kind)
      (if (string? name)
	  (if (and *auto-deffn-index* (not (member name *deffn-marks*)))
	      (begin
		 (set! *deffn-marks* (cons name *deffn-marks*))
		 `(list (mark ,(index 'pg (list `(code ,name))))
			(mark ,name)
			(table :width 1.0
			       (tr (td (bold ,name) (it ,@rest))
				   (td :align 'right ,@kind)))))
	      `(list (table :width 1.0
			    (tr (td (bold ,name) (it ,@rest))
				(td :align 'right ,@kind)))))
	  `(list (table :width 1.0
			(tr (td (it ,@rest))
			    (td :align 'right ,@kind))))))
   (define (make-deffn-title title)
      (match-case title
	 (((list . ?kind) (and ?def (? string?)))
	  (let* ((name (nthword def 0))
		 (rest (substring def
				  (+fx 1 (string-length name))
				  (string-length def))))
	     (make-kind name (list rest) kind)))
	 (((list . ?kind) (and ?def (? string?)) . ?rests)
	  (let ((name (nthword def 0)))
	     (if (not name)
		 (make-kind #f rests kind)
		 (let ((rest (substring def
					(+fx 1 (string-length name))
					(string-length def))))
		    (make-kind name (cons rest rests) kind)))))
	 (((list . ?kind) . ?rest)
	  (make-kind #f rest kind))
	 (else
	  title)))
   `(list (frame :width 1.
		 :margin 10
		 ,(make-deffn-title def)
		 ,@(map (lambda (x)
			   (match-case x
			      ((deffnx . ?rest)
			       (make-deffn-title rest))
			      (else
			       x)))
			commands))
	  (linebreak 1)))

;*---------------------------------------------------------------------*/
;*    defmethod ...                                                    */
;*---------------------------------------------------------------------*/
(define (defmethod def commands)
   (define (make-method rest)
      `(list (table :width 1.0
		    (tr (td (it ,rest))
			(td :align 'right "method")))))
   (define (make-defmethod-title title)
      (make-method (stringify title)))
   `(list (frame :width 1.
		 :margin 10
		 ,(make-defmethod-title def)
		 ,@commands)
	  (linebreak 1)))

;*---------------------------------------------------------------------*/
;*    ref ...                                                          */
;*---------------------------------------------------------------------*/
(define (ref cmd REF commands)
   (let* ((str (apply string-append commands))
	  (unspace (lambda (s)
			  (if (string? s)
			      (string-skip s #\space)
			      s)))
	  (w0 (unspace (nthword2 str 0 #\,)))
	  (w1 (unspace (nthword2 str 1 #\,)))
	  (w2 (unspace (nthword2 str 2 #\,)))
	  (w3 (nthword2 str 3 #\,)))
      (cond
	 ((not (string? w0))
	  (error "@ref" "Illegal reference" `(,cmd ,(car REF))))
	 ((not (or w1 w2 w3))
	  ;; plain reference to a local node
	  `(ref :id ,(string-replace w0 #\Newline #\Space)))
	 ((and (string? w2) (assoc w2 *info-sui*))
	  ;; a reference to another Scribe document
	  (let ((cell (cdr (assoc w2 *info-sui*)))
		(body (cond
			 ((or (pair? w1)
			      (and (string? w1) (>fx (string-length w1) 0)))
			  w1)
			 ((and (string? w0) (>fx (string-length w0) 0))
			  (string-append (prefix w2) ", " w0))
			 ((pair? w0)
			  (cons (string-append (prefix w2) ", ") w0))
			 (else
			  (prefix w2)))))
	     (if (and (string? w0) (>fx (string-length w0) 0))
		 `(ref :scribe ,cell :id ,w0 ,body)
		 `(ref :scribe ,cell ,body))))
	 ((string? w3)
	  ;; a pure tex reference
	  (let* ((base (string-append "info-file `" w3 "'"))
 		 (ebase (string-append base ", " (if w2 w2 w0))))
	     (string-replace ebase #\Newline #\Space)))
	 (else
	  ;; a reference with a customized text
	  (let* ((user-title1 (if w2 w2 w0))
		 (user-title2 (if (and (string? w1)
				       (>fx (string-length w1) 0))
				  (string-append "(" w1 "): " user-title1)
				  user-title1)))
	     `(ref :id
		   ,(string-replace w0 #\Newline #\Space)
		   ,(string-replace user-title2 #\Newline #\Space)))))))

;*---------------------------------------------------------------------*/
;*    inforef ...                                                      */
;*---------------------------------------------------------------------*/
(define (inforef commands)
   (let* ((str (apply string-append commands))
	  (w0 (nthword str 0 #\,))
	  (w1 (nthword str 1 #\,))
	  (w2 (nthword str 2 #\,)))
      (cond
	 ((not (string? w0))
	  (error "@ref" "Illegal reference" 'inforef))
	 ((not (or w1 w2))
	  ;; plain reference to a local node
	  `(ref :id ,(string-replace w0 #\Newline #\Space)))
	 ((string? w2)
	  ;; a pure tex reference
	  (let* ((base (string-append "info-file `" w2 "'"))
		 (ebase (string-append base ", " (if w2 w2 w0))))
	     (string-replace ebase #\Newline #\Space)))
	 (else
	  ;; a reference with a customized text
	  (let* ((user-title1 (if w2 w2 w0))
		 (user-title2 (if (and (string? w1)
				       (>fx (string-length (string-skip w1 #\Space)) 0))
				  (string-append "(" w1 "): " user-title1)
				  user-title1)))
	     `(ref :id
		   ,(string-replace w0 #\Newline #\Space)
		   ,(string-replace user-title2 #\Newline #\Space)))))))

;*---------------------------------------------------------------------*/
;*    printindex ...                                                   */
;*---------------------------------------------------------------------*/
(define (printindex kind)
   `(list ,@(make-index (string->symbol kind))))

;*---------------------------------------------------------------------*/
;*    math ...                                                         */
;*    -------------------------------------------------------------    */
;*    The math handling is extremelly weak. We only take care to       */
;*    _ and ^ that are replaced with (sub ...) and (sup ...).          */
;*---------------------------------------------------------------------*/
(define (math cmds)
   (let* ((str (apply string-append cmds))
	  (len (string-length str))
	  (len-1 (-fx len 1)))
      (let loop ((s 0)
		 (r 0)
		 (acc '()))
	 (if (>=fx r len-1)
	     (let ((l (reverse!
		       (if (=fx s len) acc (cons (substring str s len) acc)))))
		(cond
		   ((null? l)
		    "")
		   ((null? (cdr l))
		    (car l))
		   (else
		    `(list ,@l))))
	     (let ((c (string-ref str r)))
		(case c
		   ((#\_ #\^)
		    (let ((sub `(,(if (char=? c #\_) 'sub 'sup)
				 ,(string (string-ref str (+fx r 1))))))
		       (if (=fx s r)
			   (loop (+fx r 2)
				 (+fx r 2)
				 (cons sub acc))
			   (loop (+fx r 2)
				 (+fx r 2)
				 (cons* sub (substring str s r) acc)))))
		   (else
		    (loop s (+fx r 1) acc))))))))
      
