;; Scribe mode, and its idiosyncratic commands.
;; Copyright (C) 1986, 1987, 1988, 1990 Free Software Foundation, Inc.
;; Adapted from Lisp mode by Bill Rozas, jinx@prep.
 
;; This file is part of GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 1, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.


;; Initially a query replace of Lisp mode, except for the indentation 
;; of special forms.  Probably the code should be merged at some point 
;; so that there is sharing between both libraries.

;;; $Header: scribe.el,v 1.7 88/07/15 20:20:00 GMT cph Exp $

(provide 'scribe)
(require 'font-lock)

(defvar scribe-font-lock-keywords
  (list
   (list (concat "\(\\(let\\|let[*]\\|letrec\\|define"
		 "\\|define-markup\\|set[!]"
		 "\\|lambda\\|labels"
		 "\\|let-syntax\\|letrec-syntax"
		 "\\|regular-grammar\\|lalr-grammar"
		 "\\|if\\|begin\\|case\\|cond\\|else"
		 "\\|multiple-value-bind\\|values\\)[ :\n\t]")
	 1
	 'font-lock-keyword-face)
   (list "(\\(section\\|subsection\\|subsubsection\\|paragraph\\|table-of-contents\\|itemize\\|enumerate\\|description\\)[) \n]"
	 1
	 'font-lock-function-name-face)
   (list "(\\(document\\|chapter\\|scribe-url-index\\)[ \n]"
	 1
	 'font-lock-face-2)
   (list "(\\(column\\|include\\|margin\\|style\\|bibliography\\|table\\|tr\\|th\\|td\\|linebreak\\|bold\\|emph\\|it\\|tt\\|underline\\|item\\)[ )]"
	 1
	 'font-lock-face-5)
   (cons "\\(:[^ \n)]+\\|#![a-zA-Z]+\\)"
	 'font-lock-face-7)
   (cons "[[]\\|]"
	 'font-lock-face-3)
   (list ",(\\([^ \n()]+\\)"
	 1
	 'font-lock-face-6)))

(defvar normal-indent 0)

(defun scribe-font-lock ()
  (setq font-lock-keywords scribe-font-lock-keywords))

(defvar scribe-mode-syntax-table nil "")
(if (not scribe-mode-syntax-table)
    (let ((i 0))
      (setq scribe-mode-syntax-table (make-syntax-table))
      (set-syntax-table scribe-mode-syntax-table)

      ;; Default is atom-constituent.
      (while (< i 256)
        (modify-syntax-entry i "_   ")
        (setq i (1+ i)))

      ;; Word components.
      (setq i ?0)
      (while (<= i ?9)
        (modify-syntax-entry i "w   ")
        (setq i (1+ i)))
      (setq i ?A)
      (while (<= i ?Z)
        (modify-syntax-entry i "w   ")
        (setq i (1+ i)))
      (setq i ?a)
      (while (<= i ?z)
        (modify-syntax-entry i "w   ")
        (setq i (1+ i)))

      ;; Whitespace
      (modify-syntax-entry ?\t "    ")
      (modify-syntax-entry ?\n ">   ")
      (modify-syntax-entry ?\f "    ")
      (modify-syntax-entry ?\r "    ")
      (modify-syntax-entry ?  "    ")

      ;; These characters are delimiters but otherwise undefined.
      ;; Brackets and braces balance for editing convenience.
      (modify-syntax-entry ?[ "(]  ")
      (modify-syntax-entry ?] ")[  ")
      (modify-syntax-entry ?{ "(}  ")
      (modify-syntax-entry ?} "){  ")
      (modify-syntax-entry ?\| "    ")

      ;; Other atom delimiters
      (modify-syntax-entry ?\( "()  ")
      (modify-syntax-entry ?\) ")(  ")
      (modify-syntax-entry ?\; "<   ")
      (modify-syntax-entry ?\" "\"    ")
      (modify-syntax-entry ?' "'   ")
      (modify-syntax-entry ?` "'   ")

      ;; Special characters
      (modify-syntax-entry ?, "'   ")
      (modify-syntax-entry ?@ "'   ")
      (modify-syntax-entry ?# "'   ")
      (modify-syntax-entry ?\\ "\\   ")))


(defvar scribe-mode-abbrev-table nil "")
(define-abbrev-table 'scribe-mode-abbrev-table ())

(defun scribe-mode-variables ()
  (set-syntax-table scribe-mode-syntax-table)
  (setq local-abbrev-table scribe-mode-abbrev-table)
  (make-local-variable 'paragraph-start)
  (setq paragraph-start (concat "^$\\|" page-delimiter))
  (make-local-variable 'paragraph-separate)
  (setq paragraph-separate paragraph-start)
  (make-local-variable 'paragraph-ignore-fill-prefix)
  (setq paragraph-ignore-fill-prefix t)
  (make-local-variable 'indent-line-function)
  (setq indent-line-function 'scribe-indent-line)
  (make-local-variable 'comment-start)
  (setq comment-start ";")
  (make-local-variable 'comment-start-skip)
  (setq comment-start-skip ";+[ \t]*")
  (make-local-variable 'comment-column)
  (setq comment-column 40)
  (make-local-variable 'comment-indent-function)
  (setq comment-indent-function 'scribe-comment-indent)
  (setq mode-line-process '("" scribe-mode-line-process)))

(defvar scribe-mode-line-process "")

(defun scribe-return (&optional dummy)
  "Automatic indentation on RET."
  (interactive)
  (if (= (point) 1)
      (newline)
    (newline-and-indent)))

(defun scribe-mode-commands (map)
  (define-key map "\t" 'scribe-indent-line)
  (define-key map "\177" 'backward-delete-char-untabify)
  (define-key map "\e\C-q" 'scribe-indent-sexp)
  (define-key map "\C-m" 'scribe-return)
  (define-key map "\e\C-m" 'newline))
  

(defvar scribe-mode-map nil)
(if (not scribe-mode-map)
    (progn
      (setq scribe-mode-map (make-sparse-keymap))
      (scribe-mode-commands scribe-mode-map)))


(defun scribe-mode ()
  "Major mode for editing Scribe code.
Editing commands are similar to those of lisp-mode.

In addition, if an inferior Scribe process is running, some additional
commands will be defined, for evaluating expressions and controlling
the interpreter, and the state of the process will be displayed in the
modeline of all Scribe buffers.  The names of commands that interact
with the Scribe process start with \"xscribe-\".  For more information
see the documentation for xscribe-interaction-mode. 

Commands:
Delete converts tabs to spaces as it moves back.
Blank lines separate paragraphs.  Semicolons start comments.
\\{scribe-mode-map}
Entry to this mode calls the value of scribe-mode-hook
if that value is non-nil."
  (interactive)
  (kill-all-local-variables)
  (scribe-mode-initialize)
  (scribe-mode-variables)
  (add-hook 'font-lock-mode-hook 'scribe-font-lock)
  (scribe-font-lock)
  (turn-on-font-lock)
  (run-hooks 'scribe-mode-hook))

(defun scribe-mode-initialize ()
  (use-local-map scribe-mode-map)
  (put 'scribe-mode 'flyspell-mode-predicate 'scribe-mode-flyspell-verify)
  (setq major-mode 'scribe-mode)
  (setq mode-name "Scribe"))

(defun scribe-mode-flyspell-verify ()
  (and
   (not (save-excursion
	  (re-search-backward "^[ \t]*%%%[ \t]+Local" (point-min) t)))
   (not (save-excursion
	  (let ((this (point-marker))
		(e (progn (end-of-line) (point-marker))))
	    (beginning-of-line)
	    (if (re-search-forward "\\(?:markup\\)" e t)
		(and (>= this (match-beginning 0))
		     (<= this (match-end 0)) )))))))

(defun scribe-comment-indent (&optional pos)
  (save-excursion
    (if pos (goto-char pos))
    (cond ((looking-at ";;;") (current-column))
          ((looking-at ";;")
           (let ((tem (calculate-scribe-indent)))
             (if (listp tem) (car tem) tem)))
          ((looking-at "[[]") 0)
          (t
           (skip-chars-backward " \t")
           (max (if (bolp) 0 (1+ (current-column)))
                comment-column)))))

(defvar scribe-indent-offset nil "")
(defvar scribe-indent-hook 'scribe-indent-hook "")

(defun scribe-indent-line (&optional whole-exp)
  "Indent current line as Scribe code.
With argument, indent any additional lines of the same expression
rigidly along with this one."
  (interactive "P")
  (if (not (looking-at "[[]"))
      (let ((indent (calculate-scribe-indent)) shift-amt beg end
	    (pos (- (point-max) (point))))
	(beginning-of-line)
	(setq beg (point))
	(skip-chars-forward " \t")
	(if (or (looking-at "[ \t]*;;;") (looking-at "[ \t]*;[*]"))
	    ;; Don't alter indentation of a ;;; comment line.
	    nil
	  (if (listp indent) (setq indent (car indent)))
	  (setq shift-amt (- indent (current-column)))
	  (if (zerop shift-amt)
	      nil
	    (delete-region beg (point))
	    (indent-to indent))
	  ;; If initial point was within line's indentation,
	  ;; position after the indentation.  Else stay at same point in text.
	  (if (> (- (point-max) pos) (point))
	      (goto-char (- (point-max) pos)))
	  ;; If desired, shift remaining lines of expression the same amount.
	  (and whole-exp (not (zerop shift-amt))
	       (save-excursion
		 (goto-char beg)
		 (forward-sexp 1)
		 (setq end (point))
		 (goto-char beg)
		 (forward-line 1)
		 (setq beg (point))
		 (> end beg))
	       (indent-code-rigidly beg end shift-amt))))))

(defun calculate-scribe-indent (&optional parse-start)
  "Return appropriate indentation for current line as scribe code.
In usual case returns an integer: the column to indent to.
Can instead return a list, whose car is the column to indent to.
This means that following lines at the same level of indentation
should not necessarily be indented the same way.
The second element of the list is the buffer position
of the start of the containing expression."
  (save-excursion
    (beginning-of-line)
    (let ((indent-point (point)) state paren-depth desired-indent (retry t)
          last-sexp containing-sexp first-sexp-list-p)
      (if parse-start
          (goto-char parse-start)
        (beginning-of-defun))
      ;; Find outermost containing sexp
      (while (< (point) indent-point)
        (setq state (parse-partial-sexp (point) indent-point 0)))
      ;; Find innermost containing sexp
      (while (and retry (setq paren-depth (car state)) (> paren-depth 0))
        (setq retry nil)
        (setq last-sexp (nth 2 state))
        (setq containing-sexp (car (cdr state)))
        ;; Position following last unclosed open.
        (goto-char (1+ containing-sexp))
        ;; Is there a complete sexp since then?
        (if (and last-sexp (> last-sexp (point)))
            ;; Yes, but is there a containing sexp after that?
            (let ((peek (parse-partial-sexp last-sexp indent-point 0)))
              (if (setq retry (car (cdr peek))) (setq state peek))))
        (if (not retry)
            ;; Innermost containing sexp found
            (progn
              (goto-char (1+ containing-sexp))
              (if (not last-sexp)
                  ;; indent-point immediately follows open paren.
                  ;; Don't call hook.
                  (setq desired-indent (current-column))
                ;; Move to first sexp after containing open paren
                (parse-partial-sexp (point) last-sexp 0 t)
                (setq first-sexp-list-p (looking-at "\\s("))
                (cond
                 ((> (save-excursion (forward-line 1) (point))
                     last-sexp)
                  ;; Last sexp is on same line as containing sexp.
                  ;; It's almost certainly a function call.
                  (parse-partial-sexp (point) last-sexp 0 t)
                  (if (/= (point) last-sexp)
                      ;; Indent beneath first argument or, if only one sexp
                      ;; on line, indent beneath that.
                      (progn (forward-sexp 1)
                             (parse-partial-sexp (point) last-sexp 0 t)))
                  (backward-prefix-chars))
                 (t
                  ;; Indent beneath first sexp on same line as last-sexp.
                  ;; Again, it's almost certainly a function call.
                  (goto-char last-sexp)
                  (beginning-of-line)
                  (parse-partial-sexp (point) last-sexp 0 t)
                  (backward-prefix-chars)))))))
      ;; If looking at a list, don't call hook.
      (if first-sexp-list-p
          (setq desired-indent (current-column)))
      ;; Point is at the point to indent under unless we are inside a string.
      ;; Call indentation hook except when overriden by scribe-indent-offset
      ;; or if the desired indentation has already been computed.
      (cond ((and (numberp (car (cdr state)))
		  (save-excursion
		    (goto-char (car (cdr state)))
		    (looking-at "[[]")))
	     (setq desired-indent 0))
	    ((car (nthcdr 3 state))
             ;; Inside a string, don't change indentation.
             (goto-char indent-point)
             (skip-chars-forward " \t")
             (setq desired-indent (current-column)))
            ((and (integerp scribe-indent-offset) containing-sexp)
             ;; Indent by constant offset
             (goto-char containing-sexp)
             (setq desired-indent (+ scribe-indent-offset (current-column))))
            ((not (or desired-indent
                      (and (boundp 'scribe-indent-hook)
                           scribe-indent-hook
                           (not retry)
                           (setq desired-indent
                                 (funcall scribe-indent-hook
                                          indent-point state)))))
             ;; Use default indentation if not computed yet
             (setq desired-indent (current-column))))
      desired-indent)))

(defun scribe-indent-hook (indent-point state)
  (let ((normal-indent (current-column)))
    (save-excursion
      (goto-char (1+ (car (cdr state))))
      (re-search-forward "\\sw\\|\\s_")
      (if (/= (point) (car (cdr state)))
          (let ((function (buffer-substring (progn (forward-char -1) (point))
                                            (progn (forward-sexp 1) (point))))
                method)
            ;; Who cares about this, really?
	    ;; (if (not (string-match "\\\\\\||" function)))
            (setq function (downcase function))
            (setq method (get (intern-soft function) 'scribe-indent-hook))
            (cond
	     ((integerp method)
	      (scribe-indent-specform method state indent-point))
	     (method
	      (funcall method state indent-point))
	     ((and (> (length function) 3)
		   (let ((s (substring function 0 3)))
		     (or (string-equal s "def")
			 (string-equal s "bgl"))))
	      (scribe-indent-defform state indent-point))
	     ((string-equal function "document")
	      (scribe-block-indent state indent-point 0))
	     ((string-equal function "chapter")
	      (scribe-block-indent state indent-point 0))
	     ((string-equal function "section")
	      (scribe-block-indent state indent-point 0))
	     ((string-equal function "subsection")
	      (scribe-block-indent state indent-point 0))
	     ((string-equal function "subsubsection")
	      (scribe-block-indent state indent-point 0))
	     ((string-equal function "paragraph")
	      (scribe-block-indent state indent-point 0))))))))

(defvar scribe-body-indent 3 "")

(defun scribe-indent-specform (count state indent-point)
  (let ((containing-form-start (car (cdr state))) (i count)
        body-indent containing-form-column)
    ;; Move to the start of containing form, calculate indentation
    ;; to use for non-distinguished forms (> count), and move past the
    ;; function symbol.  scribe-indent-hook guarantees that there is at
    ;; least one word or symbol character following open paren of containing
    ;; form.
    (goto-char containing-form-start)
    (setq containing-form-column (current-column))
    (setq body-indent (+ scribe-body-indent containing-form-column))
    (forward-char 1)
    (forward-sexp 1)
    ;; Now find the start of the last form.
    (parse-partial-sexp (point) indent-point 1 t)
    (while (and (< (point) indent-point)
                (condition-case nil
                    (progn
                      (setq count (1- count))
                      (forward-sexp 1)
                      (parse-partial-sexp (point) indent-point 1 t))
                  (error nil))))
    ;; Point is sitting on first character of last (or count) sexp.
    (cond ((> count 0)
           ;; A distinguished form.  Use double scribe-body-indent.
           (list (+ containing-form-column (* 2 scribe-body-indent))
                 containing-form-start))
          ;; A non-distinguished form. Use body-indent if there are no
          ;; distinguished forms and this is the first undistinguished
          ;; form, or if this is the first undistinguished form and
          ;; the preceding distinguished form has indentation at least
          ;; as great as body-indent.
          ((and (= count 0)
                (or (= i 0)
                    (<= body-indent normal-indent)))
           body-indent)
          (t
           normal-indent))))

(defun scribe-indent-defform (state indent-point)
  (goto-char (car (cdr state)))
  (forward-line 1)
  (if (> (point) (car (cdr (cdr state))))
      (progn
        (goto-char (car (cdr state)))
        (+ scribe-body-indent (current-column)))))

(defun scribe-with-access-indent (state indent-point)
  (skip-chars-forward " \t")
  (scribe-indent-with-access-form 2 state indent-point))

(defun scribe-block-indent (state indent-point delta)
  (if (> delta 0) (skip-chars-forward " \t"))
  delta)

(defun scribe-indent-with-access-form (count state indent-point)
  (let ((containing-form-start (car (cdr state))) (i count)
        body-indent containing-form-column)
    ;; Move to the start of containing form, calculate indentation
    ;; to use for non-distinguished forms (> count), and move past the
    ;; function symbol.  scribe-indent-hook guarantees that there is at
    ;; least one word or symbol character following open paren of containing
    ;; form.
    (goto-char containing-form-start)
    (setq containing-form-column (current-column))
    (setq body-indent (+ scribe-body-indent containing-form-column))
    (forward-char 1)
    (forward-sexp 1)
    ;; Now find the start of the last form.
    (parse-partial-sexp (point) indent-point 1 t)
    (while (and (< (point) indent-point)
                (condition-case nil
                    (progn
                      (setq count (1- count))
                      (forward-sexp 1)
                      (parse-partial-sexp (point) indent-point 1 t))
                  (error nil))))
    ;; Point is sitting on first character of last (or count) sexp.
    (cond ((> count 0)
           ;; A distinguished form.  Use double scribe-body-indent.
           (list (+ containing-form-column (* 2 scribe-body-indent))
                 containing-form-start))
          ;; A non-distinguished form. Use body-indent if there are no
          ;; distinguished forms and this is the first undistinguished
          ;; form, or if this is the first undistinguished form and
          ;; the preceding distinguished form has indentation at least
          ;; as great as body-indent.
          ((and (= count 0)
                (or (= i 0)
                    (<= body-indent normal-indent)))
           body-indent)
          (t
           normal-indent))))

(defun scribe-indent-instantiate-form (count state indent-point)
  (let ((containing-form-start (car (cdr state))) (i count)
	body-indent containing-form-column)
    ;; Move to the start of containing form, calculate indentation
    ;; to use for non-distinguished forms (> count), and move past the
    ;; function symbol.  scribe-indent-hook guarantees that there is at
    ;; least one word or symbol character following open paren of containing
    ;; form.
    (goto-char containing-form-start)
    (setq containing-form-column (current-column))
    (setq body-indent (+ scribe-body-indent containing-form-column))
    (forward-char 1)
    (forward-sexp 2)
    ;; Now find the start of the last form.
    (parse-partial-sexp (point) indent-point 1 t)
    (while (and (< (point) indent-point)
		(condition-case nil
		    (progn
		      (setq count (1- count))
		      (forward-sexp 1)
		      (parse-partial-sexp (point) indent-point 1 t))
		  (error nil))))
    ;; Point is sitting on first character of last (or count) sexp.
    (cond ((> count 0)
	   ;; A distinguished form.  Use double scribe-body-indent.
	   (list (+ containing-form-column (* 2 scribe-body-indent))
		 containing-form-start))
	  ;; A non-distinguished form. Use body-indent if there are no
	  ;; distinguished forms and this is the first undistinguished
	  ;; form, or if this is the first undistinguished form and
	  ;; the preceding distinguished form has indentation at least
	  ;; as great as body-indent.
	  ((and (= count 0)
		(or (= i 0)
		    (<= body-indent normal-indent)))
	   body-indent)
	  (t
	   normal-indent))))

;;; Let is different in Scribe

(defun would-be-symbol (string)
  (not (string-equal (substring string 0 1) "(")))

(defun next-sexp-as-string ()
  ;; Assumes that protected by a save-excursion
  (forward-sexp 1)
  (let ((the-end (point)))
    (backward-sexp 1)
    (buffer-substring (point) the-end)))

;; This is correct but too slow.
;; The one below works almost always.
;;(defun scribe-let-indent (state indent-point)
;;  (if (would-be-symbol (next-sexp-as-string))
;;      (scribe-indent-specform 2 state indent-point)
;;      (scribe-indent-specform 1 state indent-point)))

(defun scribe-let-indent (state indent-point)
  (skip-chars-forward " \t")
  (if (looking-at "[a-zA-Z0-9+-*/?!@$%^&_:~]")
      (scribe-indent-specform 2 state indent-point)
      (scribe-indent-specform 1 state indent-point)))

;; (put 'begin 'scribe-indent-hook 0), say, causes begin to be indented
;; like defun if the first form is placed on the next line, otherwise
;; it is indented like any other form (i.e. forms line up under first).
;; basic forms
(put 'begin                  'scribe-indent-hook 0)
(put 'case                   'scribe-indent-hook 1)
(put 'delay                  'scribe-indent-hook 0)
(put 'do                     'scribe-indent-hook 0)
(put 'lambda                 'scribe-indent-hook 1)
(put 'cond                   'scribe-indent-hook 0)
(put 'when                   'scribe-indent-hook 1)
(put 'unless                 'scribe-indent-hook 1)

;; binding forms
(put 'let                    'scribe-indent-hook 1)
(put 'let*                   'scribe-indent-hook 1)
(put 'letrec                 'scribe-indent-hook 1)
(put 'labels                 'scribe-indent-hook 1)
(put 'let-syntax             'scribe-indent-hook 1)
(put 'letrec-syntax          'scribe-indent-hook 1)

;; output/input command
(put 'call-with-input-file   'scribe-indent-hook 1)
(put 'with-input-from-file   'scribe-indent-hook 1)
(put 'with-input-from-port   'scribe-indent-hook 1)
(put 'with-input-from-string 'scribe-indent-hook 1)
(put 'call-with-output-file  'scribe-indent-hook 1)
(put 'with-output-to-file    'scribe-indent-hook 1)
(put 'with-output-to-port    'scribe-indent-hook 1)
(put 'with-output-to-string  'scribe-indent-hook 0)
(put 'with-error-to-port     'scribe-indent-hook 1)
(put 'with-error-to-file     'scribe-indent-hook 1)
(put 'with-error-to-string   'scribe-indent-hook 0)

;; exceptions
(put 'bind-exit              'scribe-indent-hook 1)
(put 'unwind-protect         'scribe-indent-hook 0)
(put 'dynamic-wind           'scribe-indent-hook 0)

;; multiple values
(put 'multiple-value-bind    'scribe-indent-hook 1)
(put 'receive                'scribe-indent-hook 1)
(put 'call-with-values       'scribe-indent-hook 0)

;; parsing
(put 'regular-grammar        'scribe-indent-hook 'scribe-let-indent)
(put 'lalr-grammar           'scribe-indent-hook 0)

;; matching indentation
(put 'args-parse             'scribe-indent-hook 1)
(put 'match-case             'scribe-indent-hook 1)
(put 'cond-expand            'scribe-indent-hook 0)
(put 'string-case            'scribe-indent-hook 1)
(put 'match-lambda           'scribe-indent-hook 0)
(put 'syntax-rules           'scribe-indent-hook 1)

(defun scribe-indent-sexp ()
  "Indent each line of the list starting just after point."
  (interactive)
  (let ((indent-stack (list nil)) (next-depth 0) last-depth bol
        outer-loop-done inner-loop-done state this-indent)
    (save-excursion (forward-sexp 1))
    (save-excursion
      (setq outer-loop-done nil)
      (while (not outer-loop-done)
        (setq last-depth next-depth
              inner-loop-done nil)
        (while (and (not inner-loop-done)
                    (not (setq outer-loop-done (eobp))))
          (setq state (parse-partial-sexp (point) (progn (end-of-line) (point))
                                          nil nil state))
          (setq next-depth (car state))
          (if (car (nthcdr 4 state))
              (progn (indent-for-comment)
                     (end-of-line)
                     (setcar (nthcdr 4 state) nil)))
          (if (car (nthcdr 3 state))
              (progn
                (forward-line 1)
                (setcar (nthcdr 5 state) nil))
            (setq inner-loop-done t)))
        (if (setq outer-loop-done (<= next-depth 0))
            nil
          (while (> last-depth next-depth)
            (setq indent-stack (cdr indent-stack)
                  last-depth (1- last-depth)))
          (while (< last-depth next-depth)
            (setq indent-stack (cons nil indent-stack)
                  last-depth (1+ last-depth)))
          (forward-line 1)
          (setq bol (point))
	  (if (and (numberp (car (cdr state)))
		   (save-excursion
		     (goto-char (car (cdr state)))
		     (looking-at "[[]")))
	      (progn
		(skip-chars-forward " \t")		
		(setcar indent-stack (setq this-indent 0))
		(if (/= (current-column) this-indent)
		    (progn (delete-region bol (point))
			   (indent-to this-indent))))
	    (progn
	      (skip-chars-forward " \t")
	      (if (or (eobp) (looking-at "[;\n]"))
		  nil
		(if (and (car indent-stack)
			 (>= (car indent-stack) 0))
		    (setq this-indent (car indent-stack))
		  (let ((val (calculate-scribe-indent
			      (if (car indent-stack) (- (car indent-stack))))))
		    (if (integerp val)
			(setcar indent-stack
				(setq this-indent val))
		      (setcar indent-stack (- (car (cdr val))))
		      (setq this-indent (car val)))))
		(if (/= (current-column) this-indent)
		    (progn (delete-region bol (point))
			   (indent-to this-indent)))))))))))


