#include <gnome.h>

#include <glade/glade.h>

#include "editor.h"
#include "htmlfuncs.h"
#include "site.h"
#include "page.h"
#include "structure.h"
#include "preferences.h"
#include "dtd.h"

extern GtkWidget *app;
extern Site *current_site;
extern Preferences *cfg;

extern GNode *editor_parse_tree;

static gboolean block = FALSE;


static void add_node( GtkCTree *tree, GtkCTreeNode *parent, GNode *node );

static gchar* replace_node( GtkCTree *tree, GtkCTreeNode *node,
			    const gchar *text );

static void build_attribute_entries( GtkWidget *box, GList *attributes, 
				     GList *attr );

static void replace_attr( GtkWidget *entry, gchar *attr_name );


void build_page_structure_tree()
{
	GtkWidget *tree;
	GladeXML *xml;
	Page *page;
	const gchar *text;
	ScreemDTD *dtd;
	GtkWidget *widget;
	
	widget = gtk_object_get_data( GTK_OBJECT( app ), "sitebook" );
	xml = glade_get_widget_tree( widget );

	tree = glade_xml_get_widget( xml, "structure_tree" );

	gtk_clist_clear( GTK_CLIST( tree ) );

	screem_editor_buffer_text();

	page = screem_site_get_current_page( current_site );

	if( ! page )
		return;

	if( ! cfg->mpage->parse )
		return;

	text = screem_page_get_data( page );

    	gtk_clist_freeze( GTK_CLIST( tree ) );

	dtd = screem_get_doctype( text );
	if( ! editor_parse_tree )
		editor_parse_tree = screem_html_build_parse_tree( dtd,text,0 );

	add_node( GTK_CTREE( tree ), NULL, editor_parse_tree );
	gtk_clist_thaw( GTK_CLIST( tree ) );

	gtk_clist_set_column_auto_resize( GTK_CLIST( tree ), 0, TRUE );
}

static void add_node( GtkCTree *tree, GtkCTreeNode *parent, GNode *node )
{
	Node *n;
	GtkCTreeNode *this;
	gchar *item[ 2 ] = { NULL, NULL };

	if( ! node )
		return;

	n = node->data;

	if( n->name )
		item[ 0 ] = n->name;
	else
		item[ 0 ] = n->content;
	this = gtk_ctree_insert_node( tree, parent, NULL, item,
				      3, NULL, NULL, NULL, NULL, 
				      (gboolean)parent, FALSE);
	gtk_ctree_node_set_row_data( tree, this, n );

	if( node->children )
		add_node( tree, this, node->children );
	if( node->next )
		add_node( tree, parent, node->next );
}

void struct_select_row( GtkCTree *tree, GtkCTreeNode *node )
{
	Node *n;

	if( block )
		return;

	n = gtk_ctree_node_get_row_data( tree, node );

	if( n )
		screem_editor_set_pos( n->pos );
}

gboolean structure_clicked( GtkWidget *widget, GdkEventButton *event )
{
	gint row;
	gint col;
	GladeXML *xml;

	if( event->button == 3 ) {
		/* popup menu */
		xml = glade_xml_new( cfg->glade_path, "structure_popup" );
		glade_xml_signal_autoconnect( xml );
		widget = glade_xml_get_widget( xml, "structure_popup" );
		gnome_popup_menu_do_popup_modal( widget, 0, 0, event, 0 );
		gtk_widget_destroy( widget );
		
		return TRUE;
	}

	gtk_clist_get_selection_info( GTK_CLIST( widget ),
                                      event->x, event->y, &row, &col );

	if( row != - 1 ) {
		gtk_object_set_data( GTK_OBJECT( widget ),
                                     "drag_start", GINT_TO_POINTER( row ) );
        } else {
                gtk_object_set_data( GTK_OBJECT( widget ),
                                     "drag_start", NULL);
	}

	return FALSE;
}

void structure_refresh()
{
	build_page_structure_tree();
}


void structure_set_dnd_data( GtkWidget *widget, GdkDragContext *context,
			     GtkSelectionData *selectionData, guint info,
			     guint time, gpointer data)
{
	GtkCTreeNode *node;
	Node *n;
	gchar *text = NULL;

        if( ! GTK_CLIST( widget )->selection )
                return;
	
	node = ((GList*)(GTK_CLIST(widget)->selection))->data;
	if( node )
		n = gtk_ctree_node_get_row_data( GTK_CTREE(widget), node );
	
	switch( info ) {
	case TARGET_SCREEM_STRUCTURE:
		text = g_strdup( "valid" );
		break;
	default:
		text = g_strdup( "" );
		n = NULL;
	}
	/* The drop data is just an abitrary string as the real info is
	   stored as data on the clist object.  The reason this is done
	   is to simplify getting the node that was dropped, and because
	   I do not forsee any external application dropping anything of
	   this type onto the structure tree */
    	gtk_selection_data_set( selectionData, selectionData->target,
				8, text, strlen( text ) );
}

void structure_drag_begin( GtkWidget *widget, GdkDragContext *context,
			   gpointer data )
{
	GtkCTreeNode *node;
        gpointer key;
        
        key = gtk_object_get_data( GTK_OBJECT( widget ), "drag_start" );
        
        g_return_if_fail( key != NULL );
       
	node = gtk_ctree_node_nth( GTK_CTREE( widget ), 
				   GPOINTER_TO_INT( key ));
	g_return_if_fail( node != NULL );

	block = TRUE;
	gtk_ctree_select( GTK_CTREE( widget ), node );
	block = FALSE;
}

gboolean structure_drag_motion( GtkWidget *widget, 
				GdkDragContext *context, 
				gint x, gint y, guint time,
				gpointer data )
{
	GdkDragAction action;
	
	action = GDK_ACTION_MOVE; 
      
        gdk_drag_status( context, action, time );

	return TRUE;
}

void structure_drop_data( GtkWidget *widget, GdkDragContext *context,
			  gint x, gint y, GtkSelectionData *selectionData,
			  guint info, guint time )
{
	gint row;
	gint col;
	GtkCTreeNode *node = NULL;
 	Page *page;
	gchar *data;
	gchar *drop_data;

	GladeXML *xml;
	GtkWidget *menu;
	Node *n;
	
	data = selectionData->data;

	page = screem_site_get_current_page( current_site );

	if( ! page || ! strlen( data ) ) 
		return;

	gtk_clist_get_selection_info( GTK_CLIST( widget ), x, y, &row, &col );
	/* id the node we dropped onto */
	if( row != -1 )
		node = gtk_ctree_node_nth( GTK_CTREE( widget ), ( guint )row );
	else
		node = gtk_ctree_node_nth( GTK_CTREE( widget ), 0 );
	
	g_return_if_fail( node != NULL );

	n = gtk_ctree_node_get_row_data( GTK_CTREE( widget ), node );
	gtk_object_set_data( GTK_OBJECT( widget ), "node_info", n );

       	switch( info ) {
	case TARGET_SCREEM_STRUCTURE:
		/* we don't actually care about the content of the drop data
		   as it is just an arbitary string, however we need to clear
		   any previous data that will be used */
		drop_data = NULL;
		break;
	case TARGET_STRUCTURE_URL:
		/* we can drop urls onto the structure and have have a link
		   inserted at that position */
		drop_data = g_strdup_printf( "<a href=\"%s\"></a>", data );
		break;
	case TARGET_STRUCTURE_URI_LIST:
		/* just return until this is handled */
		return;
		break;
	case TARGET_STRUCTURE_TEXT:
		drop_data = g_strdup( data );
		break;
	default:
		g_free( data );
		return;
	}

	g_free( data );

	gtk_object_set_data( GTK_OBJECT( widget ), "drop_data", drop_data );

	if( n->state != FORBIDDEN ) {
		xml = glade_xml_new( cfg->glade_path, "struct_dnd_menu" );
		menu = glade_xml_get_widget( xml, "struct_dnd_menu" );
		glade_xml_signal_autoconnect( xml );
		gnome_popup_menu_do_popup_modal( menu, 0, 0, 0, 0 );
	} else
		structure_add_as_sibling( NULL, NULL );
}

Node* structure_add_get_text( gchar **text, gint *start, gint *len, gint *cpos)
{
	GladeXML *xml;
	GtkWidget *widget;
	GtkWidget *ctree;
	guint row;
	GtkCTreeNode *n;
	Node *node;
	Node *dropped;
	gint elen;
	gchar *name;
	gchar *temp;

	gchar *drop_data;
	gint junk;

	if( ! start )
		start = &junk;
	if( ! len )
		len = &junk;

	widget = gtk_object_get_data( GTK_OBJECT( app ), "sitebook" );
	xml = glade_get_widget_tree( widget );
	ctree = glade_xml_get_widget( xml, "structure_tree" );

	node = (Node*)gtk_object_get_data( GTK_OBJECT( ctree ), "node_info" );

	row = (guint)gtk_object_get_data( GTK_OBJECT( ctree ), "drag_start" );
	n = gtk_ctree_node_nth( GTK_CTREE( ctree ), row );

	dropped = gtk_ctree_node_get_row_data( GTK_CTREE( ctree ), n );
	drop_data = gtk_object_get_data( GTK_OBJECT( ctree ), "drop_data");

	if( ! drop_data ) {
		if( dropped->name )
			if( dropped->state != FORBIDDEN )
				*len = strlen( dropped->name );
			else
				*len = strlen( dropped->content ) - 1;
		else
			*len = -1;
		
		*start = dropped->pos - *len - 1;
		
		/* is dropped->cpos the position of the closing tag or a new
		   opening one? If its a closing we need to include the length
		   of that in the text we want */
		name = tag_name( dropped->name );
		
		if( dropped->name )
			if( dropped->state != FORBIDDEN )
				*len = dropped->cpos - *start;
			else
				*len = strlen( dropped->content ) + 1;
		else
			*len = strlen( dropped->content );
	
		if( name && dropped->state != FORBIDDEN ) {
			elen = strlen( "</>" ) + strlen( name );
			*text = screem_editor_get_text( *start + *len + 1,
							elen );
			if( *text && (*text)[ 1 ] == '/' ) {
				temp = g_strconcat( "</", name, ">", NULL );
				if( ! strcmp( temp, *text ) )
					*len += elen + 1;
				g_free( temp );
			}
			g_free( *text );
			g_free( name );
		}
		*text = screem_editor_get_text( *start, *len );
	} else {
		/* not a node being dropped, get drag_data */
		*text = drop_data;
		*start = -1;
		*len = -1;
	}

	if( cpos ) {
		name = tag_name( node->name );
		*cpos = node->cpos;
		if( name && node->state != FORBIDDEN ) {
			elen = strlen( "</>" ) + strlen( name );
			*text = screem_editor_get_text( *cpos + 1, elen );
			if( *text && (*text)[ 1 ] == '/' ) {
				temp = g_strconcat( "</", name, ">", NULL );
				if( ! strcmp( temp, *text ) )
					*cpos += elen + 1;
				g_free( temp );
			}
			g_free( *text );
			g_free( name );
		}
	}

	return node;
}

void structure_add_as_child( GtkWidget *item, gpointer data )
{
	gint start;
	gint len;
	Node *node;
	gint pos;
	gchar *text;

	node = structure_add_get_text( &text, &start, &len, NULL );

	/* move dropped to be a child of node, text is the data */
	pos = node->pos + 1;

	if( start != -1 && len !=- -1 ) {
		screem_editor_delete_forward( start, len );
		if( start <= pos ) {
			/* we need to subtract an offset */
			pos -= len;
		}
	}
	screem_editor_insert( pos, text );
	build_page_structure_tree();

	g_free( text );
}

void structure_add_as_sibling( GtkWidget *item, gpointer data )
{
	gint start;
	gint len;
	Node *node;
	gchar *text;
	gint cpos;

	node = structure_add_get_text( &text, &start, &len, &cpos );

	/* move dropped to be a sibling of node, text is the data */
	if( start != -1 && len != -1 ) {
		screem_editor_delete_forward( start, len );
		if( start <= cpos ) {
			/* we need to subtract an offset */
			cpos -= len;
		}
	}
	screem_editor_insert( cpos, text );
	build_page_structure_tree();

	g_free( text );
}

/* this will take the tree structure built and replace the document
   with it */
void structure_fix()
{
	GtkWidget *tree;
	GladeXML *xml;
	GtkCTreeNode *node;
	GtkWidget *widget;
	const gchar *text;
	gchar *fixed;
	Page *page;

	widget = gtk_object_get_data( GTK_OBJECT( app ), "sitebook" );
	xml = glade_get_widget_tree( widget );

	tree = glade_xml_get_widget( xml, "structure_tree" );

	node = gtk_ctree_node_nth( GTK_CTREE( tree ), 0 );

	page = screem_site_get_current_page( current_site );
	text = screem_page_get_data( page );

	fixed = replace_node( GTK_CTREE( tree ), node, text  );

	screem_editor_clear();
	screem_editor_insert( 0, fixed );
	g_free( fixed );
}

static gchar* replace_node( GtkCTree *tree, GtkCTreeNode *node, 
			    const gchar *text )
{
	gchar *next;
	gchar *fixed;
	gchar *temp;
	gchar *close;

	Node *n;

	GList *list;
	gchar *attr;

	n = gtk_ctree_node_get_row_data( tree, node );

	next = NULL;
	if( GTK_CTREE_ROW( node )->children )
		next = replace_node( tree, GTK_CTREE_ROW( node )->children, 
				     text );
	
	attr = g_strdup( "" );
	for( list = n->attributes; list; list = list->next ) {
		temp = attr;
		if( list->next->data )
			attr = g_strconcat( temp, " ", list->data, "=\"", 
					    list->next->data,
					    "\"", NULL );
		else
			attr = g_strconcat( temp, " ", list->data, NULL );
		g_free( temp );
		list = list->next;
	}

  	if( n->name ) {
		if( ! strcmp( "!--", n->name ) )
			close = " -->";
		else
			close =">";
		fixed = g_strconcat( n->content, attr, close, next, NULL );
	} else
		fixed = g_strconcat( n->content, next, NULL );

	/* if its a tag then add a close tag for it if it is allowed */
	if( n->name && n->state != FORBIDDEN ) {
		temp = fixed;
		fixed = g_strconcat( temp, "\n</", n->name, ">", NULL );
		g_free( temp );
	}

	next = NULL;
 	if( GTK_CTREE_ROW( node )->sibling )
		next = replace_node( tree, GTK_CTREE_ROW( node )->sibling, 
				     text );
	if( next ) {
		temp = fixed;
		fixed = g_strconcat( temp, next, NULL );
		g_free( temp );
		
	}

	return fixed;
}


void build_attribute_list()
{
    	GtkWidget *box;
	GtkBoxChild *bchild;
    
	GList *set_attr;

	gchar *name;

	gint pos;
	gchar *text;
	gchar *tag;

	GList *list;

	GList *attributes;
	ScreemDTD *dtd;

	static gint ppos = -1;

	box = gtk_object_get_data( GTK_OBJECT( app ), "attr_box" );

	/* id which tag we are in */
	pos = screem_editor_get_pos();
	text = screem_editor_get_text( 0, -1 );
	pos = in_tag( text, pos );
	if( ! pos ) {
		ppos = pos;
		gtk_widget_set_sensitive( box, FALSE );
		g_free( text );
		return;
	} else if( ppos == pos ) {
		g_free( text );
		return;
	}
	ppos = pos;
	

	pos --;
	tag = next_tag( text + pos, &pos, &name );
	set_attr = screem_html_build_attributes_list( tag, NULL );

	dtd = screem_get_doctype( text );
	attributes = screem_dtd_element_get_attributes( dtd, name );

	g_free( text );
	g_free( tag );
	g_free( name );

	gtk_widget_hide( box );

	while( ( list = GTK_BOX( box )->children ) ) {
		bchild = list->data;
		gtk_container_remove( GTK_CONTAINER( box ), bchild->widget );
	}

	build_attribute_entries( box, attributes, set_attr );

	/* destroy all of set_attr */
	for( list = set_attr; list; list = list->next )
		g_free( list->data );
	
	g_list_free( set_attr );

	gtk_widget_show_all( box );
	gtk_widget_set_sensitive( box, TRUE );
}

static void build_attribute_entries( GtkWidget *box, GList *attributes, 
				     GList *attr )
{
    	GtkWidget *line;
	GtkWidget *label;
	GtkWidget *combo;
	GtkWidget *entry;

	GList *combo_list;
	GList *list;

	ScreemDTDAttribute *attribute;
	gchar **values;
	gint num;

	if( ! box )
		return;

	for( ; attributes ; attributes = attributes->next ) {
		attribute = (ScreemDTDAttribute*)attributes->data;

		/* add entrys */
		line = gtk_table_new( 2, 1, FALSE );
		label = gtk_label_new( attribute->name );
		gtk_label_set_justify( GTK_LABEL( label ), GTK_JUSTIFY_LEFT );
		gtk_misc_set_alignment( GTK_MISC( label ), 0.0, 0.5 );
		gtk_table_attach( GTK_TABLE( line ), label, 0, 1, 0, 1,
				  GTK_FILL | GTK_EXPAND, 
				  GTK_FILL | GTK_EXPAND, 4, 4 );

		combo = gtk_combo_new();
		gtk_widget_set_usize( combo, 128, -1 );
		entry = GTK_COMBO( combo )->entry;
		gtk_table_attach( GTK_TABLE( line ), combo, 1, 2, 0, 1,
				  GTK_FILL, GTK_FILL, 4, 4 );
		gtk_box_pack_start( GTK_BOX( box ), line, FALSE, FALSE, 0 );
		
		/* add combo entries */
		combo_list = NULL;
		for( num = 0, values = attribute->values; values[num]; num ++ )
			combo_list = g_list_append( combo_list, values[num] );
		
		if( combo_list ) {
			gtk_combo_set_popdown_strings( GTK_COMBO( combo ), 
						       combo_list );
			g_list_free( combo_list );
		}
		
		/* 
		 * set the entry value to whatever value the
		 * attribute currently has in the tag, or make it
		 * empty if the attribute isn't used
		 */

		/* find name in the attr list */
		for( list = attr; list; list = list->next ) {
			if( ! strcmp( attribute->name, (gchar*)list->data ) )
				break;
			list = list->next;
		}
		
		if( ! list )
			gtk_entry_set_text( GTK_ENTRY( entry ), "" );
		else if( ! list->next )
			gtk_entry_set_text( GTK_ENTRY( entry ), "1" );
		else
			gtk_entry_set_text( GTK_ENTRY( entry ),
					    list->next->data );

		gtk_combo_disable_activate( GTK_COMBO( combo ) );
		gtk_signal_connect( GTK_OBJECT( entry ), "activate",
				    GTK_SIGNAL_FUNC( replace_attr ), 
				    attribute->name );

		gtk_widget_show_all( line );
	}
}

static void replace_attr( GtkWidget *entry, gchar *attr_name )
{
	gchar *text;
	gchar *attr;
	gint pos;

	text = gtk_entry_get_text( GTK_ENTRY( entry ) );

	attr = g_strconcat( attr_name, "=\"", text, "\"", NULL );

	pos = screem_editor_get_pos();
	screem_editor_insert_attribute( attr );
	screem_editor_set_pos( pos );

	g_free( attr );
}
