/*  Screem:  preview.c,
 *  code for creating/updating the preview window
 *
 *  Copyright (C) 1999, 2000  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <gnome.h>
#include <dirent.h>

#include "preferences.h"
#include "preview.h"

#ifdef HAVE_GTKHTML
#include <gtkhtml/gtkhtml.h>

#else
#include <gtk-xmhtml/gtk-xmhtml.h>

static void xm_url_clicked( GtkWidget *widget, XmHTMLAnchorCallbackStruct *cbs,
			    gboolean help );

#endif

#include "site.h"
#include "page.h"
#include "fileops.h"

#include "pageUI.h"
#include "htmlfuncs.h"

extern Site *current_site;
extern Preferences *cfg;

static GtkWidget *display;
static GtkWidget *helpdis;

static GtkWidget *screem_preview( gboolean help );
static gchar* screem_load_splash( void );

static gchar *help_dir = NULL;

static void url_requested( GtkWidget *html, const gchar *url,
			   gpointer handle );

static void url_clicked( GtkWidget *widget, const gchar *url, gboolean help );


/*
 * screem_preview_new:
 *
 * create the editor area
 *
 * return values: a GtkWidget
 */
GtkWidget *screem_preview_new()
{
	GtkWidget *preview;

	/* insert switch statement here to select from possible
	   preview widgets */
	preview = screem_preview( FALSE );

	gtk_widget_set_usize(preview, cfg->preview_width, cfg->preview_height);

	return preview;
}

static GtkWidget *screem_preview( gboolean help )
{
	GtkWidget *sw;
	GtkWidget *html;
#ifndef HAVE_GTKHTML
	sw = gtk_vbox_new( FALSE, 4 );

	html = gtk_xmhtml_new();
	gtk_widget_show( html );
	gtk_box_pack_start( GTK_BOX( sw ), html, TRUE, TRUE, 0 );

	gtk_signal_connect( GTK_OBJECT( html ), "activate",
			    GTK_SIGNAL_FUNC( xm_url_clicked ),(gpointer)help );
#else
	html = gtk_html_new();
	gtk_widget_show( html );

	sw = gtk_scrolled_window_new( GTK_LAYOUT( html )->hadjustment,
				      GTK_LAYOUT( html )->vadjustment );
	gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW( sw ), 
                                        GTK_POLICY_AUTOMATIC,
                                        GTK_POLICY_AUTOMATIC );

	gtk_container_add( GTK_CONTAINER( sw ), html );

 	gtk_signal_connect( GTK_OBJECT( html ), "url_requested",
			    GTK_SIGNAL_FUNC( url_requested ), NULL );
	gtk_signal_connect( GTK_OBJECT( html ), "link_clicked",
			    GTK_SIGNAL_FUNC( url_clicked ), (gpointer)help );
#endif
	if( ! help ) {
		display = html;
		gtk_object_set_data( GTK_OBJECT( sw ), "preview", html );
	} else {
		helpdis = html;
		gtk_object_set_data( GTK_OBJECT( sw ), "help", html );
	}

	return sw;
}

#ifndef HAVE_GTKHTML
static void xm_url_clicked( GtkWidget *widget, XmHTMLAnchorCallbackStruct *cbs,
			    gboolean help )
{
	gchar *url = cbs->href;

	url_clicked( widget, url, help );
}
#endif

static void url_requested( GtkWidget *html, const gchar *url,
			   gpointer handle )
{
	FILE *file;
	gchar buffer[ BUFSIZ + 1 ];
	gint size;

	gchar *content;
	gchar *temp;

	if( ! strlen( url ) )
		return;

	content = g_strdup( "" );
 	if( ( file = fopen( url, "r" ) ) ) {
		while( ( size = fread( buffer, 1, BUFSIZ, file ) ) ) {
#ifdef HAVE_GTKHTML
			gtk_html_write( GTK_HTML( html ), handle, 
					buffer, size );
#else
			buffer[ size ] = '\0';
			temp = content;
			content = g_strconcat( temp, buffer, NULL );
			g_free( temp );
#endif
		}
		fclose( file );
	}
#ifdef HAVE_GTKHTML
	gtk_html_end( GTK_HTML( html ), handle, GTK_HTML_STREAM_OK );
#else
#if 0
	gtk_xmhtml_freeze( GTK_XMHTML( html ) );
	gtk_xmhtml_source( GTK_XMHTML( html ), content );
	gtk_xmhtml_thaw( GTK_XMHTML( html ) );
#endif
#endif

	g_free( content );
}

/* returns text to show in the preview window when screem first loads up */
static gchar* screem_load_splash()
{
	return g_strdup_printf( _("\
<html>\
<head>\
<title>SCREEM</title>\
</head>\
<body bgcolor=\"black\" text=\"white\">\
<h1 align=\"center\">SCREEM</h1>\
<h3 align=\"center\">Site CReating and Editing EnvironMent</h3>\
<hr>\
<div align=\"center\">\
Welcome to %s version %s\
</div>\
<br>\
<h3>0.4.1 Release Notes:</h3>\
<ul>\
<li>Fixed spec file\
<li>lots of CSS Wizard fixes\
<li>The CSS Wizard will no longer remove properties it doesn't recognise\
<li>Fixed startup crash when switching off notebook toolbar for the first time\
</ul>\
<h3>0.4.0 Release Notes:</h3>\
<ul>\
<li>Please do <em>not</em> use the gnome bug track system for screem bugs<br> \
They will not be dealt with, instead use the bug tracking system on our SourceForge \
project page. (accessable from screem.org) \
<li>On the fly document parsing can now be disabled, which will sort out the speed \
problem with large documents (but will also disabled a number of other features)\
<li>Fixed small bug in page druid\
<li>Table Wizard updated\
<li>A default DTD can be set for documents which don't specify one\
<li>Inline tagging and the properties tab now take the allowed attributes \
from the DTD for the current document.\
<li>The structure tree now supports drag and drop\
<ul>\
<li>Move tags around in the tree itself\
<li>drop urls onto the tree to insert link tags\
<li>drop plain text onto the tree to insert text\
</ul>\
<li>The open page tabs now have a popup menu by right clicking on them.  \
This allows you to save/close/rename a page without viewing it.\
<li>Insert Tags menu added to editor popup, which allows you to insert valid \
tags based upon the DTD in use (or the default DTD if one is not specified) \
<li>More Translations - thanks to Christophe Merlet syncing Screem up with \
the ones in Gnome CVS, and to all those who contribute to the translations \
<li>Switching the viewbar off in the preferences will display the notebook \
tabs for the views instead (requires restart) \
<li>If a DTD is not found locally Screem will fetch it from the uri in the \
DOCTYPE (Requires gnome-vfs) \
<li>Fetched DTDs are stored in ~/.screem/dtds and are added to the catalog \
file held there. \
<li>Removed paned view - messages and file listing/tag tree etc are now dock \
items. \
<li>Dock items should remember their position properly now.\
<li>Ability to show/hide any dock item via a popup menu\
<li>Autoloading of sites which were open last time screem was in use added \
(can be switched off)\
<li>Auto loading of pages when a site is opened that were open when it was \
last closed\
<li>An improved link view primarily by Matt Colyer \
<ul>\
<li>Showing of broken links if built with gnome-vfs support.\
<li>Zoomable. \
<li>Selectable depth of links to show. \
</ul>\
</ul>"), PACKAGE, VERSION );
}

/*
 * screem_preview_display:
 *
 * displays the given text
 *
 * return values: none
 */
void screem_preview_display( const gchar *text )
{
	gboolean free_me = FALSE;
#ifndef HAVE_GTKHTML
	if( text == NULL ) {
		text = screem_load_splash();
		free_me = TRUE;
	}

	gtk_xmhtml_freeze( GTK_XMHTML( display ) );
	gtk_xmhtml_source( GTK_XMHTML( display ), text );
	gtk_xmhtml_thaw( GTK_XMHTML( display ) );
#else
	GtkHTMLStream *handle;

	if( text == NULL ) {
		text = screem_load_splash();
		free_me = TRUE;
	}

	gtk_html_set_editable( GTK_HTML( display ), FALSE );

	handle = gtk_html_begin( GTK_HTML( display ) );
	gtk_html_write( GTK_HTML( display ), handle, text, strlen( text ) );
	gtk_html_end( GTK_HTML( display ), handle, GTK_HTML_STREAM_OK );

#if 0
	/* FIXME: disabled until editing is more stable in the widget */
	gtk_html_set_editable( GTK_HTML( display ), TRUE );
#endif

#endif
	if( free_me )
		g_free( (gchar*)text );
}


/*
 * screem_preview_update :
 *
 * updates the preview, keeping the window in the same position
 *
 * return values: none
 */
void screem_preview_update( const gchar *text )
{
	screem_preview_display( text );
#if 0
	GtkAdjustment *vadj;
	GtkAdjustment *hadj;
	gfloat vval;
	gfloat hval;

     
#ifndef HAVE_GTKHTML
	vadj = GTK_ADJUSTMENT( GTK_XMHTML( display )->vsba );
	hadj = GTK_ADJUSTMENT( GTK_XMHTML( display )->hsba );
#else
	vadj = GTK_ADJUSTMENT( GTK_LAYOUT( display )->vadjustment );
	hadj = GTK_ADJUSTMENT( GTK_LAYOUT( display )->hadjustment );
#endif
	vval = vadj->value;
	hval = hadj->value;

	screem_preview_display( text );

	gtk_adjustment_set_value( vadj, vval );
	gtk_adjustment_set_value( hadj, hval );
#endif
}

GtkWidget *screem_preview_get_widget()
{
	return display;
}


static void url_clicked( GtkWidget *widget, const gchar *url, gboolean help )
{
	const gchar *site_url;
	Page *page;

	gchar *protocol;
	gchar *address;
	gchar *port;
	gchar *path;
	gchar *full;

	screem_html_parse_uri( url, &protocol, &address, &port, &path );

	if( ! help ) {
		site_url = screem_site_get_http_url( current_site );
		if( address && site_url && strcmp( address, site_url ) ) {
			/* its not a link in this site */
			g_free( path );
			g_free( protocol );
			g_free( address );
			g_free( port );
			return;
		}
	
		/* convert path to a complete pathname */
		full = relative_to_full( path );
		
		page = screem_site_locate_page( current_site, full );
		if( page ) {
			/* switch to it! */
			screem_page_insert( page );
		}
	} else {
		full = g_strconcat( help_dir, G_DIR_SEPARATOR_S, path, NULL );
		g_print( "help file: %s\n", full );
		screem_help_display( full );
	}

	g_free( full );
	g_free( protocol );
	g_free( address );
	g_free( port );
	g_free( path );
}

gint screem_preview_get_width()
{
	if( ! display )
                return cfg->preview_width;

        return display->allocation.width;
}

gint screem_preview_get_height()
{
	if( ! display )
                return cfg->preview_height;

        return display->allocation.height;
}


/****************************************************************************
 * help browser
 */
GtkWidget *screem_help_new()
{
	GtkWidget *preview;

    	preview = screem_preview( TRUE );

	gtk_widget_set_usize(preview, cfg->preview_width, cfg->preview_height);

	return preview;
}

void screem_help_display( const gchar *file )
{
	gpointer handle = NULL;
	gchar *dir;
	gchar cwd[ 16384 ];

	dir = g_dirname( file );

	getcwd( cwd, 16384 );
	chdir( dir );

	if( help_dir )
		g_free( help_dir );

	help_dir = dir;

#ifdef HAVE_GTKHTML
	handle = gtk_html_begin( GTK_HTML( helpdis ) );
#endif
	url_requested( helpdis, file, handle );

	chdir( cwd );
}

void screem_help_show( gchar *tag )
{
	gchar *path;

	path = screem_help_find_path( tag );

	if( path ) {
		screem_help_display( path );
		switch_to( NULL, "3" );
		g_free( path );
        }
}

gchar* screem_help_find_path( gchar *tag  )
{
	gchar *file;
	gchar *path;
      	struct dirent **namelist;
	gint n;
	gint i;
	gboolean found;
	const gchar *dirs[] = {
		"block",
		"fontstyle",
		"forms",
		"frames",
		"head",
		"html",
		"lists",
		"phrase",
		"special",
		"tables",
		NULL
	};

	if( tag[ 0 ] == '!' )
		tag ++;

	file = g_strconcat( tag, ".html", NULL );

     	for( found = FALSE, i = 0; dirs[ i ]; i ++ ) {
		path = g_strconcat( HELP_PATH, G_DIR_SEPARATOR_S,
				    dirs[ i ], NULL );
		n = scandir( path, &namelist, 0, alphasort );
		g_free( path );
		while( n -- ) {
			if( ! found )
				found = ( !strcmp( namelist[n]->d_name, file ) );
			free( namelist[ n ] );
		}
	       
		if( found )
			break;
	}
	if( found )
		path = g_strconcat( HELP_PATH, G_DIR_SEPARATOR_S, 
				    dirs[ i ], G_DIR_SEPARATOR_S, 
				    file, NULL );
	else
		path = NULL;

	g_free( file );

	return path;
}
