/*  Screem:  dtd.c,
 *  DTD parsing code, and utility functions which access the parsed data 
 *
 *
 *  Copyright (C) 1999, 2000  David A Knight
 *
 *  TODO:
 *  look in more paths for the dtd catalog file
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <ctype.h>
#include <gnome.h>
#include <stdio.h>
#include <string.h>
#include <dirent.h>
#include <sys/types.h>
#include <unistd.h>

#include <glade/glade.h>

#ifdef HAVE_GNOME_VFS
#include <libgnomevfs/gnome-vfs.h>
#endif

#include "dtd.h"
#include "htmlfuncs.h"
#include "preferences.h"
#include "support.h"

extern Preferences *cfg;

static GHashTable *dtd_types = NULL;
static GHashTable *loaded_dtds = NULL;		

static void dtds_insert( gpointer key, gpointer value, GtkWidget *list );

static void screem_dtd_remove_allowed( ScreemDTDAllow *allow, GString *tag );

static gchar * screem_dtd_allowed( ScreemDTD *dtd, gchar *tag, 
				   ScreemDTDAllow *allow,
				   GList *list, gint level );

static void screem_dtd_insert_markup( gchar *tag );

extern void screem_editor_insert_attribute( gchar *attribute );
extern void screem_editor_insert_markup( gchar *opentag, gchar *closetag );

ScreemDTD* screem_dtd_new()
{
	ScreemDTD *dtd;

	dtd = g_new0( ScreemDTD, 1 );

	dtd->elements = g_hash_table_new( g_str_hash, g_str_equal );
	dtd->entities = g_hash_table_new( g_str_hash, g_str_equal );

	return dtd;
}

void screem_dtd_destroy( ScreemDTD *dtd )
{
        g_hash_table_destroy( dtd->elements );
        g_hash_table_destroy( dtd->entities );

	g_free( dtd );
}

ScreemDTD* screem_dtd_load_doctype( const gchar *tag )
{
	GString *type;
	GString *url;
	ScreemDTD *dtd;

	if( ! tag )
		return NULL;

	tag = strchr( tag, '"' );

	if( ! tag )
		return NULL;

	type = g_string_new( "" );
	for( tag ++; *tag != '\0' && *tag != '"'; tag ++ )
		g_string_append_c( type, *tag );

	url = g_string_new( "" );
	if( *tag == '"' && ( tag = strchr( ++tag, '"' ) ) ) {
		for( tag ++; *tag != '\0' && *tag != '"'; tag ++ )
			g_string_append_c( url, *tag );
	}
       
	if( ! ( dtd = screem_dtd_load_type( type->str ) ) ) {
		/* we couldn't find the DTD locally */
		dtd = screem_dtd_load_remote( url->str, type->str );
	}

	if( dtd )
		dtd->url = url->str;
	
	g_string_free( url, !dtd );
	
	g_string_free( type, TRUE );

     	return dtd;
}

/* does nothing if we don't have gnome-vfs support */
ScreemDTD* screem_dtd_load_remote( const gchar *url, const gchar *type )
{
#ifdef HAVE_GNOME_VFS
	GnomeVFSHandle *handle;
	GnomeVFSResult result;

	GnomeVFSURI *uri;
	
	gchar *buffer;
	GnomeVFSFileSize red;
	GString *tmp;

	ScreemDTD *dtd;

	gchar *catalog;
	gchar *path;
	int fd;
	FILE *out;

	g_return_val_if_fail( url != NULL, NULL );
	g_return_val_if_fail( type != NULL, NULL );

	uri = gnome_vfs_uri_new( url );
	if( ! uri ) {
		/* hmm, we failed to create a uri */
		return NULL;
	}
	result = gnome_vfs_open_uri( &handle, uri, GNOME_VFS_OPEN_READ );

	if( result != GNOME_VFS_OK ) {
		gnome_vfs_uri_unref( uri );
		return NULL;
	}

	/* read the file */
	buffer = g_new0( gchar, BUFSIZ );
	tmp = g_string_new( "" );
	while( gnome_vfs_read(handle, buffer, BUFSIZ, &red) == GNOME_VFS_OK ) {
		g_string_append( tmp, buffer );
		memset( buffer, '\0', BUFSIZ );
	}
	g_string_append( tmp, buffer );

	gnome_vfs_close( handle );
	gnome_vfs_uri_unref( uri );

	/* we have the DTD, save it in ~/.screem/dtds/ and append
	   it to the users catalog, then add it to the hash table,
	   otherwise we will create a lot of network traffic fetching it
	   all the time */
	path = g_strconcat( g_get_home_dir(), G_DIR_SEPARATOR_S,
			    ".screem", G_DIR_SEPARATOR_S,
			    "dtds", G_DIR_SEPARATOR_S,
			    "XXXXXX", NULL );
	fd = mkstemp( path );
	out = fdopen( fd, "w" );
	if( out ) {
		fwrite( tmp->str, strlen( tmp->str ), 1, out );
		fclose( out );
	
		catalog = g_strconcat( g_get_home_dir(), G_DIR_SEPARATOR_S,
				       ".screem", G_DIR_SEPARATOR_S,
				       "dtds", G_DIR_SEPARATOR_S, "catalog", 
				       NULL );
		out = fopen( catalog, "a" );
		if( out ) {
			fprintf( out, "-- Added by Screem --\n" );
			fprintf( out, "PUBLIC \"%s\"\t\"%s\"\n", type, path );
			fclose( out );

			/* now add it to the hash table */
			g_hash_table_insert( dtd_types, 
					     g_strdup( type ),
					     g_strdup( path ) );

		}
		g_free( catalog );
	}
	g_free( path );

	dtd = screem_dtd_parse( tmp );

	g_string_free( tmp, TRUE );

	return dtd;
#else
	return NULL;
#endif
}

ScreemDTD* screem_dtd_load_type( const gchar *type )
{
	gchar *path;
	ScreemDTD *dtd;

	/* perform lookup */
	path = (gchar*)g_hash_table_lookup( dtd_types, type );

	if( ! path )
		return NULL;
       
	/* we have a path, perform lookup in loaded DTDs table */
	dtd = (ScreemDTD*)g_hash_table_lookup( loaded_dtds, path );
	if( ! dtd ) {
		/* it wasn't loaded, lets load it */
		dtd = screem_dtd_load( path );

		/* add to hash table */
		if( dtd )
			g_hash_table_insert( loaded_dtds, path, dtd );
	}

	return dtd;
}

ScreemDTD* screem_dtd_load( const gchar *filename )
{
	GString *dtd;

	ScreemDTD *sdtd;

	dtd = load_text_file( filename );

	sdtd = screem_dtd_parse( dtd );

	g_string_free( dtd, TRUE );

	return sdtd;
}

ScreemDTD* screem_dtd_parse( GString *dtd_data )
{
	gchar *data;
	gint pos;
	gint pos2;
	gchar *tag;

	gchar *name;
	GString *full;
	GString *extra;

	ScreemDTD *dtd;
	ScreemDTDEntity *ent;

	data = dtd_data->str;
	pos = 0;

	dtd = screem_dtd_new();

	while( data && data[ pos ] != '\0' ) {
		if( data[ pos ] == '%' ) {
			/* its an ENTITY that needs to be expanded, and
			   parsed as part of the DTD, this is needed for
			   DTD's such as -//W3C//DTD HTML 4.01 Frameset//EN
			*/
			full = g_string_new( "" );
			pos ++;
			while( data[ pos ] != ';' && data[ pos ] != '\0' )
				g_string_append_c( full, data[ pos ++ ] );
		
			if( ( ent = (ScreemDTDEntity*)
			      g_hash_table_lookup( dtd->entities, 
						   full->str ) ) ) {
				/* we have the expanded entity, get the
				   pathname */
				name = (gchar*)g_hash_table_lookup(dtd_types, 
								   ent->value);
				if( ! name ) {
					/* we couldn't find it
					   FIXME: should fetch it */
					g_string_free( full, TRUE );
					continue;
				}
				extra = load_text_file( name );
				/* loaded, insert into the data string
				   so it can be parsed */
				if( data[ pos ] != '\0' )
					pos ++;
				g_string_insert( dtd_data, pos, extra->str );
				g_string_free( extra, TRUE );
				data = dtd_data->str;
			}
			g_string_free( full, TRUE );

		} else if( data[ pos ] == '<' ) {
			/* we have a tag */
			pos2 = pos;
			if( ! ( tag = next_tag( &data[ pos ], &pos, NULL ) ) ){
				break;
			}
			
			if( ! strncmp( "<!--", tag, strlen( "<!--" ) ) ) {
				/* comment, we ignore it */
			} else if( ! g_strncasecmp( "<!ENTITY", tag, 
						    strlen( "<!ENTITY" ) ) ) {
				/* its an ENTITY */
				screem_dtd_parse_entity( dtd, tag );
			} else if( ! g_strncasecmp( "<!ELEMENT", tag, 
						    strlen( "<!ELEMENT" ) ) ) {
				/* its an ELEMENT */
				screem_dtd_parse_element( dtd, tag );
			} else if( ! g_strncasecmp( "<!ATTLIST", tag, 
						    strlen( "<!ATTLIST" ) ) ) {
				/* ATTLIST for previous ELEMENT */
				screem_dtd_parse_attlist( dtd, tag );
			} else if( ! g_strncasecmp( "<![", tag, 
						    strlen( "<![" ) ) ) {
				/* feature switch */
				name = tag + strlen( "<![ %" );
				full = g_string_new( "" );
				while( *name != ';' && *name != '\0' ) {
					g_string_append_c( full, *name );
					name ++;
				}
				if( ( ( ent = (ScreemDTDEntity*)
					g_hash_table_lookup( dtd->entities,  full->str) ) && ! strcmp( "IGNORE", ent->value ) ) || ! ent ) {
					/* entity not defined, or ignored
					   so feature not switched on */
					while( data[ pos ] != '\0' &&
					       ! ( data[ pos ] == ']' &&
						   data[ pos + 1 ] == '>' ) ) {
						pos ++;
					}
				} else {
					while( data[ pos ] != '[' )
						pos --;
				}
				g_string_free( full, TRUE );
			}
			g_free( tag );
		} else
			pos ++;
	}

	return dtd;
}

void screem_dtd_parse_entity( ScreemDTD *dtd, gchar *tag )
{
	GString *name;
	GString *value;
	gchar term;
	ScreemDTDEntity *entity;

	tag += strlen( "<!ENTITY % " );

	name = g_string_new( "" );
	while( *tag != '\0' && ! isspace( *tag ) ) {
		g_string_append_c( name, *tag );
		tag ++;
	}

	term = '\0';
      	while( *tag != '\0' ) {
		term = *tag;
		if( term == '"' || term == '\'' )
			break;
		tag ++;
	}
	value = g_string_new( "" );
	tag ++;
	while( *tag != '\0' && *tag != term ) {
		g_string_append_c( value, *tag );
		tag ++;
	}

	/* add name to hash table, only if we don't already have
	   an entity with a matching name */
	if( ! g_hash_table_lookup( dtd->entities, name->str ) ) {
		entity = g_new0( ScreemDTDEntity, 1 );
		entity->name = name->str;
		if( ! value->str ) {
			g_string_free( name, TRUE );
			g_string_free( value, TRUE );
		} else {
			entity->value = g_strchug( value->str );
			g_hash_table_insert( dtd->entities, name->str, entity );
			g_string_free( name, FALSE );
			g_string_free( value, FALSE );
		}
	} else {
		g_string_free( name, TRUE );
		g_string_free( value, TRUE );
	}
}

void screem_dtd_parse_element( ScreemDTD *dtd, gchar *tag )
{
	GString *name;
	gchar **split;
	gint i;
   	ScreemDTDElement *element;

	tag += strlen( "<!ELEMENT " );

	name = g_string_new( "" );
	tag = screem_dtd_parse_name( dtd, tag, name );

	if( ! strlen( name->str ) ) {
		/* discard this element as we didn't find a name */
		g_string_free( name, TRUE );
		return;
	}

	/* handle the name */
	split = g_strsplit( name->str, "|", 1024 );
	g_string_free( name, TRUE );

	i = 0;
	element = NULL;

	if( ( element = (ScreemDTDElement*)
	      g_hash_table_lookup( dtd->elements, split[ 0 ] ) ) ) {
		g_strfreev( split );
		return;
	}

	for( i = 0; split[ i ]; i ++ ) {
		element = g_new0( ScreemDTDElement, 1 );
		element->name = g_strdup( split[ i ] );
		g_assert( element->name );
		element->name = g_strchug( element->name );
		element->name = g_strchomp( element->name );
		if( ! strlen( element->name ) ) {
			g_free( element->name );
			g_free( element );
			continue;
		}
		element->can_close = TRUE;
		element->must_close = TRUE;
		g_hash_table_insert( dtd->elements, element->name, element );
	}
	g_strfreev( split );

	/* get allowed sub elements */
      	tag = screem_dtd_parse_allowed_elements( dtd, element, tag );

}

void screem_dtd_parse_attlist( ScreemDTD *dtd, gchar *tag )
{
	GString *name;
	gchar **split;
	gint i;
	gchar *aname;
	GList *list;
	ScreemDTDElement *element;

	tag += strlen( "<!ATTLIST " );
	name = g_string_new( "" );
	tag = screem_dtd_parse_name( dtd, tag, name );

	if( ! strlen( name->str ) ) {
		/* discard this element as we didn't find a name */
		g_string_free( name, TRUE );
		return;
	}

	list = screem_dtd_parse_attribute( dtd, tag, NULL );
	split = g_strsplit( name->str, "|", 1024 );

	for( i = 0; split[ i ]; i ++ ) {
		aname = g_strdup( split[ i ] );
		if( ! strlen( aname ) ) {
			g_free( aname );
			continue;
		}
		g_assert( aname );
		aname = g_strchug( aname );
		aname = g_strchomp( aname );
		if( ! strlen( aname ) ) {
			g_free( aname );
			continue;
		}
		/* add the attributes to the element matching aname */
		element = (ScreemDTDElement*)
			g_hash_table_lookup( dtd->elements, aname );
		if( element ) {
			element->attributes = list;
		} else {
			g_print( "\
Screem DTD parsing: Unknown element (%s) for ATTLIST\n", aname );
		}
		g_free( aname );
	}
	g_strfreev( split );

	g_string_free( name, TRUE );
}

GList* screem_dtd_parse_attribute( ScreemDTD *dtd, gchar *tag, GList *list )
{
	GString *att_name;
	GString *att_value;
	GString *defval;

	gint state;
	gboolean isent;
	gboolean start;

	ScreemDTDAttribute *attr;
	ScreemDTDEntity *ent;
	gchar *temp;
	gchar term;

	att_name = g_string_new( "" );
	att_value = g_string_new( "" );
	defval = g_string_new( "" );

	start = TRUE;
	state = ANEXT;
	isent = FALSE;
	g_assert( tag );
     	tag = g_strchug( tag );
	while( tag && *tag != '\0' ) {
		switch( state ) {
		case ANAME:
			if( isent && *tag != ';' ) {
				g_string_append_c( att_name, *tag );
				tag ++;
				break;
			} else if( ! isent && ! isspace( *tag ) ) {
				g_string_append_c( att_name, *tag );
				tag ++;
				break;
			}
			/* end of name */
			state = AVAL;
			
			/* is the name an entity? */
			if( isent ) {
				tag ++;
				ent = (ScreemDTDEntity*)
					g_hash_table_lookup( dtd->entities,
							     att_name->str );
				state = ANEXT;
				if( ent ) {
					/* entity found */
					temp = g_strdup( ent->value );
					list =screem_dtd_parse_attribute(dtd,
									 temp,
									 list);
					g_free( temp );
				} else
					g_print( "couldn't find entity for %s\n", att_name->str );
			}
			g_assert( tag );
			tag = g_strchug( tag );
			break;
		case AVAL:
			tag = screem_dtd_parse_name( dtd, tag, att_value );
			state = ADEF;
			if( tag )
				tag = g_strchug( tag );
			break;
		case ADEF:
			if( ! isspace( *tag ) ) {
				g_string_append_c( defval, *tag );
				tag ++;
				break;
			} else if( ! strcmp( "#FIXED", defval->str ) ) {
				/* the default value is a fixed one,
				   change att_value to be the next thing
				   we read in as it is the only
				   value allowed */
				g_assert( tag );
				tag = g_strchug( tag );
				term = *tag;
				tag ++;
				g_string_assign( att_value, "" );
				while( *tag != '\0' && *tag != term ) {
					g_string_append_c( att_value, *tag );
					tag ++;	
				}
				if( *tag == term )
					tag ++;
			}
			/* end of default, move to next */
			state = ANEXT;
			g_assert( tag );
			tag = g_strchug( tag );
			break;
		case ANEXT:
			if( ! start && ! isent ) {
				/* build the attribute now */
				attr = g_new0( ScreemDTDAttribute, 1 );
				attr->name = g_strdup( att_name->str );
				attr->defval = g_strdup( defval->str );
				attr->values = g_strsplit( att_value->str,
							   "|", 1024 );
				list = g_list_append( list, attr );
				start = TRUE;
			}
			if( *tag == '-' && *(tag + 1) =='-' ) {
				tag = strstr( tag + 2, "--" );
				if( ! tag )
					break;
				tag += 2;
				g_assert( tag );
				tag = g_strchug( tag );
			} else {
				start = FALSE;
				state = ANAME;
				g_string_assign( att_name, "" );
				g_string_assign( att_value, "" );
				g_string_assign( defval, "" );
				if( ( isent = ( *tag == '%' ) ) )
					tag ++;
			}
			break;
		}
	}

	return list;
}

gchar* screem_dtd_parse_name( ScreemDTD *dtd, gchar *tag, GString *full )
{
	gchar *temp;
	gint state;
	gboolean stop;
	GString *sub;
	ScreemDTDEntity *ent;

	/* skip white space at start */
	g_assert( tag );
	temp = g_strchug( tag );

	sub = NULL;

	switch( *temp ) {
	case '(':
		state = EXPR;
		temp ++;
		break;
	case '%':
		state = ENTITY;
		temp ++;
		sub = g_string_new( "" );
		break;
	default:
		state = NAME;
		break;
	}

	stop = FALSE;
	while( ! stop && *temp != '\0' && *temp != '>' ) {
		/* bypass comments */
		if( *temp == '-' && *(temp + 1) == '-' ) {
			temp += 2;
			temp = strstr( temp, "--" );
			if( ! temp )
				break;
			else
				temp += strlen( "--" );
		}
		switch( state ) {
		case NAME:
			if( *temp == '(' ) {
				sub = g_string_new( "" );
				temp = screem_dtd_parse_name( dtd, temp, sub );
				g_string_append( full, sub->str );
			} else if( *temp == '%' ) {
				sub = g_string_new( "" );
				temp = screem_dtd_parse_name( dtd, temp, sub );
				g_string_append( full, sub->str );
			} else if( ! isspace( *temp ) && *temp != '>' )
				g_string_append_c( full, *temp );
			else
				stop = TRUE;
			break;
		case ENTITY:
			if( *temp != ';' )
				g_string_append_c( sub, *temp );
			else {
				/* do lookup */
				ent = (ScreemDTDEntity*)
					g_hash_table_lookup( dtd->entities, 
							     sub->str );
				g_string_assign( sub, "" );
				/* entity expansion may contain other
				   entities */
				if( *ent->value != '(' )
					tag = g_strconcat( "(", ent->value, 
							   ")",NULL );
				else
					tag = g_strdup( ent->value );
				screem_dtd_parse_name( dtd, tag, sub );
				g_free( tag );
				g_string_append_c( full, '|' );
				g_string_append( full, sub->str );
				g_string_free( sub, TRUE );
				stop = TRUE;
			}
			break;
		case EXPR:
			switch( *temp ) {
			case ')':
				stop = TRUE;
				break;
			case '%':
				temp = screem_dtd_parse_name( dtd,temp, full );
				temp --;
				break;
			default:
				g_string_append_c( full, *temp );
				break;
			}
			break;
		}
		temp ++;
	}
	return temp;
}

static void screem_dtd_remove_allowed( ScreemDTDAllow *allow, GString *tag )
{
	GList *list;

	if( ! allow || ! tag )
		return;

	while( allow->prev )
		allow = allow->prev;
	
	while( allow ) {
		for( list = allow->elements; list; list = list->next ) {
			if( ! strcmp( tag->str, (gchar*)list->data ) ) {
				allow->elements = g_list_remove_link( allow->elements, list );
			}
		}
		allow = allow->next;
	}
}

static gchar * screem_dtd_allowed( ScreemDTD *dtd, gchar *tag, 
				   ScreemDTDAllow *allow,
				   GList *list, gint level )
{
	GString *name = g_string_new( "" );
	ScreemDTDEntity *ent;
	gboolean inent = FALSE;
	static gboolean no = FALSE;

	if( ! allow )
		return NULL;

	if( level == 0 )
		no = FALSE;

	while( tag && *tag != '>' ) {
		switch( *tag ) {
		case '-':
			if( *(tag  + 1) == '-' ) {
				tag = strstr( tag + 2, "--" );
				if( tag )
					tag += 2;
			} else {
				/* don't add the next ones to the list of
				   allowed elements */
				no = TRUE;
			}
			break;
		case '(':
			if( level == 0 && ! no ) {
				allow->next = g_new0( ScreemDTDAllow, 1 );
				allow->next->prev = allow;
				tag = screem_dtd_allowed( dtd, tag + 1, 
							  allow->next, 
							  allow->elements, 
							  level + 1 );
			} else {
				tag = screem_dtd_allowed( dtd, tag + 1, allow, 
							  allow->elements, 
							  level + 1 );
			}
			break;
		case '%':
			/* entity */
			inent = TRUE;
			break;
		case ';':
			/* if we aren't in an entity, ie there is a rogue ';' 
			   then do nothing */
			if( ! inent )
				break;
			
			/* end entity, expand it and get the ScreemDTDAllow 
			   list for it */
			ent = (ScreemDTDEntity*)
				g_hash_table_lookup( dtd->entities,name->str );
			if( ent ) {
				screem_dtd_allowed( dtd, ent->value, 
						    allow, allow->elements,
						    level );
			}
			g_string_assign( name, "" );
			inent = FALSE;
			break;
		case '*':
			allow->number = -1;
			/* as this ends an expression block we return */
			if( level == 0 ) {
				return tag + 1;
			} else {
				allow->next = g_new0( ScreemDTDAllow, 1 );
				allow->next->prev = allow;
				allow = allow->next;
			}
			break;
		case '+':
			if( strlen( name->str ) > 0 )
				allow->number = 0;

			if( level == 0 ) {
				return tag + 1;
			} else {
				allow->next = g_new0( ScreemDTDAllow, 1 );
				allow->next->prev = allow;
				allow = allow->next;
			}
			break;
		case '?':
			allow->number = 1;
			/* as this ends an expression block we return */
			if( level == 0 ) {
				return tag + 1;
			} else {
				allow->next = g_new0( ScreemDTDAllow, 1 );
				allow->next->prev = allow;
				allow = allow->next;
			}
			break;
		case '\0':
		case ')':
			/* end of an expression */
		case ' ':
		case '|':
		case ',':
			if( no && ( strlen( name->str ) > 0 ) ) {
				/* remove this word */
				screem_dtd_remove_allowed( allow, name );
				g_string_assign( name, "" );
			} else if( strlen( name->str ) > 0 ) {
				/* end of element name */
				if( strcmp( "EMPTY", name->str ) )
					allow->elements = g_list_append( allow->elements, g_strdup( name->str ) );
				g_string_assign( name, "" );
			}
			if( level == 0 )
				no = FALSE;
			break;
		default:
			g_string_append_c( name, *tag );
			break;
		}
		if( *tag == '\0' )
			break;
		if( tag )
			tag ++;
	}

	return tag;
}

gchar* screem_dtd_parse_allowed_elements( ScreemDTD *dtd, 
					  ScreemDTDElement *element,
					  gchar *tag )
{
	/* get the allowed elements, if the open tag is
	   required/optional, and if the close tag is 
	   required/optional/forbidden */
	ScreemDTDAllow *allow;

	allow = g_new0( ScreemDTDAllow, 1 );
	g_assert( tag );
	tag = g_strchug( tag );

	if( ! strncmp( "O O ", tag, strlen( "O O " ) ) ) {
		element->must_close = FALSE;
	} else if( ! strncmp( "- O EMPTY", tag, strlen( "- O EMPTY" ) )
		   || ! strncmp( "EMPTY", tag, strlen( "EMPTY" ) ) ) {
		element->can_close = FALSE;
		element->must_close = FALSE;
	}


	while( tag && *tag != '\0' && *tag != '>' )
		tag = screem_dtd_allowed( dtd, tag, allow, allow->elements, 0);
       
	element->allow = allow;

	return tag;
}

/****************************************************************************
 * End of parsing section                                                   *
 ****************************************************************************/

CloseState screem_dtd_element_get_close_state( ScreemDTD *dtd, gchar *name )
{
	ScreemDTDElement *element;
	GString *hack;

	if( ! dtd ) {
		dtd = screem_dtd_load_type( cfg->mpage->default_dtd );
		if( ! dtd )
			return REQUIRED;
	}

	if( ! name )
		return REQUIRED;

	element = screem_dtd_check_element( dtd, name );

	if( ! element )
		return REQUIRED;
	
	if( element->must_close )
		return REQUIRED;
	else if( element->can_close )
		return OPTIONAL;
	else
		return FORBIDDEN;
}

GList* screem_dtd_element_get_attributes( ScreemDTD *dtd, gchar *name )
{
	ScreemDTDElement *element;
	GString *hack;

	if( ! dtd ) {
		dtd = screem_dtd_load_type( cfg->mpage->default_dtd );
		if( ! dtd )
			return NULL;
	}

	if( ! name )
		return NULL;

	element = screem_dtd_check_element( dtd, name );

	if( ! element )
		return NULL;
       
	return element->attributes;
}


void screem_dtd_load_catalog()
{
	gchar *paths[] = {
		SCREEM_DTD_PATH,
		"/usr/lib/sgml",                      /* Debian, RedHat */
		"/usr/share/sgml",                    /* SuSE */
		"/usr/share/lib/sgml/locale/C/dtds",  /* Solaris, use the 
							 current locale at 
							 runtime? */
		NULL,                                 /* replaced with the
							 users .screem dir at
							 runtime */
		NULL
	};

	gint i;
	gchar *buffer;
	FILE *file;

	gchar *id;
	gchar *path;

	gchar *tid;
	gchar *tpath;
	gint ret;

	gchar cwd[ 16384 ];
	DIR *dir;
	struct dirent *entry;

	paths[ 4 ] = g_strconcat( g_get_home_dir(), G_DIR_SEPARATOR_S,
				  ".screem", G_DIR_SEPARATOR_S,
				  "dtds", NULL );

	dtd_types = g_hash_table_new( g_str_hash, g_str_equal );
	loaded_dtds = g_hash_table_new( g_str_hash, g_str_equal );

	buffer = g_new0( gchar, BUFSIZ );
	id = g_new0( gchar, BUFSIZ );
	path = g_new0( gchar, BUFSIZ );

	dir = NULL;
	getcwd( cwd, 16384 );
	i = 0;

	while( paths[ i ] ) {
		if( ! dir ) {
			if( ( dir = opendir( paths[ i ] ) ) )
				chdir( paths[ i ] );
			else {
				i ++;
				continue;
			}
		}

		/* get next file */
		while( ( entry = readdir( dir ) ) ) {
			if( ( ! g_strcasecmp( "catalog", entry->d_name ) ) ||
			    ( ! g_strncasecmp( "catalog.", 
					       entry->d_name, 
					       strlen( "catalog." ) ) ) )
				break;  /* its a catalog file */
		}
		if( ! entry ) {
			closedir( dir );
			dir = NULL;
			i ++;
			continue;
		}

		/* we have a catalog file */
		file = fopen( entry->d_name, "r" );
		if( ! file ) {
			/* this should not actually occur, but its best to 
			   check anyway */
			continue;
		}

		while( fgets( buffer, BUFSIZ, file ) ) {
			g_assert( buffer );
			buffer = g_strchug( buffer );
			if( strncmp( "PUBLIC ", buffer, strlen( "PUBLIC " ) ) )
				continue;
			ret = sscanf( buffer, "PUBLIC \"%[^\"]\" \"%[^\"]\"", 
				      id, path );
			if( ! ret )
				break;
			else if( ret == 1 ) {
				/* ok try a different match */
				if( sscanf( buffer, "PUBLIC \"%[^\"]\" %s", 
					    id, path ) != 2 )
					break;
			}
			
			/* is id already in the hash table? */
			if( g_hash_table_lookup( dtd_types, id ) ) {
				/* yes it is, we ignore this one then */
				g_print( "Screem: multiple catalog entries for %s\n", id );
				g_print( "Screem: This may be due to the Screem provided DTDs\n" );
			} else {
				tid = g_strdup( id );
				if( ! g_path_is_absolute( path ) )
					tpath = g_strconcat( paths[ i ],
							     G_DIR_SEPARATOR_S,
							     path, NULL );
				else
					tpath = g_strdup( path );
				
				g_hash_table_insert( dtd_types, tid, tpath );
			}
		}
		fclose( file );
	}

	chdir( cwd );
	
	g_free( buffer );
	g_free( id );
	g_free( path );

	g_free( paths[ 4 ] );
}

GtkWidget *screem_dtd_build_attribute_menu( ScreemDTD *dtd, gchar *name,
					    gboolean sub )
{
	ScreemDTDElement *element;
	ScreemDTDAttribute *attribute;
	GtkWidget *item;
	GtkWidget *menu;
	GtkWidget *subitem;
	GtkWidget *submenu;
	GtkWidget *valitem;
	GList *list;
	gchar *title;
	gint i;
	gint j;
	gchar *value;
	gchar *temp;

	gboolean single;
	GString *hack;

	gint num;
	
	if( ! name )
		return NULL;

	if( ! dtd ) {
		dtd = screem_dtd_load_type( cfg->mpage->default_dtd );
		if( ! dtd )
			return NULL;
	}

	element = screem_dtd_check_element( dtd, name );

	if( ! element )
		return NULL;

	/* we have the element */
	menu = gtk_menu_new();

	if( sub ) {
		title = g_strdup_printf( "<%s>", name );
		item = gtk_menu_item_new_with_label( title );
		gtk_widget_show( item );
       		gtk_menu_item_set_submenu( GTK_MENU_ITEM( item ), menu );
	} else
		item = menu;

	for( num = 0, list = element->attributes; list; list = list->next ) {
		attribute = (ScreemDTDAttribute*) list->data;
		
		if( ++ num > 10 ) {
			subitem = gtk_menu_item_new_with_label( _( "More" ) );
			gtk_widget_show( subitem );
			gtk_menu_prepend( GTK_MENU( menu ), subitem );
			menu = gtk_menu_new();
			gtk_menu_item_set_submenu( GTK_MENU_ITEM( subitem ), 
						   menu );
			num = 0;
		}

		/* add the attribute name */
		subitem = gtk_menu_item_new_with_label( attribute->name );
		gtk_widget_show( subitem );
		gtk_menu_append( GTK_MENU( menu ), subitem );

		/* add the attributes into a submenu */
		submenu = NULL;
		single = FALSE;
		for( i = 0; attribute->values[ i ]; i ++ ) {
			value = g_strdup( attribute->values[ i ] );
			g_assert( value );
			value = g_strchug( value );
			value = g_strchomp( value );
			if( ! strlen( value ) ) {
				g_free( value );
				continue;
			}
			/* only add if it is a fixed set of values that can
			   be entered */
			if( ! strcmp( "NUMBER", value ) ) {
				if( ! submenu )
					submenu = gtk_menu_new();
				for( j = 0; j != 10; j ++ ) {
					temp = g_strdup_printf( "%i", j );
					valitem = gtk_menu_item_new_with_label(temp );	
					gtk_widget_show( valitem );
					gtk_menu_append( GTK_MENU( submenu ), 
							 valitem );
					/* connect activate signal */
					g_free( temp );
					temp = g_strdup_printf("%s=\"%i\"",
							       attribute->name,
							       j );
					gtk_signal_connect_full( GTK_OBJECT( valitem ),
								 "activate",
								 GTK_SIGNAL_FUNC( screem_editor_insert_attribute ),
								 NULL, temp,
								 g_free, 1, 0 );
				}
			} else if( ! strcmp( value, attribute->name ) ) {
				/* hmm, same name as attribute, assume that
				   it has no real values then, this may be
				   the wrong thing to do but it appears
				   correct */
				single = TRUE;
				temp = g_strdup( attribute->name );
				gtk_signal_connect_full( GTK_OBJECT( subitem ),
							 "activate",
							 GTK_SIGNAL_FUNC( screem_editor_insert_attribute ),
							 NULL, temp,
							 g_free, 1, 0 );
			} else if( strcmp( "CDATA", value ) &&
			    strcmp( "NAME", value ) &&
			    strcmp( "ID", value ) &&
			    strcmp( "#PCDATA", value ) ) {
				valitem = gtk_menu_item_new_with_label( value);
				gtk_widget_show( valitem );
				if( ! submenu )
					submenu = gtk_menu_new();
				gtk_menu_append( GTK_MENU( submenu ), valitem);
				/* connect activate signal */
				temp = g_strdup_printf( "%s=\"%s\"",
							attribute->name,
							value );
				gtk_signal_connect_full( GTK_OBJECT( valitem ),
							 "activate",
							 GTK_SIGNAL_FUNC( screem_editor_insert_attribute ),
							 NULL, temp,
							 g_free, 1, 0 );
			}
			g_free( value );
		}
		if( submenu )
			gtk_menu_item_set_submenu( GTK_MENU_ITEM( subitem ),
						   submenu );
		else if( ! single ) {
			/* no submenu, connect activate signal */
			temp = g_strdup_printf( "%s=\"\"", attribute->name );
			gtk_signal_connect_full( GTK_OBJECT( subitem ),
						 "activate",
						 GTK_SIGNAL_FUNC( screem_editor_insert_attribute ),
						 NULL, temp,
						 g_free, 1, 0 );
		}
	}

	return item;
}

void screem_loaded_dtds()
{
	GladeXML *xml;
	GtkWidget *widget;

	xml = glade_xml_new( cfg->glade_path, "loaded_dtds" );

	widget = glade_xml_get_widget( xml, "dtd_list" );

	g_hash_table_foreach( dtd_types, (GHFunc)dtds_insert, widget );

	glade_xml_signal_autoconnect( xml );

	widget = glade_xml_get_widget( xml, "loaded_dtds" );
	gnome_dialog_run_and_close( GNOME_DIALOG( widget ) );
}

void screem_dtd_fill_combo( GtkWidget *combo )
{
	GList *list;

	list = g_list_alloc();
	gtk_object_set_data( GTK_OBJECT( combo ), "list", list );
	g_hash_table_foreach( dtd_types, (GHFunc)dtds_insert, combo );
	gtk_combo_set_popdown_strings( GTK_COMBO( combo ), list );
	g_list_free( list );
}

static void dtds_insert( gpointer key, gpointer value, GtkWidget *widget )
{
	gchar *items[] = { NULL, NULL, NULL };
	GList *list;


	if( GTK_IS_CLIST( widget ) ) {
		items[ 0 ] = (gchar*)key;
		items[ 1 ] = (gchar*)value;
		gtk_clist_append( GTK_CLIST( widget ), items );
	} else if( GTK_IS_COMBO( widget ) ) {
		list = (GList*)gtk_object_get_data( GTK_OBJECT( widget ), 
						    "list" );
		g_list_append( list, key );
	}
}

GtkWidget *screem_dtd_allowed_tags_menu( ScreemDTD *dtd, gchar *name )
{
	ScreemDTDElement *element;
	ScreemDTDElement *element2;
	ScreemDTDAllow *allow;
	GString *hack;
	GList *list;

	GtkWidget *item;
	GtkWidget *subitem;
	GtkWidget *menu;

	gint num;
	gchar *temp;

	if( ! name )
		return NULL;

	if( ! dtd ) {
		dtd = screem_dtd_load_type( cfg->mpage->default_dtd );
		if( ! dtd )
			return NULL;
	}

	element = screem_dtd_check_element( dtd, name );

	if( ! element )
		return NULL;

	/* we have the element */
       	item = gtk_menu_item_new_with_label( _( "Insert Tag" ) );
	gtk_widget_show( item );
	
	menu = gtk_menu_new();
	gtk_menu_item_set_submenu( GTK_MENU_ITEM( item ), menu );
	subitem = NULL;

	/* note we ignore the allowed number for now, what we should
	   do is check to see if there is already a tag of a limited
	   number inserted and if so not offer it in the menu */
	for( num = 0, allow = element->allow; allow; allow = allow->next ) {
		for( list = allow->elements; list; list = list->next ) {
			element2 = (ScreemDTDElement*)
				g_hash_table_lookup( dtd->elements,
						     (gchar*)list->data );
			if( ! element2 )
				continue;

			if( ++ num > 10 ) {
				subitem = gtk_menu_item_new_with_label( _( "More" ) );
				gtk_widget_show( subitem );
				gtk_menu_prepend( GTK_MENU( menu ), subitem );
				menu = gtk_menu_new();
				gtk_menu_item_set_submenu( GTK_MENU_ITEM( subitem ), menu );
				num = 0;
			}

			subitem = gtk_menu_item_new_with_label(element2->name);
			temp = g_strdup_printf( "<%s>", element2->name );
			gtk_signal_connect_full( GTK_OBJECT( subitem ),
						 "activate",
						 GTK_SIGNAL_FUNC( screem_dtd_insert_markup ),
						 NULL, temp, g_free, 1, 0 );
			gtk_widget_show( subitem );
			gtk_menu_append( GTK_MENU( menu ), subitem );
		}
	}

	if( ! subitem ) {
		gtk_widget_destroy( item );
		item = NULL;
	}

	return item;
}

/* acts as a proxy for the allowed tag menu */
static void screem_dtd_insert_markup( gchar *tag )
{
	screem_editor_insert_markup( tag, NULL );
}

ScreemDTDAttribute *screem_dtd_check_attribute( ScreemDTD *dtd, gchar *name )
{
	return NULL;
}

ScreemDTDElement *screem_dtd_check_element( ScreemDTD *dtd, gchar *name )
{
	ScreemDTDElement *element;
	GString *hack = NULL;

	if( ! dtd ) {
		dtd = screem_dtd_load_type( cfg->mpage->default_dtd );
		if( ! dtd )
			return NULL;
	}

	if( *name == '/' )
		name ++;

	element = g_hash_table_lookup( dtd->elements, name );

	/* shouldn't really do the rest of this if the document is xml */
	if( ! element ) {
		/* try all upper */
		hack = g_string_new( name );
		g_string_up( hack );
		element = g_hash_table_lookup( dtd->elements,hack->str );
	}
	if( ! element ) {
		/* try all lower */
		g_string_down( hack );
		element = g_hash_table_lookup( dtd->elements, hack->str );
	}
	if( ! element ) {
		/* capitalise first letter, Docbook 3.0 DTD Elements are like this */
		*hack->str = toupper( *hack->str );
		element = g_hash_table_lookup( dtd->elements, hack->str );
	}

	if( hack )
		g_string_free( hack, TRUE );

	return element;
}

gchar *screem_dtd_build_doctype_tag( const gchar *type )
{
	ScreemDTD *dtd;
	GString *tag;
	gchar *retval;

	tag = g_string_new( "<!DOCTYPE PUBLIC \"" );
	
	g_string_append( tag, type );

	g_string_append_c( tag, '"' );

	/* find the uri for the doctype */
	dtd = screem_dtd_load_type( type );

	if( dtd ) {
		/* we managed to get the dtd, append the uri */
		g_string_append( tag, " \"" );
		/* it shouldn't be NULL but lets check anyway */
		if( dtd->url )
			g_string_append( tag, dtd->url );
		g_string_append_c( tag, '"' );
	}

	g_string_append( tag, ">" );

	retval = tag->str;
	g_string_free( tag, FALSE );
	return retval;
}
