/*  Screem:  uploadWizard.c,
 *  This file handles the uploading of a site
 *
 *  This file Copyright (C) 1999 Lee Mallabone.
 *  Original 'sitecopy' by Joe Orton.
 *  Sitecopy functionality integrated into transfer.c by Lee Mallabone.
 *
 *  TODO:
 *  add fetch site functionality
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <errno.h>
#include <gmodule.h>
#include <gnome.h>
#include <unistd.h>
#include <stdio.h>
#include <pthread.h>

#include <semaphore.h>
#include <setjmp.h>
#include <signal.h>

#include <sys/stat.h>

#include <glade/glade.h>

/* add sitecopy functionality */
#include "common.h"
#include "sites.h"
#include "rcfile.h"
#include "frontend.h"
#include "ftp.h"

/* screem stuff */
#include "site.h"
#include "uploadWizard.h"

int my_abortable_transfer_wrapper(struct site *site,
                                  enum site_op operation );
void handle_abort( int sig );
void do_abort(void);
void abort_site_update(GtkWidget * button, gpointer data);
int fe_gtk_question(char *question, GnomeReplyCallback yes_action);
void fe_status(char *message);
void fe_login_clicked( GnomeDialog *dialog, gint number );
void fe_login_close( GnomeDialog *dialog );

static gboolean create_sitecopy_directory( void );
static void destroy_fnlists( struct site *site );
static void destroy_fnlist( struct fnlist *list );


extern GtkWidget *app;
extern Site *current_site;

GladeXML *xml;
GladeXML *exml;

static 	struct site *site;
extern struct site *all_sites;

volatile int in_critical_section = 0, want_abort = 0;
sigjmp_buf abort_buf;
sem_t *update_semaphore = NULL;
pthread_t update_tid = 0;
gboolean prompting = false;
gboolean closing;

static gint button;

/* the drivers */
extern const struct proto_driver ftp_driver;
extern const struct proto_driver dav_driver;
extern const struct proto_driver rsh_driver;
extern const struct proto_driver local_driver;

/* The site we're working with during all these functions. */
static float upload_total;
static float uploaded_bytes = 0.0;

G_MODULE_EXPORT const gchar* g_module_check_init( GModule *module );
G_MODULE_EXPORT void g_module_unload( GModule *module );
G_MODULE_EXPORT const gchar *html_tag( gchar **wizard_name );
G_MODULE_EXPORT void init( void );
G_MODULE_EXPORT gboolean screem_site_get_sync_status( GHashTable **table );

static struct site* screem_site_to_sitecopy_site( Site *site, 
						  gboolean show_alert );
static void upload_wizard_error( gchar *message );
static void upload_site( void );
static gboolean should_init( void );
static gboolean verify_site( struct site *site );
static void create_upload_dialog( void );
static void create_error_log_window( void );

static void *update_thread( void *arg );

void upload_button_clicked( GtkWidget *button, gpointer user_data );
void error_button_clicked( GtkButton *button, gpointer user_data );
void close_button_clicked(GtkButton *button, gpointer user_data );
static void confirmInit( GtkWidget *widget, gint button, gpointer data);

#define WIZARD_NAME "sitecopy"
#define TAG "sitecopy_hack"

/* hack to id as the sitecopy upload wizard */
G_MODULE_EXPORT const gchar *html_tag( gchar **wizard_name )
{
        *wizard_name = g_strdup( WIZARD_NAME ); 
        return g_strdup( TAG );
}



G_MODULE_EXPORT const gchar* g_module_check_init( GModule *module )
{
	g_print("uploadWizard: check-init\n");
	return NULL;
}

G_MODULE_EXPORT void g_module_unload( GModule *module )
{
	g_print( "uploadWizard: unloaded\n" );
}

G_MODULE_EXPORT void init()
{
	GtkWidget *uploadButton;
	GtkWidget *toolbar;
	
	GnomeUIInfo menuinfo[] = {
		{
			GNOME_APP_UI_ITEM, N_("Upload"),
			N_("Upload the site"),
			upload_site, NULL, NULL,
			GNOME_APP_PIXMAP_STOCK,
			GNOME_STOCK_MENU_BLANK,
			0,
			GDK_CONTROL_MASK, NULL
		},
		GNOMEUIINFO_END
	};
	
	
	/* place menu item after image under insert */
	gnome_app_insert_menus( GNOME_APP( app ),
				_("_Tools/"), menuinfo);
	
	toolbar = gtk_object_get_data( GTK_OBJECT( app ), "wizardbar" );
	
	/* place a button on the wizards toolbar,
	   I need to expose another toolbar as it shouldn't really be
	   put on the wizard one */
	uploadButton = gnome_stock_new_with_icon( GNOME_STOCK_PIXMAP_REDO );
	gtk_toolbar_append_item( GTK_TOOLBAR( toolbar ), "",
				 _("Upload Wizard"), "", uploadButton ,
				 upload_site, 0 );
	
	fe_initialize();

	update_semaphore = g_new( sem_t, 1 );
	sem_init(update_semaphore, 0, 0);

	closing = FALSE;

	g_print( "uploadWizard: initialised\n" );
}

static struct site* screem_site_to_sitecopy_site( Site *ssite,
						  gboolean show_alert )
{
	struct site *sitecopy_site;
	struct site_host *host;
	const gchar *string;
	GList *list;
	struct fnlist *exclude;
	struct fnlist *ignores;
	struct fnlist *ascii;

	const gchar* home;

	gchar *temp;

	sitecopy_site = g_new0( struct site, 1 );

	host = &sitecopy_site->server;

	/* the site name */
	string = screem_site_get_name( ssite );
	sitecopy_site->name = string;

	string = screem_site_get_remote_url( ssite );
	host->hostname = string;
	host->username = screem_site_get_remote_user( ssite );
	host->password = screem_site_get_remote_pass( ssite );

	if( host->username && strlen( host->username ) == 0 )
		host->username = NULL;

	if( host->password && strlen( host->password ) == 0 )
		host->password = NULL;

	if( host->hostname ) {
		if( ( temp = strchr( host->hostname, ':' ) ) ) {
			host->port = atoi( temp + 1 );
			temp = g_strndup( host->hostname, 
					 temp - host->hostname );
			host->hostname = temp;
			/* FIXME: memory leak */
		}
	}

	/* the view url */
	string = screem_site_get_http_url( ssite );
	sitecopy_site->url = string;

	/* get transfer options */
	switch( screem_site_get_remote_method( ssite ) ) {
	case FTP:
		sitecopy_site->protocol = siteproto_ftp;
		sitecopy_site->proto_string = "ftp";
		sitecopy_site->driver = &ftp_driver;
		break;
	case WEBDAV:
		sitecopy_site->protocol = siteproto_dav;
		sitecopy_site->proto_string = "dav";
		sitecopy_site->driver = &dav_driver;
		break;
	case RSH:
		sitecopy_site->protocol = siteproto_rsh;
		sitecopy_site->proto_string = "rsh";
		sitecopy_site->driver = &rsh_driver;
		sitecopy_site->rsh_cmd = "rsh";
		sitecopy_site->rcp_cmd = "rcp";
		break;
	case SSH:
		sitecopy_site->protocol = siteproto_rsh;
                sitecopy_site->proto_string = "ssh";
                sitecopy_site->driver = &rsh_driver;
		sitecopy_site->rsh_cmd = "ssh";
		sitecopy_site->rcp_cmd = "scp";
                break;
	case LOCAL:
	default:
		sitecopy_site->protocol = siteproto_local;
		sitecopy_site->proto_string = "local";
		sitecopy_site->driver = &local_driver;
		break;
	}

	switch( screem_site_get_permissions( ssite ) ) {
	case PERMS_EXEC:
		sitecopy_site->perms = sitep_exec;
		break;
	case PERMS_IGNORE:
		sitecopy_site->perms = sitep_ignore;
		break;
	case PERMS_ALL:
		sitecopy_site->perms = sitep_all;
		break;
	}

	switch( screem_site_get_symlinks( ssite ) ) {
	case SYM_IGNORE:
		sitecopy_site->symlinks = sitesym_ignore;
		break;
	case SYM_FOLLOW:
		sitecopy_site->symlinks = sitesym_follow;
		break;
	case SYM_MAINTAIN:
		sitecopy_site->symlinks = sitesym_maintain;
		break;
	}

	/* the remote path */
	string = screem_site_get_remote_path( ssite );
	sitecopy_site->remote_root_user = string;

	sitecopy_site->remote_isrel = ( string[ 0 ] == '~' );
	if( string[ 0 ] != '~' && string[ 0 ] != G_DIR_SEPARATOR ) {
		/* must start with ~ or /  */
		if( show_alert )
			upload_wizard_error( _("Remote path must being with ~ or /") );
		g_free( sitecopy_site );
		return NULL;
	}

	/* the local pathname */
	string = screem_site_get_pathname( ssite );
	sitecopy_site->local_root_user = string;
	sitecopy_site->local_isrel = ( string[ 0 ] == '~' );
	if( string[ 0 ] != '~' && string[ 0 ] != G_DIR_SEPARATOR ) {
		/* should never occur, screem should not allow this
		   case to happen */
		if( show_alert )
			upload_wizard_error( _("Local path must being with ~ or /") );
		g_free( sitecopy_site );
		return NULL;
	}

	/* misc settings */
	sitecopy_site->ftp_pasv_mode = screem_site_get_passive_ftp( ssite );
	sitecopy_site->nodelete = screem_site_get_no_delete( ssite );
	sitecopy_site->checkmoved = screem_site_get_check_moved( ssite );
	sitecopy_site->nooverwrite = screem_site_get_no_overwrite( ssite );

	/* setup excludes */
	exclude = g_new0( struct fnlist, 1 );
	exclude->haspath = FALSE;
	exclude->pattern = "*.screem";
	exclude->prev = g_new0( struct fnlist, 1 );
	exclude->prev->next = exclude;
	exclude = exclude->prev;
	exclude->haspath = FALSE;
	exclude->pattern = ".project.screem";
	for(list = screem_site_get_excludes( ssite ); list; list = list->next){
		exclude->prev = g_new0( struct fnlist, 1 );
		exclude->prev->next = exclude;
		exclude = exclude->prev;
		exclude->haspath = TRUE;
		exclude->pattern = ((gchar*)list->data) + strlen( string );
	}
	sitecopy_site->excludes = exclude;

	/* setup ignores */
	for( ignores = NULL, list = screem_site_get_ignores( ssite ); list;
	     list = list->next ) {
		if( ! ignores ) {
			ignores = g_new0( struct fnlist, 1 );
		} else {
			ignores->prev = g_new0( struct fnlist, 1 );
			ignores->prev->next = ignores;
			ignores = ignores->prev;
		}
		ignores->haspath = TRUE;
		ignores->pattern = ((gchar*)list->data) + strlen( string );
	}
	sitecopy_site->ignores = ignores;

	/* setup asciis */
	for( ascii = NULL, list = screem_site_get_asciis( ssite ); list;
	     list = list->next ) {
		if( ! ascii ) {
			ascii = g_new0( struct fnlist, 1 );
		} else {
			ascii->prev = g_new0( struct fnlist, 1 );
			ascii->prev->next = ascii;
			ascii = ascii->prev;
		}
		ascii->haspath = TRUE;
		ascii->pattern = ((gchar*)list->data) + strlen( string );
	}
	sitecopy_site->asciis = ascii;

	sitecopy_site->use_this = TRUE;

	/* FIXME: this should be an option */
	sitecopy_site->state_method = state_timesize;
	sitecopy_site->stored_state_method = state_timesize;

	/* set info file path */
	home = g_get_home_dir();
       	sitecopy_site->infofile = g_strconcat( home, G_DIR_SEPARATOR_S, 
					       ".sitecopy",
					       G_DIR_SEPARATOR_S, 
					       sitecopy_site->name, NULL );

	return sitecopy_site;
}

static void upload_page()
{
	GladeXML *sxml;
	GtkWidget *dialog;
	GtkWidget *widget;
	gint button;

	Page *page;
	gchar *file;
	gchar *rfile;

	ftp_session *session;
	struct site_host *server;

	page = screem_site_get_current_page( current_site );
	
	file = g_basename( screem_page_get_pathname( page ) );

	sxml = glade_xml_new( GLADE_PATH, "single_upload" );
	dialog = glade_xml_get_widget( sxml, "single_upload" );
	
	button = gnome_dialog_run( GNOME_DIALOG( dialog ) );
	
	if( button == 0 ) {
		server = g_new0( struct site_host, 1 );
		widget = glade_xml_get_widget( sxml, "path" );
		rfile = gtk_entry_get_text( GTK_ENTRY( widget ) );

		widget = glade_xml_get_widget( sxml, "uri" );
		server->hostname = gtk_entry_get_text( GTK_ENTRY( widget ) );
		server->port = 21;
		widget = glade_xml_get_widget( sxml, "user" );
		server->username = gtk_entry_get_text( GTK_ENTRY( widget ) );
		widget = glade_xml_get_widget( sxml, "pass" );
		server->password = gtk_entry_get_text( GTK_ENTRY( widget ) );

		/* do upload, FIXME: handle reporting of errors +
		   give the user some feed back */
		session = ftp_init();
		ftp_set_server( session, server );
		
		rfile = g_strconcat( rfile, "/", file, NULL );
		ftp_put( session, file, rfile, TRUE );
		ftp_finish( session );
		g_free( rfile );
		g_free( server );
	}

	gtk_widget_destroy( dialog );
}

static void upload_site()
{
	struct stat s;

	if( screem_site_get_fake_flag( current_site ) ) {
		/* upload a single page, we can use the ftp code
		   in sitecopy */

		upload_page();
		return;
	}

	/* create ~/.sitecopy if necessary */
	if( ! create_sitecopy_directory() ) {
		g_free( site );
		return;
	}

	if( ! ( site = screem_site_to_sitecopy_site( current_site,TRUE ) ) )
		return;

	all_sites = site;

	/* verify that the site is setup properly */
	if( ! verify_site( site ) ) {
		g_free( site->infofile );
		g_free( site );
		return;
	}

	/* info file exist? */
	if( stat( site->infofile, &s ) < 0 ) {
		/* Displays a dialog asking if the user wants to init or
		   catchup their website. */
		site_write_stored_state( site );

		if( should_init() )
			site_initialize( site );
		else if ( (site_readfiles (site)) != SITE_OK) {
			gnome_error_dialog (_("Could not retrieve information about your local files."));
			g_free( site->infofile );
			g_free( site );
			return;
		} else {
			site_catchup( site );
			site_write_stored_state( site );
			/* If the site is up to date, we really don't need 
			   to upload it! */
			g_free( site->infofile );
			g_free( site );
			return;
		}
	}

	site->files = NULL;
	site->files_tail = NULL;

	if( site_readfiles( site ) ) {
		/* couldn't get info on local site files */
		upload_wizard_error( _("Couldn't retrieve information about local files") );
		g_free( site->infofile );
		g_free( site );
		return;
	} else if( ! site->remote_is_different ) {
		/* nothing needs uploading */
		g_print( "diff: %d\n", site->local_is_different );
		g_free( site->infofile );
		g_free( site );
		return;
	}

	/* reset transfer counter */
	upload_total = site->totalnew + site->totalchanged;
	g_print( "upload_total: %f", upload_total );
	uploaded_bytes = 0.0;

	sem_init(update_semaphore, 0, 0);
	closing = FALSE;

	pthread_create(&update_tid, NULL, update_thread, NULL);

       	/* all setup, display upload dialog */
	create_upload_dialog();
}

static gboolean verify_site( struct site *site )
{
	gint ret;
	gchar *message = NULL;
	
	switch( ( ret = rcfile_verify( site ) ) ) {
	case SITE_NONAME:
		message =  _("No site name given");
		break;
	case SITE_NOSERVER:
		message = _("No server name given.");
		break;
	case SITE_NOREMOTEDIR:
		message = _("No remote directory given.");
		break;
	case SITE_NOLOCALDIR:
		message = _("No local directory given.");
		break;
	case SITE_ACCESSLOCALDIR:
		message = _("The local dir couldn't be accessed.");
		break;
	case SITE_INVALIDPORT:
		message = _("There was a problem with the port number for this site.");
		break;
	case SITE_NOMAINTAIN:
		message = _("Sorry, the symbolic links option you chose is not supported\nby this transfer protocol. Please choose another option.");
		break;
	case SITE_NOREMOTEREL:
		if( site->remote_isrel )
			message = _("This protocol does not support relative remote directories.");
		break;
	case SITE_NOPERMS:
		message = _("The protocol you are attempting to use does\nnot currently support maintaining permissions.");
        break;

	case SITE_NOLOCALREL:
		message = _("The local dir is invalid.");
		break;
	case 0:
		/* Site integrity is okay. */
		break;
	default:
		message = _("There was an undetermined problem verifying the correctness of your site definition. Please report this to the maintainer.");
		break;
	}

	if( message )
		upload_wizard_error( message );
	
	return ( ret == 0 );
}


static gboolean should_init()
{
	GtkWidget *state_question;
	GtkWidget *dialog_vbox2;
	GtkWidget *button3;
	GtkWidget *button4;
	GtkWidget *dialog_action_area2;
	gint button;
	
	state_question = gnome_message_box_new( _("This appears to be the first time you have attempted to upload\nthis website. What is the state of the remote site?"), GNOME_MESSAGE_BOX_GENERIC, NULL);

	gtk_object_set_data (GTK_OBJECT (state_question), "state_question", state_question);
	GTK_WINDOW (state_question)->type = GTK_WINDOW_DIALOG;
	gtk_window_set_modal (GTK_WINDOW (state_question), TRUE);
	gtk_window_set_policy (GTK_WINDOW (state_question), FALSE, FALSE, FALSE);
	gnome_dialog_set_close (GNOME_DIALOG (state_question), TRUE);
	
	dialog_vbox2 = GNOME_DIALOG (state_question)->vbox;
	gtk_object_set_data (GTK_OBJECT (state_question), "dialog_vbox2", dialog_vbox2);
	gtk_widget_show (dialog_vbox2);
	
	gnome_dialog_append_button_with_pixmap (GNOME_DIALOG (state_question),
						_("Empty"), GNOME_STOCK_PIXMAP_NEW);
	button3 = g_list_last (GNOME_DIALOG (state_question)->buttons)->data;
	gtk_widget_ref (button3);
	gtk_object_set_data_full (GTK_OBJECT (state_question), "button3", button3,
				  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (button3);
	GTK_WIDGET_SET_FLAGS (button3, GTK_CAN_DEFAULT);
	
	gnome_dialog_append_button_with_pixmap (GNOME_DIALOG (state_question),
						_("Already uploaded"), GNOME_STOCK_PIXMAP_MULTIPLE);
	button4 = g_list_last (GNOME_DIALOG (state_question)->buttons)->data;
	gtk_widget_ref (button4);
	gtk_object_set_data_full (GTK_OBJECT (state_question), "button4", button4,
				  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (button4);
	GTK_WIDGET_SET_FLAGS (button4, GTK_CAN_DEFAULT);
	
	dialog_action_area2 = GNOME_DIALOG (state_question)->action_area;
	gtk_widget_ref (dialog_action_area2);
	gtk_object_set_data_full (GTK_OBJECT (state_question), "dialog_action_area2", dialog_action_area2,
				  (GtkDestroyNotify) gtk_widget_unref);
	
	gtk_object_set_data (GTK_OBJECT(state_question), "dialog", &state_question);
	gtk_signal_connect (GTK_OBJECT(state_question), "clicked", 
			    confirmInit, &button);
	gtk_widget_show(state_question);
	button = -1;
	while (state_question)
		gtk_main_iteration();

	return ( button != 1 );
}

static void confirmInit( GtkWidget *widget, gint button, gpointer data) 
{
	gpointer *prompt = gtk_object_get_data(GTK_OBJECT( widget ), "dialog");
	gint *b = (gint*)data;
	/*  gtk_widget_destroy(*prompt); */
	*prompt = NULL;
	*b = button;
}

static void *update_thread( void *arg ) 
{
	int ret;
	gboolean tmp;
	GtkWidget *keep_going_button;
	GtkWidget *begin_button;
	GtkWidget *stop_button;
	GtkWidget *close_button;
	GtkWidget *update_dialog;

	pthread_detach(pthread_self());

	for( ;; ) {
		/* sleep */
		sem_wait( update_semaphore );

		if( closing )
			break;
		
		gdk_threads_enter();
		keep_going_button = glade_xml_get_widget(xml, 
							 "keep_going_button" );
		begin_button = glade_xml_get_widget( xml, "upload_button" );
		close_button = glade_xml_get_widget( xml, "close_button" );
		stop_button = glade_xml_get_widget( xml, "stop_button" );
		update_dialog = glade_xml_get_widget( xml, "update_dialog" );

		tmp = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( keep_going_button ) );

		gtk_widget_set_sensitive(begin_button, FALSE);
		gtk_widget_set_sensitive(keep_going_button, FALSE);
		gdk_threads_leave();

		ret = my_abortable_transfer_wrapper(site, site_op_update);

		gdk_threads_enter();

		switch( ret ) {
		case SITE_OK:
			/* be sure to add some 'free' commands in here */
			site_write_stored_state( site );
			/*rescan_selected(1);*/
			break;
		case SITE_AUTH:
			gnome_warning_dialog (_("There was a problem authenticating with the remote server."));
			break;
		case SITE_LOOKUP:
			gnome_warning_dialog ( _("Could not resolve the remote site's hostname."));
			break;
		case SITE_CONNECT:
			gnome_warning_dialog ( _("Could not establish a connection to the remote site."));
			break;
		case SITE_FAILED:
			gnome_warning_dialog( _("Update Failed (Authentication problems)" ));
			break;
		case SITE_ABORTED:
			gnome_warning_dialog( _("Update was aborted." ));
			break;
		case SITE_ERRORS:
			/* replace this with a custom dialog containing the errors, a la Xsitecopy */
			gnome_warning_dialog (_("There were errors."));      
			break;
		}
		gtk_widget_hide (stop_button);
		gtk_widget_show (close_button); 
		gtk_widget_set_sensitive(close_button, TRUE);
		gtk_widget_set_sensitive(begin_button, TRUE);
		gtk_window_set_modal(GTK_WINDOW(update_dialog), FALSE);
		gdk_threads_leave();
	}

	return NULL;
}

/* GUI part */

static void upload_wizard_error( gchar *message )
{
	gnome_error_dialog( message );
}

static void create_upload_dialog()
{
	GtkWidget *stop_button;

	xml = glade_xml_new( GLADE_PATH, "update_dialog" );

	stop_button = glade_xml_get_widget( xml, "stop_button" );
	gtk_widget_hide( stop_button );

	glade_xml_signal_autoconnect( xml );
}

static void create_error_log_window()
{
	GtkWidget *dialog;

	gdk_threads_enter();

	exml = glade_xml_new( GLADE_PATH, "error_log_window" );

	dialog = glade_xml_get_widget( exml, "error_log_window" );

	glade_xml_signal_autoconnect( xml );
	gnome_dialog_run_and_close( GNOME_DIALOG( dialog ) );

	gdk_threads_leave();
}

/* callbacks */
void upload_button_clicked( GtkWidget *button, gpointer user_data )
{
	/* We don't want to begin again. Also, there doesn't seem to be any way
	 * to explictly terminate a pthread, so we don't want the dialog to be
	 * closed while the upload would carry on in the background.
	 * We also don't want the user changing files while we're doing an 
	 * upload, so we make the upload dialog modal.
	 */

	GtkWidget *update_dialog;
	GtkWidget *close_button;
	GtkWidget *stop_button;
	GtkWidget *upload_button;

	update_dialog = glade_xml_get_widget( xml, "update_dialog" );
	upload_button = glade_xml_get_widget( xml, "upload_button" );
	close_button = glade_xml_get_widget( xml, "close_button" );
	stop_button = glade_xml_get_widget( xml, "stop_button" );

	gtk_widget_hide( close_button );
	gtk_widget_show( stop_button );
	gtk_widget_set_sensitive(upload_button, FALSE);
	gtk_window_set_modal( GTK_WINDOW( update_dialog ), TRUE );
	sem_post(update_semaphore);
}

void error_button_clicked( GtkButton *button, gpointer user_data )
{
	create_error_log_window();
}

void close_button_clicked(GtkButton *button, gpointer user_data )
{
	GtkWidget *update_dialog;

	update_dialog = glade_xml_get_widget( xml, "update_dialog" );

	closing = TRUE;
	sem_post(update_semaphore);
       
	/* the update thread will terminate itself as the for loop
	   is broken out of */
	
	/* Free the memory created in site_readfiles. */
	site_destroy( site );

	/* clean up */
	g_free( site );
	all_sites = NULL;

	gtk_widget_destroy( update_dialog );
}

/* Signal handler */
void handle_abort( int sig ) 
{
	if( in_critical_section ) {
		/* Can't abort now, so remember we want to for later */
		want_abort = 1;
	} else {
		do_abort();
	}
}

void abort_site_update(GtkWidget * button, gpointer data)
{
	/*gtk_label_set(GTK_LABEL(status_label), "Aborting...");*/
	
	/* Free the memory created in site_readfiles. */
	site_destroy( site );

	pthread_kill(update_tid, SIGUSR1);
}


int my_abortable_transfer_wrapper(struct site *site,
                                  enum site_op operation ) 
{
	int ret;
	signal( SIGUSR1, handle_abort );
	if( !sigsetjmp( abort_buf, 1 ) ) {
		/* Normal execution */
		switch (operation) {
			case site_op_update:
				ret = site_update(site);
				break;
			case site_op_fetch:
				ret = site_fetch(site);
				break;
			case site_op_resync:
				ret = site_synch(site);
				break;
			default:
				break;
			}
	} else {
		/* The update was aborted */
		ret = SITE_ABORTED;
	}
	signal( SIGUSR1, SIG_IGN );

	return ret;
}

/* Actually abort the update */
void do_abort(void) {
    want_abort = 0;
    siglongjmp( abort_buf, 1 );
}



/********************************************************************/
/**** Implementations of all the functions defined in frontend.h ****/
/**** These functions should be considered 'private' and never   ****/
/**** be called directly.                                        ****/
/********************************************************************/

int fe_gtk_question(char *question, GnomeReplyCallback yes_action)
{
    GtkWidget *question_window;

    gdk_threads_enter();
    question_window = gnome_app_question(GNOME_APP(app),
                                         (const gchar *) question,
                                         yes_action, NULL);
    gtk_widget_show(question_window);
    gdk_threads_leave();

    return 1;
}

void fe_status(char *message)
{
	gdk_threads_enter();
	gnome_app_message(GNOME_APP(app), (const gchar *) message);
	gdk_threads_leave();
}

/* Enter critical section */
void fe_disable_abort(struct site *site)
{
	in_critical_section = 1;
}

/* Leave critical section */
void fe_enable_abort(struct site *site)
{
	in_critical_section = 0;
	/* Carry out the abort if we were aborted while in the
	 * critical section */
	if( want_abort ) {
		do_abort();
	}
}

void fe_warning(const char *description, const char *subject,
                const char *error)
{
/*    gnome_warning_dialog(description);*/
}

/** Informs the user when we're trying to (re)establish a connection
 *  or lookup a hostname. 
 */
void fe_connection( sock_status status, const char *info ) 
{
	GtkWidget *label;
	gchar *text = "";

	gdk_threads_enter();

	label = glade_xml_get_widget( xml, "status_label" );

	switch( status ) {
	case (sock_namelookup):
		text = _("Looking up hostname...");
		break;
	case (sock_connecting):
		text = _("Attempting to connect...");
		break;
	case (sock_connected):
		text = _("Connected.");
		break;
	}

	gtk_label_set( GTK_LABEL( label ), text );

	gdk_threads_leave();
}


bool fe_can_update( const struct site_file *file ) 
{
   return TRUE;
}

void fe_updating( const struct site_file *file ) 
{
	gchar *file_status;
	GtkWidget *status_label;

	GtkWidget *op_label;
	GtkWidget *file_label;
	GtkWidget *dir_label;

	gdk_threads_enter();

	status_label = glade_xml_get_widget( xml, "status_label" );
	op_label = glade_xml_get_widget( xml, "op_label" );
	file_label = glade_xml_get_widget( xml, "file_label" );
	dir_label = glade_xml_get_widget( xml, "dir_label" );

	file_status = g_strdup_printf(_("Commiting updates to %s..."),
				      site->server.hostname);
	gtk_label_set(GTK_LABEL(status_label), file_status);
	/* Check this */
	g_free(file_status);
	if (file->type == file_dir) {
		if (file->diff == file_new) {
			gtk_label_set(GTK_LABEL(op_label), _("Creating directory..."));
			gtk_label_set(GTK_LABEL(file_label), file_name(file));
			/*         gtk_label_set (GTK_LABEL (dir_label), file->directory); */
			gtk_label_set(GTK_LABEL(dir_label), "");
		} else {
			/* can we move dirs yet? */
			gtk_label_set(GTK_LABEL(op_label), _("Deleting directory..."));
			gtk_label_set(GTK_LABEL(dir_label), "");
		}
	} else {
		switch (file->diff) {
		case file_changed:
		case file_new:
			gtk_label_set(GTK_LABEL(op_label), _("Uploading..."));
			gtk_label_set(GTK_LABEL(file_label), file_name(file));
			gtk_label_set(GTK_LABEL(dir_label), dir_name(file_name(file)));
			break;
		case file_deleted:
			gtk_label_set(GTK_LABEL(op_label), _("Deleting..."));
			gtk_label_set(GTK_LABEL(file_label), file_name(file));
			gtk_label_set(GTK_LABEL(dir_label), "");
			break;
		case file_moved:
			gtk_label_set(GTK_LABEL(op_label), _("Moving..."));
			gtk_label_set(GTK_LABEL(file_label), file_name(file));
			/* FIXME: Check this, I think it's dodgy. */
			gtk_label_set(GTK_LABEL(dir_label), dir_name(file_name(file)));
			break;
		case file_unchanged:
			gtk_label_set(GTK_LABEL(op_label), _("ARG! The file hasn't changed, we shouldn't be doing anything!"));
		}
	}

	gdk_threads_leave();

}

/* Once a file has been updated, any errors with it are recorded in the 
 * error list, and the progress bar is reset. 
 */
void fe_updated( const struct site_file *file, const bool success,
		 const char *error )
{
	GtkWidget *mainbar;
	gchar *temp;

	gdk_threads_enter();

	mainbar = glade_xml_get_widget( xml, "main_progressbar" );

	if( ! success ) {
		temp = g_strdup_printf( "%s: %s", file_name( file ), error );
		screem_show_warning( temp );
		g_free( temp );
	}

	gtk_progress_bar_update(GTK_PROGRESS_BAR(mainbar), 0.0);
	uploaded_bytes += (float) file->local.size;

	gdk_threads_leave();
}

void fe_transfer_progress( off_t progress, off_t total )
{
	float div1, div2;
	size_t upload_size = 0;

	GtkWidget *mainbar;
	GtkWidget *jobbar;

	gdk_threads_enter();

	mainbar = glade_xml_get_widget( xml, "main_progressbar" );
	jobbar = glade_xml_get_widget( xml, "job_progressbar" );

	/* Calculate the amount to display the current file's progress */
	div1 = (float) progress;
	div2 = (float) total;

	if( div2 ) {
		g_print( "%f / %f = %f\n", div1, div2, div1/div2 );
		gtk_progress_bar_update(GTK_PROGRESS_BAR(mainbar), 
					div1 / div2);
	}
      
	gtk_progress_bar_update(GTK_PROGRESS_BAR(jobbar),
				(uploaded_bytes + (float) progress) / 
				(float) upload_total);

	gdk_threads_leave();
}

/* These can be implemented if we ever want to use the respective 
 * functionality.
 */
void fe_synching( const struct site_file *file )
{

}

void fe_synched( const struct site_file *file, const bool success,
		 const char *error )
{

}

void fe_setting_perms(const struct site_file *file)
{
}
void fe_set_perms(const struct site_file *file, int success,
                  const char *error)
{
}

void fe_checksumming( const char *filename )
{

}

void fe_checksummed(const char *file, int success, const char *err)
{
}

void fe_fetch_found(const struct site_file *file) 
{
}

int fe_login(fe_login_context ctx, const char *realm, const char *hostname,
             char **username, char **password) 
{
	const char *server = 
		(ctx==fe_login_server)?N_("server"):N_("proxy server");
	char *pass;

	gchar *message;
	GtkWidget *widget;
	GladeXML *lxml;

	g_print( "fe_login\n" );

	gdk_threads_enter();

	lxml = glade_xml_new( GLADE_PATH, "auth_dialog" );
	widget = glade_xml_get_widget( lxml, "message_label" );

	if (realm) {
		message = g_strdup_printf( _("Authentication required for %s on %s `%s':\n"), realm, server, hostname );
	} else {
		message = g_strdup_printf( _("Authentication required on %s `%s':\n"), server, hostname );
	}

	gtk_label_set( GTK_LABEL( widget ), message );
	g_free( message );

	if( *username ) {
		widget = glade_xml_get_widget( lxml, "username" );
		gtk_entry_set_text( GTK_ENTRY( widget ), *username );
	}
	if( *password ) {
		widget = glade_xml_get_widget( lxml, "password" );
		gtk_entry_set_text( GTK_ENTRY( widget ), *password );
	}

	widget = glade_xml_get_widget( lxml, "auth_dialog" );
	gtk_widget_show_all( widget );

	glade_xml_signal_autoconnect( lxml );

	/* sleep, we will be woken up by one of the dialogs callbacks */
	button = -1 ;
	gdk_threads_leave();

	sem_wait( update_semaphore );

	if( *username )
		free( *username );
	if( *password )
		free( *password );

	if( button != 0 ) {
		/* canceled */
		gtk_widget_destroy( widget );
		return -1;
	}
	gdk_threads_enter();
	widget = glade_xml_get_widget( lxml, "username" );
	*username = g_strdup( gtk_entry_get_text( GTK_ENTRY( widget ) ) );

	widget = glade_xml_get_widget( lxml, "password" );
	*password = g_strdup( gtk_entry_get_text( GTK_ENTRY( widget ) ) );

	widget = glade_xml_get_widget( lxml, "auth_dialog" );
	gtk_widget_destroy( widget );

	gdk_threads_leave();

	return 0;
}

void fe_login_clicked( GnomeDialog *dialog, gint number )
{
	button = number;
	sem_post(update_semaphore);
}
void fe_login_close( GnomeDialog *dialog )
{
	sem_post(update_semaphore);
}

static gboolean create_sitecopy_directory()
{
	const gchar *home;
	gchar *dir;
	struct stat s;

	home = g_get_home_dir();
	dir = g_strconcat( home, G_DIR_SEPARATOR_S, ".sitecopy", NULL );
	if( stat( dir, &s ) != 0 ) {
		if( errno != ENOENT ) {
			/* couldn't access it */
			upload_wizard_error(_("Couldn't access ~/.sitecopy"));
			g_free( dir );
			return FALSE;
		}
		/* it doesn't exist so lets make it */
		if( mkdir( dir, S_IRWXU ) != 0 ) {
			/* couldn't make it */
			upload_wizard_error(_("Couldn't create ~/.sitecopy"));
			g_free( dir );
			return FALSE;
		}
	}
	g_free( dir );
	return TRUE;
}

gboolean screem_site_get_sync_status( GHashTable **table )
{
	struct site *site;
	struct site_file *files;
	struct file_state *file;
	gchar *filename;

	g_return_val_if_fail( table != NULL, FALSE );

	*table = NULL;

	if( screem_site_get_fake_flag( current_site ) ) {
		/* fake sites have no sync status */
		return FALSE;
	}

	if( ! create_sitecopy_directory() )
		return FALSE;

	site = screem_site_to_sitecopy_site( current_site, FALSE );

	if( ! site )
		return FALSE;

	if( ! verify_site( site ) ) {
		destroy_fnlists( site );
		g_free( site->infofile );
		g_free( site );
		return FALSE;
	}

	if( site_readfiles( site ) ) {
		/* couldn't get info on local site files, this probably
		   means the user hasn't attempted to upload yet */
		destroy_fnlists( site );
		g_free( site->infofile );
		g_free( site );
		return FALSE;
	}

	*table = g_hash_table_new( g_str_hash, g_str_equal );

	/* ok we've got the info, and there are changes */
	files = site->files;
	while( files ) {
		file = &files->local;
		
		if( file->filename ) {
			filename = file_full_local( file, site );
			
			/* add to hash table with the status */
			g_hash_table_insert( *table, filename, 
					     GINT_TO_POINTER( files->diff ) );
		}

		files = files->next;
	}

	site_destroy( site );

	destroy_fnlists( site );
	g_free( site->infofile );
	g_free( site );

	return TRUE;
}

/* destroy the fnlists in the sitecopy site object,
   we don't want to free that pattern though */
static void destroy_fnlists( struct site *site )
{
	struct fnlist *list;

	list = site->excludes;
	destroy_fnlist( list );
	list = site->ignores;
	destroy_fnlist( list );
	list = site->asciis;
	destroy_fnlist( list );
	
}

static void destroy_fnlist( struct fnlist *list )
{
	if( ! list )
		return;

	for( ;; ) {
		if( list->next ) {
			list = list->next;
			g_free( list->prev );
		} else {
			g_free( list );
			break;
		}
	}
}
