/*  Screem:  screem-window-menus.c
 *
 *  Handle menus/toolbars
 *
 *  Copyright (C) 2001 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */
#include <config.h>

#include <pwd.h>
#include <string.h>
#include <sys/types.h>

#include <glib/gi18n.h>
#include <libgnome/gnome-help.h>
#include <libgnome/gnome-exec.h>
#include <libgnome/gnome-util.h>

#include <libgnomeui/gnome-about.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <libgnomeui/gnome-file-entry.h>
#include <libgnomeui/gnome-entry.h>
#include <libgnomeui/gnome-stock-icons.h>

#include <libgnomevfs/gnome-vfs-mime-handlers.h>
#include <libgnomevfs/gnome-vfs-mime-utils.h>
#include <libgnomevfs/gnome-vfs-utils.h>

#include <glade/glade.h>


#include <gtk/gtk.h>

#include <glib-object.h>
#include <glib/glist.h>

#include <gdk-pixbuf/gdk-pixbuf.h>

#include "screem-application.h"
#include "screem-window.h"
#include "screem-window-private.h"

#include "screem-site.h"
#include "screem-site-ui.h"
#include "screem-site-druid.h"

#include "screem-editor.h"
#include "screem-tree-view.h"

#include "screem-site-view.h"    /* don't like including this */

#include "screem-preferences.h"

#include "screem-page.h"
#include "pageUI.h"

#include "screem-markup.h"

#include "screem-todo.h"

#include "screem-cvs.h"

#include "screem-search.h"

#include "support.h"

#include "fileops.h"

#include "gdl.h"

#include "egg-editable-toolbar.h"
#include "egg-toolbars-model.h"
#include "egg-toolbar-editor.h"

extern void screem_debug_console( ScreemApplication *app );


static void file_menu_save_document_as_callback( GtkAction *action, 
					   	 gpointer user_data );
static void edit_menu_goto_parent_callback( GtkAction *action, 
				       gpointer user_data );


static void add_widget( GtkUIManager *merge, GtkWidget *widget, ScreemWindow *window )
{
       	if( GTK_IS_MENU_SHELL( widget ) ) {
		gtk_box_pack_start( GTK_BOX( window->content_hbox ), widget,
				    FALSE, TRUE, 0 );
		gtk_box_reorder_child( GTK_BOX( window->content_hbox ), widget,
					0 );
	} else if( GTK_IS_TOOLBAR( widget ) ) {
		guint toolbars;
		
		toolbars = GPOINTER_TO_UINT( g_object_get_data( G_OBJECT( merge ),
						"toolbars" ) );
		
		toolbars ++;
		
		gtk_toolbar_set_show_arrow( GTK_TOOLBAR( widget ), TRUE );
		gtk_toolbar_set_tooltips( GTK_TOOLBAR( widget ), TRUE );
		
		gtk_box_pack_start( GTK_BOX( window->content_hbox ), widget,
				    FALSE, TRUE, 0 );
		gtk_box_reorder_child( GTK_BOX( window->content_hbox ), widget,
					toolbars );
		g_object_set_data( G_OBJECT( merge ), "toolbars",
					GUINT_TO_POINTER( toolbars ) );
	} 
}

static void item_select_cb( GtkMenuItem *item, gpointer data )
{
	ScreemWindow *window;
	GtkAction *action;
	gchar *msg;
	
	window = SCREEM_WINDOW( data );

	action = g_object_get_data( G_OBJECT( item ), "gtk-action" );
	g_return_if_fail( action != NULL );

	g_object_get( G_OBJECT( action ), "tooltip", &msg, NULL );
	if( msg ) {
		gtk_statusbar_push( GTK_STATUSBAR( window->status_text ),
					0, msg );
		g_free( msg );
	}
}

static void item_deselect_cb( GtkMenuItem *item, gpointer data )
{
	ScreemWindow *window;
	
	window = SCREEM_WINDOW( data );

	gtk_statusbar_pop( GTK_STATUSBAR( window->status_text ), 0 );
}

static void connect_proxy_cb( GtkUIManager *ui,
		GtkAction *action, GtkWidget *proxy,
		gpointer data )
{
	if( GTK_IS_MENU_ITEM( proxy ) ) {
		g_signal_connect( proxy, "select",
				G_CALLBACK( item_select_cb ),
				data );
		g_signal_connect( proxy, "deselect",
				G_CALLBACK( item_deselect_cb ),
				data );
	}
}

static void disconnect_proxy_cb( GtkUIManager *ui,
		GtkAction *action, GtkWidget *proxy,
		gpointer data )
{
	if( GTK_IS_MENU_ITEM( proxy ) ) {
		g_signal_handlers_disconnect_by_func( proxy,
				G_CALLBACK( item_select_cb ), data );
		g_signal_handlers_disconnect_by_func( proxy,
				G_CALLBACK( item_deselect_cb ), data );
	}
}

static gboolean file_menu_new_window_after_cb( gpointer data )
{
	ScreemWindow *window;
	ScreemApplication *app;
	ScreemSite *site;
	
	window = SCREEM_WINDOW( data );
	app = SCREEM_APPLICATION( window->application );
	site = screem_application_get_default_site( app );

	gdk_threads_enter();
	screem_window_set_current( window, site );
	gdk_threads_leave();
	
	return FALSE;
}

static void file_menu_new_window_callback( GtkAction *action, 
					   gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *app;

	window = SCREEM_WINDOW( user_data );
	app = SCREEM_APPLICATION( window->application );

	window = screem_application_create_window( app );

	g_idle_add( file_menu_new_window_after_cb, window );

	gtk_widget_show( GTK_WIDGET( window ) );
}

static void file_menu_new_site_callback( GtkAction *action, 
					 gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *application;

	window = SCREEM_WINDOW( user_data );
	application = SCREEM_APPLICATION( window->application );

	screem_site_druid_begin( application, window );
}

static void file_menu_new_document_callback( GtkAction *action,
					     gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *application;

	window = SCREEM_WINDOW( user_data );
	application = SCREEM_APPLICATION( window->application );

	screem_page_druid_begin( application, window );
}

static void file_menu_new_blank_document_callback( GtkAction *action, 
						   gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *app;
	ScreemSite *site;
	ScreemPage *page;

	window = SCREEM_WINDOW( user_data );
	site = screem_window_get_current( window );
	g_object_get( G_OBJECT( site ), "app", &app, NULL );
	
	page = screem_page_new( G_OBJECT( app ) );
	screem_page_set_data( page, NULL );

	screem_site_add_page( site, page );
	g_object_unref( page );

	screem_window_set_document( window, page );
	g_object_unref( app );
}

static void file_menu_new_site_template_callback( GtkAction *action, 
						  gpointer user_data )
{
	ScreemWindow *window;

	window = SCREEM_WINDOW( user_data );
	screem_site_create_template( window );
}

static void file_menu_open_document_callback( GtkAction *action, 
					      gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *application;
	ScreemSite *dsite;
	ScreemSite *site;
	const gchar *spath;
	GSList *filenames;
	
	GSList *filters;
	GtkFileFilter *filter;
	static const gchar *filternames[] = {
		N_( "All Text Files" ),
		N_( "CSS Files" ),
		N_( "Javascript Files" ),
		N_( "Markup Files" )
	};
	gint i;
	static const gint num = G_N_ELEMENTS( filternames );

	window = SCREEM_WINDOW( user_data );
	application = SCREEM_APPLICATION( window->application );
	dsite = screem_application_get_default_site( application );

	site = screem_window_get_current( window );
	spath = screem_site_get_pathname( site );

	filters = NULL;
	for( i = 0; i < num; ++ i ) {
		filter = screem_get_file_filter( gettext( filternames[ i ] ) );
		filters = g_slist_prepend( filters, (gpointer)filter );
	}
	
	filenames = screem_fileselect_multi( _( "Open document(s) - Screem" ), 
					GTK_WINDOW( window ),
					GTK_FILE_CHOOSER_ACTION_OPEN,
					spath, filters );
	g_slist_free( filters );

	if( filenames ) {
		GSList *tmp;
		gchar *filename;
		
		for( tmp = filenames; tmp; tmp = tmp->next ) {
			filename = (gchar*)tmp->data;
			site = screem_window_get_current( window );
			if( ! screem_site_locate_page( site, filename ) ) {
				site = dsite;
			}
			screem_page_open_with_filename( site, 
						window, 
						filename );
			g_free( filename );
		}
		g_slist_free( filenames );
	}
	
}

static void file_menu_open_location_callback( GtkAction *action,
						gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *application;
	ScreemWindowDetails *details;
	GladeXML *xml;
	gchar *glade_path;
	GtkWidget *widget;

	gchar *name;
	gboolean opage;
	gchar *txt;
	const gchar *prompt;

	const gchar *widg;
	
	g_object_get( G_OBJECT( action ), "name", &name, NULL );
	if( ! strcmp( "Open Location", name ) ) {
		opage = TRUE;
		widg = "open_file_location";
	} else {
		opage = FALSE;
		widg = "open_site_location";
	}
	g_free( name );

	window = SCREEM_WINDOW( user_data );
	application = SCREEM_APPLICATION( window->application );
	details = window->details;
	
	glade_path = screem_get_glade_path();	
	xml = glade_xml_new( glade_path, widg, NULL );
	g_free( glade_path );

	widget = glade_xml_get_widget( xml, "prompt" );
	if( opage ) {
		prompt = _( "Enter the URL of the file to open" );
	} else {
		prompt = _( "Enter the URL of the site to open" );
	}
	txt = g_strconcat( "<b>", prompt, "</b>", NULL );
	gtk_label_set_markup( GTK_LABEL( widget ), txt );
	g_free( txt );

	glade_xml_signal_autoconnect( xml );
	
	widget = glade_xml_get_widget( xml, widg );
	if( gtk_dialog_run( GTK_DIALOG( widget ) ) == GTK_RESPONSE_OK ) {
		const gchar *url;
		ScreemSite *site;
		
		widget = glade_xml_get_widget( xml, "url" );
		url = gtk_entry_get_text( GTK_ENTRY( widget ) );
		
		if( opage ) {
			site = screem_window_get_current( window );
			if( ! screem_site_locate_page( site, url ) ) {
				site = screem_application_get_default_site( application );
			}
			screem_page_open_with_filename( site, window,
							url );
		} else {
			screem_site_open_with_filename( window, 
						application, url );
		}
	}
	widget = glade_xml_get_widget( xml, widg );
	gtk_widget_destroy( widget );

	g_object_unref( G_OBJECT( xml ) );
}

static void file_menu_save_document_callback( GtkAction *action, 
					      gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *application;
	ScreemPage *page;
	const gchar *filepath;
	gchar *temp;
	gchar *primary;
	GError *error;

	window = SCREEM_WINDOW( user_data );
	application = SCREEM_APPLICATION( window->application );

	page = g_object_get_data( G_OBJECT( window ), "tab_popup" );
	if( ! page ) {
		page = screem_window_get_document( window );
	}
	g_object_set_data( G_OBJECT( window ), "tab_popup", NULL );

	/* Check if the page has a file path. */
	filepath = screem_page_get_pathname(page);
	if( filepath && *filepath != '\0' ) {
		/* Filename is normal, try a save straight away */
		if( ! screem_page_save( page, &error ) ) {
			/* failed to save! */
			temp = gnome_vfs_unescape_string_for_display( filepath );
			primary = g_strdup_printf( _( "Failed to save: %s" ), temp );
			screem_hig_alert( GTK_STOCK_DIALOG_ERROR,
				primary, error->message,
				GTK_WIDGET( user_data ) );
			g_free( primary );
			g_free( temp );
			g_error_free( error );
		}
	} else {
		/* If there's no file name, request one. */
		file_menu_save_document_as_callback( action, user_data );
	}
}

static void file_menu_save_document_as_callback( GtkAction *action, 
						 gpointer user_data )
{
	ScreemWindow *window;
	ScreemPage *page;
	
	window = SCREEM_WINDOW( user_data );

	page = g_object_get_data( G_OBJECT( window ), "tab_popup" );
	if( ! page ) {
		page = screem_window_get_document( window );
	}
	g_object_set_data( G_OBJECT( window ), "tab_popup", NULL );

	if( screem_page_save_as( page, window ) == SAVE_AS_FAILED ) {
		GtkWidget *widget;

		widget = gtk_message_dialog_new( GTK_WINDOW( window ),
						 GTK_DIALOG_MODAL,
						 GTK_MESSAGE_ERROR,
						 GTK_BUTTONS_OK,
						 _( "Failed to save page" ) );
		gtk_window_set_transient_for( GTK_WINDOW( widget ),
					      GTK_WINDOW( window ) );
		gtk_dialog_run( GTK_DIALOG( widget ) );
		gtk_widget_destroy( widget );
	} 
}

static void file_menu_save_template_as_callback( GtkAction *action, 
						 gpointer user_data )
{
	ScreemWindow *window;
	ScreemPage *page;
	const gchar *pathname;
	gchar *pdata;
	gchar *ndata;
	gint status;
	GError *error;
	gchar *temp;
	gchar *primary;
	
	window = SCREEM_WINDOW( user_data );

	page = g_object_get_data( G_OBJECT( window ), "tab_popup" );
	if( ! page ) {
		page = screem_window_get_document( window );
	}
	g_object_set_data( G_OBJECT( window ), "tab_popup", NULL );

	status = screem_page_save_as( page, window );
	if( status == SAVE_AS_FAILED ) {
		GtkWidget *widget;

		widget = gtk_message_dialog_new( GTK_WINDOW( window ),
						 GTK_DIALOG_MODAL,
						 GTK_MESSAGE_ERROR,
						 GTK_BUTTONS_OK,
						 _( "Failed to save template" ) );
		gtk_window_set_transient_for( GTK_WINDOW( widget ),
					      GTK_WINDOW( window ) );
		gtk_dialog_run( GTK_DIALOG( widget ) );
		gtk_widget_destroy( widget );
	} else if( status == SAVE_AS_OK ) {
		pathname = screem_page_get_pathname( page );
		pdata = screem_page_get_data( page );

		ndata = g_strconcat( "<!-- #BeginTemplate \"", 
				pathname, "\" -->\n",
				pdata, "\n<!-- #EndTemplate -->", NULL );
		g_free( pdata );
		screem_page_set_data( page, ndata );
		g_free( ndata );
		if( ! screem_page_save( page, &error ) ) {
			/* failed to save template */
			temp = gnome_vfs_unescape_string_for_display( pathname );
			primary = g_strdup_printf( _( "Failed to save template: %s" ), temp );
			screem_hig_alert( GTK_STOCK_DIALOG_ERROR,
				primary, error->message,
				GTK_WIDGET( user_data ) );
			g_free( primary );
			g_free( temp );
			g_error_free( error );
		}
	}
}

static void file_menu_save_document_copy_callback( GtkAction *action, 
		gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *app;
	gchar *filename;
	const gchar *pathname;
	GSList *filters;
	GtkFileFilter *filter;
	static const gchar *filternames[] = {
		N_( "All Text Files" ),
		N_( "CSS Files" ),
		N_( "Javascript Files" ),
		N_( "Markup Files" )
	};
	gint i;
	static const gint num = G_N_ELEMENTS( filternames );
	ScreemSite *site;
	ScreemPage *page;
	ScreemPage *npage;
	gchar *data;
	GError *error;
	gchar *temp;
	gchar *primary;

	window = SCREEM_WINDOW( user_data );
	g_object_get( G_OBJECT( window ), "app", &app, NULL );
	
	page = screem_window_get_document( window );
	site = screem_window_get_current( window );
	pathname = screem_site_get_pathname( site );

	filters = NULL;
	for( i = 0; i < num; ++ i ) {
		filter = screem_get_file_filter( gettext( filternames[ i ] ) );
		filters = g_slist_prepend( filters, (gpointer)filter );
	}
	filename = screem_fileselect( _( "Select filename to save as" ), 
					GTK_WINDOW( window ),
					GTK_FILE_CHOOSER_ACTION_SAVE,
					pathname, filters );
	if( filename ) {
		npage = screem_page_new( G_OBJECT( app ) );
		data = screem_page_get_data( page );
		screem_page_set_data( npage, data );
		g_free( data );
		screem_page_set_pathname( npage, filename );
		if( ! screem_page_save( npage, &error ) ) {
			/* failed to save template */
			temp = gnome_vfs_unescape_string_for_display( pathname );
			primary = g_strdup_printf( _( "Failed to save copy to: %s" ), temp );
			screem_hig_alert( GTK_STOCK_DIALOG_ERROR,
				primary, error->message,
				GTK_WIDGET( data ) );
			g_free( primary );
			g_free( temp );
			g_error_free( error );
		}
		g_object_unref( npage );

		g_free( filename );
	}

	g_object_unref( G_OBJECT( app ) );
}

static void file_menu_close_document_callback( GtkAction *action, 
					       gpointer user_data )
{
	ScreemWindow *window;
	ScreemPage *page;
	ScreemSite *site;
	gboolean ret;
	
	window = SCREEM_WINDOW( user_data );
	
	site = screem_window_get_current( window );

	page = g_object_get_data( G_OBJECT( window ), "tab_popup" );
	if( ! page ) {
		page = screem_window_get_document( window );
	}
	g_object_set_data( G_OBJECT( window ), "tab_popup", NULL );
	ret = screem_page_close_page( page, window, TRUE );
	if( ret ) {
		screem_site_save( site );

		if( screem_site_get_fake_flag( site ) ) {
			screem_site_remove_page( site, page );
		}
	}
}

static void file_menu_close_all_callback( GtkAction *action, 
				       gpointer user_data )
{
	ScreemWindow *window;
	ScreemPage *page;
	ScreemSite *site;
	GList *tmp;
	GList *tmp2;
	
	window = SCREEM_WINDOW( user_data );

	site = screem_window_get_current( window );

	tmp = g_list_copy( screem_mdi_get_document_list( window->details->mdi ) );
	screem_mdi_remove_all( window->details->mdi );
	
	if( screem_site_get_fake_flag( site ) ) {
		for( tmp2 = tmp; tmp2; tmp2 = tmp2->next ) {
			page = SCREEM_PAGE( tmp2->data );
			screem_site_remove_page( site, page );
		}
	}
	g_list_free( tmp );
}

static void file_menu_open_site_callback( GtkAction *action, 
					  gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *app;
	
	gchar *filename;

	window = SCREEM_WINDOW( user_data );
	g_object_get( G_OBJECT( window ), "app", &app, NULL );
	
	
	filename = screem_fileselect( _( "Open Folder as Site - Screem" ), 
				GTK_WINDOW( window ),
				GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER,
				NULL, NULL );
	if( filename ) {
		screem_site_open_with_filename( window, app, filename );

		g_free( filename );
	}
	g_object_unref( app );
}

static void file_menu_save_all_callback( GtkAction *action, 
					 gpointer user_data )
{
	ScreemWindow *window;
	ScreemSite *site;

	/* FIXME: need error reporting should any saves fail */
	
	window = SCREEM_WINDOW( user_data );
	site = screem_window_get_current( window );
	screem_site_foreach_page( site, (GFunc)screem_page_save, NULL );
}


static void file_menu_close_site_callback( GtkAction *action, 
					   gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *application;
	ScreemEditor *editor;
	ScreemSite *site;

	window = SCREEM_WINDOW( user_data );
	application = SCREEM_APPLICATION( window->application );

	editor = SCREEM_EDITOR( window->details->editor );

	site = screem_window_get_current( window );

	if( screem_site_get_fake_flag( site ) ) {
		g_assert( FALSE );
	} else if( screem_site_save_confirmation( site ) ) {
		ScreemSite *next;
		screem_site_close_site( site );
	
		screem_application_close_site( application, site );
	
		next = screem_application_get_default_site( application );
		screem_window_set_current( window, next );

		g_object_unref( site );
	}  
}

static void file_menu_print_callback( GtkAction *action, 
				      gpointer user_data )
{
	ScreemWindow *window;

	window = SCREEM_WINDOW( user_data );

	screem_window_print( window, FALSE );
}

static void file_menu_print_preview_callback( GtkAction *action, 
					      gpointer user_data )
{
	ScreemWindow *window;

	window = SCREEM_WINDOW( user_data );

	screem_window_print( window, TRUE );
}

static void file_menu_page_setup_callback( GtkAction *action,
					   gpointer user_data )
{
	screem_preferences_page_setup();
}

static void file_menu_offline_callback( GtkAction *action, 
					gpointer user_data )
{
	ScreemWindow *window;

	window = SCREEM_WINDOW( user_data );

	screem_window_toggle_online_status( window, GINT_TO_POINTER( 1 ) );
}

static void file_menu_exit_application_callback( GtkAction *action, 
						 gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *application;

	window = SCREEM_WINDOW( user_data );
	application = SCREEM_APPLICATION( window->application );

	screem_application_close_all_windows( application );
}

static void edit_menu_undo_callback( GtkAction *action, 
				     gpointer user_data )
{
	ScreemWindow *window;

	window = SCREEM_WINDOW( user_data );

	screem_view_undo( SCREEM_VIEW( window->details->current ) );
}

static void edit_menu_redo_callback( GtkAction *action, 
				     gpointer user_data )
{
	ScreemWindow *window;

	window = SCREEM_WINDOW( user_data );

	screem_view_redo( SCREEM_VIEW( window->details->current ) );
}

static void edit_menu_cut_callback( GtkAction *action, 
				    gpointer user_data )
{
	ScreemWindow *window;
	ScreemEditor *editor;

	window = SCREEM_WINDOW( user_data );
	
	editor = SCREEM_EDITOR( window->details->editor );
	
	if( GTK_WIDGET( editor ) == window->details->current ) {
		screem_editor_cut( editor );
	}
}

static void edit_menu_copy_callback( GtkAction *action, 
				     gpointer user_data )
{
	ScreemWindow *window;
	ScreemEditor *editor;

	window = SCREEM_WINDOW( user_data );
	
	editor = SCREEM_EDITOR( window->details->editor );
	
	if( GTK_WIDGET( editor ) == window->details->current ) {
		screem_editor_copy( editor );
	}
}

static void edit_menu_paste_callback( GtkAction *action, 
				      gpointer user_data )
{
	ScreemWindow *window;
	ScreemEditor *editor;

	window = SCREEM_WINDOW( user_data );
	
	editor = SCREEM_EDITOR( window->details->editor );
	
	if( GTK_WIDGET( editor ) == window->details->current ) {
		screem_editor_paste( editor );
	}
}

static void edit_menu_paste_text_callback( GtkAction *action, 
				      gpointer user_data )
{
	ScreemWindow *window;
	ScreemEditor *editor;

	window = SCREEM_WINDOW( user_data );
	
	editor = SCREEM_EDITOR( window->details->editor );
	
	if( GTK_WIDGET( editor ) == window->details->current ) {
		screem_editor_paste_unformatted( editor );
	}
}


static void edit_menu_paste_enc_callback( GtkAction *action, 
					  gpointer user_data )
{
	ScreemWindow *window;
	ScreemEditor *editor;

	window = SCREEM_WINDOW( user_data );
	
	editor = SCREEM_EDITOR( window->details->editor );
	
	if( GTK_WIDGET( editor ) == window->details->current ) {
		screem_editor_paste_encoded( editor );
	}
}

static void edit_menu_clear_callback( GtkAction *action, 
				      gpointer user_data )
{
	ScreemWindow *window;
	ScreemEditor *editor;

	window = SCREEM_WINDOW( user_data );
	
	editor = SCREEM_EDITOR( window->details->editor );
	
	if( GTK_WIDGET( editor ) == window->details->current ) {
		screem_editor_clear_selection( editor );
	}
}

static void edit_menu_select_all_callback( GtkAction *action, 
					   gpointer user_data )
{
	ScreemWindow *window;
	ScreemEditor *editor;

	window = SCREEM_WINDOW( user_data );
	
	editor = SCREEM_EDITOR( window->details->editor );
	
	if( GTK_WIDGET( editor ) == window->details->current ) {
		screem_editor_select_region( editor, 0, 0 );
	}
}

static void edit_menu_select_context_callback( GtkAction *action, 
					       gpointer user_data )
{
	ScreemWindow *window;
	ScreemEditor *editor;
	ScreemPage *page;

	guint pos;
	guint start;
	guint end;
	guint len;

	guint selstart;
	guint selend;
	
	window = SCREEM_WINDOW( user_data );
	
	editor = SCREEM_EDITOR( window->details->editor );
	page = screem_window_get_document( window );

	pos = screem_editor_get_pos( editor );

	if( screem_page_select_context( page, 
				pos, &start, &end, FALSE ) ) {

		/* hmm, why does this need to be -1 ? if not we
		 * can end up selecting an extra character */
		len = end - start - 1;
	
		selstart = 0;
		selend = 0;
		if( screem_editor_has_selection( editor, 
					&selstart, &selend ) &&
			selstart == start && selend == ( start + len ) ) {
			/* go to parent then reselect */
			edit_menu_goto_parent_callback( NULL, user_data );
			edit_menu_select_context_callback( action,
					user_data );
		} else {
			screem_editor_select_region( editor, start, len ); 
		}
	}
}

static void edit_menu_select_content_callback( GtkAction *action, 
					       gpointer user_data )
{
	ScreemWindow *window;
	ScreemEditor *editor;
	ScreemPage *page;

	guint pos;
	guint start;
	guint end;

	window = SCREEM_WINDOW( user_data );
	
	editor = SCREEM_EDITOR( window->details->editor );
	page = screem_window_get_document( window );

	pos = screem_editor_get_pos( editor );

	if( screem_page_select_content( page, 
				pos, &start, &end, FALSE ) ) {
		screem_editor_select_region( editor, start, 
					end - start );	
	}
}


static void edit_menu_mark_editable_callback( GtkAction *action, 
					      gpointer user_data )
{
	ScreemWindow *window;
	ScreemWindowDetails *details;
	GladeXML *xml;
	gchar *glade_path;
	GtkWidget *widget;
	
	window = SCREEM_WINDOW( user_data );
	details = window->details;
	
	glade_path = screem_get_glade_path();	
	xml = glade_xml_new( glade_path, "mark_editable", NULL );
	g_free( glade_path );
	
	widget = glade_xml_get_widget( xml, "mark_editable" );
	if( gtk_dialog_run( GTK_DIALOG( widget ) ) == GTK_RESPONSE_OK ) {
		const gchar *name;
		gchar *start;
		
		widget = glade_xml_get_widget( xml, "region_name" );
		name = gtk_entry_get_text( GTK_ENTRY( widget ) );
		
		start = g_strconcat( "<!-- #BeginEditable \"", name,
					"\" -->\n", NULL );
		
		screem_editor_insert_markup( SCREEM_EDITOR( details->editor ),
						start, 
						"<!-- #EndEditable -->\n" );
		g_free( start );
	}
	widget = glade_xml_get_widget( xml, "mark_editable" );
	gtk_widget_destroy( widget );

	g_object_unref( G_OBJECT( xml ) );
}

static void edit_menu_auto_indent_callback( GtkAction *action,
					gpointer user_data )
{
	ScreemWindow *window;
	ScreemEditor *editor;
	guint pos;
	
	window = SCREEM_WINDOW( user_data );
	editor = SCREEM_EDITOR( window->details->editor );

	pos = screem_editor_get_pos( editor );
	screem_editor_auto_indent( editor, pos );
}

static void edit_menu_indent_callback( GtkAction *action,
					gpointer user_data )
{
	ScreemWindow *window;
	ScreemEditor *editor;
	guint pos;
	
	window = SCREEM_WINDOW( user_data );
	editor = SCREEM_EDITOR( window->details->editor );

	pos = screem_editor_get_pos( editor );
	screem_editor_indent( editor, pos );
}

static void edit_menu_unindent_callback( GtkAction *action,
					gpointer user_data )
{
	ScreemWindow *window;
	ScreemEditor *editor;
	guint pos;
	
	window = SCREEM_WINDOW( user_data );
	editor = SCREEM_EDITOR( window->details->editor );

	pos = screem_editor_get_pos( editor );
	screem_editor_unindent( editor, pos );
}

static void edit_menu_encode_entities_callback( GtkAction *action,
						gpointer user_data )
{
	ScreemWindow *window;
	ScreemEditor *editor;
	
	window = SCREEM_WINDOW( user_data );
	editor = SCREEM_EDITOR( window->details->editor );
	
	screem_editor_encode_text( editor, FALSE );
}

static void edit_menu_url_encode_callback( GtkAction *action,
					gpointer user_data )
{
	ScreemWindow *window;
	ScreemEditor *editor;
	
	window = SCREEM_WINDOW( user_data );
	editor = SCREEM_EDITOR( window->details->editor );
	
	screem_editor_encode_text( editor, TRUE );
}

static void edit_menu_case_callback( GtkAction *action,
				gpointer user_data )
{
	ScreemWindow *window;
	ScreemEditor *editor;
	ScreemPage *page;
	gchar *name;
	guint start;
	guint end;
	gchar *text;
	gchar *tmp;
	
	window = SCREEM_WINDOW( user_data );
	editor = SCREEM_EDITOR( window->details->editor );
	page = screem_window_get_document( window );

	if( page ) {
		if( ! screem_editor_has_selection( editor, 
						&start, &end ) ) {
			start = 0;
			end = gtk_text_buffer_get_char_count( GTK_TEXT_BUFFER( page ) );
		}

		g_object_get( G_OBJECT( action ), "name", &name, NULL );
		text = screem_editor_get_text( editor, start,
				end - start );
		tmp = screem_markup_change_case( text,
				! strcmp( "TagUpper", name ) );
		screem_editor_delete_forward( editor, start,
				end - start );
		screem_editor_insert( editor, start, tmp );
		g_free( text );
		g_free( tmp );
		g_free( name );
	}
}

static void edit_menu_bookmarks_callback( GtkAction *action, 
					  gpointer user_data )
{
	ScreemWindow *window;
	ScreemEditor *editor;
	ScreemPage *page;

	window = SCREEM_WINDOW( user_data );
	editor = SCREEM_EDITOR( window->details->editor );

	page = screem_window_get_document( window );

	screem_page_edit_bookmarks( window, editor, page );
}

static void edit_menu_search_callback( GtkAction *action, 
				       gpointer user_data )
{
	ScreemWindow *window;
	gchar *name;

	window = SCREEM_WINDOW( user_data );
	g_object_get( G_OBJECT( action ), "name", &name, NULL );

	if( ! strcmp( "Find", name ) ) {
		screem_search_show_dialog( window->search, FALSE );
	} else if( ! strcmp( "FindAgain", name ) ) {
		screem_search_set_from( window->search,
				SCREEM_SEARCH_CURRENT );
		screem_search_do_find( window->search );	
	} else {
		screem_search_show_dialog( window->search, TRUE );
	}

	g_free( name );
}

static void edit_menu_goto_line_callback( GtkAction *action, 
				       gpointer user_data )
{
	ScreemWindow *window;
	ScreemWindowDetails *details;
	GladeXML *xml;
	gchar *glade_path;
	GtkWidget *widget;
	ScreemPage *page;
	guint lines;
	ScreemEditor *editor;
	
	window = SCREEM_WINDOW( user_data );
	details = window->details;
	
	page = screem_window_get_document( window );

	if( ! page ) {
		return;
	}
	
	glade_path = screem_get_glade_path();	
	xml = glade_xml_new( glade_path, "gotoline", NULL );
	g_free( glade_path );

	widget = glade_xml_get_widget( xml, "line" );
	lines = gtk_text_buffer_get_line_count( GTK_TEXT_BUFFER( page ) );
	gtk_spin_button_set_range( GTK_SPIN_BUTTON( widget ),
				0.0, (gdouble)lines );
	gtk_entry_set_activates_default( GTK_ENTRY( widget ), TRUE );
	
	widget = glade_xml_get_widget( xml, "gotoline" );
	if( gtk_dialog_run( GTK_DIALOG( widget ) ) == GTK_RESPONSE_OK ) {
		widget = glade_xml_get_widget( xml, "line" );
		lines = (guint)gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON( widget ) );
	
		editor = SCREEM_EDITOR( window->details->editor );
		screem_editor_goto_line( editor, lines );
	}
	widget = glade_xml_get_widget( xml, "gotoline" );
	gtk_widget_destroy( widget );

	g_object_unref( G_OBJECT( xml ) );
}
static void edit_menu_goto_open_callback( GtkAction *action, 
				       gpointer user_data )
{
	ScreemWindow *window;
	ScreemEditor *editor;
	ScreemPage *page;

	guint pos;
	guint start;
	guint end;

	window = SCREEM_WINDOW( user_data );
	
	editor = SCREEM_EDITOR( window->details->editor );
	page = screem_window_get_document( window );

	pos = screem_editor_get_pos( editor );

	if( screem_page_select_context( page, 
				pos, &start, &end, FALSE ) ) {
		screem_editor_set_pos( editor, start );
	}
}
static void edit_menu_goto_close_callback( GtkAction *action, 
				       gpointer user_data )
{
	ScreemWindow *window;
	ScreemEditor *editor;
	ScreemPage *page;

	guint pos;
	guint start;
	guint end;

	window = SCREEM_WINDOW( user_data );
	
	editor = SCREEM_EDITOR( window->details->editor );
	page = screem_window_get_document( window );

	pos = screem_editor_get_pos( editor );

	if( screem_page_select_context( page, 
				pos, &start, &end, FALSE ) ) {
		if( end > 0 ) {
			end --;
		}
		screem_editor_set_pos( editor, end );
	}
}

static void edit_menu_goto_parent_callback( GtkAction *action, 
				       gpointer user_data )
{
	ScreemWindow *window;
	ScreemEditor *editor;
	ScreemPage *page;

	guint pos;
	guint start;
	guint end;

	window = SCREEM_WINDOW( user_data );
	
	editor = SCREEM_EDITOR( window->details->editor );
	page = screem_window_get_document( window );

	pos = screem_editor_get_pos( editor );

	if( screem_page_select_parent_context( page, 
				pos, &start, &end, FALSE ) ) {
		screem_editor_set_pos( editor, start );
	}
}

static void view_menu_dock_show_callback( GtkAction *action,
					  gpointer user_data )
{
	ScreemWindow *window;
	GdlDockItem *item;
	gboolean show;	
	gchar *name;
		
	window = SCREEM_WINDOW( user_data );
	g_object_get( G_OBJECT( action ), "name", &name, NULL );
	
	show = gtk_toggle_action_get_active( GTK_TOGGLE_ACTION( action ) );
	item = GDL_DOCK_ITEM( g_object_get_data( G_OBJECT( action ), "item" ) );

	/* check item here as it is NULL when we set the initial toggle state */
	if( item ) {
		if( show ) {
			gdl_dock_item_show_item( item );
		} else {
			gdl_dock_item_hide_item( item );
		}
	} else if( ! strcmp( "ViewMainToolbar", name ) ) {
		if( show ) {
			egg_editable_toolbar_show( EGG_EDITABLE_TOOLBAR( window->toolbar ), "Main Toolbar" );
		} else { 
			egg_editable_toolbar_hide( EGG_EDITABLE_TOOLBAR( window->toolbar ), "Main Toolbar" );
			
		}
		gconf_client_set_bool( window->application->client,
					"/apps/screem/ui/show_main_toolbar",
					show, NULL );
	} else if( ! strcmp( "ViewWizardToolbar", name ) ) {
		GtkWidget *widget;
	
		widget = gtk_ui_manager_get_widget( GTK_UI_MANAGER( window->merge ),
							"/Wizards Toolbar" );	
		if( widget ) {
			if( show ) {
				gtk_widget_show( widget );
			} else { 
				gtk_widget_hide( widget );
			}
		}
		gconf_client_set_bool( window->application->client,
					"/apps/screem/ui/show_wizard_toolbar",
					show, NULL );
	}
	g_free( name );
}

static void view_menu_view_callback( GtkRadioAction *action,
				GtkRadioAction *current,
				gpointer user_data )
{
	ScreemWindow *window = SCREEM_WINDOW( user_data );
	
	switch( gtk_radio_action_get_current_value( current ) ) {
		case 0:
			screem_window_change_view( window, 
					window->details->editor );
			break;
		case 1:
			screem_window_change_view( window, 
					window->details->preview );
			break;
		case 2:
			screem_window_change_view( window, 
					window->details->link_view );
			break;
		case 3:
			screem_window_change_view( window, 
					window->details->tree_view );
			break;
		default:
			g_assert( FALSE );
			break;
	}
}

static void view_menu_toolbar_callback( GtkRadioAction *action,
				GtkRadioAction *current,
				gpointer user_data )
{
	GtkWidget *toolbar;
	GtkToolbarStyle style;
	EggToolbarsModel *model;
	EggTbModelFlags flags;
	gint bars;
	gint i;
	gint value;
	GConfClient *client;
	ScreemWindow *window = SCREEM_WINDOW( user_data );

	toolbar = gtk_ui_manager_get_widget( GTK_UI_MANAGER( window->merge ),
			"/Wizards Toolbar" );	
	
	model = EGG_TOOLBARS_MODEL( window->toolbar_model );
	bars = egg_toolbars_model_n_toolbars( model );
	
	value = gtk_radio_action_get_current_value( current );

	client = gconf_client_get_default();
	gconf_client_set_int( client, "/apps/screem/ui/toolbar_style", 
			value, NULL );
	g_object_unref( client );
	
	switch( value ) {
		case 1:
			style = GTK_TOOLBAR_ICONS;
			flags = EGG_TB_MODEL_ICONS_ONLY;
			break;
		case 2: 
			style = GTK_TOOLBAR_BOTH;
			flags = EGG_TB_MODEL_ICONS_TEXT;
			break;
		case 3:
			style = GTK_TOOLBAR_BOTH_HORIZ;
			flags = EGG_TB_MODEL_ICONS_TEXT_HORIZ;
			break;
		case 4:
			style = GTK_TOOLBAR_TEXT;
			flags = EGG_TB_MODEL_TEXT_ONLY;
			break;
		case 0:
		default:
			if( toolbar ) {
				gtk_toolbar_unset_style( GTK_TOOLBAR( toolbar ) );
			}
			egg_toolbars_model_set_flags( model, 
				EGG_TB_MODEL_NOT_REMOVABLE, 0 );
			for( i = 1; i < bars; ++ i ) {
				egg_toolbars_model_set_flags( model, 
						0, i );
			}
			return;
			break;
	}
	if( toolbar ) {
		gtk_toolbar_set_style( GTK_TOOLBAR( toolbar ), style );
	}
	egg_toolbars_model_set_flags( model, 
			EGG_TB_MODEL_NOT_REMOVABLE | flags, 0 );
	for( i = 1; i < bars; ++ i ) {
		egg_toolbars_model_set_flags( model, flags, i );
	}
}

static void insert_doctype_sel_callback( GtkTreeSelection *sel,
					 GtkWidget *entry )
{
	GtkTreeModel *model;
	GtkTreeIter it;
	gchar *public;
	
	if( gtk_tree_selection_get_selected( sel, &model, &it ) ) {

		gtk_tree_model_get( model, &it,
				0, &public,
				-1 );
		
		gtk_entry_set_text( GTK_ENTRY( entry ), public );
		g_free( public );
	}
}

static void insert_doctype_activate_callback( GtkTreeView *view,
					GtkTreePath *path,
					GtkTreeViewColumn *col,
					GtkDialog *dialog )
{
	gtk_dialog_response( dialog, GTK_RESPONSE_OK );
}

static void insert_menu_doctype_callback( GtkAction *action, 
					  gpointer user_data )
{
	ScreemWindow *window = SCREEM_WINDOW( user_data );
	ScreemApplication *application;
	ScreemSession *session;
	ScreemDTDDB *db;
	GladeXML *xml;
	gchar *glade_path;
	gchar *defaultdtd;
	GtkWidget *widget;
	gint button;

	GtkCellRenderer *rend;
	GtkTreeViewColumn *col;
	GtkListStore *store;
	GtkTreeSelection *sel;
	GtkWidget *dialog;

	GtkTreeIter *iter;
	GtkTreePath *path;
	
	application = SCREEM_APPLICATION( window->application );
	session = screem_application_get_session( application );

	glade_path = screem_get_glade_path();
	xml = glade_xml_new( glade_path, "insertdtd", NULL );
	dialog = glade_xml_get_widget( xml, "insertdtd" );
	g_free( glade_path );

	db = screem_application_get_dtd_db( application );
	store = screem_dtd_db_get_store( db );

	widget = glade_xml_get_widget( xml, "doctypes" );
	rend = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_pack_start( col, rend, TRUE );
	gtk_tree_view_column_set_resizable( col, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( widget ), col );
	gtk_tree_view_column_set_attributes( col, rend,
					     "text", 0, NULL ); 
	gtk_tree_view_column_set_title( col, "PUBLIC" );

	rend = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_pack_start( col, rend, TRUE );
	gtk_tree_view_column_set_resizable( col, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( widget ), col );
	gtk_tree_view_column_set_attributes( col, rend,
					     "text", 1, NULL ); 
	gtk_tree_view_column_set_title( col, "SYSTEM" );
	gtk_tree_view_set_model( GTK_TREE_VIEW( widget ), 
			GTK_TREE_MODEL( store ) );

	g_signal_connect( G_OBJECT( widget ), "row_activated",
			  G_CALLBACK(insert_doctype_activate_callback),
			  dialog );
	
	sel = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );
	widget = glade_xml_get_widget( xml, "public" );
	g_signal_connect( G_OBJECT( sel ), "changed",
			  G_CALLBACK( insert_doctype_sel_callback ),
			  widget );
	
	defaultdtd = gconf_client_get_string(application->client,
					     "/apps/screem/editor/default_dtd",
					     NULL );
	if( ! defaultdtd ) {
		defaultdtd = g_strdup( "-//W3C//DTD HTML 4.01 Transitional//EN" );
	}
	gtk_entry_set_text( GTK_ENTRY( widget ), defaultdtd );

	iter = screem_support_find_in_list( GTK_LIST_STORE( store ),
			0, defaultdtd );
	if( iter ) {
		gtk_tree_selection_select_iter( sel, iter );
		widget = glade_xml_get_widget( xml, "doctypes" );
		path = gtk_tree_model_get_path( GTK_TREE_MODEL( store ),
				iter );
		gtk_tree_view_scroll_to_cell( GTK_TREE_VIEW( widget ),
				path, col, FALSE, 0.0, 0.0 );
		gtk_tree_path_free( path );
		gtk_tree_iter_free( iter );
	}
	
	g_free( defaultdtd );

	gtk_window_set_transient_for( GTK_WINDOW( dialog ),
				      GTK_WINDOW( window ) );

	screem_session_restore_dialog( session, dialog );

	button = gtk_dialog_run( GTK_DIALOG( dialog ) );

	screem_session_store_dialog( session, dialog );
	
	if( button == GTK_RESPONSE_OK ) {
		ScreemEditor *editor;
		const gchar *txt;

		editor = SCREEM_EDITOR( window->details->editor );

		widget = glade_xml_get_widget( xml, "public" );
		txt = gtk_entry_get_text( GTK_ENTRY( widget ) );

		if( ! strncmp( "<!DOCTYPE", txt, strlen( "<!DOCTYPE" ) ) ) {
			/* full doctype tag was entered, insert */
			screem_editor_insert( editor, 0, txt );
		} else {
			/* build doctype tag */
			gchar *tag;
			gchar *systemid;
			gchar *root;
			ScreemDTD *dtd;
		
			dtd = screem_dtd_db_get_dtd( db, txt, NULL ); 
			root = screem_dtd_db_get_root( db, txt, NULL );
			if( ! root ) {
				root = g_strdup( "unknown" );
			}

			systemid = screem_dtd_db_get_system_id( db,
							     txt,
							     NULL );
		
			tag = g_strconcat( "<!DOCTYPE ", root,
					   " PUBLIC \"", txt, 
					   "\" \"", systemid,
					   "\">", NULL );
			screem_editor_insert( editor, 0, tag );
			g_free( root );
			g_free( systemid );
			g_free( tag );
		}
	}

	gtk_widget_destroy( dialog );
	g_object_unref( G_OBJECT( xml ) );
}

static void insert_menu_file_callback( GtkAction *action, 
				       gpointer user_data )
{
	ScreemWindow *window;
	ScreemWindowDetails *details;
	gchar *filename;
	const gchar *spath;
	ScreemSite *site;

	GSList *filters;
	GtkFileFilter *filter;
	static const gchar *filternames[] = {
		N_( "All Text Files" ),
		N_( "CSS Files" ),
		N_( "Javascript Files" ),
		N_( "Markup Files" )
	};
	gint i;
	static const gint num = G_N_ELEMENTS( filternames );
	ScreemEditor *editor;
	
	window = SCREEM_WINDOW( user_data );
	details = window->details;

	site = screem_window_get_current( window );
	spath = screem_site_get_pathname( site );

	filters = NULL;
	for( i = 0; i < num; ++ i ) {
		filter = screem_get_file_filter( gettext( filternames[ i ] ) );
		filters = g_slist_prepend( filters, (gpointer)filter );
	}
	
	filename = screem_fileselect( _( "Open document - Screem" ), 
			GTK_WINDOW( window ),
			GTK_FILE_CHOOSER_ACTION_OPEN,
			spath, filters );
	g_slist_free( filters );
	
	if( filename && details->current == details->editor ) {
		editor = SCREEM_EDITOR( details->editor );
		screem_editor_insert_file( editor, filename );
	}

	g_free( filename );
}

static void site_menu_site_settings_callback( GtkAction *action, 
						gpointer user_data )
{
	ScreemWindow *window;
	ScreemSite *site;

	window = SCREEM_WINDOW( user_data );

	site = screem_window_get_current( window );

	screem_site_settings_dialog( site, window );
}

static void toolbar_editor_destroy_cb( GtkWidget *editor,
		ScreemWindow *window )
{	
	GtkWidget *widget;
	
	egg_editable_toolbar_set_edit_mode( EGG_EDITABLE_TOOLBAR( window->toolbar ), FALSE );

	widget = gtk_ui_manager_get_widget( GTK_UI_MANAGER( window->merge ),
							"/Wizards Toolbar" );	
	if( widget ) {
		gtk_widget_set_sensitive( widget, TRUE );
	}
}

static void
toolbar_editor_response_cb (GtkDialog  *dialog,
			    gint response_id,
			    gpointer data)
{
	ScreemWindow *window;
	ScreemApplication *app;
	ScreemSession *session;
	EggToolbarsModel *model;
	int n;
	gchar *temp;
	gchar *file;
	
	window = SCREEM_WINDOW( data );
	app = SCREEM_APPLICATION( window->application );
	session = screem_application_get_session( app );
	model = EGG_TOOLBARS_MODEL( window->toolbar_model );
	
	switch (response_id)
	{
	case 0: /*RESPONSE_ADD_TOOLBAR*/
		n = egg_toolbars_model_n_toolbars (EGG_TOOLBARS_MODEL (model));
		egg_toolbars_model_add_toolbar (EGG_TOOLBARS_MODEL (model),
						n, "UserCreated");
		break;
	case GTK_RESPONSE_CLOSE:
	default:
		screem_session_store_dialog( session, 
				GTK_WIDGET( dialog ) );
	
		gtk_widget_destroy (GTK_WIDGET (dialog));
		break;
	}


	temp = screem_get_dot_dir();
	file = g_build_filename( temp, "screem-editable-toolbars.xml",
			NULL );
	g_free( temp );
	egg_toolbars_model_save( model, file, "1.0" );
	g_free( file );
}

static void edit_menu_toolbars_callback( GtkAction *action, 
					 gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *app;
	ScreemSession *session;
	GtkWidget *editor;
	GtkWidget *t;
	GtkWidget *dialog;
	EggToolbarsModel *model;

	gchar *temp;
	gchar *file;

	temp = screem_get_dot_dir();
	file = g_build_filename( temp, "screem-editable-toolbars.xml",
			NULL );
	g_free( temp );

	window = SCREEM_WINDOW( user_data );
	app = SCREEM_APPLICATION( window->application );
	session = screem_application_get_session( app );
	t = window->toolbar;
	model = EGG_TOOLBARS_MODEL( window->toolbar_model );
		
	dialog = gtk_dialog_new ();
	gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);
	gtk_window_set_title (GTK_WINDOW (dialog), _("Toolbar Editor"));
        gtk_window_set_transient_for (GTK_WINDOW (dialog),
				      GTK_WINDOW (window));
	gtk_window_set_role( GTK_WINDOW( dialog ), "screem_toolbar_editor" );

	editor = egg_toolbar_editor_new
		(GTK_UI_MANAGER (window->merge),
		 EGG_TOOLBARS_MODEL (model));
	egg_toolbar_editor_load_actions (EGG_TOOLBAR_EDITOR (editor),
					UIDATADIR"/screem-editable-toolbars.xml" );
	g_free( file );

	gtk_container_set_border_width (GTK_CONTAINER (EGG_TOOLBAR_EDITOR (editor)), 5);
	gtk_box_set_spacing (GTK_BOX (EGG_TOOLBAR_EDITOR (editor)), 5);
	gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), editor);
	g_signal_connect (editor, "destroy",
			  G_CALLBACK (toolbar_editor_destroy_cb),
			  window );
	gtk_widget_show (editor);
	
	gtk_box_set_spacing (GTK_BOX (GTK_DIALOG (dialog)->vbox), 2);

	gtk_container_set_border_width (GTK_CONTAINER (GTK_DIALOG (dialog)), 5);
	
	gtk_dialog_add_button (GTK_DIALOG (dialog),
			       _("_Add a New Toolbar"), 0 );/*RESPONSE_ADD_TOOLBAR);*/
	gtk_dialog_add_button (GTK_DIALOG (dialog),
			       GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE);
	
	g_signal_connect (G_OBJECT (dialog), "response",
			  G_CALLBACK (toolbar_editor_response_cb), 
			  window );
	gtk_window_set_default_size( GTK_WINDOW( dialog ),
			500, 325 );

	screem_session_restore_dialog( session, dialog );

	gtk_widget_show (dialog);
	
	egg_editable_toolbar_set_edit_mode (EGG_EDITABLE_TOOLBAR (t), TRUE);
	t = gtk_ui_manager_get_widget( GTK_UI_MANAGER( window->merge ),
					"/Wizards Toolbar" );	
	if( t ) {
		gtk_widget_set_sensitive( t, FALSE );
	}

}

static void edit_menu_preferences_callback( GtkAction *action, 
					    gpointer user_data )
{
	ScreemApplication *app;

	g_object_get( G_OBJECT( user_data ), "app", &app, NULL );
	
	screem_preferences_edit( app );

	g_object_unref( app );
}

/* FIXME: put some place sensible, doesn't belong in ScreemCVS though */
static gboolean screem_cvs_dialog( ScreemWindow *window,
				   gchar **log,
				   gchar **root,
				   gchar **module,
				   gchar **base,
				   gchar **flags )
{
	ScreemApplication *application;
       	GladeXML *xml;
	gchar *glade_path;
	GtkWidget *dialog;
	GtkWidget *widget;
	GtkTextBuffer *buffer;
	GtkTextIter it;
	GtkTextIter eit;
	const gchar *value;
	gint response;

	application = SCREEM_APPLICATION( window->application );

	glade_path = screem_get_glade_path();
	xml = glade_xml_new( glade_path, "cvs_dialog", NULL );
	g_free( glade_path );

	dialog = glade_xml_get_widget( xml, "cvs_dialog" );

	if( log ) {
		widget = glade_xml_get_widget( xml, "logframe" );
		gtk_widget_show( widget );
	}

	if( root || module || base ) {
		widget = glade_xml_get_widget( xml, "checkoutframe" );
		gtk_widget_show( widget );
	}

	widget = glade_xml_get_widget( xml, "root" );
	gtk_widget_set_sensitive( widget, root != NULL );
	if( root ) {
		widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
		gtk_entry_set_text( GTK_ENTRY( widget ), *root );
	}
	widget = glade_xml_get_widget( xml, "module" );
	gtk_widget_set_sensitive( widget, module != NULL );
	if( module ) {
		widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
		gtk_entry_set_text( GTK_ENTRY( widget ), *module );
	}
	widget = glade_xml_get_widget( xml, "checkoutto" );
	gtk_widget_set_sensitive( widget, base != NULL );
	if( base ) {
		widget = gnome_file_entry_gtk_entry( GNOME_FILE_ENTRY(widget));
		gtk_entry_set_text( GTK_ENTRY( widget ), *base );
	}
	
	if( flags ) {
		widget = glade_xml_get_widget( xml, "flagframe" );
		gtk_widget_show( widget );

		widget = glade_xml_get_widget( xml, "cvsflags" );
		widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
		gtk_entry_set_text( GTK_ENTRY( widget ), *flags );
	}

	glade_xml_signal_autoconnect( xml );
	
	gtk_window_set_transient_for( GTK_WINDOW( dialog ),
				      GTK_WINDOW( window ) );
	response = gtk_dialog_run( GTK_DIALOG( dialog ) );

	if( response == GTK_RESPONSE_OK ) {
		if( log ) {
			widget = glade_xml_get_widget( xml, "cvslog" );
			buffer = gtk_text_view_get_buffer( GTK_TEXT_VIEW( widget ) );
			gtk_text_buffer_get_start_iter( buffer, &it );
			gtk_text_buffer_get_end_iter( buffer, &eit );
			*log = gtk_text_buffer_get_text( buffer, &it, &eit,
							  TRUE );
		}

		if( root ) {
			widget = glade_xml_get_widget( xml, "root" );
			widget = gnome_entry_gtk_entry( GNOME_ENTRY(widget) );
			value = gtk_entry_get_text( GTK_ENTRY( widget ) );
			*root = g_strdup( value );
		}
		if( module ) {
			widget = glade_xml_get_widget( xml, "module" );
			widget = gnome_entry_gtk_entry( GNOME_ENTRY(widget) );
			value = gtk_entry_get_text( GTK_ENTRY( widget ) );
			*module = g_strdup( value );
		}
		if( base ) {
			widget = glade_xml_get_widget( xml, "checkoutto" );
			*base = gtk_editable_get_chars( GTK_EDITABLE(widget),
							0, -1 );
		}
		
		if( flags ) {
			widget = glade_xml_get_widget( xml, "cvsflags" );
			widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget) );
			value = gtk_entry_get_text( GTK_ENTRY( widget ) );
			*flags = g_strdup( value );
		}
	}
	gtk_widget_destroy( dialog );
	g_object_unref( G_OBJECT( xml ) );

	return ( response == GTK_RESPONSE_OK );
}

static void cvs_menu_checkout_site_callback( GtkAction *action, 
						gpointer user_data )
{
	ScreemWindow *window;
	ScreemCVS *cvs;
	gchar *flags;
	gchar *module;
	gchar *root;
	gchar *base;

	window = SCREEM_WINDOW( user_data );
	cvs = window->details->cvs;

	flags = "";
	module = "";
	root = (gchar*)g_getenv( "CVSROOT" );
	if( ! root ) {
		root = "";
	}
	base = "";

	if( screem_cvs_dialog( window, NULL, &root, &module, &base, 
			       &flags ) ) {
		
		g_object_set( G_OBJECT( cvs ), 
			      "root", root,
			      "base", base, NULL );
		screem_cvs_checkout( cvs, flags, module );
		
		g_free( base );
		g_free( flags );
		g_free( module );
		g_free( root );
	}
}

static void cvs_menu_update_site_callback( GtkAction *action, 
					   gpointer user_data )
{
	ScreemWindow *window;
	ScreemCVS *cvs;
	ScreemSite *site;

	gchar *root;
	gchar *module;
	gchar *base;
	gchar *flags;

	window = SCREEM_WINDOW( user_data );
	cvs = window->details->cvs;

	site = screem_window_get_current( window );

	root = (gchar*)screem_site_get_cvs_root( site );
	module = (gchar*)screem_site_get_name( site );
	base = (gchar*)screem_site_get_pathname( site );
	flags = "";

	if( screem_cvs_dialog( window, NULL, &root,
			       &module, &base, &flags ) ) {

		g_object_set( G_OBJECT( cvs ),
			      "root", root,
			      "base", base, NULL );

		screem_cvs_update( cvs, NULL, flags );

		g_free( module );
		g_free( base );
		g_free( flags );
	}
}


static void cvs_menu_update_page_callback( GtkAction *action, 
					   gpointer user_data )
{
	ScreemWindow *window;
	ScreemCVS *cvs;
	ScreemSite *site;
	ScreemPage *page;

	const gchar *pathname;

	gchar *root;
	gchar *module;
	gchar *flags;
	gchar *base;
	
	gchar *temp;

	gchar *name;
	
	window = SCREEM_WINDOW( user_data );
	cvs = window->details->cvs;

	g_object_get( G_OBJECT( action ), "name", &name, NULL );
	
	if( strcmp( "UpdateFile", name ) ) {
		page = screem_window_get_document( window );
		
		pathname = screem_page_get_pathname( page );
	} else {
		GtkWidget *view;
		GtkTreeModel *model;
		GtkTreeSelection *selection;
		GtkTreeIter it;
		GValue value = { 0 };
		ScreemSiteViewNodeInfo *info;

		view = GTK_WIDGET( g_object_get_data( G_OBJECT( window ), 
							"siteview" ) );
		model = gtk_tree_view_get_model( GTK_TREE_VIEW( view ) );
		selection = gtk_tree_view_get_selection( GTK_TREE_VIEW(view) );

		gtk_tree_selection_get_selected( selection, &model, &it );
		gtk_tree_model_get_value( model, &it, FILE_BROWSER_DATA_COL, &value );
		info = g_value_get_pointer( &value );
		if( info ) {
			pathname = info->fullname;
		} else {
			pathname = NULL;
		}
		g_value_unset( &value );
	}

	site = screem_window_get_current( window );
	if( ! screem_site_get_fake_flag( site ) ) {
		root = (gchar*)screem_site_get_cvs_root( site );
	       	module = (gchar*)screem_site_get_name( site );
		base = (gchar*)screem_site_get_pathname( site );
		temp = NULL;
	} else {
		root = (gchar*)g_getenv( "CVSROOT" );
		if( ! root ) {
			root = "";
		}
		module = "";
		temp = base = g_path_get_dirname( pathname );
	}
	flags = "";

	if( pathname && screem_cvs_dialog( window, NULL, 
					   &root, &module, &base,
					   &flags ) ) {

		g_object_set( G_OBJECT( cvs ),
			      "root", root,
			      "base", base, NULL );

		screem_cvs_update( cvs, pathname, flags );

		g_free( root );
		g_free( module );
		g_free( base );
		g_free( flags );
	} 

	if( temp ) {
		g_free( temp );
	}

	g_free( name );
}

static void cvs_menu_import_site_callback( GtkAction *action, 
					   gpointer user_data )
{
	ScreemWindow *window;
	ScreemCVS *cvs;
	ScreemSite *site;

	gchar *log;
	gchar *root;
	gchar *module;
	gchar *base;
	gchar *flags;

	window = SCREEM_WINDOW( user_data );
	cvs = window->details->cvs;

	site = screem_window_get_current( window );

	log = "";
	root = (gchar*)screem_site_get_cvs_root( site );
	if( ! root ) {
		root = (gchar*)g_getenv( "CVSROOT" );
		if( ! root ) {
			root = "";
		}
	}
	module = (gchar*)screem_site_get_name( site );
	base = (gchar*)screem_site_get_pathname( site );
	flags = "";

	if( screem_cvs_dialog( window, &log, 
			       &root, &module, &base, &flags ) ) {

		g_object_set( G_OBJECT( cvs ),
			      "root", root,
			      "base", base, NULL );
	
		screem_cvs_import( cvs, flags, log, module );

		g_free( log );
		g_free( root );
		g_free( module );
		g_free( base );
		g_free( flags );
	}
}

static void cvs_menu_commit_site_callback( GtkAction *action, 
					   gpointer user_data )
{
	ScreemWindow *window;
	ScreemCVS *cvs;
	ScreemSite *site;

	gchar *log;
	const gchar *root;
	const gchar *base;
	gchar *flags;

	window = SCREEM_WINDOW( user_data );
	cvs = window->details->cvs;

	site = screem_window_get_current( window );

	log = "";
	root = screem_site_get_cvs_root( site );
	if( ! root ) {
		root = g_getenv( "CVSROOT" );
		if( ! root ) {
			root = "";
		}
	}
	base = screem_site_get_pathname( site );
	flags = "";

	if( screem_cvs_dialog( window, &log, 
			       NULL, NULL, NULL, &flags ) ) {

		g_object_set( G_OBJECT( cvs ),
			      "root", root,
			      "base", base, NULL );
	
		screem_cvs_commit( cvs, NULL, flags, log );

		g_free( log );
		g_free( flags );
	}
}

static void cvs_menu_commit_page_callback( GtkAction *action, 
					   gpointer user_data )
{
	ScreemWindow *window;
	ScreemCVS *cvs;
	ScreemSite *site;
	ScreemPage *page;

	const gchar *pathname;

	gchar *log;
	const gchar *root;
	const gchar *base;
	gchar *module;
	gchar *flags;

	gchar *temp;

	gchar **troot;
	gchar **tbase;
	gchar **tmodule;

	gchar *name;
	
	window = SCREEM_WINDOW( user_data );
	cvs = window->details->cvs;

	g_object_get( G_OBJECT( action ), "name", &name, NULL );
	
	if( strcmp( "CommitFile", name ) ) {
		page = screem_window_get_document( window );
		pathname = screem_page_get_pathname( page );
	} else {
		GtkWidget *view;
		GtkTreeModel *model;
		GtkTreeSelection *selection;
		GtkTreeIter it;
		GValue value = { 0 };
		ScreemSiteViewNodeInfo *info;

		view = GTK_WIDGET( g_object_get_data( G_OBJECT( window ), 
							"siteview" ) );
		model = gtk_tree_view_get_model( GTK_TREE_VIEW( view ) );
		selection = gtk_tree_view_get_selection( GTK_TREE_VIEW(view) );

		gtk_tree_selection_get_selected( selection, &model, &it );
		gtk_tree_model_get_value( model, &it, FILE_BROWSER_DATA_COL, &value );
		info = g_value_get_pointer( &value );
		if( info ) {
			pathname = info->fullname;
		} else {
			pathname = NULL;
		}
		g_value_unset( &value );
	}

	site = screem_window_get_current( window );

	if( ! screem_site_get_fake_flag( site ) ) {
		log = "";
		root = screem_site_get_cvs_root( site );
		if( ! root ) {
			root = g_getenv( "CVSROOT" );
			if( ! root ) {
				root = "";
			}
		}
		base = screem_site_get_pathname( site );
		troot = NULL;
		tmodule = NULL;
		tbase = NULL;
		temp = NULL;
	} else {
		root = g_getenv( "CVSROOT" );
		if( ! root ) {
			root = "";
		}
		if( pathname ) {
			temp =  g_path_get_dirname( pathname );
		} else {
			temp = NULL;
		}
		base = (const gchar*)temp;

		module = "";
		tbase = (gchar**)&base;
		tmodule = &module;
		troot = (gchar**)&root;
	}
	flags = "";

	if( pathname && screem_cvs_dialog( window, &log, 
					   troot, tmodule, tbase,
					   &flags ) ) {

		g_object_set( G_OBJECT( cvs ),
			      "root", root,
			      "base", base, NULL );
	
		screem_cvs_commit( cvs, pathname, flags, log );

		g_free( log );
		g_free( flags );

		if( troot ) {
			g_free( *troot );
		}
		if( tmodule ) {
			g_free( *tmodule );
		}
		if( tbase ) {
			g_free( *tbase );
		}
	} 

	if( temp ) {
		g_free( temp );
	}

	g_free( name );
}

static void cvs_menu_add_page_callback( GtkAction *action, 
					gpointer user_data )
{
	ScreemWindow *window;
	ScreemCVS *cvs;
	ScreemSite *site;
	ScreemPage *page;

	const gchar *pathname;

	gchar *log;
	const gchar *root;
	const gchar *base;
	gchar *module;
	gchar *flags;

	gchar *temp;

	gchar **troot;
	gchar **tbase;
	gchar **tmodule;

	gchar *name;
	
	window = SCREEM_WINDOW( user_data );
	cvs = window->details->cvs;

	g_object_get( G_OBJECT( action ), "name", &name, NULL );
	
	if( strcmp( "AddFile", name ) ) {
		page = screem_window_get_document( window );
		pathname = screem_page_get_pathname( page );
	} else {
		GtkWidget *view;
		GtkTreeModel *model;
		GtkTreeSelection *selection;
		GtkTreeIter it;
		GValue value = { 0 };
		ScreemSiteViewNodeInfo *info;

		view = GTK_WIDGET( g_object_get_data( G_OBJECT( window ), 
							"siteview" ) );
		model = gtk_tree_view_get_model( GTK_TREE_VIEW( view ) );
		selection = gtk_tree_view_get_selection( GTK_TREE_VIEW(view) );

		gtk_tree_selection_get_selected( selection, &model, &it );
		gtk_tree_model_get_value( model, &it, FILE_BROWSER_DATA_COL, &value );
		info = g_value_get_pointer( &value );
		if( info ) {
			pathname = info->fullname;
		} else {
			pathname = NULL;
		}
		g_value_unset( &value );
	}

	site = screem_window_get_current( window );

	if( ! screem_site_get_fake_flag( site ) ) {
		log = "";
		root = screem_site_get_cvs_root( site );
		if( ! root ) {
			root = g_getenv( "CVSROOT" );
			if( ! root ) {
				root = "";
			}
		}
		base = screem_site_get_pathname( site );
		troot = NULL;
		tmodule = NULL;
		tbase = NULL;
		temp = NULL;
	} else {
		root = g_getenv( "CVSROOT" );
		if( ! root ) {
			root = "";
		}
		if( pathname ) {
			temp =  g_path_get_dirname( pathname );
		} else {
			temp = NULL;
		}
		base = (const gchar*)temp;

		module = "";
		tbase = (gchar**)&base;
		tmodule = &module;
		troot = (gchar**)&root;
	}
	flags = "";

	if( pathname && screem_cvs_dialog( window, &log, 
					   troot, tmodule, tbase,
					   &flags ) ) {

		g_object_set( G_OBJECT( cvs ),
			      "root", root,
			      "base", base, NULL );
	
		screem_cvs_add( cvs, pathname, flags, log );

		g_free( log );
		g_free( flags );

		if( troot ) {
			g_free( *troot );
		}
		if( tmodule ) {
			g_free( *tmodule );
		}
		if( tbase ) {
			g_free( *tbase );
		}
	} 

	if( temp ) {
		g_free( temp );
	}

	g_free( name );
}

static void cvs_menu_login_callback( GtkAction *action, 
				     gpointer user_data )
{
	ScreemWindow *window;
	ScreemCVS *cvs;

	gchar *root;
	gchar *flags;

	window = SCREEM_WINDOW( user_data );
	cvs = window->details->cvs;

	root = (gchar*)g_getenv( "CVSROOT" );
	if( ! root ) {
		root = "";
	}
	flags = "";

	if( screem_cvs_dialog( window, NULL, 
			       &root, NULL, NULL, &flags ) ) {

		g_object_set( G_OBJECT( cvs ),
			      "root", root, NULL );
	
		/*screem_cvs_import( cvs, flags, log, module );*/

		g_free( root );
		g_free( flags );
	}
}

static void tool_menu_fix_links_callback( GtkAction *action, 
					  gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *application;
	gchar *glade_path;
	GladeXML *xml;
	GtkWidget *widget;
	
	window = SCREEM_WINDOW( user_data );
	application = SCREEM_APPLICATION( window->application );

	glade_path = screem_get_glade_path();
	xml = glade_xml_new( glade_path, "fixlinks", NULL );
	g_free( glade_path );
	glade_xml_signal_autoconnect( xml );

	widget = glade_xml_get_widget( xml, "fixlinks" ); 
	if( gtk_dialog_run( GTK_DIALOG( widget ) ) == GTK_RESPONSE_OK ) {
		ScreemSite *site;
		GtkWidget *entry;
		const gchar *sitepath;
		gchar *src;
		gchar *dest;
		
		site = screem_window_get_current( window );
		g_assert( ! screem_site_get_fake_flag( site ) );
		sitepath = screem_site_get_pathname( site );
		
		entry = glade_xml_get_widget( xml, "original" );
		src = (gchar*)gtk_entry_get_text( GTK_ENTRY( entry ) );
		if( strncmp( sitepath, src, strlen( sitepath ) ) ) {
			src = g_strconcat( sitepath, src + 1, NULL );
		} else {
			src = g_strdup( src );
		}
		entry = glade_xml_get_widget( xml, "new" );
		dest = (gchar*)gtk_entry_get_text( GTK_ENTRY( entry ) );
		if( strncmp( sitepath, dest, strlen( sitepath ) ) ) {
			dest = g_strconcat( sitepath, dest + 1 , NULL );
		} else {
			dest = g_strdup( dest );
		}		
		screem_site_file_change( site, src, dest );
		g_free( src );
		g_free( dest );
	}
	gtk_widget_destroy( widget );
	g_object_unref( G_OBJECT( xml ) );
}

static void tool_menu_template_update_callback( GtkAction *action, 
					  gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *app;
	ScreemPage *page;
	gchar *tag;
	gchar *path;
	const gchar *pathname;
	ScreemPage *template;
	GSList *blocks;
	ScreemSite *site;
	
	window = SCREEM_WINDOW( user_data );
	g_object_get( G_OBJECT( window ), "app", &app, NULL );
	site = screem_window_get_current( window );
	page = screem_window_get_document( window );
	if( page && ( path = screem_page_template_path( page ) ) ) { 

		/* the page appears to use/be a template, check which */
			
		pathname = screem_page_get_pathname( page );
			
		if( pathname && strcmp( pathname, path ) ) {
				
			/* it is using a template */
			
			template = screem_page_new( G_OBJECT( app ) );
			screem_page_set_pathname( template, path );
			/* we need the blocks from template, which
			 * may be different to those in the current page,
			 * which is why we didn't get them before */
		
			g_free( path );
			blocks = NULL;
			if( screem_page_is_template( template, 
							&tag, &path,
							&blocks ) ) {
				screem_markup_update_from_template( site, template, page, tag, blocks );
				g_slist_foreach( blocks, 
						(GFunc)g_free, NULL );
				g_slist_free( blocks );
				g_free( tag );
			}
			g_object_unref( template );
		}
		g_free( path );
	}
	g_object_unref( app );
}

static void edit_macros_callback( GtkAction *action,
				gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *app;

	window = SCREEM_WINDOW( user_data );
	g_object_get( G_OBJECT( window ), "app", &app, NULL );
	screem_preferences_edit_macros( app );
	g_object_unref( app );
}

static void edit_helpers_callback( GtkAction *action,
				gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *app;

	window = SCREEM_WINDOW( user_data );
	g_object_get( G_OBJECT( window ), "app", &app, NULL );
	screem_preferences_edit_helpers( app );
	g_object_unref( app );
}

static void doc_menu_previous_callback( GtkAction *action, 
				     gpointer user_data )
{
	ScreemWindow *window;
	ScreemWindowDetails *details;
	
	window = SCREEM_WINDOW( user_data );
	details = window->details;
	
	gtk_notebook_prev_page( GTK_NOTEBOOK( details->notebook ) );
}

static void doc_menu_next_callback( GtkAction *action, 
				     gpointer user_data )
{
	ScreemWindow *window;
	ScreemWindowDetails *details;
	
	window = SCREEM_WINDOW( user_data );
	details = window->details;
		
	gtk_notebook_next_page( GTK_NOTEBOOK( details->notebook ) );
}


static void help_menu_help_callback( GtkAction *action, 
				     gpointer user_data )
{
	GError *error;

	error = NULL;
	gnome_help_display( "screem", NULL, &error );

	if( error ) {
		screem_hig_alert( GTK_STOCK_DIALOG_ERROR,
				_( "Could not display help for Screem." ),
				error->message,
				GTK_WIDGET( user_data ) );
		g_error_free (error);
	}
}

static void help_menu_about_screem_callback( GtkAction *action, 
					     gpointer user_data )
{
	static GtkWidget *about = NULL;

	const gchar *COPYRIGHT = "1999-2003 David A Knight";
	const gchar *AUTHORS[] = {
		"David A Knight - Main Author",
		"Lee Mallabone - Sitecopy Integration / General ",
		"Joe Orton - Sitecopy",
		"Sven Liessem - General",
		"Matt Colyer - General",
		"Sven Salzwedel - General",
		NULL
	};
	const gchar *DESCRIPTION = _( "\
SCREEM (Site CReation and Editing EnvironMent)\n\
Screem provides an integrated development environment for creating and maintaining websites\n\
SCREEM is licensed under the GNU General Public License version 2 or above" );
        GdkPixbuf *logo = NULL;
	const gchar *DOCUMENTERS[] = {
		"Matt Colyer",
		"David A Knight",
		NULL
	};
	const gchar *TRANSLATERS = _( "Translater Credits" );

	if( ! about ) {
		logo = gdk_pixbuf_new_from_file( SPLASH_DIR"/splash.png", NULL );
	
		about = gnome_about_new( "Screem", VERSION,
					 COPYRIGHT,
					 DESCRIPTION,
					 AUTHORS,
					 DOCUMENTERS,
					 strcmp( TRANSLATERS,
						 "Translater Credits" ) != 0 ? TRANSLATERS : NULL,
				 	logo );
		g_signal_connect_swapped( G_OBJECT( about ), "delete_event",
					G_CALLBACK( g_nullify_pointer ),
					&about );
		g_signal_connect_swapped( G_OBJECT( about ), "destroy",
					G_CALLBACK( g_nullify_pointer ),
					&about );
	}
	gtk_window_set_transient_for( GTK_WINDOW( about ),
				      	GTK_WINDOW( user_data ) );
	gtk_widget_show( about );
	gdk_window_raise( GDK_WINDOW( about->window ) );
}

static void help_menu_tip_callback( GtkAction *action, 
				    gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *application;

	window = SCREEM_WINDOW( user_data );
	application = SCREEM_APPLICATION( window->application );

	gtk_window_set_transient_for(GTK_WINDOW(application->hint),
				     GTK_WINDOW(window));
	
	gtk_widget_show( application->hint );
}

static void help_menu_debug_callback( GtkAction *action, 
				    gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *application;

	window = SCREEM_WINDOW( user_data );
	application = SCREEM_APPLICATION( window->application );

	screem_debug_console( application );
}


/* Site view menu: FIXME doesn't belong here */

static void create_dir_callback( const gchar *string, gpointer data )
{
	ScreemWindow *window;
	ScreemSite *site;
	GtkWidget *widget;

	window = SCREEM_WINDOW( data );
	site = screem_window_get_current( window );

	widget = GTK_WIDGET( g_object_get_data( G_OBJECT( window ),
						"siteview" ) );

	if( ! string ) {
                screem_window_show_message( window,
					    _( "No directory name specified." ), FALSE );
	} else if( ! screem_site_view_create_dir( site, widget, string ) ) {
                screem_window_show_message( window,
					    _( "Directory could not be created." ), FALSE );
        }
}

static void site_view_menu_new_dir_callback( GtkAction *action, 
					     gpointer user_data )
{
	GtkWidget *widget;

	widget = gnome_request_dialog( FALSE, 
				       _("Enter the directory name to create:"),
				       "", 255, 
				       (GnomeStringCallback)create_dir_callback,
				       user_data, NULL );
	gtk_window_set_title( GTK_WINDOW( widget ), 
				"Create Directory: - Screem" );
}

static void site_view_menu_delete_file_callback( GtkAction *action, 
						 gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *app;
	ScreemSite *site;
	GtkWidget *view;
	GtkTreeModel *model;
	GtkTreeSelection *selection;
	GtkTreeIter it;
	gchar *uri;
	GtkWidget *confirm;
	GtkWidget *label;
	gchar *glade_path;
	gchar *txt;
	gchar *base;
	gchar *temp;
	GladeXML *xml;
		
	window = SCREEM_WINDOW( user_data );
	app = window->application;
	
	site = screem_window_get_current( window );
	view = GTK_WIDGET( g_object_get_data( G_OBJECT( window ), 
							"siteview" ) );
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( view ) );
	
	gtk_tree_selection_get_selected( selection, &model, &it );
	
	gtk_tree_model_get( model, &it, 
			FILE_BROWSER_URI_COL, &uri, -1 );
	if( uri ) {		
		glade_path = screem_get_glade_path();
		xml = glade_xml_new( glade_path, "delete_file", NULL );
		g_free( glade_path );
	
		confirm = glade_xml_get_widget( xml, "delete_file" );
		label = glade_xml_get_widget( xml, "delete_label" );
		base = g_path_get_basename( uri );
		temp = gnome_vfs_unescape_string_for_display( base );
		txt = g_strdup_printf( _( "Are you sure you want to permanently delete \"%s\"?" ),
					temp );
		g_free( temp );
		g_free( base );
		gtk_label_set( GTK_LABEL( label ), txt );
		g_free( txt );
		
		gtk_window_set_wmclass( GTK_WINDOW( confirm ),
					"Screem",
					"delete_confirm_prompt" );
		gtk_window_set_transient_for( GTK_WINDOW( confirm ),
					      GTK_WINDOW( window ) );
		gtk_window_set_type_hint( GTK_WINDOW( confirm ), 
					  GDK_WINDOW_TYPE_HINT_DIALOG );
		if(gtk_dialog_run( GTK_DIALOG( confirm ) ) == GTK_RESPONSE_OK){
			delete_dir( uri, screem_application_file_op, app );
		}
		gtk_widget_destroy( confirm );
		g_object_unref( G_OBJECT( xml ) );
		g_free( uri );
	}
}

static void site_view_menu_file_props_callback( GtkAction *action, 
						gpointer user_data )
{
	ScreemWindow *window;
	ScreemWindowDetails *details;
	ScreemSite *site;
	GtkTreeModel *model;
	GtkWidget *view;
	GtkTreeSelection *selection;
	GtkTreeIter it;
	gchar *uri;

	window = SCREEM_WINDOW( user_data );
	details = window->details;

	site = screem_window_get_current( window );
	view = GTK_WIDGET( g_object_get_data( G_OBJECT( window ), 
						"siteview" ) );
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( view ) );

	gtk_tree_selection_get_selected( selection, &model, &it );
	
	gtk_tree_model_get( model, &it, 
			FILE_BROWSER_URI_COL, &uri, -1 );
	
	if( uri ) {
		/* display info on info fullname */
		screem_site_view_show_properties( site, uri,
						  GTK_WINDOW( window ) );
		g_free( uri );
	}
}

void screem_window_recent_page_cb( GtkAction *action, gpointer data )
{
	EggRecentItem *item;
	ScreemWindow *window;
	ScreemApplication *application;
	ScreemSite *site;
	gchar *uri;
	gchar *tmp;
	gchar *base;
	gchar *primary;
	gchar *secondary;

	item = g_object_get_data( G_OBJECT( action ), "egg_recent_uri" );
	window = SCREEM_WINDOW( data );
	application = SCREEM_APPLICATION( window->application );
	site = screem_application_get_default_site( application );

	uri = egg_recent_item_get_uri( item );
	if( uri_exists( uri, NULL ) ) {
		screem_page_open_with_filename( site, window, uri );
	} else {
		tmp = gnome_vfs_unescape_string_for_display( uri );
		base = g_path_get_basename( tmp );
		primary = g_strdup_printf( _( "Could not open recent page: %s" ), base );
		secondary = g_strdup_printf( _( "%s no longer exists" ),
				tmp );
		screem_hig_alert( GTK_STOCK_DIALOG_ERROR,
				primary, secondary,
				GTK_WIDGET( data ) );
		g_free( base );
		g_free( primary );
		g_free( secondary );
		g_free( tmp );
		egg_recent_model_delete( window->details->recent_page_model, uri );
	}
	g_free( uri );
}

void screem_window_recent_site_cb( GtkAction *action, gpointer data )
{
	EggRecentItem *item;
	ScreemWindow *window;
	ScreemApplication *app;
	gchar *uri;
	gchar *tmp;
	gchar *base;
	gchar *primary;
	gchar *secondary;
	
	item = g_object_get_data( G_OBJECT( action ), "egg_recent_uri" );
	window = SCREEM_WINDOW( data );
	g_object_get( G_OBJECT( window ), "app", &app, NULL );
	
	uri = egg_recent_item_get_uri( item );

	if( uri_exists( uri, NULL ) ) {
		screem_site_open_with_filename( window, app, uri );
	} else {
		tmp = gnome_vfs_unescape_string_for_display( uri );
		base = g_path_get_basename( tmp );
		primary = g_strdup_printf( _( "Could not open recent site: %s" ), base );
		secondary = g_strdup_printf( _( "%s no longer exists" ),
				tmp );
	
		screem_hig_alert( GTK_STOCK_DIALOG_ERROR,
				primary, secondary,
				GTK_WIDGET( data ) );
		g_free( base );
		g_free( primary );
		g_free( secondary );
		g_free( tmp );
		egg_recent_model_delete( window->details->recent_site_model, uri );
	}
	g_free( uri );
	g_object_unref( app );
}

/* toolbar callback stuff */
typedef enum {
	LEFT, CENTRE, JUSTIFY, RIGHT
} Align;

static void
toolbar_task_list_callback( GtkAction *action, gpointer user_data )
{
	ScreemWindow *window;
	ScreemSite *site;
	ScreemTodo *todo;

	window = SCREEM_WINDOW( user_data );
	site = screem_window_get_current( window );
	todo = SCREEM_TODO( screem_site_get_todo( site ) );

	screem_todo_view( todo );
}

static void screem_toolbar_spell_check( GtkAction *action, gpointer user_data )
{
	ScreemWindow *window;
	ScreemPage *page;

	window = SCREEM_WINDOW( user_data );
	page = screem_window_get_document( window );

	g_object_set_data( G_OBJECT( window->details->spell ), "page", page );
	screem_spell_check_interactive( window->details->spell );
}

static void screem_toolbar_external_browser_notify( GConfClient *client,
						    guint cnxn_id,
						    GConfEntry *entry,
						    gpointer data )
{
	ScreemApplication *application;
	ScreemWindow *window;
	GSList *list;

	gchar *browser;
	gchar *label;
	gchar *conf;
	gchar *tip;
	gchar *stock_id;
	gint i;
	gboolean set;
	
	GtkAction *action;
	
	window = SCREEM_WINDOW( data );
	application = SCREEM_APPLICATION( window->application );

	list = window->details->browser_notify_handles;

	i = g_slist_index( list, GINT_TO_POINTER( cnxn_id ) );
	i /= 2;
	i ++;

	browser = g_strdup_printf( "Browser%i", i );

	conf = g_strdup_printf( "/apps/screem/Browser%i_path", i );
	tip = gconf_client_get_string( application->client, conf,
				       NULL );
	g_free( conf );
	conf = g_strdup_printf( "/apps/screem/Browser%i_name", i );
	label = gconf_client_get_string( application->client, conf,
					 NULL );
	if( ( ! label || *label == '\0' ) && ( tip && *tip != '\0' ) ) {
		gchar *tmp;
		gunichar c;
			
		label = g_path_get_basename( tip );
		tmp = label;
		for( c = g_utf8_get_char( tmp );
			! g_unichar_isspace( c ) && c != '\0';
			tmp = g_utf8_next_char( tmp ),
			c = g_utf8_get_char( tmp ) ) { }
		if( c != '\0' ) {
			*tmp = '\0';
		}
	}
	g_free( conf );

	set = TRUE;
	if( ! tip || *tip == '\0' ) {
		set = FALSE;
		tip = g_strdup_printf( _( "External Browser %i Not Set" ),
				       i );
		stock_id = g_strdup( GTK_STOCK_EXECUTE );
	} else {
		stock_id = g_strconcat( "Screem_", browser, NULL );
	}
	if( ! label || *label == '\0' ) {
		label = g_strdup_printf( _( "External Browser %i" ), i );
	}
		
	action = gtk_action_group_get_action( GTK_ACTION_GROUP( window->action_group ),
						browser );
	g_object_set( G_OBJECT( action ), 
			"label", label,
			"tooltip", tip,
			"sensitive", set, 
			"stock_id", stock_id, NULL );
	g_object_set_data( G_OBJECT( action ),
			"active_browser",
			GINT_TO_POINTER( set ) );
	
	g_free( label );
	g_free( tip );
	g_free( stock_id );
	g_free( browser );
}

void screem_toolbar_external_browser( GtkAction *action, gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *application;
	ScreemPage *page;
	const gchar *pagepath;

	gchar *conf;
	gchar *browserpath;
	gboolean use_url;
	gboolean dynamic;
	gboolean needs_terminal;
	gchar *uri;

	gchar *name;
	ScreemSite *site;
	
	window = SCREEM_WINDOW( user_data );
	g_object_get( G_OBJECT( action ), "name", &name, NULL );

	/* name holds the browser we want, eg Browser1, Browser2 etc */

	application = SCREEM_APPLICATION( window->application );

	/* get browser settings */
	conf = g_strconcat( "/apps/screem/", name, "_path", NULL );
	browserpath = gconf_client_get_string( application->client,
					       conf, NULL );
	g_free( conf );

	needs_terminal = FALSE;
	if( ! browserpath && ! strcmp( "Browser0", name ) ) {
		/* default browser, hasn't been overridden for
		   SCREEM, so use gnome default,
		   /desktop/gnome/url-handlers/http/command */
		   
		browserpath = gconf_client_get_string( application->client,
				"/desktop/gnome/url-handlers/http/command", NULL );
		needs_terminal = gconf_client_get_bool( application->client,
				"/desktop/gnome/url-handlers/http/needs_terminal", NULL );
	}

	page = screem_window_get_document( window );
	dynamic = screem_page_is_dynamic( page );

	if( ! dynamic ) {
		conf = g_strconcat( "/apps/screem/", name, "_static", NULL );
	} else {
		conf = g_strconcat( "/apps/screem/", name, "_dynamic", NULL );
	}
	use_url = gconf_client_get_bool( application->client, conf, NULL );
	g_free( conf );

	/* can only use url if we are working on a site, otherwise
	 * we don't have a base for the url */
	site = screem_window_get_current( window );
	use_url &= ( ! screem_site_get_fake_flag( site ) );
	
	if( ! browserpath ) {
		/* FIXME: hmm, no browserpath available, alert user */
	} else if( (screem_page_save_confirmation( page, FALSE ) != 
		    GTK_RESPONSE_CANCEL) ) {
		pagepath = screem_page_get_pathname( page );

		if( use_url ) {
			const gchar *sitepath;
				
			sitepath = screem_site_get_pathname( site );
			
			uri = g_strconcat( browserpath, " http://",
					   screem_site_get_http_url( site ),
					   pagepath + strlen( sitepath ), 
					   NULL );
		} else {
			gchar *file;
			gchar *rest;
			gint pos;

			file = strstr( browserpath, "%s" );
			if( file ) {
				pos = file - browserpath;
				rest = file + 2;
				if( rest > ( browserpath + strlen( browserpath ) ) )
					rest = "";
				file = g_strndup( browserpath, pos );
			} else {
				/* no %s, assume path just goes on the end */
				file = g_strconcat( browserpath, " ", NULL );
				rest = "";
			}
			uri = g_strconcat( file, pagepath, rest,
					   NULL );
			g_free( file );
		}
		if( ! needs_terminal ) {
			gnome_execute_shell( NULL, uri );
		} else {
			gnome_execute_terminal_shell( NULL, uri );
		}
		g_free( uri );
	}
	if( browserpath ) {
		g_free( browserpath );
	}
	g_free( name );
}

static void screem_toolbar_create_browsers( ScreemWindow *window )
{
	gint i;

	ScreemApplication *application;
	gchar *conf;
	const int NO_OF_CUSTOM_BROWSERS = 3;

	GtkActionEntry *entry;
	GError *error;
	
	static GtkActionEntry browser_action_entries[] = {
		{ "Action Name", NULL, "Action Label",
	  	   NULL, "tip",
	  	   G_CALLBACK( NULL ) }
	};
	
	application = SCREEM_APPLICATION( window->application );

	for( i = 1; i <= NO_OF_CUSTOM_BROWSERS; ++ i ) {
		gchar *data;
		gchar *label;
		gchar *tip;
		gchar *browser;
		gchar *menu;
		gint handle;
		gboolean set;
		GtkAction *action;
		
		browser = g_strdup_printf( "Browser%i", i );
		menu = g_strdup_printf( "<ui><menubar><menu action=\"View\"><menuitem action=\"%s\" /></menu></menubar><!-- toolbar name=\"Main Toolbar\"><toolitem action=\"%s\" /></toolbar --></ui>",
					browser, browser );
		
		conf = g_strdup_printf( "/apps/screem/Browser%i_icon", i );
		data = gconf_client_get_string( application->client, conf,
						NULL );
		g_free( conf );
		if( data && *data != '\0' ) {
			/* change default stock icon */
			GdkPixbuf *pixbuf;
			gchar *id = g_strconcat( "Screem_", browser, NULL );
			GtkIconSet *set = gtk_icon_factory_lookup_default( id );
			GtkIconSource *src = gtk_icon_source_new();
				
			pixbuf = gdk_pixbuf_new_from_file( data, NULL );
			g_free( data );
			if( pixbuf ) {
				gtk_icon_source_set_pixbuf( src, pixbuf );
			
				gtk_icon_source_set_direction_wildcarded( src, TRUE );
				gtk_icon_source_set_state_wildcarded( src, TRUE );
				gtk_icon_source_set_size_wildcarded( src, TRUE );
				gtk_icon_source_set_size( src, GTK_ICON_SIZE_BUTTON );
				gtk_icon_set_add_source( set, src );
				
				g_object_unref( G_OBJECT( pixbuf ) );
				gtk_icon_source_free( src );
				data = id;
			} else {
				data = g_strdup( GTK_STOCK_EXECUTE );
				g_free( id );
			}
		} else {
			data = g_strdup( GTK_STOCK_EXECUTE );
		}

		conf = g_strdup_printf( "/apps/screem/Browser%i_path", i );
		tip = gconf_client_get_string( application->client, conf,
					       NULL );
		g_free( conf );
		set = TRUE;

		conf = g_strdup_printf( "/apps/screem/Browser%i_name", i );
		label = gconf_client_get_string( application->client, conf,
						 NULL );
		if( ( ! label || *label == '\0' ) && ( tip && *tip != '\0' ) ) {
			gchar *tmp;
			gunichar c;
			
			label = g_path_get_basename( tip );
			tmp = label;
			for( c = g_utf8_get_char( tmp );
				! g_unichar_isspace( c ) && c != '\0';
				tmp = g_utf8_next_char( tmp ),
				c = g_utf8_get_char( tmp ) ) { }
			if( c != '\0' ) {
				*tmp = '\0';
			}
		}
		g_free( conf );
			
		if( ! tip || *tip == '\0' ) {
			set = FALSE;
			g_free( tip );
			tip = g_strdup_printf( _( "External Browser %i Not Set" ),
					       i );
		}
		if( ! label  || *label == '\0' ) {
			label = g_strdup_printf( _( "External Browser %i" ), i );
		}
				
		entry = browser_action_entries;
		entry->name = browser;
		entry->label = label;
		entry->tooltip = tip;
		entry->stock_id = data;
		entry->accelerator = "";
		entry->callback = G_CALLBACK( screem_toolbar_external_browser );
		gtk_action_group_add_actions( GTK_ACTION_GROUP( window->action_group ),
						entry, 1, window );
		g_free( label );
		g_free( tip );
		g_free( data );
		
		if( ! gtk_ui_manager_add_ui_from_string( GTK_UI_MANAGER( window->merge ),
							menu, strlen( menu ), &error ) ) {
			g_message( "External browser ui string error = %s", error->message );
			g_error_free( error );
		}
		
		g_free( menu );
		
		action = gtk_action_group_get_action( GTK_ACTION_GROUP( window->action_group ),
							browser );
		g_object_set( G_OBJECT( action ), "sensitive", FALSE, NULL );
		g_object_set_data( G_OBJECT( action ),
				"active_browser",
				GINT_TO_POINTER( set ) );
		
		g_free( browser );

		browser = g_strdup_printf( "/apps/screem/Browser%i_icon",
					   i );
		handle = gconf_client_notify_add( application->client, browser,
						  
						  screem_toolbar_external_browser_notify, 
						  window, NULL, NULL );
		window->details->browser_notify_handles =
			g_slist_append(window->details->browser_notify_handles,
				       GINT_TO_POINTER( handle ) );
		g_free( browser );

		browser = g_strdup_printf( "/apps/screem/Browser%i_path",
					   i );
		handle = gconf_client_notify_add( application->client, browser,
						  
						  screem_toolbar_external_browser_notify, 
						  window, NULL, NULL );
		window->details->browser_notify_handles =
			g_slist_append(window->details->browser_notify_handles,
				       GINT_TO_POINTER( handle ) );
		g_free( browser );

	}
}

void screem_toolbar_markup( ScreemEditor *editor, const gchar *tag )
{
	gchar *open;
	gchar *close;

	open = g_strconcat( "<", tag, ">", NULL );
	close = g_strconcat( "</", tag, ">", NULL );

	screem_editor_insert_markup( editor, open, close );

	g_free( close );
	g_free( open );
}

void screem_toolbar_align( ScreemEditor *editor, Align a )
{
	const gchar *open = NULL;
	const gchar *close = NULL;

	switch( a ) {
	case CENTRE:
		open = "<center>";
		close = "</center>";
		break;
	case RIGHT:
		open = "<div align=\"right\">";
		close = "</div>";
		break;
	case JUSTIFY:
		open = "<div align=\"justify\">";
		close = "</div>";
		break;
	case LEFT:
		open = "<div align=\"left\">";
		close = "</div>";
		break;
	}
	if( open && close )
		screem_editor_insert_markup( editor, open, close );
}


/* normal tag insertion handling */
static void quickbar_bold_callback( GtkAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "b" );
}
static void quickbar_italic_callback( GtkAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "i" );
}
static void quickbar_underline_callback( GtkAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "u" );
}
static void quickbar_para_callback( GtkAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "p" );
}
static void quickbar_bullet_callback( GtkAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "ul" );
}
static void quickbar_numbered_callback( GtkAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "ol" );
}

/* align handling */

static void quickbar_alignleft_callback( GtkAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_align( editor, LEFT );
}
static void quickbar_aligncentre_callback( GtkAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_align( editor, CENTRE );
}
static void quickbar_alignright_callback( GtkAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_align( editor, RIGHT );
}
static void quickbar_justify_callback( GtkAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_align( editor, JUSTIFY );
}


static void fontbar_font_callback( GtkAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "font" );
}
static void fontbar_pre_callback( GtkAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "pre" );
}
static void fontbar_subscript_callback( GtkAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "sub" );
}

static void fontbar_superscript_callback( GtkAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "sup" );
}



static void tablebar_table_callback( GtkAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "table" );
}

static void tablebar_tablebody_callback( GtkAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "tbody" );
}

static void tablebar_tr_callback( GtkAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "tr" );
}

static void tablebar_th_callback( GtkAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "th" );
}

static void tablebar_td_callback( GtkAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "td" );
}

static void tablebar_caption_callback( GtkAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "caption" );
}

static void tablebar_summary_callback( GtkAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "summary" );
}


static void formbar_textarea_callback( GtkAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "textarea" );	
}

static void formbar_input_callback( GtkAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_editor_insert_markup( editor, "<input name=\"\">", NULL );
}

static void formbar_fileinput_callback( GtkAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_editor_insert_markup( editor, "<input type=\"file\" name=\"\">",
					NULL );
}

static void formbar_button_callback( GtkAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_toolbar_markup( editor, "button" );
}

static void formbar_checkbutton_callback( GtkAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_editor_insert_markup( editor, 
					"<input type=\"checkbox\" name=\"\">",
					NULL );
}

static void formbar_radiobutton_callback( GtkAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_editor_insert_markup( editor, 
					"<input type=\"radio\" name=\"\">", NULL );
}

static void formbar_optionmenu_callback( GtkAction *action, gpointer user_data )
{
	ScreemEditor *editor;
	editor = SCREEM_EDITOR( SCREEM_WINDOW( user_data )->details->editor );
	screem_editor_insert_markup( editor, 
					"<select><option>",
					"</option></select>" );
}

static void insert_object_flash_callback( GtkAction *action,
				gpointer user_data )
{
	ScreemWindow *window;
	ScreemPage *page;
	gboolean xhtml;
	ScreemEditor *editor;
	
	const gchar *obj = "\
<object type=\"application/x-shockwave-flash\" data=\"\" \
width=\"\" height=\"\">\n\t\
<param name=\"movie\" value=\"\">\n\t\
<img src=\"\" alt=\"no flash support, image alt text\">\n";
	const gchar *xobj = "\
<object type=\"application/x-shockwave-flash\" data=\"\" \
width=\"\" height=\"\">\n\t\
<param name=\"movie\" value=\"\" />\n\t\
<img src=\"\" alt=\"no flash support, image alt text\" />\n";
	const gchar *txt;
	GConfClient *client;
	gfloat tabwidth;
	gchar *spaces;
	gchar *tmp;
	
	window = SCREEM_WINDOW( user_data );
	page = screem_window_get_document( window );

	if( ! page ) {
		return;
	}
	editor = SCREEM_EDITOR( window->details->editor );

	xhtml = screem_page_is_xml( page );
	if( ! xhtml ) {
		txt = obj;
	} else {
		txt = xobj;
	}

	client = gconf_client_get_default();
	tabwidth = gconf_client_get_float( client,
			"/apps/screem/editor/tabwidth",
			NULL );
	if( tabwidth == 0.0 ) {
		tabwidth = 8.0;
	}
	if( gconf_client_get_bool( client,
				"/apps/screem/editor/spaces_instead_of_tabs",
				NULL ) ) {
		spaces = g_new0( gchar, (gint)(tabwidth + 1) );
		memset( spaces, ' ', tabwidth );
		tmp = find_text( txt, "\t", spaces, NULL );
 		g_free( spaces );
	} else {
		tmp = g_strdup( txt );
	}
	g_object_unref( client );
	
	screem_editor_insert_markup( editor, tmp, "</object>" );
	g_free( tmp );
}

static void insert_object_applet_callback( GtkAction *action,
				gpointer user_data )
{
	ScreemWindow *window;
	ScreemPage *page;
	gboolean xhtml;
	ScreemEditor *editor;
	
	const gchar *obj = "\
<object codetype=\"application/java\" classid=\"\" \
width=\"\" height=\"\">\n\t\
<img src=\"\" alt=\"no java support, image alt text\">\n";
	const gchar *xobj = "\
<object codetype=\"application/java\" classid=\"\" \
width=\"\" height=\"\">\n\t\
<img src=\"\" alt=\"no java support, image alt text\" />\n";
	const gchar *txt;
	GConfClient *client;
	gfloat tabwidth;
	gchar *spaces;
	gchar *tmp;
	
	window = SCREEM_WINDOW( user_data );
	page = screem_window_get_document( window );

	if( ! page ) {
		return;
	}
	editor = SCREEM_EDITOR( window->details->editor );

	xhtml = screem_page_is_xml( page );
	if( ! xhtml ) {
		txt = obj;
	} else {
		txt = xobj;
	}

	client = gconf_client_get_default();
	tabwidth = gconf_client_get_float( client,
			"/apps/screem/editor/tabwidth",
			NULL );
	if( tabwidth == 0.0 ) {
		tabwidth = 8.0;
	}
	if( gconf_client_get_bool( client,
				"/apps/screem/editor/spaces_instead_of_tabs",
				NULL ) ) {
		spaces = g_new0( gchar, (gint)(tabwidth + 1) );
		memset( spaces, ' ', tabwidth );
		tmp = find_text( txt, "\t", spaces, NULL );
 		g_free( spaces );
	} else {
		tmp = g_strdup( txt );
	}
	g_object_unref( client );
	
	screem_editor_insert_markup( editor, tmp, "</object>" );
	g_free( tmp );

}



static void insert_basic_html_callback( GtkAction *action, 
				gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *application;
	ScreemDTDDB *db;
	ScreemEditor *editor;
	ScreemPage *page;
	const gchar *pathname;
	gchar *doctype;
	gchar *text;
	gboolean xhtml;
	const gchar *head;
	gint pos;
	const gchar *metas[] = {
		"Generator",
		"Author",
		"Content-Type",
		"Content-Script-Type",
		"Content-Style-Type",
		NULL
	};
	
	window = SCREEM_WINDOW( user_data );

	application = SCREEM_APPLICATION( window->application );
	db = screem_application_get_dtd_db( application );

	editor = SCREEM_EDITOR( window->details->editor );

	page = screem_window_get_document( window );
	pathname = screem_page_get_pathname( page );
	
	xhtml = screem_page_is_xml( page );

	doctype = gconf_client_get_string( application->client,
				     "/apps/screem/editor/default_dtd",
					     NULL );
	text = screem_markup_basic_html( db, doctype, FALSE, xhtml );
	g_free( doctype );

	screem_editor_insert( editor, -1, text );

	head = strstr( text, "</head>" );
	if( head ) {
		pos = ( head - text );
		screem_editor_set_pos( editor, pos );
		for( pos = 0; metas[ pos ]; ++ pos ) {
			action = gtk_action_group_get_action( GTK_ACTION_GROUP( window->action_group ), metas[ pos ] );
			gtk_action_activate( action );
			screem_editor_insert( editor, -1, "\n" );
		}
	}

	/* auto indent the inserted text */
	screem_editor_select_region( editor, 0, 0 );
	screem_editor_auto_indent( editor, 0 );
	
	g_free( text );
}

static void insert_closing_tag_callback( GtkAction *action, 
				gpointer user_data )
{
	ScreemWindow *window;
	ScreemEditor *editor;
	ScreemPage *page;
	gint pos;
	gchar *tag;
	gchar *tmp;
	guint start;
	guint end;
	gchar *data;
	
	window = SCREEM_WINDOW( user_data );
	editor = SCREEM_EDITOR( window->details->editor );
	
	page = screem_window_get_document( window );
	if( page ) {
		pos = screem_editor_get_pos( editor );
		
		tag = screem_page_query_context( page, pos, FALSE,
				TRUE, NULL, &start, &end, NULL );
		if( tag ) {
			data = screem_page_get_data( page );
			
			if( ! screem_markup_next_tag_close( data, tag,
						start + 1 ) ) {
				tmp = g_strconcat( "</", tag, ">", NULL );
				screem_editor_insert( editor, pos, tmp );
				g_free( tmp );
			}
			g_free( tag );
			g_free( data );
		}
	}
}
	
static void meta_insert_callback( GtkAction *action, gpointer user_data )
{
	ScreemWindow *window;
	ScreemEditor *editor;
	ScreemPage *page;
	gchar *name;
	GString *markup;
	const gchar *pattern1 = "<meta name=\"%s\" content=\"%s\" ";
	const gchar *pattern2 = "<meta http-equiv=\"%s\" content=\"%s\" ";
	const gchar *pattern;
	const gchar *type;
	gchar *content;
	gboolean xhtml;
	gchar *charset;
	
	window = SCREEM_WINDOW( user_data );
	editor = SCREEM_EDITOR( window->details->editor );
	
	page = screem_window_get_document( window );

	if( ! page ) {
		return;
	}
	
	xhtml = screem_page_is_xml( page );

	g_object_get( G_OBJECT( action ), "name", &name, NULL );

	markup = g_string_new( NULL );

	pattern = pattern1;
	type = "";
	content = NULL;
	if( ! strcmp( "Generator", name ) ) {
		pattern = pattern1;
		type = "generator";
		content = g_strdup( PACKAGE_STRING );
	} else if( ! strcmp( "Author", name ) ) {
		pattern = pattern1;
		type = "author";
		content = g_strdup( g_get_real_name() );
	} else if( ! strcmp( "Content-Type", name ) ) {
		pattern = pattern2;
		type = "Content-Type";
		charset = screem_page_get_charset( page );
		content = g_strdup_printf( "%s;charset=%s",
				screem_page_get_mime_type( page ),
				charset );
		g_free( charset );
	} else if( ! strcmp( "Content-Script-Type", name ) ) {
		pattern = pattern2;
		type = "Content-Script-Type";
		content = g_strdup( "text/javascript" );
	} else if( ! strcmp( "Content-Style-Type", name ) ) {
		pattern = pattern2;
		type = "Content-Style-Type";
		content = g_strdup( "text/css" );
	} else if( ! strcmp( "MetaRefresh", name ) ) {
		pattern = pattern2;
		type = "Refresh";
		content = g_strdup( "10" );
	}

	if( ! content ) {
		content = g_strdup( "" );
	}
	g_string_append_printf( markup, pattern,
				type, content );

	if( xhtml ) {
		g_string_append( markup, "/>" );
	} else {
		g_string_append( markup, ">" );
	}

	screem_editor_insert_markup( editor, markup->str, NULL );

	g_string_free( markup, TRUE );

	g_free( content );
	g_free( name );
}

/* initialise stuff */
static GtkToggleActionEntry screem_window_toggle_action_entries[] = {
	{ "ViewMessages", NULL, N_("Messages"),
	  "<Control>F1", N_("Show/Hide the Messages dockitem"),
	  G_CALLBACK( view_menu_dock_show_callback ) },
	  
	{ "ViewErrors", NULL, N_("Errors"),
	  "<Control>F2", N_("Show/Hide the Messages dockitem"),
	  G_CALLBACK( view_menu_dock_show_callback ) },
	  
	{ "ViewFiles", NULL, N_("Files"),
	  "<Control>F4", N_("Show/Hide the file browser"),
	  G_CALLBACK( view_menu_dock_show_callback ) },
	  
	{ "ViewResources", NULL, N_("Resources"),
	  "<Control>F5", N_("Show/Hide the resources browser"),
	  G_CALLBACK( view_menu_dock_show_callback ) },
	  
	{ "ViewTree", NULL, N_("Structure"),
	  "<Control>F6", N_("Show/Hide the document structure browser"),
	  G_CALLBACK( view_menu_dock_show_callback ) },
	  
	{ "ViewAttributes", NULL, N_("Attributes"),
	  "<Control>F7", N_("Show/Hide the tag attribute editor"),
	  G_CALLBACK( view_menu_dock_show_callback ) },
	  
	{ "ViewSymbols", NULL, N_("Symbols"),
	  "<Control>F8", N_("Show/Hide the ctags browser"),
	  G_CALLBACK( view_menu_dock_show_callback ) },
	  
	{ "ViewMainToolbar", NULL, N_( "Main Toolbar" ),
	  "<Shift><Control>T", N_("Show/Hide the Main Toolbar" ),
	  G_CALLBACK( view_menu_dock_show_callback ) },
	  
  	{ "ViewWizardToolbar", NULL, N_( "Wizard Toolbar" ),
	  NULL, N_("Show/Hide the Wizards Toolbar" ),
	  G_CALLBACK( view_menu_dock_show_callback ) },

	{ "ExcludeFlag", NULL, N_("Exclude"),
	  NULL, NULL, 
	  G_CALLBACK( NULL ) },

	{ "IgnoreFlag", NULL, N_("Ignore"),
	  NULL, NULL,
	  G_CALLBACK( NULL ) },

	{ "ASCIIFlag", NULL, N_("ASCII"),
	  NULL, NULL,
	  G_CALLBACK( NULL ) }

};
static guint screem_window_toggle_action_n_entries=G_N_ELEMENTS(screem_window_toggle_action_entries);

static GtkRadioActionEntry screem_window_view_action_entries[] = {
	{ "SwitchEditor", NULL, N_("_Editor"),
	  "<control>1", N_("Switch to the editor view"), 0 },

	{ "SwitchPreview", NULL, N_("_Preview"),
	  "<control>2", N_("Switch to the preview view"), 1 },

	{ "SwitchLinkView", "Screem_LinkView", N_("_Link View"),
	  "<control>3", N_("Switch to the link view"), 2 },

	{ "SwitchTreeView", NULL, N_("_Tree View"),
	  "<control>4",  N_("Switch to the tree view"), 3 }

};
static guint screem_window_view_action_n_entries=G_N_ELEMENTS(screem_window_view_action_entries);

static GtkRadioActionEntry screem_window_toolbar_entries[] = {
	{ "ToolbarDefault", NULL, N_( "_Desktop Default" ),
	  NULL, N_( "Set toolbar button style according to desktop Menu and Toolbar Preferences" ), 0 },
	{ "ToolbarIcons", NULL, N_( "_Icons Only" ),
		NULL, N_( "Show only icons in the toolbars" ), 1 },
	{ "ToolbarBoth", NULL, N_( "Text for _All Icons" ), NULL,
		N_( "Show text below every icon in the toolbars" ), 2 },
	{ "ToolbarImportant", NULL, N_( "Text for I_mportant Icons" ),
		NULL,
		N_( "Show text only besides important icons in the toolbars" ), 3 },
	{ "ToolbarText", NULL, N_( "_Text Only" ),
		NULL, N_( "Show only text in the toolbars" ), 4 }
};
static guint screem_window_toolbar_n_entries = G_N_ELEMENTS( screem_window_toolbar_entries );

static GtkActionEntry screem_window_action_entries[] = {
	{ "File Menu", NULL, N_("_File"), NULL, NULL, NULL },
	{ "New Menu", GTK_STOCK_NEW, N_("_New"), "", NULL, NULL },
	{ "RecentPages", NULL, N_("Recent Pages"), NULL, NULL, NULL },
	{ "Edit", NULL, N_("_Edit"), NULL, NULL, NULL },
	{ "Editable Regions", NULL, N_("Editable Regions"), NULL, NULL, NULL },
	{ "Block Operations", NULL, N_("Block Operations"), NULL, NULL, NULL },
	{ "View", NULL, N_("_View"), NULL, NULL, NULL },
	{ "Search", NULL, N_("Se_arch"), NULL, NULL, NULL },
	{ "Insert", NULL, N_("_Insert"), NULL, NULL, NULL },
	{ "Wizards", NULL, N_("Wizards"), NULL, NULL, NULL },
	{ "HTML", NULL, N_("HTML"), NULL, NULL, NULL },
	{ "Formatting", NULL, N_("Formatting"), NULL, NULL, NULL },
	{ "Fonts", NULL, N_("Fonts"), NULL, NULL, NULL },
	{ "Tables", NULL, N_("Tables"), NULL, NULL, NULL },
	{ "Forms", NULL, N_("Forms"), NULL, NULL, NULL },
	{ "Objects", NULL, N_("Objects"), NULL, NULL, NULL },
	{ "Site", NULL, N_("_Site"), NULL, NULL, NULL },
	{ "RecentSites", NULL, N_("Recent Sites"), NULL, NULL, NULL },
	{ "SwitchTo", NULL, N_("Switch to site"), NULL, NULL, NULL },
	{ "Tools", NULL, N_("_Tools"), NULL, NULL, NULL },
	{ "CVS", NULL, N_("CVS"), NULL, NULL, NULL },
	{ "HelpMenu", NULL, N_("_Help"), NULL, NULL, NULL },
	{ "Helpers", NULL, N_( "_Helpers" ), NULL, NULL, NULL },
	{ "Documents", NULL, N_("_Documents"), NULL, NULL, NULL },

	{ "ToolbarStyle", NULL, N_( "Toolbar Appearence" ), NULL, NULL,
		NULL },

	{ "tabmenu", NULL, "", NULL, NULL, NULL },
	{ "editormenu", NULL, "", NULL, NULL, NULL },
	{ "siteviewmenu", NULL, "", NULL, NULL, NULL },
	{ "linkviewmenu", NULL, "", NULL, NULL, NULL },
	
	{ "New Window", GTK_STOCK_NEW, N_("New _Window"),
	  "", N_("Open a new window"),
	  G_CALLBACK( file_menu_new_window_callback ) },

	{ "New Site", GTK_STOCK_NEW, N_("New _Site..."),
	  "", N_("Create a new site"),
	  G_CALLBACK( file_menu_new_site_callback ) },

	{ "New Document", GTK_STOCK_NEW, N_("New Document..."),
	  "<Control>N", N_("Create a new document"),
	  G_CALLBACK( file_menu_new_document_callback ) },

	{ "New Blank Document", GTK_STOCK_NEW, N_("New Blank Document"),
	  "", N_("Create a new empty document"),
	  G_CALLBACK( file_menu_new_blank_document_callback ) },

	{ "New Site Template", GTK_STOCK_NEW, N_("New Site Template..."),
	  "", N_("Create a new site template"), 
	  G_CALLBACK( file_menu_new_site_template_callback ) },

	{ "Open Site", GTK_STOCK_OPEN, N_("Open..."),
	  "", N_("Open a site"), 
	  G_CALLBACK( file_menu_open_site_callback ) },

	{ "Open Site Location", GTK_STOCK_OPEN, N_("Open Location..."),
	  "", N_("Open a site from a specified location"), 
	  G_CALLBACK( file_menu_open_location_callback ) },

	{ "Save All", GTK_STOCK_SAVE, N_("Save All"),
	  "<Control><Shift>L",
	  N_("Save All modified documents in the current site"),
	  G_CALLBACK( file_menu_save_all_callback ) },

	{ "Close Site", GTK_STOCK_CLOSE, N_("Close"),
	  "", N_("Close the current site"), 
	  G_CALLBACK( file_menu_close_site_callback ) },

	{ "Open Document", GTK_STOCK_OPEN, N_("Open..."),
	  "<Control>O", N_("Open a document"), 
	  G_CALLBACK( file_menu_open_document_callback ) },

	  { "Open Location", GTK_STOCK_OPEN, N_("Open Location..."),
	   "<Control>L", N_("Open a document from a specified location"), 
	  G_CALLBACK( file_menu_open_location_callback ) },

	{ "Save Document", GTK_STOCK_SAVE, N_("Save"),
	  "<Control>S",  N_("Save a document"), 
	  G_CALLBACK( file_menu_save_document_callback ) },

	{ "Save Document As", GTK_STOCK_SAVE_AS, N_("Save As..."),
	  "<Shift><Control>S", N_("Save document as"), 
	  G_CALLBACK( file_menu_save_document_as_callback ) },

	{ "Save Copy", NULL, N_("Save Copy..."),
	  "", N_("Save a copy of the document"), 
	  G_CALLBACK( file_menu_save_document_copy_callback ) },

	  { "Save Template", NULL, N_("Save As Template..."),
	  "", N_("Save document as a template"), 
	  G_CALLBACK( file_menu_save_template_as_callback ) },

	{ "Close Document", GTK_STOCK_CLOSE, N_("Close"),
	  "<Control>W", N_("Close the current document"),
	  G_CALLBACK( file_menu_close_document_callback ) },

	{ "CloseAll", GTK_STOCK_CLOSE, N_( "Close All" ),
	  "<Control><Shift>W",
	  N_( "Close all open documents in the current site" ),
	  G_CALLBACK( file_menu_close_all_callback ) },
	  
	{ "Print", GTK_STOCK_PRINT, N_("Print..."),
	  "<Control>P",  N_("Print the current view"),
	  G_CALLBACK( file_menu_print_callback ) },

	{ "PrintPreview", GTK_STOCK_PRINT_PREVIEW, N_("Print Preview"),
	  "<Shift><Control>P",
	  N_("Preview Printing of the current view"), 
	  G_CALLBACK( file_menu_print_preview_callback ) },

	{ "PageSetup", GTK_STOCK_PROPERTIES, N_("Page Set_up"),
	  "",
	  N_("Setup page printing settings"), 
	  G_CALLBACK( file_menu_page_setup_callback ) },

	{ "Offline", "Screem_Offline", N_("Work Offline"),
	  NULL,  N_("Prevent Screem from attempting to load remote files"),
	  G_CALLBACK( file_menu_offline_callback ) },

	{ "Online", "Screem_Online", N_("Work Online"),
	  NULL, N_("Allow Screem to attempt to load remote files"), 
	  G_CALLBACK( file_menu_offline_callback ) },

	{ "Exit", GTK_STOCK_QUIT, N_("_Quit"),
	  "<Control>Q",  N_("Quit Screem"), 
	  G_CALLBACK( file_menu_exit_application_callback ) },

	{ "Undo", GTK_STOCK_UNDO, N_("Undo"),
	  "<Control>Z", N_("Undo"),
	  G_CALLBACK( edit_menu_undo_callback ) },

	{ "Redo", GTK_STOCK_REDO, N_("Redo"),
	  "<Shift><Control>Z", N_("Redo"), 
	  G_CALLBACK( edit_menu_redo_callback ) },

	{ "Cut", GTK_STOCK_CUT, N_("Cut"),
	  "<Control>X", N_("Cut current selection"), 
	  G_CALLBACK( edit_menu_cut_callback ) },

	{ "Copy", GTK_STOCK_COPY, N_("Copy"),
	  "<Control>C", N_("Copy current selection"), 
	  G_CALLBACK( edit_menu_copy_callback ) },

	{ "Paste", GTK_STOCK_PASTE, N_("Paste"),
	  "<Control>V", N_("Paste clipboard contents"),
	  G_CALLBACK( edit_menu_paste_callback ) },
	
	{ "PasteText", GTK_STOCK_PASTE, N_("Paste Unformatted"),
	  "", N_("Paste clipboard contents"),
	  G_CALLBACK( edit_menu_paste_text_callback ) },

	{ "PasteEncoded", GTK_STOCK_PASTE, N_("Paste Encoded"),
	  "",
	  N_("Paste clipboard contents, encoding html entites where needed"),
	  G_CALLBACK( edit_menu_paste_enc_callback ) },

	{ "Clear", GTK_STOCK_CLEAR, N_("Clear"),
	  "",  N_("Clear the current selection"), 
	  G_CALLBACK( edit_menu_clear_callback ) },

	{ "SelectAll", NULL, N_("Select All"),
	  "<Control>A",  N_("Select All"), 
	  G_CALLBACK( edit_menu_select_all_callback ) },

	{ "SelectContext", NULL, N_("Select Context"),
	  "<Shift><Control>A",
	  N_("Select the element which the cursor is currently within"),
	  G_CALLBACK( edit_menu_select_context_callback ) },

	{ "SelectContent", NULL, N_("Select Element Content"),
	  "<Control><Shift>C",
	  N_("Select the content of the element which the cursor is currently within"),
	  G_CALLBACK( edit_menu_select_content_callback ) },

	
	{ "Find", GTK_STOCK_FIND, N_("Find..."),
	  "<Control>F", N_("Find matches in document(s)"),
	  G_CALLBACK( edit_menu_search_callback ) },

	{ "FindAgain", GTK_STOCK_FIND, N_("Find Again"),
	  "<Control>G", N_("Find more matches in the current document"),
	 G_CALLBACK( edit_menu_search_callback ) },
	  
	{ "Replace", GTK_STOCK_FIND_AND_REPLACE, N_("Replace..."),
	  "<Control>H",
	  N_("Find and Replace matches in document(s)"),
	  G_CALLBACK( edit_menu_search_callback ) },

	{ "Find XPath", GTK_STOCK_FIND, N_("Find XPath..."),
	  "", N_("Find matches for an XPath"),
	  G_CALLBACK( NULL ) },
	{ "Find XPath Again", GTK_STOCK_FIND, N_("Find XPath Again"),
	  "", N_("Find next match for an XPath"),
	  G_CALLBACK( NULL ) },
	  
	{ "GotoLine", GTK_STOCK_JUMP_TO, N_( "Goto _Line..." ),
	  "<Control>I",  N_( "Goto a specified line" ),
	  G_CALLBACK( edit_menu_goto_line_callback ) },

	{ "GotoContextStart", GTK_STOCK_JUMP_TO, N_( "Goto Context Start" ),
	  "<Control>bracketleft",  N_( "Goto to the opening tag of the current element" ),
	  G_CALLBACK( edit_menu_goto_open_callback ) },
	  
	{ "GotoContextEnd", GTK_STOCK_JUMP_TO, N_( "Goto Context End" ),
	  "<Control>bracketright",  N_( "Goto to the closing tag of the current element" ),
	  G_CALLBACK( edit_menu_goto_close_callback ) },

	{ "GotoParent", GTK_STOCK_JUMP_TO, N_( "Goto Parent Element" ),
	  "<Alt>Up",  N_( "Goto to the parent element of the current one" ),
	  G_CALLBACK( edit_menu_goto_parent_callback ) },

	
	{ "Bookmarks", "Screem_Bookmarks", N_("Bookmarks"),
	  "<Control>B", N_("Edit Bookmarks"), 
	  G_CALLBACK( edit_menu_bookmarks_callback ) },

	{ "MarkEditable", NULL, N_("Create Editable Region..."),
	  NULL,
	  N_("Add a new editable region, or make a region out of the selected text"),
	  G_CALLBACK( edit_menu_mark_editable_callback ) },

	{ "Auto Indent", NULL, N_( "Auto Indent" ),
	  NULL,
	  N_("Indent block according to the document tree"),
	  G_CALLBACK( edit_menu_auto_indent_callback ) },
	  
	{ "Indent", NULL, N_( "Indent" ),
	  "<Shift><Control>I",
	  N_("Indent block"),
	  G_CALLBACK( edit_menu_indent_callback ) },
	  
	{ "UnIndent", NULL, N_( "Unindent" ),
	  "<Shift><Control>U",
	  N_("Unindent block"),
	  G_CALLBACK( edit_menu_unindent_callback ) },
	  
	{ "Encode Entities", NULL, N_( "Encode Entities" ),
	  NULL,
	  N_("Encode necessary characters as entities"),
	  G_CALLBACK( edit_menu_encode_entities_callback ) },
	  
	{ "URL Encode", NULL, N_( "URL Encode" ),
	  NULL,
	  N_("Encode using URL encoding. eg. %20 for space"),
	  G_CALLBACK( edit_menu_url_encode_callback ) },
	  
	{ "TagLower", NULL, N_( "Tags to lowercase" ),
	  NULL,
	  N_("Change all tags to lowercase"),
	  G_CALLBACK( edit_menu_case_callback ) },
	  
	{ "TagUpper", NULL, N_( "Tags to uppercase" ),
	  NULL,
	  N_("Change all tags to uppercase"),
	  G_CALLBACK( edit_menu_case_callback ) },
	  
	{ "BonoboCustomize", NULL, N_("Toolbars"),
	  NULL,
	  N_("Edit toolbar display and appearance"), 
	  G_CALLBACK( edit_menu_toolbars_callback ) },

	{ "EditMacros", NULL, N_( "Macros" ),
	  NULL,
	  N_( "Edit Macros" ),
	  G_CALLBACK( edit_macros_callback ) },
	  
	{ "Preferences", GTK_STOCK_PREFERENCES, N_("_Preferences"),
	  "",  N_("Edit Preferences for Screem"), 
	  G_CALLBACK( edit_menu_preferences_callback ) },

	{ "DockItems", NULL, N_("Dock Items"),
	  NULL, NULL,
	  G_CALLBACK( NULL ) },
	  
	{ "Browser0", "Screem_Preview", N_("External Preview"),
	  NULL, N_("Preview in Default external browser"),
	  G_CALLBACK( screem_toolbar_external_browser ) },

	{ "TaskList", "Screem_Todo", N_("Site Tasks"),
	  NULL,  N_("Tasks to be performed on the site"),
	  G_CALLBACK( toolbar_task_list_callback ) },

	{ "SpellCheck", GTK_STOCK_SPELL_CHECK, N_("Spell Check"),
	  "", N_("Spell check the current document"), 
	  G_CALLBACK( screem_toolbar_spell_check ) },

	{"Bold", GTK_STOCK_BOLD, N_("Bold"),
	  "",
	 N_("Bold Text"), G_CALLBACK( quickbar_bold_callback ) },

	{"Italic", GTK_STOCK_ITALIC, N_("Italic"),
	 "",
	 N_("Italic Text"), G_CALLBACK( quickbar_italic_callback ) },

	{"Underline", GTK_STOCK_UNDERLINE, N_("Underline"),
	 "",
	 N_("Underlined Text"), G_CALLBACK( quickbar_underline_callback ) },

	{"Para", "Screem_Paragraph", N_("Paragraph"),
	 NULL,
	 N_("Paragraph"), G_CALLBACK( quickbar_para_callback ) },

	{"Bullet", GNOME_STOCK_TEXT_BULLETED_LIST, N_("Bullet List"),
	 "",
	 N_("Bulleted List"), G_CALLBACK( quickbar_bullet_callback ) },

	{"Numbered", GNOME_STOCK_TEXT_NUMBERED_LIST, N_("Numbered List"),
	 "",
	 N_("Numbered List"), G_CALLBACK( quickbar_numbered_callback ) },

	{"AlignLeft", GTK_STOCK_JUSTIFY_LEFT, N_("Left Align"),
	 "",
	 N_("Align Text Left"), G_CALLBACK( quickbar_alignleft_callback ) },

	{"AlignCentre", GTK_STOCK_JUSTIFY_CENTER, N_("Center Align"),
	 "",
	 N_("Align Text Center"), G_CALLBACK( quickbar_aligncentre_callback ) },

	{"AlignRight", GTK_STOCK_JUSTIFY_RIGHT, N_("Right Align"),
	 "",
	 N_("Align Text Right"), G_CALLBACK( quickbar_alignright_callback ) },

	{"Justify", GTK_STOCK_JUSTIFY_FILL, N_("Justify"),
	 "",
	 N_("Justify Text"), G_CALLBACK( quickbar_justify_callback ) },

	{"Font", GTK_STOCK_SELECT_FONT, N_("Font"),
	 "",
	 N_("Font Attributes"), G_CALLBACK( fontbar_font_callback ) },

	{"Pre", "Screem_Pre", N_("Pre"),
	 NULL,
	 N_("Preformatted Text"), G_CALLBACK( fontbar_pre_callback ) },

	{"Subscript", "Screem_Sub", N_("Subscript"),
	 NULL,
	 N_("Subscripted Text"), G_CALLBACK( fontbar_subscript_callback ) },

	{"Superscript", "Screem_Sup", N_("Superscript"),
	 NULL,
	 N_("Superscripted Text"), G_CALLBACK( fontbar_superscript_callback ) },

	{"Table", "Screem_Table", N_("Table"),
	 NULL,
	 N_("Table"), G_CALLBACK( tablebar_table_callback ) },

	{"TBody", "Screem_Table", N_("Table Body"),
	 NULL,
	 N_("Table Body"), G_CALLBACK( tablebar_tablebody_callback ) },

	{"Tr", "Screem_Tr", N_("Table Row"),
	 NULL,
	 N_("Table Row"), G_CALLBACK( tablebar_tr_callback ) },

	{"Th", "Screem_Th", N_("Table Heading"),
	 NULL,
	 N_("Table Heading"), G_CALLBACK( tablebar_th_callback ) },

	{"Td", "Screem_Td", N_("Table Column"),
	 NULL,
	 N_("Table Column"), G_CALLBACK( tablebar_td_callback ) },

	{"Caption", "Screem_Caption", N_("Table Caption"),
	 NULL,
	 N_("Table Caption"), G_CALLBACK( tablebar_caption_callback ) },

	{"Summary", "Screem_Caption", N_("Table Summary"),
	 NULL,
	 N_("Table Summary"), G_CALLBACK( tablebar_summary_callback ) },

	{"Textarea", "Screem_Text", N_("Textarea"),
	 NULL,
	 N_("Text Area"), G_CALLBACK( formbar_textarea_callback ) },

	{"Input", "Screem_Entry", N_("Input"),
	 NULL,
	 N_("Text Input"), G_CALLBACK( formbar_input_callback ) },

	{"Fileinput", "Screem_FileEntry", N_("File Input"),
	 NULL,
	 N_("File Input"), G_CALLBACK( formbar_fileinput_callback ) },

	{"Button", "Screem_Button", N_("Button"),
	 NULL,
	 N_("Button"), G_CALLBACK( formbar_button_callback ) },

	{"Checkbutton", "Screem_Checkbutton", N_("Checkbutton"),
	 NULL,
	 N_("Checkbutton"), G_CALLBACK( formbar_checkbutton_callback ) },

	{"Radiobutton", "Screem_Radiobutton", N_("Radiobutton"),
	 NULL,
	 N_("Radiobutton"), G_CALLBACK( formbar_radiobutton_callback ) },

	{"Optionmenu", "Screem_Optionmenu", N_("Optionmenu"),
	 NULL,
	 N_("Optionmenu"), G_CALLBACK( formbar_optionmenu_callback ) },

	{ "flash object", NULL, N_( "Flash Movie" ), 
		NULL,
		N_( "Insert a Flash movie, set the data=\"\" and value=\"\" to the file to view" ), 
		G_CALLBACK( insert_object_flash_callback ) },
	{ "applet object", NULL, N_( "Java Applet" ), 
		NULL,
		N_( "Insert a Java applet, set the classid=\"\" to java:YourFile.class" ), 
		G_CALLBACK( insert_object_applet_callback ) },
	 
	{ "Basic Structure", NULL, N_( "Basic Structure" ),
	  NULL,
	  N_( "Insert basic HTML page, based on the default doctype" ),
	  G_CALLBACK( insert_basic_html_callback ) },

	{ "Closing Tag", NULL, N_( "Closing Tag" ),
	  "<Control>Return",
	  N_( "Insert the closing tag for the current element" ),
	  G_CALLBACK( insert_closing_tag_callback ) },
	
	{ "Generator", NULL, N_( "Generator meta" ),
	  NULL,
	  N_( "Insert a generator meta tag" ),
	  G_CALLBACK( meta_insert_callback ) },

	{ "Author", NULL, N_( "Author meta" ),
	  NULL,
	  N_( "Insert an author meta tag" ),
	  G_CALLBACK( meta_insert_callback ) },

	{ "Content-Type", NULL, N_( "Content-Type meta" ),
	  NULL,
	  N_( "Insert a content type meta tag" ),
	  G_CALLBACK( meta_insert_callback ) },

	{ "Content-Script-Type", NULL, N_( "Content-Script-Type meta" ),
	  NULL,
	  N_( "Insert a script type meta tag" ),
	  G_CALLBACK( meta_insert_callback ) },

	{ "Content-Style-Type", NULL, N_( "Content-Style-Type meta" ),
	  NULL,
	  N_( "Insert a style type meta tag" ),
	  G_CALLBACK( meta_insert_callback ) },
	
	{ "MetaRefresh", NULL, N_( "Refresh meta" ),
	  NULL,
	  N_( "Insert a refresh meta tag" ),
	  G_CALLBACK( meta_insert_callback ) },
	  
	{"InsertDoctype", NULL, N_("_Doctype tag..."),
	 NULL,
	 N_("Insert a <!DOCTYPE>"), G_CALLBACK( insert_menu_doctype_callback ) },

	{"InsertFile", NULL, N_("_From File..."),
	 NULL,
	 N_("Insert text from a file"), G_CALLBACK( insert_menu_file_callback ) },

	{"CheckoutSite", "Screem_CVSCheckout", N_("Checkout Site"),
	 NULL,
	 N_("Checkout a Site from CVS"), G_CALLBACK( cvs_menu_checkout_site_callback ) },

	{"UpdateSite", "Screem_CVSUpdate", N_("Update Site"),
	 NULL,
	 N_("Update a Site from CVS"), G_CALLBACK( cvs_menu_update_site_callback ) },

	{"UpdatePage", "Screem_CVSUpdate", N_("Update Page"),
	 NULL,
	 N_("Update a Page from CVS"), G_CALLBACK( cvs_menu_update_page_callback ) },

	{"ImportSite", NULL, N_("Import Site"),
	 NULL,
	 N_("Place a Site in CVS"), G_CALLBACK( cvs_menu_import_site_callback ) },

	{"CommitSite", NULL, N_("Commit Site"),
	 NULL,
	 N_("Commit Site changes to CVS"), G_CALLBACK( cvs_menu_commit_site_callback ) },

	{"CommitPage", NULL, N_("Commit Page"),
	 NULL,
	 N_("Commit Page changes to CVS"), G_CALLBACK( cvs_menu_commit_page_callback ) },

	{"AddPage", "Screem_CVSAdd", N_("Add Page"),
	 NULL,
	 N_("Add Page to CVS"), G_CALLBACK( cvs_menu_add_page_callback )},

	{ "AddFile", "Screem_CVSAdd", N_("Add File"),
	  NULL, N_("Add File to CVS"),
	  G_CALLBACK( cvs_menu_add_page_callback ) },

	{ "CommitFile", NULL, N_("Commit File"),
	  NULL, N_("Commit File to CVS"),
	  G_CALLBACK( cvs_menu_commit_page_callback ) },

	{ "UpdateFile", "Screem_CVSUpdate", N_("Update File"),
	  NULL, N_("Update File from CVS"),
	  G_CALLBACK( cvs_menu_update_page_callback ) },

	{ "CVSLogin", NULL, N_("Login"),
	  NULL, N_("Login to a CVS server"),
	  G_CALLBACK( cvs_menu_login_callback ) },

	{ "FixLinks", NULL, N_( "Fix Links..." ),
	  NULL, 
	  N_( "Update all links to a source to point to a new location" ),
	  G_CALLBACK( tool_menu_fix_links_callback ) },

	{ "TemplateUpdate", NULL, N_( "Update from Template" ),
	  NULL, 
	  N_( "Updates the page from the template it is using" ),
	  G_CALLBACK( tool_menu_template_update_callback ) },

	{ "EditHelpers", NULL, N_( "Edit Helpers" ),
	  NULL,
	  N_( "Edit helpers" ),
	  G_CALLBACK( edit_helpers_callback ) },

	{ "PreviousDocument", GTK_STOCK_GO_BACK, N_( "_Previous Document" ),
	  "<Control>Page_Up", 
	  N_( "Switch to the previous open document" ),
	  G_CALLBACK( doc_menu_previous_callback ) },

	{ "NextDocument", GTK_STOCK_GO_FORWARD, N_( "_Next Document" ),
	  "<Control>Page_Down", 
	  N_( "Switch to the next open document" ),
	  G_CALLBACK( doc_menu_next_callback ) },
	  
	{ "Help", GTK_STOCK_HELP, N_("Contents"),
	  "F1", N_("Help On this application"), 
	  G_CALLBACK( help_menu_help_callback ) },

	{ "About Screem", GNOME_STOCK_ABOUT, N_("_About Screem"),
	  "",
	  N_("Display credits for the creators of Screem"), 
	  G_CALLBACK( help_menu_about_screem_callback ) },

	{ "Tip Of The Day", GTK_STOCK_DIALOG_INFO, N_("Tip Of The Day"),
	  "",  N_("Show the Tip Of The Day"),
	  G_CALLBACK( help_menu_tip_callback ) },

	{ "Debug", GTK_STOCK_DIALOG_INFO, N_("Debug"),
	  "",  N_("Debug"),
	  G_CALLBACK( help_menu_debug_callback ) },

	{ "New Directory", GTK_STOCK_NEW, N_("New Directory"),
	  "", NULL, 
	  G_CALLBACK( site_view_menu_new_dir_callback ) },

	{ "Delete File", GTK_STOCK_DELETE, N_("Delete"),
	  "", NULL, 
	  G_CALLBACK( site_view_menu_delete_file_callback ) },

	{ "File Properties", GTK_STOCK_PROPERTIES, N_("Properties"),
	  "", NULL, 
	  G_CALLBACK( site_view_menu_file_props_callback ) },

	{ "SiteSettings", GTK_STOCK_PROPERTIES, N_("_Site Settings"),
	  "", N_("Edit Settings for the current Site"), 
	  G_CALLBACK( site_menu_site_settings_callback ) },
};

static 
guint screem_window_action_n_entries=G_N_ELEMENTS(screem_window_action_entries);


void screem_window_initialise_menus( ScreemWindow *window )
{
	GtkUIManager *merge;
	GtkAction *action;
	gint major;
	gint minor;
	gint patch;
	gboolean show;
	GError *error;

	EggToolbarsModel *model;
	GtkWidget *toolbar;

	gchar *editablefile;
	gchar *temp;
	
	window->action_group = gtk_action_group_new( "ScreemActions" );
	gtk_action_group_set_translation_domain( window->action_group, 
			GETTEXT_PACKAGE );	

	gtk_action_group_add_actions( GTK_ACTION_GROUP( window->action_group ),
			screem_window_action_entries,
			screem_window_action_n_entries,
			window );
	gtk_action_group_add_toggle_actions( GTK_ACTION_GROUP( window->action_group ),
			screem_window_toggle_action_entries,
			screem_window_toggle_action_n_entries,
			window );
	gtk_action_group_add_radio_actions( GTK_ACTION_GROUP( window->action_group ),
			screem_window_view_action_entries,
			screem_window_view_action_n_entries,
			0,
			G_CALLBACK( view_menu_view_callback ), 
			window );
	gtk_action_group_add_radio_actions( GTK_ACTION_GROUP( window->action_group ),
			screem_window_toolbar_entries,
			screem_window_toolbar_n_entries,
			0,
			G_CALLBACK( view_menu_toolbar_callback ), 
			window );
	
	/* disable debug action for stable versions */
	action = gtk_action_group_get_action( GTK_ACTION_GROUP( window->action_group ), "Debug" );
	show = TRUE;
	major = minor = patch = 0;
	if( sscanf( VERSION, "%i.%i.%i", &major, &minor, &patch ) > 2 ){
		show = ( minor % 2 );
	}
	g_object_set( G_OBJECT( action ),
			"visible", show, NULL );
	
	
	merge = gtk_ui_manager_new ();
	g_signal_connect( merge, "connect_proxy",
		G_CALLBACK( connect_proxy_cb ), window );
	g_signal_connect( merge, "disconnect_proxy",
		G_CALLBACK( disconnect_proxy_cb ), window );
	
	gtk_ui_manager_insert_action_group( merge, 
					    GTK_ACTION_GROUP( window->action_group ), 
					    0 );
	window->merge = G_OBJECT( merge );
	
	g_signal_connect( merge, "add_widget", G_CALLBACK( add_widget ), window );
	
	error = NULL;
  	if( ! gtk_ui_manager_add_ui_from_file( merge, 
					 UIDATADIR"/screem-window-menu.xml", 
					 &error ) ) {
		g_message( "Main ui file error = %s", error->message );
		g_error_free( error );
		exit( 0 );
	}
	
	error = NULL;
	if( ! gtk_ui_manager_add_ui_from_file( merge,
					 UIDATADIR"/screem-window-toolbar.xml",
					 &error ) ) {
		g_message( "Toolbar ui file error = %s", error->message );
		g_error_free( error );
		exit( 0 );
	}
	
	error = NULL;
	if( ! gtk_ui_manager_add_ui_from_file( merge,
					 UIDATADIR"/screem-window-popups.xml",
					 &error ) ) {
		g_message( "Popup ui file error = %s", error->message );
		g_error_free( error );
		exit( 0 );
	}
		
	screem_toolbar_create_browsers( window );

	temp = screem_get_dot_dir();
	editablefile = g_build_filename( temp, "screem-editable-toolbars.xml", NULL );
	g_free( temp );
	temp = editablefile;
	if( ! uri_exists( editablefile, NULL ) ) {
		editablefile = g_strdup( UIDATADIR"/screem-editable-toolbars.xml" );
	} 

	model = egg_toolbars_model_new();
	window->toolbar_model = G_OBJECT( model );
	egg_toolbars_model_load( model, editablefile );
	g_free( editablefile );

	egg_toolbars_model_set_flags( model, 
			EGG_TB_MODEL_NOT_REMOVABLE, 0 );

	/* make sure we have the copy in ~/.screem if we didn't
	 * to start with */
	if( temp != editablefile ) {
		egg_toolbars_model_save( model, temp, "1.0" );
		g_free( temp );
	}

	window->toolbar = toolbar = egg_editable_toolbar_new( merge, 
								model );
	gtk_widget_show( toolbar );
	gtk_box_pack_start( GTK_BOX( window->content_hbox ), toolbar,
			    FALSE, TRUE, 0 );
	gtk_box_reorder_child( GTK_BOX( window->content_hbox ), 
				toolbar, 0 );
	egg_editable_toolbar_show( EGG_EDITABLE_TOOLBAR( toolbar ),
			"Main Toolbar" );
	g_object_set_data( G_OBJECT( merge ), "toolbars",
				GUINT_TO_POINTER( 1 ) );
	
	gtk_window_add_accel_group( GTK_WINDOW( window ), 
			gtk_ui_manager_get_accel_group( merge ) );
}

