/*  Screem:  screem-site-druid.c,
 *  A site druid for guiding users through the process of creating a site
 *
 *  Copyright (C) 2002  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <pwd.h>
#include <unistd.h>
#include <sys/types.h>

#include <gtk/gtkentry.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkspinbutton.h>
#include <gtk/gtknotebook.h>
#include <gtk/gtkimage.h>
#include <gtk/gtkdialog.h>
#include <gtk/gtkmessagedialog.h>

#include <glib/gi18n.h>

#include <libgnomeui/libgnomeui.h>
#include <libgnomevfs/gnome-vfs.h>
#include <glade/glade.h>


#include "screem-application.h"
#include "screem-window.h"
#include "screem-site.h"
#include "screem-page.h"

#include "screem-site-druid.h"

#include "fileops.h"
#include "support.h"

static void screem_site_druid_create_site( ScreemSite *site, GladeXML *xml );

static gboolean screem_site_druid_create_initial_page( ScreemWindow *window );

void screem_site_druid_begin( ScreemApplication *application, 
			      ScreemWindow *window )
{
	GladeXML *xml;
	gchar *glade_path;
	GtkWidget *druid;
	ScreemSite *site;
	ScreemSession *session;
	
	session = screem_application_get_session( application );

	glade_path = screem_get_glade_path();	
	xml = glade_xml_new( glade_path, "new_site_druid", NULL );

	druid = glade_xml_get_widget( xml, "new_site_druid" );
	site = screem_site_new( G_OBJECT( application ) );

	g_object_set_data( G_OBJECT( druid ), "site", site );
	g_object_set_data( G_OBJECT( druid ), "application", application );
	g_object_set_data( G_OBJECT( druid ), "window", window );

	glade_xml_signal_autoconnect( xml );

	gtk_window_set_transient_for( GTK_WINDOW( druid ),
				      GTK_WINDOW( window ) );

	screem_session_restore_dialog( session, druid );

	gtk_widget_show_all( druid );
}

void screem_site_druid_finish( GnomeDruidPage *druid_page,
		GtkWidget *widget )
{
	ScreemSite *site;
	ScreemWindow *window;
	ScreemApplication *application;
	GladeXML *xml;
	ScreemSession *session;

	xml = glade_get_widget_tree( GTK_WIDGET( druid_page ) );
	widget = glade_xml_get_widget( xml, "new_site_druid" );

	site = SCREEM_SITE( g_object_get_data( G_OBJECT( widget ), "site" ) );
	application = SCREEM_APPLICATION( g_object_get_data( G_OBJECT( widget),
							     "application" ) );
	window = SCREEM_WINDOW( g_object_get_data( G_OBJECT( widget ),
						   "window" ) );

	screem_site_druid_create_site( site, xml );

	if( screem_site_create( site ) ) {
		screem_application_add_site( application, site );
		screem_window_set_current( window, site );
		g_idle_add_full( G_PRIORITY_LOW,
				 (GSourceFunc)screem_site_druid_create_initial_page,
			   	 window, NULL );
	} else {
		GtkWidget *alert;

		alert = gtk_message_dialog_new( GTK_WINDOW( window ),
						GTK_DIALOG_MODAL |
						GTK_DIALOG_DESTROY_WITH_PARENT,
						GTK_MESSAGE_ERROR,
						GTK_BUTTONS_OK,
						_( "Failed to create %s" ),
						screem_site_get_name( site ) );
		gtk_dialog_run( GTK_DIALOG( alert ) );
		gtk_widget_destroy( alert );
		g_object_unref( site );
	}
	session = screem_application_get_session( application );
	screem_session_store_dialog( session, widget );

	gtk_widget_destroy( widget );
	g_object_unref( xml );
}

void screem_site_druid_cancel( GnomeDruidPage *druid_page,
		GtkWidget *widget )
{
	ScreemApplication *app;
	ScreemSession *session;
	ScreemSite *site;
	GladeXML *xml;
	GtkWidget *druid;

	xml = glade_get_widget_tree( GTK_WIDGET( druid_page ) );
	druid = glade_xml_get_widget( xml, "new_site_druid" );
	
	site = SCREEM_SITE( g_object_get_data( G_OBJECT( druid ), "site" ) );
	g_object_unref( G_OBJECT( site ) );
	app = SCREEM_APPLICATION( g_object_get_data( G_OBJECT( druid),
				"application" ) );
	session = screem_application_get_session( app );
	screem_session_store_dialog( session, druid );

	gtk_widget_destroy( druid );
	g_object_unref( xml );
}

gboolean screem_site_druid_start_next( GnomeDruidPage *druid_page,
		GtkWidget *druid )
{
	GladeXML *xml;
	GtkWidget *widget;
	GnomeDruidPage *page;

	xml = glade_get_widget_tree( GTK_WIDGET( druid_page ) );

	widget = glade_xml_get_widget( xml, "druid1" );
	page = GNOME_DRUID_PAGE( glade_xml_get_widget( xml, 
				"detailspage" ) );
	gnome_druid_set_page( GNOME_DRUID( widget ), page );
	gnome_druid_set_buttons_sensitive( GNOME_DRUID( widget ),
			TRUE, FALSE, TRUE, TRUE );

	return TRUE;
}

void screem_site_druid_details_prepare( GnomeDruidPage *druid_page,
		GtkWidget *druid )
{
	GladeXML *xml;
	GtkWidget *widget;

	xml = glade_get_widget_tree( GTK_WIDGET( druid_page ) );

	widget = glade_xml_get_widget( xml, "druid1" );
	gnome_druid_set_buttons_sensitive( GNOME_DRUID( widget ),
			TRUE, FALSE, TRUE, TRUE );

}

void screem_site_druid_details_validate( GtkWidget *widget )
{
	GladeXML *xml;
	const gchar *value;
	gboolean valid;
	
	xml = glade_get_widget_tree( widget );
	
	widget = glade_xml_get_widget( xml, "sitename" );
	widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
	value = gtk_entry_get_text( GTK_ENTRY( widget ) );
	valid = ( value && *value != '\0' );

	widget = glade_xml_get_widget( xml, "sitepath" );
	widget = gnome_file_entry_gtk_entry( GNOME_FILE_ENTRY( widget ) );
	value = gtk_entry_get_text( GTK_ENTRY( widget ) );
	valid = valid && ( value && *value != '\0' );
	
	widget = glade_xml_get_widget( xml, "druid1" );
	gnome_druid_set_buttons_sensitive( GNOME_DRUID( widget ),
			TRUE, valid, TRUE, TRUE );
}

gboolean screem_site_druid_details_next( GnomeDruidPage *druid_page,
		GtkWidget *druid )
{
	GladeXML *xml;
	gboolean ret;

	xml = glade_get_widget_tree( GTK_WIDGET( druid_page ) );
	ret = FALSE;

	return ret;
}

gboolean screem_site_druid_management_next( GnomeDruidPage *druid_page,
		GtkWidget *druid )
{
	GladeXML *xml;
	gboolean ret;

	xml = glade_get_widget_tree( GTK_WIDGET( druid_page ) );
	ret = FALSE;

	return ret;
}

void screem_site_druid_publishing_prepare( GnomeDruidPage *druid_page,
		GtkWidget *druid )
{
	GladeXML *xml;
	GtkWidget *widget;

	xml = glade_get_widget_tree( GTK_WIDGET( druid_page ) );

	widget = glade_xml_get_widget( xml, "username" );
	widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
	gtk_entry_set_text( GTK_ENTRY( widget ), g_get_user_name() );

	widget = glade_xml_get_widget( xml, "remote_method" );
	gtk_combo_box_set_active( GTK_COMBO_BOX( widget ), FTP );
}

gboolean screem_site_druid_publishing_next( GnomeDruidPage *druid_page,
		GtkWidget *druid )
{
	GladeXML *xml;
	gboolean ret;

	xml = glade_get_widget_tree( GTK_WIDGET( druid_page ) );
	ret = FALSE;

	return ret;
}

gboolean screem_site_druid_options_next( GnomeDruidPage *druid_page,
		GtkWidget *druid )
{
	GladeXML *xml;
	gboolean ret;

	xml = glade_get_widget_tree( GTK_WIDGET( druid_page ) );
	ret = FALSE;

	return ret;
}

static void screem_site_druid_create_site( ScreemSite *site, GladeXML *xml )
{
	GtkWidget *widget;
	const gchar *value;
	gboolean flag;
	ScreemSitePerms sp;
	ScreemSiteSymlinks sl;
	GnomeVFSURI *uri;
	
	/* process details page */
	widget = glade_xml_get_widget( xml, "sitename" );
	widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
	value = gtk_entry_get_text( GTK_ENTRY( widget ) );
	screem_site_set_name( site, value );

	widget = glade_xml_get_widget( xml, "sitepath" );
	widget = gnome_file_entry_gtk_entry( GNOME_FILE_ENTRY( widget ) );
	value = gtk_entry_get_text( GTK_ENTRY( widget ) );
	value = gtk_entry_get_text( GTK_ENTRY( widget ) );
	screem_site_set_pathname( site, value );

	widget = glade_xml_get_widget( xml, "usesitetemplate" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		widget = glade_xml_get_widget( xml, "sitetemplate" );
		widget = gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY(widget));
		value = gtk_entry_get_text( GTK_ENTRY( widget ) );
		screem_site_set_site_template( site, value );
	}

	widget = glade_xml_get_widget( xml, "usesitecvs" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		widget = glade_xml_get_widget( xml, "sitecvs" );
		widget = gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY(widget));
		value = gtk_entry_get_text( GTK_ENTRY( widget ) );
		screem_site_set_cvs_root( site, value );
	}

	widget = glade_xml_get_widget( xml, "autocvsupdate" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		screem_site_set_auto_update( site, TRUE );
	}

	widget = glade_xml_get_widget( xml, "autoctagsupdate" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		screem_site_set_use_ctags( site, TRUE );
	}
	
	widget = glade_xml_get_widget( xml, "usepagetemplate" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		widget = glade_xml_get_widget( xml, "sitepagetemplate" );
		widget = gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY(widget));
		value = gtk_entry_get_text( GTK_ENTRY( widget ) );
		screem_site_set_template_path( site, value );
	}

	/* process publish settings 1 page */
	widget = glade_xml_get_widget( xml, "remote_method" );
	screem_site_set_remote_method( site,
			gtk_combo_box_get_active( GTK_COMBO_BOX( widget ) ) );

	widget = glade_xml_get_widget( xml, "remote_location" );
	widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
	value = gtk_entry_get_text( GTK_ENTRY( widget ) );

	uri = gnome_vfs_uri_new( value );
	if( uri ) {
		value = gnome_vfs_uri_get_host_name( uri );
		screem_site_set_remote_url( site, value );

		value = gnome_vfs_uri_get_path( uri );
		screem_site_set_remote_path( site, value );
	}	
	widget = glade_xml_get_widget( xml, "baseuri" );
	widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
	value = gtk_entry_get_text( GTK_ENTRY( widget ) );
	screem_site_set_http_url( site, value );

	widget = glade_xml_get_widget( xml, "username" );
	widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
	value = gtk_entry_get_text( GTK_ENTRY( widget ) );
	screem_site_set_remote_user( site, value );

	widget = glade_xml_get_widget( xml, "password" );
	value = gtk_entry_get_text( GTK_ENTRY( widget ) );
	screem_site_set_remote_pass( site, value );

	/* process publish settings 2 page */
	widget = glade_xml_get_widget( xml, "passiveftp" );
	flag = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );
	screem_site_set_passive_ftp( site, flag );

	widget = glade_xml_get_widget( xml, "nodelete" );
	flag = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );
	screem_site_set_no_delete( site, flag );

	widget = glade_xml_get_widget( xml, "checkmoved" );
	flag = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );
	screem_site_set_check_moved( site, flag );

	widget = glade_xml_get_widget( xml, "nooverwrite" );
	flag = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );
	screem_site_set_no_overwrite( site, flag );

	sp = PERMS_IGNORE;
	widget = glade_xml_get_widget( xml, "p_exec" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) )
		sp = PERMS_EXEC;
	widget = glade_xml_get_widget( xml, "p_all" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) )
		sp = PERMS_ALL;
	screem_site_set_permissions( site, sp );

	sl = SYM_IGNORE;
	widget = glade_xml_get_widget( xml, "s_follow" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) )
		sp = SYM_FOLLOW;
	widget = glade_xml_get_widget( xml, "s_maintain" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) )
		sp = SYM_MAINTAIN;
	screem_site_set_symlinks( site, sl );
}

static gboolean screem_site_druid_create_initial_page( ScreemWindow *window )
{
	GConfClient *client;
	ScreemApplication *app;
	ScreemSite *site;
	ScreemPage *page;
	const gchar *template;
	gchar *default_file;
	gchar *tmp;
	GError *error;

	client = gconf_client_get_default();
	
	site = screem_window_get_current( window );
	g_object_get( G_OBJECT( site ), "app", &app, NULL );
	
	tmp = gconf_client_get_string( client,
				       "/apps/screem/default_filename",
				       NULL );
	if( ! tmp ) {
		tmp = g_strdup( "index.html" );
	}

	default_file = g_build_filename( screem_site_get_pathname( site ),
			tmp, NULL );
	g_free( tmp );

	page = screem_page_new( G_OBJECT( app ) );

	template = screem_site_get_template_path( site );

	if( template ) {
		screem_page_set_pathname( page, template );
		if( ! screem_page_load( page, NULL ) ) {
			/* FIXME: report error */
		}
	}
	       
	screem_page_set_pathname( page, default_file );
	screem_page_set_changed( page, TRUE );
	gdk_threads_enter();
	error = NULL;
	if( overwrite( default_file ) && screem_page_save( page, &error ) ) {

		/* if we have overwritten a page with the same
		 * filename as the initial one the page will already
		 * be in the site so don't add it */
		if( ! screem_site_locate_page( site, default_file ) ) {
			screem_site_add_page( site, page );
		}

		screem_window_set_document( window, page );
	} else {
		/* failed to save the page */
		gchar *primary;
		const gchar *secondary;

		primary = g_strdup_printf( _( "Failed to create initial page: %s" ),
				default_file );
		secondary = "";
		if( error ) {
			secondary = error->message;
		}
		screem_hig_alert( GTK_STOCK_DIALOG_ERROR,
				primary, secondary,
				GTK_WIDGET( window ) );
		if( error ) {
			g_error_free( error );
		}
	}
	g_object_unref( page );

	gdk_threads_leave();
	g_free( default_file );
	g_object_unref( client );
	g_object_unref( app );

	return FALSE;
}
