/*******************************************************************************
 * Score Reading Trainer                                                       *
 * Copyright (C) 2004 by Jos Pablo Ezequiel Fernndez <pupeno@pupeno.com>     *
 *                                                                             *
 * This program is free software; you can redistribute it and/or               *
 * modify it under the terms of the GNU General Public License                 *
 * version 2 as published by the Free Software Foundation.                     *
 *                                                                             *
 * This program is distributed in the hope that it will be useful,             *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 * GNU General Public License for more details.                                *
 *                                                                             *
 * You should have received a copy of the GNU General Public License           *
 * along with this program; if not, write to the Free Software                 *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *******************************************************************************/


#include <qlayout.h>
#include <qhbox.h>
#include <qvaluevector.h>

#include <kpushbutton.h>
#include <kapplication.h>
#include <klocale.h>
#include <kdebug.h>
#include <kconfig.h>

#include "ScoreReadingTrainerView.h"

ScoreReadingTrainerView::ScoreReadingTrainerView(QWidget *parent) : QWidget(parent){
	//kdDebug() << "Running ScoreReadingTrainerView::ScoreReadingTrainerView(QWidget *parent)" << endl;

	availableNotes = new QValueVector<int>;
	QVBoxLayout *mainLayout = new QVBoxLayout( this );
	mainLayout->setAutoAdd(true);

	// Create a Score widget which is the one that does the hard work
	score = new Score(this, "Score");
}

ScoreReadingTrainerView::~ScoreReadingTrainerView(){
	//kdDebug() << "Running ScoreReadingTrainerView::~ScoreReadingTrainerView()" << endl;
}

void ScoreReadingTrainerView::start(){
	updateAvailableNotes();
	//FIXME this should fill all the space with notes, not just generate some of them
	for(int a = 0 ; a <= 20 ; a++){
		generateRandomNote();
	}
	started = true;
}

void ScoreReadingTrainerView::stop(){
	started = false;
	score->clearNotes();
}

void ScoreReadingTrainerView::setAutomatic(bool t){
	kdDebug() << "Running ScoreReadingTrainerView::setAutomatic(bool t = true)" << endl;
	mAutomatic = t;
}

Score::Clefs ScoreReadingTrainerView::clef(){
	kdDebug() << "Running ScoreReadingTrainerView::clef()" << endl;
	return score->clef();
}

void ScoreReadingTrainerView::setClef(Score::Clefs pClef){
	kdDebug() << "Running ScoreReadingTrainerView::setClef(Score::Clefs pClef)" << endl;
	score->setClef(pClef);
}

Score::KeySignatures ScoreReadingTrainerView::keySignature(){
	kdDebug() << "Running ScoreReadingTrainerView::keySignature()" << endl;
	return score->keySignature();
}

void ScoreReadingTrainerView::setKeySignature(Score::KeySignatures pKeySignature){
	kdDebug() << "Running ScoreReadingTrainerView::setKeySignature(Score::KeySignature pKeySignature)" << endl;
	score->setKeySignature(pKeySignature);
}

int ScoreReadingTrainerView::bottomLimit(){
	kdDebug() << "Running ScoreReadingTrainerView::bottomLimit()" << endl;
	return mBottomLimit;
}

void ScoreReadingTrainerView::setBottomLimit(int pBottomLimit){
	kdDebug() << "Running ScoreReadingTrainerView::setBottomLimit(int)" << endl;
	mBottomLimit = pBottomLimit;
}

int ScoreReadingTrainerView::topLimit(){
	kdDebug() << "Running ScoreReadingTrainerView::topLimit()" << endl;
	return mTopLimit;
}

void ScoreReadingTrainerView::setTopLimit(int pTopLimit){
	kdDebug() << "Running ScoreReadingTrainerView::setTopLimit(int)" << endl;
	mTopLimit = pTopLimit;
}

void ScoreReadingTrainerView::setAccidentals(bool pAccidentals){
	//kdDebug() << "Running ScoreReadingTrainerView::setAccidentals(bool pAccidentals)" << endl;
	mAccidentals = pAccidentals;
}

void ScoreReadingTrainerView::generateRandomNote(bool remove){
	//kdDebug() << "Running ScoreReadingTrainerView::generateRandomNote()" << endl;

	// If remove is true, remove one note.
	if(remove){
		score->dequeueNote();
	}

	//Calculate a random note, a random note among the available notes
	int randomNote = availableNotes->at(KApplication::random() % availableNotes->count());

	// simplify note to be only the note itself without octaves.
	// We add 12 and re-module 12 to avoid having negative values.
	int simplifiedRandomNote = ((randomNote % 12) + 12) % 12;

	Score::NoteValues note = Score::C;
	Score::Accidentals accidental = Score::None;

	// Using the simplified note, separate into note and accidental.
	switch(simplifiedRandomNote){
		case(ScoreReadingTrainerView::C):{
			note = Score::C;
			accidental = Score::None;
			break;
		}
		case(ScoreReadingTrainerView::CSharp):{
			note = Score::C;
			accidental = Score::Sharp;
			break;
		}
		case(ScoreReadingTrainerView::D):{
			note = Score::D;
			accidental = Score::None;
			break;
		}
		case(ScoreReadingTrainerView::DSharp):{
			note = Score::D;
			accidental = Score::Sharp;
			break;
		}
		case(ScoreReadingTrainerView::E):{
			note = Score::E;
			accidental = Score::None;
			break;
		}
		case(ScoreReadingTrainerView::F):{
			note = Score::F;
			accidental = Score::None;
			break;
		}
		case(ScoreReadingTrainerView::FSharp):{
			note = Score::F;
			accidental = Score::Sharp;
			break;
		}
		case(ScoreReadingTrainerView::G):{
			note = Score::G;
			accidental = Score::None;
			break;
		}
		case(ScoreReadingTrainerView::GSharp):{
			note = Score::G;
			accidental = Score::Sharp;
			break;
		}
		case(ScoreReadingTrainerView::A):{
			note = Score::A;
			accidental = Score::None;
			break;
		}
		case(ScoreReadingTrainerView::ASharp):{
			note = Score::A;
			accidental = Score::Sharp;
			break;
		}
		case(ScoreReadingTrainerView::B):{
			note = Score::B;
			accidental = Score::None;
			break;
		}
	}

	// Convert sharps into flat when needed, 50% of time, nothing more.
	if(KApplication::random() % 2 == 0 and mAccidentals){	
		if(accidental == Score::Sharp){
			switch(note){
				case(Score::C):{
					note = Score::D;
					accidental = Score::Flat;
					break;
				}
				case(Score::D):{
					note = Score::E;
					accidental = Score::Flat;
					break;
				}
				case(Score::E):{
					note = Score::F;
					accidental = Score::Flat;
					break;
				}
				case(Score::F):{
					note = Score::G;
					accidental = Score::Flat;
					break;
				}
				case(Score::G):{
					note = Score::A;
					accidental = Score::Flat;
					break;
				}
				case(Score::A):{
					note = Score::B;
					accidental = Score::Flat;
					break;
				}
				case(Score::B):{
					note = Score::C;
					accidental = Score::Flat;
					break;
				}
			}
		}
	}

	// Transform the calculated note to what's necesary acording to the current key signature
	if(note == Score::F and score->keySignature() >= Score::KeySignatureG ){
		if(accidental == Score::Sharp){
			accidental = Score::None;
		} else if(accidental == Score::None){
			accidental = Score::Natural;
		}
	}
	if(note == Score::C and score->keySignature() >= Score::KeySignatureD){
		if(accidental == Score::Sharp){
			accidental = Score::None;
		} else if(accidental == Score::None){
			accidental = Score::Natural;
		}
	}
	if(note == Score::G and score->keySignature() >= Score::KeySignatureA){
		if(accidental == Score::Sharp){
			accidental = Score::None;
		} else if(accidental == Score::None){
			accidental = Score::Natural;
		}
	}
	if(note == Score::D and score->keySignature() >= Score::KeySignatureE){
		if(accidental == Score::Sharp){
			accidental = Score::None;
		} else if(accidental == Score::None){
			accidental = Score::Natural;
		}
	}
	if(note == Score::A and score->keySignature() >= Score::KeySignatureB){
		if(accidental == Score::Sharp){
			accidental = Score::None;
		} else if(accidental == Score::None){
			accidental = Score::Natural;
		}
	}
	if(note == Score::E and score->keySignature() >= Score::KeySignatureFSharp){
		if(accidental == Score::Sharp){
			accidental = Score::None;
		} else if(accidental == Score::None){
			accidental = Score::Natural;
		}
	}
	if(note == Score::B and score->keySignature() >= Score::KeySignatureCSharp){
		if(accidental == Score::Sharp){
			accidental = Score::None;
		} else if(accidental == Score::None){
			accidental = Score::Natural;
		}
	}

	if(note == Score::B and score->keySignature() <= Score::KeySignatureF){
		if(accidental == Score::Flat){
			accidental = Score::None;
		} else if(accidental == Score::None){
			accidental = Score::Natural;
		}
	}

	if(note == Score::E and score->keySignature() <= Score::KeySignatureBFlat){
		if(accidental == Score::Flat){
			accidental = Score::None;
		} else if(accidental == Score::None){
			accidental = Score::Natural;
		}
	}

	if(note == Score::A and score->keySignature() <= Score::KeySignatureEFlat){
		if(accidental == Score::Flat){
			accidental = Score::None;
		} else if(accidental == Score::None){
			accidental = Score::Natural;
		}
	}

	if(note == Score::D and score->keySignature() <= Score::KeySignatureAFlat){
		if(accidental == Score::Flat){
			accidental = Score::None;
		} else if(accidental == Score::None){
			accidental = Score::Natural;
		}
	}

	if(note == Score::G and score->keySignature() <= Score::KeySignatureDFlat){
		if(accidental == Score::Flat){
			accidental = Score::None;
		} else if(accidental == Score::None){
			accidental = Score::Natural;
		}
	}

	if(note == Score::C and score->keySignature() <= Score::KeySignatureGFlat){
		if(accidental == Score::Flat){
			accidental = Score::None;
		} else if(accidental == Score::None){
			accidental = Score::Natural;
		}
	}

	if(note == Score::F and score->keySignature() <= Score::KeySignatureCFlat){
		if(accidental == Score::Flat){
			accidental = Score::None;
		} else if(accidental == Score::None){
			accidental = Score::Natural;
		}
	}

	// There's a problem when reaching the last two notes of each key signature, the jumps are not homogenous anymore and need the follow workarround.
	// To ilustrate the problem with an example, when the key signature is C#, the set of notes are C#, D#, F (which is drawn as E#), F#, G#, D# and C (which is drawn as B#), that is, all sharpened notes because the key signature C# has # for all the notes.
	// What we have here is a little problem because the system won't draw C as B# but as C natural (with the natural accidental), so, we have to force the drawing of B# but the note, in the inside of the program, is still C natural.
	if(score->keySignature() >= Score::KeySignatureFSharp and note == Score::F and accidental == Score::Natural){
		note = Score::E;
		accidental = Score::None;
	}
	if(score->keySignature() >= Score::KeySignatureCSharp and note == Score::C and accidental == Score::Natural){
		note = Score::B;
		accidental = Score::None;
	}
	if(score->keySignature() <= Score::KeySignatureGFlat and note == Score::B and accidental == Score::Natural){
		note = Score::C;
		accidental = Score::None;
	}
	if(score->keySignature() <= Score::KeySignatureGFlat and note == Score::E and accidental == Score::Natural){
		note = Score::F;
		accidental = Score::None;
	}

	// Calculate the octave where the note will be
	int octave = static_cast<int>(randomNote / 12);
	if(randomNote < 0){
		octave--;
	}

	score->enqueueNote(Note(note, accidental, octave));
}

void ScoreReadingTrainerView::updateAvailableNotes(){
	// Clean the list before building it.
	availableNotes->clear();

	// Since the center of the limits (that is, the 0 of them) is on the note the clef marks, we have to do some clef compensation here
	int clefCompensation = 0;
	if(score->clef() == Score::ClefG2){
		clefCompensation += 7;
	} else if(score->clef() == Score::ClefF4){
		clefCompensation += 17;
	} else if(score->clef() == Score::ClefC3){
		clefCompensation += 12;
	} else if(score->clef() == Score::ClefC4){
		clefCompensation += 12;
	}

	for(int a = -mBottomLimit+clefCompensation; a <= mTopLimit+clefCompensation; ++a){
		if(mAccidentals){
			availableNotes->push_back(a);
		} else {
			int simplifiedA = ((a % 12) + 12) % 12;
			if(score->keySignature() == Score::KeySignatureC){
				if(simplifiedA == ScoreReadingTrainerView::C or
				   simplifiedA == ScoreReadingTrainerView::D or
				   simplifiedA == ScoreReadingTrainerView::E or
				   simplifiedA == ScoreReadingTrainerView::F or
				   simplifiedA == ScoreReadingTrainerView::G or
				   simplifiedA == ScoreReadingTrainerView::A or
				   simplifiedA == ScoreReadingTrainerView::B){
					availableNotes->push_back(a);
				}
			} else if(score->keySignature() == Score::KeySignatureG){
				if(simplifiedA == ScoreReadingTrainerView::C or
				   simplifiedA == ScoreReadingTrainerView::D or
				   simplifiedA == ScoreReadingTrainerView::E or
				   simplifiedA == ScoreReadingTrainerView::FSharp or
				   simplifiedA == ScoreReadingTrainerView::G or
				   simplifiedA == ScoreReadingTrainerView::A or
				   simplifiedA == ScoreReadingTrainerView::B){
					availableNotes->push_back(a);
				}
			} else if(score->keySignature() == Score::KeySignatureD){
				if(simplifiedA == ScoreReadingTrainerView::CSharp or
				   simplifiedA == ScoreReadingTrainerView::D or
				   simplifiedA == ScoreReadingTrainerView::E or
				   simplifiedA == ScoreReadingTrainerView::FSharp or
				   simplifiedA == ScoreReadingTrainerView::G or
				   simplifiedA == ScoreReadingTrainerView::A or
				   simplifiedA == ScoreReadingTrainerView::B){
					availableNotes->push_back(a);
				}
			} else if(score->keySignature() == Score::KeySignatureA){
				if(simplifiedA == ScoreReadingTrainerView::CSharp or
				   simplifiedA == ScoreReadingTrainerView::D or
				   simplifiedA == ScoreReadingTrainerView::E or
				   simplifiedA == ScoreReadingTrainerView::FSharp or
				   simplifiedA == ScoreReadingTrainerView::GSharp or
				   simplifiedA == ScoreReadingTrainerView::A or
				   simplifiedA == ScoreReadingTrainerView::B){
					availableNotes->push_back(a);
				}
			} else if(score->keySignature() == Score::KeySignatureE){
				if(simplifiedA == ScoreReadingTrainerView::CSharp or
				   simplifiedA == ScoreReadingTrainerView::DSharp or
				   simplifiedA == ScoreReadingTrainerView::E or
				   simplifiedA == ScoreReadingTrainerView::FSharp or
				   simplifiedA == ScoreReadingTrainerView::GSharp or
				   simplifiedA == ScoreReadingTrainerView::A or
				   simplifiedA == ScoreReadingTrainerView::B){
					availableNotes->push_back(a);
				}
			} else if(score->keySignature() == Score::KeySignatureB){
				if(simplifiedA == ScoreReadingTrainerView::CSharp or
				   simplifiedA == ScoreReadingTrainerView::DSharp or
				   simplifiedA == ScoreReadingTrainerView::E or
				   simplifiedA == ScoreReadingTrainerView::FSharp or
				   simplifiedA == ScoreReadingTrainerView::GSharp or
				   simplifiedA == ScoreReadingTrainerView::ASharp or
				   simplifiedA == ScoreReadingTrainerView::B){
					availableNotes->push_back(a);
				}
			} else if(score->keySignature() == Score::KeySignatureFSharp){
				if(simplifiedA == ScoreReadingTrainerView::CSharp or
				   simplifiedA == ScoreReadingTrainerView::DSharp or
				   simplifiedA == ScoreReadingTrainerView::F or
				   simplifiedA == ScoreReadingTrainerView::FSharp or
				   simplifiedA == ScoreReadingTrainerView::GSharp or
				   simplifiedA == ScoreReadingTrainerView::ASharp or
				   simplifiedA == ScoreReadingTrainerView::B){
					availableNotes->push_back(a);
				}
			} else if(score->keySignature() == Score::KeySignatureCSharp){
				if(simplifiedA == ScoreReadingTrainerView::CSharp or
				   simplifiedA == ScoreReadingTrainerView::DSharp or
				   simplifiedA == ScoreReadingTrainerView::F or
				   simplifiedA == ScoreReadingTrainerView::FSharp or
				   simplifiedA == ScoreReadingTrainerView::GSharp or
				   simplifiedA == ScoreReadingTrainerView::ASharp or
				   simplifiedA == ScoreReadingTrainerView::C){
					availableNotes->push_back(a);
				}
			} else if(score->keySignature() == Score::KeySignatureF){
				if(simplifiedA == ScoreReadingTrainerView::C or
				   simplifiedA == ScoreReadingTrainerView::D or
				   simplifiedA == ScoreReadingTrainerView::E or
				   simplifiedA == ScoreReadingTrainerView::F or
				   simplifiedA == ScoreReadingTrainerView::G or
				   simplifiedA == ScoreReadingTrainerView::A or
				   simplifiedA == ScoreReadingTrainerView::ASharp){
					availableNotes->push_back(a);
				}
			} else if(score->keySignature() == Score::KeySignatureBFlat){
				if(simplifiedA == ScoreReadingTrainerView::C or
				   simplifiedA == ScoreReadingTrainerView::D or
				   simplifiedA == ScoreReadingTrainerView::DSharp or
				   simplifiedA == ScoreReadingTrainerView::F or
				   simplifiedA == ScoreReadingTrainerView::G or
				   simplifiedA == ScoreReadingTrainerView::A or
				   simplifiedA == ScoreReadingTrainerView::ASharp){
					availableNotes->push_back(a);
				}
			} else if(score->keySignature() == Score::KeySignatureEFlat){
				if(simplifiedA == ScoreReadingTrainerView::C or
				   simplifiedA == ScoreReadingTrainerView::D or
				   simplifiedA == ScoreReadingTrainerView::DSharp or
				   simplifiedA == ScoreReadingTrainerView::F or
				   simplifiedA == ScoreReadingTrainerView::G or
				   simplifiedA == ScoreReadingTrainerView::GSharp or
				   simplifiedA == ScoreReadingTrainerView::ASharp){
					availableNotes->push_back(a);
				}
			} else if(score->keySignature() == Score::KeySignatureAFlat){
				if(simplifiedA == ScoreReadingTrainerView::C or
				   simplifiedA == ScoreReadingTrainerView::CSharp or
				   simplifiedA == ScoreReadingTrainerView::DSharp or
				   simplifiedA == ScoreReadingTrainerView::F or
				   simplifiedA == ScoreReadingTrainerView::G or
				   simplifiedA == ScoreReadingTrainerView::GSharp or
				   simplifiedA == ScoreReadingTrainerView::ASharp){
					availableNotes->push_back(a);
				}
			} else if(score->keySignature() == Score::KeySignatureDFlat){
				if(simplifiedA == ScoreReadingTrainerView::C or
				   simplifiedA == ScoreReadingTrainerView::CSharp or
				   simplifiedA == ScoreReadingTrainerView::DSharp or
				   simplifiedA == ScoreReadingTrainerView::F or
				   simplifiedA == ScoreReadingTrainerView::FSharp or
				   simplifiedA == ScoreReadingTrainerView::GSharp or
				   simplifiedA == ScoreReadingTrainerView::ASharp){
					availableNotes->push_back(a);
				}
			} else if(score->keySignature() == Score::KeySignatureGFlat){
				if(simplifiedA == ScoreReadingTrainerView::B or
				   simplifiedA == ScoreReadingTrainerView::CSharp or
				   simplifiedA == ScoreReadingTrainerView::DSharp or
				   simplifiedA == ScoreReadingTrainerView::F or
				   simplifiedA == ScoreReadingTrainerView::FSharp or
				   simplifiedA == ScoreReadingTrainerView::GSharp or
				   simplifiedA == ScoreReadingTrainerView::ASharp){
					availableNotes->push_back(a);
				}
			} else if(score->keySignature() == Score::KeySignatureCFlat){
				if(simplifiedA == ScoreReadingTrainerView::B or
				   simplifiedA == ScoreReadingTrainerView::CSharp or
				   simplifiedA == ScoreReadingTrainerView::DSharp or
				   simplifiedA == ScoreReadingTrainerView::E or
				   simplifiedA == ScoreReadingTrainerView::FSharp or
				   simplifiedA == ScoreReadingTrainerView::GSharp or
				   simplifiedA == ScoreReadingTrainerView::ASharp){
					availableNotes->push_back(a);
				}
			}
		}
	}

}

void ScoreReadingTrainerView::checkNote(int note){
	kdDebug() << "Running ScoreReadingTrainerView::checkNote(Notes note)" << endl;
	
	// Check if we are started.
	if(!started){
		// We are not running, so, don't do anything.
		return;
	}
	// If note is not a note... what are we doing ?
	if(note >= ScoreReadingTrainerView::C and note <= ScoreReadingTrainerView::B){
		// Get the note from the score
		Note sNote = score->note();

		// Compensate the note acording to the key signature
		if(score->keySignature() > Score::KeySignatureC){
			// Nothing is needed to be done for key signature C
			if(score->keySignature() >= Score::KeySignatureG and sNote.noteValue() == Score::F and sNote.accidental() == Score::None){
				//kdDebug() << "Compensationg F to F#" << endl;
				sNote.setAccidental(Score::Sharp);
			}
			if(score->keySignature() >= Score::KeySignatureD and sNote.noteValue() == Score::C and sNote.accidental() == Score::None){
				//kdDebug() << "Compensationg C to C#" << endl;
				sNote.setAccidental(Score::Sharp);
			}
			if(score->keySignature() >= Score::KeySignatureA and  sNote.noteValue() == Score::G and sNote.accidental() == Score::None){
				//kdDebug() << "Compensationg G to G#" << endl;
				sNote.setAccidental(Score::Sharp);
			}
			if(score->keySignature() >= Score::KeySignatureE and  sNote.noteValue() == Score::D and sNote.accidental() == Score::None){
				//kdDebug() << "Compensationg D to D#" << endl;
				sNote.setAccidental(Score::Sharp);
			}
			if(score->keySignature() >= Score::KeySignatureB and  sNote.noteValue() == Score::A and sNote.accidental() == Score::None){
				//kdDebug() << "Compensationg A to A#" << endl;
				sNote.setAccidental(Score::Sharp);
			}
			if(score->keySignature() >= Score::KeySignatureFSharp and  sNote.noteValue() == Score::E and sNote.accidental() == Score::None){
				//kdDebug() << "Compensationg E to E#" << endl;
				sNote.setAccidental(Score::Sharp);
			}
			if(score->keySignature() >= Score::KeySignatureCSharp and  sNote.noteValue() == Score::B and sNote.accidental() == Score::None){
				//kdDebug() << "Compensationg B to B#" << endl;
				sNote.setAccidental(Score::Sharp);
			}
		} else if(score->keySignature() < Score::KeySignatureC){
			if(score->keySignature() <= Score::KeySignatureF and  sNote.noteValue() == Score::B and sNote.accidental() == Score::None){
				sNote.setAccidental(Score::Flat);
			}
			if(score->keySignature() <= Score::KeySignatureBFlat and  sNote.noteValue() == Score::E and sNote.accidental() == Score::None){
				sNote.setAccidental(Score::Flat);
			}
			if(score->keySignature() <= Score::KeySignatureEFlat and  sNote.noteValue() == Score::A and sNote.accidental() == Score::None){
				sNote.setAccidental(Score::Flat);
			}
			if(score->keySignature() <= Score::KeySignatureAFlat and  sNote.noteValue() == Score::E and sNote.accidental() == Score::None){
				sNote.setAccidental(Score::Flat);
			}
			if(score->keySignature() <= Score::KeySignatureDFlat and  sNote.noteValue() == Score::G and sNote.accidental() == Score::None){
				sNote.setAccidental(Score::Flat);
			}
			if(score->keySignature() <= Score::KeySignatureGFlat and  sNote.noteValue() == Score::C and sNote.accidental() == Score::None){
				sNote.setAccidental(Score::Flat);
			}
			if(score->keySignature() <= Score::KeySignatureCFlat and  sNote.noteValue() == Score::F and sNote.accidental() == Score::None){
				sNote.setAccidental(Score::Flat);
			}
		}

		// Test for the right note and remove the note if right (also generate a new note)
		switch(note){
			case(ScoreReadingTrainerView::C):{
				if((sNote.noteValue() == Score::C and (sNote.accidental() == Score::None or sNote.accidental() == Score::Natural)) or
				   (sNote.noteValue() == Score::B and sNote.accidental() == Score::Sharp) or
				   (sNote.noteValue() == Score::D and sNote.accidental() == Score::DoubleFlat)){
					generateRandomNote(true);
				}
				break;
			}
			case(ScoreReadingTrainerView::CSharp):{
				if((sNote.noteValue() == Score::C and sNote.accidental() == Score::Sharp) or
				   (sNote.noteValue() == Score::B and sNote.accidental() == Score::DoubleSharp) or
				   (sNote.noteValue() == Score::D and sNote.accidental() == Score::Flat)){
					generateRandomNote(true);
				}
				break;
			}
			case(ScoreReadingTrainerView::D):{
				if((sNote.noteValue() == Score::D and (sNote.accidental() == Score::None or sNote.accidental() == Score::Natural)) or
				   (sNote.noteValue() == Score::C and sNote.accidental() == Score::DoubleSharp) or
				   (sNote.noteValue() == Score::E and sNote.accidental() == Score::DoubleFlat)){
					generateRandomNote(true);
				}
				break;
			}
			case(ScoreReadingTrainerView::DSharp):{
				if((sNote.noteValue() == Score::D and sNote.accidental() == Score::Sharp) or
				   (sNote.noteValue() == Score::E and sNote.accidental() == Score::Flat) or
				   (sNote.noteValue() == Score::F and sNote.accidental() == Score::DoubleFlat)){
					generateRandomNote(true);
				}
				break;
			}
			case(ScoreReadingTrainerView::E):{
				if((sNote.noteValue() == Score::E and (sNote.accidental() == Score::None or sNote.accidental() == Score::Natural)) or
				   (sNote.noteValue() == Score::D and sNote.accidental() == Score::DoubleSharp) or
				   (sNote.noteValue() == Score::F and sNote.accidental() == Score::Flat)){
					generateRandomNote(true);
				}
				break;
			}
			case(ScoreReadingTrainerView::F):{
				if((sNote.noteValue() == Score::F and (sNote.accidental() == Score::None or sNote.accidental() == Score::Natural)) or
				   (sNote.noteValue() == Score::E and sNote.accidental() == Score::Sharp) or
				   (sNote.noteValue() == Score::G and sNote.accidental() == Score::DoubleFlat)){
					generateRandomNote(true);
				}
				break;
			}
			case(ScoreReadingTrainerView::FSharp):{
				if((sNote.noteValue() == Score::F and sNote.accidental() == Score::Sharp) or
				   (sNote.noteValue() == Score::E and sNote.accidental() == Score::DoubleSharp) or
				   (sNote.noteValue() == Score::G and sNote.accidental() == Score::Flat)){
					generateRandomNote(true);
				}
				break;
			}
			case(ScoreReadingTrainerView::G):{
				if((sNote.noteValue() == Score::G and (sNote.accidental() == Score::None or sNote.accidental() == Score::Natural)) or
				   (sNote.noteValue() == Score::F and sNote.accidental() == Score::DoubleSharp) or
				   (sNote.noteValue() == Score::A and sNote.accidental() == Score::DoubleFlat)){
					generateRandomNote(true);
				}
				break;
			}
			case(ScoreReadingTrainerView::GSharp):{
				if((sNote.noteValue() == Score::G and sNote.accidental() == Score::Sharp) or
				   (sNote.noteValue() == Score::A and sNote.accidental() == Score::Flat)){
					generateRandomNote(true);
				}
				break;
			}
			case(ScoreReadingTrainerView::A):{
				if((sNote.noteValue() == Score::A and (sNote.accidental() == Score::None or sNote.accidental() == Score::Natural)) or
				   (sNote.noteValue() == Score::G and sNote.accidental() == Score::DoubleSharp) or
				   (sNote.noteValue() == Score::B and sNote.accidental() == Score::DoubleFlat)){
					generateRandomNote(true);
				}
				break;
			}
			case(ScoreReadingTrainerView::ASharp):{
				if((sNote.noteValue() == Score::A and sNote.accidental() == Score::Sharp) or
				   (sNote.noteValue() == Score::B and sNote.accidental() == Score::Flat) or
				   (sNote.noteValue() == Score::C and sNote.accidental() == Score::DoubleFlat)){
					generateRandomNote(true);
				}
				break;
			}
			case(ScoreReadingTrainerView::B):{
				if((sNote.noteValue() == Score::B and (sNote.accidental() == Score::None or sNote.accidental() == Score::Natural)) or
				   (sNote.noteValue() == Score::A and sNote.accidental() == Score::DoubleSharp) or
				   (sNote.noteValue() == Score::C and sNote.accidental() == Score::Flat)){
					generateRandomNote(true);
				}
				break;
			}
		}
	}

}


#include "ScoreReadingTrainerView.moc"
