#  Copyright (C) 1999-2012
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

proc ExportDef {} {
    global export

    set export(base) ds9
    set export(format) jpeg
    set export(jpeg,quality) 75
    set export(tiff,compress) none
}

proc ExportDialog {} {
    global export
    global exportfbox
    global ed
    global current

    set w {.export}

    set ed(ok) 0
    set ed(format) $export(format)

    DialogCreate $w [msgcat::mc {Export}] ed(ok)

    # Param
    set f [ttk::frame $w.param]
    ttk::label $f.title -text [msgcat::mc {Format}]
    ttk::radiobutton $f.gif -text {GIF} \
	-variable ed(format) -value gif
    ttk::radiobutton $f.jpeg -text {JPEG} \
	-variable ed(format) -value jpeg
    ttk::radiobutton $f.tiff -text {TIFF RGB 24-bit} \
	-variable ed(format) -value tiff
    ttk::radiobutton $f.png -text {PNG 24-bit (Portable Network Graphics)} \
	-variable ed(format) -value png
    ttk::radiobutton $f.ppm -text {PPM RGB (Portable Pixmap)} \
	-variable ed(format) -value ppm
    grid $f.gif -padx 2 -pady 2 -sticky w
    grid $f.jpeg -padx 2 -pady 2 -sticky w
    grid $f.tiff -padx 2 -pady 2 -sticky w
    grid $f.png -padx 2 -pady 2 -sticky w
    grid $f.ppm -padx 2 -pady 2 -sticky w

    # Buttons
    set f [ttk::frame $w.buttons]
    ttk::button $f.ok -text [msgcat::mc {OK}] -command {set ed(ok) 1} \
	-default active 
    ttk::button $f.cancel -text [msgcat::mc {Cancel}] -command {set ed(ok) 0}
    pack $f.ok $f.cancel -side left -expand true -padx 2 -pady 4

    bind $w <Return> {set ed(ok) 1}

    # Fini
    ttk::separator $w.sep -orient horizontal
    pack $w.buttons $w.sep -side bottom -fill x
    pack $w.param -side top -fill both -expand true

    DialogCenter $w 
    DialogWait $w ed(ok)
    DialogDismiss $w

    if {$ed(ok)} {
	set export(format) $ed(format)
	set exportfbox(types) [FormatToFBox $export(format)]
 	FileLast exportfbox "$export(base).$export(format)"
 	set fn [SaveFileDialog exportfbox]

	if {$fn != {}} {
 	    set export(base) [file rootname [file tail $fn]]

	    set ok 1
	    switch -- $export(format) {
		gif {}
		jpeg {set ok [SaveJPEGParams export(jpeg,quality)]}
		tiff {set ok [SaveTIFFParams export(tiff,compress)]}
		png {}
		ppm {}
	    }

	    if {$ok} {
		Export $fn
	    }
	}
    }

    set rr $ed(ok)
    unset ed
    return $rr
}

proc Export {fn} {
    global current
    global export

    if {$fn == {} || ![$current(frame) has fits]} {
	return
    }

    switch -- [$current(frame) get type] {
	base {ExportPhoto $fn}
	rgb -
	3d {}
    }
}

# Support

proc ExportPhoto {fn} {
    global ds9
    global export
    global current

    if {[catch {image create photo} ph]} {
	Error [msgcat::mc {An error has occurred while creating image.}]
	return
    }

    $current(frame) save photo $ph

    switch -- $export(format) {
	gif {$ph write $fn -format $export(format)}
	jpeg {$ph write $fn -format \
		  [list $export(format) -quality $export(jpeg,quality)]}
	tiff {$ph write $fn -format \
		  [list $export(format) -compression $export(tiff,compress)]}
	png {$ph write $fn -format $export(format)}
	ppm {$ph write $fn -format $export(format)}
    }

    image delete $ph
}

proc ProcessExportCmd {varname iname} {
    upvar $varname var
    upvar $iname i

    # we need to be realized
    ProcessRealizeDS9

    set format {}
    set fn [lindex $var $i]
    if {$fn == {}} {
	return
    }

    # backward compatibility
    switch $fn {
	gif -
	tif -
	tiff -
	jpg -
	jpeg -
	png -
	ppm {
	    set format $fn
	    set fn {}
	    incr i
	}
    }

    # one last time
    if {$fn == {}} {
	set fn [lindex $var $i]
	if {$fn == {}} {
	    return
	}
    }

    global export
    if {$format == {}} {
	set export(format) [ExtToFormat $fn]
    }

    set param [string tolower [lindex $var [expr $i+1]]]
    switch $export(format) {
	gif -
	png -
	ppm {}
	jpeg {
	    if {$param != {} && [string is integer $param]} {
		set export(jpeg,quality) $param
		incr i
	    }
	}
	tiff {
	    switch $param {
		none -
		jpeg -
		packbits -
		deflate {
		    set export(tiff,compress) $param
		    incr i
		}
	    }
	}
    }

    Export $fn
    FileLast exportfbox $fn
}
