// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

%{
#define YYPARSE_PARAM fr
#define YYDEBUG 1

#define FR ((FrameBase*)fr)
#define FITSPTR (FR->findFits(1))

#include <math.h>
#include <string.h>
#include <iostream.h>

#include "framebase.h"
#include "fitsimage.h"
#include "util.h"
#include "vector.h"
#include "list.h"
#include "basemarker.h"
#include "coord.h"

extern int xylex(void);
extern void xyerror(const char*);

int xysign;
int xysign2;

static CoordSystem globalSystem;
static SkyFrame globalSky;
static CoordSystem localSystem;
static SkyFrame localSky;

static List<Tag> taglist;
%}

%union {
  double real;
  int integer;
  char str[256];
  double vector[3];
}

%type <real> numeric

%type <real> sexagesimal
%type <vector> coord
%type <integer> coordSystem
%type <integer> skyFrame

%token <integer> INT
%token <real> REAL

%token <integer> HOUR
%token <integer> MINUTE
%token <real> SECOND
%token <real> DEGREE
%token <real> ARCMINUTE
%token <real> ARCSECOND

%token <str> SEXSTR

%token AMPLIFIER_
%token B1950_
%token CCD_
%token DEBUG_
%token DETECTOR_
%token ECLIPTIC_
%token FK4_
%token FK5_
%token GALACTIC_
%token ICRS_
%token IMAGE_
%token J2000_
%token LOGICAL_
%token OFF_
%token ON_
%token PHYSICAL_
%token VERSION_
%token WCS_
%token WCSA_
%token WCSB_
%token WCSC_
%token WCSD_
%token WCSE_
%token WCSF_
%token WCSG_
%token WCSH_
%token WCSI_
%token WCSJ_
%token WCSK_
%token WCSL_
%token WCSM_
%token WCSN_
%token WCSO_
%token WCSP_
%token WCSQ_
%token WCSR_
%token WCSS_
%token WCST_
%token WCSU_
%token WCSV_
%token WCSW_
%token WCSX_
%token WCSY_
%token WCSZ_

%%

start	: {
	    globalSystem = FR->getMarkerDefaultSystem(); 
	    globalSky = FR->getMarkerDefaultSky();
	  } commands
	;

commands: commands command terminator
	| command terminator
	;

command : /* empty */
	| DEBUG_ debug
	| VERSION_ {cerr << "X Y Format 1.0" << endl;}
	| coordSystem {globalSystem = (CoordSystem)$1;}
	| coordSystem skyFrame 
	  {globalSystem = (CoordSystem)$1; globalSky = (SkyFrame)$2;}
	| {localSystem = globalSystem; localSky = globalSky;} shape comment
	;

comment	: /* empty */
	| junks
	;

junks	: junks junk
	| junk
	;

junk	:  numeric {}
	;

terminator: '\n'
	| ';'
	;

numeric	: REAL {$$=$1;}
	| INT {$$=$1;}
	;

debug	: ON_ {yydebug=1;}
	| OFF_ {yydebug=0;}
	;

sp	: /* empty */
	| ','
	;

sexagesimal: SEXSTR {$$ = parseDMS($1);}
	;

coord	: sexagesimal sp sexagesimal
	{
	  Vector r;
	  if (localSky == GALACTIC || localSky == ECLIPTIC) 
	    r = FITSPTR->mapToRef(Vector($1,$3),localSystem,localSky);
	  else
	    r = FITSPTR->mapToRef(Vector($1*360./24.,$3),localSystem,localSky);

	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	| HOUR MINUTE SECOND {xysign2 = xysign;} sp INT ARCMINUTE ARCSECOND
        {
	  Vector r = FITSPTR->mapToRef(
	    Vector(hmsToDegree(xysign2,$1,$2,$3), 
	    dmsToDegree(xysign,$6,$7,$8)),
	    localSystem, localSky);
	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	| INT ARCMINUTE ARCSECOND {xysign2 = xysign;} sp 
	  INT ARCMINUTE ARCSECOND
        {
	  Vector r = FITSPTR->mapToRef(
	    Vector(dmsToDegree(xysign2,$1,$2,$3),
	    dmsToDegree(xysign,$6,$7,$8)),
	    localSystem, localSky);
	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	| numeric sp numeric 
	{
	  Vector r = FITSPTR->mapToRef(Vector($1,$3),localSystem,localSky);
	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	| DEGREE sp DEGREE
	{
	  Vector r = FITSPTR->mapToRef(Vector($1,$3),localSystem,localSky);
	  $$[0] = r[0];
	  $$[1] = r[1];
	  $$[2] = r[2];
	}
	;

coordSystem : IMAGE_ {$$ = IMAGE;}
	| LOGICAL_ {$$ = IMAGE;}
	| PHYSICAL_ {$$ = PHYSICAL;}
	| CCD_ {$$ = PHYSICAL;}
	| AMPLIFIER_ {$$ = AMPLIFIER;}
	| DETECTOR_ {$$ = DETECTOR;}
	| WCS_ {$$ = WCS;}
	| WCSA_ {$$ = WCSA;}
	| WCSB_ {$$ = WCSB;}
	| WCSC_ {$$ = WCSC;}
	| WCSD_ {$$ = WCSD;}
	| WCSE_ {$$ = WCSE;}
	| WCSF_ {$$ = WCSF;}
	| WCSG_ {$$ = WCSG;}
	| WCSH_ {$$ = WCSH;}
	| WCSI_ {$$ = WCSI;}
	| WCSJ_ {$$ = WCSJ;}
	| WCSK_ {$$ = WCSK;}
	| WCSL_ {$$ = WCSL;}
	| WCSM_ {$$ = WCSM;}
	| WCSN_ {$$ = WCSN;}
	| WCSO_ {$$ = WCSO;}
	| WCSP_ {$$ = WCSP;}
	| WCSQ_ {$$ = WCSQ;}
	| WCSR_ {$$ = WCSR;}
	| WCSS_ {$$ = WCSS;}
	| WCST_ {$$ = WCST;}
	| WCSU_ {$$ = WCSU;}
	| WCSV_ {$$ = WCSV;}
	| WCSW_ {$$ = WCSW;}
	| WCSX_ {$$ = WCSX;}
	| WCSY_ {$$ = WCSY;}
	| WCSZ_ {$$ = WCSZ;}
	;

skyFrame : FK4_ {$$ = FK4;}
	| B1950_  {$$ = FK4;}
	| FK5_ {$$ = FK5;}
	| J2000_  {$$ = FK5;}	
	| ICRS_ {$$ = ICRS;}
	| GALACTIC_ {$$ = GALACTIC;}
	| ECLIPTIC_ {$$ = ECLIPTIC;}
	;

shape	: coord {FR->createBoxCirclePointCmd(Vector($1),
	    "green",1,"helvetica 10 normal","",
	    Marker::SELECT | Marker::EDIT | Marker::MOVE | Marker::ROTATE | 
	    Marker::DELETE | Marker::HIGHLITE | 
	    Marker::INCLUDE | Marker::SOURCE,
	    "",taglist);
	}
	;

%%
