// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "polygon.h"
#include "framebase.h"
#include "fitsimage.h"
#include "util.h"

Polygon::Polygon(const Polygon& a) : Marker(a)
{
  flip = a.flip;
  vertex = a.vertex;
}

Polygon::Polygon(FrameBase* p, const Vector& ctr, const Vector& b,
		 const char* clr, int w, const char* f,
		 const char* t, unsigned short prop, const char* c,
		 const List<Tag>& tag)
  : Marker(p, ctr, clr, w, f, t, prop, c, tag)
{
  angle = 0;
  strcpy(type, "polygon");

  Vector bb = b;
  vertex.append(new Vertex(-bb[0],-bb[1]));
  vertex.append(new Vertex( bb[0],-bb[1]));
  vertex.append(new Vertex( bb[0], bb[1]));
  vertex.append(new Vertex(-bb[0], bb[1]));

  if (!parent->isIIS())
    flip = FlipY();

  updateBBox();
}

Polygon::Polygon(FrameBase* p, const List<Vertex>& v, 
		 const char* clr, int w, const char* f,
		 const char* t, unsigned short prop, const char* c,
		 const List<Tag>& tag)
  : Marker(p, Vector(0,0), clr, w, f, t, prop, c, tag)
{
  // Vertex list is in ref coords

  angle = 0;
  strcpy(type, "polygon");
  vertex = v;

  // check to see if the first and last node are the same

  if (vertex.head()->vector[0] == vertex.tail()->vector[0] &&
      vertex.head()->vector[1] == vertex.tail()->vector[1])
    delete vertex.pop();

  // find center

  center = Vector(0,0);
  vertex.head();
  do
    center += vertex.current()->vector;
  while (vertex.next());
  center /= vertex.count();

  if (!parent->isIIS())
    flip = FlipY();

  // vertices are relative

  vertex.head();
  do
    vertex.current()->vector *= Translate(-center) * flip; // no rotation
  while (vertex.next());

  updateBBox();
}

void Polygon::updateBBox()
{
  // generate handles

  numHandle = 4 + vertex.count();
  if (handle)
    delete [] handle;
  handle = new Vector[numHandle];

  Matrix m = Rotate(angle) * flip * Translate(center) * parent->refToCanvas;

  // the first four are our control handles

  BBox bb;
  vertex.head();
  do
    bb.bound(vertex.current()->vector);
  while (vertex.next());

  Vector r = Vector(8,0)/parent->getZoom();
  bb.expand(r[0]); // give us more room

  handle[0] = bb.ll   * m;
  handle[1] = bb.lr() * m;
  handle[2] = bb.ur   * m;
  handle[3] = bb.ul() * m;

  // and the rest are vertices

  int i=4;
  vertex.head();
  do
    handle[i++] = vertex.current()->vector * m;
  while (vertex.next());

  // now, calc bbox

  Vector c = center * parent->refToCanvas;
  bbox = BBox(c,c);
  {
    for (int i=0; i<numHandle; i++)
      bbox.bound(handle[i]);
  }
  bbox.expand(3); // give us room for control handles

  // calculate overall bbox

  calcAllBBox();
}

void Polygon::updateCoords(const Matrix& m)
{
  Scale s(m);
  vertex.head();
  do
    vertex.current()->vector *= s;
  while (vertex.next());
  
  Marker::updateCoords(m);
}

void Polygon::edit(const Vector& v, int h)
{
  if (h < 5) {
    Vector s1 = v * Translate(-center) * flip * Rotate(-angle);
    Vector s2 = handle[h-1] * parent->canvasToRef * 
      Translate(-center) * flip * Rotate(-angle);

    if (s1[0] != 0 && s1[1] != 0 && s2[0] != 0 && s2[1] != 0) {
      double a = fabs(s1[0]/s2[0]);
      double b = fabs(s1[1]/s2[1]);
      double s = a > b ? a : b;

      vertex.head();
      do
	vertex.current()->vector *= Scale(s);
      while (vertex.next());
    }

    updateBBox();
    doCallBack(&editCB);
  }
  else {
    moveVertex(v,h);

    updateBBox();
    doCallBack(&editCB);
    doCallBack(&moveCB); // center can change
  }
}

void Polygon::rotate(const Vector& v, int h)
{
  if (h < 5)
    Marker::rotate(v,h);
  else {
    // we need to check this here, because we are really rotating
    if (canEdit()) { 
      moveVertex(v,h);

      updateBBox();
      doCallBack(&editCB);
      doCallBack(&moveCB); // center can change
    }
  }
}

void Polygon::reset(const Vector& b)
{
  angle = 0;
  vertex.deleteAll();

  Vector bb = b;
  vertex.append(new Vertex(-bb[0],-bb[1]));
  vertex.append(new Vertex( bb[0],-bb[1]));
  vertex.append(new Vertex( bb[0], bb[1]));
  vertex.append(new Vertex(-bb[0], bb[1]));

  if (!parent->isIIS())
    flip = FlipY();

  updateBBox();
}

#if __GNUC__ >= 3
void Polygon::ps(int mode)
{
  Marker::ps(mode);

  Matrix m = Rotate(angle) * flip * Translate(center) * parent->refToCanvas;

  char buf[256];
  vertex.head();
  int first = 1;
  do {
    ostringstream str;
    Vector v =  vertex.current()->vector * m;
    if (first) {
      str << "newpath " << endl
	  << v.TkCanvasPs(parent->canvas) << " moveto" << endl << ends;
      first = 0;
    }
    else
      str << v.TkCanvasPs(parent->canvas) << " lineto" << endl << ends;

    Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
  } while (vertex.next());

  ostringstream str;
  str << "closepath stroke" << endl << ends;
  Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);

  // remember, bbox is in canvas coords
  if (!(properties & INCLUDE)) {
    psColor(mode, "red");

    ostringstream str;
    str << "newpath " 
	<< bbox.ll.TkCanvasPs(parent->canvas) << "moveto"
	<< bbox.ur.TkCanvasPs(parent->canvas) << "lineto"
	<< " stroke" << endl << ends;
    Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
  }
}
#else
void Polygon::ps(int mode)
{
  Marker::ps(mode);

  Matrix m = Rotate(angle) * flip * Translate(center) * parent->refToCanvas;

  char buf[256];
  vertex.head();
  int first = 1;
  do {
    ostrstream str(buf,256);
    Vector v =  vertex.current()->vector * m;
    if (first) {
      str << "newpath " << endl
	  << v.TkCanvasPs(parent->canvas) << " moveto" << endl << ends;
      first = 0;
    }
    else
      str << v.TkCanvasPs(parent->canvas) << " lineto" << endl << ends;

    Tcl_AppendResult(parent->interp, buf, NULL);
  } while (vertex.next());

  ostrstream str(buf,256);
  str << "closepath stroke" << endl << ends;
  Tcl_AppendResult(parent->interp, buf, NULL);

  // remember, bbox is in canvas coords
  if (!(properties & INCLUDE)) {
    psColor(mode, "red");

    ostrstream str(buf,256);
    str << "newpath " 
	<< bbox.ll.TkCanvasPs(parent->canvas) << "moveto"
	<< bbox.ur.TkCanvasPs(parent->canvas) << "lineto"
	<< " stroke" << endl << ends;
    Tcl_AppendResult(parent->interp, buf, NULL);
  }
}
#endif

int Polygon::isIn(const Vector& vv)
{
  /*
    v[0]-- x value of point being tested
    v[1]-- y value of point being tested

    This algorithm is from "An Introduction to Ray Tracing", Academic Press,
    1989, edited by Andrew Glassner, pg 53
    -- a point lies in a polygon if a line is extended from the point to 
    infinite in any direction and the number of intersections with the 
    polygon is odd.
    This is valid for both concave and convex polygons.
    Points on a vertex are considered inside.
    Points on a edge are considered inside.
  */

  Vector v = vv * parent->canvasToRef; // xform to ref coords
  int crossings = 0;   // number of crossings

  vertex.head();
  Vector v1;
  Vector v2 = 
    (vertex.current()->vector * Rotate(angle) * flip * Translate(center)) - v;

  int sign = ((v2[1])>=0) ? 1 : -1; // init sign

  // for all edges

  int done = 0;

  do {
    // look at next two vertices
    v1 = v2;
    if (!vertex.next()) {
      done = 1;
      vertex.head();
    }

    v2 = (vertex.current()->vector*Rotate(angle)*flip*Translate(center)) - v;

    int nextSign = (v2[1]>=0) ? 1 : -1; // sign holder for p2

    if (sign != nextSign) {
      if (v1[0]>0 && v2[0]>0)
	crossings++;
      else if (v1[0]>0 || v2[0]>0) {
	if (v1[0]-(v1[1]*(v2[0]-v1[0])/(v2[1]-v1[1])) > 0)
	  crossings++;
      }
      sign = nextSign;
    }
  } while (!done);

  return fmod(float(crossings),float(2)) ? 1 : 0; // if odd, point is inside
}

void Polygon::createVertex(int which, const Vector& v)
{
  // which segment (1 to n)
  // v is in ref coords

  int seg = which-1;
  if (seg>=0 && seg<vertex.count()) {
    Vertex* n = new Vertex(v * Translate(-center) * flip * Rotate(-angle));
    vertex.insert(seg,n);

    recalcCenter();

    updateBBox();
    doCallBack(&editCB);
    doCallBack(&moveCB); // center can change
  }
}

void Polygon::deleteVertex(int h)
{
  if (h>4) {
    int hh = h-4-1;

    if (vertex.count() > 3) {
      Vertex* v = vertex[hh];
      if (v) {
	vertex.extractNext(v);
	delete v;

	recalcCenter();

	updateBBox();
	doCallBack(&editCB);
	doCallBack(&moveCB); // center can change
      }  
    }
  }
}

int Polygon::getSegment(const Vector& v)
{
  // v is in canvas coords

  Matrix m = Rotate(angle) * flip * Translate(center);

  vertex.head();
  Vector v1;
  Vector v2 = vertex.current()->vector * m;
  int done = 0;
  int i = 1;
  do {
    v1 = v2;
    if (!vertex.next()) {
      vertex.head();
      done = 1;
    }
    v2 = vertex.current()->vector * m;

    Vector l1 = v1*parent->refToCanvas;
    Vector l2 = v2*parent->refToCanvas;
    double a = parent->isIIS() ? -(l2-l1).angle() : (l2-l1).angle();
    Matrix mm = Translate(-l1) * flip * Rotate(-a); 
    Vector end = l2*mm;
    Vector vv = v*mm;
    
    if (vv[0]>0 && vv[0]<end[0] && vv[1]>-3 && vv[1]<3)
      return i;

    i++;
  } while (!done);

  return 0;
}

// private

void Polygon::render(Drawable drawable, const Matrix& matrix, double zoom, 
		     RenderMode mode)
{
  setGC(mode);

  Matrix m = Rotate(angle) * flip * Translate(center) * matrix;

  vertex.head();
  Vector v1;
  Vector v2 = (vertex.current()->vector * m).round();
  int done = 0;

  do {
    if (!vertex.next()) {
      done = 1;
      vertex.head();
    }
    v1 = v2;
    v2 = (vertex.current()->vector * m).round();
    XDRAWLINE(display, drawable, gc, (int)v1[0], (int)v1[1], 
	      (int)v2[0], (int)v2[1]);
  } while (!done);

  // remember, bbox is in canvas coords
  if (!(properties & INCLUDE)) {
    BBox b(bbox);
    b.shrink(3);

    Vector ll = (b.ll * parent->getCanvasToRef() * matrix).round();
    Vector ur = (b.ur * parent->getCanvasToRef() * matrix).round();

    if (mode==SRC)
      XSetForeground(display, gc, parent->getRedColor());

    XDRAWLINE(display, drawable, gc, (int)ll[0], (int)ll[1], 
	      (int)ur[0], (int)ur[1]);    
  }
}

void Polygon::moveVertex(const Vector& v, int h)
{
  if (vertex[h-5])
    vertex.current()->vector = v * Translate(-center) * flip * Rotate(-angle);

  recalcCenter();
}

void Polygon::recalcCenter()
{
  // recalculate center

  Vector nc;
  vertex.head();
  do
    nc += vertex.current()->vector * Rotate(angle) * flip;
  while (vertex.next());
  nc /= vertex.count();

  center += nc;

  // update all vertices

  vertex.head();
  do
    vertex.current()->vector -= nc * flip * Rotate(-angle);
  while (vertex.next());
}

// list

void Polygon::list(ostream& str, CoordSystem sys, SkyFrame sky,
		   SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(center);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      listPre(str,sys,sky,ptr);

      str << type << '(';
      int first=1;
      vertex.head();
      do {
	if (!first)
	  str << ',';
	first=0;

	Vector w = vertex.current()->vector * 
	  Rotate(angle) * flip * Translate(center);
	Vector v = ptr->mapFromRef(w,sys);
	str << setprecision(8) << v[0] << ',' << v[1];
      }
      while (vertex.next());
      str << ')';

      listPost(str,delim);
    }
    break;
  default:
    if (ptr->hasWCS(sys)) {
      listPre(str,sys,sky,ptr);

      if (ptr->hasWCSEqu(sys)) {
	switch (format) {
	case DEGREES:
	  {
	    str << type << '(';
	    int first=1;
	    vertex.head();
	    do {
	      if (!first)
		str << ',';
	      first=0;

	      Vector w = vertex.current()->vector * 
		Rotate(angle) * flip * Translate(center);
	      Vector v = ptr->mapFromRef(w,sys,sky);
	      str << setprecision(8) << v[0] << ',' << v[1];
	    }
	    while (vertex.next());
	    str << ')';
	  }
	  break;
	case SEXAGESIMAL:
	  {
	    char buf[64];
	    char ra[16];
	    char dec[16];

	    str << type << '(';
	    int first=1;
	    vertex.head();
	    do {
	      if (!first)
		str << ',';
	      first=0;

	      Vector w = vertex.current()->vector * 
		Rotate(angle) * flip * Translate(center);
	      ptr->mapFromRef(w,sys,sky,format,buf,64);
#if __GNUC__ >= 3
	      string x(buf);
	      istringstream wcs(x);
#else
	      istrstream wcs(buf,64);
#endif
	      wcs >> ra >> dec;
	      str << ra << ',' << dec;
	    }
	    while (vertex.next());
	    str << ')';
	  }
	  break;
	}
      }
      else {
	str << type << '(';
	int first=1;
	vertex.head();
	do {
	  if (!first)
	    str << ',';
	  first=0;

	  Vector w = vertex.current()->vector * 
	    Rotate(angle) * flip * Translate(center);
	  Vector v = ptr->mapFromRef(w,sys);
	  str << setprecision(8) << v[0] << ',' << v[1];
	}
	while (vertex.next());
	str << ')';
      }

      listPost(str,delim);
    }
    else
      str << "";
    break;
  }
}

void Polygon::listCiao(ostream& str, CoordSystem sys, SkyFrame sky, 
		       SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(1);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      listCiaoPre(str);

      str << type << '(';
      int first=1;
      vertex.head();
      do {
	if (!first)
	  str << ',';
	first=0;

	Vector w = vertex.current()->vector * 
	  Rotate(angle) * flip * Translate(center);
	Vector v = ptr->mapFromRef(w,PHYSICAL);
	str << setprecision(8) << v[0] << ',' << v[1];
      }
      while (vertex.next());
      str << ')' << delim;
    }
    break;
  default:
    if (ptr->hasWCSEqu(sys)) {
      listCiaoPre(str);

      char buf[64];
      char ra[16];
      char dec[16];

      str << type << '(';
      int first=1;
      vertex.head();
      do {
	if (!first)
	  str << ',';
	first=0;

	Vector w = vertex.current()->vector * 
	  Rotate(angle) * flip * Translate(center);
	ptr->mapFromRef(w,sys,FK5,SEXAGESIMAL,buf,64);
#if __GNUC__ >= 3
	string x(buf);
	istringstream wcs(x);
#else
	istrstream wcs(buf,64);
#endif
	wcs >> ra >> dec;
	str << ra << ',' << dec;
      }
      while (vertex.next());
      str << ')' << delim;
    }
    break;
  }
}

void Polygon::listPros(ostream& str, CoordSystem sys, SkyFrame sky,
		       SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(1);

  switch (sys) {
  case IMAGE:
  case DETECTOR:
  case AMPLIFIER:
    sys = IMAGE;
  case PHYSICAL:
    {
      listProsCoordSystem(str,sys,sky);
      str << "; " << type;
      vertex.head();
      do {
	Vector w = vertex.current()->vector * 
	  Rotate(angle) * flip * Translate(center);
	Vector v = ptr->mapFromRef(w,sys);
	str << setprecision(8) << v;
      }
      while (vertex.next());

      str << delim;
    }
    break;
  default:
    if (ptr->hasWCSEqu(sys)) {
      listProsCoordSystem(str,sys,sky);
      str << "; " << type << ' ';

      switch (format) {
      case DEGREES:
	{
	  vertex.head();
	  do {
	    Vector w = vertex.current()->vector * 
	      Rotate(angle) * flip * Translate(center);
	    Vector v = ptr->mapFromRef(w,sys,sky);
	    str << setprecision(8) << v[0] << "d " << v[1] << "d ";
	  }
	  while (vertex.next());
	}
	break;
      case SEXAGESIMAL:
	{
	  char buf[64];
	  char ra[16];
	  char dec[16];

	  vertex.head();
	  do {
	    Vector w = vertex.current()->vector *
	      Rotate(angle) * flip * Translate(center);
	    ptr->mapFromRef(w,sys,sky,format,buf,64);
#if __GNUC__ >= 3
	string x(buf);
	istringstream wcs(x);
#else
	istrstream wcs(buf,64);
#endif
	    wcs >> ra >> dec;
	    if (dec[0]=='+')
	      str << ' ' << ra << ' ' << dec+1;
	    else
	      str << ' ' << ra << ' ' << dec;
	  }
	  while (vertex.next());
	}
	break;
      }
      str << delim;
    }
    else
      str << "";
    break;
  }
}

void Polygon::listSAOtng(ostream& str, CoordSystem sys, SkyFrame sky,
			 SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(1);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      listSAOtngPre(str,delim);

      str << type << '(';
      int first=1;
      vertex.head();
      do {
	if (!first)
	  str << ',';
	first=0;

	Vector w = vertex.current()->vector * 
	  Rotate(angle) * flip * Translate(center);
	Vector v = ptr->mapFromRef(w,sys);
	str << setprecision(8) << v[0] << ',' << v[1];
      }
      while (vertex.next());
      str << ')';

      listSAOtngPost(str,delim);
    }
    break;
  default:
    if (ptr->hasWCSEqu(sys)) {
      listSAOtngPre(str,delim);

      switch (format) {
      case DEGREES:
	{
	  str << type << '(';
	  int first=1;
	  vertex.head();
	  do {
	    if (!first)
	      str << ',';
	    first=0;

	    Vector w = vertex.current()->vector * 
	      Rotate(angle) * flip * Translate(center);
	    Vector v = ptr->mapFromRef(w,sys,sky);
	    str << setprecision(8) << v[0] << ',' << v[1];
	  }
	  while (vertex.next());
	  str << ')';
	}
	break;
      case SEXAGESIMAL:
	{
	  char buf[64];
	  char ra[16];
	  char dec[16];

	  str << type << '(';
	  int first=1;
	  vertex.head();
	  do {
	    if (!first)
	      str << ',';
	    first=0;

	    Vector w = vertex.current()->vector * 
	      Rotate(angle) * flip * Translate(center);
	    ptr->mapFromRef(w,sys,sky,format,buf,64);
#if __GNUC__ >= 3
	    string x(buf);
	    istringstream wcs(x);
#else
	    istrstream wcs(buf,64);
#endif
	    wcs >> ra >> dec;
	    str << ra << ',' << dec;
	  }
	  while (vertex.next());
	  str << ')';
	}
	break;
      }

      listSAOtngPost(str,delim);
    }
    else
      str << "";
      break;
  }
}

void Polygon::listSAOimage(ostream& str, CoordSystem sys, SkyFrame sky,
			   SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(1);

  if (!(properties&INCLUDE))
    str << '-';

  str << type << '(';
  int first=1;
  vertex.head();
  do {
    if (!first)
      str << ',';
    first=0;

    Vector w = vertex.current()->vector * 
      Rotate(angle) * flip * Translate(center);
    Vector v = ptr->mapFromRef(w,IMAGE);
    str << setprecision(8) << v[0] << ',' << v[1];
  }
  while (vertex.next());
  str << ')';

  str << delim;
}
