#  Copyright (C) 1999-2004
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

proc ChangeDATASEC {} {
    global current
    global scale
    global rgb

    if {$current(frame) != ""} {
	SetWatchCursor
	RGBEvalLock rgb(lock,scale) "$current(frame) datasec $scale(datasec)"
	UpdateScaleDialog
	UpdateGraphYAxis
	UnsetWatchCursor
    }
}

proc ChangeScale {} {
    global current
    global scale
    global rgb

    if {$current(frame) != ""} {
	SetWatchCursor
	RGBEvalLock rgb(lock,scale) "$current(frame) colorscale $scale(type)"
	UnsetWatchCursor
    }
}

proc ChangeScaleMode {} {
    global current
    global scale
    global rgb

    if {$current(frame) != ""} {
	SetWatchCursor
	RGBEvalLock rgb(lock,scale) "$current(frame) clip mode $scale(mode)"
	UpdateScaleDialog
	UpdateGraphYAxis
	UnsetWatchCursor
    }
}

proc ChangeScaleLimit {} {
    global current
    global scale

    if {$current(frame) != ""} {
	SetWatchCursor
	set scale(mode) user
	$current(frame) clip user $scale(usermin) $scale(usermax)
	$current(frame) clip mode $scale(mode)
	UpdateScaleDialog
	UpdateGraphYAxis
	UnsetWatchCursor
    }
}

proc ChangeScaleScope {} {
    global current
    global scale
    global rgb

    if {$current(frame) != ""} {
	SetWatchCursor
	RGBEvalLock rgb(lock,scale) "$current(frame) clip scope $scale(scope)"
	UpdateGraphYAxis
	UnsetWatchCursor
    }
}

proc PreserveScale {} {
    global current
    global scale
    global rgb

    if {$current(frame) != {}} {
	SetWatchCursor
	RGBEvalLock rgb(lock,scale) \
	    "$current(frame) clip preserve $scale(preserve)"
	UpdateGraphYAxis
	UnsetWatchCursor
    }
}

proc ChangeMinMaxMode {} {
    global current
    global minmax
    global rgb

    if {$current(frame) != ""} {
	SetWatchCursor
	RGBEvalLock rgb(lock,scale) "$current(frame) clip minmax mode $minmax(mode)"
	UpdateScaleDialog
	UpdateGraphYAxis
	UnsetWatchCursor
    }
}

proc ChangeMinMaxSample {} {
    global current
    global minmax
    global rgb

    if {$current(frame) != ""} {
	SetWatchCursor
	RGBEvalLock rgb(lock,scale) "$current(frame) clip minmax sample $minmax(sample)"
	UpdateScaleDialog
	UpdateGraphYAxis
	UnsetWatchCursor
    }
}

proc UpdateScaleMenu {} {
    global current
    global scale
    global buttons
    global ds9
    global menu

    global debug
    if {$debug(tcl,update)} {
	puts "UpdateScaleMenu"
    }

    if {$current(frame) != ""} {
	if {![$current(frame) has iis]} {
	    $ds9(mb) entryconfig $menu(scale) -state normal
	    $buttons(name).major.scale configure -state normal

	    set scale(type) [$current(frame) get colorscale]
	    set scale(scope) [$current(frame) get clip scope]
	    set scale(mode) [$current(frame) get clip mode]
	    set scale(datasec) [$current(frame) get datasec]
	    set scale(preserve) [$current(frame) get clip preserve]
	    set minmax(mode) [$current(frame) get clip minmax mode]
	    set minmax(sample) [$current(frame) get clip minmax sample]

	    if {[$current(frame) has fits]} {
		if {[$current(frame) has fits mosaic] || \
			[$current(frame) has fits cube]} {
		    $ds9(mb).scale entryconfig "Scope" -state normal
		} else {
		    $ds9(mb).scale entryconfig "Scope" -state disabled
		}

		if [$current(frame) has datasec] {
		    $ds9(mb).scale entryconfig "use DATASEC" -state normal
		} else {
		    $ds9(mb).scale entryconfig "use DATASEC" -state disabled
		}

		if [$current(frame) has datamin] {
		    $ds9(mb).scale.minmax \
			entryconfig "DATAMIN DATAMAX" -state normal
		} else {
		    $ds9(mb).scale.minmax \
			entryconfig "DATAMIN DATAMAX" -state disabled
		}

		if [$current(frame) has irafmin] {
		    $ds9(mb).scale.minmax \
			entryconfig "IRAF-MIN IRAF-MAX" -state normal
		} else {
		    $ds9(mb).scale.minmax \
			entryconfig "IRAF-MIN IRAF-MAX" -state disabled
		}
	    } else {
		$ds9(mb).scale entryconfig "Scope" -state normal
		$ds9(mb).scale entryconfig "use DATASEC" -state normal
		$ds9(mb).scale.minmax entryconfig "DATAMIN DATAMAX" \
		    -state normal
		$ds9(mb).scale.minmax entryconfig "IRAF-MIN IRAF-MAX" \
		    -state normal
	    }

	    return
	}
    }

    # default
    $ds9(mb) entryconfig $menu(scale) -state disabled
    $buttons(name).major.scale configure -state disabled
}

proc ScaleDialog {} {
    global scale
    global scaledlg
    global current
    global graph
    global menu
    global ds9
    global minmax

    # see if we already have a window visible

    if [winfo exist $scale(top)] {
	raise $scale(top)
	return
    }

    set w $scale(top)
    set title Scale

    # create the window

    toplevel $w -colormap $ds9(main)
    wm title $w $title
    wm iconname $w $title
    wm group $w $ds9(top)
    wm protocol $w WM_DELETE_WINDOW ScaleDestroyDialog

    $w configure -menu $scale(mb)

    menu $scale(mb) -tearoff 0
    $scale(mb) add cascade -label File -menu $scale(mb).file
    $scale(mb) add cascade -label Scale -menu $scale(mb).scale
    $scale(mb) add cascade -label Limits -menu $scale(mb).limit
    $scale(mb) add cascade -label Scope -menu $scale(mb).scope
    $scale(mb) add cascade -label MinMax -menu $scale(mb).minmax
    $scale(mb) add cascade -label Parameters -menu $scale(mb).param
    $scale(mb) add cascade -label Graph -menu $scale(mb).graph

    menu $scale(mb).file -tearoff 0 -selectcolor $menu(selectcolor)
    $scale(mb).file add command -label "Apply" -command ScaleApplyDialog
    $scale(mb).file add separator
    $scale(mb).file add command -label "Close" -command ScaleDestroyDialog

    menu $scale(mb).scale -tearoff 0 -selectcolor $menu(selectcolor)
    $scale(mb).scale add radiobutton -label "Linear" \
	-variable scale(type) -command ChangeScale -value linear
    $scale(mb).scale add radiobutton -label "Log" \
	-variable scale(type) -command ChangeScale -value log
    $scale(mb).scale add radiobutton -label "Squared" \
	-variable scale(type) -command ChangeScale -value squared
    $scale(mb).scale add radiobutton -label "Square Root" \
	-variable scale(type) -command ChangeScale -value sqrt
    $scale(mb).scale add radiobutton -label "Histogram Equalization" \
	-variable scale(type) -command ChangeScale -value histequ

    menu $scale(mb).limit -tearoff 0 -selectcolor $menu(selectcolor)
    $scale(mb).limit add radiobutton -label MinMax \
	-variable scale(mode) -command ChangeScaleMode -value minmax
    $scale(mb).limit add separator
    $scale(mb).limit add radiobutton -label "99.5%" \
	-variable scale(mode) -command ChangeScaleMode -value 99.5
    $scale(mb).limit add radiobutton -label "99%" \
	-variable scale(mode) -command ChangeScaleMode -value 99
    $scale(mb).limit add radiobutton -label "98%" \
	-variable scale(mode) -command ChangeScaleMode -value 98
    $scale(mb).limit add radiobutton -label "95%" \
	-variable scale(mode) -command ChangeScaleMode -value 95
    $scale(mb).limit add radiobutton -label "90%" \
	-variable scale(mode) -command ChangeScaleMode -value 90
    $scale(mb).limit add separator
    $scale(mb).limit add radiobutton -label "ZScale" \
	-variable scale(mode) -command ChangeScaleMode -value zscale
    $scale(mb).limit add radiobutton -label "ZMax" \
	-variable scale(mode) -command ChangeScaleMode -value zmax

    menu $scale(mb).scope -tearoff 0 -selectcolor $menu(selectcolor)
    $scale(mb).scope add radiobutton -label "Global" \
	-variable scale(scope) -command ChangeScaleScope -value global
    $scale(mb).scope add radiobutton -label "Local" \
	-variable scale(scope) -command ChangeScaleScope -value local

    menu $scale(mb).minmax -tearoff 0 -selectcolor $menu(selectcolor)
    $scale(mb).minmax add radiobutton -label "Auto" \
	-variable minmax(mode) -value auto -command ChangeMinMaxMode
    $scale(mb).minmax add radiobutton -label "Scan Data" \
	-variable minmax(mode) -value scan -command ChangeMinMaxMode
    $scale(mb).minmax add radiobutton -label "Sample Data" \
	-variable minmax(mode) -value sample -command ChangeMinMaxMode
    $scale(mb).minmax add radiobutton -label "DATAMIN DATAMAX" \
	-variable minmax(mode) -value datamin -command ChangeMinMaxMode
    $scale(mb).minmax add radiobutton -label "IRAF-MIN IRAF-MAX" \
	-variable minmax(mode) -value irafmin -command ChangeMinMaxMode
    $scale(mb).minmax add separator
    $scale(mb).minmax add command -label "Sample Increment..." \
	-command "MinMaxSampleDialog minmax(sample) 1"

    menu $scale(mb).param -tearoff 0 -selectcolor $menu(selectcolor)
    $scale(mb).param add checkbutton -label "use DATASEC" \
	-variable scale(datasec) -command ChangeDATASEC
    $scale(mb).param add separator
    $scale(mb).param add command -label "ZScale..." \
	-command ZScaleDialog

    menu $scale(mb).graph -tearoff 0 -selectcolor $menu(selectcolor)
    $scale(mb).graph add radiobutton -label Linear -value linear \
	-variable scale(yaxis) -command ScaleYAxisDialog
    $scale(mb).graph add radiobutton -label Log -value log \
	-variable scale(yaxis) -command ScaleYAxisDialog
    $scale(mb).graph add separator
    $scale(mb).graph add radiobutton -label "Full Range" -value full \
	-variable scale(xaxis) -command ScaleXAxisDialog
    $scale(mb).graph add radiobutton -label "Current Range" -value current \
	-variable scale(xaxis) -command ScaleXAxisDialog

    frame $w.top  -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.top $w.buttons -ipadx 4 -ipady 4 -fill x -expand true

    # graph
    
    set bgcolor [$w.top cget -background]
    set scale(hist) [blt::barchart $w.top.chart \
			 -width 400 -height 200 -title \
			 "Pixel Distribution" -plotrelief groove \
			 -plotbackground $bgcolor -plotpadx 0 -plotpady 0]

    set fgcolor [$scale(hist) cget -foreground]
    $scale(hist) legend configure -hide yes
    $scale(hist) element create bar1 -hide yes \
	-foreground $fgcolor -background $fgcolor -relief flat
    $scale(hist) xaxis configure -title "" -hide yes \
	-color $fgcolor -tickfont $graph(font) -ticklength 4 
    $scale(hist) yaxis configure -title "" -hide yes

    pack $scale(hist)

    # Cut Lines

    $scale(hist) marker bind min <B1-Motion> \
	"ScaleMotionDialog %x %y scaledlg(usermin)"
    $scale(hist) marker bind max <B1-Motion> \
	"ScaleMotionDialog %x %y scaledlg(usermax)"
    $scale(hist) marker bind up <ButtonRelease-1> ScaleReleaseDialog

    set scale(histmin) [$scale(hist) marker create line -element bar1 \
			    -outline red -bindtags [list min up]]
    set scale(histmax) [$scale(hist) marker create line -element bar1 \
			    -outline green -bindtags [list max up]]

    # Cut Levels

    frame $w.top.low
    frame $w.top.high
    pack $w.top.low $w.top.high -side left -padx 10

    label $w.top.low.title -text "Low"
    entry $w.top.low.value -textvariable scaledlg(usermin) -width 9
    pack $w.top.low.title $w.top.low.value -side left -padx 4

    label $w.top.high.title -text "High"
    entry $w.top.high.value -textvariable scaledlg(usermax) -width 9
    pack $w.top.high.title $w.top.high.value -side left -padx 4

    # Buttons

    button $w.buttons.apply -text "Apply" -command ScaleApplyDialog
    button $w.buttons.close -text "Close" -command ScaleDestroyDialog
    pack $w.buttons.apply $w.buttons.close -side left -padx 10 -expand true

    UpdateScaleDialog
}

proc ScaleApplyDialog {} {
    global scale
    global scaledlg
    global current

    if {$current(frame) != "" && 
	$scaledlg(usermin) != "" && 
	$scaledlg(usermax) != ""} {

	$scale(hist) marker configure $scale(histmin) \
	    -coords "$scaledlg(usermin) -Inf $scaledlg(usermin) Inf"
	$scale(hist) marker configure $scale(histmax) \
	    -coords "$scaledlg(usermax) -Inf $scaledlg(usermax) Inf"

	set scale(usermin) $scaledlg(usermin)
	set scale(usermax) $scaledlg(usermax)

	SetWatchCursor
	set scale(mode) user
	$current(frame) clip user $scale(usermin) $scale(usermax)
	$current(frame) clip mode $scale(mode)
	UpdateGraphYAxis
	UnsetWatchCursor
    }
}

proc ScaleDestroyDialog {} {
    global scale

    destroy $scale(top)
    destroy $scale(mb)

    unset scale(hist)
    unset scale(histmin)
    unset scale(histmax)
}

proc ScaleMotionDialog {x y varname} {
    upvar $varname var
    global scale
    global scaledlg

    set var [lindex [$scale(hist) invtransform $x $y] 0]
    if {$scaledlg(usermin) < $scaledlg(minmin)} {
	set scaledlg(usermin) $scaledlg(minmin)
    }
    if {$scaledlg(usermax) > $scaledlg(minmax)} {
	set scaledlg(usermax) $scaledlg(minmax)
    }

    if {$scaledlg(usermin) > $scaledlg(usermax)} {
	if {$varname == "scaledlg(usermin)"} {
	    set var $scaledlg(usermax)
	} else {
	    set var $scaledlg(usermin)
	}
    }

    $scale(hist) marker configure $scale(histmin) \
	-coords "$scaledlg(usermin) -Inf $scaledlg(usermin) Inf"
    $scale(hist) marker configure $scale(histmax) \
	-coords "$scaledlg(usermax) -Inf $scaledlg(usermax) Inf"
}

proc ScaleReleaseDialog {} {
    global scale
    global scaledlg
    global current

    set scale(usermin) $scaledlg(usermin)
    set scale(usermax) $scaledlg(usermax)

    SetWatchCursor
    set scale(mode) user
    $current(frame) clip user $scale(usermin) $scale(usermax)
    $current(frame) clip mode $scale(mode)
    UpdateGraphYAxis
    UnsetWatchCursor

    ScaleXAxisDialog
}

proc UpdateScaleDialog {} {
    global current
    global scale
    global scaledlg
    global histX histY

    global debug
    if {$debug(tcl,update)} {
	puts "UpdateScaleDialog"
    }

    if {[winfo exist $scale(top)]} {
	if {$current(frame) != ""} {
	    set limits [$current(frame) get clip]
	    set scaledlg(usermin) [lindex $limits 0]
	    set scaledlg(usermax) [lindex $limits 1]

	    if {[$current(frame) has fits] & ![$current(frame) has iis]} {

		set limits [$current(frame) get minmax]
		set scaledlg(minmin) [lindex $limits 0]
		set scaledlg(minmax) [lindex $limits 1]

		set width [$current(frame) get histogram histX histY]

		# add an offset for log scaling if y min is zero
		
		if {$histY(min) == 0} {
		    histY set [histY + .001]
		}

		set stepsize [format "%.3f" [expr ($histX(max)-$histX(min))/5]]

		$scale(hist) element configure bar1 -xdata histX -ydata histY \
		    -hide no -barwidth $width
		$scale(hist) xaxis configure -hide no -stepsize $stepsize \
		    -min [expr $histX(min)-$width] \
		    -max [expr $histX(max)+$width]
		$scale(hist) yaxis configure -min 1

		$scale(hist) marker configure $scale(histmin) \
		    -coords "$scaledlg(usermin) -Inf $scaledlg(usermin) Inf"
		$scale(hist) marker configure $scale(histmax) \
		    -coords "$scaledlg(usermax) -Inf $scaledlg(usermax) Inf"

		if {[$current(frame) has fits mosaic] || \
			[$current(frame) has fits cube]} {
		    $scale(mb) entryconfig "Scope" -state normal
		} else {
		    $scale(mb) entryconfig "Scope" -state disabled
		}

		if [$current(frame) has datamin] {
		    $scale(mb).minmax \
			entryconfig "DATAMIN DATAMAX" -state normal
		} else {
		    $scale(mb).minmax \
			entryconfig "DATAMIN DATAMAX" -state disabled
		}
		if [$current(frame) has irafmin] {
		    $scale(mb).minmax \
			entryconfig "IRAF-MIN IRAF-MAX" -state normal
		} else {
		    $scale(mb).minmax \
			entryconfig "IRAF-MIN IRAF-MAX" -state disabled
		}

		if [$current(frame) has datasec] {
		    $scale(mb).param entryconfig "use DATASEC" -state normal
		} else {
		    $scale(mb).param entryconfig "use DATASEC" -state disabled
		}

		ScaleYAxisDialog
		ScaleXAxisDialog

		return
	    }

	    $scale(hist) element configure bar1 -hide yes
	    $scale(hist) xaxis configure -hide yes

	    $scale(mb) entryconfig "Scope" -state normal
	    $scale(mb).minmax entryconfig "DATAMIN DATAMAX" -state normal
	    $scale(mb).minmax entryconfig "IRAF-MIN IRAF-MAX" -state normal

	    return
	}

	set scaledlg(usermin) {}
	set scaledlg(usermax) {}
	$scale(hist) element configure bar1 -hide yes
	$scale(hist) xaxis configure -hide yes
	$scale(mb) entryconfig "Scope" -state normal
	$scale(mb).minmax entryconfig "DATAMIN DATAMAX" -state normal
	$scale(mb).minmax entryconfig "IRAF-MIN IRAF-MAX" -state normal
	$scale(mb).param entryconfig "use DATASEC" -state normal
    }
}

proc ScaleYAxisDialog {} {
    global scale

    switch -- $scale(yaxis) {
	linear {$scale(hist) yaxis configure -logscale 0}
	log {$scale(hist) yaxis configure -logscale 1 -min 1}
    }
}

proc ScaleXAxisDialog {} {
    global scale
    global scaledlg

    switch -- $scale(xaxis) {
	full {$scale(hist) xaxis configure -min "" -max ""}
	current {
	    set diff [expr $scaledlg(usermax)-$scaledlg(usermin)]
	    set per .25
	    set a [expr $scaledlg(usermin)-($diff*$per)]
	    if {$a < $scaledlg(minmin)} {
		set a ""
	    }
	    set b [expr $scaledlg(usermax)+($diff*$per)]
	    if {$b > $scaledlg(minmax)} {
		set b ""
	    }
	    $scale(hist) xaxis configure -min $a -max $b
	}
    }
}

proc ZScaleDialog {} {
    global current
    global ed
    global rgb

    set w ".zscale"
    set length 300
    set params [$current(frame) get clip zscale param]

    set ed(ok) 0
    set ed(contrast) [lindex $params 0]
    set ed(size) [lindex $params 1]
    set ed(line) [lindex $params 2]

    DialogCreate $w "ZScale Parameters" -borderwidth 2
    frame $w.zscale  -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.zscale $w.buttons -ipadx 4 -ipady 4 -fill x -expand true

    scale $w.zscale.contrast -from 0 -to 1 -length $length \
	-variable ed(contrast) -orient horizontal -label "Contrast" \
	-tickinterval .2 -showvalue true -resolution .01
    scale $w.zscale.size -from 0 -to 1000 -length $length \
	-variable ed(size) -orient horizontal -label "Number of Samples" \
	-tickinterval 200 -showvalue true -resolution 10
    scale $w.zscale.line -from 0 -to 500 -length $length \
	-variable ed(line) -orient horizontal -label "Samples per Line" \
	-tickinterval 100 -showvalue true -resolution 5
    pack $w.zscale.contrast $w.zscale.size $w.zscale.line -side top

    button $w.buttons.ok -text "OK" -default active -command {set ed(ok) 1}
    button $w.buttons.cancel -text "Cancel" -command {set ed(ok) 0}
    pack $w.buttons.ok -side left -padx 10
    pack $w.buttons.cancel -side right -padx 10

    bind $w <Return> {set ed(ok) 1}
    bind $w <Alt-o> "tkButtonInvoke $w.buttons.ok"
    bind $w <Alt-c> "tkButtonInvoke $w.buttons.cancel"

    DialogCenter $w 
    DialogWait $w ed(ok)
    DialogDismiss $w

    if {$ed(ok)} {
	SetWatchCursor
	if {$ed(line) == 0} {
	    set ed(line) 1
	}
	RGBEvalLock rgb(lock,scale) "$current(frame) clip zscale param $ed(contrast) $ed(size) $ed(line)"
	UpdateGraphYAxis
	UnsetWatchCursor
    }

    # we want to destroy this window

    destroy $w 
    unset ed
}

proc MinMaxSampleDialog {varname doit} {
    upvar $varname var
    global minmax
    global prefs
    global ed

    set w ".sample"
    set length 300

    set ed(ok) 0
    set ed(value) $var

    DialogCreate $w "Sample Parameters" -borderwidth 2
    frame $w.sample  -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.sample $w.buttons -ipadx 4 -ipady 4 -fill x -expand true

    scale $w.sample.incr -from 0 -to 1000 -length $length \
	-variable ed(value) -orient horizontal -label "Sample Increment" \
	-tickinterval 200 -showvalue true
    pack $w.sample.incr

    button $w.buttons.ok -text "OK" -default active -command {set ed(ok) 1}
    button $w.buttons.cancel -text "Cancel" -command {set ed(ok) 0}
    pack $w.buttons.ok -side left -padx 10
    pack $w.buttons.cancel -side right -padx 10

    bind $w <Return> {set ed(ok) 1}
    bind $w <Alt-o> "tkButtonInvoke $w.buttons.ok"
    bind $w <Alt-c> "tkButtonInvoke $w.buttons.cancel"

    DialogCenter $w 
    DialogWait $w ed(ok)
    DialogDismiss $w

    if {$ed(ok)} {
	if {$ed(value) == 0} {
	    set ed(value) 1
	}
	set var $ed(value)

	if {$doit} {
	    ChangeMinMaxSample
	}
    }

    # we want to destroy this window

    destroy $w 
    unset ed
}

proc MatchScales {} {
    global current

    SetWatchCursor
    if {[$current(frame) get type] == "rgb"} {
	MatchScalesRGB
    } else {
	MatchScalesBase
    }
    UnsetWatchCursor
}

proc MatchScalesRGB {} {
    global ds9
    global current
    global colorbar

    set ch [$current(frame) get rgb channel]
    foreach c {red green blue} {
	$current(frame) rgb channel $c

	set type [$current(frame) get colorscale]
	set mode [$current(frame) get clip mode]
	set scope [$current(frame) get clip scope]
	set mmmode [$current(frame) get clip minmax mode]
	set mmsample [$current(frame) get clip minmax sample]

	set limits [$current(frame) get clip]
	set usermin [lindex $limits 0]
	set usermax [lindex $limits 1]

	set params [$current(frame) get clip zscale param]
	set zscontrast [lindex $params 0]
	set zssize [lindex $params 1]
	set zsline [lindex $params 2]

	foreach f $ds9(frames) {
	    if {$f != $current(frame) && [$f get type] == "rgb"} {
		set chh [$f get rgb channel]
		$f rgb channel $c

		$f colorscale $type
		$f clip mode $mode
		$f clip scope $scope
		$f clip minmax mode $mmmode
		$f clip minmax sample $mmsample
		$f clip user $usermin $usermax
		$f clip zscale param $zscontrast $zssize $zsline

		$f rgb channel $chh
	    }
	}
    }

    $current(frame) rgb channel $ch
}

proc MatchScalesBase {} {
    global ds9
    global current
    global colorbar

    set type [$current(frame) get colorscale]
    set mode [$current(frame) get clip mode]
    set scope [$current(frame) get clip scope]
    set mmmode [$current(frame) get clip minmax mode]
    set mmsample [$current(frame) get clip minmax sample]

    set limits [$current(frame) get clip]
    set usermin [lindex $limits 0]
    set usermax [lindex $limits 1]

    set params [$current(frame) get clip zscale param]
    set zscontrast [lindex $params 0]
    set zssize [lindex $params 1]
    set zsline [lindex $params 2]

    foreach f $ds9(frames) {
	if {$f != $current(frame) && [$f get type] == "base"} {
	    $f colorscale $type
	    $f clip mode $mode
	    $f clip scope $scope
	    $f clip minmax mode $mmmode
	    $f clip minmax sample $mmsample
	    $f clip user $usermin $usermax
	    $f clip zscale param $zscontrast $zssize $zsline
	}
    }
}

proc ProcessScaleCmd {varname iname} {
    upvar $varname var
    upvar $iname i

    global scale

    switch -- [string tolower [lindex $var $i]] {
	linear -
	log -
	squared -
	sqrt -
	histequ {
	    set scale(type) [string tolower [lindex $var $i]]
	    ChangeScale
	}
	datasec {
	    incr i
	    set scale(datasec) [FromYesNo [lindex $var $i]]
	    ChangeDATASEC
	}
	limits {
	    incr i
	    set scale(usermin) [lindex $var $i]
	    incr i
	    set scale(usermax) [lindex $var $i]
	    ChangeScaleLimit
	}
	minmax -
	zscale -
	zmax -
	user {
	    set scale(mode) [string tolower [lindex $var $i]]
	    ChangeScaleMode
	}
	mode {
	    incr i
	    set scale(mode) [string tolower [lindex $var $i]]
	    ChangeScaleMode
	}
	local -
	global {
	    set scale(scope) [string tolower [lindex $var $i]]
	    ChangeScaleScope
	}
	scope {
	    incr i
	    set scale(scope) [string tolower [lindex $var $i]]
	    ChangeScaleScope
	}
    }
}

proc ProcessMinMaxCmd {varname iname} {
    upvar $varname var
    upvar $iname i

    global minmax
    global scale

    switch -- [string tolower [lindex $var $i]] {
	scan -
	sample -
	datamin -
	irafmin {
	    set minmax(mode) [string tolower [lindex $var $i]]
	    ChangeMinMaxMode
	}
	mode {
	    incr i
	    set minmax(mode) [string tolower [lindex $var $i]]
	    ChangeMinMaxMode
	}
	interval {
	    incr i
	    set minmax(sample) [lindex $var $i]
	    ChangeMinMaxSample
	}
	default {
	    # for backward compatibility
	    set scale(mode) minmax
	    ChangeScaleMode
	    incr i -1
	}
    }
}

