#  Copyright (C) 1999-2004
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

proc NSVRDialog {} {
    global nsvr
    global ds9
    global menu

    if [winfo exists $nsvr(top)] {
	raise $nsvr(top)
	return
    }

    set w $nsvr(top)
    set title "Name Resolution"

    # variables

    set nsvr(system) wcs
    set nsvr(sky) fk5

    # create the window

    toplevel $w -colormap $ds9(main)
    wm title $w $title
    wm iconname $w $title
    wm group $w $ds9(top)
    wm protocol $w WM_DELETE_WINDOW NSVRDestroyDialog

    # menu

    $w configure -menu $nsvr(mb)

    menu $nsvr(mb) -tearoff 0
    $nsvr(mb) add cascade -label File -menu $nsvr(mb).file
    $nsvr(mb) add cascade -label Edit -menu $nsvr(mb).edit
    $nsvr(mb) add cascade -label "Name Server" -menu $nsvr(mb).name
    $nsvr(mb) add cascade -label Format -menu $nsvr(mb).format

    menu $nsvr(mb).file -tearoff 0 -selectcolor $menu(selectcolor)
    $nsvr(mb).file add command -label Resolve -command NSVRApplyDialog
    $nsvr(mb).file add command -label Cancel -command NSVRCancelDialog
    $nsvr(mb).file add separator
    $nsvr(mb).file add command -label "Pan To" -command NSVRPanDialog
    $nsvr(mb).file add command -label "Crosshair To" \
	-command NSVRCrosshairDialog
    $nsvr(mb).file add separator
    $nsvr(mb).file add command -label Close -command NSVRDestroyDialog

    menu $nsvr(mb).edit -tearoff 0 -selectcolor $menu(selectcolor)
    $nsvr(mb).edit add command -label Cut -state disabled
    $nsvr(mb).edit add command -label Copy -state disabled
    $nsvr(mb).edit add command -label Paste -state disabled
    $nsvr(mb).edit add command -label Clear -command NSVRClearDialog

    menu $nsvr(mb).name -tearoff 0 -selectcolor $menu(selectcolor)
    $nsvr(mb).name add radiobutton -label "NED@SAO" \
	-variable nsvr(server) -value ned-sao
    $nsvr(mb).name add radiobutton -label "NED@ESO" \
	-variable nsvr(server) -value ned-eso
    $nsvr(mb).name add radiobutton -label "SIMBAD@SAO" \
	-variable nsvr(server) -value simbad-sao
    $nsvr(mb).name add radiobutton -label "SIMBAD@ESO" \
	-variable nsvr(server) -value simbad-eso

    menu $nsvr(mb).format -tearoff 0 -selectcolor $menu(selectcolor)
    $nsvr(mb).format add radiobutton -label "Degrees" \
	-variable nsvr(skyformat) -value degrees
    $nsvr(mb).format add radiobutton -label "Sexagesimal" \
	-variable nsvr(skyformat) -value sexagesimal

    frame $w.param -relief groove -borderwidth 2
    frame $w.status -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.param $w.status $w.buttons -fill x -expand true -ipadx 4 -ipady 4

    label $w.param.nametitle -text "Name"
    entry $w.param.name -textvariable nsvr(name) -width 32

    label $w.param.xtitle -text "a" -font {symbol 12} -width 1
    label $w.param.x -textvariable nsvr(x) -width 14 -relief groove
    label $w.param.ytitle -text "d" -font {symbol 12} -width 1
    label $w.param.y -textvariable nsvr(y) -width 14 -relief groove
    label $w.param.system -textvariable nsvr(sky) -width 10 -relief groove

    grid rowconfigure $w.param 0 -pad 4
    grid rowconfigure $w.param 1 -pad 4
    grid rowconfigure $w.param 2 -pad 4

    grid $w.param.nametitle -row 0 -column 0 -padx 4 -pady 1 -sticky w
    grid $w.param.name -row 0 -column 1 -columnspan 4 -padx 4 -pady 1 -sticky w

    grid $w.param.xtitle -row 1 -column 0 -padx 4 -pady 1 -sticky w
    grid $w.param.x -row 1 -column 1 -padx 4 -sticky w
    grid $w.param.ytitle -row 1 -column 2 -padx 4 -pady 1 -sticky w
    grid $w.param.y -row 1 -column 3 -padx 4 -sticky w
    grid $w.param.system -row 1 -column 4 -padx 4 -pady 1 -sticky w

    label $w.status.item -textvariable nsvr(status)
    pack $w.status.item -anchor w -pady 4

    set nsvr(apply) \
	[button $w.buttons.apply -text "Resolve" -command NSVRApplyDialog]
    set nsvr(cancel) \
	[button $w.buttons.cancel -text "Cancel" -command NSVRCancelDialog \
	     -state disabled]

    button $w.buttons.close -text "Close" -command NSVRDestroyDialog
    pack $w.buttons.apply $w.buttons.cancel $w.buttons.close \
	-side left -padx 10 -expand true

    NSVRStatus {}
}

proc NSVRDestroyDialog {} {
    global nsvr

    if {[info exists nsvr(token)]} {
	set nsvr(state) 0
	http::reset $nsvr(token)
    }

    destroy $nsvr(top)
    destroy $nsvr(mb)

    unset nsvr(status)
    unset nsvr(cancel)
    unset nsvr(apply)
}

proc NSVRApplyDialog {} {
    global nsvr

    NSVRStatus {}
    $nsvr(apply) configure -state disabled
    $nsvr(cancel) configure -state normal

    if {$nsvr(name)!=""} {
	NSVRStatus "Looking up $nsvr(name)"
	set nsvr(x) {}
	set nsvr(y) {}
	NameSVR
    } else {
	NSVRStatus "Please specify an Object Name"

	$nsvr(apply) configure -state normal
	$nsvr(cancel) configure -state disabled
    }
}

proc NSVRCancelDialog {} {
    global nsvr

    if {[info exists nsvr(token)]} {
	set nsvr(state) 0
	http::reset $nsvr(token)
    }
}

proc NameSVR {} {
    global nsvr
    global http

    switch -- $nsvr(server) {
	ned-sao {
	    set query "[regsub -all {\ } $nsvr(name) {%20}]"
	    set url "http://cfa-www.harvard.edu/catalog/ned"
	    set cmd NEDSAO
	}
	ned-eso {
	    set query "&o=[regsub -all {\ } $nsvr(name) {%20}]"
	    set url "http://archive.eso.org/skycat/servers/ned-server"
	    set cmd NEDESO
	}
	simbad-sao {
	    set query "[regsub -all {\ } $nsvr(name) {%20}]"
	    set url "http://cfa-www.harvard.edu/catalog/simbad-cfa"
	    set cmd SIMBADSAO
	}
	simbad-eso {
	    set query "&o=[regsub -all {\ } $nsvr(name) {%20}]"
	    set url "http://archive.eso.org/skycat/servers/sim-server"
	    set cmd SIMBADESO
	}
    }

    set nsvr(token) [eval "http::geturl $url?$query -command $cmd -headers \{[ProxyHTTP]\}"]
    set nsvr(state) 1
}

proc NSVRParse {data} {
    global nsvr

    set nsvr(x) [lindex $data 0]
    set nsvr(y) [lindex $data 1]

    if {($nsvr(x) == "") || ($nsvr(y) == "")} {
	set nsvr(x) {}
	set nsvr(y) {}
	NSVRStatus "$nsvr(name) not found"
    } else {
	NSVRStatus {Done}
    }
}

proc NSVRReset {} {
    global nsvr

    set nsvr(state) 0
    catch {unset nsvr(token)}

    $nsvr(apply) configure -state normal
    $nsvr(cancel) configure -state disabled
}

proc NSVRError {code} {
    NSVRStatus "Error code $code was returned"
    NSVRReset
}

proc NEDSAO {token} {
    global nsvr

    HTTPLog $token
    if {$nsvr(state)} {
	# check for error
	set code [http::ncode $token]
	if {$code != "200"} {
	    NSVRError $code
	    return
	}

	# data is in sexagesmal
	set data [http::data $token]
	set r [lindex $data 0]
	set d  [lindex $data 1]
	if {($r != "") && ($d != "")} {
	    switch -- $nsvr(skyformat) {
		degrees {NSVRParse "[expr h2d([Sex2H $r])] [Sex2D $d]"}
		sexagesimal {NSVRParse "$r $d"}
	    }
	} else {
	    NSVRParse {}
	}
    } else {
	NSVRStatus {Cancelled}
    }
    NSVRReset
}

proc NEDESO {token} {
    global nsvr

    HTTPLog $token
    if {$nsvr(state)} {
	# check for error
	set code [http::ncode $token]
	if {$code != "200"} {
	    NSVRError $code
	    return
	}

	# data is in degrees
	set data [http::data $token]
	set l [llength $data]
	set r [lindex $data [expr $l-3]]
	set d [lindex $data [expr $l-2]]
	if {[string is double -strict $r] && [string is double -strict $d]} {
	    switch -- $nsvr(skyformat) {
		degrees {NSVRParse "$r $d"}
		sexagesimal {
		    NSVRParse "[format %.3@ [expr d2h($r)]] [format %.3@ $d]"
		}
	    }
	} else {
	    NSVRParse {}
	}
    } else {
	NSVRStatus {Cancelled}
    }
    NSVRReset
}

proc SIMBADSAO {token} {
    global nsvr

    HTTPLog $token
    if {$nsvr(state)} {
	# check for error
	set code [http::ncode $token]
	if {$code != "200"} {
	    NSVRError $code
	    return
	}

	# data is in sexagesmal
	set data [http::data $token]
	set r [lindex $data 0]
	set d  [lindex $data 1]
	if {($r != "") && ($d != "")} {
	    switch -- $nsvr(skyformat) {
		degrees {NSVRParse "[expr h2d([Sex2H $r])] [Sex2D $d]"}
		sexagesimal {NSVRParse $data}
	    }
	} else {
	    NSVRParse {}
	}
    } else {
	NSVRStatus {Cancelled}
    }
    NSVRReset
}

proc SIMBADESO {token} {
    global nsvr

    HTTPLog $token
    if {$nsvr(state)} {
	# check for error
	set code [http::ncode $token]
	if {$code != "200"} {
	    NSVRError $code
	    return
	}

	# data is in degrees
	set data [http::data $token]
	set l [llength $data]
	set r [lindex $data [expr $l-3]]
	set d [lindex $data [expr $l-2]]
	if {[string is double -strict $r] && [string is double -strict $d]} {
	    switch -- $nsvr(skyformat) {
		degrees {NSVRParse "$r $d"}
		sexagesimal {
		    NSVRParse "[format %.3@ [expr d2h($r)]] [format %.3@ $d]"
		}
	    }
	} else {
	    NSVRParse {}
	}
    } else {
	NSVRStatus {Cancelled}
    }
    NSVRReset
}

# Other procedures

proc NSVRStatus {message} {
    global nsvr

    set nsvr(status) "Status: $message"
}

proc NSVRClearDialog {} {
    global nsvr

    set nsvr(name) {}
    set nsvr(x) {}
    set nsvr(y) {}
    NSVRStatus {}
}

proc NSVRPanDialog {} {
    global nsvr

    if {$nsvr(x) != ""} {
	PanTo $nsvr(x) $nsvr(y) $nsvr(system) $nsvr(sky)
    }
}

proc NSVRCrosshairDialog {} {
    global nsvr
    global current
    global ds9

    if {($current(frame) != "") && ($nsvr(x) != "")} {
	set ds9(mode) crosshair
	ChangeMode
	$current(frame) crosshair $nsvr(system) $nsvr(sky) $nsvr(x) $nsvr(y)
    }
}

proc ProcessNSVRCmd {varname iname} {
    upvar $varname var
    upvar $iname i

    global nsvr

    switch -- [string tolower [lindex $var $i]] {
	server {
	    incr i
	    set nsvr(server) [lindex $var $i]
	}
	format -
	skyformat {
	    incr i
	    set nsvr(skyformat) [lindex $var $i]
	}
	name {
	    incr i
	    set nsvr(name) [lindex $var $i]
	    NSVRDialog
	    NSVRApplyDialog
	}
	default {
	    set nsvr(name) [lindex $var $i]
	    NSVRDialog
	    NSVRApplyDialog
	}
    }
}
