#  Copyright (C) 1999-2004
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

# Public Procedures

proc InitCanvas {} {
    global canvas

    # Bindings

    bind $canvas(name) <Configure> [list LayoutFrames]
    bind $canvas(name) <Tab> [list NextFrame]
    bind $canvas(name) <Shift-Tab> [list PrevFrame]
    bind $canvas(name) <Enter> [list focus $canvas(name)]
    bind $canvas(name) <Leave> [list focus {}]

    # backward compatible bindings

    bind $canvas(name) <Button-3> {Button3Canvas %x %y}
    bind $canvas(name) <B3-Motion> {Motion3Canvas %x %y}
    bind $canvas(name) <ButtonRelease-3> {Release3Canvas %x %y}

    # freeze

    bind $canvas(name) <f> {ToggleFreeze}
}

proc Button3Canvas {x y} {
    global ds9

    set ds9(b3) 1
    ButtonColormap $x $y
}

proc Motion3Canvas {x y} {
    MotionColormap $x $y
}

proc Release3Canvas {x y} {
    global ds9

    set ds9(b3) 0
    ReleaseColormap $x $y
}

proc FreezeCanvas {} {
    global ds9

    foreach f $ds9(active) {
	UnBindEventsFrame $f
    }
}

proc UnFreezeCanvas {} {
    global ds9
    global current

    switch -- $ds9(display,mode) {
	single -
	blink {BindEventsFrame $current(frame)}
	tile {
	    foreach f $ds9(active) {
		BindEventsFrame $f
	    }
	}
    }
}

proc CreateFrame {} {
    CreateNamedFrame Frame base
}

proc CreateRGBFrame {} {
    CreateNamedFrame Frame rgb
    RGBDialog
}

proc CreateNamedFrame {name type} {
    global ds9

    # find the first open slot

    set num 1
    while {1} {
	if {[lsearch $ds9(frames) "$name$num"]==-1} {
	    CreateNameNumberFrame "$name$num" $type
	    return
	}
	incr num
    }
}

proc CreateGotoFrame {num type} {
    global ds9
    global active

    set which "Frame$num"
    if {[lsearch $ds9(frames) $which]==-1} {
	CreateNameNumberFrame $which $type
    } else {
	if {$active($which)==0} {
	    set active($which) 1
	    UpdateActiveFrames
	}
	set ds9(next) $which
	GotoFrame
    }
}

proc CreateNameNumberFrame {which type} {
    global ds9
    global active
    global current
    global canvas
    global panner
    global magnifier
    global bin
    global contour
    global scale
    global wcs
    global colorbar
    global minmax
    global message
    global marker
    global panzoom

    set ds9(next) $which

    # update frame lists
    lappend ds9(frames) $ds9(next)
    set ds9(frames) [lsort -dictionary $ds9(frames)]
    lappend ds9(active) $ds9(next)
    set ds9(active) [lsort -dictionary $ds9(active)]
    set active($ds9(next)) 1

    # and update frame list menus
    UpdateFrameMenuItems

    # and create the frame
    switch -- $ds9(visual) {
	pseudocolor {
	    if {$type == "rgb"} {
		Error "$message(error,frame,rgb)"
	    }
	    $canvas(name) create frame$ds9(visual)$ds9(depth) \
		-x 0 -y 0 -anchor nw -command $ds9(next) -tag $ds9(next)
	    $ds9(next) colormap [colorbar get colormap]
	}
    	truecolor {
	    switch -- $type {
		base {
		    $canvas(name) create frame$ds9(visual)$ds9(depth) \
			-x 0 -y 0 -anchor nw -command $ds9(next) \
			-tag $ds9(next)
		    $ds9(next) colormap [colorbar get colormap]
		}
		rgb {
		    $canvas(name) create framergb$ds9(visual)$ds9(depth) \
			-x 0 -y 0 -anchor nw -command $ds9(next) \
			-tag $ds9(next)
		    $ds9(next) colormap [colorbarrgb get colormap]
		}
	    }
	}
    }

    $ds9(next) panner 'panner' $panner(size) $panner(size)
    $ds9(next) panner wcs $panner(compass,wcs,system) $panner(compass,wcs,sky)
    $ds9(next) magnifier 'magnifier' $magnifier(size) $magnifier(size)

    $ds9(next) zoom to $current(zoom)
    $ds9(next) rotate to $current(rotate)
    $ds9(next) orient $current(orient)
    $ds9(next) pan preserve $panzoom(preserve)

    $ds9(next) wcs align $wcs(align) $wcs(align,system) $wcs(align,sky)

    $ds9(next) datasec $scale(datasec)
    $ds9(next) colormap area $colorbar(area) $colorbar(mode)
    $ds9(next) mosaic fast $ds9(mosaic,fast)
    $ds9(next) bg color $ds9(bg,color)
    $ds9(next) nan color $ds9(nan,color)

    $ds9(next) marker fg show $marker(show)
    $ds9(next) marker preserve $marker(preserve)

    # channel dependent items
    if {[$ds9(next) get type] == "rgb"} {
	foreach c {red green blue} {
	    $ds9(next) rgb channel $c

	    $ds9(next) colorscale $scale(type)

	    $ds9(next) clip scope $scale(scope)
	    $ds9(next) clip mode $scale(mode)
	    $ds9(next) clip user $scale(usermin) $scale(usermax)
	    $ds9(next) clip preserve $scale(preserve)

	    $ds9(next) clip minmax mode $minmax(mode)
	    $ds9(next) clip minmax sample $minmax(sample)

	    $ds9(next) bin function $bin(function)
	    $ds9(next) bin factor to $bin(factor)
	    $ds9(next) bin depth $bin(depth)
	    $ds9(next) bin buffer size $bin(buffersize)
	    $ds9(next) bin smooth $bin(smooth)
	    $ds9(next) bin smooth function $bin(smooth,function)
	    $ds9(next) bin smooth radius $bin(smooth,radius)
	}
	$ds9(next) rgb channel red
	
    } else {
	$ds9(next) colorscale $scale(type)

	$ds9(next) clip scope $scale(scope)
	$ds9(next) clip mode $scale(mode)
	$ds9(next) clip user $scale(usermin) $scale(usermax)
	$ds9(next) clip preserve $scale(preserve)

	$ds9(next) clip minmax mode $minmax(mode)
	$ds9(next) clip minmax sample $minmax(sample)

	$ds9(next) bin function $bin(function)
	$ds9(next) bin factor to $bin(factor)
	$ds9(next) bin depth $bin(depth)
	$ds9(next) bin buffer size $bin(buffersize)
	$ds9(next) bin smooth $bin(smooth)
	$ds9(next) bin smooth function $bin(smooth,function)
	$ds9(next) bin smooth radius $bin(smooth,radius)
    }

    if {$ds9(mode) == "crosshair"} {
	$ds9(next) crosshair on
    }

    set current(frame) $ds9(next)
    DisplayMode
}

proc DeleteFrameMenu {} {
    global buttons

    if {[tk_messageBox -type okcancel -default cancel \
	     -message "Delete this Frame?" -icon question] == "ok"} {
	DeleteCurrentFrame
    }
}

proc DeleteAllFramesMenu {} {
    global buttons

    if {[tk_messageBox -type okcancel -default cancel \
	     -message "Delete All Frames?" -icon question] == "ok"} {
	DeleteAllFrames
    }
}

proc DeleteCurrentFrame {} {
    global current

    if {$current(frame) != ""} {
	DeleteSingleFrame $current(frame)
    }
}

proc DeleteAllFrames {} {
    global ds9

    # we can not just cycle thru all frames, since some frames maybe slaves
    # foreach f $ds9(frames)
    set f [lindex $ds9(frames) 0]
    while {$f != {}} {
	DeleteFrame $f
	set f [lindex $ds9(frames) 0]
    }

    UpdateActiveFrames

    ClearInfoBox
    ClearPixelTable
    ClearGraphData
}

proc DeleteSingleFrame {which} {
    DeleteFrame $which

    UpdateActiveFrames

    ClearInfoBox
    ClearPixelTable
    ClearGraphData
}

proc DeleteFrame {which} {
    global ds9
    global active
    global header
    global current
    global canvas
    global contour
    global marker

    # contour copy
    if {$contour(copy) == $which} {
	set contour(copy) {}
    }
    # marker copy
    if {$marker(copy) == $which} {
	set marker(copy) {}
    }
    
    # kill the header dialog (if needed)
    DestroyHeader $which

    # delete any slave frames
    DeleteExamine $which

    # delete canvas widget
    $canvas(name) delete $which

    # delete it from the frame list
    set i [lsearch $ds9(frames) $which]
    set ds9(frames) [lreplace $ds9(frames) $i $i]
    unset active($which)

    # delete menu entries
    # NOTE: we can't use 'menuBar.frame delete' since there is a bug
    UpdateFrameMenuItems
}

proc UpdateCurrentFrame {} {
    global current

    UpdateFrame $current(frame)
}

proc UpdateAllFrame {} {
    global ds9

    foreach f $ds9(frames) {
	UpdateFrame $f
    }
}

proc UpdateFrame {which} {
    SetWatchCursor
    if {$which != ""} {
	$which update
    }
    UnsetWatchCursor
}

proc ChangeBackgroundColor {} {
    global ds9

    foreach f $ds9(frames) {
	$f bg color $ds9(bg,color)
    }
}

proc ChangeNaNColor {} {
    global ds9

    foreach f $ds9(frames) {
	$f nan color $ds9(nan,color)
    }
}

proc ChangeMosaicFast {} {
    global ds9

    foreach f $ds9(frames) {
	$f mosaic fast $ds9(mosaic,fast)
    }
}

# Event Processing

proc BindEventsFrame {which} {
    global canvas
    global analysis

    $canvas(name) bind $which <Motion> [list MotionFrame $which %x %y]
    $canvas(name) bind $which <Shift-Motion> \
	[list ShiftMotionFrame $which %x %y]
    $canvas(name) bind $which <Control-Motion> \
	[list ControlMotionFrame $which %x %y]

    $canvas(name) bind $which <Enter> [list EnterFrame $which %x %y]
    $canvas(name) bind $which <Leave> [list LeaveFrame $which]

    $canvas(name) bind $which <Button-1> [list Button1Frame $which %x %y]
    $canvas(name) bind $which <Shift-Button-1> \
	[list ShiftButton1Frame $which %x %y]
    $canvas(name) bind $which <Control-Button-1> \
	[list ControlButton1Frame $which %x %y]
    $canvas(name) bind $which <Control-Shift-Button-1> \
	[list ControlShiftButton1Frame $which %x %y]
    $canvas(name) bind $which <B1-Motion> [list Motion1Frame $which %x %y]
    $canvas(name) bind $which <ButtonRelease-1> \
	[list Release1Frame $which %x %y]

    $canvas(name) bind $which <Double-1> [list Double1Frame $which %x %y]
    $canvas(name) bind $which <Double-ButtonRelease-1> \
	[list DoubleRelease1Frame $which %x %y]

    $canvas(name) bind $which <Button-2> [list Button2Frame $which %x %y]
    $canvas(name) bind $which <Shift-Button-2> \
	[list ShiftButton2Frame $which %x %y]
    $canvas(name) bind $which <B2-Motion> [list Motion2Frame $which %x %y]
    $canvas(name) bind $which <ButtonRelease-2> \
	[list Release2Frame $which %x %y]

    $canvas(name) bind $which <Delete> [list DeleteKeyFrame $which %x %y]
    $canvas(name) bind $which <BackSpace> [list DeleteKeyFrame $which %x %y]

    $canvas(name) bind $which <Up> [list ArrowKeyFrame $which 0 -1]
    $canvas(name) bind $which <Down> [list ArrowKeyFrame $which 0 1]
    $canvas(name) bind $which <Left> [list ArrowKeyFrame $which -1 0]
    $canvas(name) bind $which <Right> [list ArrowKeyFrame $which 1 0]

    $canvas(name) bind $which <k> [list ArrowKeyFrame $which 0 -1]
    $canvas(name) bind $which <j> [list ArrowKeyFrame $which 0 1]
    $canvas(name) bind $which <h> [list ArrowKeyFrame $which -1 0]
    $canvas(name) bind $which <l> [list ArrowKeyFrame $which 1 0]

    # don't forget to disable for imexamine in iis.tcl

    $canvas(name) bind $which <c> [list DisplayCoordDialog $which %x %y canvas]
    $canvas(name) bind $which <i> [list $which marker property include 1 %x %y]
    $canvas(name) bind $which <e> [list $which marker property include 0 %x %y]
    $canvas(name) bind $which <s> [list $which marker property source 1 %x %y]
    $canvas(name) bind $which <b> [list $which marker property source 0 %x %y]
    $canvas(name) bind $which <g> GroupCreate
    $canvas(name) bind $which <G> GroupCreateSilent

    $canvas(name) bind $which <plus> [list CubeNext]
    $canvas(name) bind $which <minus> [list CubePrev]

    # analysis

    for {set i 0} {$i<$analysis(bind,count)} {incr i} {
	$canvas(name) bind $which "$analysis(bind,$i,item)" \
	    "AnalysisTask $i bind $which %x %y"
    }
}

proc UnBindEventsFrame {which} {
    global canvas
    global analysis

    $canvas(name) bind $which <Motion> {}
    $canvas(name) bind $which <Shift-Motion> {}
    $canvas(name) bind $which <Control-Motion> {}

    $canvas(name) bind $which <Enter> {}
    $canvas(name) bind $which <Leave> {}

    $canvas(name) bind $which <Button-1> {}
    $canvas(name) bind $which <Shift-Button-1> {}
    $canvas(name) bind $which <Control-Button-1> {}
    $canvas(name) bind $which <Control-Shift-Button-1> {}
    $canvas(name) bind $which <B1-Motion> {}
    $canvas(name) bind $which <ButtonRelease-1> {}

    $canvas(name) bind $which <Double-1> {}
    $canvas(name) bind $which <Double-ButtonRelease-1> {}

    $canvas(name) bind $which <Button-2> {}
    $canvas(name) bind $which <Shift-Button-2> {}
    $canvas(name) bind $which <B2-Motion> {}
    $canvas(name) bind $which <ButtonRelease-2> {}

    $canvas(name) bind $which <Delete> {}
    $canvas(name) bind $which <BackSpace> {}

    $canvas(name) bind $which <Up> {}
    $canvas(name) bind $which <Down> {}
    $canvas(name) bind $which <Left> {}
    $canvas(name) bind $which <Right> {}

    $canvas(name) bind $which <k> {}
    $canvas(name) bind $which <j> {}
    $canvas(name) bind $which <h> {}
    $canvas(name) bind $which <l> {}

    $canvas(name) bind $which <c> {}
    $canvas(name) bind $which <i> {}
    $canvas(name) bind $which <e> {}
    $canvas(name) bind $which <s> {}
    $canvas(name) bind $which <b> {}
    $canvas(name) bind $which <g> {}
    $canvas(name) bind $which <G> {}

    $canvas(name) bind $which <+> {}
    $canvas(name) bind $which <-> {}

    # analysis

    for {set i 0} {$i<$analysis(bind,count)} {incr i} {
	$canvas(name) bind $which "$analysis(bind,$i,item)" {}
    }
}

proc EnterFrame {which x y} {
    global ds9
    global canvas
    global info
    global magnifier

    global debug
    if {$debug(tcl,events)} {
	puts "EnterFrame"
    }

    # check to see if this event was generated while processing other events
    if {$ds9(b1) || $ds9(sb1) || $ds9(cb1) || $ds9(csb1) || $ds9(b2) || $ds9(b3)} {
	return
    }

    $canvas(name) focus $which

    switch -- $ds9(mode) {
	crosshair {
	    set coord [$which get crosshair canvas]
	    set x [lindex $coord 0]
	    set y [lindex $coord 1]
	}
	none -
	pointer -
	colorbar -
	pan -
	zoom -
	rotate -
	examine -
	illustrate -
	imexam  {}
    }

    EnterInfoBox $which $x $y canvas
    UpdatePixelTable $which $x $y canvas
    UpdateGraph $which $x $y canvas
    UpdateMagnifier $which $x $y

    if {$magnifier(view)} {
	$which magnifier on
    }

    UpdateEditMenu
}

proc LeaveFrame {which} {
    global ds9
    global canvas
    global info
    global magnifier

    global debug
    if {$debug(tcl,events)} {
	puts "LeaveFrame"
    }

    # check to see if this event was generated while processing other events
    if {$ds9(b1) || $ds9(sb1) || $ds9(cb1) || $ds9(csb1) || $ds9(b2) || $ds9(b3)} {
	return
    }

    $canvas(name) focus {}

    switch -- $ds9(mode) {
	crosshair {}
	none -
	pointer -
	colorbar -
	pan -
	zoom -
	rotate -
	examine -
	illustrate -
	imexam {
	    LeaveInfoBox
	    ClearPixelTable
	    ClearGraphData
	}
    }

    $which magnifier off
    magnifier clear
}

proc MotionFrame {which x y} {
    global debug
    if {$debug(tcl,events)} {
	puts "MotionFrame"
    }

    DoMotion $which $x $y sizing fleur
}

proc ShiftMotionFrame {which x y} {
    global debug
    if {$debug(tcl,events)} {
	puts "ShiftMotionFrame"
    }

    DoMotion $which $x $y exchange fleur
}

proc ControlMotionFrame {which x y} {
    global debug
    if {$debug(tcl,events)} {
	puts "ControlMotionFrame"
    }

    DoMotion $which $x $y sizing draped_box
}

proc DoMotion {which x y cursor1 cursor2} {
    global ds9
    global current
    global panner

    # if button 3 is down, ignore this event, we are doing something already

    if {$ds9(b3) || $ds9(b2)} {
	return
    }

    switch -- $ds9(mode) {
	crosshair {}
	pointer {
	    if {$which == $current(frame)} {
		CursorMarker $which $x $y $cursor1 $cursor2
	    }

	    MotionInfoBox $which $x $y canvas
	    UpdatePixelTable $which $x $y canvas
	    UpdateGraph $which $x $y canvas
	}
	none -
	colorbar -
	pan -
	zoom -
	rotate -
	examine -
	illustrate -
	imexam {
	    MotionInfoBox $which $x $y canvas
	    UpdatePixelTable $which $x $y canvas
	    UpdateGraph $which $x $y canvas
	}
    }

    UpdateMagnifier $which $x $y
}

proc Button1Frame {which x y} {
    global ds9
    global current
    global marker

    global debug
    if {$debug(tcl,events)} {
	puts "Button1Frame"
    }

    # let others know that the mouse is down

    set ds9(b1) 1

    switch -- $ds9(mode) {
	none {
	    UpdateMagnifier $which $x $y
	}
	pointer {
	    if {$which == $current(frame)} {
		ButtonMarker $which $x $y
	    } else {
		# we need this cause MotionMarker maybe called, 
		# and we don't want it
		set marker(motion) none
		set marker(handle) -1

		set ds9(next) $which
		GotoFrame
	    }
	    UpdateMagnifier $which $x $y
	}
	crosshair {
	    ButtonCrosshair $which $x $y

	    UpdateInfoBox $which $x $y canvas
	    UpdatePixelTable $which $x $y canvas
	    UpdateGraph $which $x $y canvas
	    UpdateMagnifier $which $x $y
	}
	colorbar {}
	pan {
	    ButtonPan $which $x $y
	    UpdateMagnifier $which $x $y
	}
	zoom {
	    ButtonZoom $which $x $y
	    UpdateMagnifier $which $x $y
	}
	rotate {ButtonRotate $which $x $y}
	examine {ButtonExamine $which $x $y}
	illustrate {}
	imexam {
	    # we need this cause MotionMarker maybe called, 
	    # and we don't want it
	    set marker(motion) none
	    set marker(handle) -1

	    set ds9(imexam,which) $which
	    set ds9(imexam,x) $x
	    set ds9(imexam,y) $y

	    set ds9(mode) $ds9(imexam,mode)
	    set ds9(imexam) 0
	}
    }
}

proc ShiftButton1Frame {which x y} {
    global ds9
    global current

    global debug
    if {$debug(tcl,events)} {
	puts "ShiftButton1Frame"
    }

    # let others know that the mouse is down

    set ds9(sb1) 1

    switch -- $ds9(mode) {
	none {}
	pointer {
	    if {$which == $current(frame)} {
		ShiftMarker $which $x $y
	    }
	    UpdateMagnifier $which $x $y
	}
	crosshair {}
	colorbar {}
	pan {}
	zoom {ShiftZoom $which}
	rotate {}
	examine {}
	illustrate {}
	imexam {}
    }
}

proc ControlButton1Frame {which x y} {
    global ds9
    global current
    global marker

    global debug
    if {$debug(tcl,events)} {
	puts "ControlButton1Frame"
    }

    # let others know that the mouse is down

    set ds9(cb1) 1

    switch -- $ds9(mode) {
	none {}
	pointer {
	    if {$which == $current(frame)} {
		ControlMarker $which $x $y
	    } else {
		# we need this cause MotionMarker maybe called, 
		# and we don't want it
		set marker(motion) none
		set marker(handle) -1
	    }
	    UpdateMagnifier $which $x $y
	}
	crosshair {}
	colorbar {}
	pan {}
	zoom {}
	rotate {}
	examine {}
	illustrate {}
	imexam {}
    }
}

proc ControlShiftButton1Frame {which x y} {
    global ds9
    global current
    global marker

    global debug
    if {$debug(tcl,events)} {
	puts "ControlShiftButton1Frame"
    }

    # let others know that the mouse is down

    set ds9(csb1) 1

    switch -- $ds9(mode) {
	none {}
	pointer {
	    if {$which == $current(frame)} {
		ControlShiftMarker $which $x $y
	    } else {
		# we need this cause MotionMarker maybe called, 
		# and we don't want it
		set marker(motion) none
		set marker(handle) -1
	    }
	    UpdateMagnifier $which $x $y
	}
	crosshair {}
	colorbar {}
	pan {}
	zoom {}
	rotate {}
	examine {}
	illustrate {}
	imexam {}
    }
}

proc Motion1Frame {which x y} {
    global ds9
    global current
    global magnifier

    global debug
    if {$debug(tcl,events)} {
	puts "Motion1Frame"
    }

    # abort if we are here by accident (such as a double click)

    if {($ds9(b1) == 0) && ($ds9(sb1) == 0) && \
	    ($ds9(cb1) == 0) && ($ds9(csb1) == 0)} {
	return
    }

    switch -- $ds9(mode) {
	none {
	    UpdateMagnifier $which $x $y
	}
	pointer {
	    if {$which == $current(frame)} {
		MotionMarker $which $x $y
	    }
	    UpdateMagnifier $which $x $y
	}
	crosshair {
	    MotionCrosshair $which $x $y

	    UpdateInfoBox $which $x $y canvas
	    UpdatePixelTable $which $x $y canvas
	    UpdateGraph $which $x $y canvas
	    UpdateMagnifier $which $x $y
	}
	colorbar {}
	pan {
	    MotionPan $which $x $y
	    UpdateMagnifier $which $x $y
	}
	zoom {
	    UpdateMagnifier $which $x $y
	}
	rotate {MotionRotate $which $x $y}
	examine {}
	illustrate {}
	imexam {}
    }
}

proc Release1Frame {which x y} {
    global ds9
    global current

    global debug
    if {$debug(tcl,events)} {
	puts "Release1Frame"
    }

    # abort if we are here by accident (such as a double click)
    if {($ds9(b1) == 0) && ($ds9(sb1) == 0) && \
	    ($ds9(cb1) == 0) && ($ds9(csb1) == 0)} {
	return
    }

    switch -- $ds9(mode) {
	none {}
	pointer {
	    if {$which == $current(frame)} {
		ReleaseMarker $which $x $y
	    }
	}
	crosshair {
	    ReleaseCrosshair $which $x $y

	    UpdateInfoBox $which $x $y canvas
	    UpdatePixelTable $which $x $y canvas
	    UpdateGraph $which $x $y canvas
	}
	colorbar {}
	pan {ReleasePan $which $x $y}
	zoom {}
	rotate {ReleaseRotate $which $x $y}
	examine {}
	illustrate {}
	imexam {}
    }

    # let others know that the mouse is up
    set ds9(b1) 0
    set ds9(sb1) 0
    set ds9(cb1) 0
    set ds9(csb1) 0

    UpdateEditMenu
    UpdateMagnifier $which $x $y
}

proc Double1Frame {which x y} {
    global ds9
    global current

    global debug
    if {$debug(tcl,events)} {
	puts "Double1Frame"
    }

    switch -- $ds9(mode) {
	none {}
	pointer {
	    if {$which == $current(frame)} {
		DoubleMarker $which $x $y
		UpdateMagnifier $which $x $y
	    }
	}
	crosshair {}
	colorbar {}
	pan {}
	zoom {}
	rotate {}
	examine {}
	illustrate {}
	imexam {}
    }
}

proc DoubleRelease1Frame {which x y} {
    global ds9

    global debug
    if {$debug(tcl,events)} {
	puts "DoubleRelease1Frame"
    }

    switch -- $ds9(mode) {
	none {}
	pointer {}
	crosshair {}
	colorbar {}
	pan {}
	zoom {}
	rotate {}
	examine {}
	illustrate {}
	imexam {}
    }

    UpdateEditMenu
    UpdateMagnifier $which $x $y
}

proc Button2Frame {which x y} {
    global ds9

    global debug
    if {$debug(tcl,events)} {
	puts "Button2Frame"
    }

    set ds9(b2) 1
    ButtonPan $which $x $y
}

proc ShiftButton2Frame {which x y} {
    global ds9

    global debug
    if {$debug(tcl,events)} {
	puts "ShiftButton2Frame"
    }

    set ds9(sb2) 1
}

proc Motion2Frame {which x y} {
    global debug
    if {$debug(tcl,events)} {
	puts "Motion2Frame"
    }

    MotionPan $which $x $y
}

proc Release2Frame {which x y} {
    global ds9

    global debug
    if {$debug(tcl,events)} {
	puts "Release2Frame"
    }

    ReleasePan $which $x $y

    # let others know that the mouse is up

    set ds9(b2) 0
    set ds9(sb2) 0
}

proc DeleteKeyFrame {which x y} {
    global ds9

    global debug
    if {$debug(tcl,events)} {
	puts "DeleteKeyFrame"
    }

    switch -- $ds9(mode) {
	none {}
	pointer {DeleteMarker $which $x $y}
	crosshair {}
	colorbar {}
	pan {}
	zoom {}
	rotate {}
	examine {}
	illustrate {}
	imexam {}
    }

    UpdateEditMenu
}

proc ArrowKeyFrame {which x y} {
    global ds9

    global debug
    if {$debug(tcl,events)} {
	puts "ArrowKeyFrame"
    }

    switch -- $ds9(mode) {
	none {$which warp $x $y}
	pointer {
	    $which warp $x $y
	    $which marker move $x $y
	}
	pan {Pan [expr -$x] [expr -$y] canvas {}}
	zoom -
	rotate {}
	crosshair {
	    ArrowKeyCrosshair $which $x $y

	    set coord [$which get crosshair canvas]
	    set X [lindex $coord 0]
	    set Y [lindex $coord 1]

	    UpdateInfoBox $which $X $Y canvas
	    UpdatePixelTable $which $X $Y canvas
	    UpdateGraph $which $X $Y canvas
	}
	colorbar {}
	examine {}
	illustrate {}
	imexam {}
    }

    UpdateEditMenu
}

# Other Public Routines

proc CopyFrame {} {
    global current
    global marker

    if {$current(frame) != ""} {
	$current(frame) marker copy
	set marker(copy) $current(frame)
    }
    UpdateEditMenu
}

proc CutFrame {} {
    global current
    global marker

    if {$current(frame) != ""} {
	$current(frame) marker cut
	set marker(copy) $current(frame)
    }
    UpdateEditMenu
    UpdateGroupDialog
}

proc PasteFrame {} {
    global current
    global marker

    if {$current(frame) != {}} {
	# same frame?
	if {$current(frame) == $marker(copy)} {
	    $current(frame) marker paste

	    # any special callbacks?
	    foreach id [$current(frame) get marker select] {
		switch -- [$current(frame) get marker $id type] {
		    projection {SetupProjectionPlot $id}
		}
	    }
	} else {
	    PasteFrameDialog
	}
    }
	
    UpdateEditMenu
    UpdateGroupDialog
}

proc PasteFrameDialog {} {
    global mkld
    global current
    global marker
    global menu
    global ds9

    set w ".mkld"

    set mkld(ok) 0
    set mkld(system) WCS
    set mkld(sky) fk5

    DialogCreate $w "Paste Regions" -borderwidth 2
    frame $w.param -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.param $w.buttons  -side top -ipadx 4 -ipady 4 -fill x -expand true

    label $w.param.coordtitle -text "Coordinate System: "
    menubutton $w.param.coordbutton -relief raised \
	-menu $w.param.coordbutton.menu -textvariable mkld(system) \
	-width 8

    menu $w.param.coordbutton.menu -tearoff 0 -selectcolor $menu(selectcolor)
    $w.param.coordbutton.menu add radiobutton -label "WCS" \
	-variable mkld(system) -value "WCS"
    $w.param.coordbutton.menu add cascade -label "Multiple WCS" \
	-menu $w.param.coordbutton.menu.wcs
    $w.param.coordbutton.menu add separator
    $w.param.coordbutton.menu add radiobutton -label "Image" \
	-variable mkld(system) -value "Image"
    $w.param.coordbutton.menu add radiobutton -label "Physical" \
	-variable mkld(system) -value "Physical"
    if {$ds9(amp,det)} {
	$w.param.coordbutton.menu add radiobutton -label "Amplifier" \
	    -variable mkld(system) -value "Amplifier"
	$w.param.coordbutton.menu add radiobutton -label "Detector" \
	    -variable mkld(system) -value "Detector"
    }
    $w.param.coordbutton.menu add separator
    $w.param.coordbutton.menu add radiobutton -label "Equatorial B1950" \
	-variable mkld(sky) -value fk4
    $w.param.coordbutton.menu add radiobutton -label "Equatorial J2000" \
	-variable mkld(sky) -value fk5
    $w.param.coordbutton.menu add radiobutton -label "ICRS" \
	-variable mkld(sky) -value icrs
    $w.param.coordbutton.menu add radiobutton -label "Galactic" \
	-variable mkld(sky) -value galactic
    $w.param.coordbutton.menu add radiobutton -label "Ecliptic" \
	-variable mkld(sky) -value ecliptic

    menu $w.param.coordbutton.menu.wcs -tearoff $menu(tearoff) \
	-selectcolor $menu(selectcolor)
    foreach l {A B C D E F G H I J K L M N O P Q R S T U V W X Y Z} {
	$w.param.coordbutton.menu.wcs add radiobutton -label "WCS $l" \
	    -variable mkld(system) -value "wcs$l"
    }

    grid rowconfigure $w.param 0 -pad 0
    grid $w.param.coordtitle $w.param.coordbutton -padx 4 -sticky w

    button $w.buttons.ok -text "OK" -default active -command {set mkld(ok) 1}
    button $w.buttons.cancel -text "Cancel" -command {set mkld(ok) 0}
    pack $w.buttons.ok -side left -padx 10
    pack $w.buttons.cancel -side right -padx 10

    bind $w <Return> {set mkld(ok) 1}
    bind $w <Alt-o> "tkButtonInvoke $w.buttons.ok"
    bind $w <Alt-c> "tkButtonInvoke $w.buttons.cancel"

    DialogCenter $w 
    DialogWait $w mkld(ok)
    DialogDismiss $w

    if {$mkld(ok)} {
	global cmd
	set cmd "[$marker(copy) marker paste $mkld(system) $mkld(sky)]"
	$current(frame) marker fg command ds9 var cmd
    }

    # we want to destroy this window
    destroy $w 

    unset mkld
}

proc UndoFrame {} {
    global current

    if {$current(frame) != ""} {
	$current(frame) marker undo
    }
    UpdateEditMenu
    UpdateGroupDialog
}

proc FirstFrame {} {
    global ds9
    set ds9(next) [lindex $ds9(active) 0]
    GotoFrame
}

proc PrevFrame {} {
    global ds9
    global current

    set i [lsearch $ds9(active) $current(frame)]
    if {$i>0} {
	set ds9(next) [lindex $ds9(active) [expr $i-1]]
    } else {
	set last [expr [llength $ds9(active)] -1]
	set ds9(next) [lindex $ds9(active) $last]
    }
    GotoFrame
}

proc NextFrame {} {
    global ds9
    global current

    set i [lsearch $ds9(active) $current(frame)]
    set last [expr [llength $ds9(active)] -1]
    if {$i<$last} {
	set ds9(next) [lindex $ds9(active) [expr $i+1]]
    } else {
	set ds9(next) [lindex $ds9(active) 0]
    }
    GotoFrame
}

proc LastFrame {} {
    global ds9
    set ds9(next) [lindex $ds9(active) [expr [llength $ds9(active)]-1]]
    GotoFrame
}

proc UpdateActiveFrames {} {
    global ds9
    global active
    global current

    # reset active list

    set ds9(active) ""
    foreach f $ds9(frames) {
	if {$active($f)} {
	    lappend ds9(active) $f
	    $ds9(mb).frame entryconfig $f -state normal
	} else {
	    $ds9(mb).frame entryconfig $f -state disabled
	}

    }

    # New layout if needed

    if {[llength $ds9(active)] > 0} {
	if {[lsearch $ds9(active) $current(frame)] == -1} {
	    set current(frame) [lindex $ds9(active) 0]
	    set ds9(next) $current(frame)
	}
    }

    SetWatchCursor
    DisplayMode
    LayoutFrames
    UnsetWatchCursor
}

proc ActiveFrameAll {} {
    global ds9
    global active

    foreach f $ds9(frames) {
	set active($f) 1
    }
    UpdateActiveFrames
}

proc ActiveFrameNone {} {
    global ds9
    global active

    foreach f $ds9(frames) {
	set active($f) 0
    }
    UpdateActiveFrames
}

proc GotoFrame {} {
    global ds9
    global current
    global active

    if {$current(frame) != {} && $current(frame) != $ds9(next)} {
#	$current(frame) marker unselect all
	$current(frame) highlite off
	$current(frame) panner off

	switch -- $ds9(display,mode) {
	    blink -
	    single {
		$current(frame) hide
		UnBindEventsFrame $current(frame)
	    }
	    tile {}
	}
    }

    if {$current(frame) != $ds9(next)} {
	set current(frame) $ds9(next)
	FrameToFront $current(frame)
    }
}

proc DisplayMode {} {
    global ds9
    global blink

    switch -- $ds9(display,user) {
	single {set ds9(display,mode) $ds9(display,user)}
	tile -
	blink {
	    if {[llength $ds9(active)] > 1} {
		set ds9(display,mode) $ds9(display,user)
	    } else {
		set ds9(display,mode) single
	    }
	}
    }

    switch -- $ds9(display,mode) {
	single -
	tile {
	    # turn off blink if on
	    if {$blink(id)>0} {
		after cancel $blink(id)
		set blink(id) 0
		set blink(index) -1
	    }

	    SetWatchCursor
	    LayoutFrames
	    UnsetWatchCursor
	}
	blink {
	    # ignore if we are already blinking
	    if {$blink(id)==0} {
		SetWatchCursor
		LayoutFrames
		UnsetWatchCursor

		BlinkTimer
	    }
	}
    }
}

proc BlinkTimer {} {
    global blink
    global ds9

    if {$ds9(active) != ""} {
	incr blink(index)
	if {$blink(index) >= [llength $ds9(active)]} {
	    set blink(index) 0
	}
	set ds9(next) [lindex $ds9(active) $blink(index)]
	GotoFrame
    }

    set blink(id) [after $blink(interval) BlinkTimer]
}

proc ResetCurrentFrame {} {
    global current

    ResetFrame $current(frame)
}

proc ResetAllFrame {} {
    global ds9

    foreach f $ds9(frames) {
	ResetFrame $f
    }
}

proc ResetFrame {which} {
    if {$which != {}} {
	SetWatchCursor
	$which reset

	RefreshInfoBox $which
	ClearPixelTable
	ClearGraphData

	UnsetWatchCursor

	GridUpdateZoom
	UpdatePanZoomMenu
	UpdatePanZoomDialog
	UpdateScaleMenu
	UpdateScaleDialog

	UpdateGraphYAxis
	UpdateGraphXAxis
    }
}

proc ClearCurrentFrame {} {
    global current

    ClearFrame $current(frame)
}

proc ClearAllFrame {} {
    global ds9

    foreach f $ds9(frames) {
	ClearFrame $f
    }
}

proc ClearFrame {which} {
    global current
    global header

    if {$which != ""} {

	# delete any slave frames
	DeleteExamine $which

	DestroyHeader $which

	SetWatchCursor
	$which clear
	UnsetWatchCursor

	ClearInfoBox
	ClearPixelTable
	ClearGraphData

	UpdateDS9
    }
}

proc MatchFrames {which} {
    global ds9
    global current
    global wcs

    if {$current(frame) != ""} {
	SetWatchCursor

	set pan [$current(frame) get pan $which scientific]

	set wcs(align) 0
	$current(frame) wcs align $wcs(align) \
	    $wcs(align,system) $wcs(align,sky)
	foreach f $ds9(frames) {
	    if {$f != $current(frame)} {
		$f wcs align $wcs(align) \
		    $wcs(align,system) $wcs(align,sky)

		$f pan to $which $pan
		$f zoom to $current(zoom)
		$f orient $current(orient)
		$f rotate to $current(rotate)
	    }
	}

	UnsetWatchCursor
    }
}

proc MatchFramesWCS {} {
    global ds9
    global current
    global wcs

    if {$current(frame) != ""} {
	if  {[$current(frame) has wcs $wcs(align,system)]} {
	    SetWatchCursor

	    set wcs(align) 1
	    $current(frame) wcs align $wcs(align) \
		$wcs(align,system) $wcs(align,sky)
	    set zoom [$current(frame) get wcs zoom \
			  $wcs(align,system) scientific]
	    set pan [$current(frame) get pan $wcs(align,system) scientific]

	    foreach f $ds9(frames) {
		if {$f != $current(frame) && [$f has wcs $wcs(align,system)]} {
		    $f pan to $wcs(align,system) $pan
		    $f zoom to $current(zoom)
		    $f orient $current(orient)
		    $f rotate to $current(rotate)

		    $f wcs align $wcs(align) $wcs(align,system) $wcs(align,sky)
		    $f wcs zoom $wcs(align,system) $zoom
		}
	    }
	    UnsetWatchCursor
	}
    }
}

# Private Procedures

proc FrameToFront {which} {
    global ds9
    global current
    global canvas
    global colorbar
    global panner
    global info
    global blink

    global debug
    if {$debug(tcl,layout)} {
	puts "FrameToFront"
    }

    # process proper colorbar
    switch -- $ds9(visual) {
	pseudocolor {
	    colorbar show
	    set current(colorbar) colorbar
	    set colorbar(map) [colorbar get name]
	    set colorbar(invert) [colorbar get invert]
	    colorbar set colormap [$which get colormap]
	}
	truecolor {
	    switch -- [$which get type] {
		base {
		    colorbar show
		    colorbarrgb hide
		    set current(colorbar) colorbar
		    set colorbar(map) [colorbar get name]
		    set colorbar(invert) [colorbar get invert]
		    colorbar set colormap [$which get colormap]
		}
		rgb {
		    colorbar hide
		    colorbarrgb show
		    set current(colorbar) colorbarrgb
		    colorbarrgb set colormap rgb [$which get colormap]
		    colorbarrgb rgb channel [$which get rgb channel]
		}
	    }
	}
    }

    $canvas(name) raise $which

    $which show
    switch -- $ds9(display,mode) {
	single -
	blink {
	    RefreshInfoBox $which
	    if {!$ds9(freeze)} {
		BindEventsFrame $current(frame)
	    }
	}
	tile {$which highlite on}
    }

    if {$panner(view)} {
	$which panner on
    }

    UpdateDS9
}

proc LayoutFrames {} {
    global ds9
    global current
    global canvas
    global tile
    global panner
    global magnifier
    global colorbar

    global debug
    if {$debug(tcl,layout)} {
	puts "LayoutFrames"
    }

    set canvas(width) [expr [winfo width $canvas(name)] - 4]
    set canvas(height) [expr [winfo height $canvas(name)] - 4]

    # turn everything off

    foreach f $ds9(frames) {
	$f hide
	$f highlite off
	$f panner off
	$f magnifier off
	UnBindEventsFrame $f
    }

    set n [llength $ds9(active)]
    if {$n > 0} {
	switch -- $ds9(display,mode) {
	    single {TileOne}
	    tile {
		switch -- $tile(mode) {
		    row {TileRect 1 $n $tile(grid,gap)}
		    column {TileRect $n 1 $tile(grid,gap)}
		    grid {
			switch -- $tile(grid,mode) {
			    automatic {
				TileRect [expr int(sqrt($n-1))+1] \
				    [expr int(sqrt($n)+.5)] \
				    $tile(grid,gap)
			    }
			    manual {
				TileRect $tile(grid,row) \
				    $tile(grid,col) \
				    $tile(grid,gap)
			    }
			}
		    }
		}
	    }
	    blink {TileOne}
	}
    } else {
	set current(frame) ""
	set ds9(next) ""

	# panner
	if {$panner(view)} {
	    panner clear
	}

	# magnifier
	if {$magnifier(view)} {
	    magnifier clear
	}

	# process proper colorbar
	switch -- $ds9(visual) {
	    pseudocolor {
		colorbar show
		set current(colorbar) colorbar
		set colorbar(map) [colorbar get name]
		set colorbar(invert) [colorbar get invert]
	    }
	    truecolor {
		colorbar show
		colorbarrgb hide
		set current(colorbar) colorbar
		set colorbar(map) [colorbar get name]
		set colorbar(invert) [colorbar get invert]
	    }
	}

	# update menus/dialogs
	UpdateDS9
    }
}

# This procedure is called when we have only 1 frames to display
# Canvas Coordinate system is 1 to canvas(width), canvas(height)

proc TileOne {} {
    global ds9
    global current
    global canvas

    global debug
    if {$debug(tcl,layout)} {
	puts "TileOne"
    }

    set w $canvas(width)
    set h $canvas(height)

    set x [expr int($canvas(width)/2.) + 2]
    set y [expr int($canvas(height)/2.) + 2]

    foreach f $ds9(active) {
	$f configure -x $x -y $y -width $w -height $h -anchor center
    }
    # only show the current frame
    $current(frame) show
    FrameToFront $current(frame)
}

proc TileRect {numx numy gap} {
    global ds9
    global canvas

    global debug
    if {$debug(tcl,layout)} {
	puts "TileRect"
    }

    set w [expr $canvas(width) /$numx - $gap]
    set h [expr $canvas(height)/$numy - $gap]
    
    for {set i 0; set b 1} {$i < $numy} {incr i; incr b 2} {
	for {set j 0; set a 1} {$j < $numx} {incr j; incr a 2} {
	    set c [expr $i*$numx + $j]
	    set x($c) [expr int($canvas(width) *$a/(2.*$numx)) + 2]
	    set y($c) [expr int($canvas(height)*$b/(2.*$numy)) + 2]
	}
    }

    TileIt $w $h x y [expr $numx*$numy]
}

proc TileIt {w h xx yy nn} {
    upvar $xx x
    upvar $yy y
    global ds9
    global current
    global canvas

    global debug
    if {$debug(tcl,layout)} {
	puts "TileIt"
    }

    set i 0
    foreach f $ds9(active) {
	if {$i<$nn} {
	    $f configure -x $x($i) -y $y($i) -width $w -height $h \
		-anchor center
	    $f show
	    $canvas(name) raise $f
	    if {!$ds9(freeze)} {
		BindEventsFrame $f
	    }
	}
	incr i
    }

    # if manual grid, current frame could be not included
    if {[llength $ds9(active)] > $nn} {
	set current(frame) [lindex $ds9(active) 0]
    }
    FrameToFront $current(frame)
}

proc UpdateEditMenu {} {
    global ds9
    global current
    global buttons
    global marker

    global debug
    if {$debug(tcl,update)} {
	puts "UpdateEditMenu"
    }

    switch -- $ds9(mode) {
	pointer {
	    if {$current(frame) != ""} {
		set l [$current(frame) has marker undo]
		if {$l != ""} {
		    $ds9(mb).edit entryconfig 1 \
			-label "Undo [string totitle $l]" -state normal

		    $buttons(name).edit.undo configure -state normal
		} else {
		    $ds9(mb).edit entryconfig 1 -label "Undo" -state disabled
		    $buttons(name).edit.undo configure -state disabled
		}

		if {[$current(frame) has marker select]} {
		    $ds9(mb).edit entryconfig "Cut" -state normal
		    $ds9(mb).edit entryconfig "Copy" -state normal
		    $buttons(name).edit.cut configure -state normal
		    $buttons(name).edit.copy configure -state normal
		} else {
		    $ds9(mb).edit entryconfig "Cut" -state disabled
		    $ds9(mb).edit entryconfig "Copy" -state disabled
		    $buttons(name).edit.cut configure -state disabled
		    $buttons(name).edit.copy configure -state disabled
		}

		if {$marker(copy) != {} } {
		    if {[$marker(copy) has marker paste]} {
			$ds9(mb).edit entryconfig "Paste" -state normal
			$buttons(name).edit.paste configure -state normal
		    } else {
			$ds9(mb).edit entryconfig "Paste" -state disabled
			$buttons(name).edit.paste configure -state disabled
		    }
		} else {
		    $ds9(mb).edit entryconfig "Paste" -state disabled
		    $buttons(name).edit.paste configure -state disabled
		}
	    } else {
		$ds9(mb).edit entryconfig 1 -label "Undo" -state disabled
		$ds9(mb).edit entryconfig "Cut" -state disabled
		$ds9(mb).edit entryconfig "Copy" -state disabled
		$ds9(mb).edit entryconfig "Paste" -state disabled

		$buttons(name).edit.undo configure -state disabled
		$buttons(name).edit.cut configure -state disabled
		$buttons(name).edit.copy configure -state disabled
		$buttons(name).edit.paste configure -state disabled
	    }
	}
	none -
	crosshair -
	colorbar -
	pan -
	zoom -
	rotate -
	examine -
	illustrate -
	imexam {$ds9(mb).edit entryconfig 1 -label "Undo" -state disabled}
    }
}

proc UpdateFrameMenu {} {
    global ds9
    global current
    global menu
    global buttons
    global rgb

    global debug
    if {$debug(tcl,update)} {
	puts "UpdateFrameMenu"
    }

    switch -- $ds9(visual) {
	pseudocolor {
	    $ds9(mb).frame entryconfig $menu(frame,newrgb) -state disabled
	    $buttons(name).frame.newrgb configure -state disabled
	}
	truecolor {
	    $ds9(mb).frame entryconfig $menu(frame,newrgb) -state normal
	    $buttons(name).frame.newrgb configure -state normal
	}
    }

    if {$current(frame) != ""} {
	$ds9(mb).frame entryconfig $menu(frame,delete) -state normal
	$ds9(mb).frame entryconfig $menu(frame,deleteall) -state normal
	$ds9(mb).frame entryconfig $menu(frame,reset) -state normal
	$ds9(mb).frame entryconfig $menu(frame,refresh) -state normal
	$ds9(mb).frame entryconfig $menu(frame,single) -state normal
	$ds9(mb).frame entryconfig $menu(frame,tile) -state normal
	$ds9(mb).frame entryconfig $menu(frame,blink) -state normal
	$ds9(mb).frame entryconfig $menu(frame,tileparam) -state normal
	$ds9(mb).frame entryconfig $menu(frame,blinkparam) -state normal
	$ds9(mb).frame entryconfig $menu(frame,matchcolorbars) -state normal

	if {[$current(frame) get type] == "rgb"} {
	    $ds9(mb).frame entryconfig $menu(frame,rgb) -state normal
	} else {
	    $ds9(mb).frame entryconfig $menu(frame,rgb) -state disabled
	}

	$buttons(name).frame.delete configure -state normal
	$buttons(name).frame.deleteall configure -state normal
	$buttons(name).frame.reset configure -state normal
	$buttons(name).frame.single configure -state normal
	$buttons(name).frame.tile configure -state normal
	$buttons(name).frame.blink configure -state normal
	$buttons(name).frame.first configure -state normal
	$buttons(name).frame.prev configure -state normal
	$buttons(name).frame.next configure -state normal
	$buttons(name).frame.last configure -state normal

    } else {
	$ds9(mb).frame entryconfig $menu(frame,delete) -state disabled
	$ds9(mb).frame entryconfig $menu(frame,deleteall) -state disabled
	$ds9(mb).frame entryconfig $menu(frame,reset) -state disabled
	$ds9(mb).frame entryconfig $menu(frame,refresh) -state disabled
	$ds9(mb).frame entryconfig $menu(frame,single) -state disabled
	$ds9(mb).frame entryconfig $menu(frame,tile) -state disabled
	$ds9(mb).frame entryconfig $menu(frame,blink) -state disabled
	$ds9(mb).frame entryconfig $menu(frame,tileparam) -state disabled
	$ds9(mb).frame entryconfig $menu(frame,blinkparam) -state disabled
	$ds9(mb).frame entryconfig $menu(frame,matchcolorbars) -state disabled
	$ds9(mb).frame entryconfig $menu(frame,rgb) -state disabled

	$buttons(name).frame.delete configure -state disabled
	$buttons(name).frame.deleteall configure -state disabled
	$buttons(name).frame.reset configure -state disabled
	$buttons(name).frame.single configure -state disabled
	$buttons(name).frame.tile configure -state disabled
	$buttons(name).frame.blink configure -state disabled
	$buttons(name).frame.first configure -state disabled
	$buttons(name).frame.prev configure -state disabled
	$buttons(name).frame.next configure -state disabled
	$buttons(name).frame.last configure -state disabled
    }
}

# Frame Menu Delete
proc UpdateFrameMenuItems {} {
    global ds9

    if {[$ds9(mb).frame index end]>28} {
	$ds9(mb).frame delete 29 end
    }
    if {[$ds9(mb).frame.active index end]>3} {
	$ds9(mb).frame.active delete 4 end
    }

    foreach f $ds9(frames) {
	$ds9(mb).frame add radiobutton -label $f \
	    -variable ds9(next) -value $f -command GotoFrame
	$ds9(mb).frame.active add checkbutton -label $f \
	    -variable active($f) -command UpdateActiveFrames
    }
}

proc TileDialog {} {
    global tile
    global ds9
    global menu

    # see if we already have a window visible
    if [winfo exist $tile(top)] {
	raise $tile(top)
	return
    }

    set w $tile(top)
    set title "Tile Parameters"

    toplevel $w -colormap $ds9(main)
    wm title $w $title
    wm iconname $w $title
    wm group $w $ds9(top)
    wm protocol $w WM_DELETE_WINDOW TileDestroyDialog

    $w configure -menu $tile(mb)

    set tile(gd,mode) $tile(grid,mode)
    set tile(gd,row) $tile(grid,row)
    set tile(gd,col) $tile(grid,col)
    set tile(gd,gap) $tile(grid,gap)

    menu $tile(mb) -tearoff 0
    $tile(mb) add cascade -label "File" -menu $tile(mb).file

    menu $tile(mb).file -tearoff 0 -selectcolor $menu(selectcolor)
    $tile(mb).file add command -label "Close" -command TileDestroyDialog

    frame $w.param -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.param $w.buttons -side top -ipadx 4 -ipady 4 -fill x -expand true

    label $w.param.t1 -text "Grid"
    radiobutton $w.param.auto -text "Automatic" \
	-selectcolor $menu(selectcolor) \
	-variable tile(gd,mode) -value automatic
    radiobutton $w.param.manual -text "Manual" \
	-selectcolor $menu(selectcolor) \
	-variable tile(gd,mode) -value manual

    label $w.param.t2 -text "Frame Layout"
    entry $w.param.row -textvariable tile(gd,row) -width 6
    label $w.param.t2x -text "x"
    entry $w.param.col -textvariable tile(gd,col) -width 6

    label $w.param.t3 -text "Frame Gap"
    entry $w.param.gap -textvariable tile(gd,gap) -width 6
    label $w.param.t4 -text "pixels"

    grid $w.param.t1 $w.param.auto - - $w.param.manual \
	-padx 4 -pady 2 -sticky w
    grid $w.param.t2 $w.param.row $w.param.t2x $w.param.col \
	-padx 4 -pady 2 -sticky w
    grid $w.param.t3 $w.param.gap - $w.param.t4 -padx 4 -pady 2 -sticky w

    # Buttons

    button $w.buttons.apply -text "Apply" -command TileApplyDialog
    button $w.buttons.close -text "Close" -command TileDestroyDialog
    pack $w.buttons.apply $w.buttons.close -side left -padx 10 -expand true
}

proc TileDestroyDialog {} {
    global tile

    destroy $tile(top)
    destroy $tile(mb)

    unset tile(gd,mode)
    unset tile(gd,row)
    unset tile(gd,col)
    unset tile(gd,gap)
}

proc TileApplyDialog {} {
    global tile

    set tile(mode) grid
    set tile(grid,mode) $tile(gd,mode)
    set tile(grid,row) $tile(gd,row)
    set tile(grid,col) $tile(gd,col)
    set tile(grid,gap) $tile(gd,gap)
    
    DisplayMode
}

# Process Cmds

proc ProcessFrameCmd {varname iname} {
    upvar $varname var
    upvar $iname i

    global current
    global ds9
    global active
    global rgb

    catch {
	switch -- [string tolower [lindex $var $i]] {
	    center {
		incr i
		switch -- [lindex $var $i] {
		    all {CenterAllFrame}
		    {} {CenterCurrentFrame; incr i -1}
		    default {
			set f "Frame[lindex $var $i]"
			CenterFrame $f
		    }
		}
	    }
	    clear {
		incr i
		switch -- [lindex $var $i] {
		    all {ClearAllFrame}
		    {} {ClearCurrentFrame; incr i -1}
		    default {
			set f "Frame[lindex $var $i]"
			ClearFrame $f
		    }
		}
	    }
	    delete {
		incr i
		switch -- [lindex $var $i] {
		    all {DeleteAllFrames}
		    {} {DeleteCurrentFrame; incr i -1}
		    default {
			set f "Frame[lindex $var $i]"
			DeleteSingleFrame $f
		    }
		}
	    }
	    new {
		incr i
		switch -- [lindex $var $i] {
		    rgb {CreateRGBFrame}
		    default {CreateFrame; incr i -1}
		}
	    }
	    reset {
		incr i
		switch -- [lindex $var $i] {
		    all {ResetAllFrame}
		    {} {ResetCurrentFrame; incr i -1}
		    default {
			set f "Frame[lindex $var $i]"
			ResetFrame $f
		    }
		}
	    }
	    refresh {
		incr i
		switch -- [lindex $var $i] {
		    all {UpdateAllFrame}
		    {} {UpdateCurrentFrame; incr i -1}
		    default {
			set f "Frame[lindex $var $i]"
			UpdateFrame $f
		    }
		}
	    }
	    hide {
		incr i
		switch -- [lindex $var $i] {
		    all {ActiveFrameNone}
		    {} {
			set active($current(frame)) 0
			UpdateActiveFrames
			incr i -1
		    }
		    default {
			set f "Frame[lindex $var $i]"
			set active($f) 0
			UpdateActiveFrames
		    }
		}
	    }
	    show {
		incr i
		switch -- [lindex $var $i] {
		    all {ActiveFrameAll}
		    default {
			set f "Frame[lindex $var $i]"
			set active($f) 1
			UpdateActiveFrames
		    }
		}
	    }
	    first {FirstFrame}
	    next {NextFrame}
	    prev {PrevFrame}
	    last {LastFrame}
	    frameno {
		incr i
		CreateGotoFrame [lindex $var $i] base
	    }
	    default {CreateGotoFrame [lindex $var $i] base}
	}
    }
}

proc ProcessSingleCmd {varname iname} {
    upvar $varname var
    upvar $iname i

    global ds9
    set ds9(display,user) single
    DisplayMode
}

proc ProcessTileCmd {varname iname} {
    upvar $varname var
    upvar $iname i

    global ds9
    global tile

    switch -- [string tolower [lindex $var $i]] {
	mode {
	    incr i
	    set tile(mode) [lindex $var $i]
	}
	grid {
	    incr i
	    switch -- [string tolower [lindex $var $i]] {
		mode {
		    incr i
		    set tile(grid,mode) [lindex $var $i]
		}
		layout {
		    incr i
		    set tile(grid,row) [lindex $var $i]
		    incr i
		    set tile(grid,col) [lindex $var $i]
		}
		gap {
		    incr i
		    set tile(grid,gap) [lindex $var $i]
		}
		default {
		    set tile(mode) grid
		}
	    }
	}
	column {
	    set tile(mode) column
	}
	row {
	    set tile(mode) row
	}

	yes -
	true -
	1 -
	no -
	false -
	0 {
	    if {[FromYesNo [lindex $var $i]]} {
		set ds9(display,user) tile
	    } else {
		set ds9(display,user) single
	    }
	}
	default {
	    set ds9(display,user) tile
	    incr i -1
	}
    }
    DisplayMode
}

proc ProcessBlinkCmd {varname iname} {
    upvar $varname var
    upvar $iname i

    global ds9
    set ds9(display,user) blink
    DisplayMode
}

proc ProcessMatchCmd {varname iname} {
    upvar $varname var
    upvar $iname i

    RealizeDS9
    switch -- [string tolower [lindex $var $i]] {
	frames {
	    incr i
	    switch -- [string tolower [lindex $var $i]] {
		image -
		physical {MatchFrames [lindex $var $i]}
		wcs {MatchFramesWCS}
	    }
	}
	colorbars {MatchColorbars}
	scales {MatchScales}
    }
}

proc ProcessLockCmd {varname iname} {
    upvar $varname var
    upvar $iname i

    global crosshair

    RealizeDS9
    switch -- [string tolower [lindex $var $i]] {
	crosshairs {
	    incr i
	    switch -- [string range [string tolower [lindex $var $i]] 0 2] {
		ima -
		phy -
		amp -
		det -
		wcs {set crosshair(lock) [lindex $var $i]}
	    }
	}
    }
}

