#  Copyright (C) 1999-2004
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

proc EllipseAnnulusDialog {frame id} {
    global marker
    global ds9

    # see if we already have a header window visible

    set w ".marker$id"
    set mb ".mb$id"

    if [winfo exist $w] {
	raise $w
	return
    }

    # Init common items

    InitMarkerDialog $frame $id
    PropMenuEllipseAnnulusDialog $frame $mb $id
    CoordMenuMarkerDialog $frame $mb $id CoordEllipseAnnulusCB
    CoordMarkerCB $frame $id

    # file menu items

    $mb.file add command -label "Apply" \
	-command "ApplyEllipseAnnulusDialog $frame $id"
    $mb.file add separator
    $mb.file add command -label "Close" \
	-command "CloseEllipseAnnulusDialog $frame $id"

    # annulus menu items

    set marker($frame,$id,rcoord) $marker(dialog,dist,system)
    set marker($frame,$id,rlabel) $marker(dialog,dist,system)
    set marker($frame,$id,rformat) $marker(dialog,dist,format)

    DistMenuMarkerDialog $frame $mb $id DistEllipseAnnulusCB radius \
	rcoord rformat
    $mb add cascade -label Radius -menu $mb.radius

    set marker($frame,$id,method) dist
    AnnuliMenuMarkerDialog $frame $mb $id method
    $mb add cascade -label Method -menu $mb.method

    # annulus specific callbacks

    $frame marker $id callback edit EditEllipseAnnulusCB $frame
    $frame marker $id callback rotate RotateMarkerCB $frame
    $frame marker $id callback delete DeleteEllipseAnnulusDialog $frame

    # Dialog

    set type [string toupper "[$frame get marker $id type]"]

    toplevel $w -colormap $ds9(main)
    wm title $w $type
    wm iconname $w $type
    wm group $w $ds9(top)
    wm protocol $w WM_DELETE_WINDOW "CloseEllipseAnnulusDialog $frame $id"

    $w configure -menu $mb

    frame $w.ref
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.ref -fill x
    pack $w.buttons -fill x -ipadx 4 -ipady 4

    frame $w.ref.left
    frame $w.ref.right -relief groove -borderwidth 2
    pack $w.ref.left $w.ref.right -side left -fill y

    frame $w.ref.left.top -relief groove -borderwidth 2
    frame $w.ref.left.top.f
    frame $w.ref.left.bottom -relief groove -borderwidth 2
    frame $w.ref.left.bottom.f
    pack $w.ref.left.top.f $w.ref.left.bottom.f -anchor w -padx 4 -pady 4
    pack $w.ref.left.top $w.ref.left.bottom -fill both -expand true

    # ID

    label $w.ref.left.top.f.idTitle -text "Id"
    label $w.ref.left.top.f.idValue -text "$id"

    # Text

    label $w.ref.left.top.f.textTitle -text "Text"
    entry $w.ref.left.top.f.textValue -textvariable marker($frame,$id,text) \
	-width 40 

    grid $w.ref.left.top.f.idTitle $w.ref.left.top.f.idValue -padx 4 -sticky w
    grid $w.ref.left.top.f.textTitle $w.ref.left.top.f.textValue \
	-padx 4 -sticky w

    # Center

    label $w.ref.left.bottom.f.centerTitle -text "Center"
    entry $w.ref.left.bottom.f.centerX -textvariable marker($frame,$id,x) \
	-width 13 
    entry $w.ref.left.bottom.f.centerY -textvariable marker($frame,$id,y) \
	-width 13 
    label $w.ref.left.bottom.f.centerCoord -relief groove -width 9 \
	-padx 4 -textvariable marker($frame,$id,system)

    # Radius

    label $w.ref.left.bottom.f.majorTitle -text "Major"
    label $w.ref.left.bottom.f.minorTitle -text "Minor"

    label $w.ref.left.bottom.f.innerTitle -text "Outer Radius"
    entry $w.ref.left.bottom.f.radius1 \
	-textvariable marker($frame,$id,radius1) \
	-width 13 
    entry $w.ref.left.bottom.f.radius2 \
	-textvariable marker($frame,$id,radius2) \
	-width 13 
    label $w.ref.left.bottom.f.radiusformat \
	-textvariable marker($frame,$id,rlabel) -relief groove -width 9 -padx 4

    label $w.ref.left.bottom.f.outerTitle -text "Inner Radius"
    entry $w.ref.left.bottom.f.radius3 \
	-textvariable marker($frame,$id,radius3) -width 13 

    label $w.ref.left.bottom.f.annuliTitle -text "Annuli"
    entry $w.ref.left.bottom.f.annuli -textvariable marker($frame,$id,annuli) \
	-width 13 

    label $w.ref.left.bottom.f.angleTitle -text "Angle"
    entry $w.ref.left.bottom.f.angle -textvariable marker($frame,$id,angle) \
	-width 13 
    label $w.ref.left.bottom.f.angleUnit -relief groove -width 9 \
	-padx 4 -text "degrees"

    grid $w.ref.left.bottom.f.centerTitle $w.ref.left.bottom.f.centerX \
	$w.ref.left.bottom.f.centerY $w.ref.left.bottom.f.centerCoord \
	-padx 4 -sticky w
    grid x $w.ref.left.bottom.f.majorTitle $w.ref.left.bottom.f.minorTitle \
	-padx 4 -sticky w
    grid $w.ref.left.bottom.f.innerTitle $w.ref.left.bottom.f.radius1 \
	$w.ref.left.bottom.f.radius2 $w.ref.left.bottom.f.radiusformat \
	-padx 4 -sticky w
    grid $w.ref.left.bottom.f.outerTitle $w.ref.left.bottom.f.radius3 \
	-padx 4 -sticky w
    grid $w.ref.left.bottom.f.annuliTitle $w.ref.left.bottom.f.annuli \
	-padx 4 -sticky w

    grid $w.ref.left.bottom.f.angleTitle $w.ref.left.bottom.f.angle x \
	$w.ref.left.bottom.f.angleUnit -padx 4 -sticky w

    # Right

    frame $w.ref.right.title
    frame $w.ref.right.value
    pack $w.ref.right.title $w.ref.right.value -side top -padx 4 -pady 4

    label $w.ref.right.title.title -text "Annuli"
    label $w.ref.right.title.radiusformat \
	-textvariable marker($frame,$id,rlabel) -relief groove -width 9 -padx 4

    grid $w.ref.right.title.title $w.ref.right.title.radiusformat \
	-padx 4 -sticky news

    set txtscr $w.ref.right.value.yscroll
    set marker($frame,$id,annulitxt) $w.ref.right.value.txt
    text $marker($frame,$id,annulitxt) -height 10 -width 15 -wrap none \
	-font {courier 12} -yscrollcommand "$txtscr set"
    scrollbar $txtscr -command [list $marker($frame,$id,annulitxt) yview] \
	-orient vertical

    grid $marker($frame,$id,annulitxt) $txtscr -sticky news

    # Buttons

    button $w.buttons.apply -text "Apply" \
	-command "ApplyEllipseAnnulusDialog $frame $id"
    button $w.buttons.generate -text "Generate" \
	-command "EllipseAnnulusUpdateRadius $frame $id"
    button $w.buttons.close -text "Close" \
	-command "CloseEllipseAnnulusDialog $frame $id"
    pack $w.buttons.apply $w.buttons.generate $w.buttons.close \
	-side left -padx 10 -expand true

    # some window managers need a hint
    raise $w

    # Init annulus variables

    EditEllipseAnnulusCB $frame $id
    RotateMarkerCB $frame $id
    DistEllipseAnnulusCB $frame $id
}

proc PropMenuEllipseAnnulusDialog {frame mb id} {
    global menu
    global marker

    menu $mb.properties -tearoff 0 -selectcolor $menu(selectcolor)
    $mb.properties add checkbutton -label "Can Edit" \
	-variable marker($frame,$id,edit) \
	-command "PropertyMarkerDialog $frame $id edit"
    $mb.properties add checkbutton -label "Can Move" \
	-variable marker($frame,$id,move) \
	-command "PropertyMarkerDialog $frame $id move"
    $mb.properties add checkbutton -label "Can Delete" \
	-variable marker($frame,$id,delete) \
	-command "PropertyMarkerDialog $frame $id delete"
    $mb.properties add checkbutton -label "Fixed in Size" \
	-variable marker($frame,$id,fixed) \
	-command "PropertyMarkerDialog $frame $id fixed"
    $mb.properties add separator
    $mb.properties add radiobutton -label Include \
	-variable marker($frame,$id,include) -value 1 \
	-command "PropertyMarkerDialog $frame $id include"
    $mb.properties add radiobutton -label Exclude \
	-variable marker($frame,$id,include) -value 0 \
	-command "PropertyMarkerDialog $frame $id include"
    $mb.properties add separator
    $mb.properties add radiobutton -label Source \
	-variable marker($frame,$id,source) -value 1 \
	-command "PropertyMarkerDialog $frame $id source"
    $mb.properties add radiobutton -label Background \
	-variable marker($frame,$id,source) -value 0 \
	-command "PropertyMarkerDialog $frame $id source"
}

proc ApplyEllipseAnnulusDialog {frame id} {
    EditEllipseAnnulusDialog $frame $id
    RotateMarkerDialog $frame $id
    ApplyMarkerDialog $frame $id
}

proc CloseEllipseAnnulusDialog {frame id} {
    global marker

    $frame marker $id delete callback edit EditEllipseAnnulusCB
    $frame marker $id delete callback rotate RotateMarkerCB
    $frame marker $id delete callback delete DeleteEllipseAnnulusDialog
    DeleteMarkerCBs $frame $id

    DeleteEllipseAnnulusDialog $frame $id
}

proc DeleteEllipseAnnulusDialog {frame id} {
    global marker

    DeleteMarkerDialog $frame $id

    unset marker($frame,$id,radius1)
    unset marker($frame,$id,radius2)
    unset marker($frame,$id,radius3)
    unset marker($frame,$id,annuli)
    unset marker($frame,$id,annulitxt)
    unset marker($frame,$id,rcoord)
    unset marker($frame,$id,rformat)
    unset marker($frame,$id,rlabel)
    unset marker($frame,$id,angle)
    unset marker($frame,$id,method)
}

proc EditEllipseAnnulusDialog {frame id} {
    global marker

    set levels ""
    regsub -all "\n" "[$marker($frame,$id,annulitxt) get 1.0 end]" " " levels
    # and trim any trailing spaces
    set levels [string trimright $levels " "]

    if {$levels != ""} {
	$frame marker $id ellipse annulus radius "\{$levels\}" \
	    $marker($frame,$id,rcoord) $marker($frame,$id,rformat)
    }
}

proc EditEllipseAnnulusCB {frame id} {
    global marker

    set t [$frame get marker $id ellipse annulus radius \
	       $marker($frame,$id,rcoord) $marker($frame,$id,rformat)]

    set last [llength $t]
    set marker($frame,$id,annuli) [expr $last/2-1]
    set marker($frame,$id,radius1) [lindex $t [expr $last-2]]
    set marker($frame,$id,radius2) [lindex $t [expr $last-1]]
    set marker($frame,$id,radius3) [lindex $t 0]

    $marker($frame,$id,annulitxt) delete 1.0 end
    $marker($frame,$id,annulitxt) insert end "$t"
}

proc CoordEllipseAnnulusCB {frame id} {
    CoordMarkerCB $frame $id

    MoveMarkerCB $frame $id
    EditEllipseAnnulusCB $frame $id
    RotateMarkerCB $frame $id
}

proc DistEllipseAnnulusCB {frame id} {
    global marker
    set mb ".mb$id"

    AdjustDist $frame marker($frame,$id,rcoord)

    set marker($frame,$id,rlabel) $marker($frame,$id,rcoord)
    switch -- $marker($frame,$id,rcoord) {
	image -
	physical -
	amplifier -
	detector {}
	default {
	    if [$frame has wcs $marker($frame,$id,rcoord)] {
		if [$frame has wcs equatorial $marker($frame,$id,rcoord)] {
		    set marker($frame,$id,rlabel) $marker($frame,$id,rformat)
		} else {
		    set name [$frame get wcs name $marker($frame,$id,rcoord)]
		    if {$name != ""} {
			set marker($frame,$id,rlabel) $name
		    }
		}
	    }
	}
    }

    MoveMarkerCB $frame $id
    EditEllipseAnnulusCB $frame $id
}

proc EllipseAnnulusUpdateRadius {frame id} {
    global marker

    $marker($frame,$id,annulitxt) delete 1.0 end

    if {$marker($frame,$id,annuli) < 1} {
	set marker($frame,$id,annuli) 1
    }

    set radius1 $marker($frame,$id,radius1)
    set radius2 $marker($frame,$id,radius2)
    set radius3 $marker($frame,$id,radius3)
    set annuli $marker($frame,$id,annuli)

    if {($radius1 != "") && ($radius2 != "") && \
        ($radius3 != "") && ($annuli != "")} {

	switch -- $marker($frame,$id,method) {
	    dist {
		for {set i 0} {$i<=$annuli} {incr i} {
		    set major [expr ((($radius1-$radius3)/double($annuli))*$i)\
				   +$radius3]
		    set minor [expr $major*$radius2/$radius1]
		    $marker($frame,$id,annulitxt) insert end "$major $minor\n"
		}
	    }

	    area {
		set pi 3.14159265358979323846
		set r [expr double($radius2)/$radius1]
		set area [expr $pi*(($radius1*$radius2)-($radius3*$radius3*$r))\
			      /$annuli]

		set major0 $radius3
		set minor0 [expr $radius3*$r]
		$marker($frame,$id,annulitxt) insert end "$major0 $minor0\n"
		for {set i 0} {$i<$annuli} {incr i} {
		    set major1 [expr sqrt(($area+($pi*$major0*$minor0)) / \
					      ($pi*$r))]
		    set minor1 [expr $major1*$r]
		    $marker($frame,$id,annulitxt) insert end \
			[format "%.4f %.4f\n" $major1 $minor1]
		    set major0 $major1
		    set minor0 $minor1
		}
	    }
	}
    }
}

proc EllipseAnnulusDefaultDialog {} {
    global marker
    global ed

    set w ".ellipseannulus"

    set ed(ok) 0
    set ed(radius1) $marker(ellipseannulus,radius1)
    set ed(radius2) $marker(ellipseannulus,radius2)
    set ed(radius3) $marker(ellipseannulus,radius3)
    set ed(annuli) $marker(ellipseannulus,annuli)

    DialogCreate $w "Default Elliptical Annulus" -borderwidth 2
    frame $w.ed  -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.ed $w.buttons -fill x -ipadx 4 -ipady 4

    label $w.ed.majorTitle -text "Major"
    label $w.ed.minorTitle -text "Minor"

    label $w.ed.innerTitle -text "Inner Radius"
    entry $w.ed.radius1 -textvariable ed(radius1) -width 10 	
    entry $w.ed.radius2 -textvariable ed(radius2) -width 10 
    label $w.ed.unit -text "image" -relief groove -width 8

    label $w.ed.outerTitle -text "Outer Radius"
    entry $w.ed.radius3 -textvariable ed(radius3) -width 10 

    label $w.ed.annuliTitle -text "Annuli"
    entry $w.ed.annuli -textvariable ed(annuli) -width 10 
    
    grid x $w.ed.majorTitle $w.ed.minorTitle -padx 4 -sticky w
    grid $w.ed.innerTitle $w.ed.radius1 $w.ed.radius2 $w.ed.unit \
	-padx 4 -sticky w
    grid $w.ed.outerTitle $w.ed.radius3 -padx 4 -sticky w
    grid $w.ed.annuliTitle $w.ed.annuli -padx 4 -sticky w

    button $w.buttons.ok -text "OK" -default active -command {set ed(ok) 1}
    button $w.buttons.cancel -text "Cancel" -command {set ed(ok) 0}
    pack $w.buttons.ok -side left -padx 10
    pack $w.buttons.cancel -side right -padx 10

    bind $w <Return> {set ed(ok) 1}
    bind $w <Alt-o> "tkButtonInvoke $w.buttons.ok"
    bind $w <Alt-c> "tkButtonInvoke $w.buttons.cancel"

    DialogCenter $w 
    $w.ed.radius1 select range 0 end
    DialogWait $w ed(ok) $w.ed.radius1
    DialogDismiss $w

    if {$ed(ok)} {
	set marker(ellipseannulus,radius1) $ed(radius1)
	set marker(ellipseannulus,radius2) $ed(radius2)
	set marker(ellipseannulus,radius3) $ed(radius3)
	set marker(ellipseannulus,annuli) $ed(annuli)
    }

    unset ed
}
