# Samizdat request handling
#
#   Copyright (c) 2002-2009  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 3 or later.
#
# vim: et sw=2 sts=2 ts=8 tw=0

require 'samizdat/engine'
require 'zlib'

# CGI request and response handler
#
# todo: refactor deployment functions into this
#
class Request < SimpleDelegator
  # wrapper for CGI#cookies: add cookie name prefix, return first value
  #
  def cookie(name)
    @cgi.cookies[@cookie_prefix + name][0]
  end

  # create cookie and add it to the HTTP response
  #
  # cookie name prefix is configured via config.yaml; set expires to non-nil
  # value to create persistent cookie, use #forever as defined in
  # engine/helpers.rb to set a long-term cookie
  #
  def set_cookie(name, value = nil, expires = nil)
    headers['cookie'] = {} unless headers['cookie'].kind_of? Hash
    headers['cookie'][name] = CGI::Cookie.new({
      'name' => @cookie_prefix + name,
      'value' => value,
      'path' => @uri_prefix + '/',
      'expires' => expires.kind_of?(Numeric) ? Time.now + expires : nil,
      'secure' => (@env['HTTPS'] and 'session' == name)
    })
  end

  # set cookie to expire now
  #
  def unset_cookie(name)
    set_cookie(name, nil, 0)
  end

  # get cached session, refresh or clear it
  #
  # don't cache guest and stale sessions
  #
  def cached_session
    c = cookie('session')

    if c and c =~ Session::COOKIE_PATTERN
      key = Session.cache_key(c)

      session = cache.fetch_or_add(key) do
        s = Session.new(c)
        s.member ? s : nil
      end

      if session and session.member and session.fresh!
        set_cookie('session', c, config['timeout']['last'])
      else
        cache.delete(key)
        unset_cookie('session')
      end
    end

    session or Session.new(nil)
  end

  # set language of user interface
  #
  def language=(lang)
    lang = default_language if
      lang.nil? or not config['locale']['languages'].include?(lang)
    lang.untaint
    if defined? GetText
      samizdat_bindtextdomain(lang, config['locale']['path'])
    end
    @language = lang
  end

  # current language
  attr_reader :language

  # set default CGI headers (set charset to UTF-8)
  #
  # set id and refresh session if there's a valid session cookie,
  # set credentials to guest otherwise
  #
  def initialize(cgi)
    @cgi = cgi

    # hack to get through to CGI variables
    class << @cgi
      public :env_table
    end
    @env = @cgi.env_table

    @host = (@env['HTTP_X_FORWARDED_HOST'] or
      @env['SERVER_NAME'] or
      @env['HTTP_HOST'])

    @site_name, @uri_prefix, uri_tail =
      SamizdatSites.instance.find(@host, @env['REQUEST_URI'])

    # drop the request if it can't be attributed to any of the configured sites
    # todo: fallback to pwd-based operation when lookup in global config fails
    throw :finish unless @site_name.kind_of?(String) and
      @uri_prefix.kind_of?(String) and uri_tail.kind_of?(String)

    # hack for deployment classes (should really be Thread.current['request'],
    # but that doesn't work with DRb)
    $samizdat_current_request = self

    DRb.start_service("druby://localhost:0")
    # fixme: localhost -> config.drb_local

    @route = uri_tail.split('?', 2)[0]   # drop GET parameters

    @headers = {'charset' => 'utf-8', 'cookie' => {}}
    @headers['Content-Location'] = '/' + @uri_prefix

    @cookie_prefix = config['site']['cookie_prefix'] + '_'

    set_language(config['locale']['languages'], @env['HTTP_ACCEPT_LANGUAGE'])

    # construct @base
    proto = @env['HTTPS'] ? 'https' : 'http'
    port = defined?(MOD_RUBY) ?
      Apache.request.connection.local_port :
      @env['SERVER_PORT']
    port = (port.to_i == {'http' => 80, 'https' => 443}[proto]) ? '': ':' + port.to_s
    @base = proto + '://' + @host + port + @uri_prefix + '/'

    # select CSS style
    @style = cookie('style')
    @style = config['style'][0] unless config['style'].include?(style)

    @session = cached_session

    super @cgi
  end

  # raw CGI environment variables
  attr_reader :env

  # server name
  attr_reader :host

  # name of Samizdat site
  attr_reader :site_name

  # URI prefix
  attr_reader :uri_prefix

  # route to controller action
  attr_reader :route

  # base URI of the site, used to compose absolute links
  attr_reader :base

  # Session object
  attr_reader :session

  # list of languages in user's order of preference
  attr_reader :accept_language

  # preferred stylesheet
  attr_reader :style

  # HTTP response headers
  attr_reader :headers

  # web server document root (untainted as it's assumed to be safe)
  #
  def document_root
    @document_root ||= @env['DOCUMENT_ROOT'].dup.untaint
  end

  # translate location to a real file name
  #
  def filename(location)
    if defined?(MOD_RUBY)
      Apache.request.lookup_uri(location).filename
    else
      document_root + location
    end
  end

  # location of content directory relative to host root
  #
  def content_location
    if config['site']['content'].kind_of? String then
      File.join(@uri_prefix, config['site']['content'])
    else
      ''
    end
  end

  def advanced_ui?
    'advanced' == cookie('ui')
  end

  # true if the user has and is wielding their moderator priviledges
  #
  def moderate?
    @session.moderator? and ('yes' == cookie('moderate'))
  end

  # return role name for the current user
  # ('moderator', 'member', or 'guest')
  #
  def role
    if moderate? then 'moderator'
    elsif @session.member then 'member'
    else 'guest'
    end
  end

  def _rgettext_hack   # :nodoc:
    [ _('moderator'), _('member'), _('guest') ]
  end

  def alt_styles
    config['style'].reject {|alt| alt == @style }
  end

  # raw CGI#params hash
  #
  def cgi_params
    @cgi.params
  end

  # always imitate CGI#[] from Ruby 1.8, normalize value
  #
  def [](key)
    normalize_parameter(@cgi.params[key][0])
  end

  # plant a fake CGI parameter
  #
  def []=(key, value)
    @cgi.params[key] = [value]
  end

  # return list of normalized values of CGI parameters with given names
  #
  def values_at(keys)
    keys.collect {|key| self[key] }
  end

  # dump CGI parameters (except passwords) for error report
  #
  def dump_params
    @cgi.params.dup.delete_if {|k, v|
      k =~ /^password/
    }.inspect
  end

  # print header and optionally content, then clean-up and exit
  #
  def response(headers = {}, body = nil)
    if @notice
      set_cookie('notice', @notice)
    end

    @headers.update(headers)
    if body
      @cgi.out(@headers) { compress(body) }
    else
      print @cgi.header(@headers)
    end

    DRb.stop_service
    $samizdat_current_request = nil
  end

  # make sure 'redirect_when_done' cookie value is set to a relative location,
  # so that a secure action doesn't redirect back to an unencrypted page
  #
  def set_redirect_when_done_cookie
    unless cookie('redirect_when_done')
      location = (@cgi.referer or '').sub(/\A#{@base}/, '')
      location = '/' if location.empty? or 'member/login' == location
      set_cookie('redirect_when_done', location)
    end
  end

  # 'redirect_when_done' cookie overrides value of referer header
  #
  def redirect_when_done
    location = cookie('redirect_when_done')
    if location
      unset_cookie('redirect_when_done')
      location = '' if '/' == location
      redirect(location)
    else
      redirect(@cgi.referer)
    end
  end

  # send a redirect header and finish the request processing
  #
  # +location+ defaults to referer; site base is prepended to relative links
  #
  def redirect(location = nil)
    if location.nil?
      location = referer
    elsif not absolute_url?(location)
      location = base + location.to_s
    end
    response({'status' => 'REDIRECT', 'location' => location})
    throw :finish
  end

  # record a notice to be displayed on next request
  #
  def add_notice(notice)
    if @notice
      @notice << notice
    else
      @notice = notice
    end
  end

  def notice
    cookie('notice')
  end

  def reset_notice
    unset_cookie('notice')
  end

  private

  # check size limit, read multipart field into memory, transform empty value
  # to +nil+
  #
  def normalize_parameter(value)
    raise UserError, _('Input size exceeds content size limit') if
      value.respond_to?(:size) and
      value.size > config['limit']['content']

    case value
    when StringIO, Tempfile
      io = value
      value = io.read
      io.rewind
    end

    (value =~ /[^\s]/) ? value : nil
  end

  # see #compress
  #
  MIN_GZ_SIZE = 1024

  # do gzip compression and check ETag when supported by client
  #
  def compress(body)
    if body.length > MIN_GZ_SIZE and @env.has_key?('HTTP_ACCEPT_ENCODING')
      enc =
        case @env['HTTP_ACCEPT_ENCODING']
        when /x-gzip/ then 'x-gzip'
        when /gzip/   then 'gzip'
        end
      unless enc.nil?
        io = ''   # primitive StringIO replacement
        class << io
          def write(str)
            self << str
          end
        end
        gz = Zlib::GzipWriter.new(io)
        gz.write(body)
        gz.close
        body = io
        @headers['Content-Encoding'] = enc
        @headers['Vary'] = 'Accept-Encoding'
      end
    end

    # check ETag
    if @env.has_key?('HTTP_IF_NONE_MATCH')
      etag = '"' + digest(body) + '"'
      @headers['ETag'] = etag
      catch(:ETagFound) do
        @env['HTTP_IF_NONE_MATCH'].each(',') do |e|
          if etag == e.strip.delete(',')
            @headers['status'] = 'NOT_MODIFIED'
            body = ''   # don't send body
            throw :ETagFound
          end
        end
      end
    end

    body
  end

  # parse and store user's accept-language preferences, determine the UI
  # language from the results
  #
  def set_language(site_languages, http_accept_language)
    @accept_language = []   # [[lang, q]...]

    if http_accept_language
      http_accept_language.scan(/([^ ,;]+)(?:;q=([^ ,;]+))?/).collect {|l, q|
        [l, (q ? q.to_f : 1.0)]
      }.sort_by {|l, q| -q }.each {|l, q|
        unless site_languages.include? l
          # try converting full locale (language tag) to ISO-639 language only
          # http://www.w3.org/Protocols/rfc2616/rfc2616-sec14.html#sec14.4
          l = Locale::Object.new(l).language
        end
        @accept_language.push l if site_languages.include? l
      }
    end

    if lang = cookie('lang') and site_languages.include? lang
      @accept_language.unshift lang   # lang cookie overrides Accept-Language
    elsif @accept_language.empty?
      @accept_language.push site_languages.first
    end

    self.language = @accept_language.first   # set interface language
  end
end
