# Samizdat search query construction
#
#   Copyright (c) 2002-2008  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#
# vim: et sw=2 sts=2 ts=8 tw=0

class QueryController < Controller

  def index
    query = (@request['q'] or DEFAULT_QUERY)
    q = validate_query(query)

    @title = _('Edit Query')
    @content_for_layout = box(@title, edit_box(q))
  end

  def run(query = @request['q'])
    if query.nil?
      @request.redirect('query')
    end
    q = validate_query(query)

    page = (@request['page'] or 1).to_i

    @title = _('Search Result') + page_number(page)

    dataset = SqlDataSet.new(q.to_sql[0])
    results = dataset[page - 1].collect {|id,| Resource.new(@request, id).list_item }

    if results.size > 0
      unless page > 1
        feed = rss_link(query)
        @feeds = { @title => feed }
      end

      if @request.advanced_ui? and not page > 1
        foot = form('message/publish',
          [:hidden, 'format', 'application/x-squish'],
          [:hidden, 'body', query],
          [:submit, 'publish', _('Publish This Query')])
      end

      body = list(results,
        nav_rss(feed) <<
        nav(dataset, 'page', page,
          %{query/run?q=#{CGI.escape(query)}&amp;}) <<
        foot.to_s)
    else
      body = '<p>' << _('No matching resources found.') << '</p>'
    end

    @content_for_layout = box(@title, body)
    @content_for_layout << edit_box(q) unless page > 1
  end

  # search messages by a substring
  #
  def search
    substring = @request['substring']

    if substring.nil?
      @request.redirect('query')
    end

    query = %{SELECT ?msg
WHERE (dc::date ?msg ?date)
      (dc::title ?msg ?title)
LITERAL ?title ILIKE '%#{substring.gsub(/%/, '\\\\\\%')}%'
ORDER BY ?date DESC}

    run(query)
  end

  # regenerate query from form data
  #
  def update
    nodes, literal, order, order_dir, using =
      @request.values_at %w[nodes literal order order_dir using]

    # generate namespaces
    namespaces = config['ns']   # standard namespaces
    namespaces.update(Hash[*using.split]) if using

    using = {}

    # generate query pattern
    pattern = []
    @request.keys.grep(/\A(predicate|subject|object)_(\d{1,2})\z/) do |key|
      value = @request[key]
      next unless value
      i = $2.to_i - 1
      pattern[i] = [] unless pattern[i].kind_of?(Array)
      pattern[i][ %w[predicate subject object].index($1) ] = value
      namespaces.each do |p, uri|
        if /\A#{p}::/ =~ value
          using[p] = uri   # only leave used namespace prefixes
          break
        end
      end
    end

    query = "SELECT #{nodes}\nWHERE " <<
      pattern.collect {|predicate, subject, object|
        "(#{predicate} #{subject} #{object})" if
          predicate and subject and object and predicate !~ /\s/
          # whitespace is only present in 'BLANK CLAUSE'
        }.compact.join("\n      ")
    query << "\nLITERAL #{literal}" if literal
    query << "\nORDER BY #{order} #{order_dir}" if order
    query << "\nUSING " + using.to_a.collect {|n|
      n.join(' FOR ')
    }.join("\n      ")

    run(query)
  end

  # RSS feed of a query run
  #
  def rss
    query = @request['q']
    q = validate_query(query)

    @request.headers['type'] = 'application/xml'

    require 'rss/maker'
    rss = RSS::Maker.make("1.0") do |maker|
      maker.channel.about = @request.base + rss_link(query)
      title = config['site']['name'] + ' / ' + _('Search Result')
      maker.channel.title = title
      maker.channel.description = CGI.escapeHTML(query)
      maker.channel.link = @request.base +
        %{query/run?q=#{CGI.escape(query)}}

      if config['site']['icon']
        maker.image.title = title
        maker.image.url = @request.base + config['site']['icon']
      end

      SqlDataSet.new(q.to_sql[0])[0].each do |id,|
        resource = Resource.new(@request, id)
        if 'Message' == resource.type
          Message.cached(id).rss(maker, @request)
        else
          item = maker.items.new_item
          item.link = @request.base + id.to_s
          item.date = rdf.get_property(id, 'dc::date').to_time
          item.title = resource.title
        end
      end
    end

    @content_for_layout = rss.to_s
    @layout = nil
  end

  private

  DEFAULT_QUERY = %{SELECT ?resource\nWHERE (dc::date ?resource ?date)\nORDER BY ?date DESC}

  # validate query (syntax, must-bind list size, number of clauses)
  #
  def validate_query(query)
    begin
      q = Samizdat::SquishQuery.new(query, rdf.ns)
      sql, = rdf.select(q)
    rescue Samizdat::ProgrammingError
      raise UserError, _('Error in your query: ') + CGI.escapeHTML($!.message)
    end

    (q.nodes.size != 1) and raise UserError,
      _('Must-bind list should contain only one blank node, filters based on queries with a complex answer pattern are not implemented')

    (q.pattern.size > config['limit']['pattern']) and raise UserError,
      sprintf(_('User-defined query pattern should not be longer than %s clauses'), config['limit']['pattern'])

    q
  end

  def edit_box(q)
    editbox = box(_('Search By Substring'),
      form('query/search',
        [:text, 'substring', @request['substring']],
        [:submit, 'search', _('Search')]))

    if @request.advanced_ui?
      edit = box(_('Select Target'),
        form_field(:select, 'nodes', q.pattern.collect {|p, s, o| s }.uniq))

      i = 0
      edit << box( _('Query Pattern (predicate subject object)'),
        ((q.pattern.size >= config['limit']['pattern'])?
          q.pattern :
          q.pattern + [['']]
        ).collect {|clause|
          predicate, subject, object = clause.first(3).collect {|uri| q.ns_shrink(uri) }
          i += 1

          %{#{i}. } +
          # todo: add external properties to the options list
          form_field(:select, "predicate_#{i}",
            [_('BLANK CLAUSE ')] + config['map'].keys.sort, predicate) +
            # make sure 'BLANK CLAUSE' includes whitespace in all translations
          form_field(:text, "subject_#{i}", subject) +
          form_field(:text, "object_#{i}", q.substitute_literals(object)) +
          form_field(:br)
          # todo: select subject and object from variables and known urirefs
        }.join
      )

      edit << box(_('Literal Condition'),
        form_field(:text, 'literal', q.substitute_literals(q.literal)))

      edit << box(_('Order By'),
        form_field(:select, 'order', q.variables, q.order) +
        form_field(:select, 'order_dir',
          [['ASC', _('Ascending')], ['DESC', _('Descending')]], q.order_dir))

      edit << box(nil, '<pre>USING ' +
        q.ns.to_a.collect {|n| n.join(' FOR ') }.join("\n      ") + '</pre>' +
        form_field(:hidden, 'using', q.ns.to_a.flatten.join(' ')))

      edit << form_field(:submit, 'update', _('Update'))

      editbox << box(_('Construct Query'),
        %{<form action="query/update" method="post">#{edit}</form>})

      editbox << box(_('Edit Raw Query'),
        form('query/run',
          [:textarea, 'q', q.to_s],
          [:label],
          [:submit, 'run', _('Run')]))
    end

    editbox = q.substitute_literals(editbox)
    box(_('Edit Query'), editbox)
  end

  def rss_link(query)
    %{query/rss?q=#{CGI.escape(query)}}
  end
end
