#include "jacobi.h"
#include "givens.h"

using namespace RUMBA;

namespace
{
// both of these functions assume a symmetric matrix !!!
double off(const ManifoldMatrix& A);
double sumSquares (const ManifoldMatrix& A);
}

void RUMBA::sym_shur2(
		ManifoldMatrix A, 
		unsigned int p, 
		unsigned int q, 
		double& c, 
		double& s
)
{
	double tau, t;
	if (A.element(p,q) != 0)
	{
		tau = (A.element(q,q) - A.element(p,p)) * ( 1 / (2*A.element(p,q)));
		if (tau>=0)
			t = 1/(tau + std::sqrt(1+tau*tau));
		else
			t = -1/(-1*tau + std::sqrt(1+tau*tau));

		c = 1/std::sqrt(1+t*t);
		s = t*c;
	}
	else
	{
		c = 1;
		s = 0;
	}
}

ManifoldMatrix RUMBA::jacobi 
( 
 ManifoldMatrix& A, 
 double t 
 )
{
	double c=0,s=0;
	ManifoldMatrix V (RUMBA::identityMatrix(A.rows()));
	double epsilon = t * std::sqrt(sumSquares(A));

	while (off(A) > epsilon)
	{
		for ( int p = 0; p < A.rows()-1; ++p )
		{
			for ( int q = p+1; q < A.cols(); ++q )
			{
				sym_shur2(A,p,q,c,s);
				apply_left_givens(A,p,q,c,s);
				apply_right_givens(A,p,q,c,s);
				apply_right_givens(V,p,q,c,s);
			}
		}
	}

	return V;
}

namespace 
{

double sumSquares (const ManifoldMatrix& A)
{
	double tmp = 0, sum = 0;
	int n = A.rows();
	for ( int i = 0; i < n; ++i )
	{
		tmp = A.element(i,i);
		sum += tmp * tmp;
	}
	return sum + off(A);
}

// sum off-diagonal elements in a symmetric matrix
double off(const ManifoldMatrix& A)
{
	double sum = 0;
	int n = A.rows(); 
	for ( int i = 0; i < n-1; ++i )
		for ( int j = i+1; j < n; ++j )
				sum += A.element(i,j) * A.element(i,j);
	return 2*sum;
}


}
