#ifndef RUMBA_FACTORY_H
#define RUMBA_FACTORY_H

#include <iostream>
#include <map>
#include <rumba_config.h>
#include <rumba/log.h>


/**
  * \file factory.h
  * \author Donovan Rebbechi, Ben Bly, Giorgio Grasso
  * Copyright Ben Bly
  * This file is released under the artistic license. See the file
  * COPYING for details.
  */




namespace RUMBA
{

class IOHandlerBase;
// class BaseManifold;


/** 
  *	Building blocks for manifold. This is currently used as a dummy argument
  * so that other classes can indirectly request a certain templated type (like 
  *	IOHandler<int>) without having to "know" about that class, because coupling
  * a class with templated code adds excessive compile time dependencies.
  */


class Factory
{
private:
	std::string id;
protected:
	static std::map<std::string, Factory*> TypeList;
	void add_type()
	{
		TypeList.insert(std::make_pair(id,this));
	}

	RUMBA::Log log;
	Factory(std::string id):id(id),log("Factory" ) { add_type(); }

	virtual ~Factory(){}
public:
	static Factory* lookup(std::string id)
	{
		std::map<std::string, Factory*>::const_iterator it = TypeList.find(id);
		if ( it == TypeList.end() )
			return 0;
		else
			return it->second;
	}


	virtual IOHandlerBase* makeIOHandler 
		(
		 const char* name, 
		 int Size, 
		 std::ios_base::openmode mode 
		 ) const = 0 ; 
	//	virtual BaseManifold* makeManifold = 0;
};

class CharFactory : public Factory
{
public:
	virtual IOHandlerBase* makeIOHandler 
		( 
		 const char* name, 
		 int Size,
		 std::ios_base::openmode mode=std::ios::in 
		 ) const;
	static CharFactory* get(){ static CharFactory x; return &x; }
private:
	CharFactory& operator=(const CharFactory& ) {return *this;}
	CharFactory() : Factory("char") 
	{ 
		log.logName() << "Creating CharFactory at " << this << "\n"; 
	}

};

class ShortFactory : public Factory
{
public:
	virtual IOHandlerBase* makeIOHandler 
		( 
			const char* name, 
			int Size,
			std::ios_base::openmode mode=std::ios::in 
		) const;
	static ShortFactory* get(){ static ShortFactory x; return &x; }
private:
	ShortFactory& operator=(const ShortFactory& ) {return *this;}
	ShortFactory() : Factory("int16")
	{ 
		log.logName() << "Creating ShortFactory at " << this << "\n"; 
	}

};

class IntFactory : public Factory
{
public:
	virtual IOHandlerBase* makeIOHandler 
		( 
		 const char* name, 
		 int Size, 
		 std::ios_base::openmode mode=std::ios::in 
		 ) const;
	static IntFactory* get(){ static IntFactory x; return &x; }
private:
	IntFactory& operator=(const IntFactory& ) {return *this;}
	IntFactory() : Factory("int32") 
	{ 
		log.logName() << "Creating IntFactory at " << this << "\n"; 
	}
};

class FloatFactory : public Factory
{
public:
	virtual IOHandlerBase* makeIOHandler 
		( 
		 const char* name, 
		 int Size,
		 std::ios_base::openmode mode=std::ios::in 
		 ) const;
	static FloatFactory* get(){ static FloatFactory x; return &x; }
private:
	FloatFactory& operator=(const FloatFactory& ) {return *this;}
	FloatFactory() : Factory("float32") { log.logName() << "Creating FloatFactory at "<< this << "\n"; }

};

class DoubleFactory : public Factory
{
public:
	virtual IOHandlerBase* makeIOHandler 
		( 
		 const char* name, 
		 int Size,
		 std::ios_base::openmode mode=std::ios::in 
		 ) const;
	static DoubleFactory* get(){ static DoubleFactory x; return &x; }
private:
	DoubleFactory& operator=(const DoubleFactory& ) {return *this;}
	DoubleFactory() : Factory("float64") { log.logName() << "Creating DoubleFactory at "<< this << "\n"; }
};



} // rumba

#endif
