// Strings and streams
#include <iostream>
#include <iomanip>
#include <fstream>
#include <strstream>
#include <string>

// STL containers

#include <map>


// rumba specific headers


#include <rumba/exception.h>
#include <rumba/rumba_cast.h>
#include <rumba/specialisations.h>
#include <rumba/rumba_system.h>

#include <rumba/manifoldFile.h>







template <class TYPE>
RUMBA::Manifold<TYPE>::Manifold() 
: BaseManifold(), DataArray(NULL), RefCount(new unsigned long(1))
{	
	HeaderData["datatype"] = HeaderData["normalized_datatype"] = getDataString();
}

template <class TYPE>
RUMBA::Manifold<TYPE>::Manifold(string filename) 
: BaseManifold(), DataArray(NULL),RefCount(new unsigned long(1))
{	
	load(filename);
	HeaderData["datatype"] = HeaderData["normalized_datatype"] = getDataString();
}

template <class TYPE>
RUMBA::Manifold<TYPE>::Manifold(const intPoint& e )
: BaseManifold(), DataArray(NULL), RefCount(new unsigned long(1))
{	
	allocate(e);
	HeaderData["normalized_datatype"] = getDataString();
	HeaderData["datatype"] = getDataString();
}


template <class TYPE>
RUMBA::Manifold<TYPE>::Manifold(const RUMBA::Orientation& o )
: BaseManifold(), DataArray(NULL), RefCount(new unsigned long(1))
{	
	allocate(o.extent());
	Orient = o;
	HeaderData["normalized_datatype"] = getDataString();
	HeaderData["datatype"] = getDataString();
}



template <class TYPE>
RUMBA::Manifold<TYPE>::~Manifold()
{	
#ifdef DEBUG
	log.logName() << "Calling destructor" << std::endl;
#endif
	release();
}


template<class TYPE>
void RUMBA::Manifold<TYPE>::release()
{
#ifdef DEBUG
	log.logName() << "Calling release()" << std::endl;
	log.logName() << "Refcount is :" << *RefCount << std::endl;
	log.logName() << "Refcount is :" << RefCount << std::endl;
	log.logName() << "DataArray is :" << DataArray << std::endl;

#endif
	if (!--*RefCount)
	{
#ifdef DEBUG
		log.logName() << "RefCount down to 0, deleting" << std::endl;
#endif
		delete RefCount;
		delete []DataArray;
#ifdef DEBUG
		log.logName() << "Deleted" << std::endl;
#endif
	}
}





template <class TYPE>
void RUMBA::Manifold<TYPE>::print(std::ostream& stream) const
{	
	stream << "Manifold info: ";
	stream << Extent.x() << "x";
	stream << Extent.y() << "x";
	stream << Extent.z() << "x";
	stream << Extent.t() << " ";
	stream << "(" << sizeof(TYPE)*8 << " bpv) = ";
	stream << size()*sizeof(TYPE) << " bytes (";
	stream << VoxelSize.x() << "x";
	stream << VoxelSize.y() << "x";
	stream << VoxelSize.z() << " mm)" << std::endl;

	for ( 
			std::map<std::string, RUMBA::Splodge>::const_iterator it = 
			HeaderData.begin();
			it != HeaderData.end();
			++it 
			)
		stream << it->first << ": " << it->second << std::endl;
}



template <class TYPE>
RUMBA::Manifold<TYPE>::Manifold(const Manifold<TYPE>& manifold)
: BaseManifold(), DataArray(manifold.DataArray), RefCount(manifold.RefCount)
{
#ifdef DEBUG
	log.logName() <<  "Copy construct" << std::endl;
#endif
	++*RefCount;
#ifdef DEBUG
	log.logName() <<  "Increasing ref count to " << *RefCount << std::endl;
#endif
	copyHeader(manifold);
//	allocate(manifold.extent());

}

template <class TYPE>
RUMBA::Manifold<TYPE>
RUMBA::Manifold<TYPE>::copy() const
{
#ifdef DEBUG
	log.logName() <<  "COPY" << std::endl;
#endif

	Manifold<TYPE> result(Extent);
	memcpy ( (void*)result.DataArray, (void*)DataArray, sizeof(TYPE) * size() );

	result.copyHeader(*this);
	return result;
}

template <class TYPE> template <class TYPEa>
RUMBA::Manifold<TYPE>::Manifold(const Manifold<TYPEa>& manifold)
	: BaseManifold(), RefCount(new unsigned long(1))
{
	DataArray = NULL;
#ifdef DEBUG
	log.logName() << "CONVERSION CONSTRUCT" << std::endl;
#endif
	copyHeader(manifold);

	int s = size();
	const TYPEa *srcDataArray = manifold.data();
	TYPE *dstDataArray = DataArray;
#ifdef USE_RUMBA_CAST
	while (s--) 
		*(dstDataArray++) = rumba_cast<TYPE>( *(srcDataArray++));
#else
	while (s--) *(dstDataArray++) = (TYPE) *(srcDataArray++);
#endif
}

template<class TYPE>
const TYPE *RUMBA::Manifold<TYPE>::data() const
{	return DataArray;
}

template<class TYPE>
const TYPE& RUMBA::Manifold<TYPE>::operator[] ( int pos ) const
{
	return DataArray[pos];
}

template<class TYPE>
TYPE& RUMBA::Manifold<TYPE>::operator[] ( int pos ) 
{
	return DataArray[pos];
}



template<class TYPE>
TYPE RUMBA::Manifold<TYPE>::getElement(int pos) const
{	
	return DataArray[pos];
}

template<class TYPE>
int RUMBA::Manifold<TYPE>::getElementInt(int pos) const
{	
	return rumba_cast<int> ( DataArray[pos] );
}

template<class TYPE>
double RUMBA::Manifold<TYPE>::getElementDouble(int pos) const
{	
	return rumba_cast<double> ( TYPE(DataArray[pos]) );
}



template<class TYPE>
TYPE RUMBA::Manifold<TYPE>::getElement(int x, int y, int z, int t) const
{	return DataArray[x*Skip.x()+y*Skip.y()+z*Skip.z()+t*Skip.t()];
}

template<class TYPE>
int RUMBA::Manifold<TYPE>::getElementInt(int x, int y, int z, int t) const
{	
	return rumba_cast<int> ( DataArray[x*Skip.x()+y*Skip.y()+z*Skip.z()+t*Skip.t() ] );
}

template<class TYPE>
double RUMBA::Manifold<TYPE>::getElementDouble(int x, int y, int z, int t) const
{	
	return rumba_cast<double>( DataArray[x*Skip.x()+y*Skip.y()+z*Skip.z()+t*Skip.t()] );
}

template<class TYPE>
TYPE RUMBA::Manifold<TYPE>::getElement(const intPoint& p) const
{	return DataArray[p.x()*Skip.x()+p.y()*Skip.y()+p.z()*Skip.z()+p.t()*Skip.t()];
}

template<class TYPE>
int RUMBA::Manifold<TYPE>::getElementInt(const intPoint& p) const
{	
	return rumba_cast<int> ( DataArray[p.x()*Skip.x()+p.y()*Skip.y()+p.z()*Skip.z()+p.t()*Skip.t() ] );
}

template<class TYPE>
double RUMBA::Manifold<TYPE>::getElementDouble(const intPoint& p) const
{	
	return rumba_cast<double>( DataArray[p.x()*Skip.x()+p.y()*Skip.y()+p.z()*Skip.z()+p.t()*Skip.t()] );
}



template<class TYPE>
RUMBA::Manifold<TYPE>* 
RUMBA::Manifold<TYPE>::get(int x,int y,int z, int t, int dx, int dy, int dz, int dt) const
{	
	if ((x<0 || y<0 || z<0 || t<0) ||
	    (x>=Extent.x() || y>=Extent.y() || z>=Extent.z() || t>=Extent.t()))
		throw RUMBA::Exception("Extents out of range");

	if ((dx<0 || dy<0 || dz<0 || dt<0) ||
	    (dx>x+Extent.x() || dy>y+Extent.y() || dz>z+Extent.z() || dt>t+Extent.t()))
		throw RUMBA::Exception("Extents out of range");

	Manifold<TYPE>* res = new Manifold<TYPE>(intPoint(dx,dy,dz,dt));

	for (int i=0; i<dx; i++)
		for (int j=0; j<dy; j++)
			for (int k=0; k<dz; k++)
				for (int l=0; l<dt; l++)
					res->setElement(i,j,k,l, getElement(x+i,y+j,z+k,t+l));

	return res;
}

template<class TYPE>
RUMBA::Manifold<TYPE>* 
RUMBA::Manifold<TYPE>::get(const intPoint& o,const intPoint& e) const
{		
	return get(o.x(),o.y(),o.z(),o.t(),e.x(),e.y(),e.z(),e.t());
}

template<class TYPE>
void RUMBA::Manifold<TYPE>::setElement(int pos, TYPE val) 
{	
	DataArray[pos] = val;
}

template<class TYPE>
void RUMBA::Manifold<TYPE>::setElementDouble( int pos, double val )
{
	DataArray[pos] = rumba_cast<TYPE> ( val );
}
template<class TYPE>
void RUMBA::Manifold<TYPE>::setElementInt( int pos, int val )
{
	DataArray[pos] = rumba_cast<TYPE> ( val );
}

template<class TYPE>
void RUMBA::Manifold<TYPE>::setElement(int x, int y, int z, int t, TYPE val)
{	
	DataArray[x*Skip.x()+y*Skip.y()+z*Skip.z()+t*Skip.t()] = val;
}

template<class TYPE>
void RUMBA::Manifold<TYPE>::setElementDouble(int x, int y, int z, int t, double val)
{	
	DataArray[x*Skip.x()+y*Skip.y()+z*Skip.z()+t*Skip.t()] = rumba_cast<TYPE> ( val );
}
template<class TYPE>
void RUMBA::Manifold<TYPE>::setElementInt(int x, int y, int z, int t, int val)
{	
	DataArray[x*Skip.x()+y*Skip.y()+z*Skip.z()+t*Skip.t()] = rumba_cast<TYPE> ( val );
}

template<class TYPE>
void RUMBA::Manifold<TYPE>::setElement(const intPoint& p, TYPE val)
{	
	DataArray[p.x()*Skip.x()+p.y()*Skip.y()+p.z()*Skip.z()+p.t()*Skip.t()] = val;
}

template<class TYPE>
void RUMBA::Manifold<TYPE>::setElementDouble(const intPoint& p, double val)
{	
	DataArray[p.x()*Skip.x()+p.y()*Skip.y()+p.z()*Skip.z()+p.t()*Skip.t()] = rumba_cast<TYPE> ( val );
}
template<class TYPE>
void RUMBA::Manifold<TYPE>::setElementInt(const intPoint& p, int val)
{	
	DataArray[p.x()*Skip.x()+p.y()*Skip.y()+p.z()*Skip.z()+p.t()*Skip.t()] = rumba_cast<TYPE> ( val );
}

template<class TYPE>
void RUMBA::Manifold<TYPE>::set(Manifold<TYPE>& manifold, int pos)
{	
	int tmp = pos;
	//convert pos to x,y,z,t
	int x,y,z,t;
	x = tmp % Extent.x();
	tmp /= Extent.x();
	y = tmp % Extent.y();
	tmp /= Extent.y();
	z = tmp % Extent.z();
	tmp /= Extent.z();
	t = tmp % Extent.t();

	set ( manifold,x,y,z,t );
}

template<class TYPE>
void RUMBA::Manifold<TYPE>::set(Manifold<TYPE>& manifold, int x, int y, int z, int t)
{
#ifdef USE_EXCEPTIONS
	if ( 
		manifold.Extent.x() + x > Extent.x() ||
		manifold.Extent.y() + y > Extent.y() ||
		manifold.Extent.z() + z > Extent.z() ||
		manifold.Extent.t() + t > Extent.t()
		)
	throw RUMBA::Exception ( "Insertion out of range\n" );
#endif		

	for ( int i =0; i<manifold.Extent.x(); ++i )
		for ( int j = 0; j < manifold.Extent.y(); ++j )
			for ( int k = 0; k < manifold.Extent.z(); ++k )
				for ( int l = 0; l < manifold.Extent.t(); ++l )
					setElement ( x+i, y+j, z+k, t+l, manifold.getElement(i,j,k,l));
}

template<class TYPE> template<class TYPEa>
void RUMBA::Manifold<TYPE>::set(Manifold<TYPEa>& manifold, int pos)
{	
	//convert pos to x,y,z,t
	int x,y,z,t;
	x = tmp % Extent.x();
	tmp /= Extent.x();
	y = tmp % Extent.y();
	tmp /= Extent.y();
	z = tmp % Extent.z();
	tmp /= Extent.z();
	t = tmp % Extent.t();

	set ( manifold,x,y,z,t );

}

template<class TYPE> template<class TYPEa>
void RUMBA::Manifold<TYPE>::set(Manifold<TYPEa>& manifold, int x, int y, int z, int t )
{
#ifdef USE_EXCEPTIONS
	// can't use manifold's data memebers -- Manifold<TYPEa> is *not* the same class as 
	// Manifold<TYPE>
	if ( 
		manifold.width() + x > width() ||
		manifold.height() + y > height() ||
		manifold.depth() + z > depth() ||
		manifold.timepoints() > timepoints()
		)
	throw RUMBA::Exception ( "Insertion out of range\n" );
#endif		

	int w = manifold.width(), h = manifold.height(), d = manifold.depth(), 
		tp = manifold.timepoints();
	for ( int i =0; i<w; ++i )
		for ( int j = 0; j < h; ++j )
			for ( int k = 0; k < d; ++k )
				for ( int l = 0; l < tp; ++l )
					setElement ( x+i, y+j, z+k, t+l, manifold.getElement(i,j,k,l));

}


template <class TYPE>
RUMBA::Manifold<TYPE>& RUMBA::Manifold<TYPE>::operator= (const Manifold<TYPE>& manifold)
{
#ifdef DEBUG
	log.logName() <<  "Copy-assign" << std::endl;
#endif
	copyHeader(manifold);

	if (this != (const Manifold<TYPE> *) &manifold) 
	{
		release();
		RefCount = manifold.RefCount;
		++*RefCount;
		DataArray = manifold.DataArray;
	}
#ifdef DEBUG
	log.logName() <<  "After assign, refcount: " << *RefCount << std::endl;
#endif

	return *this;
}

template <class TYPE> template <class TYPEa>
RUMBA::Manifold<TYPE>& 
RUMBA::Manifold<TYPE>::operator= (const Manifold<TYPEa>& manifold)
{
#ifdef DEBUG
	log.logName() <<  "ASSIGN" <<std::endl;
#endif
	if (this == (Manifold<TYPE> *) &manifold) return *this;
	copyHeader(manifold);

	release();
	RefCount = new unsigned long(1);
	allocate(manifold.extent());

	VoxelSize = M.voxelSize();
		
	int s = size();
	const TYPEa *srcDataArray = manifold.data();
	TYPE *dstDataArray = DataArray;
	while (s--) 
		*(dstDataArray++) = rumba_cast<TYPE> ( *(srcDataArray++));
	return *this;
}

template <class TYPE>
RUMBA::Manifold<TYPE>& RUMBA::Manifold<TYPE>::fill (const TYPE val)
{	
	if (size() == 0)
		std::cerr << "Manifold: cannot assign a scalar value to an empty manifold" << std::endl;

	int s = size();
	TYPE *dstDataArray = DataArray;
	while (s--) *(dstDataArray++) = val;

	return *this;
}


template <class TYPE>
RUMBA::Manifold<TYPE> RUMBA::Manifold<TYPE>::getVolume (int pos) const
{	
	if (pos > Extent.t() )
		std::cerr << "Manifold: index out of range" << std::endl;
	intPoint p(Extent);
	p.t() = 1;
	Manifold<TYPE> res (p);
	res.setVoxelSize(VoxelSize);
	int i;
	int s = res.size();
	const TYPE *srcDataArray = data();
// 	TYPE *dstDataArray = res.DataArray;
	srcDataArray += Skip.t()*pos;
	for ( i =0; i<s; ++i ) res[i]=DataArray[i];
	
	return res;
}

template <class TYPE>
RUMBA::Manifold<TYPE> operator+ (const RUMBA::Manifold<TYPE>& manifold1, const RUMBA::Manifold<TYPE>& manifold2)
{	
	return manifold1.copy() += manifold2;
}

template<class TYPE>
RUMBA::Manifold<TYPE> operator+ (const RUMBA::Manifold<TYPE>& manifold, const TYPE val)
{	
	return manifold.copy() += val;
}

template<class TYPE>
RUMBA::Manifold<TYPE> operator+ (const TYPE val, const RUMBA::Manifold<TYPE>& manifold)
{	
	return manifold.copy() += val;
}

template<class TYPE>
RUMBA::Manifold<TYPE> operator- (const RUMBA::Manifold<TYPE>& manifold1, const RUMBA::Manifold<TYPE>& manifold2)
{	
	return manifold1.copy() -= manifold2;
}

template<class TYPE>
RUMBA::Manifold<TYPE> operator- (const RUMBA::Manifold<TYPE>& manifold, const TYPE val)
{	
	return manifold.copy() -= val;
}

template<class TYPE>
RUMBA::Manifold<TYPE> 
operator- (const TYPE val, const RUMBA::Manifold<TYPE>& manifold)
{	
	return manifold.copy()  *= -1 += val;
}

template<class TYPE>
RUMBA::Manifold<TYPE> 
operator* (const RUMBA::Manifold<TYPE>& manifold1, const RUMBA::Manifold<TYPE>& manifold2)
{	
	return manifold1.copy() *= manifold2;
}



template<class TYPE>
RUMBA::Manifold<TYPE> 
operator* (const RUMBA::Manifold<TYPE>& manifold, const TYPE val)
{	
	return manifold.copy() *= val;
}

template<class TYPE>
RUMBA::Manifold<TYPE> 
operator* (const TYPE val, const RUMBA::Manifold<TYPE>& manifold)
{	
	return manifold.copy() *= val;
}

template<class TYPE>
RUMBA::Manifold<TYPE> 
operator/ (const RUMBA::Manifold<TYPE>& manifold1, const RUMBA::Manifold<TYPE>& manifold2)
{	
	return manifold1.copy() /= manifold2;
}

template<class TYPE>
RUMBA::Manifold<TYPE> operator/ (const RUMBA::Manifold<TYPE>& manifold, const TYPE val)
{	
	#ifdef USE_EXCEPTIONS
	if ( val == 0 )
		throw  RUMBA::DivByZero();
	#endif
	if ( val == 0 )
		return manifold;
	return manifold.copy() /= val;
}

template<class TYPE>
RUMBA::Manifold<TYPE> operator/ (const TYPE val, const RUMBA::Manifold<TYPE>& manifold)
{	
	RUMBA::Manifold<TYPE> res (manifold);

	int s = res.size();
	int i;
//	TYPE *dstDataArray = res.data();
	#ifdef USE_EXCEPTIONS
	for ( i = 0; i<s;++i ){
		if ( manifold[i] == 0 ){
			throw RUMBA::DivByZero();
		}
		res[i] = val / manifold[i];
	}
	#else	
	for ( i=0; i<s;++i )
		res[i] = val / manifold[i];
	#endif

	return res;
}

template<class TYPE>
RUMBA::Manifold<TYPE>& RUMBA::Manifold<TYPE>::operator+= (const Manifold<TYPE>& manifold)
{	int s = size();
	const_iterator src = manifold.begin();
	iterator dest = begin();
	while (s--) *(dest++) += *(src++);
	return *this;
}

template<class TYPE>
RUMBA::Manifold<TYPE>& RUMBA::Manifold<TYPE>::operator+= (TYPE val)
{	int s = size();
	iterator dest = begin();
	while (s--) *(dest++)+= val;
	return *this;
}

template<class TYPE>
RUMBA::Manifold<TYPE>& RUMBA::Manifold<TYPE>::operator-= (const Manifold<TYPE>& manifold)
{	int s = size();
	const_iterator src = manifold.begin();
	iterator dest = begin();
	while (s--) *(dest++) -= *(src++);
	return *this;
}

template<class TYPE>
RUMBA::Manifold<TYPE>& RUMBA::Manifold<TYPE>::operator-= (TYPE val)
{	int s = size();
	iterator dest = begin();
	while (s--) *(dest++) -= val;
	return *this;
}

template<class TYPE>
RUMBA::Manifold<TYPE>& RUMBA::Manifold<TYPE>::operator*= (const Manifold<TYPE>& manifold)
{	
	int s = size();
	const_iterator src = manifold.begin();
	iterator dest = begin();
	while (s--) *(dest++) *= *(src++);
	return *this;
}

template<class TYPE>
RUMBA::Manifold<TYPE>& RUMBA::Manifold<TYPE>::operator*= (TYPE val)
{	
	int s = size();
	iterator dest = begin();
	while (s--) *(dest++) *= val;
	return *this;
}

template<class TYPE>
RUMBA::Manifold<TYPE>& RUMBA::Manifold<TYPE>::operator/= (const Manifold<TYPE>& manifold)
{	
	int s = size();
	const TYPE *srcDataArray = manifold.data();
	TYPE *dstDataArray = DataArray;
	while (s--) {
	#ifdef USE_EXCEPTIONS
		if ( *(srcDataArray) == 0 )
			throw RUMBA::DivByZero();
	#endif
		*(dstDataArray++) /= *(srcDataArray++);
	}
	return *this;
}

template<class TYPE>
RUMBA::Manifold<TYPE>& RUMBA::Manifold<TYPE>::operator/= (TYPE val)
{	
	#ifdef USE_EXCEPTIONS
	if ( val == 0 )
		throw RUMBA::DivByZero();
	#endif

	int s = size();
	TYPE *dstDataArray = DataArray;
	while (s--) *(dstDataArray++) /= val;
	return *this;
}

template <class TYPE>
void RUMBA::Manifold<TYPE>::threshold(const TYPE val)
{	
	int s = size();
	TYPE *dstDataArray = DataArray;
	while (s--) {	
		if (*(dstDataArray) < val) 
			*(dstDataArray) = rumba_cast<TYPE>( 0 );
		dstDataArray++;
	}
}

template <class TYPE>
RUMBA::Manifold<TYPE>* RUMBA::Manifold<TYPE>::crop(const TYPE val)
{	
	intPoint p (-1,-1,-1,-1);
//	int xx=-1, yy=-1, zz=-1, tt=-1;
	bool cropped = false;

	for (int i=0; i<Extent.x(); i++)
		for (int j=0; j<Extent.y(); j++)
			for (int k=0; k<Extent.z(); k++)
				for (int l=0; l<Extent.t(); l++)
					if (getElement(i,j,k,l) > val) {
						if (i<Extent.x()) p.x() = i;
						if (j<Extent.y()) p.y() = j;
						if (k<Extent.z()) p.z() = k;
						if (l<Extent.t()) p.t() = l;
						if (i>p.x()) p.x() = i;
						if (j>p.y()) p.y() = j;
						if (k>p.z()) p.z() = k;
						if (l>p.t()) p.t() = l;
						cropped = true;
					}

	Manifold<TYPE>* res;
	if (cropped) res = get(Extent.x(), Extent.y(), Extent.z(), Extent.t(), p.x()-Extent.x()+1, p.y()-Extent.y()+1, p.z()-Extent.z()+1, p.t()-Extent.t()+1);

	return res;
}





template <class TYPE>
void RUMBA::Manifold<TYPE>::allocate(const intPoint& p)
{	
#ifdef DEBUG
	log.logName() << "Calling Manifold<TYPE>::allocate(" 
		<< p.x() << "," << p.y() << "," << p.z() << "," << p.t() << ")"
		<< std::endl;
#endif

	if ( DataArray != NULL )
	{
		 log.logName() <<  "WARNING: calling Manifold<TYPE>::allocate while DataArray not null\n"	;
		#ifdef USE_EXCEPTIONS
		throw RUMBA::Exception ("allocate called while DataArray not null\n");
		#endif
	}

#ifdef DEBUG
	std::cerr << "ALLOCATING: " << (void *) DataArray << std::endl;
#endif

	Extent.x() = p.x() ? p.x() : 1; 
	Extent.y() = p.y() ? p.y() : 1; 
	Extent.z() = p.z() ? p.z() : 1; 
	Extent.t() = p.t() ? p.t() : 1;
	Skip=skip();
#ifdef DEBUG
	log.logName() << "Size allocated: " << size() << std::endl;
#endif
	DataArray = new TYPE[size()];
	std::fill(DataArray, DataArray+size(), 0);

#ifdef DEBUG
	log.logName() << "Leaving Manifold<TYPE>::allocate()"<<std::endl;
#endif

}



/*
template <class TYPE> int RUMBA::Manifold<TYPE>::getDataType() const { return -1; }
*/
template <class TYPE> std::string RUMBA::Manifold<TYPE>::getDataString() const { return ""; }



/**
*
*			Methods for loading files
*
*/


template <class TYPE>
void RUMBA::Manifold<TYPE>::loadData(ManifoldFile* f)
{	
#ifdef DEBUG
	log.logName() <<  "Calling Manifold<TYPE>::loadData" ;
#endif
	assert(size());
//	f->seekg(0);
//	f->cGet(const_cast<TYPE*>(DataArray),size());	
	for ( int i = 0; i<size(); i += Extent.x() )
	{
		f->seekg(i);
		f->cGet(const_cast<TYPE*>(DataArray+i),Extent.x());
	}
#ifdef DEBUG
	log.logName() <<  "Leaving Manifold<TYPE>::loadData" <<std::endl;
#endif

}










/*------------------------------------------------------------------------*/
//
//			Methods for saving files
//
/*------------------------------------------------------------------------*/







template <class TYPE>
void RUMBA::Manifold<TYPE>::saveData( ManifoldFile* f ) const
{
//	log.logName() << "Size is:" << size() << std::endl;
//	f->seekp(0);
//	f->cPut(DataArray,size() );

	for ( int i = 0; i<size(); i += Extent.x() )
	{
		f->seekp(i);
		f->cPut((DataArray+i),Extent.x());
	}

}







template<class TYPE>
void RUMBA::Manifold<TYPE>::readFromStream( std::istream& in, int start, int nelts)
{
	char* ptr = (char*) ( DataArray + start );
	in.read ( ptr, nelts*sizeof(TYPE) );
}

template<class TYPE>
void RUMBA::Manifold<TYPE>::
writeToStream( std::ostream& out, int start, int nelts) const
{
	char* ptr = (char*) ( DataArray + start );
	out.write( ptr, nelts*sizeof(TYPE) );
}

template <class TYPE>
void RUMBA::Manifold<TYPE>::test(int& PASS, int& FAIL, int& COUNT)
{
	//------------------------------------------------------------
	//  Tests for reference counts
	//------------------------------------------------------------
	RUMBA::Manifold<TYPE> M;
	++COUNT;

	if (*M.RefCount == 1)
	{
		++PASS;
	}
	else
	{
		++FAIL;
		std::cerr << "Default constructed manifold has reference count "
			<< "not equal to 1" << std::endl;
	}

	RUMBA::Manifold<TYPE> N(M);

	++COUNT;
	if (*M.RefCount == 2 && *N.RefCount == 2)
	{
		++PASS;
	}
	else
	{
		++FAIL;
		std::cerr << "Copy constructed manifold has reference count "
			<< "not equal to 2" << std::endl;
	}

	RUMBA::Manifold<TYPE> O;

	N=O;
	++COUNT;
	if (*M.RefCount == 1 && *N.RefCount == 2 && *O.RefCount == 2)
	{
		++PASS;
	}
	else
	{
		++FAIL;
		std::cerr << "Assignment test failed" << std::endl;
	}

	//---------------------- end reference count tests ------------



}
