#ifndef RUMBA_ARRAY_CAST_H
#define RUMBA_ARRAY_CAST_H


/**
  * \file arraycast.h
  * \author Donovan Rebbechi, Ben Bly, Giorgio Grasso
  * Copyright Ben Bly
  * This file is released under the artistic license. See the file
  * COPYING for details.
  */



namespace RUMBA
{
/** 
*  convert one array of binary data to another. 
*  dest is the destination, src is the data source, 
*  src_le and dest_le are the endian-nesses of the source and destination.
*  These stay constant throughout the lifetime of the object, and they're
*  assigned to the constructor so that users don't need to deal with the 
*  unwieldy syntax of function_type()
*  length is the array size.
*/
class ArrayCast
{
public:
	typedef void (*function_type)(char*, bool, const char*, bool, int length);
	ArrayCast( function_type f, bool dest_le, bool src_le ) 
		: f(f), DestLE(dest_le), SrcLE(src_le) 
	{}
	void operator()(char* dest, const char* src, int length)
	{
		f(dest,DestLE,src,SrcLE,swap);
	}
private:
	function_type f;	
	const bool SrcLE;
	const bool DestLE;
};


/**
  *	convertor<TYPE, TYPEa> reads TYPEa and writes TYPE.
  */
template<class TYPE, class TYPEa>
void array_convertor(char* dest, bool dest_le, const char* src, bool src_le, int length)
{
	TYPEa in;
	size_t inSize = sizeof(TYPEa);

	TYPE out;
	size_t outSize = sizeof(TYPE);

	while (length--)
	{
		memcpy ( (void*) &in, (void*) src, inSize );
		src += inSize;
		out = rumba_cast<TYPE>(in);
		memcpy ( (void*) dest, (void*) &out, outSize );
		dest += outSize;
	}
}

} // namespace

#endif
