#include <StimulateFile.h>
#include <rumba/factory.h>
#include <rumba/exception.h>
#include <rumba/parse.h>
#include <rumba/rumba_system.h>
#include <fstream>
#include <strstream>

using RUMBA::StimulateFile;
using RUMBA::ManifoldFile;


StimulateFile StimulateFileInstance(Exemplar("StimulateFile"));
StimulateFile* StimulateFileExemplar = &StimulateFileInstance;

StimulateFile::StimulateFile()
:ManifoldFile("StimulateFile")
{
	loadRc();
}

ManifoldFile* StimulateFile::getNew()
{
	return new StimulateFile;
}

StimulateFile::StimulateFile( Exemplar a )
:ManifoldFile(a,"StimulateFile")
{
	log.logName() << "loading rc ... ";
	loadRc(); // important !! we use this in isMine()
	log.logName() << "done ";

	log.logName() << "Calling " << classname << "(Exemplar)\n";
}

bool StimulateFile::isMine( std::string filename )
{
	log.logName() << "Calling " << classname << "::isMine(" << filename << ")\n";
	string ext;
	std::string::size_type p = filename.find_last_of(".");
	if (p==std::string::npos)
		return false;
	ext.assign( filename, p, filename.length()); 
	return( ext == header_extension || ext == data_extension );
}






// initialise Data
void StimulateFile::initFile(std::ios_base::openmode mode)
{
	log.logName() << "Calling initFile()" << "\n";
	Factory* f;	
	if ( ! HeaderData.count("normalized_datatype") )
	{
		HeaderData["normalized_datatype"] = normalizeDataType ( HeaderData["datatype"].asString() );
		f = getFactoryFromDataType ( HeaderData["normalized_datatype"].asString() );
	}
	else
	{
		f = getFactoryFromDataType  ( normalizeDataType(types["defaulttype"] ));
	}


	Data = f->makeIOHandler ( DataFile.c_str(), size(), mode );
}

void StimulateFile::loadRc()
{
	std::ifstream conf_in;

	std::string basename = classname + std::string(".rc");
	std::string conf = RUMBA::find_file_modules() + string("/") + basename; 
	log.logName() << "Opening rc file: " << conf << "\n";

	conf_in.open(conf.c_str());
	if ( !conf_in )
	{
		log.logName() << "Couldn't open file: " << conf << "\n";
		throw RUMBA::BadFile(conf);
	}
	
	RUMBA::rcFind(conf_in, "data extension", data_extension );
	RUMBA::rcFind(conf_in, "header extension", header_extension );
//	RUMBA::rcFind(conf_in, "header size", header_size );
	



	types["char"] = "";
	types["int16"] = "";
	types["int32"] = "";
	types["float32"] = "";
	types["float64"] = "";
	types["defaulttype"] = "";

	RUMBA::rcFind(conf_in, "char", types["char"] );
	RUMBA::rcFind(conf_in, "int16", types["int16"] );
	RUMBA::rcFind(conf_in, "int32", types["int32"] );
	RUMBA::rcFind(conf_in, "float32", types["float32"] );
	RUMBA::rcFind(conf_in, "float64", types["float64"] );
	RUMBA::rcFind(conf_in, "defaulttype", types["defaulttype"] );

}



RUMBA::Factory* StimulateFile::getFactoryFromDataType(std::string t)
{
	log.logName() << "In getFactoryFromDataType() : t is " << t << "\n";
	log.logName() << "types[float64] " << types["float64"] << "\n";
	if ( t == types["char"] || t == "char" )
		return CharFactory::get();
	else if (  t == "int16" )
		return ShortFactory::get();
	else if (  t == "int32" )
		return IntFactory::get();
	else if (  t == "float32" )
		return FloatFactory::get();
	else if (  t == "float64" )
		return DoubleFactory::get();

	throw RUMBA::Exception("getFactoryFromDataType failed");

}

std::string StimulateFile::normalizeDataType(std::string t)
{
	log.logName() << "In normalizeDataType() : t is " << t << "\n";
	if ( t == types["char"] || t == "char" )
		return "char";
	else if ( t == types["int16"] || t == "int16" )
		return "int16";
	else if ( t == types["int32"] || t == "int32" )
		return "int32";
	else if ( t == types["float32"] || t == "float32" )
		return "float32";
	else if ( t == types["float64"] || t == "float64" )
		return "float64";

	throw RUMBA::Exception("normalizeDataType failed");

}

void RUMBA::StimulateFile::saveHeader(const char* filename) 
{
	log.logName() << "Calling RUMBA::StimulateFile::saveHeader( " 
	   << filename << "	\n"; 
	copyHeaderDataFromManifold();

	int nDim;

	std::ofstream fout ( filename );

	// This is wrong ! fix me.
	if ( Extent.y() <= 1 && Extent.z()  <= 1 && Extent.t() <= 1) 
	{
		nDim = 1;
	}
	else if ( Extent.z() <= 1 && Extent.t() <= 1)
	{
		nDim = 2;
	}
	else if ( Extent.t() <= 1 )
	{
		nDim = 3;
	}
	else 
	{
		nDim = 4;
	}

	fout << "numDim:	" << nDim << "\n";

	fout << "dim:	";
	fout << Extent.x() << " ";
	if ( nDim >=2 ) { 
		fout << Extent.y() << " "; 
	}
	if ( nDim >= 3 ) {
		fout << Extent.z() << " ";
	}
	if ( nDim >= 4 ) {
		fout << Extent.t();
	}
	fout << "\n";

	fout << "dataType:	" << types[HeaderData["datatype"].asString()] << std::endl;


	fout.close();

}

void RUMBA::StimulateFile::loadHeader ( const char* filename )
{
	LittleEndian = false;
	char buf [ 256 ];
	std::ifstream headerFile ( filename );
	std::istrstream* inputStream;
	string name;
	int i,j;
	int nDim = 0;
	float d;
	float interval[4];
	char tmp;
	bool recognised = false;
	string dt;	// Data type
	string StdOrient;

	VoxelSize.x() = VoxelSize.y() = VoxelSize.z() = 1;

	while ( headerFile.getline ( buf, 256 ) )
	{
		inputStream = new std::istrstream ( buf );
		*inputStream >> name;
		recognised = true;
		if ( name == "numDim:"  ) // so ... which is it ?
		{
			*inputStream >> nDim;
			if ( nDim > 4 ) log.logName() << "Warning: More than 4 dimensions\n";
		} 
		else if ( name == "fidName:" )
		{
			*inputStream >> DataFile;
		}
		else if ( name == "sdtOrient:" )
		{
			*inputStream >> StdOrient;
		}
		else if ( name == "dataType:" )
		{
			*inputStream >> dt;
			HeaderData["datatype"] = dt;
		}
		else if ( name == "displayRange:" )
		{
			*inputStream >> DisplayRange[0];
			*inputStream >> DisplayRange[1];
		}
		else if ( name == "dim:" || name == "Dim:" )
		{
			for ( i = 0; i < nDim; ++i )
			{
				*inputStream >> j;
				switch (i)
				{
					case 0: Extent.x() = j; break;
					case 1: Extent.y() = j; break;
					case 2: Extent.z() = j; break;
					case 3: Extent.t() = j; break;
				}
			}
		}
		else if ( name == "origin:" )
		{
			for ( i = 0; i < nDim; ++i )
			{
				*inputStream >> d;
				switch (i)
				{
					case 0: Origin.x() = d; break;
					case 1: Origin.y() = d; break;
					case 2: Origin.z() = d; break;
					case 3: Origin.t() = d; break;
				}

			}
		}
		else if ( name == "fov:" )
		{
			for ( i = 0; i < nDim; ++i )
			{
				*inputStream >> d;
				switch (i)
				{
					case 0: VoxelSize.x() = d; break;
					case 1: VoxelSize.y() = d; break;
					case 2: VoxelSize.z() = d; break;
					// case 3: Should always be 1.
				}

			}
		}
		else if ( name == "interval:" )
		{
			for ( i = 0; i < nDim; ++i )
			{
				*inputStream >> d;
				if ( i < 4 ) interval[i] = d;
			}

		}
		else
		{
			recognised = false;
		}
	
		if ( recognised )
		{
			if ( inputStream->fail() )
				log.logName() << "Warning: Fail at line\n" << buf << "\n";
			if ( *inputStream >> tmp ) 
				log.logName() << "Warning: Trailing garbage in line\n" << buf << "\n";
		}
		delete inputStream;
	
	} /* while */	

	// Use interval to set VoxelSize.x,Y,Z
	if ( VoxelSize.x() == 0 && Extent.x() != 0 ) 
		VoxelSize.x() = interval[0] / Extent.x();
	if ( VoxelSize.y() == 0 && Extent.y() != 0 ) 
		VoxelSize.y() = interval[1] / Extent.y();
	if ( VoxelSize.z() == 0 && Extent.z() != 0 ) 
		VoxelSize.z() = interval[2] / Extent.z();

	headerFile.close();
	copyHeaderDataToManifold();
	Skip=skip();

}



