#ifndef BASE_MANIFOLD_H
#define BASE_MANIFOLD_H


/**
  * \file baseManifold.h
  * \author Donovan Rebbechi, Ben Bly, Giorgio Grasso
  * Copyright Ben Bly
  * This file is released under the artistic license. See the file
  * COPYING for details.
  */



// #include <fstream>
#include <string>
#include <map>
#include <rumba/splodge.h>
#include <rumba/log.h>
#include <rumba/point.hpp>

#include <rumba/orientation.h>


namespace RUMBA {

class ManifoldFile; // forward declaration





/**
 *	The BaseManifold is a class that Manifold inherits from.
 *	Its main purpose is to enable certain operations on data
 *	obtained from files without having to coerce the data into
 *	a certain format. That is, the BaseManifold class enables 
 *	dynamic instantiation of a Manifold, and the type of the 
 *	Manifold is determined by the file.
 *
 *	Note that BaseManifold is an abstract class -- you don't 
 *	instantiate a BaseManifold. What you do is create a Manifold
 *	using BaseManifold::construct, and operate via that pointer 
 *	without having to worry about what "type" of Manifold you have.
 *
 */



class BaseManifold {


public:
	class proxy;
	class const_proxy;
	class const_iterator;
	class iterator;

	class iterator
	{
	public:
		typedef std::random_access_iterator_tag iterator_category;
		typedef double value_type;
		typedef proxy reference;
		typedef int difference_type;
		typedef void pointer;

		iterator(BaseManifold* M, int i) : M(M), i(i) {}
		reference operator*(){ return proxy(M,i); }
		iterator operator++(){ ++i; return *this; }
		iterator operator++(int){ iterator it(M,i); ++i; return it; }
		iterator operator--(){ --i; return *this; }
		iterator operator--(int);		
		iterator& operator+=(int j){ i+=j; return *this; }
		iterator& operator-=(int j){ i-=j; return *this; }
		iterator operator+(int j){ return iterator(M,i+j); }
		iterator operator-(int j){ return iterator(M,i-j); }
		friend bool operator==(iterator,iterator);
		friend bool operator!=(iterator,iterator);
		friend difference_type operator-(iterator,iterator);
		friend class const_iterator;
	private:
		BaseManifold* M;
		int i;			
	};

	class const_iterator
	{
	public:
		typedef std::random_access_iterator_tag iterator_category;
		typedef double value_type;
		typedef const_proxy reference;
		typedef int difference_type;
		typedef void pointer;

		const_iterator(iterator it):M(it.M),i(it.i) {}
		const_iterator(const BaseManifold* M, int i) : M(M), i(i) {}
		const_proxy operator*(){ return const_proxy(M,i); }
		const_iterator operator++(){ ++i; return *this; }
		const_iterator operator++(int){ const_iterator it(M,i); ++i; return it; }
		const_iterator operator--(){ --i; return *this; }
		const_iterator operator--(int);		
		const_iterator& operator+=(int j){ i+=j; return *this; }
		const_iterator& operator-=(int j){ i-=j; return *this; }
		const_iterator operator+(int j){ return const_iterator(M,i+j); }
		const_iterator operator-(int j){ return const_iterator(M,i-j); }
		friend bool operator==(const_iterator,const_iterator);
		friend bool operator!=(const_iterator,const_iterator);
		friend difference_type operator-(const_iterator,const_iterator);
	private:
		const BaseManifold* M;
		int i;			
	};


	class proxy
	{
	public:	
		friend class const_proxy;
		proxy(BaseManifold* M, int i):M(M),i(i){}
		proxy& operator=(double val)
		{
			M->setElementDouble(i, val);
			return *this;
		}
		operator double() const
		{
			return M->getElementDouble(i);
		}
		proxy& operator=(int val)
		{
			M->setElementInt(i,val);
			return *this;
		}
		operator int() const
		{
			return M->getElementInt(i);
		}
		proxy& operator=(const proxy& p)
		{
			M->setElementDouble(i, static_cast<double>(p) );
			return *this;
		}
	private: 
		BaseManifold* M; 
		int i;
	};

	class const_proxy
	{
	public:	
		const_proxy(const BaseManifold* M, int i):M(M),i(i){}

		operator double()
		{
			return M->getElementDouble(i);
		}
		operator int()
		{
			return M->getElementInt(i);
		}
		const_proxy& operator=(const proxy& p)
		{
			M=p.M;
			i=p.i;
			return *this;
		}
		const_proxy& operator=(const const_proxy& p)
		{
			M=p.M;
			i=p.i;
			return *this;
		}

	private: 
		const BaseManifold* M; 
		int i;
	};



	proxy operator[](int i)
	{
		return proxy(this,i);
	}

	const_proxy operator[](int i) const
	{
		return const_proxy(this,i);
	}

	iterator begin()
	{
		return iterator(this,0);
	}
	iterator end()
	{
		return iterator(this,size());
	}
	const_iterator begin() const
	{
		return const_iterator(this,0);
	}
	const_iterator end() const
	{
		return const_iterator(this,size());
	}
	/* 	Constructors/Destructors	*/


/**
  *		Copy header fields and dimensions
  *
  */

	void copyHeader(const BaseManifold&);
/**
*	Loads a manifold from a file. use loadImage=false if you just want to
*	scan the headers without loading the image data.
*/
	void load ( std::string filename, bool loadImage = true );
/**
*	Load data from an image file. Usually, you would just use load ( filename )
*	but you can use this if you want to scan the headers and load data later.
*/
	virtual	void loadData ( ManifoldFile* f ) = 0;

/**
*	Write Manifold to file. This method determines the desired filetype from
*	the filename.
*/
	void save( std::string filename, bool save_image = true ) const;
/**
*	Print out info about the Manifold. For debugging purposes.
*/
	virtual void print(std::ostream& stream = std::cout) const = 0;

/**
*	This method lets you read from the BaseManifold without knowing what
*	the underlying data type is. The position is the position in the array.
*/
	virtual int getElementInt ( int pos ) const = 0;
/**
*	Get element at specified x,y,z,t coordinates, expressed as an integer.
*/
	virtual int getElementInt ( int x, int y, int z, int t ) const =0;
/**
 * 	Get element as specified point
 */
	virtual int getElementInt ( const intPoint& p ) const =0;


/**
*	Retrieve element by position in the array as a double.
*/
	virtual double getElementDouble ( int pos ) const = 0;
/**
*	Get element at specified x,y,z,t coordinates, expressed as a double.
*/
	virtual double getElementDouble ( int x, int y, int z, int t ) const =0;
/**
 * 	Get element as specified point
 */
	virtual double getElementDouble ( const intPoint& p ) const =0;


/**
*	Set element at pos in the array to value val. val is converted into the
*	appropriate data type.
*/	
	virtual void setElementDouble ( int pos,double val )=0;
/**
*	Set element at coordinates (x,y,z,t) to value val. val is converted into the
*	appropriate data type.
*/
	virtual void setElementDouble ( int x, int y, int z, int t, double val )=0;
/**
*	Set element at coordinates (x,y,z,t) to value val. val is converted into the
*	appropriate data type.
*/
	virtual void setElementDouble ( const intPoint& p, double val )=0;

/**
*	Set element at pos in the array to value val. val is converted into the
*	appropriate data type.
*/	
	virtual void setElementInt( int pos,int val )=0;
/**
*	Set element at coordinates (x,y,z,t) to value val. val is converted into the
*	appropriate data type.
*/
	virtual void setElementInt ( int x, int y, int z, int t, int val )=0;
/**
*	Set element at coordinates (x,y,z,t) to value val. val is converted into the
*	appropriate data type.
*/
	virtual void setElementInt ( const intPoint& p, int val )=0;


/**
  *	Return dimensions of the manifold
  */
	intPoint extent() const;

/**
*	Specify the dimensions of the Manifold. This does *NOT* allocate data.
*/
	void setExtent(const intPoint&);
/**
*	Returns the width of the Manifold ( number of points on the x-axis ).
*/
	int width() const;
/**
*	Returns the height of the Manifold ( number of points on the y-axis )
*/
	int height() const;
/**
*	Returns the depth of the Manifold ( number of points on the z axis )
*/
	int depth() const;
/**
*	Return the number of time points in the Manifold.
*/
	int timepoints() const;

/**
  * Same as width()*height()*depth()
  */
	int pixels() const;
/**
*	Return the number of points on the manifold. 
*/
	int size() const;
/**
*	Return the number of units between voxels
*/
	doublePoint voxelSize() const;

/**
*	Set the number of units between voxels in the x direction
*/
	void setVoxelSize(const doublePoint&);

/**
  *	Returns the number of pixels to skip to travel 1 unit in a given direction
  */
	intPoint skip() const;


/**
  *	Read/write data from/to an i/o stream. Does NOT do byteswapping or
  *	error checking. First parameter is start, second is number of elements
  */
	virtual void readFromStream ( std::istream&, int , int ) {} 
	virtual void writeToStream ( std::ostream&, int , int ) const {} 
	virtual ~BaseManifold() = 0;

/**
  * Allow clients direct access to header data.
  * This method can't be constant since it gives out a handle.
  */
	std::map<std::string, RUMBA::Splodge> & headerData();


/*
   * const version of the above.
   *
   */
	const std::map<std::string, RUMBA::Splodge> & headerData() const;

	bool littleEndian();
	void setLittleEndian(bool b);

	virtual BaseManifold* get(const intPoint& o, const intPoint& e) const = 0;
	virtual BaseManifold* 
		get(int x, int y, int z, int t, int dx, int dy, int dz, int dt) const= 0;
	Orientation orientation()  const;


protected:
	BaseManifold();

	virtual void allocate(const intPoint& ) { log.logName()<< "Warning: allocate() stub called" << "\n"; }
	virtual void saveData (ManifoldFile*) const {}

	virtual void copyHeaderDataToManifold();// copies  header fields to Extent/VoxelSize 
	virtual void copyHeaderDataFromManifold(); // copies Extent/VoxelSize to header fields


	/*--------------------------------------------------*/
	/*				Data Members						*/
	/*--------------------------------------------------*/
	mutable Log log;
	bool LittleEndian; 
	intPoint Extent;
	doublePoint VoxelSize;
	Orientation Orient;
	doublePoint Origin;
	intPoint Skip; // redundant, but speeds up computations.
	std::map < std::string, RUMBA::Splodge > HeaderData;
};

inline bool operator==
(BaseManifold::iterator left, BaseManifold::iterator right)
{ return  left.i == right.i; }

inline bool operator!=
(BaseManifold::iterator left, BaseManifold::iterator right)
{ return  left.i != right.i;  }

inline int operator-
(BaseManifold::iterator left, BaseManifold::iterator right)
{ return  left.i - right.i;  }

inline bool operator==
(BaseManifold::const_iterator left, BaseManifold::const_iterator right)
{ return  left.i == right.i; }

inline bool operator!=
(BaseManifold::const_iterator left, BaseManifold::const_iterator right)
{ return  left.i != right.i;  }

inline int operator-
(BaseManifold::const_iterator left, BaseManifold::const_iterator right)
{ return  left.i - right.i;  }




} // namespace RUMBA



#endif
