#include <algorithm>

#include <rumba/manifoldmatrix.h>
#include <rumba/orientation.h>
#include <rumba/orientation_impl.h>

//------------------------- implementation code

using RUMBA::BaseManifold;
using RUMBA::intPoint;

namespace
{
// From format definition.
const char PERMUTATION_SAC = 1;
const char PERMUTATION_ASC = 2;
const char PERMUTATION_ACS = 3;
const char PERMUTATION_CSA = 5;
const char PERMUTATION_SCA = 6;
const char PERMUTATION_CAS = 7;

// given a representation  such as (x,y,z) where x y z are PATIENT_LEFT ... etc,
// we want the property that  (x|y|z)&7 gives the parity field of the spec.
const char PATIENT_RIGHT = 1; // note: same as lowest order parity bit
const char PATIENT_FRONT = 2; // note: same as middle order parity bit
const char PATIENT_FOOT  = 4; // note: same as high order parity bit

//const char PATIENT_LEFT  = 8;
//const char PATIENT_HEAD  = 16;
//const char PATIENT_BACK  = 32;
//
const char PATIENT_LEFT  = 8;
const char PATIENT_BACK  = 16;
const char PATIENT_HEAD  = 32;

const char PATIENT_TIME  = 64;

const char SAGITAL = PATIENT_LEFT | PATIENT_RIGHT;
const char CORONAL = PATIENT_FRONT| PATIENT_BACK;
const char AXIAL   = PATIENT_HEAD | PATIENT_FOOT;
}

namespace RUMBA
{

pdir_t patient_left() { return PATIENT_LEFT; }
pdir_t patient_right() { return PATIENT_RIGHT; }
pdir_t patient_head() { return PATIENT_HEAD; }
pdir_t patient_foot() { return PATIENT_FOOT; }
pdir_t patient_front() { return PATIENT_FRONT; }
pdir_t patient_back() { return PATIENT_BACK; }
pdir_t patient_time() { return PATIENT_TIME; }

}

RUMBA::OrientationImpl::~OrientationImpl() {}

char RUMBA::OrientationImpl::toChar() const
{
	return Rep;
}



void RUMBA::OrientationImpl::convert_to_array ( char c, char* arr, bool* Xyzt )
{
	char parity = c&7;

	arr[0] = (parity & SAGITAL) ? SAGITAL & 7 : SAGITAL & 56;
	arr[1] = (parity & CORONAL) ? CORONAL & 7 : CORONAL & 56;
	arr[2] = (parity & AXIAL  ) ? AXIAL   & 7 : AXIAL   & 56;

	char permutation = (c>>3)&7;

	// MSB: coronal before axial
	if (!(permutation & 4)) std::swap(arr[1],arr[2]);

	// LSB: sagital not first.  Middle bit: sagital not second
	// Note that we preserve coronal/axial order here
	switch (permutation & 3) 
	{
		case 0:
			throw RUMBA::Exception 
				("Ill formed: LSB and MB cannot both be zero " );
			break;
		case 1:
			std::swap(arr[0],arr[1]); 
			break;
		case 2: 
			break;
		case 3:
			std::swap(arr[0],arr[1]);
			std::swap(arr[1],arr[2]);
			break;
	}

	if (Xyzt)
		*Xyzt = c|64 ? true : false;

}

char RUMBA::OrientationImpl::convert_to_char(const char* arr, bool b)
{
	char parity = (arr[0]|arr[1]|arr[2])&7; 
	char perm = 0;

	// LSB : first axis is not sagital
	if (!(arr[0]&SAGITAL)) perm+=1; 

	// middle bit: second axis is not sagital
	if (!(arr[1]&SAGITAL)) perm+=2; 

	// MSB: coronal before axial <=> coronal first or axial last
	if ( arr[2]&AXIAL || arr[0]&CORONAL )
		perm += 4;

	return (perm<<3|parity)|(b?64:0);
	
}

RUMBA::Orientation::Orientation()
	
	: impl ( new RUMBA::OrientationImpl( 
				patient_right(), patient_front(), patient_foot(), 
				intPoint(),true)
			)
{

}

char RUMBA::Orientation::toChar() const
{
	if (impl) 
		return impl->toChar();
	else
		return 0;
}

RUMBA::Orientation::Orientation(const RUMBA::Orientation& o)
	: impl ( new OrientationImpl(*o.impl) )
{

}

RUMBA::Orientation& RUMBA::Orientation::operator=(const RUMBA::Orientation& o)
{
	RUMBA::OrientationImpl* x = 0;
   if (o.impl) 
	   x = new OrientationImpl(*o.impl);
   impl = x;
   return *this;
}

RUMBA::Orientation::Orientation ( pdir_t o1,pdir_t o2,pdir_t o3,
		const intPoint& e, bool xyzt)
: impl ( new RUMBA::OrientationImpl(o1,o2,o3,e,xyzt) )
{
}

RUMBA::Orientation::~Orientation() {}

RUMBA::intPoint RUMBA::Orientation::extent() const
{
	return impl->extent();
}

void RUMBA::Orientation::setExtent(const RUMBA::intPoint& p) 
{
	impl->setExtent(p);
}

int RUMBA::Orientation::operator[](int x) const
{
	return impl->operator[](x);
}

char RUMBA::OrientationImpl::operator[](int x)
{
	char arr[3]; bool flag;
	convert_to_array(Rep, arr, &flag);
	if (x>3)
		throw RUMBA::Exception("Argument out of range");
	return arr[x];

}





RUMBA::PointTransform RUMBA::Orientation::transform(const Orientation& o) const
{
	RUMBA::PointTransform 
		t1((RUMBA::PointTransformImpl*)buildTransform(*this, impl->extent()));
	RUMBA::PointTransform 
		t2((RUMBA::PointTransformImpl*)buildTransform(o,o.impl->extent()));

//	return RUMBA::PointTransform::compose ( t2, t1.invert() );
	return RUMBA::PointTransform::compose ( t1, t2.invert() );




}


RUMBA::PointTransformImpl::~PointTransformImpl(){}

RUMBA::PointTransform::PointTransform() : impl (0){}
RUMBA::PointTransform::PointTransform(PointTransformImpl* impl):impl(impl){}
RUMBA::PointTransform::PointTransform(const PointTransform& pt)
	: impl(pt.impl->clone())
{
}
RUMBA::PointTransform& 
RUMBA::PointTransform::operator=(const PointTransform& pt)
{
	if (&pt != this) 
	{
		RUMBA::PointTransformImpl* tmp = impl; 
		impl = pt.impl->clone();
		delete tmp; // delete only when memry allocation in clone() is succesful
	}
	return *this;
}
RUMBA::PointTransform::~PointTransform() { delete impl; }

RUMBA::intPoint RUMBA::PointTransform::operator() (const RUMBA::intPoint& p)
{
	RUMBA::intPoint res; 
	impl->operator()(p, res);
	return res;
}

void RUMBA::PointTransform::operator() 
(const RUMBA::intPoint& p, RUMBA::intPoint& q)
{
	impl->operator()(p, q);
}

RUMBA::intPoint RUMBA::PointTransform::map_extent (const RUMBA::intPoint& p)
{
	RUMBA::intPoint res; 
	impl->map_extent(p, res);
	return res;
}

RUMBA::PointTransform RUMBA::PointTransform::invert() const
{
	return RUMBA::PointTransform(impl->invert());
}




RUMBA::PointTransform 
RUMBA::PointTransform::compose
(
 const RUMBA::PointTransform& left, 
 const RUMBA::PointTransform& right
 )
{
	return PointTransform(left.impl->compose (right.impl));
}

RUMBA::BoundingBox 
RUMBA::PointTransform::operator() (const RUMBA::BoundingBox& b)
{
	RUMBA::intPoint s,e;
	impl->operator() ( b.start, s );
	impl->operator() ( b.extent, e );
	return RUMBA::BoundingBox(s,e,b.M);
}

void RUMBA::PointTransform::debug() 
{
	impl->debug();
}



RUMBA::PermutationTransform::PermutationTransform
(const PermutationTransform& t)
	: PointTransformImpl()
{
	memmove ( Rep, t.Rep, sizeof(Rep) );
	memmove ( Parity, t.Parity, sizeof(Parity) );
	extent = t.extent;

	
}

RUMBA::PermutationTransform::PermutationTransform() 
{
	Rep[0]=0; Rep[1]=1; Rep[2]=2; Rep[3] = 3;
	Parity[0] = Parity[1] = Parity[2] = Parity[3] = 1;
}

RUMBA::PermutationTransform::
	PermutationTransform(const int* perm, const char* par, const intPoint& extent)
		: extent(extent)
{
	memcpy (Rep, perm, sizeof(Rep));
	memcpy (Parity, par, sizeof(Parity));
}

void RUMBA::PermutationTransform::operator() (const intPoint& p, intPoint& q)
{
	q[0] = p[Rep[0]];
	q[1] = p[Rep[1]];
	q[2] = p[Rep[2]];
	q[3] = p[Rep[3]];

	if(Parity[0]==-1) q[0] = extent[0] - q[0] -1;
	if(Parity[1]==-1) q[1] = extent[1] - q[1] -1;
	if(Parity[2]==-1) q[2] = extent[2] - q[2] -1;
	if(Parity[3]==-1) q[3] = extent[3] - q[3] -1;

}

void RUMBA::PermutationTransform::map_extent (const intPoint& p, intPoint& q)
{
	q[0] = p[Rep[0]];
	q[1] = p[Rep[1]];
	q[2] = p[Rep[2]];
	q[3] = p[Rep[3]];
}

RUMBA::PermutationTransform* 
RUMBA::PermutationTransform::clone() 
{
	PermutationTransform* res = new PermutationTransform(*this);
	return res;
}

RUMBA::PermutationTransform* 
RUMBA::PermutationTransform::invert() const
{
	// first invert permutation.
	int arr[4];
	char par[4];
	RUMBA::intPoint e;
	for ( int i = 0; i < 4; ++i ) arr[Rep[i]] = i;

	// now apply this permutation to parity.

	for ( int i = 0; i < 4; ++i ) par[Rep[i]] = Parity[i];

	// also need to apply it to transform extents

//	for ( int i = 0; i < 4; ++i ) e[i] = extent[Rep[i]];
	for ( int i = 0; i < 4; ++i ) e[Rep[i]] = extent[i];

	PermutationTransform* tmp = new PermutationTransform(arr,par,e);

	return tmp;

}

RUMBA::PermutationTransform* 
RUMBA::PermutationTransform::compose(const RUMBA::PointTransformImpl* t)
{
	const PermutationTransform* x;
	if (( x = dynamic_cast<const PermutationTransform*>(t)))
		return compose_impl(x);
	else
		return 0;
}

RUMBA::PermutationTransform* 
RUMBA::PermutationTransform::compose_impl(const RUMBA::PermutationTransform* t)
{
	int arr[4];
	char par[4];
	RUMBA::intPoint e;

	// first compute the new permutation
	for ( int i = 0; i < 4; ++i )
		arr[i] = Rep[t->Rep[i]];

	// now the fun part: work out parity.
	for ( int i = 0; i < 4; ++i )
		par[i] = t->Parity[i] * Parity[t->Rep[i]];

//	for ( int i = 0; i < 4; ++i )
//		par[i] = Parity[i] * t->Parity[Rep[i]];


	// If the first transforms extent has non-zero dimension, use it. 
	// Otherwise, we'd better use the second transforms extent

	if (t->extent.x()!=0 || t->extent.y()!=0||t->extent.z()!=0||t->extent.t()!=0)
	{
		e = t->extent;
	}
	else
	{
		// fix extent

		for (int i = 0; i < 4; ++i )
		for (int j = 0; j < 4; ++j )
			if (Rep[i]==t->Rep[j])
				e[i]=t->extent[j];


		for (int i = 0; i < 4; ++i )
			e[i] = extent[ t->Rep[i] ];




	}


	PermutationTransform* tmp = new PermutationTransform(arr,par,e);
	return tmp;

}


// build transform map from orientation o to reference orientation.
RUMBA::PermutationTransform*
buildTransform(const RUMBA::Orientation& o, const RUMBA::intPoint& ext)
{
	bool b=false; char arr[3] = {0,0,0};
	RUMBA::OrientationImpl::convert_to_array(o.toChar(), arr, &b);	


	int perm_arg[4];
	char parity_arg[4];
	RUMBA::intPoint e;

	int offset = b ? 0: 1;

	// time
	if (b) parity_arg[0]=1; else parity_arg[3]=1;

	// space
	parity_arg[offset+0]= arr[0]&7 ? 1:-1;
	parity_arg[offset+1]= arr[1]&7 ? 1:-1;
	parity_arg[offset+2]= arr[2]&7 ? 1:-1;

	for ( int i = 0; i < 3; ++i )
	{
		if (arr[i] & SAGITAL) 
			perm_arg[i+offset] = 0;
		else if (arr[i] & CORONAL) 
			perm_arg[i+offset] = 1;
		else if (arr[i] & AXIAL) 
			perm_arg[i+offset] = 2;
		else
			std::cerr << "couldn't identify perm_argtype" << std::endl;
	}



	if (b)
		perm_arg[3] = 3;
	else
		perm_arg[3] = 0;

	// invert permutation for perm_arg
	int perm_arg2[4];
	char parity_arg2[4];
	for ( int i = 0; i < 4; ++i )
	{
		perm_arg2[perm_arg[i]] = i;
		parity_arg2[perm_arg[i]] = parity_arg[i];
	}

	/* fix extent so it matches image space */
	for (int i = 0; i < 4; ++i )
		e[i] = ext[perm_arg2[i]];




	RUMBA::PermutationTransform* tmp = 
		new RUMBA::PermutationTransform(perm_arg2, parity_arg2, e);
	return tmp;

};


void RUMBA::volume_transform (
		const BaseManifold& in, int in_vol, BaseManifold& out, int out_vol)
{
	RUMBA::intPoint p,q;
	RUMBA::PointTransform f;
	p.t() = in_vol;
	q.t() = out_vol;

   	f = RUMBA::transform( in.orientation(), out.orientation() );
	for ( p.x() = 0; p.x() < in.width(); ++p.x() )
		for ( p.y() = 0; p.y() < in.height(); ++p.y() )
			for ( p.z() = 0; p.z() < in.depth(); ++p.z() )
			{
				f(p,q);
				q.t() = out_vol;
				out.setElementDouble( q, in.getElementDouble (p) );
			}
}

void RUMBA::volume_transform( const BaseManifold& in, BaseManifold& out)
{
	if (in.timepoints()!=out.timepoints())
		throw RUMBA::Exception("Incompatible sizes in volume_transform()");
	for (int i = 0; i < in.timepoints(); ++i )
		volume_transform(in,i,out,i);
}
