/*
 *
 *   attr.c - Ruby interface for ImageMagick
 *
 *   Copyright (C) Ryuichi Tamura (tam@kais.kyoto-u.ac.jp)
 *
 *   $Date: 2001/07/22 08:16:43 $
 *   $Revision: 1.15 $
 *
 */

#include "magick.h"


/******************************************************************************

     S E T   A T T R I B U T E S

*******************************************************************************/

static void
set_adjoin(const char *key, MgkImage *mgk, Image *image,  VALUE bool_val)
{
    ImageInfo *info;

    GetInfoStruct(mgk, info);
    info->adjoin = RTEST(bool_val);
}

static void
set_antialias(const char *key, MgkImage *mgk, Image *image, VALUE bool_val)
{
    ImageInfo *info;

    GetInfoStruct(mgk, info);
    info->antialias = RTEST(bool_val);
}

static void
set_background (const char *key, MgkImage *mgk, Image *image, VALUE val)
{
    PixelPacket target_color;
    ImageInfo *info;

    Check_Type(val, T_STRING);
    GetInfoStruct(mgk, info);
    if (QueryColorDatabase(RSTRING(val)->ptr, &target_color))
        info->background_color = target_color;
    if (!image)
        return;
    image->background_color = target_color;
}

static void
set_blue_primary(const char *key, MgkImage *mgk, Image *image,  VALUE val)
{
    char *str;
    double x, y;

    if (!image)
        return;
    Check_Type(val, T_STRING);
    str = RSTRING(val)->ptr;
    sscanf(str, "%lf,%lf", &x, &y);
    image->chromaticity.blue_primary.x = x;
    image->chromaticity.blue_primary.y = y;
}

static void
set_bordercolor(const char *key, MgkImage *mgk, Image *image, VALUE val)
{
    PixelPacket target_color;
    ImageInfo *info;

    Check_Type(val, T_STRING);
    GetInfoStruct(mgk, info);
    if (QueryColorDatabase(RSTRING(val)->ptr, &target_color))
        info->border_color = target_color;
    if (!image)
        return;
    image->border_color = target_color;
}

static void
set_cache_threshold(const char *key, MgkImage *mgk, Image *image, VALUE val)
{
    SetCacheThreshold(NUM2INT(val));
}

static void
set_colormap(const char *key, MgkImage *mgk, Image *image, VALUE color_val)
{
    PixelPacket *color;
    int i, red, green, blue;
    char *arg;

    if (!image)
        return;
    Check_Type(color_val, T_STRING);
    arg = RSTRING(color_val)->ptr;
    i=0;
    sscanf(key,"%*[^[][%d",&i);

    if (i > image->colors)
        i %= image->colors;
    if ( !strchr(arg, ',') )
        QueryColorDatabase(arg, image->colormap+i);
    else {
        color=image->colormap+i;
        red=color->red;
        green=color->green;
        blue=color->blue;
        sscanf(arg, "%d,%d,%d", &red, &green, &blue);
        color->red = ((red < 0) ? 0 : (red > MaxRGB) ? MaxRGB : red);
        color->green = ((green < 0) ? 0 : (green > MaxRGB) ? MaxRGB : green);
        color->blue=((blue < 0) ? 0 : (blue > MaxRGB) ? MaxRGB : blue);
    }
}


static void
set_color_space(const char *key, MgkImage *mgk, Image *image, VALUE colorspace_type)
{
    ImageInfo *info;

    GetInfoStruct(mgk, info);
    info->colorspace = FIX2UINT(colorspace_type);
    if (!image)
        return;
    RGBTransformImage(image, FIX2UINT(colorspace_type));
}

static void
set_compress(const char *key, MgkImage *mgk, Image *image, VALUE compression_type)
{
    ImageInfo *info;

    GetInfoStruct(mgk, info);
    info->compression = FIX2UINT(compression_type);
    if (!image)
        return;
    image->compression = FIX2UINT(compression_type);
}

static void
set_delay(const char *key, MgkImage *mgk, Image *image, VALUE val)
{
    if (!image)
        return;
    image->delay = FIX2UINT(val);
}

static void
set_density(const char *key, MgkImage *mgk, Image *image, VALUE val)
{
    ImageInfo *info;
    char *geom;
    int count;

    Check_Type(val, T_STRING);
    geom = RSTRING(val)->ptr;
    GetInfoStruct(mgk, info);
    if (!IsGeometry(geom)){
        MagickWarning(OptionWarning,"Invalid geometry on density", geom);

    }
    CloneString(&info->density, geom);

    if (!image)
        return;
    count = sscanf(info->density, "%lfx%lf",
                   &image->x_resolution, &image->y_resolution);
    if (count != 2)
        image->y_resolution = image->x_resolution;
}

static void
set_depth(const char *key, MgkImage *mgk, Image *image, VALUE val)
{
    ImageInfo *info;
    unsigned int depth;

    GetInfoStruct(mgk, info);
    depth = FIX2UINT(val);
    info->depth = depth;

    if (!image)
        return;
    image->depth = depth;
    SetImageDepth(image, depth);
}

static void
set_dispose(const char *key, MgkImage *mgk, Image *image, VALUE val)
{
    Image *im;

    if (!image)
        return;
    for (im = mgk->ref->image; im; im = im->next)
        im->dispose = FIX2UINT(val);
}

static void
set_dither(const char *key, MgkImage *mgk, Image *image, VALUE bool_val)
{
    ImageInfo *info;

    GetInfoStruct(mgk, info);
    info->dither = RTEST(bool_val);
}

static void
set_display(const char *key, MgkImage *mgk, Image *image, VALUE display_name)
{
    ImageInfo *info;

    Check_Type(display_name, T_STRING);
    GetInfoStruct(mgk, info);
    CloneString(&info->server_name, RSTRING(display_name)->ptr);
}

static void
set_file(const char *key, MgkImage *mgk, Image *image, VALUE val)
{
    ImageInfo *info;
    OpenFile *fptr;

    GetInfoStruct(mgk, info);
    Check_Type(val, T_FILE);
    GetOpenFile(val, fptr);
    info->file = fptr->f;
}

static void
set_filename(const char *key, MgkImage *mgk, Image *image, VALUE val)
{
    ImageInfo *info;

    Check_Type(val, T_STRING);
    GetInfoStruct(mgk, info);
    strncpy(info->filename, RSTRING(val)->ptr, MaxTextExtent-1);
}

static void
set_filter(const char *key, MgkImage *mgk, Image *image, VALUE filter_type)
{
    if (!image)
        return;
    image->filter = FIX2UINT(filter_type);
}

static void
set_font(const char *key, MgkImage *mgk, Image *image, VALUE val)
{
    ImageInfo *info;

    Check_Type(val, T_STRING);
    GetInfoStruct(mgk, info);
    CloneString(&info->font, RSTRING(val)->ptr);
}


static void
set_fuzz(const char *key, MgkImage *mgk, Image *image, VALUE val)
{
    ImageInfo *info;

    Check_Type(val, T_FIXNUM);
    GetInfoStruct(mgk, info);
    info->fuzz = FIX2INT(val);

    if (!image)
        return;
    for ( ; image; image = image->next)
        image->fuzz = FIX2INT(val);
}

static void
set_geometry(const char *key, MgkImage *mgk, Image *image, VALUE str)
{
    if (!image)
        return;
    Check_Type(str, T_STRING);
    CloneString(&image->geometry, RSTRING(str)->ptr);
}


static void
set_green_primary(const char *key, MgkImage *mgk, Image *image, VALUE val)
{
    if (!image)
        return;
    Check_Type(val, T_STRING);
    sscanf(RSTRING(val)->ptr, "%lf, %lf",
           &image->chromaticity.green_primary.x,
           &image->chromaticity.green_primary.y);
}

static void
set_index(const char *key, MgkImage *mgk, Image *image, VALUE val)
{
    int x, y, index;
    IndexPacket *indexes;
    PixelPacket *pixel;

    if (!image)
        return;

    if (!(TYPE(val) == T_FIXNUM || TYPE(val) == T_FLOAT))
        rb_raise(rb_eTypeError, "Fixnum or Float expected");
    if (image->class != PseudoClass)

    x = 0;
    y = 0;

    sscanf(key, "%*[^[][%d,%d",&x,&y);
    pixel = GetImagePixels(image, x % image->columns, y % image->rows,
                           1, 1);
    if (!pixel)
        return;

    indexes = GetIndexes(image);
    index = NUM2DBL(val);
    if ((index >= 0) && (index < image->colors))
        *indexes = index;
    SyncImagePixels(image);
}

static void
set_interlace(const char *key, MgkImage *mgk, Image *image, VALUE intl_type)
{
    ImageInfo *info;

    GetInfoStruct(mgk, info);
    info->interlace = FIX2UINT(intl_type);
    if (!image)
        return;
    image->interlace = FIX2UINT(intl_type);
}

static void
set_iterations(const char *key, MgkImage *mgk, Image *image, VALUE val)
{
    if (!image)
        return;
    image->iterations = FIX2UINT(val);
}

static void
set_label(const char *key, MgkImage *mgk, Image *image, VALUE val)
{
    Check_Type(val, T_STRING);
    if (image){
        while (SetImageAttribute(image, "Label", (char *)NULL) != 0)
            ;
        SetImageAttribute(image, "Label", RSTRING(val)->ptr);
    }
}

/* alias for set_interations */
static void
set_loop(const char *key, MgkImage *mgk, Image *image, VALUE val)
{
    set_iterations(key, mgk, image, val);
}

static void
set_magick(const char *key, MgkImage *mgk, Image *image, VALUE format)
{
    ImageInfo *info;

#if MagickLibVersion >= 0x0534
    ExceptionInfo e;
    GetExceptionInfo(&e);
#endif /* MagickLibVersion */

    Check_Type(format, T_STRING);
    GetInfoStruct(mgk, info);
    FormatString(info->filename, "%.1024s:", RSTRING(format)->ptr);

#if MagickLibVersion >= 0x0534
    SetImageInfo(info, 1, &e);
#else
    SetImageInfo(info, 1);
#endif /* MagickLibVersion */

    if (*info->magick == '\0') {
        MagickWarning(OptionWarning, "unrecognized image format", info->filename);
    }
    else {
        if (!image)
            return;
        strcpy(image->magick, info->magick);
    }
}

static void
set_matte(const char *key, MgkImage *mgk, Image *image, VALUE bool_val)
{
    if (!image)
        return;
    image->matte = RTEST(bool_val);
}

static void
set_mattecolor(const char *key, MgkImage *mgk, Image *image, VALUE val)
{
    PixelPacket target_color;
    ImageInfo *info;

    Check_Type(val, T_STRING);
    GetInfoStruct(mgk, info);
    QueryColorDatabase(RSTRING(val)->ptr, &target_color);
    info->matte_color = target_color;
    if (!image)
        return;
    image->matte_color = target_color;
}

static void
set_monochrome(const char *key, MgkImage *mgk, Image *image, VALUE bool_val)
{
    ImageInfo *info;

    GetInfoStruct(mgk, info);
    info->monochrome = bool_val;
}

static void
set_page(const char *key, MgkImage *mgk, Image *image, VALUE val)
{
    char *p;
    ImageInfo *info;

    Check_Type(val, T_STRING);
    p = PostscriptGeometry(RSTRING(val)->ptr);
    if (!p)
       return;

    GetInfoStruct(mgk, info);
    CloneString(&info->page, p);
    if (!image)
        return;
    ParseImageGeometry(p,
                       &image->page.x, &image->page.y,
                       &image->page.width, &image->page.height);
#if MagickLibVersion >= 0x0536
    LiberateMemory((void **) &p);
#else
    DestroyPostscriptGeometry(p);
#endif /* MagickLibVersion */
}

static void
set_pen(const char *key, MgkImage *mgk, Image *image, VALUE val)
{
    ImageInfo *info;

    Check_Type(val, T_STRING);
    GetInfoStruct(mgk, info);
    QueryColorDatabase(RSTRING(val)->ptr, &info->pen);
}

static void
set_pixel(const char *key, MgkImage *mgk, Image *image, VALUE val)
{
    int x, y, red, green, blue, opacity;
    PixelPacket *pixel;
    char *color;

    if (!image)
        return;
    Check_Type(val, T_STRING);
    color = RSTRING(val)->ptr;

    x = 0;
    y = 0;
    sscanf(key, "%*[^[][%d,%d",&x,&y);
    pixel = GetImagePixels(image,  x % image->columns, y % image->rows,
                           1, 1);
    if (!pixel)
        return;
    image->storage_class = DirectClass;
    if (strchr(color, ',') == 0)
        QueryColorDatabase(color, pixel);
    else {
        red = pixel->red;
        green = pixel->green;
        blue = pixel->blue;
        opacity = pixel->opacity;

        sscanf(color,"%d,%d,%d,%d", &red, &green, &blue, &opacity);
        pixel->red = ((red < 0) ? 0 : (red > MaxRGB) ? MaxRGB : red);
        pixel->green = ((green < 0) ? 0 : (green > MaxRGB) ? MaxRGB : green);
        pixel->blue = ((blue < 0) ? 0 : (blue > MaxRGB) ? MaxRGB : blue);
        pixel->opacity = ((opacity < 0) ? 0 : (opacity > MaxRGB) ? MaxRGB : opacity);
    }
    SyncImagePixels(image);
}

static void
set_pointsize(const char *key, MgkImage *mgk, Image *image, VALUE val)
{
    ImageInfo *info;

    Check_Type(val, T_FIXNUM);
    GetInfoStruct(mgk, info);
    info->pointsize = NUM2DBL(val);
}

static void
set_preview(const char *key, MgkImage *mgk, Image *image, VALUE preview_type)
{
    ImageInfo *info;
    GetInfoStruct(mgk, info);
    info->preview_type = FIX2UINT(preview_type);
}

static void
set_quality(const char *key, MgkImage *mgk, Image *image, VALUE val)
{
    ImageInfo *info;
    GetInfoStruct(mgk, info);

    Check_Type(val, T_FIXNUM);
    if( (info->quality = FIX2UINT(val)) <= 0 ){
        info->quality = 75;
    }
}

static void
set_red_primary(const char *key, MgkImage *mgk, Image *image, VALUE val)
{
    if (!image)
        return;
    Check_Type(val, T_STRING);
    sscanf(RSTRING(val)->ptr,"%lf,%lf",
           &image->chromaticity.red_primary.x,
           &image->chromaticity.red_primary.y);

}

static void
set_rendering_intent(const char *key, MgkImage *mgk, Image *image,  VALUE rendering_intent)
{
    if (!image)
        return;
    image->rendering_intent = FIX2UINT(rendering_intent);
}

static void
set_scene(const char *key, MgkImage *mgk, Image *image, VALUE val)
{
    if (!image)
        return;
    image->scene = FIX2UINT(val);
}

static void
set_server(const char *key, MgkImage *mgk, Image *image, VALUE val)
{
    if (!image)
        return;
    set_display(key, mgk, image, val);
}

static void
set_size(const char *key, MgkImage *mgk, Image *image, VALUE val)
{
    ImageInfo *info;
    char *geom;

    Check_Type(val, T_STRING);
    GetInfoStruct(mgk, info);
    geom = RSTRING(val)->ptr;
    if (!IsGeometry(geom)) {
        MagickWarning(OptionWarning,"Invalid geometry on size", geom);
        return;
    }
    CloneString(&info->size, geom);

}

static void
set_subimage(const char *key, MgkImage *mgk, Image *image, VALUE val)
{
    ImageInfo *info;

    GetInfoStruct(mgk, info);
    if (info)
        info->subimage = NUM2UINT(val);
}

static void
set_subrange(const char *key, MgkImage *mgk, Image *image, VALUE val)
{
    ImageInfo *info;

    GetInfoStruct(mgk, info);
    info->subrange = NUM2UINT(val);
}

static void
set_texture(const char *key, MgkImage *mgk, Image *image, VALUE val)
{
    ImageInfo *info;

    Check_Type(val, T_STRING);
    GetInfoStruct(mgk, info);
    CloneString(&info->texture, RSTRING(val)->ptr);
}

static void
set_tile(const char *key, MgkImage *mgk, Image *image, VALUE val)
{
    ImageInfo *info;

    Check_Type(val, T_STRING);
    GetInfoStruct(mgk, info);
    CloneString(&info->tile, RSTRING(val)->ptr);
}

static void
set_units(const char *key, MgkImage *mgk, Image *image, VALUE resolution_type)
{
    ImageInfo *info;

    GetInfoStruct(mgk, info);
    info->units = NUM2INT(resolution_type);
    if (!image)
        return;
    image->units = NUM2INT(resolution_type);
}

static void
set_verbose(const char *key, MgkImage *mgk, Image *image, VALUE bool_val)
{
    ImageInfo *info;

    GetInfoStruct(mgk, info);
    info->verbose = RTEST(bool_val);
}

static void
set_view(const char *key, MgkImage *mgk, Image *image, VALUE val)
{
    ImageInfo *info;

    Check_Type(val, T_STRING);
    GetInfoStruct(mgk, info);
    CloneString(&info->view, RSTRING(val)->ptr);
}

static void
set_white_point(const char *key, MgkImage *mgk, Image *image, VALUE val)
{
    if (!image)
        return;
    Check_Type(val, T_STRING);
    sscanf(RSTRING(val)->ptr, "%lf,%lf",
           &image->chromaticity.white_point.x,
           &image->chromaticity.white_point.y);
}

struct _Attr_Set_List
{
    char *name;
    int  pos;
    void (*attr_set_func)(const char*, MgkImage*, Image*, VALUE);
};

typedef struct _Attr_Set_List Attr_Set_List;

static
const Attr_Set_List AttrSetList[] ={
    {"adjoin",             2, set_adjoin},
    {"antialias",          2, set_antialias},
    {"background",         2, set_background},
    {"blue-primary",       2, set_blue_primary},
    {"bordercolor",        2, set_bordercolor},
    {"cache-threshold",    2, set_cache_threshold},
    {"colormap",           6, set_colormap},
    {"colorspace",         6, set_color_space},
    {"compress",           3, set_compress},
    {"delay",              3, set_delay},
    {"density",            3, set_density},
    {"depth",              3, set_depth},
    {"display",            5, set_display},
    {"dispose",            5, set_dispose},
    {"dither",             3, set_dither},
    {"filename",           5, set_filename},
    {"file",               4, set_file},
    {"filter",             4, set_filter},
    {"font",               2, set_font},
    {"fuzz",               2, set_fuzz},
    {"geometry",           2, set_geometry},
    {"green-primary",      2, set_green_primary},
    {"index",              3, set_index},
    {"interlace",          3, set_interlace},
    {"iterations",         2, set_iterations},
    {"label",              2, set_label},
    {"loop",               2, set_loop},
    {"magick",             3, set_magick},
    {"mattecolor",         6, set_mattecolor},
    {"matte",              5, set_matte},
    {"monochrome",         2, set_monochrome},
    {"page",               3, set_page},
    {"pen",                3, set_pen},
    {"pixel",              2, set_pixel},
    {"pointsize",          2, set_pointsize},
    {"preview",            2, set_preview},
    {"quality",            1, set_quality},
    {"red-primary",        3, set_red_primary},
    {"rendering-intent",   3, set_rendering_intent},
    {"scene",              2, set_scene},
    {"server",             2, set_server},
    {"size",               2, set_size},
    {"subimage",           4, set_subimage},
    {"subrange",           4, set_subrange},
    {"texture",            2, set_texture},
    {"tile",               2, set_tile},
    {"units",              1, set_units},
    {"verbose",            2, set_verbose},
    {"view",               2, set_view},
    {"white-point",        2, set_white_point},
    {NULL,                 0, NULL}
};


void
mgk_set_attr(MgkImage *mgk, const VALUE hash)
{
    const Attr_Set_List *attr;

    VALUE keys;
    int i;

    keys = rb_funcall(hash, rb_intern("keys"), 0);
    for (i=0; i<=(RARRAY(keys)->len)-1; i++){
        int found;
        char *key;
        key = STR2CSTR(*(RARRAY(keys)->ptr+i));
        found = 1;
        for (attr = AttrSetList; attr->name != (char *)NULL; attr++){
            VALUE val;
            if ( (found = LocaleNCompare(attr->name, key, attr->pos)) == 0){
                val = rb_funcall(hash, rb_intern("[]"),
                                 1, rb_str_new2(key));
                (attr->attr_set_func)(key, mgk, mgk->ptr, val);
                break;
            }
        }
        if (found != 0)
            MagickWarning(OptionWarning, "no such attribute", key);
    }
}





/******************************************************************************

     G E T   A T T R I B U T E S

*******************************************************************************/

static VALUE
get_adjoin(const char *arg, MgkImage* mgk, Image* image)
{
    ImageInfo *info;

    GetInfoStruct(mgk, info);
    return (info->adjoin != 0) ? Qtrue : Qfalse;
}

static VALUE
get_antialias(const char *arg, MgkImage* mgk, Image* image)
{
    ImageInfo *info;

    GetInfoStruct(mgk, info);
    return (info->antialias != 0) ? Qtrue : Qfalse;
}


static VALUE
get_background(const char *arg, MgkImage* mgk, Image* image)
{
    char color[MaxTextExtent];

    if (!image)
        return Qnil;

    FormatString(color,"%u,%u,%u,%u",image->background_color.red,
                 image->background_color.green,image->background_color.blue,
                 image->background_color.opacity);

    return rb_str_new2(color);
}


static VALUE
get_base_column(const char *arg, MgkImage* mgk, Image* image)
{
    if (!image)
        return Qnil;
    return INT2FIX(image->magick_columns);
}


static VALUE
get_base_filename(const char *arg, MgkImage* mgk, Image* image)
{

    if (!image)
        return Qnil;
    return rb_str_new2(image->magick_filename);
}


static VALUE
get_base_rows(const char *arg, MgkImage* mgk, Image* image)
{
    if (!image)
        return Qnil;
    return INT2FIX(image->magick_rows);
}


static VALUE
get_blue_primary(const char *arg, MgkImage* mgk, Image* image)
{
    char color[MaxTextExtent];

    if (!image)
        return Qnil;
    FormatString(color,"%g, %g",
                 image->chromaticity.blue_primary.x,
                 image->chromaticity.blue_primary.y);
    return rb_str_new2(color);
}


static VALUE
get_bordercolor(const char *arg, MgkImage* mgk, Image* image)
{
    char color[MaxTextExtent];

    if (!image)
        return Qnil;
    FormatString(color,"%u,%u,%u,%u",
                 image->border_color.red,
                 image->border_color.green,
                 image->border_color.blue,
                 image->border_color.opacity);
    return rb_str_new2(color);
}

/* what to return?? */
static VALUE
get_cache_threshold(const char *arg, MgkImage* mgk, Image* image)
{
    ImageInfo *image_info;

    GetInfoStruct(mgk, image_info);
    return Qnil; /*INT2NUM(WriteCacheInfo(image));*/
}


static VALUE
get_class(const char *arg, MgkImage* mgk, Image* image)
{
    char *ClassType[] = {"Undefined","Direct","Pseudo"};
    if (!image)
        return Qnil;
    return rb_str_new2(ClassType[image->storage_class]);
}


static VALUE
get_colormap(const char *arg, MgkImage* mgk, Image* image)
{
    int j;
    char color[MaxTextExtent];

    if (!image || !image->colormap)
        return Qnil;

    sscanf(arg,"%*[^[][%d",&j);
    if (j > image->colors)
        j%=image->colors;
    FormatString(color,"%u,%u,%u,%u",
                 image->colormap[j].red,
                 image->colormap[j].green,
                 image->colormap[j].blue,
                 image->colormap[j].opacity);

    return rb_str_new2(color);
}

static VALUE
get_colors(const char *arg, MgkImage* mgk, Image* image)
{
    if (!image)
        return Qnil;
    return(INT2NUM(GetNumberColors(image, (FILE *)NULL)));
}


static VALUE
get_colorspace(const char *arg, MgkImage* mgk, Image* image)
{
    char *ColorspaceType[] = {"Undefined","RGB","GRAY","Transparent",
                              "OHTA","XYZ","YCbCr","YCC","YIQ",
                              "YPbPr","YUV","CMYK","S_RGB"};
    if (!image)
        return Qnil;
    return rb_str_new2(ColorspaceType[image->colorspace]);
}


static VALUE
get_columns(const char *arg, MgkImage* mgk, Image* image)
{
    if (!image)
        return Qnil;
    return INT2NUM(image->columns);
}


static VALUE
get_comment(const char *arg, MgkImage* mgk, Image* image)
{
    ImageAttribute *attr;

    if (!image)
        return Qnil;
    attr = GetImageAttribute(image, "Comment");
    if (!attr)
        return Qnil;

    return rb_str_new2(attr->value);
}


static VALUE
get_compress(const char *arg, MgkImage* mgk, Image* image)
{
    int j;
    ImageInfo *info;
    char *CompressionType[] = {"Undefined","No","Bzip","Fax",
                               "Group4","JPEG","LZW","RunlengthEncoded", "Zip"};
    if (!image)
        return Qnil;
    GetInfoStruct(mgk, info);
    j = info ? info->compression : image->compression;
    if (info)
        if (info->compression == UndefinedCompression)
            j = image->compression;

    return rb_str_new2(CompressionType[j]);
}


static VALUE
get_delay(const char *arg, MgkImage* mgk, Image* image)
{
    if (!image)
        return Qnil;
    return INT2FIX(image->delay);
}


static VALUE
get_density(const char *arg, MgkImage* mgk, Image* image)
{
    ImageInfo *info;

    GetInfoStruct(mgk, info);
    if (!(info && info->density))
        return Qnil;

    return rb_str_new2(info->density);
}


static VALUE
get_depth(const char *arg, MgkImage* mgk, Image* image)
{
    ImageInfo *info;
    VALUE ret;

    if (!image){
        GetInfoStruct(mgk, info);
        ret = INT2NUM(info->depth);
    }
    else {
        ret = INT2NUM(image->depth);
    }
    return ret;
}


static VALUE
get_directory(const char *arg, MgkImage* mgk, Image* image)
{
    if (!image)
        return Qnil;
    if (image->directory)
        return rb_str_new2(image->directory);

    return Qnil;
}


static VALUE
get_display(const char *arg, MgkImage* mgk, Image* image)
{
    ImageInfo *info;

    GetInfoStruct(mgk, info);
    if (info->server_name)
        return rb_str_new2(info->server_name);

    return Qnil;
}


static VALUE
get_dispose(const char *arg, MgkImage* mgk, Image* image)
{
    if (!image)
        return Qnil;
    return INT2FIX(image->dispose);
}


static VALUE
get_dither(const char *arg, MgkImage* mgk, Image* image)
{
    ImageInfo *info;

    GetInfoStruct(mgk, info);
    return info->dither > 0 ? Qtrue : Qfalse;
}


static VALUE
get_error(const char *arg, MgkImage* mgk, Image* image)
{
    if (!image)
        return Qnil;
    return (rb_float_new(image->mean_error_per_pixel));
}


static VALUE
get_file(const char *arg, MgkImage* mgk, Image* image)
{
    return Qnil; /* not implemented */
}


static VALUE
get_filename(const char *arg, MgkImage* mgk, Image* image)
{
    ImageInfo *info;

    GetInfoStruct(mgk, info);
    if (image)
        return rb_str_new2(image->filename);

    if (info && info->filename
        && *info->filename)
        return rb_str_new2(info->filename);

    return Qnil; /* not reached */
}


static VALUE
get_filesize(const char *arg, MgkImage* mgk, Image* image)
{
    if (!image)
        return Qnil;
    return (INT2NUM(image->filesize));
}

static VALUE
get_filter(const char *arg, MgkImage* mgk, Image* image)
{
    char *FilterType[] = {"Undefined","Point","Triangle","Hermite","Hanning",
                          "Hamming","Blackman","Gaussian","Quadratic","Cubic",
                          "Catrom","Mitchell","Lanczos","Bessel","Sinc"};
    if (!image)
        return Qnil;
    return rb_str_new2(FilterType[image->filter]);
}

static VALUE
get_font(const char *arg, MgkImage* mgk, Image* image)
{
    ImageInfo *info;

    GetInfoStruct(mgk, info);
    if (!(info && info->font))
        return Qnil;

    return rb_str_new2(info->font);
}


static VALUE
get_format(const char *arg, MgkImage* mgk, Image* image)
{
    MagickInfo *magick_info;
    ImageInfo *info;
    char *ret;

    magick_info = (MagickInfo *)NULL;
    ret = (char *)NULL;

    GetInfoStruct(mgk, info);
    if (info && (*info->magick != '\0')){
#if MagickLibVersion >= 0x0534
        magick_info = GetMagickInfo(info->magick, &mgk->ptr->exception);
#else
        magick_info = GetMagickInfo(info->magick);
#endif /* MagickLibVersion */
        ret = magick_info->description;
    }
    else {
        if (image){
#if MagickLibVersion >= 0x0534
            magick_info = (MagickInfo *) GetMagickInfo(image->magick,
                                                       &mgk->ptr->exception);
#else
            magick_info = (MagickInfo *) GetMagickInfo(image->magick);
#endif /* MagickLibVersion */
        }
        if ((magick_info != (MagickInfo *)NULL) &&
            (*magick_info->description = '\0')) {
            ret = magick_info->description;
        }
    }

    return rb_str_new2(ret);
}


static VALUE
get_fuzz(const char *arg, MgkImage* mgk, Image* image)
{
    ImageInfo *info;

    GetInfoStruct(mgk, info);
    if (info){
        return INT2FIX(info->fuzz);
    }
    else {
        return INT2FIX(image->fuzz);
    }

    return Qnil; /* not reached */
}


static VALUE
get_gamma(const char *arg, MgkImage* mgk, Image* image)
{
    if (!image)
        return Qnil;
    return rb_float_new(image->gamma);
}


static VALUE
get_geometry(const char *arg, MgkImage* mgk, Image* image)
{
    if (!image || !image->geometry)
        return Qnil;

    return rb_str_new2(image->geometry);
}


static VALUE
get_green_primary(const char *arg, MgkImage* mgk, Image* image)
{
    char color[MaxTextExtent];

    if (!image)
        return Qnil;
    FormatString(color,"%g,%g",image->chromaticity.green_primary.x,
                 image->chromaticity.green_primary.y);
    return rb_str_new2(color);
}


static VALUE
get_height(const char *arg, MgkImage* mgk, Image* image)
{
    if (!image)
        return Qnil;
    return INT2NUM(image->rows);
}


static VALUE
get_index(const char *arg, MgkImage* mgk, Image* image)
{
    char name[MaxTextExtent];
    int x, y;
    IndexPacket *indexes;
    PixelPacket pixel;

    if (!image)
        return Qnil;
    if (image->class != PseudoClass)
        return Qnil;

    x = 0;
    y = 0;

    sscanf(arg, "%*[^[][%d,%d", &x, &y);
    pixel=GetOnePixel(image,x % image->columns,y % image->rows);
    indexes=GetIndexes(image);
    FormatString(name,"%u",*indexes);

    return rb_str_new2(name);
}


static VALUE
get_interlace(const char *arg, MgkImage* mgk, Image* image)
{
    int j;
    ImageInfo *info;
    char *InterlaceType[] = {"No","Line","Plane","Partition"};

    GetInfoStruct(mgk, info);
    j = info ? info->interlace : image->interlace;
    return rb_str_new2(InterlaceType[j]);
}


static VALUE
get_iterations(const char *arg, MgkImage* mgk, Image* image)
{
    if (!image)
        return Qnil;
    return INT2NUM(image->iterations);
}


static VALUE
get_label(const char *arg, MgkImage* mgk, Image* image)
{
    ImageAttribute *attr;

    if (!image)
        return Qnil;
    attr = GetImageAttribute(image, "Label");
    if (!attr)
        return Qnil;

    return rb_str_new2(attr->value);
}


static VALUE
get_loop(const char *arg, MgkImage* mgk, Image* image)
{
    return INT2NUM(image->iterations);
}


static VALUE
get_magick(const char *arg, MgkImage* mgk, Image* image)
{
    ImageInfo *info;

    GetInfoStruct(mgk, info);

    if (info && *info->magick){
        return rb_str_new2(info->magick);
    }
    else {
        if (!image)
            return Qnil;
        return rb_str_new2(image->magick);
    }

    return Qnil;
}


static VALUE
get_matte(const char *arg, MgkImage* mgk, Image* image)
{
    if (!image)
        return Qnil;
    return INT2FIX(image->matte);
}


static VALUE
get_mattecolor(const char *arg, MgkImage* mgk, Image* image)
{
    char color[MaxTextExtent];

    if (!image)
        return Qnil;
    FormatString(color,"%u,%u,%u,%u",
                 image->matte_color.red,
                 image->matte_color.green,
                 image->matte_color.blue,
                 image->matte_color.opacity);

    return rb_str_new2(color);
}


static VALUE
get_maximum_error(const char *arg, MgkImage* mgk, Image* image)
{
    if (!image)
        return Qnil;
    return rb_float_new(image->normalized_maximum_error);
}


static VALUE
get_mean_error(const char *arg, MgkImage* mgk, Image* image)
{
    if (!image)
        return Qnil;
    return rb_float_new(image->normalized_mean_error);
}


static VALUE
get_montage(const char *arg, MgkImage* mgk, Image* image)
{
    if (!image || !image->montage)
        return Qnil;
    return rb_str_new2(image->montage);
}


static VALUE
get_monochrome(const char *arg, MgkImage* mgk, Image* image)
{
    int j;
    ImageInfo *info;

    GetInfoStruct(mgk, info);
    j = info ? info->monochrome : IsMonochromeImage(image);
    return j > 0 ? Qtrue : Qfalse;
}


static VALUE
get_page(const char *arg, MgkImage* mgk, Image* image)
{
    ImageInfo *info;
    VALUE ret;

    GetInfoStruct(mgk, info);
    if (info && info->page) {
        ret = rb_str_new2(info->page);
    }
    else {
        if (image) {
            char geometry[MaxTextExtent];

            FormatString(geometry,"%ux%u%+d%+d",image->page.width,
                         image->page.height,image->page.x,image->page.y);
            ret = rb_str_new2(geometry);
        }
        else {
            ret = Qnil;
        }
    }

    return ret;
}


static VALUE
get_pen(const char *arg, MgkImage* mgk, Image* image)
{
    return Qnil; /* not implemented */
}


static VALUE
get_pixel(const char *arg, MgkImage* mgk, Image* image)
{
    char name[MaxTextExtent];
    int x, y;
    PixelPacket pixel;

    if (!image)
        return Qnil;
    x = 0;
    y = 0;
    sscanf(arg, "%*[^[][%d,%d",&x, &y);
    pixel=GetOnePixel(image,x % image->columns,y % image->rows);
    FormatString(name,"%u,%u,%u,%u",
                 pixel.red,
                 pixel.green,
                 pixel.blue,
                 pixel.opacity);
    return rb_str_new2(name);
}


static VALUE
get_pointsize(const char *arg, MgkImage* mgk, Image* image)
{
    ImageInfo *info;

    GetInfoStruct(mgk, info);
    return INT2NUM(info->pointsize);
}


static VALUE
get_preview(const char *arg, MgkImage* mgk, Image* image)
{
    ImageInfo *info;
    char *PreviewType[] = {"Undefined","Rotate","Shear","Roll",
                         "Hue","Saturation","Brightness","Gamma",
                         "Spiff","Dull","Grayscale","Quantize",
                         "Despeckle","ReduceNoise","AddNoise",
                         "AddNoise","Sharpen","Blur","Threshold",
                         "EdgeDetect","Spread","Solarize","Shade",
                         "Raise","Segment","Swirl","Implode",
                         "Wave","Oilpaint","CharcoalDrawing","JPEG"};
    GetInfoStruct(mgk, info);
    return rb_str_new2(PreviewType[info->preview_type]);
}


static VALUE
get_quality(const char *arg, MgkImage* mgk, Image* image)
{
    ImageInfo *info;

    GetInfoStruct(mgk, info);
    return INT2FIX(info->quality);
}


static VALUE
get_red_primary(const char *arg, MgkImage* mgk, Image* image)
{
    char color[MaxTextExtent];

    if (!image)
        return Qnil;
    FormatString(color,"%g,%g",
                 image->chromaticity.red_primary.x,
                 image->chromaticity.red_primary.y);

    return rb_str_new2(color);
}


static VALUE
get_rendering_intent(const char *arg, MgkImage* mgk, Image* image)
{
    int j;
    char *RenderingIntent[] = {"Undefined","Satulation","Perceptual",
                               "Absolute","Relative"};
    if (!image)
        return Qnil;
    j = image->rendering_intent;
    return rb_str_new2(RenderingIntent[j]);
}


static VALUE
get_rows(const char *arg, MgkImage* mgk, Image* image)
{
    if (!image)
        return Qnil;
    return INT2NUM(image->rows);
}


static VALUE
get_scene(const char *arg, MgkImage* mgk, Image* image)
{
    if (!image)
        return Qnil;
    return INT2NUM(image->scene);
}


static VALUE
get_server(const char *arg, MgkImage* mgk, Image* image)
{
    ImageInfo *info;

    GetInfoStruct(mgk, info);
    if (!(info && info->server_name) )
        return Qnil;

    return rb_str_new2(info->server_name);
}


static VALUE
get_signature(const char *arg, MgkImage* mgk, Image* image)
{
    ImageAttribute *attr;

    if (!image)
        return Qnil;
    SignatureImage(image);
    attr = GetImageAttribute(image, "Signature");
    if (!attr)
        return Qnil;

    return rb_str_new2(attr->value);
}


static VALUE
get_size(const char *arg, MgkImage* mgk, Image* image)
{
    ImageInfo *info;

    GetInfoStruct(mgk, info);
    if (!(info && info->size) )
        return Qnil;

    return rb_str_new2(info->size);
}


static VALUE
get_subimage(const char *arg, MgkImage* mgk, Image* image)
{
    ImageInfo *info;

    GetInfoStruct(mgk, info);
    return(INT2FIX(info->subimage));
}


static VALUE
get_subrange(const char *arg, MgkImage* mgk, Image* image)
{
    ImageInfo *info;

    GetInfoStruct(mgk, info);
    return(INT2FIX(info->subrange));
}


static VALUE
get_taint(const char *arg, MgkImage* mgk, Image* image)
{
    if (!image)
        return Qnil;

    return IsImageTainted(image) ? Qtrue : Qfalse;
}


static VALUE
get_texture(const char *arg, MgkImage* mgk, Image* image)
{
    ImageInfo *info;

    GetInfoStruct(mgk, info);
    if (!(info && info->texture))
        return Qnil;

    return rb_str_new2(info->texture);
}


static VALUE
get_tile(const char *arg, MgkImage* mgk, Image* image)
{
    ImageInfo *info;

    GetInfoStruct(mgk, info);
    if (!(info && info->tile))
        return Qnil;

    return rb_str_new2(info->tile);
}


static VALUE
get_type(const char *arg, MgkImage* mgk, Image* image)
{
    int j;
    char *ImageType[] = {"Undefined","Bilevel","Grayscale","Palette",
                         "PaletteMatte","TrueColor","TrueColorMatte", "CMYK"};
    if (!image)
        return Qnil;

    j = GetImageType(image);
    return rb_str_new2(ImageType[j]);
}


static VALUE
get_units(const char *arg, MgkImage* mgk, Image* image)
{
    int j;
    ImageInfo *info;
    char *ResolutionType[] = {"Undefined","PixelsPerInch","PixelsPerCentimeter"};
    GetInfoStruct(mgk, info);
    j = info ? info->units : image->units;

    if (info){
        if (info->units == UndefinedResolution){
            if (image)
                j = image->units;
        }
    }
    return rb_str_new2(ResolutionType[j]);
}


static VALUE
get_verbose(const char *arg, MgkImage* mgk, Image* image)
{
    ImageInfo *info;

    GetInfoStruct(mgk, info);
    if (!info)
        return Qnil;

    return info->verbose > 0 ? Qtrue : Qfalse;
}


static VALUE
get_view(const char *arg, MgkImage* mgk, Image* image)
{
    ImageInfo *info;

    GetInfoStruct(mgk, info);
    if (! (info && info->view) )
        return Qnil;

    return rb_str_new2(info->view);
}


static VALUE
get_white_point(const char *arg, MgkImage* mgk, Image* image)
{
    char color[MaxTextExtent];

    if (!image)
        return Qnil;
    FormatString(color,"%g,%g",image->chromaticity.white_point.x,
                 image->chromaticity.white_point.y);

    return rb_str_new2(color);
}


static VALUE
get_width(const char *arg, MgkImage* mgk, Image* image)
{
    if (!image)
        return Qnil;
    return INT2NUM(image->columns);
}


static VALUE
get_x_resolution(const char *arg, MgkImage* mgk, Image* image)
{
    if (!image)
        return Qnil;
    return INT2NUM(image->x_resolution);
}


static VALUE
get_y_resolution(const char *arg, MgkImage* mgk, Image* image)
{
    if (!image)
        return  Qnil;
    return INT2NUM(image->y_resolution);
}





struct Attr_Get_List
{
    char *name;
    int   pos;
    VALUE (*attr_get_func)(const char *arg, MgkImage*, Image*);
};

typedef struct Attr_Get_List Attr_Get_List;

static
const Attr_Get_List AttrGetList[] ={
    {"adjoin",            2, get_adjoin},
    {"antialias",         2, get_antialias},
    {"background",        3, get_background},
    {"base-column",       6, get_base_column},
    {"base-filename",     6, get_base_filename},
    {"base-rows",         6, get_base_rows},
    {"blue-primary",      2, get_blue_primary},
    {"bordercolor",       2, get_bordercolor},
    {"cache-threshold",   2, get_cache_threshold},
    {"class",             2, get_class},
    {"colormap",          6, get_colormap},
    {"colorspace",        7, get_colorspace},
    {"colors",            6, get_colors},
    {"columns",           4, get_columns},
    {"comment",           4, get_comment},
    {"compress",          4, get_compress},
    {"delay",             3, get_delay},
    {"density",           3, get_density},
    {"depth",             3, get_depth},
    {"directory",         3, get_directory},
    {"display",           5, get_display},
    {"dispose",           5, get_dispose},
    {"dither",            3, get_dither},
    {"error",             1, get_error},
    {"filename",          5, get_filename},
    {"file",              4, get_file},
    {"filesize",          5, get_filesize},
    {"filter",            4, get_filter},
    {"font",              3, get_font},
    {"format",            3, get_format},
    {"fuzz",              2, get_fuzz},
    {"gamma",             2, get_gamma},
    {"geometry",          2, get_geometry},
    {"green-primary",     2, get_green_primary},
    {"height",            1, get_height},
    {"index",             3, get_index},
    {"interlace",         3, get_interlace},
    {"iterations",        2, get_iterations},
    {"label",             2, get_label},
    {"loop",              2, get_loop},
    {"magick",            3, get_magick},
    {"mattecolor",        6, get_mattecolor},
    {"matte",             5, get_matte},
    {"maximun-error",     3, get_maximum_error},
    {"mean-error",        2, get_mean_error},
    {"montage",           4, get_montage},
    {"monochrome",        4, get_monochrome},
    {"page",              2, get_page},
    {"pen",               2, get_pen},
    {"pixel",             2, get_pixel},
    {"pointsize",         2, get_pointsize},
    {"preview",           2, get_preview},
    {"quality",           1, get_quality},
    {"red-primary",       3, get_red_primary},
    {"rendering-intent",  4, get_rendering_intent},
    {"rows",              2, get_rows},
    {"scene",             2, get_scene},
    {"server",            2, get_server},
    {"signature",         3, get_signature},
    {"size",              3, get_size},
    {"subimage",          4, get_subimage},
    {"subrange",          4, get_subrange},
    {"taint",             2, get_taint},
    {"texture",           2, get_texture},
    {"tile",              2, get_tile},
    {"type",              3, get_type},
    {"units",             1, get_units},
    {"verbose",           2, get_verbose},
    {"view",              2, get_view},
    {"x-resolution",      1, get_x_resolution},
    {"y-resolution",      1, get_y_resolution},
    {"white-point",       2, get_white_point},
    {"width",             2, get_width},
    {NULL,                0, NULL}
};

VALUE
mgk_get_attr(MgkImage *mgk, const VALUE arg)
{
    const Attr_Get_List *attr;
    VALUE retval = Qnil;

    char *str = RSTRING(arg)->ptr;
    int found = 1;

    for (attr = AttrGetList; attr->name != '\0'; attr++){
        if ( (found = LocaleNCompare(attr->name, str, attr->pos)) == 0 ){
            retval = (attr->attr_get_func)(str, mgk, mgk->ptr);
            break;
        }
    }
    if (found != 0) {
        MagickWarning(OptionWarning, "no such attribute", str);
        retval = Qnil;
    }
    return retval;
}
