/*
*  Rubrica
*  file: utils.c
*
*  Copyright (C) 2000-2003 Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <config.h>
#include <gnome.h>
#include <libxml/tree.h>
#include <libxml/parser.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <time.h>
#include <errno.h>
#include <ctype.h>
#include <math.h>

#include "utils.h"
#include "file.h"
#include "prefer.h"
#include "log.h"
#include "rubrica_type.h"


gboolean start = FALSE;

static GList *pixmaps_directories = NULL;
static GList *tooltip_list = NULL;
GList *netlist = NULL, *tellist = NULL;


gboolean rubrica_utils_is_rubrica_doc(gchar *filename)
{
  xmlDocPtr doc;

  doc = rubrica_file_open_doc(filename);  

  return rubrica_file_is_rubrica_doc(doc, filename);
}

gboolean rubrica_utils_is_start(void)
{
  return start;
}

void rubrica_utils_set_start(gboolean state)
{
  start = state;
}


void rubrica_utils_file_split(gchar *file, gchar **f, gchar **ext, gint *n)
{
  gchar *buf, *tmp, **split;
  gint i = 0, num = 0;

  buf = g_strdup(g_basename(file));
  split = g_strsplit(buf, ".", 1);

  *ext = g_strdup(split[1]);

  buf = split[0];
  g_strreverse(buf);

  while (isdigit(buf[i]))
    {
      num = num + (buf[i] - 48) * pow(10,i);
      i++;
    }
  num+=1;
  g_strreverse(buf);
  tmp = g_strndup(split[0], strlen(split[0]) - i);
  
  *f = tmp;
  *n = num;
}


int rubrica_utils_str2int(gchar *str)
{
  int len;
  int ret = 0;
  int i = 0;
  
  if (str)
    {
      len = strlen(str)-1;
      
      while (str[len])
	{
	  if (isdigit(str[len]))
	    {
	      ret = ret + (str[len] - '0') * pow(10, i);
	      i++;
	    }
	  len--;
	}
    }
  
  return ret;
}

void rubrica_utils_check_home(gchar *home)
{
  gchar *folder;
  gchar *message = NULL;
  gint err;
  
  folder = g_strdup_printf("%s/.rubrica", home);
  if (!g_file_exists(folder))
    {
      err = rubrica_utils_make_rubrica_home(home);  
      
      if (!err)
	message = g_strdup_printf(_("\nRubrica warning *** Can't find folder: "
				    "%s (may be it was deleted). I have "
				    "recreated it, but other data are lost. "
				    "Rubrica will start with an empty "
				    "addressbook."), folder);
      else
	message = g_strdup_printf(_("\nRubrica error *** Can't find folder: "
				    "%s (may be it was deleted). I "
				    "unsuccessfully tried to recreate, "
				    "I got this error: %s"), 
				  folder, strerror(errno));
      
      rubrica_preferences_set_have_default_book(FALSE);
      rubrica_preferences_set_book("");
    }

  if (message)
    {
      g_print(message);
      rubrica_log_write_log(message);  
      g_free(message);
    }  

  g_free(folder);
}


void rubrica_utils_check_dictionary_home(gchar *home)
{
  gchar *message = NULL;
  gchar *folder;
  gint err;

  folder = g_strdup_printf("%s/.rubrica/dictionary", home);
  if (!g_file_exists(folder))
    {
      err = rubrica_utils_make_dictionary_home(home);

      if (!err)
	message = g_strdup_printf(_("\nRubrica warning *** Can't find folder: "
				    "%s (may be it was deleted)."
				    "I have recreated it."), folder);  
      else
	message = g_strdup_printf(_("\nRubrica error *** Can't find folder: "
				    "%s (may be it was deleted)."
				    "I unsuccessfully tried to recreate, "
				    "I got this error: %s"),
				  folder, strerror(errno));

      rubrica_log_write_log(message);  
      g_print(message);
      g_free(message);
    }

  g_free(folder);
}


void rubrica_utils_check_folders(gchar *home)
{
  gchar *message = NULL;
  gchar *folder;
  gint err;

  g_print("\nRubrica: checking rubrica's folders");

  rubrica_utils_check_home(home);
  rubrica_log_init_log_file();       

  rubrica_utils_check_dictionary_home(home);

  if (rubrica_preferences_get_have_default_folder())
    {
      folder = rubrica_preferences_get_default_folder();
      
      if (!g_file_exists(folder))
	{
	  err = rubrica_utils_make_user_rubrica_home(folder);

	  if (!err)
	    message = g_strdup_printf(_("\nRubrica warning *** Can't find "
					"folder: %s (may be it was deleted)."
					"I have recreated it."), folder);
	  else
	    message = g_strdup_printf(_("\nRubrica error *** Can't find "
					"folder: %s (may be it was deleted)."
					"I unsuccessfully tried to recreate, "
					"I got this error: %s"),
				      folder, strerror(errno));

	  rubrica_log_write_log(message);  
	  g_print(message);
	  g_free(message);      
	}
    }

  g_print("\nRubrica: rubrica's folders checked");
}


gint rubrica_utils_make_user_rubrica_home(gchar *folder)
{
  return rubrica_utils_make_home(folder);  
}


gint rubrica_utils_make_rubrica_home(gchar *home)
{
  gint err;
  gchar *folder;

  folder = g_strdup_printf("%s/.rubrica", home);
  err = rubrica_utils_make_home(folder);

  rubrica_preferences_set_rubrica_home(folder);  

  g_free(folder);
  
  return err;
}


gint rubrica_utils_make_dictionary_home(gchar *home)
{
  gint err;
  gchar *dicthome; 
      
  dicthome = g_strdup_printf("%s/.rubrica/dictionary", home); 
  err = rubrica_utils_make_home(dicthome);

  rubrica_preferences_set_dictionary_home(dicthome); 
  
  g_free(dicthome);
  
  return err;
}


gint rubrica_utils_make_home(gchar *folder)
{
  gchar *message; 
  gint err;

  err = mkdir(folder, S_IRUSR | S_IWUSR | S_IXUSR);

  if (!err)
    {
      if (rubrica_run_first_time())    
	{
	  message = g_strdup_printf(_("\nfolder: %s doesn't exist.Created\n"), 
				    folder);
	  
	  g_print(message);       
	  g_free(message);      
	}
    }

  return err;
}


void rubrica_utils_copy_script(void)
{
  gchar *message; 
  gchar *source_script;
  gchar *dest_script;
  gchar *gnome_home;
  gchar *nautilus_scripts_home;
  	
  gnome_home = g_strdup(gnome_user_dir_get());	
  if (!g_file_exists(gnome_home))
    {
      message = g_strdup_printf("\n%s doesn't exist. May be you never run "
				"Gnome. I can't continue", 
				gnome_home);
      g_print(message);
      rubrica_log_write_log(message);
      
      g_free(message);
      g_free(gnome_home);
      return;
    }
  
  nautilus_scripts_home = g_strdup_printf("%s%s%s", gnome_home, 
					  G_DIR_SEPARATOR_S, 
					  "nautilus-scripts");
  if (!g_file_exists(nautilus_scripts_home))
    {
      message = g_strdup_printf("\n%s doesn't exist. May be you never "
				"run Nautilus. I can't continue", 
				nautilus_scripts_home);

      g_print(message);
      rubrica_log_write_log(message);
      
      g_free(message);      
      g_free(gnome_home);
      g_free(nautilus_scripts_home);
      return;
    }

  

  source_script = g_strdup_printf("%s%s%s", RUBRICA_SCRIPT_DIR, 
				  G_DIR_SEPARATOR_S, 
				  "nautilus-script/rubrica2html.py");

  dest_script = g_strdup_printf("%s/%s", gnome_home, 
				"nautilus-scripts/rubrica2html.py");
  
  g_free(gnome_home);
  g_free(nautilus_scripts_home);

  if(g_file_exists(source_script))
    {
      FILE *in, *out;
      
      if((in = fopen(source_script, "r")) == NULL)
	{
	  message = g_strdup("\ncan't read script file");
	  g_print(message);
	  rubrica_log_write_log(message); 
	  
	  g_free(message);
	  g_free(source_script);
	  g_free(dest_script);

	  return;
	}
      
      if((out = fopen(dest_script, "w")) == NULL)
	{
	  message = g_strdup("\ncan't write script file");
	  g_print(message);
	  rubrica_log_write_log(message); 
	  
	  fclose(in);
	  g_free(message);	  
	  g_free(source_script);
	  g_free(dest_script);

	  return;
	}	  
      
      while (!feof(in))
	fputc(fgetc(in), out);

      chmod(dest_script, S_IRWXU);
    }
  else
    {
      message = g_strdup_printf("\nI can't find nautilus script in %s."
				"\nScript not installed", source_script);
      
      g_print(message);
      rubrica_log_write_log(message); 
      
      g_free(message);	  
      g_free(source_script);
      g_free(dest_script);
      
      return;      
    }
  
  message = g_strdup_printf(_("\nNautilus script %s installed in %s"),  
			    source_script, dest_script);
  g_print(message);
  rubrica_log_write_log(message); 
  
  g_free(message); 
  g_free(source_script);
  g_free(dest_script);
}


void 
rubrica_utils_add_tooltip(GtkTooltips *tooltip)
{
  tooltip_list = g_list_prepend(tooltip_list, tooltip);
}

void 
rubrica_utils_enable_tooltips(void)
{
  GList *alias = tooltip_list;

  while (alias)
    {
      gtk_tooltips_enable(GTK_TOOLTIPS(alias->data));

      alias = g_list_next(alias);
    }
}


void 
rubrica_utils_disable_tooltips(void)
{
  GList *alias = tooltip_list;

  while (alias)
    {
      gtk_tooltips_disable(GTK_TOOLTIPS(alias->data));

      alias = g_list_next(alias);
    }  
}


/* derived from code build by glade 
*/
void 
rubrica_utils_add_pixmap_directory(const gchar *directory)
{
  pixmaps_directories = g_list_prepend (pixmaps_directories, 
					g_strdup (directory));
}


/* derived from code build by glade 
*/
gchar*
rubrica_utils_find_pixmap_file (const gchar *filename)
{
  GList *elem;

  if (!filename)
    return NULL;

  /* We step through each of the pixmaps directory to find it. */
  elem = pixmaps_directories;
  while (elem)
    {
      gchar *pathname = g_strdup_printf ("%s%s%s", (gchar*)elem->data, 
					 G_DIR_SEPARATOR_S, filename);

      if (g_file_test (pathname, G_FILE_TEST_EXISTS))
        return pathname;
      g_free (pathname);
      elem = elem->next;
    }
  return NULL;
}


/* This is used to set ATK action descriptions. 
 */
void
rubrica_utils_set_atk_action_description (AtkAction *action, 
					  const gchar *action_name,
					  const gchar *description)
{
  gint n_actions, i;

  n_actions = atk_action_get_n_actions (action);
  for (i = 0; i < n_actions; i++)
    {
      if (!strcmp (atk_action_get_name (action, i), action_name))
        atk_action_set_description (action, i, description);
    }
}


/*
 */
GList*
rubrica_utils_get_all_data(GtkWidget *tree, RubricaTreeType type)
{
  GtkTreeModel *model;

  model = gtk_tree_view_get_model(GTK_TREE_VIEW(tree));
  netlist = tellist = NULL;  /* global variables */

  switch (type)
    {
    case CARD_TREE:
      break;

    case ACCOUNT_TREE:
      break;

    case NET_TREE:
      gtk_tree_model_foreach(GTK_TREE_MODEL(model), 
			     rubrica_utils_extract_net_data, NULL);  

      return netlist;
      break;

    case PHONE_TREE:
      gtk_tree_model_foreach(GTK_TREE_MODEL(model), 
			     rubrica_utils_extract_phone_data, NULL);

      return tellist;
      break;

    default:
      break;
    }
  
  return NULL;
}


gboolean 
rubrica_utils_extract_net_data(GtkTreeModel *model, GtkTreePath *path,
			       GtkTreeIter *iter, gpointer data)
{
  gchar *str;

  gtk_tree_model_get(model, iter, URI_COLUMN, &str, -1);
  netlist = g_list_append(netlist, str);
  
  return FALSE;
}


gboolean 
rubrica_utils_extract_phone_data(GtkTreeModel *model, GtkTreePath *path,
				 GtkTreeIter *iter, gpointer data)
{
  TelNum *tt;
  gchar *number;
  gchar *type;

  tt = (TelNum *) g_malloc(sizeof(TelNum));

  gtk_tree_model_get(model, iter, NUMBER_COLUMN, &number, 
		     TYPE_COLUMN, &type, 
		     -1);
  
  tt->number = g_strdup(number);
  tt->type = g_strdup(type);
  tellist = g_list_append(tellist, tt);

  return FALSE;
}


/* 
*/
RubricaTelephoneType rubrica_utils_telephone2int(gchar *tipo)
{
  g_return_val_if_fail(tipo != NULL, TELEPHONE_UNKNOWN);

  if ((strcasecmp(tipo, "home") == 0) ||      
      (strcasecmp(tipo, "Casa") == 0))
    return TELEPHONE_HOME;

  if ((strcasecmp(tipo, "work") == 0) ||
      (strcasecmp(tipo, "Ufficio") == 0))
    return TELEPHONE_WORK;

  if ((strcasecmp(tipo, "fax") == 0) ||
      (strcasecmp(tipo, "Fax") == 0))
    return TELEPHONE_FAX;
  
  if ((strcasecmp(tipo, "cellphone") == 0) ||
      (strcasecmp(tipo, "Cellulare") == 0))
    return TELEPHONE_CELLPHONE;

  if ((strcasecmp(tipo, "other") == 0) ||
      (strcasecmp(tipo, "Altro") == 0))
    return TELEPHONE_OTHER;

  return TELEPHONE_UNKNOWN;
}

gchar *rubrica_utils_bool2str(gchar *str)
{
  if(!str)
    return "???";
  
  if (strcasecmp(str, "true") == 0)
    return _("yes");

  return _("no");
}

gboolean rubrica_utils_str2bool(gchar *str)
{
  if (!str)
    return FALSE;

  if (strcasecmp(str, "true") == 0)
    return TRUE;

  return FALSE; 
}


gchar* rubrica_utils_decode_date(gchar *str)
{
  return rubrica_utils_convert_time(atoi(str), FALSE);
}

gchar*
rubrica_utils_convert_time(time_t t,/*  RubricaTimeFormatType type,*/ 
			   gboolean has_hour)
{
  struct tm *loctime;
  gchar *buffer;
  
  buffer = g_malloc(sizeof(gchar)*256);
  
  if (t == 0)
    {
      return "";
    }

  loctime = localtime(&t);
  if (has_hour)
    strftime(buffer, 256, "%m/%d/%Y %H:%M:%S %Z", loctime);
  else
    strftime(buffer, 256, "%m/%d/%Y", loctime);

/*
  switch (type)
    {
    case RT_DMY:
      if (has_hour)
	strftime(buffer, 256, "%d/%m/%Y %H:%M:%S %Z", loctime);
      else
	strftime(buffer, 256, "%d/%m/%Y", loctime);
      break;

    case RT_MDY:
      if (has_hour)
	strftime(buffer, 256, "%m/%d/%Y %H:%M:%S %Z", loctime);
      else
	strftime(buffer, 256, "%m/%d/%Y", loctime);
      break;

    default:
      break;
    }
*/
  return buffer;     
}



gint rubrica_utils_option_len(RubricaOptionType *option)
{
  RubricaOptionType *opt = option;
  gint i = 0;

  while(opt->pixmap)
    {
      i++;
      opt++;
    }

  return i;
}



/*  Debug functions, ignore :)
*/
void 
rubrica_utils_debug_net_list(GList *lst)
{
  while (lst)
    {
      if (lst && lst->data)
	g_print("\n*** DEBUG NET LIST -- elemento: %s --  ***", 
		(gchar *)lst->data);

      lst = lst->next;
    }
}


void 
rubrica_utils_debug_tel_list(GList *lst)
{
  while (lst)
    {
      TelNum *tn;

      if (lst && lst->data)
	{
	  tn = (TelNum *) lst->data;

	  g_print("\n*** DEBUG TEL LIST -- elemento: %s %s --  ***", 
		  (gchar *)tn->number, 
		  (gchar *)tn->type);
	}

      lst = lst->next;
    }
}
