/*				       	-*- c-file-style: "bsd" -*-
 *
 * $Id: http.c,v 1.13 2000/08/13 10:40:32 mbp Exp $
 * 
 * Copyright (C) 1999, 2000 by Martin Pool
 * Copyright (C) 1999 by Andrew Tridgell
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"
#include "sysheaders.h"
#include <stdio.h>
#include <ctype.h>

#include "rproxy.h"

#include "util.h"
#include "urllib.h"
#include "header.h"
#include "msgpage.h"
#include "trace.h"
#include "error.h"
#include "statusline.h"

static void
request_lookup_method(request_t *req)
{
    char const *n = req->method_name;
    int id;
    
    /* RFC2616: s5.1.1: The method is case-sensitive.  */

    if (!strcmp(n, "GET"))
        id = METHOD_GET;
    else if (!strcmp(n, "POST"))
        id = METHOD_POST;
    else if (!strcmp(n, "PUT"))
        id = METHOD_PUT;
    else if (!strcmp(n, "CONNECT"))
        id = METHOD_CONNECT;
    else
        id = METHOD_OTHER;

    req->method_id = id;
}

/*
 * Split up an HTTP Request-Line, filling in the METHOD, URL, and
 * HTTP_VERSION fields of the REQ structure.  According to RFC2068,
 * the format is:
 *
 *   Request-Line   = Method SP Request-URI SP HTTP-Version CRLF
 *
 * This means that there must be single spaces and no garbage.
 * Admirably simple!
 *
 * The returned copies are alloc'd and should be freed by the caller.
 */
int
request_parse_line(request_t * req)
{
    char const     *start, *end;
    char const     *const line = req->req_line;

    assert(line);

    end = strchr(line, ' ');
    if (!end || end == line)
	goto fail;
    req->method_name = xstrndup(line, end - line);
    request_lookup_method(req);

    start = end + 1;

    end = strchr(start, ' ');
    if (!end) {
	/* this might look like "GET /", which is OK. */
	end = strchr(start, '\0');
    }
    req->url = xstrndup(start, end - start);

    if (!*end) {
	/* nothing after path. */
	/* XXX: OK to assume this? */
	req->http_version = "HTTP/0.9";
    } else {
	start = end + 1;
	end = strchr(start, 0);
	req->http_version = xstrndup(start, end - start);
    }

    trace(LOGAREA_PARSE,
	  __FUNCTION__ ": split request into method=%s method_id=%d "
          "url=%s ver=%s",
	  req->method_name, req->method_id, req->url, req->http_version);

    return DONE;

  fail:
    rp_request_failed(req,
		   HTTP_BAD_REQUEST,
		   __FUNCTION__
		   ": can't extract URL from request " "\"%s\"", line);
}



/*
 * Read the request line from the client.  This will look something
 * like this:
 * 
 * "GET URL HTTP/1.0"
 *
 * The request line is stored in newly allocated memory in
 * req->req_line, and is not parsed yet.
 */
int
read_request_line(request_t * req)
{
    size_t          size = 0;

    req->req_line = NULL;

    if (getline(&req->req_line, &size, req->f_from_client) == -1) {
	rp_request_failed(req, HTTP_BAD_REQUEST, "failed to read request\n");
    }
    trim_crlf(req->req_line);

    trace(LOGAREA_HTTP, "req< %s", req->req_line);
    return 0;
}


/*
 * Work out whether the connection to the client is in such a state that we
 * can send them an error page.  
 *
 * This is a problem with HTTP/1.0: if we've already started sending a
 * request back to the client, then there's no way to indicate that
 * we've encountered a problem and can't continue.  So, the best we
 * can do is abort with no error and leave them with a truncated
 * instance.  Fortunately, this doesn't really happen all that often
 * compared to the background noise of dropped network connections.
 */
bool_t req_can_send_client_error_p(request_t * req)
{
    /* TODO: Hmm, check the RFC: are we allowed to send an error page
     * in response to a HEAD request?  */
    return (req->state < sending_response_headers) && req->f_to_client;
}


void
set_request_status(request_t * req, int code, char const *str)
{
    req->status = code;
    if (req->status_msg)
	free(req->status_msg);
    req->status_msg = xstrdup(str);
}


/* Read and digest the response from the upstream server. */
int
read_response_status_line(request_t * req)
{
    size_t          size;
    char           *p;
    char           *line;

    size = 0;
    if (getline(&line, &size, req->f_upstream) == -1) {
	/* this gives 'illegal seek' if the upstream disconnects. hmm. */
	rp_request_failed(req,
		       HTTP_BAD_GATEWAY,
		       PROGRAM
		       ": failed to read response from upstream: %s",
		       strerror(errno));
    }
    trim_crlf(line);

    trace(LOGAREA_HTTP, "response: %s", line);
    /* FIXME!  Don't just break on space, and perhaps ignore the extra
       parameters e.g.  "text/html; charset=iso-8859-1" */
    p = strchr(line, ' ');
    req->status = strtol(p, &p, 10);
    assert(reasonable_status(req->status));

    while (isspace(*p))
	p++;

    req->status_msg = xstrdup(p);
    free(line);

    return 0;
}





/*
 * Send the response line back to the client, e.g.
 *
 * "HTTP/1.0 200 some really good reason"
 */
void
rp_send_response_line(request_t * req)
{
    char            buf[256];
    char const *msg;

    /* If the upstream server supplied a status message, then pass it
     * down.  Otherwise use a standard message if possible. */
    
    msg = req->status_msg ? req->status_msg :
        http_status_string(req->status);
    snprintf(buf, (sizeof buf) - 1, "%s %d %s",
	     req->http_version, req->status, msg);
    trace(LOGAREA_HTTP, "resp> %s", buf);
    fprintf(req->f_to_client, "%s\r\n", buf);
}




int
will_do_version(request_t const *req)
{
    char const     *ver = req->http_version;

    return !strcasecmp(ver, "HTTP/1.0")
        || !strcasecmp(ver, "HTTP/1.1");
}
