/* 
 *  rocks/signal.c
 *
 *  Signal setup and interposition.
 *
 *  Copyright (C) 2001 Victor Zandy
 *  See COPYING for distribution terms.
 */

/* 
 When the application tries to set a signal handler:
   - If we use the signal, lie to the application - tell it the previous
     action was the default, and return as though the handler has been
     installed
   - Otherwise, call sigaction (we should be in NATIVE mode) as normal. */

#include <signal.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include "rs.h"

static sigset_t reserved;  /* signals used by reliable sockets */

void handle_sigpipe()
{
	fprintf(stderr, "YO! Got SIGPIPE\n");
	kill(getpid(), SIGSEGV);
}

int
rs_init_signal()
{
	struct sigaction sa;

	sigemptyset(&reserved);
	sigaddset(&reserved, SIGURG);
	sigaddset(&reserved, SIGALRM);
	sigaddset(&reserved, SIGPIPE);
	sigaddset(&reserved, SIGABRT);
	sigaddset(&reserved, SIGSEGV);

	sigfillset(&sa.sa_mask);
	sigdelset(&sa.sa_mask, SIGTERM);
	sigdelset(&sa.sa_mask, SIGINT);
	sa.sa_flags = SA_RESTART;
	sa.sa_restorer = 0;
	sa.sa_handler = handle_sigpipe;
	if (0 > sigaction(SIGPIPE, &sa, NULL))
		return -1;
	return 0;
}

int
rs_sigaction(int sig, const struct sigaction *act, struct sigaction *oldact)
{
	if (!sigismember(&reserved, sig))
		return sigaction(sig, act, oldact);

	/* FIXME: We should try to serve these signals to the
	   application */
	if (oldact) {
		bzero(oldact, sizeof(struct sigaction));
		oldact->sa_handler = SIG_DFL;
	}
	return 0;
}

int
rs___libc_sigaction(int sig, const struct sigaction *act, struct sigaction *oldact)
{
	return rs_sigaction(sig, act, oldact);
}

sighandler_t
rs_signal(int sig, sighandler_t handler)
{
	if (!sigismember(&reserved, sig))
		return signal(sig, handler);
	return SIG_DFL;
}

int
rs_sigprocmask(int how, const sigset_t *iset, sigset_t *oldset)
{
	sigset_t set;
	memcpy(&set, iset, sizeof(set));
	if (how == SIG_BLOCK || how == SIG_SETMASK) {
		sigdelset(&set, SIGURG);
		sigdelset(&set, SIGALRM);
	}
	return sigprocmask(how, &set, oldset);
}

int
rs_sigsuspend(const sigset_t *mask)
{
	sigset_t set;
	memcpy(&set, mask, sizeof(set));
	sigdelset(&set, SIGURG);
	sigdelset(&set, SIGALRM);
	return sigsuspend(&set);
}

unsigned int
rs_alarm(unsigned int t)
{
	/* We ignore application alarms;
	   FIXME handle application alarms */
	return 0;
}
