############################################################
#
# Module: wadg::rm::Settings
#
# Created: 20.March.2000 by Jeremy Wadsack for Wadsack-Allen Digital Group
# Copyright (C) 1999,2002 Wadsack-Allen. All rights reserved.
# Based on subs from Report Magic (19.Feb.1999-20.Mar.2000)
#
# This package is free software; you can redistribute it and/or
# modify it under the same terms as Perl itself.
############################################################
# Date        Modification                            Author
# ----------------------------------------------------------
# 16Feb1999 Created by Corey Kaye / DNS                   CK
# 26May2000 Converted to a Config::IniFiles subclass      JW
# 26May2000 Removed all file processing, now only allows
#           ini style configuration files.                JW
# 14Jun2000 Converted data structure mods to method calls JW
############################################################
package wadg::rm::Settings;
use strict;

BEGIN {
	use vars       qw($VERSION @ISA);

	$VERSION = do { my @r = (q$Revision: 2.0 $ =~ /\d+/g); sprintf "%d."."%02d" x $#r, @r }; # must be all one line, for MakeMaker

	@ISA         = qw( Config::IniFiles );
} # end BEGIN
# non-exported package globals go here
use vars      qw();
use Config::IniFiles;
use File::Basename;
use wadg::Errors;

##########################
##                      ##
##   TIEHASH Methods    ##
##                      ##
##########################
# ----------------------------------------------------------
# Sub: TIEHASH
#
# Args: {named parameter list}
#  -argv 	A list of command-line options to parse as settings
#  -<any>	Any valid Config::IniFiles parameters
#
# Description: Contructs an object tied to the hash 
#              with a tie call. The -argv list is parsed for 
#              settings file names and overrides =~ -.*
# ----------------------------------------------------------
# Date      Modification                              Author
# ----------------------------------------------------------
# 2000Mar21 Created the function                          JW
# 2000Jun14 Fixed include file processing                 JW
# ----------------------------------------------------------
sub TIEHASH {
	my $class = shift;
	my %parms = @_;

	# Get filename from command line
	my @args = @{$parms{-argv}} if defined $parms{-argv};
	delete $parms{-argv};
	my @filenames = grep /^[^-]/, @args;
	unless( @filenames ) {
		my $base;
		my $dir = File::Spec->curdir;
		( $base, undef , undef ) = fileparse( $0, '\..*' );
		if( opendir( DIR, $dir ) ) {
			@filenames = grep { /^$base\..+/i && -f File::Spec->catfile( $dir, $_ ) } readdir(DIR);
			closedir DIR;
		} # end if
	} # end if
	# But don't match the script / executable name!
	my $progname = fileparse( $0 );
	@filenames = sort( grep !/^$progname$/i, @filenames );

	# Create a new object, passing settings, and bless into our class
	# try each filename in the list until one of them comes up valid
	my $self = undef;
	while( @filenames ) {
		$parms{-file} = shift @filenames;
		my %hash;
		tie %hash, 'Config::IniFiles', ( %parms ) ;
		$self = tied(%hash);
		last if defined $self;
	} # end if
	return undef unless defined $self;
	return undef unless $self->Sections;

	bless( $self, $class );
	$self->{FILENAME} = $parms{-file};

	# Recurse the settings through Include
	while( defined $self->{v}{statistics}{Include} ) {
		$parms{-file} = $self->{v}{statistics}{Include};
		my $inc = new Config::IniFiles( %parms );
		$self->{v}{statistics}{Include} = undef;

		my( $S, $P );
		foreach $S ( $inc->Sections() ) {
			foreach $P ( $inc->Parameters( $S ) ) {
				$self->{v}{$S}{$P} = $inc->val( $S, $P ) unless defined $self->{v}{$S}{$P};
			} # end foreach
		} # end foreach
	} # end if

	# Get command line from parameters
	$self->_parse_command_line( @args );

	# Clean up the settings
	$self->_conversions();
	$self->_defaults();
	$self->_sanity_checks();
	return $self;
} # end TIEHASH
        
##########################
##                      ##
##    Public Methods    ##
##                      ##
##########################
# ----------------------------------------------------------
# Sub: getFilename
#
# Args: (None)
#
# Description: Returns the filename for the user settings file
# ----------------------------------------------------------
# Date      Modification                              Author
# ----------------------------------------------------------
# 2000Mar21 Created the function                          JW
# ----------------------------------------------------------
sub getFilename {
	my $self = shift;
	return $self->{FILENAME};
} # end getFilename

# ----------------------------------------------------------
# Sub: get_styles
#
# Args: $section
#	$section	The section (report or navigation) for which to 
#           parse settings from
#
# Description: Parses settings from a section of the settings
# into a css style sheet.
# ----------------------------------------------------------
# Date      Modification                              Author
# ----------------------------------------------------------
# 2000May26 Created method                                JW
# 2000Dec01 Now quotes font family names with whitespace  JW
# ----------------------------------------------------------
sub get_styles {
	my $self = shift;
	my( $section, $group, $elm ) = @_;
	$elm = 'body' unless defined $elm && $elm;
	
	if( defined $group && $group ) {
		$group .= '_';
	} else {
		$group = '';
	} # end if
	
	$self->newval( $section, '_styles', {} ) unless $self->val( $section, '_styles' );
	my $styles = $self->val( $section, '_styles' );
	$styles->{$elm} = {} unless defined $styles->{$elm};
	
	if( $self->val( $section, $group . 'Font' ) ) {
		# Make sure all elements with spaces are quoted
		my @families = split /\s*,\s/, $self->val( $section, $group . 'Font' );
		foreach (@families) {
			$_ = "\"$_\"" if /\s/ & !/\"/;
		} # end foreach
		$styles->{$elm}{'font-family'} = join ',', @families;
	} # end if
	if( $self->val( $section, $group . 'Font_Color' ) ) {
		$styles->{$elm}{color} = $self->val( $section, $group . 'Font_Color' );
	} # end if
	if( $self->val( $section, $group . 'Background' ) ) {
		$styles->{$elm}{'background-image'} = 'url(' . $self->val( $section, $group . 'Background' ) . ')';
	} # end if
	if( $self->val( $section, $group . 'BG_Color' ) ) {
		$styles->{$elm}{'background-color'} = $self->val( $section, $group . 'BG_Color' );
	} # end if

	return;
} # end get_styles

        
# ----------------------------------------------------------
# Sub: get_report_styles
#
# Args: $section
#	$section	The section (report letter) to get styles for
#
# Description: Convenince method to create a stylesheet for
# a given report.
# ----------------------------------------------------------
# Date      Modification                              Author
# ----------------------------------------------------------
# 2000May26 Created method                                JW
# 2000Oct03 Added check for Stylesheet attribute          JW
# ----------------------------------------------------------
sub get_report_styles {
	my $self = shift;
	my $section = shift || 'reports';
	
	# Don't set any default styles if there's a stylesheet present.
	return if $self->val( $section, 'Stylesheet' );
	
	$self->newval( $section, '_styles', {} ) unless $self->val( $section, '_styles' );
	my $styles = $self->val( $section, '_styles' );

	# Setup the style sheet for the Report pages
	$self->get_styles( $section );	

	# Duplicate the body element into p and td for Netscape
	$styles->{p} = {} unless defined $styles->{p} && ref($styles->{p}) eq 'HASH'; 
	$styles->{td} = {} unless defined $styles->{td} && ref($styles->{td}) eq 'HASH'; 
	%{$styles->{p}} = %{$styles->{body}};
	%{$styles->{td}} = %{$styles->{td}};

	$styles->{h1} = {} unless defined $styles->{h1};
	$styles->{h1}{'font-size'} = '20pt';
	$styles->{h1}{'text-align'} = 'CENTER';

	# Setup some defaults for the 'h2' style, then build styles from settings
	$styles->{h2} = {} unless defined $styles->{h2};
	$styles->{h2}{'font-size'} = '12pt';
	if( defined $self->val( $section, 'Title_BG_Color' ) ) {
		$styles->{h2}{width} = '95%';
		$styles->{h2}{padding} = '3pt';
		$styles->{h2}{border} = 'none';
		$styles->{h2}{'font-weight'} = 'bold';
	} # end if
	$self->get_styles( $section, 'Title', 'h2' );	

	$styles->{'.fineprint'} = {} unless defined $styles->{'.fineprint'};
	$styles->{'.fineprint'}{'font-size'} = '7pt';
	$styles->{'.smallfont'} = {} unless defined $styles->{'.smallfont'};
	$styles->{'.smallfont'}{'font-size'} = '8pt';

	if( $self->val( $section, 'Data_Font' ) ) {
		$styles->{td} = {} unless defined $styles->{td};
		$styles->{'td'}{'font-family'} = $self->val( $section, 'Data_Font' );
	} # end if
	if( $self->val( $section, 'Data_Font_Color_1' ) ) {
		$styles->{'td.alt1'} = {} unless defined $styles->{'td.alt1'};
		$styles->{'td.alt1'}{color} = $self->val( $section, 'Data_Font_Color_1' );
	} # end if
	if( $self->val( $section, 'Data_Font_Color_2' ) ) {
		$styles->{'td.alt2'} = {} unless defined $styles->{'td.alt2'};
		$styles->{'td.alt2'}{color} = $self->val( $section, 'Data_Font_Color_2' );
	} # end if
	if( $self->val( $section, 'Data_BG_Color_1' ) ) {
		$styles->{'td.alt1'} = {} unless defined $styles->{'td.alt1'};
		$styles->{'td.alt1'}{'background-color'} = $self->val( $section, 'Data_BG_Color_1' );
	} # end if
	if( $self->val( $section, 'Data_BG_Color_2' ) ) {
		$styles->{'td.alt2'} = {} unless defined $styles->{'td.alt2'};
		$styles->{'td.alt2'}{'background-color'} = $self->val( $section, 'Data_BG_Color_2' );
	} # end if

	$self->get_styles( $section, 'Data_Header', 'th' );	

	$self->get_styles( $section, 'Data_Total', 'th.total' );	

	return $self;
} # end get_report_styles


# ----------------------------------------------------------
# Sub: get_nav_styles
#
# Args: (None)
#
# Description: Convenience method to retrieve all the styles 
# and set defaults necessary for the Navigation stylesheet
# ----------------------------------------------------------
# Date      Modification                              Author
# ----------------------------------------------------------
# 2000May26 Created method                                JW
# 2000Oct03 Added check for Stylesheet attribute          JW
# ----------------------------------------------------------
sub get_nav_styles {
	my $self = shift;

	# Don't set any default styles if there's a stylesheet present.
	return if $self->val( 'navigation', 'Stylesheet' );
	
	$self->newval( 'navigation', '_styles', {} ) unless $self->val( 'navigation', '_styles' );
	my $styles = $self->val( 'navigation', '_styles' );

	# Setup the style sheet for the Navigation panel
	$self->get_styles( 'navigation' );	

	$styles->{body} = {} unless defined $styles->{body};
	$styles->{body}{'font-size'} = '8pt';
	$styles->{td} = {} unless defined $styles->{body};
	$styles->{td}{'font-size'} = '8pt';
	$styles->{h4} = {} unless defined $styles->{h4};
	$styles->{h4}{'font-size'} = '12pt';
	
	$styles->{a} = {} unless defined $styles->{a};
	$styles->{a}{'font-weight'} = 'normal';
	$styles->{a}{'text-decoration'} = 'none';
	if( $self->val( 'navigation', 'Font_Color' ) ) {
		$styles->{'a:link'} = {} unless defined $styles->{'a:link'};
		$styles->{'a:visited'} = {} unless defined $styles->{'a:visited'};
		$styles->{'a:active'} = {} unless defined $styles->{'a:active'};
		$styles->{'a:link'}{color} = $self->val( 'navigation', 'Font_Color' );
		$styles->{'a:visited'}{color} = $self->val( 'navigation', 'Font_Color' );
		$styles->{'a:active'}{color} = $self->val( 'navigation', 'Font_Color' );
	} # end if
	$styles->{'a:hover'} = {} unless defined $styles->{'a:hover'};
	$styles->{'a:hover'}{'font-style'} = 'italic';

	return $self;
} # end get_nav_styles


##########################
##                      ##
##   Private Methods    ##
##                      ##
##########################
# ----------------------------------------------------------
# Sub: _parse_command_line
#
# Args: @args
#	@args	A list of command-line arguments to the program
#
# Returns: The name of the file parsed for settings.
#          Settings are stored in $settings->{config}
#
# Description: Parses command-line arguments for filename 
#              and overrides and builds a hash of settings
# ----------------------------------------------------------
# Date      Modification                              Author
# ----------------------------------------------------------
# 1999Mar29 Added support for commandline filename        JW
# 1999Aug21 Added suprort for command-line overrides      JW
# 2000May26 Pulled out defaults, conversion and sanities  JW
# 2000May26 Converted to command-line parser              JW
# 2000Jun14 Fixed processing for new data model           JW
# ----------------------------------------------------------
sub _parse_command_line {
	my $self = shift;
	my $filename = undef;
	my @args = @_;
	
	foreach ( @args ) {
		if( /^--?(.+)/ ) {
			# Parse '-option' arguments
			my $name = $1;
			my $val = 1;
			if( $name =~ /([^=]+)=(.+)/ ) {
				$name = $1;
				$val = $2;
			} # end if
			if( $name =~ /(_*[^_-]+)[_-](.+)/ ) {
				$name = $1;
				my $name2 = $2;
				$name2 =~ s/-/_/g;
			#** $self->newsection( $name ) unless defined $self->Parameters( $name );
				if( defined $self->val( $name, $name2 ) ) {
					$self->setval( $name, $name2, $val );
				} else {
					$self->newval( $name, $name2, $val );
				} # end if
			} else {
				# Invalid command line argument
				wadg::Errors::warning( 'W0006', $name );
			} # end if
		} # end if
	} # end for

} # end _parse_command_line

# ----------------------------------------------------------
# Sub: _sanity_check
#
# Args: (None)
#
# Description: Runs some sanity checks on the settings and 
# forces some values based on others.
# ----------------------------------------------------------
# Date      Modification                              Author
# ----------------------------------------------------------
# 2000Apr04 Fixed bug in declarations like my( undef, ... JW
# 2000May26 Created from sections of _read_user_settings  JW
# 2000Nov29 One_File test with -d; now makes proper paths JW
# 2001Feb06 Changed directory test, cause it failed       JW
# ----------------------------------------------------------
sub _sanity_checks {
	my $self = shift;

	# 
	# Get the output directory for all the output files and 
	# force them into that directory
	#
	
	my $outDir;
	# Set statistics_One_File based on what the output file is
	(undef, $outDir, undef) = fileparse( $self->val( 'reports', 'File_Out' ) );

	if( $outDir eq  $self->val( 'reports', 'File_Out' ) ) {
		$self->newval( 'statistics', 'One_File', 0 );
		# Also, force it to proper punctuation so errors don't occur
#		$outDir = File::Spec->catdir( $self->val( 'reports', 'File_Out' ), File::Spec->curdir );
#		$self->setval( 'reports', 'File_Out', $outDir );
	} else {
		$self->newval( 'statistics', 'One_File', 1 );
	} # end if

	if( $self->val( 'navigation', 'File_Out' ) && 
	    $self->val( 'navigation', 'File_Out' ) !~ /^(LEFT|TOP|RIGHT|BOTTOM|NONE)$/i ) {;
		$self->setval( 'navigation', 'File_Out', File::Spec->catfile( $outDir, basename( $self->val( 'navigation', 'File_Out' ) ) ) );
	} # end if
	if( defined $self->val( 'statistics', 'Frame_File_Out' ) ) {
		$self->setval( 'statistics', 'Frame_File_Out', File::Spec->catfile( $outDir, basename( $self->val( 'statistics', 'Frame_File_Out' ) ) ) );
	} else {
		$self->newval( 'statistics', 'Frame_File_Out', File::Spec->catfile( $outDir, 'index.html' ) );
	} # end if

	#
	# Drop any any last '/' or '\' on website_Base_URL
	#
	my $website = $self->val( 'website', 'Base_URL' );
	if( $website =~ s/^(.*)[\/\\]$/$1/ ) {
		$self->setval( 'website', 'Base_URL', $website );
	} # end if

	#
	# Make Active Column a useful value
	#
	if( defined $self->val( 'reports', 'Active_Column' ) ) {
		# Make sure it's only the first letter.
		my $ac = substr( $self->val( 'reports', 'Active_Column' ), 0, 1 );
		# Add "_" indicator to capital columns
		$ac .= '_' if $ac eq uc($ac);
		$self->setval( 'reports', 'Active_Column', $ac );
	}  # end if

	# If outputting to STDOUT, then force non-frames mode and 
	# one-file mode and turn off Quick Summary (STDOUT cannot 
	# be reopend to insert the summary). No notices either.
	if( $self->val( 'reports', 'File_Out' ) =~ /^(-|CGI)$/i ) {
		$self->setval( 'navigation', 'File_Out', 'RIGHT' ) unless $self->val( 'navigation', 'File_Out' ) =~ /^(LEFT|TOP|RIGHT|BOTTOM)$/i;
		$self->setval( 'statistics', 'One_File', 1 );
		$self->delval( 'q', 'Rows' );
		my $v = $self->val( 'statistics', 'Verbose' );
		$v =~ s/N//;
		$self->setval( 'statistics', 'Verbose', $v );
	} # end if

	# Don't allow [statistics]File_In to contain %infile% or ${infile}
	my $infile = $self->val( 'statistics' , 'File_In' );
	$infile	=~ s/%infile%//i;
	$infile	=~ s/\$\{infile\}//i;
	$self->setval( 'statistics' , 'File_In', $infile );

	return $self;
} # end _sanity_checks


# ----------------------------------------------------------
# Sub: _conversions
#
# Args: (None)
#
# Description: Converts settings from old styles to current one
# ----------------------------------------------------------
# Date      Modification                              Author
# ----------------------------------------------------------
# 2000May26 Created from sections of _read_user_settings  JW
# 2000Jun14 Implemented __convert method for clarity      JW
# ----------------------------------------------------------
sub _conversions {
	my $self = shift;

	# Title moved from [reports] to [website]
	$self->__convert( 'website', 'Title', 'reports', 'Title' );

	# Active_Column moved from [statistics] to [reports]
	$self->__convert( 'reports', 'Active_Column', 'statistics', 'Active_Column' );

	# Graph_Font moved from [reports] to [graphs]Font
	$self->__convert( 'graphs', 'Font', 'reports', 'Graph_Font' );

	# Image_ball changed to Bullet_Image
	$self->__convert( 'navigation', 'Bullet_Image', 'navigation', 'Image_Ball' );

	# Reverse_Time moved from [statistics] to [reports]
	$self->__convert( 'reports', 'Reverse_Time', 'statistics', 'Reverse_Time' );

	# Update names of nav page styles
	$self->__convert( 'navigation', 'Font', 'navigation', 'Page_Font' );
	$self->__convert( 'navigation', 'Font_Color', 'navigation', 'Page_Font_Color' );
	$self->__convert( 'navigation', 'BG_Color', 'navigation', 'Page_BG_Color' );
	$self->__convert( 'navigation', 'Background', 'navigation', 'Page_Background' );

	# Update names of "_Color" to "_BG_Color"
	$self->__convert( 'reports', 'Data_BG_Color_1', 'reports', 'Data_Color_1' );
	$self->__convert( 'reports', 'Data_BG_Color_2', 'reports', 'Data_Color_2' );
	$self->__convert( 'reports', 'Data_Header_BG_Color', 'reports', 'Data_Header_Color' );
	$self->__convert( 'reports', 'Data_Total_BG_Color', 'reports', 'Data_Total_Color' );

	# Move "General_Rows" and "Summary_Rows" to their own sections
	$self->__convert( 'statistics', 'General_Rows', 'x', 'Rows' );
	$self->__convert( 'statistics', 'Summary_Rows', 'q', 'Rows' );

	# Equivalent boolean keywords
	my( $S, $P );
	foreach $S ( $self->Sections() ) {
		foreach $P ( $self->Parameters( $S ) ) {
			next unless defined $self->val( $S, $P );
			$self->setval( $S, $P, 1 ) if $self->val( $S, $P ) =~ /^(YES|ON|TRUE)$/i;
			$self->setval( $S, $P, 0 ) if $self->val( $S, $P ) =~ /^(NO|OFF|FALSE|NONE)$/i;
		} # end foreach
	} # end foreach
	
	return $self;
} # end _conversions

# ----------------------------------------------------------
# Sub: __convert
#
# Args: $new_section, $new_parm, $old_section, $old_parm
#
# Description: Helper method to convert settings from one
# area to method
# ----------------------------------------------------------
# Date      Modification                              Author
# ----------------------------------------------------------
# 2000Jun14 Created function                              JW
# ----------------------------------------------------------
sub __convert {
	my $self = shift;
	my( $new_section, $new_parm, $old_section, $old_parm ) = @_;
	
	unless( defined $self->val( $new_section, $new_parm ) ) {
		$self->newval( $new_section, $new_parm, $self->val( $old_section, $old_parm ) );
	} # end if
} # end __convert

# ----------------------------------------------------------
# Sub: _defaults
#
# Args: (None)
#
# Description: Sets default values for some elements
# ----------------------------------------------------------
# Date      Modification                              Author
# ----------------------------------------------------------
# 2000May26 Created from sections of _read_user_settings  JW
# 2000Oct17 Fixed bug in ALL setting in GENERAL section   JW
# 2000Nov28 Added default for Graph_Level setting         JW
# ----------------------------------------------------------
sub _defaults {
	my $self = shift;
	
	#	
	# Check GD for support of desired output format. If none
	# given or not capable, then set output format accordingly.
	#
	my $g = new GD::Image(0,0);

	if( defined $self->val( 'graphs', 'Format' ) ) {
		if( $self->val( 'graphs', 'Format' ) =~ /jpe?g/i ) {
			$self->setval( 'graphs', 'Format', 'jpeg' );
		} else {
			$self->setval( 'graphs', 'Format', lc($self->val( 'graphs', 'Format' )) );
		} # end if
		unless( $g->can( $self->val( 'graphs', 'Format' ) ) ) {
			if( $g->can( 'png' ) ) {
				$self->setval( 'graphs', 'Format', 'png' );
			} elsif( $g->can( 'gif' ) ) {
				$self->setval( 'graphs', 'Format', 'gif' );
			} elsif( $g->can( 'jpeg' ) ) {
				$self->setval( 'graphs', 'Format', 'jpeg' );
			} # end if
		} # end if
	} else {
		if( $g->can( 'png' ) ) {
			$self->newval( 'graphs', 'Format', 'png' );
		} elsif( $g->can( 'gif' ) ) {
			$self->newval( 'graphs', 'Format', 'gif' );
		} elsif( $g->can( 'jpeg' ) ) {
			$self->newval( 'graphs', 'Format', 'jpeg' );
		} # end if
	} # end if
	
	# -- If format is still not defined, then error:
	unless( defined $self->val( 'graphs', 'Format' ) ) {
		wadg::Errors::error( -1, 'E0015', 'dgsupport\@wadsack-allen.com', $GD::VERSION, $self->val( '_INTERNAL', '_VERSION' ) );
	} # end unless

	# -- Default vebosity is all
	if( !defined $self->val( 'statistics', 'Verbose' ) ) {
		$self->newval( 'statistics', 'Verbose', 'NWE' );
	} elsif( $self->val( 'statistics', 'Verbose' ) =~ /NONE/i ) {
		$self->setval( 'statistics', 'Verbose', '' );
	} # end if

	# -- Make General Summary and Quick Summary ROWS defaults
	{ # Scope block
		my $general = $self->val( 'x', 'Rows' ) || $self->val( 'GENERAL', 'Rows' ) || '';
		my $quick = $self->val( 'q', 'Rows' ) || $self->val( 'QUICK', 'Rows' ) || '';
		$self->newval( 'x', 'Rows', '' ) unless $general;
		$self->setval( 'x', 'Rows', '' ) if $general =~ /ALL/i || $general =~ /ALL/i;
		$self->delval( 'GENERAL', 'Rows' ) if $general =~ /ALL/i;
		if( $quick =~ /NONE/i ) {
			$self->delval( 'q', 'Rows' );
			$self->delval( 'QUICK', 'Rows' );
		} # end if
	} # end scope

	# -- Default Navigation Position is RIGHT
	if( defined $self->val( 'navigation', 'File_Out' ) ) {
		$self->setval( 'navigation', 'File_Out', 'RIGHT') if $self->val( 'navigation', 'File_Out' ) eq $self->val( 'reports', 'File_Out' );
	} else {
		# Since 'reports' is default, this case really never happens. 
		# So maybe this should be an assert? [JW]
		$self->newval( 'navigation', 'File_Out', 'RIGHT');
	} # end if
	$self->delval( 'navigation', 'File_Out' ) if $self->val( 'navigation', 'File_Out' ) =~ /NONE/i;
	
	# -- Deafault Graph_Level is 1
	$self->newval( 'reports', 'Graph_Level', '1') unless defined $self->val( 'reports', 'Graph_Level' );
	
	return $self;
} # end _defaults

# module clean-up code here (global destructor)
END { }

1;  # so the require or use succeeds
