/*  signals.c: signal handling */

/*  This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License , or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; see the file COPYING.  If not, write to
    the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

    You may contact the author by:
       e-mail:  hlub@knoware.nl
*/

#include "rlwrap.h"

int child_is_dead = FALSE;
int childs_exit_status = 0;

static void change_signalmask(int, int*);
static RETSIGTYPE do_nothing(int);
static RETSIGTYPE child_died(int);
static RETSIGTYPE pass_on_signal(int);
static RETSIGTYPE handle_sigtstp(int);
static RETSIGTYPE handle_segfault(int);

static int adapt_tty_winsize(int,int);
static int signals_to_be_passed_on[] = {SIGHUP, SIGINT,
					SIGQUIT, SIGABRT, SIGTERM, SIGCONT, SIGUSR1,
					SIGUSR2,  SIGWINCH,  0}; 

#ifdef DEBUG
static char *signal_names[]  = {"SIGHUP", "SIGINT", 
				"SIGQUIT", "SIGABRT", "SIGTERM", "SIGCONT", "SIGUSR1",
				"SIGUSR2",  "SIGWINCH"};
static void log_named_signal(int);

#endif


int sigterm_received = FALSE;



void
install_signal_handlers()
{
  int i;
  
  signal(SIGCHLD, &child_died);
  signal(SIGTSTP, &handle_sigtstp);
#ifndef DEBUG
  signal(SIGSEGV, &handle_segfault); /* we want core dumps when debugging, no polite excuses! */
#endif
 
  for (i=0; signals_to_be_passed_on[i]; i++) 
    signal(signals_to_be_passed_on[i], &pass_on_signal);
}



void
ignore_sigchld() 
{
  signal(SIGCHLD, &do_nothing);
}
  

/* we'll call signals whose handlers mess with readline's internal state "messy signals"
   at present SIGTSTP and SIGWINCH are considered "messy". The following 2 functions are used
   in main.c to block messy signals except when waiting for I/O */

void
block_signals(int *sigs)
{
  change_signalmask(SIG_BLOCK, sigs);
}

void
unblock_signals(int *sigs)
{
  change_signalmask(SIG_UNBLOCK, sigs);
}


static void
change_signalmask (int how, int* sigs) /* sigs should point to a *zero-terminated* list of signals */     
{
  int i;
  sigset_t mask;
  sigemptyset(&mask);
  for (i=0; sigs[i]; i++) 
    sigaddset(&mask, sigs[i]);
  sigprocmask(how, &mask, NULL);
}



static RETSIGTYPE
handle_sigtstp(int signo)  
{
  sigset_t mask;
  int saved_errno, error;

  DPRINTF0(DEBUG_SIGNALS, "got SIGTSTP");
 
  if (child_pid && (error = kill (-child_pid, SIGTSTP)) ) {
    myerror("Failed to deliver signal");
  } 
  
  if (within_line_edit)
    save_rl_state(&saved_rl_state);
  
  /* unblock SIGTSTP, before we can send it to ourself (it is stopped while we're handling it) */

  sigemptyset(&mask);
  sigaddset(&mask, SIGTSTP);
  sigprocmask(SIG_UNBLOCK, &mask, NULL);
  signal(SIGTSTP, SIG_DFL);  /* reset disposition to default (i.e. suspend) */
  /* suspend myself (and supbprocess, which is in the same process group) */
  DPRINTF0(DEBUG_SIGNALS, "sending ourselves a SIGTSTP");
  kill(getpid(), SIGTSTP);
  
  /* keyboard gathers dust, kingdoms crumble,.... */

  /* Beautiful princess types "fg", (or her father tries to kill us...) and we wake up HERE: */

  DPRINTF0(DEBUG_SIGNALS, "woken up");
  saved_errno = errno;

  signal(signo, &handle_sigtstp); /* re-install handler (old-style) */
  errno=saved_errno;
  /* we should really block SIGWINCH here ... */
  if(within_line_edit) {
    restore_rl_state (&saved_rl_state); 
    /* this is why we call SIGTSTP a "messy" signal */
  } else { 
    set_echo(FALSE);	  
    cr();
    if (slave_is_in_raw_mode() && ! always_readline)
      return;	
    write(STDOUT_FILENO, saved_rl_state.prompt, strlen(saved_rl_state.prompt));
  }
  adapt_tty_winsize(STDIN_FILENO, master_pty_fd); /* just in case */
  
}



/* signal handler, just passes the signal to the child process
   SIGWINCH is only passed after calling adapt_tty_winsize()
*/

static RETSIGTYPE pass_on_signal(int signo) {
  int ret, saved_errno = errno, pass_it_on = TRUE;
  
#ifdef DEBUG
  log_named_signal(signo);
#endif
      
  switch (signo) {
  case SIGWINCH:
    /* Make slave pty's winsize equal to that of STDIN. Pass the signal on *only if*
       winsize has changed. This is particularly important because we have the slave pty
       still open - when we pass on the signal the child will probably do a TIOCSWINSZ ioctl()
       on the slave pty  - causing the parent to get a SIGWINCH again, causing a loop */
    pass_it_on = adapt_tty_winsize(STDIN_FILENO, master_pty_fd);
    break;
  case SIGTERM:
    sigterm_received = TRUE;
    break;
  default:
    break;
  }
  if (!child_pid)
    pass_it_on = FALSE; /* no child (yet) to pass it on to */
  if (pass_it_on) { /* we resend the signal to the proces *group* of the child */
    ret= kill (-child_pid, signo);
    DPRINTF3(DEBUG_SIGNALS, "kill(%d,%d) = %d", -child_pid, signo, ret);  
    signal(signo, &pass_on_signal); /* re-install handler (old-style) */
  }
  errno=saved_errno;
}



/* This function copies from_fd's winsize to that of to_fd, and, if possible, tries to keep displayed
   line tidy.   A return value != 0 means that the size has changed*/
					 
static int
adapt_tty_winsize(int from_fd, int to_fd) 
{
   int ret, lineheight, linelength, cursor_height, i;
   
   struct winsize old_winsize = window_size; 
   ret = ioctl (from_fd, TIOCGWINSZ, &window_size);
   DPRINTF1(DEBUG_SIGNALS, "ioctl (..., TIOCGWINSZ) = %d", ret);
   if (window_size.ws_col != old_winsize.ws_col
       || window_size.ws_row != old_winsize.ws_row ) {
     ret = ioctl (to_fd, TIOCSWINSZ, &window_size);
     DPRINTF1(DEBUG_SIGNALS, "ioctl (..., TIOCSWINSZ) = %d", ret);
     rl_set_screen_size(window_size.ws_row, window_size.ws_col);   /* mess with readline internal state:
								      this is why we have SIGWINCH blocked most of the time */
     
     if (within_line_edit) { /* try to keep displayed line tidy  -  why can't we just  rl_resize_terminal()? */
       /* redisplay_multiple_lines will only ever be set if SPY_ON_READLINE is defined: otherwise we just don't know, cf rlwrap.h */
       if (redisplay_multiple_lines) { /* OK, so we know that _rl_horizontal_scroll_mode is FALSE */
	 linelength = strlen(rl_line_buffer) + strlen(rl_prompt);
	 lineheight = (linelength == 0 ? 0 : (1 + (max (rl_point, (linelength - 1)) / old_winsize.ws_col))); 
	 if (lineheight > 1
	     && term_cursor_up != NULL
	     && term_cursor_down != NULL) { /* if we have multiple rows displayed we have to clean them up first */
	   cr();
	   cursor_height = (strlen(rl_prompt) + rl_point)/old_winsize.ws_col; /* on which line is cursor now? */
	   DPRINTF2(DEBUG_SIGNALS, "lineheight: %d, cursor_height: %d", lineheight, cursor_height);
	   for(i = 1 + cursor_height; i  < lineheight  ; i++)
	     curs_down(); /* ...move it down to last line */
	   for(i = lineheight; i > 1; i--) { /* go up again, erasing every line */
	     clear_line();
	     curs_up();
	   }	    
	 }
	 clear_line();
	 cr();
	 rl_on_new_line(); /* let readline re-draw entire prompt + line */
       }
      
       my_redisplay();  
     }
     return 1;  
   } else { /* window size has not changed */
     return 0;
   }
}
   
  
static RETSIGTYPE
child_died(int unused)
{
  int             saved_errno;

  saved_errno = errno;
  while ((-1 == waitpid(-1, &childs_exit_status, WNOHANG))
	 && (errno == EINTR));
  child_is_dead = TRUE;
  child_pid = 0; /* thus we know that there is no child anymore to pass signals to */
  errno = saved_errno;
  
  return; /* allow remaining output from child to be processed in main loop */
          /* (so that we will see childs good-bye talk)                     */ 
          /* this will then clean up and terminate                          */
    
}

static RETSIGTYPE
handle_segfault(int unused) /* Even after sudden and unexpected death, leave the terminal in a tidy state */
{
  printf("\nrlwrap: Oops, segfault -  this should not have happened!\n"
	 "re-configure with --enable-debug if you want a core dump\n"
	 "Resetting terminal and cleaning up...\n");
  if (terminal_settings_saved)
    tcsetattr(STDIN_FILENO, TCSAFLUSH, &saved_terminal_settings);
  exit(EXIT_FAILURE);
}


static RETSIGTYPE
do_nothing(int unused) 
{
}


#ifdef DEBUG
static void
log_named_signal(int signo) {
  if (debug) {
    int *p,i ;
    for (i=0, p = signals_to_be_passed_on; *p; p++,i++) {
      if (*p == signo) {
	DPRINTF2(DEBUG_SIGNALS, "got signal %d (%s)", signo, signal_names[i]);
	break;
      }
    }
    if (!*p) 
      DPRINTF1(DEBUG_SIGNALS, "Huh? got signal %d", signo);
  }
}
#endif
