/*
  rlBabel: Korean (CP949 subset) settings

  Copyright (c) 2006 Peter Jolly.

  This library is free software; you can redistribute it and/or modify it under
  the terms of the GNU Lesser General Public License as published by the Free
  Software Foundation; either version 2.1 of the License, or (at your option)
  any later version.

  This library is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
  FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
  details.

  You should have received a copy of the GNU Lesser General Public License
  along with this library; if not, write to the Free Software Foundation, Inc.,
  51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

  As a special exception to the GNU Lesser General Public License (LGPL), you
  may include a publicly distributed version of the library alongside a "work
  that uses the Library" to produce a composite work that includes the library,
  and distribute that work under terms of your choice, without any of the
  additional requirements listed in clause 6 of the LGPL.

  A "publicly distributed version of the library" means either an unmodified
  binary as distributed by Haeleth, or a modified version of the library that
  is distributed under the conditions defined in clause 2 of the LGPL, and a
  "composite work that includes the library" means a RealLive program which
  links to the library, either through the LoadDLL() interface or a #DLL
  directive, and/or includes code from the library's Kepago header.

  Note that this exception does not invalidate any other reasons why any part
  of the work might be covered by the LGPL.
*/

#define DEBUG_ONLY

#include "Encodings/cp949.h"

#include <stdint.h>
#include <algorithm>  // for lower_bound()
#include <cstring>
#include <string>

Cp949::Cp949() {
  //  DesirableCharset = HANGUL_CHARSET;
  NoTransforms = false;
  //  UseUnicode = GetSystemDefaultLangID() & 0x1ff != 0x12;
}

#ifndef NO_CP949_CONVERSION

const size_t extras_length = 679;
const uint16_t extras[extras_length] = {
  0xa046, 0xa04f, 0xa057, 0xa059, 0xa06c, 0xa074, 0xa075, 0xa077,
  0xa079, 0xa086, 0xa089, 0xa08a, 0xa08b, 0xa092, 0xa095, 0xa097,
  0xa09c, 0xa0a5, 0xa0a6, 0xa0ac, 0xa0b1, 0xa0b2, 0xa0b3, 0xa0b4,
  0xa0b8, 0xa0ba, 0xa0bb, 0xa0be, 0xa0c0, 0xa0c1, 0xa0c4, 0xa0c8,
  0xa0d0, 0xa0d1, 0xa0d3, 0xa0d5, 0xa0e0, 0xa0ee, 0xa0f0, 0xa0f1,
  0xa0f2, 0xa0f9, 0xa0fb, 0xa144, 0xa147, 0xa14a, 0xa14b, 0xa153,
  0xa156, 0xa163, 0xa168, 0xa171, 0xa182, 0xa183, 0xa18b, 0xa18c,
  0xa18e, 0xa195, 0xa19a, 0xa19c, 0xa247, 0xa248, 0xa24b, 0xa24e,
  0xa24f, 0xa257, 0xa258, 0xa25a, 0xa261, 0xa268, 0xa26b, 0xa26e,
  0xa26f, 0xa277, 0xa278, 0xa27a, 0xa281, 0xa282, 0xa28d, 0xa297,
  0xa29e, 0xa34b, 0xa34d, 0xa34f, 0xa35a, 0xa361, 0xa362, 0xa364,
  0xa36a, 0xa370, 0xa371, 0xa374, 0xa378, 0xa386, 0xa387, 0xa389,
  0xa38b, 0xa39b, 0xa39c, 0xa39d, 0xa39e, 0xa3a0, 0xa441, 0xa444,
  0xa447, 0xa451, 0xa456, 0xa46b, 0xa46f, 0xa471, 0xa479, 0xa47a,
  0xa482, 0xa487, 0xa48a, 0xa54a, 0xa54d, 0xa54e, 0xa551, 0xa558,
  0xa559, 0xa562, 0xa565, 0xa566, 0xa56e, 0xa56f, 0xa571, 0xa572,
  0xa576, 0xa579, 0xa582, 0xa585, 0xa586, 0xa58e, 0xa58f, 0xa591,
  0xa592, 0xa599, 0xa59a, 0xa59d, 0xa641, 0xa649, 0xa64a, 0xa64c,
  0xa64e, 0xa655, 0xa659, 0xa661, 0xa669, 0xa66d, 0xa670, 0xa673,
  0xa681, 0xa682, 0xa684, 0xa685, 0xa68c, 0xa68f, 0xa692, 0xa693,
  0xa746, 0xa74b, 0xa754, 0xa755, 0xa756, 0xa758, 0xa763, 0xa764,
  0xa767, 0xa76b, 0xa773, 0xa774, 0xa776, 0xa784, 0xa788, 0xa78a,
  0xa791, 0xa799, 0xa79c, 0xa79f, 0xa7a0, 0xa848, 0xa849, 0xa84b,
  0xa852, 0xa853, 0xa856, 0xa859, 0xa85a, 0xa868, 0xa869, 0xa86b,
  0xa86c, 0xa86d, 0xa874, 0xa877, 0xa87a, 0xa881, 0xa889, 0xa88a,
  0xa88c, 0xa88e, 0xa895, 0xa898, 0xa89c, 0xa944, 0xa945, 0xa947,
  0xa949, 0xa950, 0xa953, 0xa956, 0xa957, 0xa965, 0xa968, 0xa96e,
  0xa96f, 0xa972, 0xa976, 0xa984, 0xa985, 0xa987, 0xa989, 0xa98a,
  0xa994, 0xa99d, 0xaa42, 0xaa46, 0xaa47, 0xaa4f, 0xaa52, 0xaa58,
  0xaa68, 0xaa6b, 0xaa6d, 0xaa79, 0xaa81, 0xaa87, 0xaa89, 0xaa9e,
  0xab49, 0xab50, 0xab52, 0xab56, 0xab62, 0xab69, 0xab6b, 0xab72,
  0xab77, 0xab78, 0xab86, 0xab87, 0xab89, 0xab8a, 0xab91, 0xab97,
  0xab9f, 0xaba0, 0xac42, 0xac4e, 0xac64, 0xac69, 0xac71, 0xac72,
  0xac74, 0xac82, 0xac95, 0xac96, 0xac9e, 0xad43, 0xad4c, 0xad53,
  0xad56, 0xad5a, 0xad68, 0xad6a, 0xad6c, 0xad77, 0xad8d, 0xad90,
  0xad93, 0xad94, 0xad9c, 0xad9d, 0xad9f, 0xada0, 0xae47, 0xae4d,
  0xae55, 0xae56, 0xae58, 0xae59, 0xae5a, 0xae67, 0xae6c, 0xae82,
  0xae8f, 0xae91, 0xae9d, 0xae9e, 0xaf41, 0xaf4d, 0xaf4e, 0xaf51,
  0xaf55, 0xaf63, 0xaf64, 0xaf66, 0xaf68, 0xaf73, 0xaf7a, 0xaf81,
  0xaf83, 0xaf85, 0xaf8b, 0xaf94, 0xaf98, 0xaf99, 0xaf9f, 0xb04b,
  0xb04c, 0xb050, 0xb054, 0xb062, 0xb063, 0xb065, 0xb06d, 0xb071,
  0xb09c, 0xb09e, 0xb09f, 0xb145, 0xb14e, 0xb152, 0xb153, 0xb155,
  0xb15a, 0xb16f, 0xb172, 0xb176, 0xb184, 0xb185, 0xb187, 0xb188,
  0xb189, 0xb194, 0xb19e, 0xb242, 0xb250, 0xb252, 0xb25a, 0xb263,
  0xb266, 0xb267, 0xb26f, 0xb270, 0xb272, 0xb27a, 0xb283, 0xb28e,
  0xb28f, 0xb291, 0xb293, 0xb29a, 0xb29d, 0xb341, 0xb349, 0xb34a,
  0xb34c, 0xb34e, 0xb359, 0xb36d, 0xb36f, 0xb374, 0xb382, 0xb383,
  0xb386, 0xb38d, 0xb390, 0xb394, 0xb39c, 0xb39d, 0xb39f, 0xb3a0,
  0xb44b, 0xb461, 0xb46e, 0xb470, 0xb472, 0xb475, 0xb482, 0xb485,
  0xb48c, 0xb494, 0xb495, 0xb498, 0xb49c, 0xb544, 0xb545, 0xb547,
  0xb549, 0xb554, 0xb563, 0xb568, 0xb56e, 0xb56f, 0xb574, 0xb57a,
  0xb585, 0xb586, 0xb587, 0xb593, 0xb595, 0xb598, 0xb59c, 0xb644,
  0xb645, 0xb647, 0xb64f, 0xb653, 0xb655, 0xb675, 0xb676, 0xb677,
  0xb68d, 0xb696, 0xb69b, 0xb69d, 0xb742, 0xb743, 0xb74b, 0xb74c,
  0xb74e, 0xb74f, 0xb756, 0xb762, 0xb76a, 0xb76d, 0xb76f, 0xb77a,
  0xb78b, 0xb78c, 0xb78d, 0xb78f, 0xb795, 0xb79d, 0xb79e, 0xb7a0,
  0xb842, 0xb862, 0xb863, 0xb86a, 0xb86f, 0xb870, 0xb878, 0xb879,
  0xb888, 0xb88b, 0xb88f, 0xb897, 0xb898, 0xb89a, 0xb89c, 0xb947,
  0xb949, 0xb957, 0xb95a, 0xb963, 0xb964, 0xb968, 0xb96c, 0xb96d,
  0xb96f, 0xb970, 0xb977, 0xb97a, 0xb983, 0xb984, 0xb98c, 0xb98d,
  0xb98f, 0xb990, 0xb991, 0xb99c, 0xba45, 0xba46, 0xba4d, 0xba4f,
  0xba5a, 0xba62, 0xba6e, 0xba70, 0xba76, 0xba83, 0xba8c, 0xba91,
  0xba93, 0xba9e, 0xbb4a, 0xbb4c, 0xbb51, 0xbb5a, 0xbb62, 0xbb68,
  0xbb74, 0xbb79, 0xbb83, 0xbb8b, 0xbb8c, 0xbb8e, 0xbb90, 0xbb97,
  0xbc57, 0xbc65, 0xbc69, 0xbc70, 0xbc79, 0xbc84, 0xbc86, 0xbc89,
  0xbc8b, 0xbc94, 0xbc9b, 0xbc9e, 0xbd41, 0xbd49, 0xbd4c, 0xbd53,
  0xbd57, 0xbd69, 0xbd6a, 0xbd6b, 0xbd6d, 0xbd70, 0xbd74, 0xbd82,
  0xbd83, 0xbd85, 0xbd8d, 0xbe49, 0xbe4f, 0xbe57, 0xbe58, 0xbe5a,
  0xbe62, 0xbe69, 0xbe6e, 0xbe76, 0xbe79, 0xbe98, 0xbe9b, 0xbe9e,
  0xbe9f, 0xbf47, 0xbf48, 0xbf4a, 0xbf4b, 0xbf52, 0xbf53, 0xbf56,
  0xbf5a, 0xbf68, 0xbf69, 0xbf6b, 0xbf6d, 0xbf74, 0xbf87, 0xbf8a,
  0xbf91, 0xbf96, 0xbf9e, 0xc047, 0xc04c, 0xc050, 0xc056, 0xc063,
  0xc064, 0xc067, 0xc06b, 0xc073, 0xc074, 0xc076, 0xc078, 0xc089,
  0xc096, 0xc097, 0xc09d, 0xc09e, 0xc0a0, 0xc141, 0xc143, 0xc145,
  0xc146, 0xc149, 0xc150, 0xc162, 0xc165, 0xc168, 0xc16c, 0xc174,
  0xc175, 0xc177, 0xc184, 0xc185, 0xc199, 0xc245, 0xc246, 0xc24b,
  0xc24d, 0xc251, 0xc252, 0xc258, 0xc267, 0xc26c, 0xc272, 0xc287,
  0xc294, 0xc296, 0xc297, 0xc298, 0xc2a0, 0xc341, 0xc343, 0xc344,
  0xc347, 0xc349, 0xc355, 0xc359, 0xc363, 0xc364, 0xc365, 0xc366,
  0xc371, 0xc372, 0xc37a, 0xc381, 0xc383, 0xc38e, 0xc396, 0xc39f,
  0xc444, 0xc44c, 0xc44f, 0xc45a, 0xc461, 0xc462, 0xc465, 0xc466,
  0xc467, 0xc46c, 0xc46d, 0xc470, 0xc483, 0xc485, 0xc486, 0xc491,
  0xc499, 0xc49a, 0xc49c, 0xc49d, 0xc548, 0xc561, 0xc562, 0xc569,
  0xc56a, 0xc570, 0xc575, 0xc579, 0xc581, 0xc586, 0xc587, 0xc588,
  0xc589, 0xc58e, 0xc597, 0xc643, 0xc64a, 0xc64c, 0xc64f };

uint16_t Cp949::JisDecode(uint16_t ch) const {
  // Special cases
  if (ch < 0x80) {
    return ch;
  } else if (ch == 0x8175) {
    return 0xa1b8;
  } else if (ch == 0x8177) {
    return 0xa1ba;
  } else if (ch == 0x8169) {
    return 0xa3a8;
  } else if (ch == 0xea40) {
    return 0x81c1;
  } else if (ch == 0xea41) {
    return 0x81c5;
  } else if (ch == 0xea42) {
    return 0x81b5;
  } else {
    // General cases
    int c1 = (ch >> 8) & 0xff, c2 = ch & 0xff;
    if (c1 <= 0x96) {
      int i = (c1 - 0x81) * 255 + c2 - 1;
      c2 = i % 177;
      return ((0x81 + i / 177) << 8) |
          (c2 + (c2 < 0x1a ? 0x41 : (c2 < 0x34 ? 0x47 : 0x4d)));
    } else if (c1 <= 0xe6) {
      int i = (c1 - (c1 <= 0x9f ? 0x97 : 0xd8)) * 255 + c2 - 1;
      return ((0xa1 + i / 94) << 8) | (0xa1 + i % 94);
    } else {
      int i = (c1 - 0xe7) * 255 + c2 - 1;
      if (i >= 0 && i < extras_length) {
        return extras[i];
      } else {
        return 0xa3bf;  // fail: return full-width question mark.
      }
    }
  }
}

void Cp949::JisEncodeString(const char* src, char* buf, size_t buflen) const {
    int srclen = std::strlen(src), k = 0, j = 0;
  while (k < srclen && j < buflen) {
    unsigned char c1 = (unsigned char) src[k++];
    if (c1 < 0x80) {
      buf[j++] = c1;
    } else {
      uint16_t ch = c1 << 8 | (unsigned char) src[k++];
      if (ch == 0xa1b8)
        ch = 0x8175;
      else if (ch == 0xa1ba)
        ch = 0x8177;
      else if (ch == 0xa3a8)
        ch = 0x8169;
      else if (ch == 0x81c1)
        ch = 0xea40;
      else if (ch == 0x81c3)
        ch = 0xea41;
      else if (ch == 0x81b5)
        ch = 0xea42;
      c1 = ch >> 8;
      unsigned char c2 = ch & 0xff;
      if (c1 <= 0x9f) {
        int i = (c1 - 0x81) * 177 + c2 -
                (c2 < 0x5b ? 0x41 : (c2 < 0x7b ? 0x47 : 0x4d));
        buf[j++] = i / 255 + 0x81;
        buf[j++] = i % 255 + 1;
      } else if (c1 >= 0xa1 && c2 >= 0xa1) {
        int i = (c1 - 0xa1) * 94 + c2 - 0xa1;
        c1 = i / 255 + 0x97;
        buf[j++] = c1 > 0x9f ? c1 + 0x41 : c1;
        buf[j++] = i % 255 + 1;
      } else {
        const uint16_t* p =
            std::lower_bound(extras, extras + extras_length, ch);
        if (*p == ch) {
          int i = p - extras;
          buf[j++] = i / 255 + 0xe7;
          buf[j++] = i % 255 + 1;
        } else {
          buf[j++] = 0x97;
          buf[j++] = 0xdb;
        }
      }
    }
  }
  buf[j] = 0;
}

const uint16_t ksc_to_uni[] = {
  0xac02, 0xac03, 0xac05, 0xac06, 0xac0b, 0xac0c, 0xac0d, 0xac0e,
  0xac0f, 0xac18, 0xac1e, 0xac1f, 0xac21, 0xac22, 0xac23, 0xac25,
  0xac26, 0xac27, 0xac28, 0xac29, 0xac2a, 0xac2b, 0xac2e, 0xac32,
  0xac33, 0xac34, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xac35, 0xac36, 0xac37, 0xac3a, 0xac3b, 0xac3d, 0xac3e, 0xac3f,
  0xac41, 0xac42, 0xac43, 0xac44, 0xac45, 0xac46, 0xac47, 0xac48,
  0xac49, 0xac4a, 0xac4c, 0xac4e, 0xac4f, 0xac50, 0xac51, 0xac52,
  0xac53, 0xac55, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xac56, 0xac57, 0xac59, 0xac5a, 0xac5b, 0xac5d, 0xac5e, 0xac5f,
  0xac60, 0xac61, 0xac62, 0xac63, 0xac64, 0xac65, 0xac66, 0xac67,
  0xac68, 0xac69, 0xac6a, 0xac6b, 0xac6c, 0xac6d, 0xac6e, 0xac6f,
  0xac72, 0xac73, 0xac75, 0xac76, 0xac79, 0xac7b, 0xac7c, 0xac7d,
  0xac7e, 0xac7f, 0xac82, 0xac87, 0xac88, 0xac8d, 0xac8e, 0xac8f,
  0xac91, 0xac92, 0xac93, 0xac95, 0xac96, 0xac97, 0xac98, 0xac99,
  0xac9a, 0xac9b, 0xac9e, 0xaca2, 0xaca3, 0xaca4, 0xaca5, 0xaca6,
  0xaca7, 0xacab, 0xacad, 0xacae, 0xacb1, 0xacb2, 0xacb3, 0xacb4,
  0xacb5, 0xacb6, 0xacb7, 0xacba, 0xacbe, 0xacbf, 0xacc0, 0xacc2,
  0xacc3, 0xacc5, 0xacc6, 0xacc7, 0xacc9, 0xacca, 0xaccb, 0xaccd,
  0xacce, 0xaccf, 0xacd0, 0xacd1, 0xacd2, 0xacd3, 0xacd4, 0xacd6,
  0xacd8, 0xacd9, 0xacda, 0xacdb, 0xacdc, 0xacdd, 0xacde, 0xacdf,
  0xace2, 0xace3, 0xace5, 0xace6, 0xace9, 0xaceb, 0xaced, 0xacee,
  0xacf2, 0xacf4, 0xacf7, 0xacf8, 0xacf9, 0xacfa, 0xacfb, 0xacfe,
  0xacff, 0xad01, 0xad02, 0xad03, 0xad05, 0xad07, 0xad08, 0xad09,
  0xad0a, 0xad0b, 0xad0e, 0xad10, 0xad12, 0xad13, 0xad14, 0xad15,
  0xad16, 0xad17, 0xad19, 0xad1a, 0xad1b, 0xad1d, 0xad1e, 0xad1f,
  0xad21, 0xad22, 0xad23, 0xad24, 0xad25, 0xad26, 0xad27, 0xad28,
  0xad2a, 0xad2b, 0xad2e, 0xad2f, 0xad30, 0xad31, 0xad32, 0xad33,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xad36, 0xad37,
  0xad39, 0xad3a, 0xad3b, 0xad3d, 0xad3e, 0xad3f, 0xad40, 0xad41,
  0xad42, 0xad43, 0xad46, 0xad48, 0xad4a, 0xad4b, 0xad4c, 0xad4d,
  0xad4e, 0xad4f, 0xad51, 0xad52, 0xad53, 0xad55, 0xad56, 0xad57,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xad59, 0xad5a,
  0xad5b, 0xad5c, 0xad5d, 0xad5e, 0xad5f, 0xad60, 0xad62, 0xad64,
  0xad65, 0xad66, 0xad67, 0xad68, 0xad69, 0xad6a, 0xad6b, 0xad6e,
  0xad6f, 0xad71, 0xad72, 0xad77, 0xad78, 0xad79, 0xad7a, 0xad7e,
  0xad80, 0xad83, 0xad84, 0xad85, 0xad86, 0xad87, 0xad8a, 0xad8b,
  0xad8d, 0xad8e, 0xad8f, 0xad91, 0xad92, 0xad93, 0xad94, 0xad95,
  0xad96, 0xad97, 0xad98, 0xad99, 0xad9a, 0xad9b, 0xad9e, 0xad9f,
  0xada0, 0xada1, 0xada2, 0xada3, 0xada5, 0xada6, 0xada7, 0xada8,
  0xada9, 0xadaa, 0xadab, 0xadac, 0xadad, 0xadae, 0xadaf, 0xadb0,
  0xadb1, 0xadb2, 0xadb3, 0xadb4, 0xadb5, 0xadb6, 0xadb8, 0xadb9,
  0xadba, 0xadbb, 0xadbc, 0xadbd, 0xadbe, 0xadbf, 0xadc2, 0xadc3,
  0xadc5, 0xadc6, 0xadc7, 0xadc9, 0xadca, 0xadcb, 0xadcc, 0xadcd,
  0xadce, 0xadcf, 0xadd2, 0xadd4, 0xadd5, 0xadd6, 0xadd7, 0xadd8,
  0xadd9, 0xadda, 0xaddb, 0xaddd, 0xadde, 0xaddf, 0xade1, 0xade2,
  0xade3, 0xade5, 0xade6, 0xade7, 0xade8, 0xade9, 0xadea, 0xadeb,
  0xadec, 0xaded, 0xadee, 0xadef, 0xadf0, 0xadf1, 0xadf2, 0xadf3,
  0xadf4, 0xadf5, 0xadf6, 0xadf7, 0xadfa, 0xadfb, 0xadfd, 0xadfe,
  0xae02, 0xae03, 0xae04, 0xae05, 0xae06, 0xae07, 0xae0a, 0xae0c,
  0xae0e, 0xae0f, 0xae10, 0xae11, 0xae12, 0xae13, 0xae15, 0xae16,
  0xae17, 0xae18, 0xae19, 0xae1a, 0xae1b, 0xae1c, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xae1d, 0xae1e, 0xae1f, 0xae20,
  0xae21, 0xae22, 0xae23, 0xae24, 0xae25, 0xae26, 0xae27, 0xae28,
  0xae29, 0xae2a, 0xae2b, 0xae2c, 0xae2d, 0xae2e, 0xae2f, 0xae32,
  0xae33, 0xae35, 0xae36, 0xae39, 0xae3b, 0xae3c, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xae3d, 0xae3e, 0xae3f, 0xae42,
  0xae44, 0xae47, 0xae48, 0xae49, 0xae4b, 0xae4f, 0xae51, 0xae52,
  0xae53, 0xae55, 0xae57, 0xae58, 0xae59, 0xae5a, 0xae5b, 0xae5e,
  0xae62, 0xae63, 0xae64, 0xae66, 0xae67, 0xae6a, 0xae6b, 0xae6d,
  0xae6e, 0xae6f, 0xae71, 0xae72, 0xae73, 0xae74, 0xae75, 0xae76,
  0xae77, 0xae7a, 0xae7e, 0xae7f, 0xae80, 0xae81, 0xae82, 0xae83,
  0xae86, 0xae87, 0xae88, 0xae89, 0xae8a, 0xae8b, 0xae8d, 0xae8e,
  0xae8f, 0xae90, 0xae91, 0xae92, 0xae93, 0xae94, 0xae95, 0xae96,
  0xae97, 0xae98, 0xae99, 0xae9a, 0xae9b, 0xae9c, 0xae9d, 0xae9e,
  0xae9f, 0xaea0, 0xaea1, 0xaea2, 0xaea3, 0xaea4, 0xaea5, 0xaea6,
  0xaea7, 0xaea8, 0xaea9, 0xaeaa, 0xaeab, 0xaeac, 0xaead, 0xaeae,
  0xaeaf, 0xaeb0, 0xaeb1, 0xaeb2, 0xaeb3, 0xaeb4, 0xaeb5, 0xaeb6,
  0xaeb7, 0xaeb8, 0xaeb9, 0xaeba, 0xaebb, 0xaebf, 0xaec1, 0xaec2,
  0xaec3, 0xaec5, 0xaec6, 0xaec7, 0xaec8, 0xaec9, 0xaeca, 0xaecb,
  0xaece, 0xaed2, 0xaed3, 0xaed4, 0xaed5, 0xaed6, 0xaed7, 0xaeda,
  0xaedb, 0xaedd, 0xaede, 0xaedf, 0xaee0, 0xaee1, 0xaee2, 0xaee3,
  0xaee4, 0xaee5, 0xaee6, 0xaee7, 0xaee9, 0xaeea, 0xaeec, 0xaeee,
  0xaeef, 0xaef0, 0xaef1, 0xaef2, 0xaef3, 0xaef5, 0xaef6, 0xaef7,
  0xaef9, 0xaefa, 0xaefb, 0xaefd, 0xaefe, 0xaeff, 0xaf00, 0xaf01,
  0xaf02, 0xaf03, 0xaf04, 0xaf05, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xaf06, 0xaf09, 0xaf0a, 0xaf0b, 0xaf0c, 0xaf0e,
  0xaf0f, 0xaf11, 0xaf12, 0xaf13, 0xaf14, 0xaf15, 0xaf16, 0xaf17,
  0xaf18, 0xaf19, 0xaf1a, 0xaf1b, 0xaf1c, 0xaf1d, 0xaf1e, 0xaf1f,
  0xaf20, 0xaf21, 0xaf22, 0xaf23, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xaf24, 0xaf25, 0xaf26, 0xaf27, 0xaf28, 0xaf29,
  0xaf2a, 0xaf2b, 0xaf2e, 0xaf2f, 0xaf31, 0xaf33, 0xaf35, 0xaf36,
  0xaf37, 0xaf38, 0xaf39, 0xaf3a, 0xaf3b, 0xaf3e, 0xaf40, 0xaf44,
  0xaf45, 0xaf46, 0xaf47, 0xaf4a, 0xaf4b, 0xaf4c, 0xaf4d, 0xaf4e,
  0xaf4f, 0xaf51, 0xaf52, 0xaf53, 0xaf54, 0xaf55, 0xaf56, 0xaf57,
  0xaf58, 0xaf59, 0xaf5a, 0xaf5b, 0xaf5e, 0xaf5f, 0xaf60, 0xaf61,
  0xaf62, 0xaf63, 0xaf66, 0xaf67, 0xaf68, 0xaf69, 0xaf6a, 0xaf6b,
  0xaf6c, 0xaf6d, 0xaf6e, 0xaf6f, 0xaf70, 0xaf71, 0xaf72, 0xaf73,
  0xaf74, 0xaf75, 0xaf76, 0xaf77, 0xaf78, 0xaf7a, 0xaf7b, 0xaf7c,
  0xaf7d, 0xaf7e, 0xaf7f, 0xaf81, 0xaf82, 0xaf83, 0xaf85, 0xaf86,
  0xaf87, 0xaf89, 0xaf8a, 0xaf8b, 0xaf8c, 0xaf8d, 0xaf8e, 0xaf8f,
  0xaf92, 0xaf93, 0xaf94, 0xaf96, 0xaf97, 0xaf98, 0xaf99, 0xaf9a,
  0xaf9b, 0xaf9d, 0xaf9e, 0xaf9f, 0xafa0, 0xafa1, 0xafa2, 0xafa3,
  0xafa4, 0xafa5, 0xafa6, 0xafa7, 0xafa8, 0xafa9, 0xafaa, 0xafab,
  0xafac, 0xafad, 0xafae, 0xafaf, 0xafb0, 0xafb1, 0xafb2, 0xafb3,
  0xafb4, 0xafb5, 0xafb6, 0xafb7, 0xafba, 0xafbb, 0xafbd, 0xafbe,
  0xafbf, 0xafc1, 0xafc2, 0xafc3, 0xafc4, 0xafc5, 0xafc6, 0xafca,
  0xafcc, 0xafcf, 0xafd0, 0xafd1, 0xafd2, 0xafd3, 0xafd5, 0xafd6,
  0xafd7, 0xafd8, 0xafd9, 0xafda, 0xafdb, 0xafdd, 0xafde, 0xafdf,
  0xafe0, 0xafe1, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xafe2, 0xafe3, 0xafe4, 0xafe5, 0xafe6, 0xafe7, 0xafea, 0xafeb,
  0xafec, 0xafed, 0xafee, 0xafef, 0xaff2, 0xaff3, 0xaff5, 0xaff6,
  0xaff7, 0xaff9, 0xaffa, 0xaffb, 0xaffc, 0xaffd, 0xaffe, 0xafff,
  0xb002, 0xb003, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xb005, 0xb006, 0xb007, 0xb008, 0xb009, 0xb00a, 0xb00b, 0xb00d,
  0xb00e, 0xb00f, 0xb011, 0xb012, 0xb013, 0xb015, 0xb016, 0xb017,
  0xb018, 0xb019, 0xb01a, 0xb01b, 0xb01e, 0xb01f, 0xb020, 0xb021,
  0xb022, 0xb023, 0xb024, 0xb025, 0xb026, 0xb027, 0xb029, 0xb02a,
  0xb02b, 0xb02c, 0xb02d, 0xb02e, 0xb02f, 0xb030, 0xb031, 0xb032,
  0xb033, 0xb034, 0xb035, 0xb036, 0xb037, 0xb038, 0xb039, 0xb03a,
  0xb03b, 0xb03c, 0xb03d, 0xb03e, 0xb03f, 0xb040, 0xb041, 0xb042,
  0xb043, 0xb046, 0xb047, 0xb049, 0xb04b, 0xb04d, 0xb04f, 0xb050,
  0xb051, 0xb052, 0xb056, 0xb058, 0xb05a, 0xb05b, 0xb05c, 0xb05e,
  0xb05f, 0xb060, 0xb061, 0xb062, 0xb063, 0xb064, 0xb065, 0xb066,
  0xb067, 0xb068, 0xb069, 0xb06a, 0xb06b, 0xb06c, 0xb06d, 0xb06e,
  0xb06f, 0xb070, 0xb071, 0xb072, 0xb073, 0xb074, 0xb075, 0xb076,
  0xb077, 0xb078, 0xb079, 0xb07a, 0xb07b, 0xb07e, 0xb07f, 0xb081,
  0xb082, 0xb083, 0xb085, 0xb086, 0xb087, 0xb088, 0xb089, 0xb08a,
  0xb08b, 0xb08e, 0xb090, 0xb092, 0xb093, 0xb094, 0xb095, 0xb096,
  0xb097, 0xb09b, 0xb09d, 0xb09e, 0xb0a3, 0xb0a4, 0xb0a5, 0xb0a6,
  0xb0a7, 0xb0aa, 0xb0b0, 0xb0b2, 0xb0b6, 0xb0b7, 0xb0b9, 0xb0ba,
  0xb0bb, 0xb0bd, 0xb0be, 0xb0bf, 0xb0c0, 0xb0c1, 0xb0c2, 0xb0c3,
  0xb0c6, 0xb0ca, 0xb0cb, 0xb0cc, 0xb0cd, 0xb0ce, 0xb0cf, 0xb0d2,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xb0d3, 0xb0d5,
  0xb0d6, 0xb0d7, 0xb0d9, 0xb0da, 0xb0db, 0xb0dc, 0xb0dd, 0xb0de,
  0xb0df, 0xb0e1, 0xb0e2, 0xb0e3, 0xb0e4, 0xb0e6, 0xb0e7, 0xb0e8,
  0xb0e9, 0xb0ea, 0xb0eb, 0xb0ec, 0xb0ed, 0xb0ee, 0xb0ef, 0xb0f0,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xb0f1, 0xb0f2,
  0xb0f3, 0xb0f4, 0xb0f5, 0xb0f6, 0xb0f7, 0xb0f8, 0xb0f9, 0xb0fa,
  0xb0fb, 0xb0fc, 0xb0fd, 0xb0fe, 0xb0ff, 0xb100, 0xb101, 0xb102,
  0xb103, 0xb104, 0xb105, 0xb106, 0xb107, 0xb10a, 0xb10d, 0xb10e,
  0xb10f, 0xb111, 0xb114, 0xb115, 0xb116, 0xb117, 0xb11a, 0xb11e,
  0xb11f, 0xb120, 0xb121, 0xb122, 0xb126, 0xb127, 0xb129, 0xb12a,
  0xb12b, 0xb12d, 0xb12e, 0xb12f, 0xb130, 0xb131, 0xb132, 0xb133,
  0xb136, 0xb13a, 0xb13b, 0xb13c, 0xb13d, 0xb13e, 0xb13f, 0xb142,
  0xb143, 0xb145, 0xb146, 0xb147, 0xb149, 0xb14a, 0xb14b, 0xb14c,
  0xb14d, 0xb14e, 0xb14f, 0xb152, 0xb153, 0xb156, 0xb157, 0xb159,
  0xb15a, 0xb15b, 0xb15d, 0xb15e, 0xb15f, 0xb161, 0xb162, 0xb163,
  0xb164, 0xb165, 0xb166, 0xb167, 0xb168, 0xb169, 0xb16a, 0xb16b,
  0xb16c, 0xb16d, 0xb16e, 0xb16f, 0xb170, 0xb171, 0xb172, 0xb173,
  0xb174, 0xb175, 0xb176, 0xb177, 0xb17a, 0xb17b, 0xb17d, 0xb17e,
  0xb17f, 0xb181, 0xb183, 0xb184, 0xb185, 0xb186, 0xb187, 0xb18a,
  0xb18c, 0xb18e, 0xb18f, 0xb190, 0xb191, 0xb195, 0xb196, 0xb197,
  0xb199, 0xb19a, 0xb19b, 0xb19d, 0xb19e, 0xb19f, 0xb1a0, 0xb1a1,
  0xb1a2, 0xb1a3, 0xb1a4, 0xb1a5, 0xb1a6, 0xb1a7, 0xb1a9, 0xb1aa,
  0xb1ab, 0xb1ac, 0xb1ad, 0xb1ae, 0xb1af, 0xb1b0, 0xb1b1, 0xb1b2,
  0xb1b3, 0xb1b4, 0xb1b5, 0xb1b6, 0xb1b7, 0xb1b8, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xb1b9, 0xb1ba, 0xb1bb, 0xb1bc,
  0xb1bd, 0xb1be, 0xb1bf, 0xb1c0, 0xb1c1, 0xb1c2, 0xb1c3, 0xb1c4,
  0xb1c5, 0xb1c6, 0xb1c7, 0xb1c8, 0xb1c9, 0xb1ca, 0xb1cb, 0xb1cd,
  0xb1ce, 0xb1cf, 0xb1d1, 0xb1d2, 0xb1d3, 0xb1d5, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xb1d6, 0xb1d7, 0xb1d8, 0xb1d9,
  0xb1da, 0xb1db, 0xb1de, 0xb1e0, 0xb1e1, 0xb1e2, 0xb1e3, 0xb1e4,
  0xb1e5, 0xb1e6, 0xb1e7, 0xb1ea, 0xb1eb, 0xb1ed, 0xb1ee, 0xb1ef,
  0xb1f1, 0xb1f2, 0xb1f3, 0xb1f4, 0xb1f5, 0xb1f6, 0xb1f7, 0xb1f8,
  0xb1fa, 0xb1fc, 0xb1fe, 0xb1ff, 0xb200, 0xb201, 0xb202, 0xb203,
  0xb206, 0xb207, 0xb209, 0xb20a, 0xb20d, 0xb20e, 0xb20f, 0xb210,
  0xb211, 0xb212, 0xb213, 0xb216, 0xb218, 0xb21a, 0xb21b, 0xb21c,
  0xb21d, 0xb21e, 0xb21f, 0xb221, 0xb222, 0xb223, 0xb224, 0xb225,
  0xb226, 0xb227, 0xb228, 0xb229, 0xb22a, 0xb22b, 0xb22c, 0xb22d,
  0xb22e, 0xb22f, 0xb230, 0xb231, 0xb232, 0xb233, 0xb235, 0xb236,
  0xb237, 0xb238, 0xb239, 0xb23a, 0xb23b, 0xb23d, 0xb23e, 0xb23f,
  0xb240, 0xb241, 0xb242, 0xb243, 0xb244, 0xb245, 0xb246, 0xb247,
  0xb248, 0xb249, 0xb24a, 0xb24b, 0xb24c, 0xb24d, 0xb24e, 0xb24f,
  0xb250, 0xb251, 0xb252, 0xb253, 0xb254, 0xb255, 0xb256, 0xb257,
  0xb259, 0xb25a, 0xb25b, 0xb25d, 0xb25e, 0xb25f, 0xb261, 0xb262,
  0xb263, 0xb264, 0xb265, 0xb266, 0xb267, 0xb26a, 0xb26b, 0xb26c,
  0xb26d, 0xb26e, 0xb26f, 0xb270, 0xb271, 0xb272, 0xb273, 0xb276,
  0xb277, 0xb278, 0xb279, 0xb27a, 0xb27b, 0xb27d, 0xb27e, 0xb27f,
  0xb280, 0xb281, 0xb282, 0xb283, 0xb286, 0xb287, 0xb288, 0xb28a,
  0xb28b, 0xb28c, 0xb28d, 0xb28e, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xb28f, 0xb292, 0xb293, 0xb295, 0xb296, 0xb297,
  0xb29b, 0xb29c, 0xb29d, 0xb29e, 0xb29f, 0xb2a2, 0xb2a4, 0xb2a7,
  0xb2a8, 0xb2a9, 0xb2ab, 0xb2ad, 0xb2ae, 0xb2af, 0xb2b1, 0xb2b2,
  0xb2b3, 0xb2b5, 0xb2b6, 0xb2b7, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xb2b8, 0xb2b9, 0xb2ba, 0xb2bb, 0xb2bc, 0xb2bd,
  0xb2be, 0xb2bf, 0xb2c0, 0xb2c1, 0xb2c2, 0xb2c3, 0xb2c4, 0xb2c5,
  0xb2c6, 0xb2c7, 0xb2ca, 0xb2cb, 0xb2cd, 0xb2ce, 0xb2cf, 0xb2d1,
  0xb2d3, 0xb2d4, 0xb2d5, 0xb2d6, 0xb2d7, 0xb2da, 0xb2dc, 0xb2de,
  0xb2df, 0xb2e0, 0xb2e1, 0xb2e3, 0xb2e7, 0xb2e9, 0xb2ea, 0xb2f0,
  0xb2f1, 0xb2f2, 0xb2f6, 0xb2fc, 0xb2fd, 0xb2fe, 0xb302, 0xb303,
  0xb305, 0xb306, 0xb307, 0xb309, 0xb30a, 0xb30b, 0xb30c, 0xb30d,
  0xb30e, 0xb30f, 0xb312, 0xb316, 0xb317, 0xb318, 0xb319, 0xb31a,
  0xb31b, 0xb31d, 0xb31e, 0xb31f, 0xb320, 0xb321, 0xb322, 0xb323,
  0xb324, 0xb325, 0xb326, 0xb327, 0xb328, 0xb329, 0xb32a, 0xb32b,
  0xb32c, 0xb32d, 0xb32e, 0xb32f, 0xb330, 0xb331, 0xb332, 0xb333,
  0xb334, 0xb335, 0xb336, 0xb337, 0xb338, 0xb339, 0xb33a, 0xb33b,
  0xb33c, 0xb33d, 0xb33e, 0xb33f, 0xb340, 0xb341, 0xb342, 0xb343,
  0xb344, 0xb345, 0xb346, 0xb347, 0xb348, 0xb349, 0xb34a, 0xb34b,
  0xb34c, 0xb34d, 0xb34e, 0xb34f, 0xb350, 0xb351, 0xb352, 0xb353,
  0xb357, 0xb359, 0xb35a, 0xb35d, 0xb360, 0xb361, 0xb362, 0xb363,
  0xb366, 0xb368, 0xb36a, 0xb36c, 0xb36d, 0xb36f, 0xb372, 0xb373,
  0xb375, 0xb376, 0xb377, 0xb379, 0xb37a, 0xb37b, 0xb37c, 0xb37d,
  0xb37e, 0xb37f, 0xb382, 0xb386, 0xb387, 0xb388, 0xb389, 0xb38a,
  0xb38b, 0xb38d, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xb38e, 0xb38f, 0xb391, 0xb392, 0xb393, 0xb395, 0xb396, 0xb397,
  0xb398, 0xb399, 0xb39a, 0xb39b, 0xb39c, 0xb39d, 0xb39e, 0xb39f,
  0xb3a2, 0xb3a3, 0xb3a4, 0xb3a5, 0xb3a6, 0xb3a7, 0xb3a9, 0xb3aa,
  0xb3ab, 0xb3ad, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xb3ae, 0xb3af, 0xb3b0, 0xb3b1, 0xb3b2, 0xb3b3, 0xb3b4, 0xb3b5,
  0xb3b6, 0xb3b7, 0xb3b8, 0xb3b9, 0xb3ba, 0xb3bb, 0xb3bc, 0xb3bd,
  0xb3be, 0xb3bf, 0xb3c0, 0xb3c1, 0xb3c2, 0xb3c3, 0xb3c6, 0xb3c7,
  0xb3c9, 0xb3ca, 0xb3cd, 0xb3cf, 0xb3d1, 0xb3d2, 0xb3d3, 0xb3d6,
  0xb3d8, 0xb3da, 0xb3dc, 0xb3de, 0xb3df, 0xb3e1, 0xb3e2, 0xb3e3,
  0xb3e5, 0xb3e6, 0xb3e7, 0xb3e9, 0xb3ea, 0xb3eb, 0xb3ec, 0xb3ed,
  0xb3ee, 0xb3ef, 0xb3f0, 0xb3f1, 0xb3f2, 0xb3f3, 0xb3f4, 0xb3f5,
  0xb3f6, 0xb3f7, 0xb3f8, 0xb3f9, 0xb3fa, 0xb3fb, 0xb3fd, 0xb3fe,
  0xb3ff, 0xb400, 0xb401, 0xb402, 0xb403, 0xb404, 0xb405, 0xb406,
  0xb407, 0xb408, 0xb409, 0xb40a, 0xb40b, 0xb40c, 0xb40d, 0xb40e,
  0xb40f, 0xb411, 0xb412, 0xb413, 0xb414, 0xb415, 0xb416, 0xb417,
  0xb419, 0xb41a, 0xb41b, 0xb41d, 0xb41e, 0xb41f, 0xb421, 0xb422,
  0xb423, 0xb424, 0xb425, 0xb426, 0xb427, 0xb42a, 0xb42c, 0xb42d,
  0xb42e, 0xb42f, 0xb430, 0xb431, 0xb432, 0xb433, 0xb435, 0xb436,
  0xb437, 0xb438, 0xb439, 0xb43a, 0xb43b, 0xb43c, 0xb43d, 0xb43e,
  0xb43f, 0xb440, 0xb441, 0xb442, 0xb443, 0xb444, 0xb445, 0xb446,
  0xb447, 0xb448, 0xb449, 0xb44a, 0xb44b, 0xb44c, 0xb44d, 0xb44e,
  0xb44f, 0xb452, 0xb453, 0xb455, 0xb456, 0xb457, 0xb459, 0xb45a,
  0xb45b, 0xb45c, 0xb45d, 0xb45e, 0xb45f, 0xb462, 0xb464, 0xb466,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xb467, 0xb468,
  0xb469, 0xb46a, 0xb46b, 0xb46d, 0xb46e, 0xb46f, 0xb470, 0xb471,
  0xb472, 0xb473, 0xb474, 0xb475, 0xb476, 0xb477, 0xb478, 0xb479,
  0xb47a, 0xb47b, 0xb47c, 0xb47d, 0xb47e, 0xb47f, 0xb481, 0xb482,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xb483, 0xb484,
  0xb485, 0xb486, 0xb487, 0xb489, 0xb48a, 0xb48b, 0xb48c, 0xb48d,
  0xb48e, 0xb48f, 0xb490, 0xb491, 0xb492, 0xb493, 0xb494, 0xb495,
  0xb496, 0xb497, 0xb498, 0xb499, 0xb49a, 0xb49b, 0xb49c, 0xb49e,
  0xb49f, 0xb4a0, 0xb4a1, 0xb4a2, 0xb4a3, 0xb4a5, 0xb4a6, 0xb4a7,
  0xb4a9, 0xb4aa, 0xb4ab, 0xb4ad, 0xb4ae, 0xb4af, 0xb4b0, 0xb4b1,
  0xb4b2, 0xb4b3, 0xb4b4, 0xb4b6, 0xb4b8, 0xb4ba, 0xb4bb, 0xb4bc,
  0xb4bd, 0xb4be, 0xb4bf, 0xb4c1, 0xb4c2, 0xb4c3, 0xb4c5, 0xb4c6,
  0xb4c7, 0xb4c9, 0xb4ca, 0xb4cb, 0xb4cc, 0xb4cd, 0xb4ce, 0xb4cf,
  0xb4d1, 0xb4d2, 0xb4d3, 0xb4d4, 0xb4d6, 0xb4d7, 0xb4d8, 0xb4d9,
  0xb4da, 0xb4db, 0xb4de, 0xb4df, 0xb4e1, 0xb4e2, 0xb4e5, 0xb4e7,
  0xb4e8, 0xb4e9, 0xb4ea, 0xb4eb, 0xb4ee, 0xb4f0, 0xb4f2, 0xb4f3,
  0xb4f4, 0xb4f5, 0xb4f6, 0xb4f7, 0xb4f9, 0xb4fa, 0xb4fb, 0xb4fc,
  0xb4fd, 0xb4fe, 0xb4ff, 0xb500, 0xb501, 0xb502, 0xb503, 0xb504,
  0xb505, 0xb506, 0xb507, 0xb508, 0xb509, 0xb50a, 0xb50b, 0xb50c,
  0xb50d, 0xb50e, 0xb50f, 0xb510, 0xb511, 0xb512, 0xb513, 0xb516,
  0xb517, 0xb519, 0xb51a, 0xb51d, 0xb51e, 0xb51f, 0xb520, 0xb521,
  0xb522, 0xb523, 0xb526, 0xb52b, 0xb52c, 0xb52d, 0xb52e, 0xb52f,
  0xb532, 0xb533, 0xb535, 0xb536, 0xb537, 0xb539, 0xb53a, 0xb53b,
  0xb53c, 0xb53d, 0xb53e, 0xb53f, 0xb542, 0xb546, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xb547, 0xb548, 0xb549, 0xb54a,
  0xb54e, 0xb54f, 0xb551, 0xb552, 0xb553, 0xb555, 0xb556, 0xb557,
  0xb558, 0xb559, 0xb55a, 0xb55b, 0xb55e, 0xb562, 0xb563, 0xb564,
  0xb565, 0xb566, 0xb567, 0xb568, 0xb569, 0xb56a, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xb56b, 0xb56c, 0xb56d, 0xb56e,
  0xb56f, 0xb570, 0xb571, 0xb572, 0xb573, 0xb574, 0xb575, 0xb576,
  0xb577, 0xb578, 0xb579, 0xb57a, 0xb57b, 0xb57c, 0xb57d, 0xb57e,
  0xb57f, 0xb580, 0xb581, 0xb582, 0xb583, 0xb584, 0xb585, 0xb586,
  0xb587, 0xb588, 0xb589, 0xb58a, 0xb58b, 0xb58c, 0xb58d, 0xb58e,
  0xb58f, 0xb590, 0xb591, 0xb592, 0xb593, 0xb594, 0xb595, 0xb596,
  0xb597, 0xb598, 0xb599, 0xb59a, 0xb59b, 0xb59c, 0xb59d, 0xb59e,
  0xb59f, 0xb5a2, 0xb5a3, 0xb5a5, 0xb5a6, 0xb5a7, 0xb5a9, 0xb5ac,
  0xb5ad, 0xb5ae, 0xb5af, 0xb5b2, 0xb5b6, 0xb5b7, 0xb5b8, 0xb5b9,
  0xb5ba, 0xb5be, 0xb5bf, 0xb5c1, 0xb5c2, 0xb5c3, 0xb5c5, 0xb5c6,
  0xb5c7, 0xb5c8, 0xb5c9, 0xb5ca, 0xb5cb, 0xb5ce, 0xb5d2, 0xb5d3,
  0xb5d4, 0xb5d5, 0xb5d6, 0xb5d7, 0xb5d9, 0xb5da, 0xb5db, 0xb5dc,
  0xb5dd, 0xb5de, 0xb5df, 0xb5e0, 0xb5e1, 0xb5e2, 0xb5e3, 0xb5e4,
  0xb5e5, 0xb5e6, 0xb5e7, 0xb5e8, 0xb5e9, 0xb5ea, 0xb5eb, 0xb5ed,
  0xb5ee, 0xb5ef, 0xb5f0, 0xb5f1, 0xb5f2, 0xb5f3, 0xb5f4, 0xb5f5,
  0xb5f6, 0xb5f7, 0xb5f8, 0xb5f9, 0xb5fa, 0xb5fb, 0xb5fc, 0xb5fd,
  0xb5fe, 0xb5ff, 0xb600, 0xb601, 0xb602, 0xb603, 0xb604, 0xb605,
  0xb606, 0xb607, 0xb608, 0xb609, 0xb60a, 0xb60b, 0xb60c, 0xb60d,
  0xb60e, 0xb60f, 0xb612, 0xb613, 0xb615, 0xb616, 0xb617, 0xb619,
  0xb61a, 0xb61b, 0xb61c, 0xb61d, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xb61e, 0xb61f, 0xb620, 0xb621, 0xb622, 0xb623,
  0xb624, 0xb626, 0xb627, 0xb628, 0xb629, 0xb62a, 0xb62b, 0xb62d,
  0xb62e, 0xb62f, 0xb630, 0xb631, 0xb632, 0xb633, 0xb635, 0xb636,
  0xb637, 0xb638, 0xb639, 0xb63a, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xb63b, 0xb63c, 0xb63d, 0xb63e, 0xb63f, 0xb640,
  0xb641, 0xb642, 0xb643, 0xb644, 0xb645, 0xb646, 0xb647, 0xb649,
  0xb64a, 0xb64b, 0xb64c, 0xb64d, 0xb64e, 0xb64f, 0xb650, 0xb651,
  0xb652, 0xb653, 0xb654, 0xb655, 0xb656, 0xb657, 0xb658, 0xb659,
  0xb65a, 0xb65b, 0xb65c, 0xb65d, 0xb65e, 0xb65f, 0xb660, 0xb661,
  0xb662, 0xb663, 0xb665, 0xb666, 0xb667, 0xb669, 0xb66a, 0xb66b,
  0xb66c, 0xb66d, 0xb66e, 0xb66f, 0xb670, 0xb671, 0xb672, 0xb673,
  0xb674, 0xb675, 0xb676, 0xb677, 0xb678, 0xb679, 0xb67a, 0xb67b,
  0xb67c, 0xb67d, 0xb67e, 0xb67f, 0xb680, 0xb681, 0xb682, 0xb683,
  0xb684, 0xb685, 0xb686, 0xb687, 0xb688, 0xb689, 0xb68a, 0xb68b,
  0xb68c, 0xb68d, 0xb68e, 0xb68f, 0xb690, 0xb691, 0xb692, 0xb693,
  0xb694, 0xb695, 0xb696, 0xb697, 0xb698, 0xb699, 0xb69a, 0xb69b,
  0xb69e, 0xb69f, 0xb6a1, 0xb6a2, 0xb6a3, 0xb6a5, 0xb6a6, 0xb6a7,
  0xb6a8, 0xb6a9, 0xb6aa, 0xb6ad, 0xb6ae, 0xb6af, 0xb6b0, 0xb6b2,
  0xb6b3, 0xb6b4, 0xb6b5, 0xb6b6, 0xb6b7, 0xb6b8, 0xb6b9, 0xb6ba,
  0xb6bb, 0xb6bc, 0xb6bd, 0xb6be, 0xb6bf, 0xb6c0, 0xb6c1, 0xb6c2,
  0xb6c3, 0xb6c4, 0xb6c5, 0xb6c6, 0xb6c7, 0xb6c8, 0xb6c9, 0xb6ca,
  0xb6cb, 0xb6cc, 0xb6cd, 0xb6ce, 0xb6cf, 0xb6d0, 0xb6d1, 0xb6d2,
  0xb6d3, 0xb6d5, 0xb6d6, 0xb6d7, 0xb6d8, 0xb6d9, 0xb6da, 0xb6db,
  0xb6dc, 0xb6dd, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xb6de, 0xb6df, 0xb6e0, 0xb6e1, 0xb6e2, 0xb6e3, 0xb6e4, 0xb6e5,
  0xb6e6, 0xb6e7, 0xb6e8, 0xb6e9, 0xb6ea, 0xb6eb, 0xb6ec, 0xb6ed,
  0xb6ee, 0xb6ef, 0xb6f1, 0xb6f2, 0xb6f3, 0xb6f5, 0xb6f6, 0xb6f7,
  0xb6f9, 0xb6fa, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xb6fb, 0xb6fc, 0xb6fd, 0xb6fe, 0xb6ff, 0xb702, 0xb703, 0xb704,
  0xb706, 0xb707, 0xb708, 0xb709, 0xb70a, 0xb70b, 0xb70c, 0xb70d,
  0xb70e, 0xb70f, 0xb710, 0xb711, 0xb712, 0xb713, 0xb714, 0xb715,
  0xb716, 0xb717, 0xb718, 0xb719, 0xb71a, 0xb71b, 0xb71c, 0xb71d,
  0xb71e, 0xb71f, 0xb720, 0xb721, 0xb722, 0xb723, 0xb724, 0xb725,
  0xb726, 0xb727, 0xb72a, 0xb72b, 0xb72d, 0xb72e, 0xb731, 0xb732,
  0xb733, 0xb734, 0xb735, 0xb736, 0xb737, 0xb73a, 0xb73c, 0xb73d,
  0xb73e, 0xb73f, 0xb740, 0xb741, 0xb742, 0xb743, 0xb745, 0xb746,
  0xb747, 0xb749, 0xb74a, 0xb74b, 0xb74d, 0xb74e, 0xb74f, 0xb750,
  0xb751, 0xb752, 0xb753, 0xb756, 0xb757, 0xb758, 0xb759, 0xb75a,
  0xb75b, 0xb75c, 0xb75d, 0xb75e, 0xb75f, 0xb761, 0xb762, 0xb763,
  0xb765, 0xb766, 0xb767, 0xb769, 0xb76a, 0xb76b, 0xb76c, 0xb76d,
  0xb76e, 0xb76f, 0xb772, 0xb774, 0xb776, 0xb777, 0xb778, 0xb779,
  0xb77a, 0xb77b, 0xb77e, 0xb77f, 0xb781, 0xb782, 0xb783, 0xb785,
  0xb786, 0xb787, 0xb788, 0xb789, 0xb78a, 0xb78b, 0xb78e, 0xb793,
  0xb794, 0xb795, 0xb79a, 0xb79b, 0xb79d, 0xb79e, 0xb79f, 0xb7a1,
  0xb7a2, 0xb7a3, 0xb7a4, 0xb7a5, 0xb7a6, 0xb7a7, 0xb7aa, 0xb7ae,
  0xb7af, 0xb7b0, 0xb7b1, 0xb7b2, 0xb7b3, 0xb7b6, 0xb7b7, 0xb7b9,
  0xb7ba, 0xb7bb, 0xb7bc, 0xb7bd, 0xb7be, 0xb7bf, 0xb7c0, 0xb7c1,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xb7c2, 0xb7c3,
  0xb7c4, 0xb7c5, 0xb7c6, 0xb7c8, 0xb7ca, 0xb7cb, 0xb7cc, 0xb7cd,
  0xb7ce, 0xb7cf, 0xb7d0, 0xb7d1, 0xb7d2, 0xb7d3, 0xb7d4, 0xb7d5,
  0xb7d6, 0xb7d7, 0xb7d8, 0xb7d9, 0xb7da, 0xb7db, 0xb7dc, 0xb7dd,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xb7de, 0xb7df,
  0xb7e0, 0xb7e1, 0xb7e2, 0xb7e3, 0xb7e4, 0xb7e5, 0xb7e6, 0xb7e7,
  0xb7e8, 0xb7e9, 0xb7ea, 0xb7eb, 0xb7ee, 0xb7ef, 0xb7f1, 0xb7f2,
  0xb7f3, 0xb7f5, 0xb7f6, 0xb7f7, 0xb7f8, 0xb7f9, 0xb7fa, 0xb7fb,
  0xb7fe, 0xb802, 0xb803, 0xb804, 0xb805, 0xb806, 0xb80a, 0xb80b,
  0xb80d, 0xb80e, 0xb80f, 0xb811, 0xb812, 0xb813, 0xb814, 0xb815,
  0xb816, 0xb817, 0xb81a, 0xb81c, 0xb81e, 0xb81f, 0xb820, 0xb821,
  0xb822, 0xb823, 0xb826, 0xb827, 0xb829, 0xb82a, 0xb82b, 0xb82d,
  0xb82e, 0xb82f, 0xb830, 0xb831, 0xb832, 0xb833, 0xb836, 0xb83a,
  0xb83b, 0xb83c, 0xb83d, 0xb83e, 0xb83f, 0xb841, 0xb842, 0xb843,
  0xb845, 0xb846, 0xb847, 0xb848, 0xb849, 0xb84a, 0xb84b, 0xb84c,
  0xb84d, 0xb84e, 0xb84f, 0xb850, 0xb852, 0xb854, 0xb855, 0xb856,
  0xb857, 0xb858, 0xb859, 0xb85a, 0xb85b, 0xb85e, 0xb85f, 0xb861,
  0xb862, 0xb863, 0xb865, 0xb866, 0xb867, 0xb868, 0xb869, 0xb86a,
  0xb86b, 0xb86e, 0xb870, 0xb872, 0xb873, 0xb874, 0xb875, 0xb876,
  0xb877, 0xb879, 0xb87a, 0xb87b, 0xb87d, 0xb87e, 0xb87f, 0xb880,
  0xb881, 0xb882, 0xb883, 0xb884, 0xb885, 0xb886, 0xb887, 0xb888,
  0xb889, 0xb88a, 0xb88b, 0xb88c, 0xb88e, 0xb88f, 0xb890, 0xb891,
  0xb892, 0xb893, 0xb894, 0xb895, 0xb896, 0xb897, 0xb898, 0xb899,
  0xb89a, 0xb89b, 0xb89c, 0xb89d, 0xb89e, 0xb89f, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xb8a0, 0xb8a1, 0xb8a2, 0xb8a3,
  0xb8a4, 0xb8a5, 0xb8a6, 0xb8a7, 0xb8a9, 0xb8aa, 0xb8ab, 0xb8ac,
  0xb8ad, 0xb8ae, 0xb8af, 0xb8b1, 0xb8b2, 0xb8b3, 0xb8b5, 0xb8b6,
  0xb8b7, 0xb8b9, 0xb8ba, 0xb8bb, 0xb8bc, 0xb8bd, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xb8be, 0xb8bf, 0xb8c2, 0xb8c4,
  0xb8c6, 0xb8c7, 0xb8c8, 0xb8c9, 0xb8ca, 0xb8cb, 0xb8cd, 0xb8ce,
  0xb8cf, 0xb8d1, 0xb8d2, 0xb8d3, 0xb8d5, 0xb8d6, 0xb8d7, 0xb8d8,
  0xb8d9, 0xb8da, 0xb8db, 0xb8dc, 0xb8de, 0xb8e0, 0xb8e2, 0xb8e3,
  0xb8e4, 0xb8e5, 0xb8e6, 0xb8e7, 0xb8ea, 0xb8eb, 0xb8ed, 0xb8ee,
  0xb8ef, 0xb8f1, 0xb8f2, 0xb8f3, 0xb8f4, 0xb8f5, 0xb8f6, 0xb8f7,
  0xb8fa, 0xb8fc, 0xb8fe, 0xb8ff, 0xb900, 0xb901, 0xb902, 0xb903,
  0xb905, 0xb906, 0xb907, 0xb908, 0xb909, 0xb90a, 0xb90b, 0xb90c,
  0xb90d, 0xb90e, 0xb90f, 0xb910, 0xb911, 0xb912, 0xb913, 0xb914,
  0xb915, 0xb916, 0xb917, 0xb919, 0xb91a, 0xb91b, 0xb91c, 0xb91d,
  0xb91e, 0xb91f, 0xb921, 0xb922, 0xb923, 0xb924, 0xb925, 0xb926,
  0xb927, 0xb928, 0xb929, 0xb92a, 0xb92b, 0xb92c, 0xb92d, 0xb92e,
  0xb92f, 0xb930, 0xb931, 0xb932, 0xb933, 0xb934, 0xb935, 0xb936,
  0xb937, 0xb938, 0xb939, 0xb93a, 0xb93b, 0xb93e, 0xb93f, 0xb941,
  0xb942, 0xb943, 0xb945, 0xb946, 0xb947, 0xb948, 0xb949, 0xb94a,
  0xb94b, 0xb94d, 0xb94e, 0xb950, 0xb952, 0xb953, 0xb954, 0xb955,
  0xb956, 0xb957, 0xb95a, 0xb95b, 0xb95d, 0xb95e, 0xb95f, 0xb961,
  0xb962, 0xb963, 0xb964, 0xb965, 0xb966, 0xb967, 0xb96a, 0xb96c,
  0xb96e, 0xb96f, 0xb970, 0xb971, 0xb972, 0xb973, 0xb976, 0xb977,
  0xb979, 0xb97a, 0xb97b, 0xb97d, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xb97e, 0xb97f, 0xb980, 0xb981, 0xb982, 0xb983,
  0xb986, 0xb988, 0xb98b, 0xb98c, 0xb98f, 0xb990, 0xb991, 0xb992,
  0xb993, 0xb994, 0xb995, 0xb996, 0xb997, 0xb998, 0xb999, 0xb99a,
  0xb99b, 0xb99c, 0xb99d, 0xb99e, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xb99f, 0xb9a0, 0xb9a1, 0xb9a2, 0xb9a3, 0xb9a4,
  0xb9a5, 0xb9a6, 0xb9a7, 0xb9a8, 0xb9a9, 0xb9aa, 0xb9ab, 0xb9ae,
  0xb9af, 0xb9b1, 0xb9b2, 0xb9b3, 0xb9b5, 0xb9b6, 0xb9b7, 0xb9b8,
  0xb9b9, 0xb9ba, 0xb9bb, 0xb9be, 0xb9c0, 0xb9c2, 0xb9c3, 0xb9c4,
  0xb9c5, 0xb9c6, 0xb9c7, 0xb9ca, 0xb9cb, 0xb9cd, 0xb9d3, 0xb9d4,
  0xb9d5, 0xb9d6, 0xb9d7, 0xb9da, 0xb9dc, 0xb9df, 0xb9e0, 0xb9e2,
  0xb9e6, 0xb9e7, 0xb9e9, 0xb9ea, 0xb9eb, 0xb9ed, 0xb9ee, 0xb9ef,
  0xb9f0, 0xb9f1, 0xb9f2, 0xb9f3, 0xb9f6, 0xb9fb, 0xb9fc, 0xb9fd,
  0xb9fe, 0xb9ff, 0xba02, 0xba03, 0xba04, 0xba05, 0xba06, 0xba07,
  0xba09, 0xba0a, 0xba0b, 0xba0c, 0xba0d, 0xba0e, 0xba0f, 0xba10,
  0xba11, 0xba12, 0xba13, 0xba14, 0xba16, 0xba17, 0xba18, 0xba19,
  0xba1a, 0xba1b, 0xba1c, 0xba1d, 0xba1e, 0xba1f, 0xba20, 0xba21,
  0xba22, 0xba23, 0xba24, 0xba25, 0xba26, 0xba27, 0xba28, 0xba29,
  0xba2a, 0xba2b, 0xba2c, 0xba2d, 0xba2e, 0xba2f, 0xba30, 0xba31,
  0xba32, 0xba33, 0xba34, 0xba35, 0xba36, 0xba37, 0xba3a, 0xba3b,
  0xba3d, 0xba3e, 0xba3f, 0xba41, 0xba43, 0xba44, 0xba45, 0xba46,
  0xba47, 0xba4a, 0xba4c, 0xba4f, 0xba50, 0xba51, 0xba52, 0xba56,
  0xba57, 0xba59, 0xba5a, 0xba5b, 0xba5d, 0xba5e, 0xba5f, 0xba60,
  0xba61, 0xba62, 0xba63, 0xba66, 0xba6a, 0xba6b, 0xba6c, 0xba6d,
  0xba6e, 0xba6f, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xba72, 0xba73, 0xba75, 0xba76, 0xba77, 0xba79, 0xba7a, 0xba7b,
  0xba7c, 0xba7d, 0xba7e, 0xba7f, 0xba80, 0xba81, 0xba82, 0xba86,
  0xba88, 0xba89, 0xba8a, 0xba8b, 0xba8d, 0xba8e, 0xba8f, 0xba90,
  0xba91, 0xba92, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xba93, 0xba94, 0xba95, 0xba96, 0xba97, 0xba98, 0xba99, 0xba9a,
  0xba9b, 0xba9c, 0xba9d, 0xba9e, 0xba9f, 0xbaa0, 0xbaa1, 0xbaa2,
  0xbaa3, 0xbaa4, 0xbaa5, 0xbaa6, 0xbaa7, 0xbaaa, 0xbaad, 0xbaae,
  0xbaaf, 0xbab1, 0xbab3, 0xbab4, 0xbab5, 0xbab6, 0xbab7, 0xbaba,
  0xbabc, 0xbabe, 0xbabf, 0xbac0, 0xbac1, 0xbac2, 0xbac3, 0xbac5,
  0xbac6, 0xbac7, 0xbac9, 0xbaca, 0xbacb, 0xbacc, 0xbacd, 0xbace,
  0xbacf, 0xbad0, 0xbad1, 0xbad2, 0xbad3, 0xbad4, 0xbad5, 0xbad6,
  0xbad7, 0xbada, 0xbadb, 0xbadc, 0xbadd, 0xbade, 0xbadf, 0xbae0,
  0xbae1, 0xbae2, 0xbae3, 0xbae4, 0xbae5, 0xbae6, 0xbae7, 0xbae8,
  0xbae9, 0xbaea, 0xbaeb, 0xbaec, 0xbaed, 0xbaee, 0xbaef, 0xbaf0,
  0xbaf1, 0xbaf2, 0xbaf3, 0xbaf4, 0xbaf5, 0xbaf6, 0xbaf7, 0xbaf8,
  0xbaf9, 0xbafa, 0xbafb, 0xbafd, 0xbafe, 0xbaff, 0xbb01, 0xbb02,
  0xbb03, 0xbb05, 0xbb06, 0xbb07, 0xbb08, 0xbb09, 0xbb0a, 0xbb0b,
  0xbb0c, 0xbb0e, 0xbb10, 0xbb12, 0xbb13, 0xbb14, 0xbb15, 0xbb16,
  0xbb17, 0xbb19, 0xbb1a, 0xbb1b, 0xbb1d, 0xbb1e, 0xbb1f, 0xbb21,
  0xbb22, 0xbb23, 0xbb24, 0xbb25, 0xbb26, 0xbb27, 0xbb28, 0xbb2a,
  0xbb2c, 0xbb2d, 0xbb2e, 0xbb2f, 0xbb30, 0xbb31, 0xbb32, 0xbb33,
  0xbb37, 0xbb39, 0xbb3a, 0xbb3f, 0xbb40, 0xbb41, 0xbb42, 0xbb43,
  0xbb46, 0xbb48, 0xbb4a, 0xbb4b, 0xbb4c, 0xbb4e, 0xbb51, 0xbb52,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xbb53, 0xbb55,
  0xbb56, 0xbb57, 0xbb59, 0xbb5a, 0xbb5b, 0xbb5c, 0xbb5d, 0x3000,
  0x3000, 0xbb60, 0x3000, 0xbb64, 0xbb65, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xbb6d, 0x3000, 0x3000, 0xbb70, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xbb74, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xbb7c, 0xbb7d, 0x3000, 0xbb7f, 0xbb80, 0xbb81, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xbb89, 0xbb8a, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xbb98, 0xbb99, 0x3000, 0xbb9b, 0xbb9c, 0xbb9d, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xbba5, 0x3000, 0x3000,
  0x3000, 0x3000, 0xbbab, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xbbb5, 0x3000, 0x3000, 0xbbb9, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xbbc1, 0x3000, 0xbbc3, 0x3000,
  0x3000, 0xbbc7, 0xbbc9, 0x3000, 0x3000, 0xbbcc, 0x3000, 0x3000,
  0xbbcf, 0xbbd1, 0x3000, 0x3000, 0xbbd5, 0x3000, 0x3000, 0x3000,
  0xbbd9, 0x3000, 0x3000, 0xbbdc, 0xbbdd, 0x3000, 0x3000, 0xbbe0,
  0x3000, 0x3000, 0x3000, 0xbbe4, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xbbec, 0xbbed, 0x3000, 0xbbef, 0x3000,
  0xbbf1, 0xbbf2, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xbc01, 0x3000, 0xbc04, 0x3000, 0x3000,
  0x3000, 0x3000, 0xbc0e, 0xbc10, 0x3000, 0x3000, 0x3000, 0x3000,
  0xbc20, 0x3000, 0x3000, 0xbc23, 0x3000, 0xbc28, 0x3000, 0xbc2b,
  0xbc2c, 0x3000, 0xbc2f, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xbc37, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xbc46, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xbc54, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xbc5c, 0x3000,
  0xbc5f, 0x3000, 0xbc61, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xbc67, 0xbc68, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xbc70, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xbc77, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xbc7d, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xbc86, 0x3000, 0x3000,
  0x3000, 0xbc8d, 0x3000, 0xbc90, 0x3000, 0x3000, 0x3000, 0x3000,
  0xbc98, 0x3000, 0xbc9c, 0xbc9d, 0x3000, 0x3000, 0xbca2, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xbcb8, 0xbcb9, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xbcc3, 0x3000, 0x3000, 0x3000,
  0xbcc8, 0x3000, 0x3000, 0x3000, 0xbccc, 0x3000, 0xbcd2, 0xbcd3,
  0xbcd4, 0x3000, 0x3000, 0xbcd9, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xbce0, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xbce8, 0xbce9, 0x3000, 0xbceb, 0x3000, 0xbced, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xbcf7, 0x3000, 0x3000,
  0xbcfb, 0xbcfd, 0x3000, 0xbcff, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xbd0a, 0xbd0b, 0x3000, 0xbd0d, 0x3000, 0xbd0f,
  0xbd11, 0x3000, 0x3000, 0x3000, 0x3000, 0xbd17, 0xbd18, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xbd20, 0xbd21, 0x3000, 0xbd23,
  0xbd25, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xbd30, 0x3000, 0x3000, 0xbd33, 0xbd34, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xbd4a,
  0x3000, 0x3000, 0x3000, 0xbd4f, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xbd5b, 0xbd5c, 0xbd5d, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xbd65, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xbd6c, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xbd74, 0xbd75, 0x3000, 0xbd77, 0x3000,
  0xbd79, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xbd82,
  0x3000, 0x3000, 0x3000, 0xbd8b, 0x3000, 0x3000, 0xbd8e, 0x3000,
  0x3000, 0x3000, 0xbd96, 0xbd97, 0xbd98, 0xbd9b, 0xbd9d, 0x3000,
  0x3000, 0xbda0, 0x3000, 0x3000, 0xbda3, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xbdac, 0xbdad, 0x3000, 0xbdaf,
  0xbdb1, 0x3000, 0x3000, 0xbdb4, 0x3000, 0x3000, 0x3000, 0xbdb9,
  0x3000, 0x3000, 0xbdbc, 0x3000, 0x3000, 0xbdbf, 0xbdc0, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xbdc8, 0xbdc9,
  0x3000, 0xbdcb, 0xbdcc, 0xbdcd, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xbddb, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xbde4, 0xbde5,
  0x3000, 0xbde7, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xbdf1, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xbdf7, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xbe01, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xbe13, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xbe15, 0x3000, 0xbe17, 0xbe18, 0x3000, 0x3000, 0xbe1b, 0x3000,
  0x3000, 0xbe21, 0x3000, 0xbe23, 0x3000, 0xbe25, 0x3000, 0xbe27,
  0xbe28, 0xbe29, 0x3000, 0x3000, 0xbe2c, 0x3000, 0x3000, 0x3000,
  0xbe30, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xbe38, 0xbe39, 0x3000, 0xbe3b, 0x3000, 0xbe3d, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xbe4b, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xbe58, 0xbe5c, 0xbe5d, 0x3000, 0xbe5f, 0xbe62, 0x3000, 0x3000,
  0x3000, 0xbe67, 0xbe69, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xbe76, 0x3000, 0x3000, 0xbe79, 0x3000, 0xbe7e, 0x3000,
  0x3000, 0x3000, 0xbe83, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xbe9c, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xbeb4, 0x3000, 0x3000, 0x3000, 0xbeb8, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xbee1,
  0x3000, 0xbee6, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xbeed,
  0x3000, 0x3000, 0xbef0, 0x3000, 0x3000, 0xbef3, 0xbef4, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xbefc, 0xbefd, 0x3000, 0xbeff,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xbf0c, 0x3000, 0x3000, 0xbf0f, 0xbf10, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xbf1f,
  0x3000, 0xbf21, 0x3000, 0x3000, 0xbf24, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xbf37,
  0xbf38, 0xbf39, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xbf47, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xbf53, 0x3000, 0x3000,
  0x3000, 0x3000, 0xbf59, 0x3000, 0xbf5b, 0xbf5c, 0x3000, 0x3000,
  0x3000, 0xbf60, 0x3000, 0x3000, 0xbf63, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xbf78, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xbf7f, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xbfa4, 0xbfa5, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xbfac, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xbfc0, 0xbfc1, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xbfd3, 0xbfd5, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xbfdd, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xbfe5, 0x3000, 0x3000, 0xbfe8, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xbfef, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc004,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xc020, 0xc021, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc043, 0xc044, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xc059, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xc05f, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc06b, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc074, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xc097, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc0a4, 0xc0a6, 0xc0a7,
  0x3000, 0x3000, 0x3000, 0xc0ab, 0xc0ae, 0x3000, 0x3000, 0xc0b7,
  0xc0b8, 0x3000, 0xc0ba, 0xc0bb, 0x3000, 0xc0c2, 0xc0c3, 0xc0c4,
  0xc0c6, 0xc0c7, 0x3000, 0xc0cb, 0x3000, 0x3000, 0xc0cf, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xc0e3, 0x3000, 0x3000, 0x3000,
  0x3000, 0xc0eb, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xc0f8, 0x3000, 0xc0fb, 0x3000, 0x3000, 0xc0fe,
  0xc0ff, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xc125, 0xc128, 0x3000, 0xc12a, 0x3000, 0x3000,
  0x3000, 0x3000, 0xc134, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xc13f, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xc14e, 0x3000, 0x3000, 0xc151, 0xc152, 0x3000,
  0x3000, 0xc157, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xc15b, 0x3000, 0x3000, 0xc15f, 0xc160, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xc171, 0x3000, 0xc173, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xc180, 0xc181, 0x3000, 0xc183,
  0xc184, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xc193, 0xc195, 0xc197, 0xc198, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xc1a3, 0x3000, 0xc1a6, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xc1b8, 0xc1b9, 0x3000, 0xc1bb,
  0xc1bc, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc1c5,
  0x3000, 0x3000, 0x3000, 0x3000, 0xc1cb, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xc1d5, 0x3000, 0xc1d9, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc1e1, 0x3000, 0x3000,
  0x3000, 0x3000, 0xc1e7, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xc1f4, 0xc1f5, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc203,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc216, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xc221, 0x3000, 0x3000, 0xc224, 0x3000,
  0x3000, 0xc227, 0x3000, 0x3000, 0xc22e, 0x3000, 0xc233, 0xc235,
  0x3000, 0x3000, 0xc238, 0x3000, 0x3000, 0xc23b, 0xc23c, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc244, 0xc245,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc247,
  0xc249, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xc257, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xc261, 0x3000, 0xc263, 0xc264, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc273,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc286, 0x3000, 0x3000,
  0x3000, 0xc28c, 0x3000, 0x3000, 0xc28f, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xc299, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xc2ab, 0xc2ae, 0xc2af, 0xc2b0, 0x3000, 0xc2b2, 0xc2b3,
  0x3000, 0x3000, 0xc2ba, 0xc2bb, 0x3000, 0x3000, 0xc2be, 0x3000,
  0xc2c0, 0xc2c1, 0x3000, 0x3000, 0xc2c4, 0x3000, 0x3000, 0x3000,
  0xc2c8, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xc2d0, 0xc2d1, 0x3000, 0xc2d3, 0x3000, 0xc2d5, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xc2de, 0x3000, 0x3000, 0xc2e2,
  0xc2e5, 0xc2e6, 0x3000, 0xc2e8, 0x3000, 0x3000, 0x3000, 0xc2f0,
  0x3000, 0xc2f3, 0xc2f4, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xc2ff, 0xc301, 0xc302, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xc30b, 0xc30e, 0x3000, 0x3000, 0xc311, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xc31b, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xc327, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xc32f, 0xc330, 0xc331, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc334, 0x3000, 0x3000,
  0xc337, 0xc338, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xc340, 0xc341, 0x3000, 0xc343, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xc34c, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc350, 0x3000,
  0x3000, 0x3000, 0xc354, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xc35c, 0x3000, 0x3000, 0x3000, 0x3000, 0xc361,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc36a, 0x3000,
  0x3000, 0x3000, 0xc36f, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xc37b, 0x3000, 0x3000, 0x3000, 0x3000, 0xc382,
  0x3000, 0xc385, 0x3000, 0x3000, 0x3000, 0x3000, 0xc38b, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc394, 0xc395,
  0x3000, 0xc397, 0xc398, 0xc399, 0x3000, 0x3000, 0x3000, 0xc39d,
  0x3000, 0x3000, 0xc3a0, 0xc3a1, 0x3000, 0x3000, 0xc3a4, 0x3000,
  0x3000, 0xc3a7, 0xc3a8, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xc3b0, 0xc3b1, 0x3000, 0xc3b3, 0xc3b4, 0xc3b5,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc3bc, 0xc3bd,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc3cc, 0xc3cd, 0x3000,
  0xc3cf, 0xc3d0, 0xc3d1, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xc3eb, 0x3000, 0x3000, 0x3000,
  0x3000, 0xc3f1, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xc3fb, 0xc3fc, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc404,
  0xc405, 0x3000, 0xc407, 0xc409, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xc411, 0x3000, 0x3000, 0xc414, 0x3000, 0x3000,
  0xc417, 0xc418, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xc423, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xc42d, 0x3000, 0x3000, 0x3000, 0x3000, 0xc433,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xc43f, 0xc440, 0xc441, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xc449, 0x3000, 0x3000, 0xc44c, 0x3000, 0x3000, 0xc44f,
  0xc450, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xc458, 0xc459, 0x3000, 0xc45b, 0x3000, 0xc45d, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xc46b, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xc477, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc47e,
  0x3000, 0xc481, 0x3000, 0x3000, 0xc484, 0x3000, 0x3000, 0xc487,
  0xc488, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xc490, 0xc491, 0x3000, 0xc493, 0xc495, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xc49d, 0x3000, 0x3000, 0xc4a0, 0x3000,
  0x3000, 0xc4a3, 0xc4a4, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xc4ac, 0xc4ad, 0x3000, 0xc4af, 0xc4b0, 0xc4b1,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc4b9, 0x3000,
  0x3000, 0x3000, 0x3000, 0xc4bf, 0xc4c0, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xc4c8, 0xc4c9, 0x3000, 0xc4cb, 0x3000, 0xc4cd,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc4d3, 0xc4d4, 0xc4d5,
  0x3000, 0x3000, 0xc4d8, 0x3000, 0x3000, 0xc4db, 0xc4dc, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xc4e4, 0xc4e5, 0x3000, 0xc4e7,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xc4f7, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xc503, 0x3000, 0xc505, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xc50d, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xc51d, 0x3000, 0xc51f, 0x3000, 0xc521, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc52f,
  0xc531, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xc53c, 0x3000, 0xc53f, 0xc540, 0x3000, 0x3000, 0xc543, 0x3000,
  0x3000, 0xc54b, 0xc54f, 0x3000, 0x3000, 0xc552, 0xc556, 0xc55a,
  0xc55b, 0x3000, 0xc55f, 0x3000, 0x3000, 0x3000, 0x3000, 0xc567,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xc579, 0xc57a, 0x3000, 0xc57e, 0x3000,
  0x3000, 0x3000, 0xc583, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xc590, 0xc592, 0x3000, 0xc594, 0x3000, 0xc599,
  0x3000, 0x3000, 0x3000, 0x3000, 0xc59f, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xc5a8, 0x3000, 0xc5ab, 0xc5ac,
  0xc5ad, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc5b6,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc5ba,
  0xc5bf, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc5cf,
  0x3000, 0x3000, 0x3000, 0x3000, 0xc5d7, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc5e4, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc5e9,
  0xc5ea, 0x3000, 0x3000, 0xc5f1, 0x3000, 0xc5f3, 0x3000, 0xc5f8,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc604, 0xc609, 0x3000,
  0x3000, 0x3000, 0x3000, 0xc60f, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xc61d, 0x3000, 0x3000, 0xc620,
  0x3000, 0x3000, 0x3000, 0xc626, 0x3000, 0x3000, 0xc62a, 0xc62b,
  0xc62f, 0x3000, 0xc632, 0x3000, 0x3000, 0xc63a, 0x3000, 0xc63d,
  0xc63e, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc647, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xc658, 0xc659, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xc663, 0xc664, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xc66d, 0x3000, 0xc670, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc67f,
  0x3000, 0xc682, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xc68c, 0x3000, 0x3000, 0x3000, 0x3000, 0xc692, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xc69b, 0xc69d, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc6ac,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc6b7,
  0x3000, 0xc6bc, 0x3000, 0x3000, 0x3000, 0xc6c2, 0x3000, 0xc6c6,
  0xc6c7, 0x3000, 0xc6c9, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc6d2, 0xc6d3,
  0x3000, 0x3000, 0x3000, 0xc6d8, 0x3000, 0x3000, 0x3000, 0x3000,
  0xc6df, 0x3000, 0x3000, 0xc6e4, 0xc6e5, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xc6ef, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xc6fb, 0xc6fc, 0x3000, 0x3000, 0x3000, 0xc701,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc70b, 0x3000,
  0xc70e, 0x3000, 0x3000, 0x3000, 0x3000, 0xc713, 0x3000, 0xc718,
  0x3000, 0x3000, 0xc71c, 0xc71d, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xc727, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xc736, 0xc738, 0xc739, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc743, 0xc745, 0xc746,
  0xc747, 0x3000, 0x3000, 0x3000, 0xc74e, 0x3000, 0xc759, 0x3000,
  0x3000, 0x3000, 0x3000, 0xc75f, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xc766, 0x3000, 0xc769, 0x3000, 0x3000, 0xc76d, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xc77b, 0x3000, 0xc780, 0x3000, 0xc782, 0xc786, 0xc78b,
  0xc78c, 0xc78d, 0xc78f, 0x3000, 0xc793, 0x3000, 0xc799, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc7a7, 0x3000, 0xc7a9,
  0xc7aa, 0xc7ab, 0x3000, 0x3000, 0x3000, 0xc7b2, 0xc7b3, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc7c2,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xc7cf, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xc7d9, 0x3000, 0xc7db, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc7eb, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc7f4, 0xc7f5, 0x3000,
  0xc7f7, 0x3000, 0xc7f9, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc802,
  0x3000, 0x3000, 0xc806, 0xc807, 0xc809, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xc814, 0x3000, 0x3000, 0xc819, 0x3000,
  0xc81b, 0x3000, 0x3000, 0x3000, 0x3000, 0xc823, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc830, 0xc832,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc839, 0x3000, 0x3000,
  0x3000, 0x3000, 0xc83f, 0xc841, 0xc842, 0xc843, 0x3000, 0x3000,
  0x3000, 0xc847, 0x3000, 0xc84b, 0xc84e, 0x3000, 0x3000, 0xc851,
  0x3000, 0xc853, 0xc855, 0x3000, 0x3000, 0xc858, 0x3000, 0x3000,
  0x3000, 0xc85c, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xc864, 0xc865, 0x3000, 0xc867, 0x3000, 0xc869, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xc877, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc890,
  0x3000, 0xc892, 0xc893, 0xc895, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xc89c, 0x3000, 0xc8a0, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xc8a9, 0x3000, 0x3000, 0xc8ac, 0x3000,
  0x3000, 0xc8af, 0xc8b0, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xc8b8, 0x3000, 0x3000, 0xc8bb, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xc8c5, 0x3000, 0x3000, 0x3000, 0x3000, 0xc8cb,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xc8d8, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xc8e7, 0xc8e8, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xc8f0, 0xc8f1, 0x3000, 0xc8f3, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xc8fb, 0x3000, 0x3000, 0x3000,
  0x3000, 0xc903, 0x3000, 0xc908, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3001, 0x3002, 0x00b7, 0x2025, 0x2026, 0x00a8, 0x3003,
  0x00ad, 0x2015, 0x2225, 0xff3c, 0x223c, 0x2018, 0x2019, 0x201c,
  0x201d, 0x3014, 0x3015, 0x3008, 0x3009, 0x300a, 0x300b, 0x300c,
  0x300d, 0x300e, 0x300f, 0x3010, 0x3011, 0x00b1, 0x00d7, 0x00f7,
  0x2260, 0x2264, 0x2265, 0x221e, 0x2234, 0x00b0, 0x2032, 0x2033,
  0x2103, 0x212b, 0xffe0, 0xffe1, 0xffe5, 0x2642, 0x2640, 0x2220,
  0x22a5, 0x2312, 0x2202, 0x2207, 0x2261, 0x2252, 0x00a7, 0x203b,
  0x2606, 0x2605, 0x25cb, 0x25cf, 0x25ce, 0x25c7, 0x25c6, 0x25a1,
  0x25a0, 0x25b3, 0x25b2, 0x25bd, 0x25bc, 0x2192, 0x2190, 0x2191,
  0x2193, 0x2194, 0x3013, 0x226a, 0x226b, 0x221a, 0x223d, 0x221d,
  0x2235, 0x222b, 0x222c, 0x2208, 0x220b, 0x2286, 0x2287, 0x2282,
  0x2283, 0x222a, 0x2229, 0x2227, 0x2228, 0xffe2, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xc917, 0xc919, 0x3000, 0x3000,
  0xc91c, 0x3000, 0x3000, 0xc91f, 0xc920, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xc928, 0xc929, 0x3000, 0xc92b,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc92d, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc935, 0x3000, 0x3000,
  0xc938, 0x3000, 0x3000, 0xc93b, 0xc93c, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xc944, 0xc945, 0x3000, 0xc947,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc948, 0xc949,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xc957, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xc965, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xc96d, 0x3000, 0x3000, 0x21d2, 0x21d4,
  0x2200, 0x2203, 0x00b4, 0xff5e, 0x02c7, 0x02d8, 0x02dd, 0x02da,
  0x02d9, 0x00b8, 0x02db, 0x00a1, 0x00bf, 0x02d0, 0x222e, 0x2211,
  0x220f, 0x00a4, 0x2109, 0x2030, 0x25c1, 0x25c0, 0x25b7, 0x25b6,
  0x2664, 0x2660, 0x2661, 0x2665, 0x2667, 0x2663, 0x2299, 0x25c8,
  0x25a3, 0x25d0, 0x25d1, 0x2592, 0x25a4, 0x25a5, 0x25a8, 0x25a7,
  0x25a6, 0x25a9, 0x2668, 0x260f, 0x260e, 0x261c, 0x261e, 0x00b6,
  0x2020, 0x2021, 0x2195, 0x2197, 0x2199, 0x2196, 0x2198, 0x266d,
  0x2669, 0x266a, 0x266c, 0x327f, 0x321c, 0x2116, 0x33c7, 0x2122,
  0x33c2, 0x33d8, 0x2121, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc97d, 0x3000,
  0xc97f, 0x3000, 0xc981, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc98f, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xc991, 0xc992, 0x3000, 0xc994,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc99e, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xc9a4, 0xc9a5, 0x3000, 0x3000, 0xc9a8,
  0x3000, 0x3000, 0x3000, 0xc9ac, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xc9b4, 0xc9b5, 0x3000, 0xc9b7, 0x3000, 0xc9b9, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xc9cf, 0xc9d2,
  0xc9d4, 0xc9d7, 0x3000, 0xc9db, 0xff01, 0xff02, 0xff03, 0xff04,
  0xff05, 0xff06, 0xff07, 0xff08, 0xff09, 0xff0a, 0xff0b, 0xff0c,
  0xff0d, 0xff0e, 0xff0f, 0xff10, 0xff11, 0xff12, 0xff13, 0xff14,
  0xff15, 0xff16, 0xff17, 0xff18, 0xff19, 0xff1a, 0xff1b, 0xff1c,
  0xff1d, 0xff1e, 0xff1f, 0xff20, 0xff21, 0xff22, 0xff23, 0xff24,
  0xff25, 0xff26, 0xff27, 0xff28, 0xff29, 0xff2a, 0xff2b, 0xff2c,
  0xff2d, 0xff2e, 0xff2f, 0xff30, 0xff31, 0xff32, 0xff33, 0xff34,
  0xff35, 0xff36, 0xff37, 0xff38, 0xff39, 0xff3a, 0xff3b, 0xffe6,
  0xff3d, 0xff3e, 0xff3f, 0xff40, 0xff41, 0xff42, 0xff43, 0xff44,
  0xff45, 0xff46, 0xff47, 0xff48, 0xff49, 0xff4a, 0xff4b, 0xff4c,
  0xff4d, 0xff4e, 0xff4f, 0xff50, 0xff51, 0xff52, 0xff53, 0xff54,
  0xff55, 0xff56, 0xff57, 0xff58, 0xff59, 0xff5a, 0xff5b, 0xff5c,
  0xff5d, 0xffe3, 0xc9de, 0x3000, 0x3000, 0xc9e3, 0x3000, 0x3000,
  0xc9e8, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xc9f7, 0x3000, 0x3000, 0x3000, 0x3000, 0xc9ff,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xca15, 0x3000, 0x3000, 0x3000,
  0xca1a, 0x3000, 0xca1c, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xca24, 0xca25, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xca27, 0x3000, 0x3000, 0x3000, 0x3000,
  0xca2d, 0x3000, 0x3000, 0xca30, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3131, 0x3132, 0x3133, 0x3134, 0x3135, 0x3136,
  0x3137, 0x3138, 0x3139, 0x313a, 0x313b, 0x313c, 0x313d, 0x313e,
  0x313f, 0x3140, 0x3141, 0x3142, 0x3143, 0x3144, 0x3145, 0x3146,
  0x3147, 0x3148, 0x3149, 0x314a, 0x314b, 0x314c, 0x314d, 0x314e,
  0x314f, 0x3150, 0x3151, 0x3152, 0x3153, 0x3154, 0x3155, 0x3156,
  0x3157, 0x3158, 0x3159, 0x315a, 0x315b, 0x315c, 0x315d, 0x315e,
  0x315f, 0x3160, 0x3161, 0x3162, 0x3163, 0x3164, 0x3165, 0x3166,
  0x3167, 0x3168, 0x3169, 0x316a, 0x316b, 0x316c, 0x316d, 0x316e,
  0x316f, 0x3170, 0x3171, 0x3172, 0x3173, 0x3174, 0x3175, 0x3176,
  0x3177, 0x3178, 0x3179, 0x317a, 0x317b, 0x317c, 0x317d, 0x317e,
  0x317f, 0x3180, 0x3181, 0x3182, 0x3183, 0x3184, 0x3185, 0x3186,
  0x3187, 0x3188, 0x3189, 0x318a, 0x318b, 0x318c, 0x318d, 0x318e,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xca53, 0x3000, 0x3000, 0xca57, 0xca58, 0x3000, 0x3000,
  0xca5b, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xca67,
  0xca69, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xca6c, 0x3000, 0x3000, 0xca6f, 0xca70, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xca78, 0xca79, 0x3000,
  0xca7b, 0xca7c, 0x3000, 0x3000, 0x3000, 0xca81, 0x3000, 0x3000,
  0xca85, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xca88, 0x3000, 0x3000, 0xca8b, 0xca8c, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xca94, 0xca95, 0x3000,
  0xca97, 0xca99, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xcaa0, 0xcaa1, 0x3000, 0x3000, 0xcaa4, 0x3000, 0x3000, 0x3000,
  0x2170, 0x2171, 0x2172, 0x2173, 0x2174, 0x2175, 0x2176, 0x2177,
  0x2178, 0x2179, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x2160,
  0x2161, 0x2162, 0x2163, 0x2164, 0x2165, 0x2166, 0x2167, 0x2168,
  0x2169, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x0391, 0x0392, 0x0393, 0x0394, 0x0395, 0x0396, 0x0397, 0x0398,
  0x0399, 0x039a, 0x039b, 0x039c, 0x039d, 0x039e, 0x039f, 0x03a0,
  0x03a1, 0x03a3, 0x03a4, 0x03a5, 0x03a6, 0x03a7, 0x03a8, 0x03a9,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x03b1, 0x03b2, 0x03b3, 0x03b4, 0x03b5, 0x03b6, 0x03b7, 0x03b8,
  0x03b9, 0x03ba, 0x03bb, 0x03bc, 0x03bd, 0x03be, 0x03bf, 0x03c0,
  0x03c1, 0x03c3, 0x03c4, 0x03c5, 0x03c6, 0x03c7, 0x03c8, 0x03c9,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xcaa8, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xcab0, 0xcab1,
  0x3000, 0xcab3, 0x3000, 0xcab5, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xcabe, 0x3000, 0x3000, 0x3000, 0xcac3, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xcac6, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xcad2, 0x3000,
  0x3000, 0x3000, 0xcad7, 0x3000, 0x3000, 0xcadc, 0x3000, 0x3000,
  0xcadf, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xcae8, 0xcae9,
  0x3000, 0xcaeb, 0xcaed, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xcaf5, 0x3000, 0x3000, 0xcaf8, 0x3000, 0x3000, 0xcafb,
  0xcafc, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x2500, 0x2502,
  0x250c, 0x2510, 0x2518, 0x2514, 0x251c, 0x252c, 0x2524, 0x2534,
  0x253c, 0x2501, 0x2503, 0x250f, 0x2513, 0x251b, 0x2517, 0x2523,
  0x2533, 0x252b, 0x253b, 0x254b, 0x2520, 0x252f, 0x2528, 0x2537,
  0x253f, 0x251d, 0x2530, 0x2525, 0x2538, 0x2542, 0x2512, 0x2511,
  0x251a, 0x2519, 0x2516, 0x2515, 0x250e, 0x250d, 0x251e, 0x251f,
  0x2521, 0x2522, 0x2526, 0x2527, 0x2529, 0x252a, 0x252d, 0x252e,
  0x2531, 0x2532, 0x2535, 0x2536, 0x2539, 0x253a, 0x253d, 0x253e,
  0x2540, 0x2541, 0x2543, 0x2544, 0x2545, 0x2546, 0x2547, 0x2548,
  0x2549, 0x254a, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xcb11, 0x3000, 0x3000, 0x3000, 0x3000, 0xcb17, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xcb23,
  0xcb24, 0xcb25, 0x3000, 0xcb27, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xcb2c, 0xcb2d,
  0x3000, 0x3000, 0xcb30, 0x3000, 0x3000, 0x3000, 0xcb34, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xcb3c, 0xcb3d,
  0x3000, 0xcb3f, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xcb4a,
  0x3000, 0x3000, 0x3000, 0xcb4f, 0x3000, 0xcb52, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xcb5b, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xcb65, 0x3000, 0x3000, 0xcb68,
  0x3000, 0x3000, 0xcb6b, 0xcb6c, 0x3395, 0x3396, 0x3397, 0x2113,
  0x3398, 0x33c4, 0x33a3, 0x33a4, 0x33a5, 0x33a6, 0x3399, 0x339a,
  0x339b, 0x339c, 0x339d, 0x339e, 0x339f, 0x33a0, 0x33a1, 0x33a2,
  0x33ca, 0x338d, 0x338e, 0x338f, 0x33cf, 0x3388, 0x3389, 0x33c8,
  0x33a7, 0x33a8, 0x33b0, 0x33b1, 0x33b2, 0x33b3, 0x33b4, 0x33b5,
  0x33b6, 0x33b7, 0x33b8, 0x33b9, 0x3380, 0x3381, 0x3382, 0x3383,
  0x3384, 0x33ba, 0x33bb, 0x33bc, 0x33bd, 0x33be, 0x33bf, 0x3390,
  0x3391, 0x3392, 0x3393, 0x3394, 0x2126, 0x33c0, 0x33c1, 0x338a,
  0x338b, 0x338c, 0x33d6, 0x33c5, 0x33ad, 0x33ae, 0x33af, 0x33db,
  0x33a9, 0x33aa, 0x33ab, 0x33ac, 0x33dd, 0x33d0, 0x33d3, 0x33c3,
  0x33c9, 0x33dc, 0x33c6, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xcb74, 0xcb75, 0x3000, 0xcb77, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xcb80, 0xcb81, 0x3000, 0x3000, 0xcb84,
  0x3000, 0x3000, 0xcb87, 0xcb88, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xcb90, 0xcb91, 0x3000, 0xcb93, 0xcb94, 0xcb95, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xcb9d, 0x3000, 0x3000,
  0xcba0, 0x3000, 0x3000, 0xcba3, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xcba4, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xcbac, 0xcbad, 0x3000, 0xcbaf, 0x3000, 0xcbb1,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xcbb9, 0x3000,
  0x3000, 0xcbbc, 0x3000, 0x3000, 0x3000, 0xcbc0, 0x3000, 0x3000,
  0x3000, 0x3000, 0x00c6, 0x00d0, 0x00aa, 0x0126, 0x3000, 0x0132,
  0x3000, 0x013f, 0x0141, 0x00d8, 0x0152, 0x00ba, 0x00de, 0x0166,
  0x014a, 0x3000, 0x3260, 0x3261, 0x3262, 0x3263, 0x3264, 0x3265,
  0x3266, 0x3267, 0x3268, 0x3269, 0x326a, 0x326b, 0x326c, 0x326d,
  0x326e, 0x326f, 0x3270, 0x3271, 0x3272, 0x3273, 0x3274, 0x3275,
  0x3276, 0x3277, 0x3278, 0x3279, 0x327a, 0x327b, 0x24d0, 0x24d1,
  0x24d2, 0x24d3, 0x24d4, 0x24d5, 0x24d6, 0x24d7, 0x24d8, 0x24d9,
  0x24da, 0x24db, 0x24dc, 0x24dd, 0x24de, 0x24df, 0x24e0, 0x24e1,
  0x24e2, 0x24e3, 0x24e4, 0x24e5, 0x24e6, 0x24e7, 0x24e8, 0x24e9,
  0x2460, 0x2461, 0x2462, 0x2463, 0x2464, 0x2465, 0x2466, 0x2467,
  0x2468, 0x2469, 0x246a, 0x246b, 0x246c, 0x246d, 0x246e, 0x00bd,
  0x2153, 0x2154, 0x00bc, 0x00be, 0x215b, 0x215c, 0x215d, 0x215e,
  0x3000, 0x3000, 0x3000, 0xcbc8, 0xcbc9, 0x3000, 0xcbcb, 0x3000,
  0xcbcd, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xcbd5,
  0x3000, 0x3000, 0xcbd8, 0x3000, 0x3000, 0xcbdb, 0xcbdc, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xcbe5, 0x3000, 0x3000, 0xcbea,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xcbf0, 0xcbf1, 0x3000,
  0x3000, 0xcbf4, 0x3000, 0x3000, 0x3000, 0xcbf8, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xcc00, 0xcc01, 0x3000, 0xcc03, 0x3000,
  0xcc05, 0xcc06, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xcc13, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xcc1f, 0x3000, 0x3000, 0x3000,
  0x00e6, 0x0111, 0x00f0, 0x0127, 0x0131, 0x0133, 0x0138, 0x0140,
  0x0142, 0x00f8, 0x0153, 0x00df, 0x00fe, 0x0167, 0x014b, 0x0149,
  0x3200, 0x3201, 0x3202, 0x3203, 0x3204, 0x3205, 0x3206, 0x3207,
  0x3208, 0x3209, 0x320a, 0x320b, 0x320c, 0x320d, 0x320e, 0x320f,
  0x3210, 0x3211, 0x3212, 0x3213, 0x3214, 0x3215, 0x3216, 0x3217,
  0x3218, 0x3219, 0x321a, 0x321b, 0x249c, 0x249d, 0x249e, 0x249f,
  0x24a0, 0x24a1, 0x24a2, 0x24a3, 0x24a4, 0x24a5, 0x24a6, 0x24a7,
  0x24a8, 0x24a9, 0x24aa, 0x24ab, 0x24ac, 0x24ad, 0x24ae, 0x24af,
  0x24b0, 0x24b1, 0x24b2, 0x24b3, 0x24b4, 0x24b5, 0x2474, 0x2475,
  0x2476, 0x2477, 0x2478, 0x2479, 0x247a, 0x247b, 0x247c, 0x247d,
  0x247e, 0x247f, 0x2480, 0x2481, 0x2482, 0x00b9, 0x00b2, 0x00b3,
  0x2074, 0x207f, 0x2081, 0x2082, 0x2083, 0x2084, 0x3000, 0xcc26,
  0x3000, 0x3000, 0x3000, 0xcc2f, 0xcc31, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xcc3f, 0x3000, 0x3000, 0xcc42,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xcc4b, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xcc5b, 0x3000, 0x3000,
  0xcc5e, 0x3000, 0xcc61, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xcc71, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xcc73, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xcc7a, 0x3000, 0xcc7c, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xcc91, 0x3000, 0x3000, 0x3041, 0x3042,
  0x3043, 0x3044, 0x3045, 0x3046, 0x3047, 0x3048, 0x3049, 0x304a,
  0x304b, 0x304c, 0x304d, 0x304e, 0x304f, 0x3050, 0x3051, 0x3052,
  0x3053, 0x3054, 0x3055, 0x3056, 0x3057, 0x3058, 0x3059, 0x305a,
  0x305b, 0x305c, 0x305d, 0x305e, 0x305f, 0x3060, 0x3061, 0x3062,
  0x3063, 0x3064, 0x3065, 0x3066, 0x3067, 0x3068, 0x3069, 0x306a,
  0x306b, 0x306c, 0x306d, 0x306e, 0x306f, 0x3070, 0x3071, 0x3072,
  0x3073, 0x3074, 0x3075, 0x3076, 0x3077, 0x3078, 0x3079, 0x307a,
  0x307b, 0x307c, 0x307d, 0x307e, 0x307f, 0x3080, 0x3081, 0x3082,
  0x3083, 0x3084, 0x3085, 0x3086, 0x3087, 0x3088, 0x3089, 0x308a,
  0x308b, 0x308c, 0x308d, 0x308e, 0x308f, 0x3090, 0x3091, 0x3092,
  0x3093, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xcc9f, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xcca7, 0x3000, 0xccae, 0x3000, 0x3000,
  0x3000, 0xccb2, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xccbb, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xccc3, 0x3000, 0xccc8, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xccd1, 0x3000, 0x3000,
  0x3000, 0x3000, 0xccd7, 0xccd8, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xcce0, 0xcce1, 0x3000, 0xcce3, 0xcce5, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xcced, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xccf4, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xccfc, 0xccfd, 0x30a1, 0x30a2, 0x30a3, 0x30a4,
  0x30a5, 0x30a6, 0x30a7, 0x30a8, 0x30a9, 0x30aa, 0x30ab, 0x30ac,
  0x30ad, 0x30ae, 0x30af, 0x30b0, 0x30b1, 0x30b2, 0x30b3, 0x30b4,
  0x30b5, 0x30b6, 0x30b7, 0x30b8, 0x30b9, 0x30ba, 0x30bb, 0x30bc,
  0x30bd, 0x30be, 0x30bf, 0x30c0, 0x30c1, 0x30c2, 0x30c3, 0x30c4,
  0x30c5, 0x30c6, 0x30c7, 0x30c8, 0x30c9, 0x30ca, 0x30cb, 0x30cc,
  0x30cd, 0x30ce, 0x30cf, 0x30d0, 0x30d1, 0x30d2, 0x30d3, 0x30d4,
  0x30d5, 0x30d6, 0x30d7, 0x30d8, 0x30d9, 0x30da, 0x30db, 0x30dc,
  0x30dd, 0x30de, 0x30df, 0x30e0, 0x30e1, 0x30e2, 0x30e3, 0x30e4,
  0x30e5, 0x30e6, 0x30e7, 0x30e8, 0x30e9, 0x30ea, 0x30eb, 0x30ec,
  0x30ed, 0x30ee, 0x30ef, 0x30f0, 0x30f1, 0x30f2, 0x30f3, 0x30f4,
  0x30f5, 0x30f6, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xccff, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xcd0f,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xcd25, 0x3000, 0x3000,
  0x3000, 0x3000, 0xcd2b, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xcd34, 0xcd35, 0x3000, 0xcd37, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xcd40, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xcd53, 0xcd54,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xcd5d,
  0x3000, 0x3000, 0x0410, 0x0411, 0x0412, 0x0413, 0x0414, 0x0415,
  0x0401, 0x0416, 0x0417, 0x0418, 0x0419, 0x041a, 0x041b, 0x041c,
  0x041d, 0x041e, 0x041f, 0x0420, 0x0421, 0x0422, 0x0423, 0x0424,
  0x0425, 0x0426, 0x0427, 0x0428, 0x0429, 0x042a, 0x042b, 0x042c,
  0x042d, 0x042e, 0x042f, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x0430, 0x0431, 0x0432, 0x0433, 0x0434, 0x0435,
  0x0451, 0x0436, 0x0437, 0x0438, 0x0439, 0x043a, 0x043b, 0x043c,
  0x043d, 0x043e, 0x043f, 0x0440, 0x0441, 0x0442, 0x0443, 0x0444,
  0x0445, 0x0446, 0x0447, 0x0448, 0x0449, 0x044a, 0x044b, 0x044c,
  0x044d, 0x044e, 0x044f, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xcd63, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xcd70, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xcd79, 0x3000, 0x3000, 0xcd7c, 0x3000, 0x3000,
  0x3000, 0xcd80, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xcd89,
  0x3000, 0xcd8b, 0x3000, 0xcd8d, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xcd9b, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xcdb1, 0x3000, 0x3000, 0xcdb4,
  0x3000, 0x3000, 0xcdb7, 0xcdb8, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xcdc0, 0xcdc1, 0x3000, 0xcdc3, 0xcdc5,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xcdcd, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xcdd4, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xcddc, 0xcddd, 0x3000, 0xcddf, 0xcde0, 0xcde1,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xcde9, 0x3000, 0x3000, 0x3000,
  0x3000, 0xcdef, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xce05,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xce15, 0x3000, 0xce17, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xce27, 0xce29, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xce2c, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xce3c, 0xce3d, 0x3000, 0x3000, 0xce40, 0x3000, 0x3000, 0x3000,
  0xce44, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xce4c, 0xce4d,
  0x3000, 0xce4f, 0x3000, 0xce51, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xce62, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xce6c, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xce6e, 0x3000, 0xce70, 0x3000,
  0xce72, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xce7b, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xce88,
  0x3000, 0x3000, 0x3000, 0xce8d, 0xce8e, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xce97, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xcea9, 0xceaa, 0x3000, 0x3000,
  0x3000, 0xceb0, 0x3000, 0x3000, 0x3000, 0xceb4, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xcebc, 0xcebd, 0x3000, 0xcebf, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xcec8, 0x3000,
  0x3000, 0x3000, 0xcecc, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xcefb, 0x3000, 0xcefd,
  0xcefe, 0x3000, 0xac00, 0xac01, 0xac04, 0xac07, 0xac08, 0xac09,
  0xac0a, 0xac10, 0xac11, 0xac12, 0xac13, 0xac14, 0xac15, 0xac16,
  0xac17, 0xac19, 0xac1a, 0xac1b, 0xac1c, 0xac1d, 0xac20, 0xac24,
  0xac2c, 0xac2d, 0xac2f, 0xac30, 0xac31, 0xac38, 0xac39, 0xac3c,
  0xac40, 0xac4b, 0xac4d, 0xac54, 0xac58, 0xac5c, 0xac70, 0xac71,
  0xac74, 0xac77, 0xac78, 0xac7a, 0xac80, 0xac81, 0xac83, 0xac84,
  0xac85, 0xac86, 0xac89, 0xac8a, 0xac8b, 0xac8c, 0xac90, 0xac94,
  0xac9c, 0xac9d, 0xac9f, 0xaca0, 0xaca1, 0xaca8, 0xaca9, 0xacaa,
  0xacac, 0xacaf, 0xacb0, 0xacb8, 0xacb9, 0xacbb, 0xacbc, 0xacbd,
  0xacc1, 0xacc4, 0xacc8, 0xaccc, 0xacd5, 0xacd7, 0xace0, 0xace1,
  0xace4, 0xace7, 0xace8, 0xacea, 0xacec, 0xacef, 0xacf0, 0xacf1,
  0xacf3, 0xacf5, 0xacf6, 0xacfc, 0xacfd, 0xad00, 0xad04, 0xad06,
  0x3000, 0x3000, 0x3000, 0x3000, 0xcf07, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xcf14, 0x3000, 0x3000,
  0x3000, 0xcf19, 0xcf1a, 0x3000, 0xcf1d, 0x3000, 0x3000, 0x3000,
  0x3000, 0xcf23, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xcf39, 0x3000,
  0x3000, 0xcf3c, 0x3000, 0x3000, 0x3000, 0xcf40, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xcf48, 0xcf49, 0x3000, 0xcf4b, 0xcf4c,
  0xcf4d, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xcf5b, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xcf6a, 0x3000, 0x3000,
  0xad0c, 0xad0d, 0xad0f, 0xad11, 0xad18, 0xad1c, 0xad20, 0xad29,
  0xad2c, 0xad2d, 0xad34, 0xad35, 0xad38, 0xad3c, 0xad44, 0xad45,
  0xad47, 0xad49, 0xad50, 0xad54, 0xad58, 0xad61, 0xad63, 0xad6c,
  0xad6d, 0xad70, 0xad73, 0xad74, 0xad75, 0xad76, 0xad7b, 0xad7c,
  0xad7d, 0xad7f, 0xad81, 0xad82, 0xad88, 0xad89, 0xad8c, 0xad90,
  0xad9c, 0xad9d, 0xada4, 0xadb7, 0xadc0, 0xadc1, 0xadc4, 0xadc8,
  0xadd0, 0xadd1, 0xadd3, 0xaddc, 0xade0, 0xade4, 0xadf8, 0xadf9,
  0xadfc, 0xadff, 0xae00, 0xae01, 0xae08, 0xae09, 0xae0b, 0xae0d,
  0xae14, 0xae30, 0xae31, 0xae34, 0xae37, 0xae38, 0xae3a, 0xae40,
  0xae41, 0xae43, 0xae45, 0xae46, 0xae4a, 0xae4c, 0xae4d, 0xae4e,
  0xae50, 0xae54, 0xae56, 0xae5c, 0xae5d, 0xae5f, 0xae60, 0xae61,
  0xae65, 0xae68, 0xae69, 0xae6c, 0xae70, 0xae78, 0x3000, 0xcf6e,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xcf81, 0x3000, 0xcf83,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xcf8d,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xcf90, 0x3000, 0x3000, 0xcf93, 0xcf94, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xcf9c, 0xcf9d, 0x3000, 0xcf9f,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xcfa9,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xcfac, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xcfb8, 0xcfb9, 0x3000, 0xcfbb, 0x3000,
  0xcfbd, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xcfc5,
  0x3000, 0x3000, 0xcfc8, 0x3000, 0x3000, 0x3000, 0xae79, 0xae7b,
  0xae7c, 0xae7d, 0xae84, 0xae85, 0xae8c, 0xaebc, 0xaebd, 0xaebe,
  0xaec0, 0xaec4, 0xaecc, 0xaecd, 0xaecf, 0xaed0, 0xaed1, 0xaed8,
  0xaed9, 0xaedc, 0xaee8, 0xaeeb, 0xaeed, 0xaef4, 0xaef8, 0xaefc,
  0xaf07, 0xaf08, 0xaf0d, 0xaf10, 0xaf2c, 0xaf2d, 0xaf30, 0xaf32,
  0xaf34, 0xaf3c, 0xaf3d, 0xaf3f, 0xaf41, 0xaf42, 0xaf43, 0xaf48,
  0xaf49, 0xaf50, 0xaf5c, 0xaf5d, 0xaf64, 0xaf65, 0xaf79, 0xaf80,
  0xaf84, 0xaf88, 0xaf90, 0xaf91, 0xaf95, 0xaf9c, 0xafb8, 0xafb9,
  0xafbc, 0xafc0, 0xafc7, 0xafc8, 0xafc9, 0xafcb, 0xafcd, 0xafce,
  0xafd4, 0xafdc, 0xafe8, 0xafe9, 0xaff0, 0xaff1, 0xaff4, 0xaff8,
  0xb000, 0xb001, 0xb004, 0xb00c, 0xb010, 0xb014, 0xb01c, 0xb01d,
  0xb028, 0xb044, 0xb045, 0xb048, 0xb04a, 0xb04c, 0xb04e, 0xb053,
  0xb054, 0xb055, 0xb057, 0xb059, 0xcfcc, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xcfd4, 0xcfd5, 0x3000, 0xcfd7,
  0x3000, 0xcfd9, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xcfe7, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xcffa, 0x3000, 0xcffd, 0x3000, 0x3000, 0x3000, 0x3000, 0xd003,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd00c, 0xd00d, 0x3000,
  0x3000, 0xd010, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xd019, 0x3000, 0x3000, 0xd01c, 0x3000, 0x3000, 0x3000, 0xd020,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd028,
  0xd029, 0x3000, 0xd02b, 0xd02c, 0xb05d, 0xb07c, 0xb07d, 0xb080,
  0xb084, 0xb08c, 0xb08d, 0xb08f, 0xb091, 0xb098, 0xb099, 0xb09a,
  0xb09c, 0xb09f, 0xb0a0, 0xb0a1, 0xb0a2, 0xb0a8, 0xb0a9, 0xb0ab,
  0xb0ac, 0xb0ad, 0xb0ae, 0xb0af, 0xb0b1, 0xb0b3, 0xb0b4, 0xb0b5,
  0xb0b8, 0xb0bc, 0xb0c4, 0xb0c5, 0xb0c7, 0xb0c8, 0xb0c9, 0xb0d0,
  0xb0d1, 0xb0d4, 0xb0d8, 0xb0e0, 0xb0e5, 0xb108, 0xb109, 0xb10b,
  0xb10c, 0xb110, 0xb112, 0xb113, 0xb118, 0xb119, 0xb11b, 0xb11c,
  0xb11d, 0xb123, 0xb124, 0xb125, 0xb128, 0xb12c, 0xb134, 0xb135,
  0xb137, 0xb138, 0xb139, 0xb140, 0xb141, 0xb144, 0xb148, 0xb150,
  0xb151, 0xb154, 0xb155, 0xb158, 0xb15c, 0xb160, 0xb178, 0xb179,
  0xb17c, 0xb180, 0xb182, 0xb188, 0xb189, 0xb18b, 0xb18d, 0xb192,
  0xb193, 0xb194, 0xb198, 0xb19c, 0xb1a8, 0xb1cc, 0xb1d0, 0xb1d4,
  0xb1dc, 0xb1dd, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xd03b, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xd051, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd061,
  0x3000, 0xd063, 0x3000, 0xd065, 0x3000, 0x3000, 0xd068, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xd072, 0x3000, 0x3000, 0xd076, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd07f, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd088, 0xd089, 0x3000,
  0x3000, 0xd08c, 0x3000, 0x3000, 0x3000, 0xd090, 0x3000, 0x3000,
  0x3000, 0x3000, 0xb1df, 0xb1e8, 0xb1e9, 0xb1ec, 0xb1f0, 0xb1f9,
  0xb1fb, 0xb1fd, 0xb204, 0xb205, 0xb208, 0xb20b, 0xb20c, 0xb214,
  0xb215, 0xb217, 0xb219, 0xb220, 0xb234, 0xb23c, 0xb258, 0xb25c,
  0xb260, 0xb268, 0xb269, 0xb274, 0xb275, 0xb27c, 0xb284, 0xb285,
  0xb289, 0xb290, 0xb291, 0xb294, 0xb298, 0xb299, 0xb29a, 0xb2a0,
  0xb2a1, 0xb2a3, 0xb2a5, 0xb2a6, 0xb2aa, 0xb2ac, 0xb2b0, 0xb2b4,
  0xb2c8, 0xb2c9, 0xb2cc, 0xb2d0, 0xb2d2, 0xb2d8, 0xb2d9, 0xb2db,
  0xb2dd, 0xb2e2, 0xb2e4, 0xb2e5, 0xb2e6, 0xb2e8, 0xb2eb, 0xb2ec,
  0xb2ed, 0xb2ee, 0xb2ef, 0xb2f3, 0xb2f4, 0xb2f5, 0xb2f7, 0xb2f8,
  0xb2f9, 0xb2fa, 0xb2fb, 0xb2ff, 0xb300, 0xb301, 0xb304, 0xb308,
  0xb310, 0xb311, 0xb313, 0xb314, 0xb315, 0xb31c, 0xb354, 0xb355,
  0xb356, 0xb358, 0xb35b, 0xb35c, 0xb35e, 0xb35f, 0xb364, 0xb365,
  0x3000, 0x3000, 0x3000, 0xd098, 0xd099, 0x3000, 0xd09b, 0x3000,
  0xd09d, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xd0ab, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xd0b8, 0x3000, 0x3000, 0x3000, 0x3000, 0xd0be,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd0c7, 0xd0ca, 0x3000,
  0x3000, 0x3000, 0x3000, 0xd0cf, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xd0da, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xd0e2, 0xd0e3, 0xd0e5, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xd0f6, 0x3000, 0xd0f9, 0x3000, 0x3000, 0xd0fc,
  0x3000, 0x3000, 0x3000, 0xd100, 0x3000, 0x3000, 0x3000, 0x3000,
  0xb367, 0xb369, 0xb36b, 0xb36e, 0xb370, 0xb371, 0xb374, 0xb378,
  0xb380, 0xb381, 0xb383, 0xb384, 0xb385, 0xb38c, 0xb390, 0xb394,
  0xb3a0, 0xb3a1, 0xb3a8, 0xb3ac, 0xb3c4, 0xb3c5, 0xb3c8, 0xb3cb,
  0xb3cc, 0xb3ce, 0xb3d0, 0xb3d4, 0xb3d5, 0xb3d7, 0xb3d9, 0xb3db,
  0xb3dd, 0xb3e0, 0xb3e4, 0xb3e8, 0xb3fc, 0xb410, 0xb418, 0xb41c,
  0xb420, 0xb428, 0xb429, 0xb42b, 0xb434, 0xb450, 0xb451, 0xb454,
  0xb458, 0xb460, 0xb461, 0xb463, 0xb465, 0xb46c, 0xb480, 0xb488,
  0xb49d, 0xb4a4, 0xb4a8, 0xb4ac, 0xb4b5, 0xb4b7, 0xb4b9, 0xb4c0,
  0xb4c4, 0xb4c8, 0xb4d0, 0xb4d5, 0xb4dc, 0xb4dd, 0xb4e0, 0xb4e3,
  0xb4e4, 0xb4e6, 0xb4ec, 0xb4ed, 0xb4ef, 0xb4f1, 0xb4f8, 0xb514,
  0xb515, 0xb518, 0xb51b, 0xb51c, 0xb524, 0xb525, 0xb527, 0xb528,
  0xb529, 0xb52a, 0xb530, 0xb531, 0xb534, 0xb538, 0x3000, 0x3000,
  0x3000, 0xd108, 0xd109, 0x3000, 0xd10b, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xd114, 0x3000, 0x3000, 0x3000,
  0xd118, 0x3000, 0xd11a, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xd137, 0xd139, 0xd13b, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xd153, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xd160, 0x3000, 0x3000, 0x3000, 0x3000,
  0xd166, 0x3000, 0xd169, 0x3000, 0x3000, 0x3000, 0xb540, 0xb541,
  0xb543, 0xb544, 0xb545, 0xb54b, 0xb54c, 0xb54d, 0xb550, 0xb554,
  0xb55c, 0xb55d, 0xb55f, 0xb560, 0xb561, 0xb5a0, 0xb5a1, 0xb5a4,
  0xb5a8, 0xb5aa, 0xb5ab, 0xb5b0, 0xb5b1, 0xb5b3, 0xb5b4, 0xb5b5,
  0xb5bb, 0xb5bc, 0xb5bd, 0xb5c0, 0xb5c4, 0xb5cc, 0xb5cd, 0xb5cf,
  0xb5d0, 0xb5d1, 0xb5d8, 0xb5ec, 0xb610, 0xb611, 0xb614, 0xb618,
  0xb625, 0xb62c, 0xb634, 0xb648, 0xb664, 0xb668, 0xb69c, 0xb69d,
  0xb6a0, 0xb6a4, 0xb6ab, 0xb6ac, 0xb6b1, 0xb6d4, 0xb6f0, 0xb6f4,
  0xb6f8, 0xb700, 0xb701, 0xb705, 0xb728, 0xb729, 0xb72c, 0xb72f,
  0xb730, 0xb738, 0xb739, 0xb73b, 0xb744, 0xb748, 0xb74c, 0xb754,
  0xb755, 0xb760, 0xb764, 0xb768, 0xb770, 0xb771, 0xb773, 0xb775,
  0xb77c, 0xb77d, 0xb780, 0xb784, 0xb78c, 0xb78d, 0xb78f, 0xb790,
  0xb791, 0xb792, 0xb796, 0xb797, 0x3000, 0xd16f, 0xd170, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd178, 0xd179,
  0x3000, 0xd17b, 0xd17d, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xd185, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd18c, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd194, 0x3000, 0x3000,
  0xd197, 0x3000, 0xd199, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd1a7, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd1b7, 0xd1b8,
  0xd1b9, 0x3000, 0xd1bd, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xd1c4, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xd1cc, 0xd1cd, 0x3000, 0xd1cf, 0xb798, 0xb799, 0xb79c, 0xb7a0,
  0xb7a8, 0xb7a9, 0xb7ab, 0xb7ac, 0xb7ad, 0xb7b4, 0xb7b5, 0xb7b8,
  0xb7c7, 0xb7c9, 0xb7ec, 0xb7ed, 0xb7f0, 0xb7f4, 0xb7fc, 0xb7fd,
  0xb7ff, 0xb800, 0xb801, 0xb807, 0xb808, 0xb809, 0xb80c, 0xb810,
  0xb818, 0xb819, 0xb81b, 0xb81d, 0xb824, 0xb825, 0xb828, 0xb82c,
  0xb834, 0xb835, 0xb837, 0xb838, 0xb839, 0xb840, 0xb844, 0xb851,
  0xb853, 0xb85c, 0xb85d, 0xb860, 0xb864, 0xb86c, 0xb86d, 0xb86f,
  0xb871, 0xb878, 0xb87c, 0xb88d, 0xb8a8, 0xb8b0, 0xb8b4, 0xb8b8,
  0xb8c0, 0xb8c1, 0xb8c3, 0xb8c5, 0xb8cc, 0xb8d0, 0xb8d4, 0xb8dd,
  0xb8df, 0xb8e1, 0xb8e8, 0xb8e9, 0xb8ec, 0xb8f0, 0xb8f8, 0xb8f9,
  0xb8fb, 0xb8fd, 0xb904, 0xb918, 0xb920, 0xb93c, 0xb93d, 0xb940,
  0xb944, 0xb94c, 0xb94f, 0xb951, 0xb958, 0xb959, 0xb95c, 0xb960,
  0xb968, 0xb969, 0x3000, 0xd1d1, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xd1ec, 0xd1ed, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xd1f5, 0x3000, 0x3000, 0x3000, 0x3000,
  0xd1fb, 0xd1fc, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xd204, 0xd205, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xd211, 0x3000, 0x3000, 0xd214, 0x3000, 0x3000, 0x3000,
  0xd218, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xd220, 0xd221, 0x3000, 0xd223, 0x3000, 0xd225, 0x3000, 0x3000,
  0x3000, 0x3000, 0xb96b, 0xb96d, 0xb974, 0xb975, 0xb978, 0xb97c,
  0xb984, 0xb985, 0xb987, 0xb989, 0xb98a, 0xb98d, 0xb98e, 0xb9ac,
  0xb9ad, 0xb9b0, 0xb9b4, 0xb9bc, 0xb9bd, 0xb9bf, 0xb9c1, 0xb9c8,
  0xb9c9, 0xb9cc, 0xb9ce, 0xb9cf, 0xb9d0, 0xb9d1, 0xb9d2, 0xb9d8,
  0xb9d9, 0xb9db, 0xb9dd, 0xb9de, 0xb9e1, 0xb9e3, 0xb9e4, 0xb9e5,
  0xb9e8, 0xb9ec, 0xb9f4, 0xb9f5, 0xb9f7, 0xb9f8, 0xb9f9, 0xb9fa,
  0xba00, 0xba01, 0xba08, 0xba15, 0xba38, 0xba39, 0xba3c, 0xba40,
  0xba42, 0xba48, 0xba49, 0xba4b, 0xba4d, 0xba4e, 0xba53, 0xba54,
  0xba55, 0xba58, 0xba5c, 0xba64, 0xba65, 0xba67, 0xba68, 0xba69,
  0xba70, 0xba71, 0xba74, 0xba78, 0xba83, 0xba84, 0xba85, 0xba87,
  0xba8c, 0xbaa8, 0xbaa9, 0xbaab, 0xbaac, 0xbab0, 0xbab2, 0xbab8,
  0xbab9, 0xbabb, 0xbabd, 0xbac4, 0xbac8, 0xbad8, 0xbad9, 0xbafc,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd233, 0x3000,
  0xd236, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd249, 0x3000,
  0x3000, 0xd24c, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xd24f, 0xd250, 0x3000, 0x3000, 0x3000, 0xd254,
  0x3000, 0x3000, 0x3000, 0xd258, 0xd259, 0x3000, 0xd25b, 0xd25d,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd265, 0x3000,
  0x3000, 0xd268, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xd26b, 0xd26c, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xd274, 0xd275, 0x3000, 0xd277, 0xd278,
  0xd279, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xd287, 0x3000, 0x3000, 0x3000, 0x3000,
  0xbb00, 0xbb04, 0xbb0d, 0xbb0f, 0xbb11, 0xbb18, 0xbb1c, 0xbb20,
  0xbb29, 0xbb2b, 0xbb34, 0xbb35, 0xbb36, 0xbb38, 0xbb3b, 0xbb3c,
  0xbb3d, 0xbb3e, 0xbb44, 0xbb45, 0xbb47, 0xbb49, 0xbb4d, 0xbb4f,
  0xbb50, 0xbb54, 0xbb58, 0xbb61, 0xbb63, 0xbb6c, 0xbb88, 0xbb8c,
  0xbb90, 0xbba4, 0xbba8, 0xbbac, 0xbbb4, 0xbbb7, 0xbbc0, 0xbbc4,
  0xbbc8, 0xbbd0, 0xbbd3, 0xbbf8, 0xbbf9, 0xbbfc, 0xbbff, 0xbc00,
  0xbc02, 0xbc08, 0xbc09, 0xbc0b, 0xbc0c, 0xbc0d, 0xbc0f, 0xbc11,
  0xbc14, 0xbc15, 0xbc16, 0xbc17, 0xbc18, 0xbc1b, 0xbc1c, 0xbc1d,
  0xbc1e, 0xbc1f, 0xbc24, 0xbc25, 0xbc27, 0xbc29, 0xbc2d, 0xbc30,
  0xbc31, 0xbc34, 0xbc38, 0xbc40, 0xbc41, 0xbc43, 0xbc44, 0xbc45,
  0xbc49, 0xbc4c, 0xbc4d, 0xbc50, 0xbc5d, 0xbc84, 0xbc85, 0xbc88,
  0xbc8b, 0xbc8c, 0xbc8e, 0xbc94, 0xbc95, 0xbc97, 0x3000, 0x3000,
  0x3000, 0x3000, 0xd293, 0xd294, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xd29d, 0x3000, 0xd29f, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd2ad,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd2af,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xd2c1, 0x3000, 0xd2c4, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xd2cd, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xd2d5, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xd2e0, 0x3000, 0x3000, 0x3000, 0x3000, 0xd2e7, 0x3000,
  0xd2e9, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xd2f7, 0x3000, 0x3000, 0xbc99, 0xbc9a,
  0xbca0, 0xbca1, 0xbca4, 0xbca7, 0xbca8, 0xbcb0, 0xbcb1, 0xbcb3,
  0xbcb4, 0xbcb5, 0xbcbc, 0xbcbd, 0xbcc0, 0xbcc4, 0xbccd, 0xbccf,
  0xbcd0, 0xbcd1, 0xbcd5, 0xbcd8, 0xbcdc, 0xbcf4, 0xbcf5, 0xbcf6,
  0xbcf8, 0xbcfc, 0xbd04, 0xbd05, 0xbd07, 0xbd09, 0xbd10, 0xbd14,
  0xbd24, 0xbd2c, 0xbd40, 0xbd48, 0xbd49, 0xbd4c, 0xbd50, 0xbd58,
  0xbd59, 0xbd64, 0xbd68, 0xbd80, 0xbd81, 0xbd84, 0xbd87, 0xbd88,
  0xbd89, 0xbd8a, 0xbd90, 0xbd91, 0xbd93, 0xbd95, 0xbd99, 0xbd9a,
  0xbd9c, 0xbda4, 0xbdb0, 0xbdb8, 0xbdd4, 0xbdd5, 0xbdd8, 0xbddc,
  0xbde9, 0xbdf0, 0xbdf4, 0xbdf8, 0xbe00, 0xbe03, 0xbe05, 0xbe0c,
  0xbe0d, 0xbe10, 0xbe14, 0xbe1c, 0xbe1d, 0xbe1f, 0xbe44, 0xbe45,
  0xbe48, 0xbe4c, 0xbe4e, 0xbe54, 0xbe55, 0xbe57, 0xbe59, 0xbe5a,
  0xbe5b, 0xbe60, 0xbe61, 0xbe64, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd308, 0x3000, 0xd30a,
  0x3000, 0x3000, 0x3000, 0x3000, 0xd313, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd323, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd326, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xd32f, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd341,
  0x3000, 0x3000, 0x3000, 0x3000, 0xd348, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd34c, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd354, 0xd355,
  0x3000, 0xd357, 0x3000, 0xd359, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xd360, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xbe68, 0xbe6a, 0xbe70, 0xbe71,
  0xbe73, 0xbe74, 0xbe75, 0xbe7b, 0xbe7c, 0xbe7d, 0xbe80, 0xbe84,
  0xbe8c, 0xbe8d, 0xbe8f, 0xbe90, 0xbe91, 0xbe98, 0xbe99, 0xbea8,
  0xbed0, 0xbed1, 0xbed4, 0xbed7, 0xbed8, 0xbee0, 0xbee3, 0xbee4,
  0xbee5, 0xbeec, 0xbf01, 0xbf08, 0xbf09, 0xbf18, 0xbf19, 0xbf1b,
  0xbf1c, 0xbf1d, 0xbf40, 0xbf41, 0xbf44, 0xbf48, 0xbf50, 0xbf51,
  0xbf55, 0xbf94, 0xbfb0, 0xbfc5, 0xbfcc, 0xbfcd, 0xbfd0, 0xbfd4,
  0xbfdc, 0xbfdf, 0xbfe1, 0xc03c, 0xc051, 0xc058, 0xc05c, 0xc060,
  0xc068, 0xc069, 0xc090, 0xc091, 0xc094, 0xc098, 0xc0a0, 0xc0a1,
  0xc0a3, 0xc0a5, 0xc0ac, 0xc0ad, 0xc0af, 0xc0b0, 0xc0b3, 0xc0b4,
  0xc0b5, 0xc0b6, 0xc0bc, 0xc0bd, 0xc0bf, 0xc0c0, 0xc0c1, 0xc0c5,
  0xc0c8, 0xc0c9, 0xc0cc, 0xc0d0, 0xc0d8, 0xc0d9, 0xc0db, 0xc0dc,
  0xc0dd, 0xc0e4, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xd383, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd38e, 0x3000,
  0x3000, 0x3000, 0xd395, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xd39f, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xd3ac, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd3b2, 0x3000, 0xd3b5,
  0x3000, 0x3000, 0xd3b9, 0x3000, 0xd3bb, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd3c7, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xd3d1, 0x3000, 0x3000, 0xd3d4,
  0x3000, 0x3000, 0xc0e5, 0xc0e8, 0xc0ec, 0xc0f4, 0xc0f5, 0xc0f7,
  0xc0f9, 0xc100, 0xc104, 0xc108, 0xc110, 0xc115, 0xc11c, 0xc11d,
  0xc11e, 0xc11f, 0xc120, 0xc123, 0xc124, 0xc126, 0xc127, 0xc12c,
  0xc12d, 0xc12f, 0xc130, 0xc131, 0xc136, 0xc138, 0xc139, 0xc13c,
  0xc140, 0xc148, 0xc149, 0xc14b, 0xc14c, 0xc14d, 0xc154, 0xc155,
  0xc158, 0xc15c, 0xc164, 0xc165, 0xc167, 0xc168, 0xc169, 0xc170,
  0xc174, 0xc178, 0xc185, 0xc18c, 0xc18d, 0xc18e, 0xc190, 0xc194,
  0xc196, 0xc19c, 0xc19d, 0xc19f, 0xc1a1, 0xc1a5, 0xc1a8, 0xc1a9,
  0xc1ac, 0xc1b0, 0xc1bd, 0xc1c4, 0xc1c8, 0xc1cc, 0xc1d4, 0xc1d7,
  0xc1d8, 0xc1e0, 0xc1e4, 0xc1e8, 0xc1f0, 0xc1f1, 0xc1f3, 0xc1fc,
  0xc1fd, 0xc200, 0xc204, 0xc20c, 0xc20d, 0xc20f, 0xc211, 0xc218,
  0xc219, 0xc21c, 0xc21f, 0xc220, 0xc228, 0xc229, 0xc22b, 0xc22d,
  0xd3d7, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xd3e0, 0x3000, 0x3000, 0xd3e5, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xd3ee, 0x3000, 0x3000, 0x3000, 0xd3f3, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xd404, 0xd405, 0xd406, 0x3000, 0xd409, 0x3000, 0x3000, 0xd40c,
  0x3000, 0x3000, 0x3000, 0xd410, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xd418, 0xd419, 0x3000, 0xd41b, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xd424, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xc22f, 0xc231, 0xc232, 0xc234, 0xc248, 0xc250, 0xc251, 0xc254,
  0xc258, 0xc260, 0xc265, 0xc26c, 0xc26d, 0xc270, 0xc274, 0xc27c,
  0xc27d, 0xc27f, 0xc281, 0xc288, 0xc289, 0xc290, 0xc298, 0xc29b,
  0xc29d, 0xc2a4, 0xc2a5, 0xc2a8, 0xc2ac, 0xc2ad, 0xc2b4, 0xc2b5,
  0xc2b7, 0xc2b9, 0xc2dc, 0xc2dd, 0xc2e0, 0xc2e3, 0xc2e4, 0xc2eb,
  0xc2ec, 0xc2ed, 0xc2ef, 0xc2f1, 0xc2f6, 0xc2f8, 0xc2f9, 0xc2fb,
  0xc2fc, 0xc300, 0xc308, 0xc309, 0xc30c, 0xc30d, 0xc313, 0xc314,
  0xc315, 0xc318, 0xc31c, 0xc324, 0xc325, 0xc328, 0xc329, 0xc345,
  0xc368, 0xc369, 0xc36c, 0xc370, 0xc372, 0xc378, 0xc379, 0xc37c,
  0xc37d, 0xc384, 0xc388, 0xc38c, 0xc3c0, 0xc3d8, 0xc3d9, 0xc3dc,
  0xc3df, 0xc3e0, 0xc3e2, 0xc3e8, 0xc3e9, 0xc3ed, 0xc3f4, 0xc3f5,
  0xc3f8, 0xc408, 0xc410, 0xc424, 0xc42c, 0xc430, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd441, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xd448, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xd450, 0xd451, 0x3000, 0xd453,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd455,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd45d, 0x3000,
  0x3000, 0x3000, 0x3000, 0xd463, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xd46c, 0x3000, 0x3000, 0xd471, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd495, 0x3000, 0x3000,
  0xd498, 0x3000, 0x3000, 0xd49b, 0xd49c, 0x3000, 0xc434, 0xc43c,
  0xc43d, 0xc448, 0xc464, 0xc465, 0xc468, 0xc46c, 0xc474, 0xc475,
  0xc479, 0xc480, 0xc494, 0xc49c, 0xc4b8, 0xc4bc, 0xc4e9, 0xc4f0,
  0xc4f1, 0xc4f4, 0xc4f8, 0xc4fa, 0xc4ff, 0xc500, 0xc501, 0xc50c,
  0xc510, 0xc514, 0xc51c, 0xc528, 0xc529, 0xc52c, 0xc530, 0xc538,
  0xc539, 0xc53b, 0xc53d, 0xc544, 0xc545, 0xc548, 0xc549, 0xc54a,
  0xc54c, 0xc54d, 0xc54e, 0xc553, 0xc554, 0xc555, 0xc557, 0xc558,
  0xc559, 0xc55d, 0xc55e, 0xc560, 0xc561, 0xc564, 0xc568, 0xc570,
  0xc571, 0xc573, 0xc574, 0xc575, 0xc57c, 0xc57d, 0xc580, 0xc584,
  0xc587, 0xc58c, 0xc58d, 0xc58f, 0xc591, 0xc595, 0xc597, 0xc598,
  0xc59c, 0xc5a0, 0xc5a9, 0xc5b4, 0xc5b5, 0xc5b8, 0xc5b9, 0xc5bb,
  0xc5bc, 0xc5bd, 0xc5be, 0xc5c4, 0xc5c5, 0xc5c6, 0xc5c7, 0xc5c8,
  0xc5c9, 0xc5ca, 0xc5cc, 0xc5ce, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xd4a4, 0xd4a5, 0x3000, 0xd4a7, 0xd4a8, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd4b0, 0xd4b1, 0x3000,
  0x3000, 0xd4b4, 0x3000, 0x3000, 0x3000, 0xd4b8, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xd4c0, 0xd4c1, 0x3000, 0xd4c3, 0x3000,
  0xd4c5, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd4cd,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xd4dd, 0x3000, 0x3000, 0xd4e1, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xd4e9, 0x3000, 0x3000, 0x3000,
  0x3000, 0xd4ef, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xd4f9, 0x3000, 0x3000, 0xc5d0, 0xc5d1, 0xc5d4, 0xc5d8,
  0xc5e0, 0xc5e1, 0xc5e3, 0xc5e5, 0xc5ec, 0xc5ed, 0xc5ee, 0xc5f0,
  0xc5f4, 0xc5f6, 0xc5f7, 0xc5fc, 0xc5fd, 0xc5fe, 0xc5ff, 0xc600,
  0xc601, 0xc605, 0xc606, 0xc607, 0xc608, 0xc60c, 0xc610, 0xc618,
  0xc619, 0xc61b, 0xc61c, 0xc624, 0xc625, 0xc628, 0xc62c, 0xc62d,
  0xc62e, 0xc630, 0xc633, 0xc634, 0xc635, 0xc637, 0xc639, 0xc63b,
  0xc640, 0xc641, 0xc644, 0xc648, 0xc650, 0xc651, 0xc653, 0xc654,
  0xc655, 0xc65c, 0xc65d, 0xc660, 0xc66c, 0xc66f, 0xc671, 0xc678,
  0xc679, 0xc67c, 0xc680, 0xc688, 0xc689, 0xc68b, 0xc68d, 0xc694,
  0xc695, 0xc698, 0xc69c, 0xc6a4, 0xc6a5, 0xc6a7, 0xc6a9, 0xc6b0,
  0xc6b1, 0xc6b4, 0xc6b8, 0xc6b9, 0xc6ba, 0xc6c0, 0xc6c1, 0xc6c3,
  0xc6c5, 0xc6cc, 0xc6cd, 0xc6d0, 0xc6d4, 0xc6dc, 0xc6dd, 0xc6e0,
  0xc6e1, 0xc6e8, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xd505, 0x3000, 0x3000, 0x3000, 0x3000, 0xd50b, 0x3000, 0x3000,
  0x3000, 0xd510, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd519,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xd520, 0xd521, 0x3000, 0x3000,
  0xd524, 0x3000, 0x3000, 0x3000, 0xd528, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xd530, 0xd531, 0x3000, 0xd533,
  0x3000, 0xd535, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xd543, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd555,
  0xd556, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd55f, 0xd561,
  0x3000, 0xd563, 0xc6e9, 0xc6ec, 0xc6f0, 0xc6f8, 0xc6f9, 0xc6fd,
  0xc704, 0xc705, 0xc708, 0xc70c, 0xc714, 0xc715, 0xc717, 0xc719,
  0xc720, 0xc721, 0xc724, 0xc728, 0xc730, 0xc731, 0xc733, 0xc735,
  0xc737, 0xc73c, 0xc73d, 0xc740, 0xc744, 0xc74a, 0xc74c, 0xc74d,
  0xc74f, 0xc751, 0xc752, 0xc753, 0xc754, 0xc755, 0xc756, 0xc757,
  0xc758, 0xc75c, 0xc760, 0xc768, 0xc76b, 0xc774, 0xc775, 0xc778,
  0xc77c, 0xc77d, 0xc77e, 0xc783, 0xc784, 0xc785, 0xc787, 0xc788,
  0xc789, 0xc78a, 0xc78e, 0xc790, 0xc791, 0xc794, 0xc796, 0xc797,
  0xc798, 0xc79a, 0xc7a0, 0xc7a1, 0xc7a3, 0xc7a4, 0xc7a5, 0xc7a6,
  0xc7ac, 0xc7ad, 0xc7b0, 0xc7b4, 0xc7bc, 0xc7bd, 0xc7bf, 0xc7c0,
  0xc7c1, 0xc7c8, 0xc7c9, 0xc7cc, 0xc7ce, 0xc7d0, 0xc7d8, 0xc7dd,
  0xc7e4, 0xc7e8, 0xc7ec, 0xc800, 0xc801, 0xc804, 0xc808, 0xc80a,
  0xd564, 0x3000, 0xd567, 0x3000, 0xd56c, 0xd56e, 0x3000, 0x3000,
  0xd571, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd57b,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xd58d, 0x3000, 0x3000, 0xd591, 0x3000, 0x3000, 0xd594,
  0x3000, 0x3000, 0x3000, 0xd598, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xd5a0, 0xd5a1, 0x3000, 0xd5a3, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xd5ab, 0xd5ac, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xd5c0, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xc810, 0xc811, 0xc813, 0xc815, 0xc816, 0xc81c, 0xc81d, 0xc820,
  0xc824, 0xc82c, 0xc82d, 0xc82f, 0xc831, 0xc838, 0xc83c, 0xc840,
  0xc848, 0xc849, 0xc84c, 0xc84d, 0xc854, 0xc870, 0xc871, 0xc874,
  0xc878, 0xc87a, 0xc880, 0xc881, 0xc883, 0xc885, 0xc886, 0xc887,
  0xc88b, 0xc88c, 0xc88d, 0xc894, 0xc89d, 0xc89f, 0xc8a1, 0xc8a8,
  0xc8bc, 0xc8bd, 0xc8c4, 0xc8c8, 0xc8cc, 0xc8d4, 0xc8d5, 0xc8d7,
  0xc8d9, 0xc8e0, 0xc8e1, 0xc8e4, 0xc8f5, 0xc8fc, 0xc8fd, 0xc900,
  0xc904, 0xc905, 0xc906, 0xc90c, 0xc90d, 0xc90f, 0xc911, 0xc918,
  0xc92c, 0xc934, 0xc950, 0xc951, 0xc954, 0xc958, 0xc960, 0xc961,
  0xc963, 0xc96c, 0xc970, 0xc974, 0xc97c, 0xc988, 0xc989, 0xc98c,
  0xc990, 0xc998, 0xc999, 0xc99b, 0xc99d, 0xc9c0, 0xc9c1, 0xc9c4,
  0xc9c7, 0xc9c8, 0xc9ca, 0xc9d0, 0xc9d1, 0xc9d3, 0x3000, 0x3000,
  0x3000, 0x3000, 0xd5cf, 0xd5d1, 0x3000, 0x3000, 0x3000, 0x3000,
  0xd5d7, 0x3000, 0xd5dc, 0x3000, 0x3000, 0x3000, 0xd5e1, 0xd5e2,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd5eb, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xd5f8, 0x3000, 0x3000, 0x3000,
  0x3000, 0xd5fe, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd607,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xd61d, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xd62c, 0x3000, 0xd62f, 0xd630, 0xd631, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd63b, 0xc9d5, 0xc9d6,
  0xc9d9, 0xc9da, 0xc9dc, 0xc9dd, 0xc9e0, 0xc9e2, 0xc9e4, 0xc9e7,
  0xc9ec, 0xc9ed, 0xc9ef, 0xc9f0, 0xc9f1, 0xc9f8, 0xc9f9, 0xc9fc,
  0xca00, 0xca08, 0xca09, 0xca0b, 0xca0c, 0xca0d, 0xca14, 0xca18,
  0xca29, 0xca4c, 0xca4d, 0xca50, 0xca54, 0xca5c, 0xca5d, 0xca5f,
  0xca60, 0xca61, 0xca68, 0xca7d, 0xca84, 0xca98, 0xcabc, 0xcabd,
  0xcac0, 0xcac4, 0xcacc, 0xcacd, 0xcacf, 0xcad1, 0xcad3, 0xcad8,
  0xcad9, 0xcae0, 0xcaec, 0xcaf4, 0xcb08, 0xcb10, 0xcb14, 0xcb18,
  0xcb20, 0xcb21, 0xcb41, 0xcb48, 0xcb49, 0xcb4c, 0xcb50, 0xcb58,
  0xcb59, 0xcb5d, 0xcb64, 0xcb78, 0xcb79, 0xcb9c, 0xcbb8, 0xcbd4,
  0xcbe4, 0xcbe7, 0xcbe9, 0xcc0c, 0xcc0d, 0xcc10, 0xcc14, 0xcc1c,
  0xcc1d, 0xcc21, 0xcc22, 0xcc27, 0xcc28, 0xcc29, 0xcc2c, 0xcc2e,
  0xcc30, 0xcc38, 0xcc39, 0xcc3b, 0xd63d, 0x3000, 0xd63f, 0xd641,
  0x3000, 0x3000, 0xd644, 0x3000, 0xd647, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xd65b, 0x3000, 0x3000, 0x3000, 0xd660, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd664, 0xd665,
  0xd666, 0xd668, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xd677, 0xd678, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd680, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xd681, 0x3000, 0xd684, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xd693, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0xd69c, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xd6a9, 0x3000, 0xcc3c, 0xcc3d, 0xcc3e, 0xcc44,
  0xcc45, 0xcc48, 0xcc4c, 0xcc54, 0xcc55, 0xcc57, 0xcc58, 0xcc59,
  0xcc60, 0xcc64, 0xcc66, 0xcc68, 0xcc70, 0xcc75, 0xcc98, 0xcc99,
  0xcc9c, 0xcca0, 0xcca8, 0xcca9, 0xccab, 0xccac, 0xccad, 0xccb4,
  0xccb5, 0xccb8, 0xccbc, 0xccc4, 0xccc5, 0xccc7, 0xccc9, 0xccd0,
  0xccd4, 0xcce4, 0xccec, 0xccf0, 0xcd01, 0xcd08, 0xcd09, 0xcd0c,
  0xcd10, 0xcd18, 0xcd19, 0xcd1b, 0xcd1d, 0xcd24, 0xcd28, 0xcd2c,
  0xcd39, 0xcd5c, 0xcd60, 0xcd64, 0xcd6c, 0xcd6d, 0xcd6f, 0xcd71,
  0xcd78, 0xcd88, 0xcd94, 0xcd95, 0xcd98, 0xcd9c, 0xcda4, 0xcda5,
  0xcda7, 0xcda9, 0xcdb0, 0xcdc4, 0xcdcc, 0xcdd0, 0xcde8, 0xcdec,
  0xcdf0, 0xcdf8, 0xcdf9, 0xcdfb, 0xcdfd, 0xce04, 0xce08, 0xce0c,
  0xce14, 0xce19, 0xce20, 0xce21, 0xce24, 0xce28, 0xce30, 0xce31,
  0xce33, 0xce35, 0x3000, 0x3000, 0x3000, 0xd6af, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd6b8, 0x3000, 0x3000,
  0xd6bd, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0xd6cb, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xd6cd, 0xd6ce, 0x3000, 0x3000, 0xd6d2, 0xd6d3,
  0xd6d5, 0x3000, 0x3000, 0x3000, 0x3000, 0xd6dc, 0xd6dd, 0x3000,
  0x3000, 0xd6e1, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xd6f1, 0x3000, 0xd6f3, 0xd6f4,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xd703, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xd70c, 0xd70d, 0x3000, 0xd70f, 0xd710, 0x3000,
  0x3000, 0x3000, 0xce58, 0xce59, 0xce5c, 0xce5f, 0xce60, 0xce61,
  0xce68, 0xce69, 0xce6b, 0xce6d, 0xce74, 0xce75, 0xce78, 0xce7c,
  0xce84, 0xce85, 0xce87, 0xce89, 0xce90, 0xce91, 0xce94, 0xce98,
  0xcea0, 0xcea1, 0xcea3, 0xcea4, 0xcea5, 0xceac, 0xcead, 0xcec1,
  0xcee4, 0xcee5, 0xcee8, 0xceeb, 0xceec, 0xcef4, 0xcef5, 0xcef7,
  0xcef8, 0xcef9, 0xcf00, 0xcf01, 0xcf04, 0xcf08, 0xcf10, 0xcf11,
  0xcf13, 0xcf15, 0xcf1c, 0xcf20, 0xcf24, 0xcf2c, 0xcf2d, 0xcf2f,
  0xcf30, 0xcf31, 0xcf38, 0xcf54, 0xcf55, 0xcf58, 0xcf5c, 0xcf64,
  0xcf65, 0xcf67, 0xcf69, 0xcf70, 0xcf71, 0xcf74, 0xcf78, 0xcf80,
  0xcf85, 0xcf8c, 0xcfa1, 0xcfa8, 0xcfb0, 0xcfc4, 0xcfe0, 0xcfe1,
  0xcfe4, 0xcfe8, 0xcff0, 0xcff1, 0xcff3, 0xcff5, 0xcffc, 0xd000,
  0xd004, 0xd011, 0xd018, 0xd02d, 0xd034, 0xd035, 0xd038, 0xd03c,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd71f,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xd73a, 0xd73b, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xd743, 0xd745, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd74d,
  0x3000, 0x3000, 0x3000, 0x3000, 0xd755, 0x3000, 0x3000, 0x3000,
  0xd75d, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xd75f, 0x3000, 0x3000, 0x3000, 0x3000, 0xd768, 0xd76a, 0xd76b,
  0xd76d, 0x3000, 0x3000, 0x3000, 0x3000, 0xd773, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd77f, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0xd044, 0xd045, 0xd047, 0xd049, 0xd050, 0xd054, 0xd058, 0xd060,
  0xd06c, 0xd06d, 0xd070, 0xd074, 0xd07c, 0xd07d, 0xd081, 0xd0a4,
  0xd0a5, 0xd0a8, 0xd0ac, 0xd0b4, 0xd0b5, 0xd0b7, 0xd0b9, 0xd0c0,
  0xd0c1, 0xd0c4, 0xd0c8, 0xd0c9, 0xd0d0, 0xd0d1, 0xd0d3, 0xd0d4,
  0xd0d5, 0xd0dc, 0xd0dd, 0xd0e0, 0xd0e4, 0xd0ec, 0xd0ed, 0xd0ef,
  0xd0f0, 0xd0f1, 0xd0f8, 0xd10d, 0xd130, 0xd131, 0xd134, 0xd138,
  0xd13a, 0xd140, 0xd141, 0xd143, 0xd144, 0xd145, 0xd14c, 0xd14d,
  0xd150, 0xd154, 0xd15c, 0xd15d, 0xd15f, 0xd161, 0xd168, 0xd16c,
  0xd17c, 0xd184, 0xd188, 0xd1a0, 0xd1a1, 0xd1a4, 0xd1a8, 0xd1b0,
  0xd1b1, 0xd1b3, 0xd1b5, 0xd1ba, 0xd1bc, 0xd1c0, 0xd1d8, 0xd1f4,
  0xd1f8, 0xd207, 0xd209, 0xd210, 0xd22c, 0xd22d, 0xd230, 0xd234,
  0xd23c, 0xd23d, 0xd23f, 0xd241, 0xd248, 0xd25c, 0x3000, 0x3000,
  0xd78f, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd797,
  0x3000, 0xd79c, 0x3000, 0x3000, 0xd7a0, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xd264, 0xd280,
  0xd281, 0xd284, 0xd288, 0xd290, 0xd291, 0xd295, 0xd29c, 0xd2a0,
  0xd2a4, 0xd2ac, 0xd2b1, 0xd2b8, 0xd2b9, 0xd2bc, 0xd2bf, 0xd2c0,
  0xd2c2, 0xd2c8, 0xd2c9, 0xd2cb, 0xd2d4, 0xd2d8, 0xd2dc, 0xd2e4,
  0xd2e5, 0xd2f0, 0xd2f1, 0xd2f4, 0xd2f8, 0xd300, 0xd301, 0xd303,
  0xd305, 0xd30c, 0xd30d, 0xd30e, 0xd310, 0xd314, 0xd316, 0xd31c,
  0xd31d, 0xd31f, 0xd320, 0xd321, 0xd325, 0xd328, 0xd329, 0xd32c,
  0xd330, 0xd338, 0xd339, 0xd33b, 0xd33c, 0xd33d, 0xd344, 0xd345,
  0xd37c, 0xd37d, 0xd380, 0xd384, 0xd38c, 0xd38d, 0xd38f, 0xd390,
  0xd391, 0xd398, 0xd399, 0xd39c, 0xd3a0, 0xd3a8, 0xd3a9, 0xd3ab,
  0xd3ad, 0xd3b4, 0xd3b8, 0xd3bc, 0xd3c4, 0xd3c5, 0xd3c8, 0xd3c9,
  0xd3d0, 0xd3d8, 0xd3e1, 0xd3e3, 0xd3ec, 0xd3ed, 0xd3f0, 0xd3f4,
  0xd3fc, 0xd3fd, 0xd3ff, 0xd401, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0xd408, 0xd41d, 0xd440, 0xd444,
  0xd45c, 0xd460, 0xd464, 0xd46d, 0xd46f, 0xd478, 0xd479, 0xd47c,
  0xd47f, 0xd480, 0xd482, 0xd488, 0xd489, 0xd48b, 0xd48d, 0xd494,
  0xd4a9, 0xd4cc, 0xd4d0, 0xd4d4, 0xd4dc, 0xd4df, 0xd4e8, 0xd4ec,
  0xd4f0, 0xd4f8, 0xd4fb, 0xd4fd, 0xd504, 0xd508, 0xd50c, 0xd514,
  0xd515, 0xd517, 0xd53c, 0xd53d, 0xd540, 0xd544, 0xd54c, 0xd54d,
  0xd54f, 0xd551, 0xd558, 0xd559, 0xd55c, 0xd560, 0xd565, 0xd568,
  0xd569, 0xd56b, 0xd56d, 0xd574, 0xd575, 0xd578, 0xd57c, 0xd584,
  0xd585, 0xd587, 0xd588, 0xd589, 0xd590, 0xd5a5, 0xd5c8, 0xd5c9,
  0xd5cc, 0xd5d0, 0xd5d2, 0xd5d8, 0xd5d9, 0xd5db, 0xd5dd, 0xd5e4,
  0xd5e5, 0xd5e8, 0xd5ec, 0xd5f4, 0xd5f5, 0xd5f7, 0xd5f9, 0xd600,
  0xd601, 0xd604, 0xd608, 0xd610, 0xd611, 0xd613, 0xd614, 0xd615,
  0xd61c, 0xd620, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  0x3000, 0x3000, 0xd624, 0xd62d, 0xd638, 0xd639, 0xd63c, 0xd640,
  0xd645, 0xd648, 0xd649, 0xd64b, 0xd64d, 0xd651, 0xd654, 0xd655,
  0xd658, 0xd65c, 0xd667, 0xd669, 0xd670, 0xd671, 0xd674, 0xd683,
  0xd685, 0xd68c, 0xd68d, 0xd690, 0xd694, 0xd69d, 0xd69f, 0xd6a1,
  0xd6a8, 0xd6ac, 0xd6b0, 0xd6b9, 0xd6bb, 0xd6c4, 0xd6c5, 0xd6c8,
  0xd6cc, 0xd6d1, 0xd6d4, 0xd6d7, 0xd6d9, 0xd6e0, 0xd6e4, 0xd6e8,
  0xd6f0, 0xd6f5, 0xd6fc, 0xd6fd, 0xd700, 0xd704, 0xd711, 0xd718,
  0xd719, 0xd71c, 0xd720, 0xd728, 0xd729, 0xd72b, 0xd72d, 0xd734,
  0xd735, 0xd738, 0xd73c, 0xd744, 0xd747, 0xd749, 0xd750, 0xd751,
  0xd754, 0xd756, 0xd757, 0xd758, 0xd759, 0xd760, 0xd761, 0xd763,
  0xd765, 0xd769, 0xd76c, 0xd770, 0xd774, 0xd77c, 0xd77d, 0xd781,
  0xd788, 0xd789, 0xd78c, 0xd790, 0xd798, 0xd799, 0xd79b, 0xd79d
};

uint16_t Cp949::Convert(uint16_t ch) const {
  if (ch <= 0x7f) {
    return ch;
  } else {
    int c1 = ((ch >> 8) & 0xff) - 0x81;
    int c2 = (ch & 0xff) - 0x41;
    return ksc_to_uni[c1 * 190 + c2];
  }
}

std::wstring Cp949::ConvertString(const std::string& in_string) const {
  std::wstring rv;
  rv.reserve(in_string.size());

  const char* s = in_string.c_str();
  while (*s) {
    if (*s < 0x80) {
      rv += Convert(*s++);
    } else {
      rv += Convert((s[0] << 8) | s[1]);
      s += 2;
    }
  }

  return rv;
}

#else

uint16_t Cp949::JisDecode(uint16_t ch) const {
  return ch;
}

void Cp949::JisEncodeString(const char* src, char* buf, size_t buflen) const {
  std::strncpy(buf, src, buflen);
}

uint16_t Cp949::Convert(uint16_t ch) const {
  return ch;
}

std::wstring Cp949::ConvertString(const std::string& s) const {
  return NULL;
}

#endif
