#ifndef _RHEO_RIESZ_H
#define _RHEO_RIESZ_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
#include "rheolef/field.h"
#include "rheolef/field_element.h"
#include "rheolef/field_assembly.h"
#include "rheolef/band.h"
#include <boost/type_traits/is_class.hpp>
#include <boost/type_traits/is_function.hpp>
#include <boost/mpl/bool.hpp>
#include <boost/mpl/and.hpp>
#include <boost/mpl/or.hpp>
#include <boost/mpl/not.hpp>

namespace rheolef { 

// --------------------------------------------------------------------------
// utility to determine whether a template arg is a function or a constant
// --------------------------------------------------------------------------
namespace details {
  // build a function that returns a constant
  template<class T1, class T2>
  struct f_constant : std::unary_function<T1,T2> {
    const T2& operator() (const T1& x) const { return c; }
    f_constant (const T2& c0) : c(c0) {}
    T2 c;
  };
  // switch for a constant, or a pointer-to-function or a class-function:
  template<class T> struct is_scalar        : boost::mpl::false_ {}; 
  template<>        struct is_scalar<int>   : boost::mpl::true_ {};
  template<>        struct is_scalar<const int>   : boost::mpl::true_ {};
  template<>        struct is_scalar<size_t>: boost::mpl::true_ {};
  template<>        struct is_scalar<Float> : boost::mpl::true_ {};

  template<class T> struct is_point                  : boost::mpl::false_ {};
  template<class T> struct is_point<point_basic<T> > : boost::mpl::true_ {};

  template<class Function>
  struct is_function :
    boost::mpl::and_ <
      boost::mpl::or_ <
        boost::is_class<Function>,
        boost::is_pointer<Function>,
        boost::is_function<Function>
      >,
      boost::mpl::not_ <
        boost::mpl::or_ <
          is_scalar<Function>,
          is_point<Function>
        >
      >
    >
  {};
  template<class Constant>
  struct constant_promote { typedef Constant type; };
  template<>        struct constant_promote<int>    { typedef Float type; };
  template<>        struct constant_promote<size_t> { typedef Float type; };
    
} // namespace details
// --------------------------------------------------------------------------
// generic implementation
// --------------------------------------------------------------------------
template <class T, class M, class Function>
field_basic<T,M>
riesz_internal (
    const space_basic<T,M>&         Xh,
    const Function&                 f,
    const quadrature_option_type&   qopt_0,
    const geo_basic<T,M>&           dom,
    const field_element<T,M>&       field_e)
{
  quadrature_option_type qopt = qopt_0;
  if (qopt.get_family() == quadrature_option_type::max_family) {
    qopt.set_family (quadrature_option_type::gauss);
    size_t k = Xh.get_numbering().degree();
    if (k == 0) k = 1;
    qopt.set_order (2*k-1); // see Raviart and Thomas, Masson 1988, page 130
    // TODO: when arg is a constant, use the central point formulae (1 node only)
  }
  field_e.initialize (Xh, qopt);
  field_basic<T,M> lh (Xh, 0.);
  field_assembly (dom, field_e, f, lh);
  return lh;
}
// --------------------------------------------------------------------------
// implementations when the template arg is a function
// --------------------------------------------------------------------------
template <class T, class M, class Function>
field_basic<T,M>
riesz_tag (
    const space_basic<T,M>&       Xh,
    const Function&               f,
    const quadrature_option_type& qopt,
    const geo_basic<T,M>&         dom,
    boost::mpl::true_)
{
  field_element<T,M> field_e;
  return riesz_internal (Xh, f, qopt, dom, field_e);
}
template <class T, class M, class Function>
field_basic<T,M>
riesz_tag (
    const space_basic<T,M>&       Xh,
    const Function&               f,
    const quadrature_option_type& qopt,
    const band_basic<T,M>&        gh,
    boost::mpl::true_)
{
  field_element<T,M> field_e;
  field_e.set_band (gh);
  return riesz_internal (Xh, f, qopt, gh.level_set(), field_e);
}
// --------------------------------------------------------------------------
// implementation when the template arg is a constant
// => thanks to a wrapper class, we go to the first case
// --------------------------------------------------------------------------
template <class T, class M, class Constant>
field_basic<T,M>
riesz_tag (
    const space_basic<T,M>&       Xh,
    const Constant&               value,
    const quadrature_option_type& qopt,
    const geo_basic<T,M>&         dom,
    boost::mpl::false_)
{
  field_element<T,M> field_e;
  typedef typename field_basic<T,M>::float_type float_type;
  typedef typename details::constant_promote<Constant>::type constant_type; // int to Float, but conserve point_basic<T> & tensor
  return riesz_internal (Xh, details::f_constant<point_basic<float_type>,constant_type>(value),
			     qopt, dom, field_e);
}
template <class T, class M, class Constant>
field_basic<T,M>
riesz_tag (
    const space_basic<T,M>&       Xh,
    const Constant&               value,
    const quadrature_option_type& qopt,
    const band_basic<T,M>&        gh,
    boost::mpl::false_)
{
  field_element<T,M> field_e;
  field_e.set_band (gh);
  typedef typename field_basic<T,M>::float_type float_type;
  typedef typename details::constant_promote<Constant>::type constant_type; // int to Float, but conserve point_basic<T> & tensor
  return riesz_internal (Xh, details::f_constant<point_basic<float_type>,constant_type>(value),
			     qopt, gh.level_set(), field_e);
}

/*Class:riesz
NAME: @code{riesz} - integrate a function by using quadrature formulae
@findex  riesz
@cindex  riesz representer
@cindex  quadrature formulae
@clindex space
@clindex field

DESCRIPTION:
  The function @code{riesz} implements the
  approximation of an integral by using quadrature formulae.
SYNOPSYS:
 template <class Function>
 field riesz (const space& Xh, const Function& f,
    	quadrature_option_type qopt = default_value);

 template <class Function>
 field riesz (const space& Xh, const Function& f,
	const geo& domain,
    	quadrature_option_type qopt = default_value);
EXAMPLE:
@noindent
  The following code compute the Riesz representant, denoted
  by @code{lh} of f(x), and the integral of f over the domain omega:
@example
  Float f(const point& x);
  ...
  space Xh (omega_h, "P1");
  field lh = riesz (Xh, f);
  Float int_f = dot(lh, 1);
@end example
@cindex  integrate
  The Riesz representer is the @code{lh} vector of values:
@example
  lh(i) = integrate f(x) phi_i(x) dx
@end example
  where phi_i is the i-th basis function in Xh
  and the integral is evaluated by using a quadrature formulae.
  By default the quadrature formule is the Gauss one with
  the order equal to the polynomial order of Xh.
  Alternative quadrature formulae and order is available
  by passing an optional variable to riesz.
OPTIONS:
  An optional argument specifies the quadrature formulae used
  for the computation of the integral.
  The domain of integration is by default the mesh associated to
  the finite element space.
  An alternative domain @code{dom}, e.g. a part of 
  the boundary can be supplied as an extra argument.
  This domain can be also a @code{band} associated to
  the banded level set method.
End: */

//<riesz:
template <class T, class M, class Function>
field_basic<T,M>
riesz (
    const space_basic<T,M>&       Xh,
    const Function&               f,
    const quadrature_option_type& qopt
       = quadrature_option_type(quadrature_option_type::max_family,0))
//>riesz:
{
  return riesz_tag (Xh, f, qopt, Xh.get_geo(), typename details::is_function<Function>::type());
}
//<riesz:
template <class T, class M, class Function>
field_basic<T,M>
riesz (
    const space_basic<T,M>&       Xh,
    const Function&               f,
    const geo_basic<T,M>&         dom,
    const quadrature_option_type& qopt
       = quadrature_option_type(quadrature_option_type::max_family,0))
//>riesz:
{
  return riesz_tag (Xh, f, qopt, dom, typename details::is_function<Function>::type());
}
//<riesz:
template <class T, class M, class Function>
field_basic<T,M>
riesz (
    const space_basic<T,M>&       Xh,
    const Function&               f,
    std::string                   dom_name,
    const quadrature_option_type& qopt
       = quadrature_option_type(quadrature_option_type::max_family,0))
//>riesz:
{
  const geo_basic<T,M>& omega = Xh.get_geo();
  return riesz (Xh, f, omega[dom_name], qopt);
}
//<riesz:
template <class T, class M, class Function>
field_basic<T,M>
riesz (
    const space_basic<T,M>&       Xh,
    const Function&               f,
    const band_basic<T,M>&        gh,
    const quadrature_option_type& qopt
       = quadrature_option_type(quadrature_option_type::max_family,0))
//>riesz:
{
  return riesz_tag (Xh, f, qopt, gh, typename details::is_function<Function>::type());
}

}// namespace rheolef
#endif // _RHEO_RIESZ_H
