#ifndef _RHEOLEF_PIOLA_H
#define _RHEOLEF_PIOLA_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
// 
// piola on a predefined point set: hat_x[q], q=0..nq
//
#include "rheolef/geo.h"
#include "rheolef/basis_on_pointset.h"
namespace rheolef { 

template<class T, class M, class BasisOnPointset>
point_basic<T>
piola_transformation (
  const geo_basic<T,M>&         omega,
  const BasisOnPointset&        piola_on_pointset,
  reference_element             hat_K,
  const std::vector<size_t>&    dis_inod,
  size_t                        q)
{
  typedef typename geo_basic<T,M>::size_type size_type;
  size_type d = omega.dimension();
  size_type loc_inod = 0;
  point_basic<T> xq;
  for (typename BasisOnPointset::const_iterator
	  iter = piola_on_pointset.begin(hat_K, q),
          last = piola_on_pointset.end  (hat_K, q);
          iter != last;
          iter++, loc_inod++) {
    const T& cnod = *iter;
    const point_basic<T>& xnod = omega.dis_node (dis_inod[loc_inod]);
    for (size_type i = 0; i < d; i++) {
      xq[i] += cnod*xnod[i];
    }
  }
  return xq;
}
// ------------------------------------------
// jacobian of the piola transformation
// at an aritrarily point hat_x
// ------------------------------------------
template<class T, class M, class BasisOnPointset>
void
jacobian_piola_transformation (
  const geo_basic<T,M>&         omega,
  const BasisOnPointset&        piola_on_pointset,
  reference_element             hat_K,
  const std::vector<size_t>&    dis_inod,
  const point_basic<T>&         hat_x,
        tensor_basic<T>&        DF)
{
  typedef typename geo_basic<T,M>::size_type size_type;
  size_type d = omega.dimension();
  size_type map_d = hat_K.dimension();
  size_type loc_inod = 0;
  DF.fill (0);
  piola_on_pointset.evaluate_grad (hat_K, hat_x);
  for (typename BasisOnPointset::const_iterator_grad
        first_grad_tr = piola_on_pointset.begin_grad(),
         last_grad_tr = piola_on_pointset.end_grad();
        first_grad_tr != last_grad_tr;
        first_grad_tr++, loc_inod++) {
    const point_basic<T>& xnod = omega.dis_node (dis_inod[loc_inod]);
    cumul_otimes (DF, xnod, *first_grad_tr, d, map_d);
  }
}
// ------------------------------------------
// jacobian of the piola transformation
// at a quadrature point
// ------------------------------------------
template<class T, class M, class BasisOnPointset>
void
jacobian_piola_transformation (
  const geo_basic<T,M>&         omega,
  const BasisOnPointset&        piola_on_pointset,
  reference_element             hat_K,
  const std::vector<size_t>&    dis_inod,
  size_t                        q,
        tensor_basic<T>&        DF)
{
  typedef typename geo_basic<T,M>::size_type size_type;
  size_type d = omega.dimension();
  size_type map_d = hat_K.dimension();
  size_type loc_inod = 0;
  DF.fill (0);
  for (typename BasisOnPointset::const_iterator_grad
  	first_grad_tr = piola_on_pointset.begin_grad (hat_K, q),
  	 last_grad_tr = piola_on_pointset.end_grad   (hat_K, q);
        first_grad_tr != last_grad_tr;
	first_grad_tr++, loc_inod++) {
    const point_basic<T>& xnod = omega.dis_node (dis_inod[loc_inod]);
    cumul_otimes (DF, xnod, *first_grad_tr, d, map_d);
  }
}
template <class T>
inline
T
det_jacobian_piola_transformation (const tensor_basic<T>& DF, size_t d , size_t map_d)
{
    if (d == map_d) {
      return DF.determinant (map_d);
    }
    /* surface jacobian: references:
     * Spectral/hp element methods for CFD
     * G. E. M. Karniadakis and S. J. Sherwin
     * Oxford university press
     * 1999
     * page 165
     */
    switch (map_d) {
      case 0: return 1;
      case 1: return norm(DF.col(0));
      case 2: return norm(vect(DF.col(0), DF.col(1)));
      default:
        error_macro ("det_jacobian_piola_transformation: unsupported element dimension "
            << map_d << " in " << d << "D mesh.");
        return 0;
    }
}
// The pseudo inverse extend inv(DF) for face in 3d or edge in 2d
//  i.e. usefull for Laplacian-Beltrami and others surfacic forms.
//
// pinvDF (hat_xq) = inv DF, if tetra in 3d, tri in 2d, etc
//                  = pseudo-invese, when tri in 3d, edge in 2 or 3d
// e.g. on 3d face : pinvDF*DF = [1, 0, 0; 0, 1, 0; 0, 0, 0]
//
// let DF = [u, v, w], where u, v, w are the column vectors of DF
// then det(DF) = mixt(u,v,w)
// and det(DF)*inv(DF)^T = [v^w, w^u, u^v] where u^v = vect(u,v)
//
// application:
// if K=triangle(a,b,c) then u=ab=b-a, v=ac=c-a and w = n = u^v/|u^v|.
// Thus DF = [ab,ac,n] and det(DF)=|ab^ac|
// and inv(DF)^T = [ac^n/|ab^ac|, -ab^n/|ab^ac|, n]
// The pseudo-inverse is obtained by remplacing the last column n by zero.
//
template<class T>
inline
tensor_basic<T>
pseudo_inverse_jacobian_piola_transformation (
    const tensor_basic<T>& DF,
    size_t d,
    size_t map_d)
{
    if (d == map_d) {
      return inv(DF, map_d);
    }
    tensor_basic<T> invDF;
    switch (map_d) {
      case 0: { // point in 1D
          invDF(0,0) = 1;
          return invDF;
      }
      case 1: { // segment in 2D
          point_basic<T> t = DF.col(0);
          invDF.set_row (t/norm2(t), 0, d);
          return invDF;
      }
      case 2: {
          point_basic<T> t0 = DF.col(0);
          point_basic<T> t1 = DF.col(1);
          point_basic<T> n = vect(t0,t1);
          T det2 = norm2(n);
          point_basic<T> v0 =   vect(t1,n)/det2;
          point_basic<T> v1 = - vect(t0,n)/det2;
          invDF.set_row (v0, 0, d);
          invDF.set_row (v1, 1, d);
          return invDF;
      }
      default:
          error_macro ("pseudo_inverse_jacobian_piola_transformation: unsupported element dimension "
	    << map_d << " in " << d << "D mesh.");
          return invDF;
    }
}
// DF: hat_x --> DF(hat_x) on hat_K
template<class T, class M>
void
jacobian_piola_transformation (
  const geo_basic<T,M>&         omega,
  reference_element             hat_K,
  const std::vector<size_t>&    dis_inod,
  const point_basic<T>&         hat_x,
        tensor_basic<T>&        DF);

// F^{-1}: x --> hat_x  on K
template<class T, class M>
point_basic<T>
inverse_piola_transformation (
  const geo_basic<T,M>&         omega,
  reference_element             hat_K,
  const std::vector<size_t>&    dis_inod,
  const point_basic<T>&         x);

// compute: P = I - nxn, the tangential projector on a map with unit normal n
template <class T>
void map_projector (const tensor_basic<T>& DF, size_t d, size_t map_d, tensor_basic<T>& P);

// axisymetric weight ?
// point_basic<T> xq = rheolef::piola_transformation (_omega, _piola_table, K, dis_inod, q);
template<class T>
inline
T
weight_coordinate_system (space_constant::coordinate_type sys_coord, const point_basic<T>& xq)
{
  switch (sys_coord) {
    case space_constant::axisymmetric_rz: return xq[0];
    case space_constant::axisymmetric_zr: return xq[1];
    case space_constant::cartesian:       return 1;
    default: {
      fatal_macro ("unsupported coordinate system `"
        <<  space_constant::coordinate_system_name (sys_coord) << "'");
      return 0;
    }
  }
}
template<class T, class M, class BasisOnPointset>
T
weight_coordinate_system (
    const geo_basic<T,M>&         omega,
    const BasisOnPointset&        piola_table,
    const geo_element&            K,
    const std::vector<size_t>&    dis_inod,
    size_t                        q)
{
  if (omega.coordinate_system() == space_constant::cartesian) return 1;
  point_basic<T> xq = rheolef::piola_transformation (omega, piola_table, K, dis_inod, q);
  return weight_coordinate_system (omega.coordinate_system(), xq);
}

}// namespace rheolef
#endif // _RHEOLEF_PIOLA_H
