#ifndef _RHEOLEF_S_GRAD_GRAD_H
#define _RHEOLEF_S_GRAD_GRAD_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

#include "rheolef/form_element.h"

namespace rheolef { 

/*Class:
NAME: @code{s_grad_grad} -- grad_grad-like operator for the Stokes stream function computation
@bfindex s_grad_grad
@apindex P1
@apindex P2
@apindex P1d
SYNOPSIS:
    @example
      form(const space V, const space& V, "s_grad_grad");
    @end example
DESCRIPTION:       
    @noindent
    Assembly the form associated to the -div(grad) variant operator on a finite element space @code{V}.
    The V space may be a either @code{P1} or @code{P2} finite element space.
    See also @ref{form class} and @ref{space class}.
    On cartesian coordinate systems, the form coincide with the "grad_grad" one (see @ref{grad_grad form}):
@iftex  
    $$ 
      a(u,v) = \int_\Omega \nabla u . \nabla v \, dx
    $$ 
@end iftex  
@ifnottex  
    @example
               /
               |
      a(u,v) = |  grad(u).grad(v) dx
               |
               / Omega
    @end example
@end ifnottex  
@cindex stream function
    @noindent
    The stream function on tri-dimensionnal cartesian coordinate systems is such that
    @example
       u = curl psi
       div psi = 0
    @end example
    where u is the velocity field. Taking the curl of the first relation,
    using the identity:
    @example
	curl(curl(psi)) = -div(grad(psi)) + grad(div(psi))
    @end example
    and using the div(psi)=0 relation leads to:
    @example
	-div(grad(psi)) = curl(u)
    @end example
    This relation leads to a variational formulation involving the
    the "grad_grad" and the "curl" forms (see @ref{grad_grad form}, @ref{curl form}).

    In the axisymmetric case, the stream function psi is scalar ans is defined from the
    velocity field u=(ur,uz) by (see Batchelor, 6th ed., 1967, p 543):
    @example       
                 d psi                       d psi
      uz = (1/r) -----    and   ur = - (1/r) -----
                  d r                         d r
    @end example
    See also http://en.wikipedia.org/wiki/Stokes_stream_function .
    Multiplying by rot(xi)=(d xi/dr, -d xi/dz), and integrating with r dr dz, we get
    a well-posed variationnal problem:
    @example       
	a(psi,xi) = b(xi,u)
    @end example
    with
    @example       
                  /
                  | (d psi d xi   d psi d xi)
      a(psi,xi) = | (----- ---- + ----- ----) dr dz
                  | ( d r  d r     d z  d z )
                  / Omega
    @end example
    and
    @example       
                /
                | (d xi      d xi   )
      b(xi,u) = | (---- ur - ---- uz) r dr dz
                | (d z       d r    )
                / Omega
    @end example
    Notice that a is symmetric definite positive, but without the 'r' weight
    as is is usual for axisymmetric standard forms. 
    The b form is named "s_curl", for the Stokes curl variant of the "curl" operator (see @ref{s_curl form})
    as it is closely related to the "curl" operator, but differs
    by the r and 1/r factors, as:
    @example       
                   (       d (r xi)     d xi )
        curl(xi) = ( (1/r) -------- ; - -----)
                   (         d r        d z  )
    @end example
    while
    @example       
                   ( d xi       d xi )
      s_curl(xi) = ( ----  ;  - ---- )
                   ( d r        d z  )
    @end example
EXAMPLE:
    @noindent
    The following piece of code build the form
    associated to the @code{P1} approximation:
    @example
   	geo g("square");
   	space V(g, "P1");
   	form a(V, V, "s_grad_grad");
    @end example
AUTHOR: 
    LMC-IMAG, 38041 Grenoble cedex 9, France
    | Pierre.Saramito@imag.fr
DATE:
    29 february 2012
End:
*/
template<class T, class M>
class s_grad_grad: public form_element_rep<T,M> {
public:
    typedef form_element_rep<T,M>       base;
    typedef typename base::size_type    size_type;
    typedef typename base::geo_type     geo_type;
    typedef typename base::space_type   space_type;

    s_grad_grad (const space_type& X, const space_type& Y, const geo_type& omega, 
        const quadrature_option_type& qopt)
	: form_element_rep<T,M> (X,Y,omega,qopt) { base::initialize_all(); }

    void operator() (const geo_element& K, ublas::matrix<T>& m) const;
    void initialize () const;
    bool is_symmetric () const { return true; }
};

}// namespace rheolef
#endif // _RHEOLEF_S_GRAD_GRAD_H
