///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "div.h"
#include "rheolef/piola.h"
namespace rheolef {

using namespace std;
using namespace ublas;

#ifdef TO_CLEAN
template<class T>
static
void
P_invDFt (
  const tensor_basic<T>& DF,
  const tensor_basic<T>& invDF,
  size_t        d,
  size_t        K_map_d,
        tensor_basic<T>& H)
{
  typedef geo_element::size_type size_type;
  if (K_map_d == d) {
    // classical case: H = trans(invDF);
    for (size_type l = 0; l < d; l++) {
      for (size_type r = 0; r < K_map_d; r++) {
        H(l,r) = invDF(r,l);
      }
    }
    return;
  }
  // here: surface divergence "div_s" : H := P*trans(invDF)
  tensor_basic<T> P;
  map_projector (DF, d, K_map_d, P); // P = I - nxn
  // H1 = P*H;
  for (size_type l = 0; l < d; l++) {
    for (size_type r = 0; r < K_map_d; r++) {
      T sum = 0;
      for (size_type m = 0; m < d; m++) {
        sum += P(l,m)*invDF(r,m);
      }
      H(l,r) = sum;
    }
  }
}
#endif // TO_CLEAN

template<class T, class M>
void
div<T,M>::operator() (const geo_element& K, matrix<T>& m) const
{
  std::vector<size_type> dis_inod_K;
  std::vector<size_type> dis_inod_L;
  base::_omega.dis_inod (K, dis_inod_K);
  reference_element hat_K   = K.variant();
  reference_element tilde_L = hat_K;
  if (base::is_on_band()) {
    // voir dans form_element.cc : fct build_grad_grad() for notations
    size_type first_dis_ie = base::_omega.sizes().ownership_by_dimension [K.dimension()].first_index();
    size_type K_ie = K.dis_ie() - first_dis_ie;
    size_type L_ie = base::get_band().sid_ie2bnd_ie (K_ie);
    const geo_element& L = base::get_band().band() [L_ie];
    base::get_band().band().dis_inod (L, dis_inod_L);
    tilde_L = L.variant();
  }
  size_type d       = base::coordinate_dimension();
  size_type K_map_d = hat_K.dimension();
  size_type L_map_d = tilde_L.dimension();
  size_type nx = base::get_first_basis().size (tilde_L);
  size_type ny = base::get_second_basis().size(tilde_L);
  m.resize (ny, d*nx);
  m.clear();
  typename quadrature<T>::const_iterator
    first = base::_quad.begin(hat_K),
    last  = base::_quad.end  (hat_K);
  tensor_basic<T> DF, DG, invDG, P, H;
  for (size_type q = 0; first != last; first++, q++) {
    rheolef::jacobian_piola_transformation (base::_omega, base::_piola_table, K, dis_inod_K, q, DF);
    T det_DF = rheolef::det_jacobian_piola_transformation (DF, d, K_map_d);
    if (!base::is_on_band()) {
      invDG = pseudo_inverse_jacobian_piola_transformation (DF, d, K_map_d);
      if (K_map_d == d) {
        H = trans(invDG); // TODO: optimize sizes: P(d,d)*trans(invDG)(d,d)
      } else {
        map_projector (DF, d, K_map_d, P); // P = I - n*n
        H = P*trans(invDG); // TODO: optimize sizes: P(d,d)*trans(invDG)(d,L_map_d)
      }
      base::_bx_table.evaluate_grad (hat_K, q);
      base::_by_table.evaluate      (hat_K, q);
    } else {
      point_basic<T> xq = rheolef::piola_transformation (base::_omega, base::_piola_table, hat_K, dis_inod_K, q);
      point_basic<T> tilde_xq = rheolef::inverse_piola_transformation (base::get_band().band(), tilde_L, dis_inod_L, xq);
      rheolef::jacobian_piola_transformation (base::get_band().band(), base::_piola_table, tilde_L, dis_inod_L, tilde_xq, DG);
      invDG = pseudo_inverse_jacobian_piola_transformation (DG, d, L_map_d);
      // apply also the tangential projection P=(I-n*n)
      map_projector (DF, d, K_map_d, P);
      H = P*trans(invDG); // TODO: optimize sizes: P(d,d)*trans(invDG)(d,L_map_d)
      base::_bx_table.evaluate_grad (tilde_L, tilde_xq);
      base::_by_table.evaluate      (tilde_L, tilde_xq);
    }
    T wq = 1;
    wq *= base::is_weighted() ? base::weight(K,q) : T(1);
    wq *= base::weight_coordinate_system (K, dis_inod_K, q);
    wq *= det_DF;
    wq *= (*first).w;
    typename base::quad_const_iterator
      phi            = base::_by_table.begin(),
      last_phi       = base::_by_table.end();
    typename base::quad_const_iterator_grad
      first_grad_psi = base::_bx_table.begin_grad(),
      last_grad_psi  = base::_bx_table.end_grad();
    for (size_type i = 0; phi != last_phi; phi++, i++) {
      typename base::quad_const_iterator_grad grad_psi = first_grad_psi;
      T phi_i = *phi;
      for (size_type j = 0; grad_psi != last_grad_psi; grad_psi++, j++) {
	const point_basic<T>& grad_psi_j = *grad_psi;
        for (size_type k = 0; k < d; k++) {
	  T sum = 0;
          for (size_type l = 0; l < L_map_d; l++) {
            sum += phi_i * H(k,l) * grad_psi_j[l];
	  }
	  m(i,j+k*nx) += sum*wq;
	}
      }
    }
  }
  if (base::coordinate_system() == space_constant::cartesian) return;
  // --------------------------------------
  // here, axisymletric: add the (ur/r) r dr dz term
  //   as : ur dr dz
  // i.e. skipping the `r' extra weight
  // --------------------------------------
  ublas::matrix<T> mr;
  base::set_use_coordinate_system_weight (false);
  base::build_scalar_mass (K, mr);
  base::set_use_coordinate_system_weight (true);
  size_type xoffset = 0;
  if (base::coordinate_system() == space_constant::axisymmetric_zr) {
    xoffset += nx;
  }
  for (size_type i = 0; i < ny; i++) {
    for (size_type j = 0; j < nx; j++) {
      m(i,j+xoffset) += mr(i,j);
    }
  }
}
template<class T, class M>
void
div<T,M>::initialize () const
{
  base::set_n_derivative(1);

  check_macro (
	space_constant::vector == base::get_first_space().valued_tag(),
	"unsupported non-vector first space for `2D' form");
  check_macro (
	space_constant::scalar == base::get_second_space().valued_tag(),
	"unsupported non-scalar second space for `2D' form");
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
template class div<Float,sequential>;

#ifdef _RHEOLEF_HAVE_MPI
template class div<Float,distributed>;
#endif // _RHEOLEF_HAVE_MPI

} // namespace rheolef
