/*
 * ckraid.c : Utility for the Linux Multiple Devices driver
 *            Copyright (C) 1997 Ingo Molnar, Miguel de Icaza, Gadi Oxman,
 *            Bradley Ward Allen
 *
 * This utility checks Linux MD RAID 1/4/5 arrays.
 *
 * This source is covered by the GNU GPL, the same as all Linux kernel
 * sources.
 */

#include "common.h"
#include "parser.h"
#include "raid_io.h"

#define FACTOR 128
#define IO_BUFFERSIZE (MD_BLK_SIZ*FACTOR)

md_cfg_entry_t *cfg_head = NULL, *cfg = NULL;
int force_flag = 1;
int do_quiet_flag = 0;

static int do_fix_flag = 0, do_continue = 0;
static char source_disk[MAX_LINE_LENGTH];
static unsigned long suggest_failed_mask=0, errcount=0;
static unsigned long suggest_fix_parity=0;
static __u32 skip = 0, blocks, current = 0, first=0;
static char fd_buffer[MD_SB_DISKS][IO_BUFFERSIZE];

static char buffer[MD_SB_DISKS][FACTOR][MD_BLK_SIZ];
static int fd[MD_SB_DISKS];

static pthread_t parallel_thread [MD_SB_DISKS];
static pthread_mutex_t start_work [MD_SB_DISKS];
static pthread_mutex_t stop_work [MD_SB_DISKS];


static void xor_block(char *dst, char *src, int size)
{
	int *dstp = (int *) dst, *srcp = (int *) src, words = size / sizeof(int), i;
	for (i = 0; i < words; i++)
		*dstp++ ^= *srcp++;
}

static inline int fixrow5 ( md_cfg_entry_t *p, md_superblock_t *sb,
	int fd[MD_SB_DISKS], int current, int failed_disk, int skip,
	char buffer[MD_SB_DISKS][FACTOR][MD_BLK_SIZ],
	char zero_buffer[MD_BLK_SIZ])
{
	md_descriptor_t *descriptor;
	int i, rel = (current-skip) & (FACTOR-1);
	char test_buffer[MD_BLK_SIZ];
	static int have_error=0;
	int chunk = sb->chunk_size >> 9, raid_disks = sb->nr_disks;
	int stripe = current / (chunk / raid_disks);
	int idx;

	memset(test_buffer, 0, MD_BLK_SIZ);
	for (i = 0; i < sb->nr_disks; i++) {
		descriptor = &p->sb_old.disks[i];
		if (descriptor->raid_disk >= sb->raid_disks)
			continue;
		xor_block(test_buffer, buffer[i][rel], MD_BLK_SIZ);
	}

	if (!memcmp(zero_buffer, test_buffer, MD_BLK_SIZ)) {
		if (have_error) {
			if (current-first!=1) {
				if (current-first-1==1)
					OUT("      (%d more block differs)                         \n", current-first-1);
				else
					OUT("      (%d more blocks differ)                         \n", current-first-1);
			}
			have_error=0;
		}
		return 0;
	}
	memset(test_buffer, 0, MD_BLK_SIZ);

	if (!have_error) {
		OUT("row #%d=0x%X incorrect", current, current);
		have_error=1;
		first=current;

		if (!do_fix_flag) {
			OUT(" ... not fixing                     \n");
		} else {
			if (suggest_fix_parity)
				OUT(" ... fixing parity                  \n");
			else if(failed_disk==-1)
				OUT("no failed disk known so cant fix    \n");
			else
				OUT(" ... fixing\n");
		}
	}

	if ((!do_fix_flag) || ((failed_disk == -1) && (!suggest_fix_parity)))
		return 0;

	if (suggest_fix_parity) {
		if (sb->level == 4)
			idx = raid_disks;
		else switch (sb->parity_algorithm) {
			case ALGORITHM_LEFT_ASYMMETRIC:
				idx = raid_disks - stripe % raid_disks - 1;
				break;
			case ALGORITHM_RIGHT_ASYMMETRIC:
				idx = stripe % raid_disks;
				break;
			case ALGORITHM_LEFT_SYMMETRIC:
				idx = raid_disks - stripe % raid_disks - 1;
				break;
			case ALGORITHM_RIGHT_SYMMETRIC:
				idx = stripe % raid_disks;
				break;
			default:
				ERR("raid5: unsupported algorithm %d\n",
					sb->parity_algorithm);
				return 0;
		}
#if 0
		OUT("current: %d\n",current);
		OUT("chunk: %d\n",chunk);
		OUT("stripe: %d\n",stripe);
		OUT("raid_disks: %d\n",raid_disks);
		OUT("sb->parity_algorithm: %d\n",sb->parity_algorithm);
		OUT("idx: %d\n",idx);
		return 0;
#endif
		failed_disk = idx;
	}

	memset(test_buffer, 0, MD_BLK_SIZ);
	for (i = 0; i < sb->nr_disks; i++) {
		if (i == failed_disk)
			continue;
		descriptor = &p->sb_old.disks[i];
		if (descriptor->raid_disk >= sb->raid_disks)
			continue;
		xor_block(test_buffer, buffer[i][rel], MD_BLK_SIZ);
	}

	raidseek (fd[failed_disk],current);
	if (write(fd[failed_disk], test_buffer, MD_BLK_SIZ) != MD_BLK_SIZ) {
		OUT( "write error on device file %s, block #%d=0x%X\n",
			p->device_name[failed_disk], current,current);
		if (suggest_fix_parity)
			failed_disk = -1;
		return 1;
	}
	if (suggest_fix_parity)
		failed_disk = -1;

	return 0;
}

static inline int fixrow1 ( md_cfg_entry_t *p, md_superblock_t *sb,
	int fd[MD_SB_DISKS], int current, int source, int skip,
	char buffer[MD_SB_DISKS][FACTOR][MD_BLK_SIZ])
{
	int i, rel = (current-skip) & (FACTOR-1);
	static int have_error=0;
	int error_flag=0;
	md_descriptor_t *descriptor;

	for (i = 0; i < sb->nr_disks; i++) {
		if (i == source)
			continue;
		descriptor = &p->sb_old.disks[i];
		if (descriptor->raid_disk >= sb->raid_disks)
			continue;
		if (!memcmp(buffer[source][rel], buffer[i][rel],
				MD_BLK_SIZ)) 
			continue;
		error_flag=1;
		if (!have_error)
			OUT("block #%d=0x%X on %s differs", current,
				current, p->device_name[i]);
		if (!do_fix_flag) {
			if (!have_error)
				OUT(" ... not fixing\n");
			continue;
		}
		if (!have_error)
			OUT(" ... fixing\n");
		memcpy(buffer[i][rel], buffer[source][rel],MD_BLK_SIZ);
		raidseek(fd[i],current);
		if (write(fd[i], buffer[i][rel], MD_BLK_SIZ) != MD_BLK_SIZ) {
			OUT("write error on device file %s, block #%d=0x%X\n",
				p->device_name[i], current,current);
			return 1;
		}
	}

	if (!error_flag) {
		if (have_error) {
			if (current-first!=1) {
				if (current-first-1==1)
					OUT("      (%d more block differs)                         \n", current-first-1);
				else
					OUT("      (%d more blocks differ)                         \n", current-first-1);
			}
			have_error=0;
		}
	} else {
		if (!have_error) {
			have_error=1;
			first=current;
		}
	}
	return 0;
}


void * reader_thread (void * data)
{
	int i = (int) data;

	while (1) {
			pthread_mutex_lock(&start_work[i]);
			read(fd[i], buffer[i][0], IO_BUFFERSIZE);
			pthread_mutex_unlock(&stop_work[i]);
	}
}

static int ckraid (md_cfg_entry_t *p)
{
	md_superblock_t *sb = &p->sb;
	int i, source = -1, failed_disk = -1,
		 nr_failed = 0, exit_status=0, n, N = sb->nr_disks;
	md_descriptor_t *descriptor;
	char zero_buffer[MD_BLK_SIZ];
	unsigned char read_error;

	for (i = 0; i < N; i++) {
		pthread_mutex_init(&start_work[i], NULL);
		pthread_mutex_lock(&start_work[i]);

		pthread_mutex_init(&stop_work[i], NULL);
		pthread_mutex_lock(&stop_work[i]);

		pthread_create(&parallel_thread[i], NULL,
				reader_thread, (void *)i);
	}

	blocks = sb->size;
	errcount=0;
	printf("checking raid level %d set %s\n", sb->level, p->md_name);

	for (i = 0; i < N; i++)
		fd[i] = 0;
	memset(zero_buffer, 0, MD_BLK_SIZ);
	print_sb(&p->sb_old);
	for (i = 0; i < N; i++) {
		unsigned char this_failed=0, do_source_thingie=0;
		if ((fd[i] = open(p->device_name[i], O_RDWR)) == 0) {
			ERR("couldn't open device file %s\n",
				p->device_name[i]);
			exit_status=1; goto abort;
		}
		descriptor = &p->sb_old.disks[i];
		if (p->sb_present[i] && (!(descriptor->state & (1 << MD_FAULTY_DEVICE))) &&
		    (descriptor->state & (1 << MD_ACTIVE_DEVICE)) &&
		    (descriptor->state & (1 << MD_SYNC_DEVICE))) {
			++do_source_thingie;
		} else if (descriptor->raid_disk >= sb->raid_disks) {
			OUT("spare device %s\n", p->device_name[i]);
			continue;
		} else {
			++this_failed;
			ERR("failed device %s\n", p->device_name[i]);
		}
		if((1<<i)&suggest_failed_mask) {
			++this_failed;
			ERR("suggested failed device %s, disk %d\n",
				p->device_name[i], i);
		}
		if(this_failed) {
			failed_disk = i;
			nr_failed++;
		} else if (do_source_thingie) {
		    	if (source == -1)
				source = i;
		}
		if (strcmp(p->device_name[i], source_disk) == 0)
			source = i;
	}

	if (sb->level == 4 || sb->level == 5)
		n = N-1;
	else if (sb->level == 1)
		n = 1;

	printf("array size: %dkB=0x%X\n", blocks*n, blocks*n*MD_BLK_SIZ);

	if (sb->level == 4 || sb->level == 5) {
		if (nr_failed > 1) {
			ERR("Unrecoverable RAID%d set (%d failed disks)\n",
			sb->level, nr_failed);
			exit_status=2; goto abort;
		}
		if (nr_failed == 1)
			ERR("Disk %d (%s) not in sync with the raid set\n",
				failed_disk, p->device_name[failed_disk]);
	} else
		printf("source disk %d (%s)\n", source, p->device_name[source]);

	if(skip) {
		OUT("skipping %u=0x%X blocks first ...\n", skip,skip);
		for (i = 0; i < N; i++) {
			if(-1==raidseek(fd[i], skip)) {
				OUT("executing skip option on disk %d\n", i);
				exit_status=7; goto abort;
			}
		}
	}
	for (current = skip, first = skip; current < blocks; current++) {

		if (!do_quiet_flag)
			progress(blocks*n, current*n);
		/*
		 * Pass 1: read FACTOR nr. of full rows
		 */
		if (!((current-skip) & (FACTOR-1))) {
			for (i = 0; i < N; i++) {
				raidseek (fd[i],current);
				pthread_mutex_unlock(&start_work[i]);
			}

			for (i = 0; i < N; i++)
				pthread_mutex_lock(&stop_work[i]);
		}
			
		if(read_error) {
			OUT("skipping block %d=0x%X because of read error\n",
				current, current);
		} else {
			if (sb->level == 5 || sb->level == 4) {
				if (fixrow5(p, sb, fd, current, failed_disk,
						skip, buffer, zero_buffer)) {
					if (!do_continue) {
						exit_status=4;
						goto abort;
					} else {
						++errcount;
						OUT("continuing next row\n");
					}
				}
			}
			if (sb->level == 1) {
				if (fixrow1(p, sb, fd, current, source,
						skip, buffer)) {
					if (!do_continue) {
						exit_status=5;
						goto abort;
					} else {
						++errcount;
						OUT("continuing next row\n");
					}
				}
			}
		}
	}
	printf("\n");
abort:
	for (i = 0; i < N; i++)
		if (fd[i])
			close(fd[i]);
	if(errcount)
		OUT("ckraid: Error count: %lu            \n", errcount);
	else
		OUT("ckraid: No errors                   \n");
	return exit_status;
}

int parse_switches(int argc, char *argv[])
{
	int i, name_index = 0;

	*source_disk = 0;
	for (i = 1; i < argc; i++) {
		if(strncmp(argv[i], "--h", 3) == 0 ||
		   strncmp(argv[i], "-h", 2) == 0) {
			return 0;	/* causes error which prints usage */
		} else if(strcmp(argv[i], "--fix") == 0) {
			do_fix_flag = 1;
			continue;
		} else if (strcmp(argv[i], "--quiet") == 0) {
			do_quiet_flag = 1;
			continue;
		} else if (strcmp(argv[i], "--force-continue") == 0) {
			do_continue = 1;
			continue;
		} else if (strcmp(argv[i], "--force-source") == 0) {
			strncpy(source_disk, argv[++i], MAX_LINE_LENGTH);
			continue;
		} else if (strcmp(argv[i], "--force-skip-blocks") == 0) {
			skip=strtoul(argv[++i],NULL,0);
			continue;
		} else if (strcmp(argv[i], "--suggest-failed-disk-mask") == 0) {
			suggest_failed_mask=strtoul(argv[++i],NULL,0);
			continue;
		} else if (strcmp(argv[i], "--suggest-fix-parity") == 0) {
			if (!suggest_failed_mask)
				suggest_fix_parity = 1;
			continue;
		} else {
			if (name_index)
				return 0;
			name_index = i;
		}
	}
	return name_index;
}

void usage (void)
{
	ERR("usage: ckraid [--fix] [--quiet] [--force-continue] [--force-source device] [--force-skip-blocks blockcount] [--suggest-failed-disk-mask mask] [--suggest-fix-parity] conf_file\n");
}

int main (int argc, char *argv[])
{
	FILE *fp = NULL;
	int name_index;
	md_cfg_entry_t *p;

	name_index = parse_switches(argc, argv);
	if (!name_index)
		goto print_usage;
	fp = fopen(argv[name_index], "r");
	if (fp == NULL) {
		ERR("Couldn't open %s -- %s\n",
			argv[name_index], strerror(errno));
		goto abort;
	}
	srand((unsigned int) time(NULL));
	printf("ckraid version %d.%d.%d\n", MKRAID_MAJOR_VERSION, MKRAID_MINOR_VERSION, MKRAID_PATCHLEVEL_VERSION);
	if (parse_config(fp))
		goto abort;
	if (analyze_sb())
		goto abort;
	if (read_sb())
		goto abort;
	for (p = cfg_head; p; p = p->next) {
		if (check_active(p))
			goto abort;
		if (ckraid(p))
			goto abort;
	}
	if (do_fix_flag) {
		if (write_sb(1))
			goto abort;
	} else
		ERR("not fixing RAID superblock\n");
	printf("ckraid: completed\n");
	fclose(fp);
	return 0;
print_usage:
	usage();
abort:
	ERR("ckraid: aborted\n");
	if (fp)
		fclose(fp);
	return 1;
}
