/*
 *  Copyright 2001-2004 Adrian Thurston <adriant@ragel.ca>
 */

/*  This file is part of Ragel.
 *
 *  Ragel is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Ragel is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Ragel; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */

%{

#include <iostream>
#include <stdlib.h>
#include <limits.h>
#include <errno.h>
#include "ragel.h"
#include "parsetree.h"
#include "buffer.h"

using std::endl;

/* Buffers that a inline code blocks are collected into. */
Buffer inlineBuf, exprBuf;

/* For collecting a name references. */
NameRef nameRef;
NameRefList nameRefList;

/* The parse data. For each fsm spec, the parser collects things that it parses
 * in data structures in here. */
ParseData *pd;

/* Bison 1.875 tries to prevent a warning, but results ends up producing a 
 * This is an ugly hack. But it works for this case. */
#ifdef BISON_V1875
#define __attribute__(arg)
#endif

/* Try to do a definition, common to assignment and instantiation. */
void tryMachineDef( const BISON_YYLTYPE &loc, char *name, 
		Join *join, bool isInstance );

extern bool inlineWhitespace;

%}

%union {
	/* General data types. */
	char c;
	char *data;
	int integer;
	Literal *literal;

	/* Tree nodes. */
	Term *term;
	FactorWithAug *factorWithAug;
	FactorWithRep *factorWithRep;
	FactorWithNeg *factorWithNeg;
	Factor *factor;
	Expression *expression;
	Join *join;

	/* Priorities and actions. */
	AugType augType;
	Action *action;
	PriorDesc *priorDesc;

	/* Regular expression items. */
	RegExpr *regExp;
	ReItem *reItem;
	ReOrBlock *reOrBlock;
	ReOrItem *reOrItem;

	/* Inline parse tree items. */
	InlineItem *ilitem;
	InlineList *illist;
}

%token TK_Section

/* General tokens. */
%token <data> TK_UInt
%token <data> TK_PInt
%token <data> TK_NInt
%token <data> TK_Hex
%token <data> TK_Word
%token <data> TK_Literal
%token <data> TK_BaseClause
%token TK_DoubleDot
%token TK_Arrow
%token TK_StarStar
%token TK_ColonEquals
%token TK_NameSep

%token TK_StartGblError
%token TK_AllGblError
%token TK_FinishGblError
%token TK_LeaveGblError
%token TK_StartFinishGblError
%token TK_FinishLeaveGblError

%token TK_StartLocalError
%token TK_AllLocalError
%token TK_FinishLocalError
%token TK_LeaveLocalError
%token TK_StartFinishLocalError
%token TK_FinishLeaveLocalError

%token TK_StartContext
%token TK_AllContext
%token TK_FinishContext
%token TK_LeaveContext
%token TK_StartFinishContext
%token TK_FinishLeaveContext

/* Regular expression tokens. */
%token RE_Slash
%token RE_SqOpen
%token RE_SqOpenNeg
%token RE_SqClose
%token RE_Dot
%token RE_Star
%token RE_Dash
%token <data> RE_Char

/* Tokens specific to inline code. */
%token <data> IL_WhiteSpace
%token <data> IL_Comment
%token <data> IL_Literal
%token <data> IL_Symbol

/* Keywords. */
%token KW_Struct
%token KW_Action
%token KW_Init
%token KW_AlphType
%token KW_Range
%token KW_Element
%token KW_GetKey
%token KW_Context

/* Specials in code blocks. */
%token KW_Hold
%token KW_PChar
%token KW_Char
%token KW_Goto
%token KW_Call
%token KW_Ret
%token KW_Stack
%token KW_CurState
%token KW_TargState
%token KW_Entry
%token KW_Next
%token KW_Exec
%token KW_Buf
%token KW_BLen

/* Special token for terminating semi-terminated code blocks. Needed because
 * semi is sent as a token in the code block rather than as a generic symbol. */
%token TK_Semi


/* Symbols. In ragel lexical space, the scanner does not pass 
 * any data along with the symbols, in inline code lexical 
 * space it does. */
%token '*' '?' '+' '!' '(' ')' ';' ',' '=' 
%token ':' '@' '%' '$' '-' '|' '&' '.' '>'

%type <augType> AugTypeBase
%type <augType> AugTypeError
%type <augType> AugTypeLocalError
%type <augType> AugTypeContext
%type <integer> PriorityAug
%type <data> PriorityAugNum
%type <action> ActionEmbed
%type <join> Join
%type <expression> Expression
%type <term> Term
%type <factorWithAug> FactorWithLabel
%type <factorWithAug> FactorWithEp
%type <factorWithAug> FactorWithAug
%type <factorWithRep> FactorWithRep
%type <integer> FactorRepNum
%type <factorWithNeg> FactorWithNeg
%type <factor> Factor
%type <literal> RangeLit
%type <data> AlphabetNum
%type <data> MachineName
%type <integer> PriorityName
%type <integer> LocalErrName
%type <integer> ContextName;
%type <data> OptBaseClause

%type <illist> InlineBlock
%type <ilitem> InlineBlockItem
%type <ilitem> InlineBlockInterpret
%type <data> InlineBlockAny
%type <data> InlineBlockSymbol

%type <illist> InlineExpr
%type <ilitem> InlineExprItem
%type <ilitem> InlineExprInterpret
%type <illist> InlineExprCommaSepExpr
%type <data> InlineExprSymbol
%type <data> InlineExprAny

%type <regExp> RegularExpr
%type <reItem> RegularExprItem
%type <reItem> RegularExprChar
%type <reOrBlock> RegularExprOrData
%type <reOrItem> RegularExprOrChar

%%

/* Input is any number of input sections. An empty file is accepted. */
input: FsmSpecList;
FsmSpecList: 		
	FsmSpecList FsmSpec |
	/* Nothing */;

/* Fsm Specification. Fsms are deliminated by the %% tokens 
 * and require a name for the fsm immediately following the open %% 
 * deliminator. */
FsmSpec:
	TK_Section TK_Word StatementList TK_Section {
		/* Set the fsm's name. */
		pd->fsmName = $2;
		pd->fsmStartSecLoc = InputLoc(@1);
		pd->fsmEndSecLoc = InputLoc(@4);

		/* If there were no errors, then generate output. */
		if ( pd->errorCount == 0 ) {
			/* Generate output, dot file or fsm spec. */
			switch ( outputFormat ) {
			case OutCCode: case OutCppCode: case OutObjCCode:
				pd->generateCode( );
				break;
			case OutGraphvizDot:
				pd->generateGraphviz( );
				break;
			case OutPrint:
				pd->printFsm();
				break;
			case OutDump:
				pd->dumpFsm();
				break;
			}
		}
		delete pd;

		/* Make a parse data for the next fsm machine. */
		pd = new ParseData( inputFile );
	};

/* Garble up a to the end of a section. */
FsmSpec: TK_Section error TK_Section { yyerrok; };


/* A NonEmpty list of statements in a fsm. */
StatementList:
	StatementList Statement |
	/* Nothing */;

/* The differnt types of statements in a fsm spec. */
Statement:	
	Assignment |
	Instantiation |
	DataSpec |
	ActionSpec |
	InitSpec |
	AlphSpec |
	ElementSpec |
	GetKeySpec |
	Context |
	RangeSpec;

/* Garble up to the next ; */
Statement: error ';' { yyerrok; };

/* Allow the user to give some code to insert into the init function. */
InitSpec:
	KW_Init '{' InlineBlock '}' {
		/* Put the code on the init code list. */
		pd->initCodeList.append( new InlineBlock( InputLoc(@2), $3, NameRefList() ) );
	};

/* Allow the user to give some declarations to put into the fsm structure This
 * block is interpreted. */
DataSpec:	
	KW_Struct OptBaseClause '{' InlineBlock '}' ';' {
		/* Only a single base clause may be specified. */
		if ( $2 != 0 ) {
			if ( outputFormat != OutCppCode && outputFormat != OutObjCCode ) {
				/* Error if using base clause in an unsupported language. */
				error(@2) << "base clause not supported by this host language" << endl;
			}
			else if ( pd->baseClause != 0 ) {
				/* Recover by ignoring the additional base clause. */
				error(@2) << "base clause already specified" << endl;
			}
			else {
				/* Save off. */
				pd->baseClause = $2;
			}
		}

		/* Put the spec for the data on the list.  */
		pd->dataList.append( new InlineBlock( 
				InputLoc(@3), $4, NameRefList() ) );
	};

OptBaseClause: 
	TK_BaseClause | 
	/* Nothing */ { $$ = 0; }

/* Allow the user to create a named fsm action that can be referenced when
 * building a machine. */
ActionSpec:	
	KW_Action TK_Word '{' InlineBlock '}' {
		if ( pd->actionDict.find( $2 ) ) {
			/* Recover by just ignoring the duplicate. */
			error(@2) << "action \"" << $2 << "\" already defined" << endl;
		}
		else {
			/* Add the action to the list of actions. */
			Action *newAction = new Action( InputLoc(@3), $2, $4, nameRefList );

			/* Insert to list and dict. */
			pd->actionList.append( newAction );
			pd->actionDict.insert( newAction );
		}
	};

/* Specifies the data type of the input alphabet. One or two words 
 * followed by a semi-colon. */
AlphSpec:
	KW_AlphType TK_Word TK_Word TK_Semi {
		if ( ! pd->setAlphType( $2, $3 ) ) {
			// Recover by ignoring the alphtype statement.
			error(@2) << "\"" << $2 << 
				" " << $3 << "\" is not an allowed alphabet type" << endl;
		}
	} |
	KW_AlphType TK_Word TK_Semi {
		if ( ! pd->setAlphType( $2 ) ) {
			// Recover by ignoring the alphtype statement.
			error(@2) << "\"" << $2 << "\" is not an allowed alphabet type" << endl;
		}
	};

ElementSpec:
	KW_Element InlineBlock TK_Semi {
		pd->elementType = $2;
	};

GetKeySpec:
	KW_GetKey InlineBlock TK_Semi {
		pd->getKeyExpr = $2;
	};

/* Declaration of context. */
Context:
	KW_Context TK_Word ';' {
		/* Get a context on word. */
		ContextMapEl *inMap = 0;
		if ( pd->contextMap.insert( $2, &inMap ) )
			inMap->value = new Context( $2, pd->nextContextId++ );
		inMap->value->declared = false;
		pd->contextList.append( inMap->value );
	};

/* Specifies a range to assume that the input characters will fall into. */
RangeSpec:
	KW_Range AlphabetNum AlphabetNum ';' {
		// Save the upper and lower ends of the range and emit the line number.
		pd->lowerNum = $2;
		pd->upperNum = $3;
		pd->rangeLowLoc = InputLoc(@2);
		pd->rangeHighLoc = InputLoc(@3);
	};

/* An assignement statement. Assigns the definition of a machine to a variable name. */
Assignment:
	MachineName '=' Join ';' {
		bool isInstance = false;

		/* Main machine must be an instance. */
		if ( strcmp($1, machineMain) == 0 ) {
			warning(@1) << "main machine will be implicitly instantiated" << endl;
			isInstance = true;
		}

		/* Generic creation of machine for instantiation and assignment. */
		tryMachineDef( @1, $1, $3, isInstance );
	};

/* An instantiation statement. Instantiates a machine and assigns it to a
 * variable name. */
Instantiation:
	MachineName TK_ColonEquals Join ';' {
		/* Generic creation of machine for instantiation and assignment. */
		tryMachineDef( @1, $1, $3, true );
	};

/* Capture the machine name for making the machine's priority name. */
MachineName:
	TK_Word {
		/* Make/get the priority key. The name may have already been referenced
		 * and therefore exist. */
		PriorDictEl *priorDictEl;
		if ( pd->priorDict.insert( $1, pd->nextPriorKey, &priorDictEl ) )
			pd->nextPriorKey += 1;
		pd->curDefPriorKey = priorDictEl->value;

		/* Make/get the local error key. */
		LocalErrDictEl *localErrDictEl;
		if ( pd->localErrDict.insert( $1, pd->nextLocalErrKey, &localErrDictEl ) )
			pd->nextLocalErrKey += 1;
		pd->curDefLocalErrKey = localErrDictEl->value;
	};

Join: 
	Join ',' Expression {
		/* Append the expression to the list and return it. */
		$1->exprList.append( $3 );
		$$ = $1;
	} |
	Expression {
		/* Create the expression list with the intial expression. */
		$$ = new Join( InputLoc(@1), $1 );
	};

/* Top level production in the parse of a fsm. The lowest precedence
 * is the '|' (or), '&' (intersection), and '-' (subtraction) operators. */
Expression:
	Expression '|' Term {
		$$ = new Expression( $1, $3, Expression::OrType );
	} |
	Expression '&' Term {
		$$ = new Expression( $1, $3, Expression::IntersectType );
	} |
	Expression '-' Term {
		$$ = new Expression( $1, $3, Expression::SubtractType );
	} |
	Term {
		$$ = new Expression( $1 );
	};

Term:
	Term FactorWithLabel {
		$$ = new Term( $1, $2 );
	} |
	Term '.' FactorWithLabel {
		$$ = new Term( $1, $3 );
	} |
	FactorWithLabel {
		$$ = new Term( $1 );
	};

FactorWithLabel:
	TK_Word ':' FactorWithLabel { 
		/* Add the label to the list and pass the factor up. */
		$3->labels.prepend( Label(@1, $1) );
		$$ = $3; 
	} |
	FactorWithEp;

FactorWithEp:
	FactorWithEp TK_Arrow LocalStateRef { 
		/* Add the target to the list and return the factor object. */
		$1->epsilonLinks.append( EpsilonLink( InputLoc(@2), nameRef ) );
		$$ = $1; 
	} |
	FactorWithAug;

/* A local state reference. Qualified name witout :: prefix. */
LocalStateRef:
	NoNameSep StateRefNames;

/* Clear the name ref structure. */
NoNameSep:
	/* Nothing */ {
		nameRef.empty();
	}

/* A qualified state reference. */
StateRef:
	OptNameSep StateRefNames;

/* Optional leading name separator. */
OptNameSep:
	TK_NameSep {
		/* Insert an inition null pointer val to indicate the existence of the
		 * initial name seperator. */
		nameRef.setAs( 0 );
	} |
	/* Nothing. */ {
		nameRef.empty();
	}

/* List of names separated by :: */
StateRefNames:
	StateRefNames TK_NameSep TK_Word {
		nameRef.append( $3 );
	} |
	TK_Word {
		nameRef.append( $1 );
	};

/* Third group up in precedence. Allow users to embed actions and to change the
 * priorities of transitions. */
FactorWithAug:
	FactorWithAug AugTypeBase ActionEmbed {
		/* Append the action to the factorWithAug, record the refernce from 
		 * factorWithAug to the action and pass up the factorWithAug. */
		$1->actions.append( ParserAction( $2, 0, $3 ) );
		$$ = $1;
	} |
	FactorWithAug AugTypeError ActionEmbed {
		/* Append the action to the factorWithAug, record the refernce from 
		 * factorWithAug to the action and pass up the factorWithAug. */
		$1->actions.append( ParserAction( $2, pd->curDefLocalErrKey, $3 ) );
		$$ = $1;
	} |
	FactorWithAug AugTypeLocalError '(' LocalErrName ',' ActionEmbed ')' {
		/* Append the action to the factorWithAug, record the refernce from
		 * factorWithAug to the action and pass up the factorWithAug. */
		$1->actions.append( ParserAction( $2, $4, $6 ) );
		$$ = $1;
	} |
	FactorWithAug AugTypeBase '(' PriorityName ',' PriorityAug ')' {
		// Append the priority using a default name.
		$1->priorityAugs.append( PriorityAug( $2, $4, $6 ) );
		$$ = $1;
	} |
	FactorWithAug AugTypeBase PriorityAug {
		// Append the named priority to the factorWthAug and pass it up.
		$1->priorityAugs.append( PriorityAug( $2, pd->curDefPriorKey, $3 ) );
		$$ = $1;
	} |
	FactorWithAug AugTypeContext ContextName {
		$1->contexts.append( ContextEmbed( $2, $3 ) );
		$$ = $1;
	} |
	FactorWithRep {
		$$ = new FactorWithAug( $1 );
	};

/* A specified priority name. Looks up the name in the current priority
 * dictionary. */
PriorityName:
	TK_Word {
		// Lookup/create the priority key.
		PriorDictEl *priorDictEl;
		if ( pd->priorDict.insert( $1, pd->nextPriorKey, &priorDictEl ) )
			pd->nextPriorKey += 1;

		// Use the inserted/found priority key.
		$$ = priorDictEl->value;
	};

LocalErrName:
	TK_Word {
		/* Lookup/create the priority key. */
		LocalErrDictEl *localErrDictEl;
		if ( pd->localErrDict.insert( $1, pd->nextLocalErrKey, &localErrDictEl ) )
			pd->nextLocalErrKey += 1;

		/* Use the inserted/found priority key. */
		$$ = localErrDictEl->value;
	};

ContextName:
	TK_Word {
		/* Get a context on word. */
		ContextMapEl *inMap = 0;
		if ( pd->contextMap.insert( $1, &inMap ) )
			inMap->value = new Context( $1, pd->nextContextId++ );
		$$ = inMap->value->id;
	};

/* Priority change specs. */
PriorityAug: 
	PriorityAugNum {
		// Convert the priority number to a long. Check for overflow.
		errno = 0;
		int aug = strtol( $1, 0, 10 );
		if ( errno == ERANGE && aug == LONG_MAX ) {
			// Priority number too large. Recover by setting the priority to 0.
			error(@1) << "priority number " << $1 << " overflows" << endl;
			$$ = 0;
		}
		else if ( errno == ERANGE && aug == LONG_MIN ) {
			// Priority number too large in the neg. Recover by using 0.
			error(@1) << "priority number " << $1 << " underflows" << endl;
			$$ = 0;
		}
		else {
			// No overflow or underflow.
			$$ = aug;
 		}
	};

PriorityAugNum:
	TK_NInt | TK_PInt | TK_UInt ;


/* Differnt places to change priorities. */
AugTypeBase:
	'@' { $$ = at_finish; } |
	'%' { $$ = at_leave; } |
	'$' { $$ = at_all; } |
	'>' { $$ = at_start; };
		
/* Global error actions. */
AugTypeError:
	AugTypeLocalError { $$ = $1; } |
	TK_StartGblError { $$ = at_start_gbl_error; } |
	TK_AllGblError { $$ = at_all_gbl_error; } |
	TK_FinishGblError { $$ = at_finish_gbl_error; } |
	TK_LeaveGblError { $$ = at_leave_gbl_error; } |
	TK_StartFinishGblError { $$ = at_start_finish_gbl_error; } |
	TK_FinishLeaveGblError { $$ = at_finish_leave_gbl_error; };

/* Local error actions. */
AugTypeLocalError:
	TK_StartLocalError { $$ = at_start_local_error; } |
	TK_AllLocalError { $$ = at_all_local_error; } |
	TK_FinishLocalError { $$ = at_finish_local_error; } |
	TK_LeaveLocalError { $$ = at_leave_local_error; } |
	TK_StartFinishLocalError { $$ = at_start_finish_local_error; } |
	TK_FinishLeaveLocalError { $$ = at_finish_leave_local_error; };

AugTypeContext:
	TK_StartContext { $$ = at_start_context; } |
	TK_AllContext { $$ = at_all_context; } |
	TK_FinishContext { $$ = at_finish_context; } |
	TK_LeaveContext { $$ = at_leave_context; } |
	TK_StartFinishContext { $$ = at_start_finish_context; } |
	TK_FinishLeaveContext { $$ = at_finish_leave_context; };


/* Different ways to embed actions. A TK_Word is reference to an action given by
 * the user as a statement in the fsm specification. An action can also be
 * specified immediately. */
ActionEmbed:
	TK_Word {
		/* Set the name in the actionDict. */
		Action *action = pd->actionDict.find( $1 );
		if ( action != 0 ) {
			/* Pass up the action element */
			$$ = action;
		}
		else {
			// Recover by returning null as the action.
			error(@1) << "action lookup of \"" << $1 << "\" failed" << endl;
			$$ = 0;
		}
	} |
	'{' InlineBlock '}' {
		/* Create the action, add it to the list and pass up. */
		Action *newAction = new Action( InputLoc(@1), 0, $2, nameRefList );
		pd->actionList.append( newAction );
		$$ = newAction;
	};

/* The fourth level of precedence. These are the trailing unary operators that
 * allow for repetition. */
FactorWithRep:
	FactorWithRep '*' {
		$$ = new FactorWithRep( InputLoc(@2), $1, 0, 0,
				FactorWithRep::StarType );
	} |
	FactorWithRep TK_StarStar {
		$$ = new FactorWithRep( InputLoc(@2), $1, 0, 0,
				FactorWithRep::StarStarType );
	} |
	FactorWithRep '?' {
		$$ = new FactorWithRep( InputLoc(@2), $1, 0, 0,
				FactorWithRep::OptionalType );
	} |
	FactorWithRep '+' {
		$$ = new FactorWithRep( InputLoc(@2), $1, 0, 0,
				FactorWithRep::PlusType );
	} |
	FactorWithRep '{' FactorRepNum '}' {
		$$ = new FactorWithRep( InputLoc(@2), $1, $3, 0,
				FactorWithRep::ExactType );
	} |
	FactorWithRep '{' ',' FactorRepNum '}' {
		$$ = new FactorWithRep( InputLoc(@2), $1, 0, $4,
				FactorWithRep::MaxType );
	} |
	FactorWithRep '{' FactorRepNum ',' '}' {
		$$ = new FactorWithRep( InputLoc(@2), $1, $3, 0,
				FactorWithRep::MinType );
	} |
	FactorWithRep '{' FactorRepNum ',' FactorRepNum '}' {
		$$ = new FactorWithRep( InputLoc(@2), $1, $3, $5,
				FactorWithRep::RangeType );
	} |
	FactorWithNeg {
		$$ = new FactorWithRep( InputLoc(@1), $1 );
	};

FactorRepNum:
	TK_UInt {
		// Convert the priority number to a long. Check for overflow.
		errno = 0;
		int rep = strtol( $1, 0, 10 );
		if ( errno == ERANGE && rep == LONG_MAX ) {
			// Repetition too large. Recover by returing repetition 1. */
			error(@1) << "repetition number " << $1 << " overflows" << endl;
			$$ = 1;
		}
		else {
			// Cannot be negative, so no overflow.
			$$ = rep;
 		}
	};

/* The fifth level up in precedence. Negation. */
FactorWithNeg:
	'!' FactorWithNeg {
		$$ = new FactorWithNeg( InputLoc(@1), $2 );
	} |
	Factor {
		$$ = new FactorWithNeg( InputLoc(@1), $1 );
	};

/* The highest level in precedence. Atomic machines such as references to other
 * machines, literal machines, regular expressions or Expressions in side of
 * parenthesis. */
Factor:
	TK_Literal {
		// Create a new factor node going to a concat literal. */
		$$ = new Factor( new Literal( InputLoc(@1), $1, Literal::LitString ) );
	} |
	AlphabetNum {
		// Create a new factor node going to a literal number. */
		$$ = new Factor( new Literal( InputLoc(@1), $1, Literal::Number ) );
	} |
	TK_Word {
		// Find the named graph.
		GraphDictEl *gdNode = pd->graphDict.find( $1 );
		if ( gdNode == 0 ) {
			// Recover by returning null as the factor node.
			error(@1) << "graph lookup of \"" << $1 << "\" failed" << endl;
			$$ = 0;
		}
		else if ( gdNode->isInstance ) {
			// Recover by retuning null as the factor node.
			error(@1) << "references to graph instantiations not allowed "
					"in expressions" << endl;
			$$ = 0;
		}
		else {
			// Create a factor node that is a lookup of an expression.
			$$ = new Factor( InputLoc(@1), gdNode->value );
		}
	} |
	RE_SqOpen RegularExprOrData RE_SqClose {
		// Create a new factor node going to an OR expression. */
		$$ = new Factor( new ReItem( InputLoc(@1), $2, ReItem::OrBlock ) );
	} |
	RE_SqOpenNeg RegularExprOrData RE_SqClose {
		// Create a new factor node going to a negated OR expression. */
		$$ = new Factor( new ReItem( InputLoc(@1), $2, ReItem::NegOrBlock ) );
	} |
	RE_Slash RegularExpr RE_Slash {
		// Create a new factor node going to a regular exp.
		$$ = new Factor( $2 );
	} |
	RangeLit TK_DoubleDot RangeLit {
		// Create a new factor node going to a range. */
		$$ = new Factor( new Range( $1, $3 ) );
	} |
	'(' Join ')' {
		$$ = new Factor( $2 );
	};

/* Garble up to the closing brace of a parenthesized expression. */
Factor: '(' error ')' { $$ = 0; yyerrok; };

/* Any form of a number that can be used as a basic machine. */
AlphabetNum:
	TK_UInt | TK_PInt | TK_NInt | TK_Hex ;

InlineBlock:
	InlineBlock InlineBlockItem {
		/* Append the item to the list, return the list. */
		$1->append( $2 );
		$$ = $1;
	} |
	/* Empty */ {
		/* Start with empty list. */
		$$ = new InlineList;
	};

/* Items in a struct block. */
InlineBlockItem:
	InlineBlockAny {
		/* Add a text segment. */
		$$ = new InlineItem( $1, InlineItem::Text );
	} |
	InlineBlockSymbol {
		/* Add a text segment, need string on heap. */
		$$ = new InlineItem( strdup($1), InlineItem::Text );
	} |
	InlineBlockInterpret {
		/* Pass the inline item up. */
		$$ = $1;
	};

/* Uninteresting tokens in a struct block. Data allocated by scanner. */
InlineBlockAny:
	IL_WhiteSpace | IL_Comment | IL_Literal | IL_Symbol |
	TK_UInt | TK_PInt | TK_NInt | TK_Hex | TK_Word;

/* Symbols in a struct block, no data allocated. */
InlineBlockSymbol:
	'[' { $$ = "["; } | ']' { $$ = "]"; } | 
	',' { $$ = ","; } | ';' { $$ = ";"; } |
	'(' { $$ = "("; } | ')' { $$ = ")"; } |
	'*' { $$ = "*"; } | TK_NameSep { $$ = "::"; };

/* Interpreted statements in a struct block. */
InlineBlockInterpret:
	InlineExprInterpret {
		/* Pass up interpreted items of inline expressions. */
		$$ = $1;
	} |
	KW_Stack SetNoWs '[' TK_UInt ']' ';' SetWs {
		$$ = new InlineItem( $4, InlineItem::Stack );
	} |
	KW_Hold SetNoWs ';' SetWs {
		$$ = new InlineItem( InlineItem::Hold );
	} |
	KW_Goto SetNoWs StateRef ';' SetWs { 
		$$ = new InlineItem( new NameRef(nameRef), InlineItem::Goto );
	} | 
	KW_Goto SetNoWs '*' SetWs InlineExpr ';' {
		$$ = new InlineItem( InlineItem::GotoE );
		$$->children = $5;
	} |
	KW_Next SetNoWs StateRef ';' SetWs { 
		$$ = new InlineItem( new NameRef(nameRef), InlineItem::Next );
	} |
	KW_Next SetNoWs '*' SetWs InlineExpr ';' {
		$$ = new InlineItem( InlineItem::NextE );
		$$->children = $5;
	} |
	KW_Call SetNoWs StateRef ';'SetWs {
		$$ = new InlineItem( new NameRef(nameRef), InlineItem::Call );
	} | 
	KW_Call SetNoWs '*' SetWs InlineExpr ';' {
		$$ = new InlineItem( InlineItem::CallE );
		$$->children = $5;
	} |
	KW_Ret SetNoWs ';' SetWs {
		$$ = new InlineItem( InlineItem::Ret );
	} |
	KW_Exec SetNoWs '(' InlineExpr ',' InlineExpr ')' ';' SetWs {
		$$ = new InlineItem( InlineItem::Exec );

		/* Make a child node for each of the expressions. */
		InlineItem *left = new InlineItem( InlineItem::Node );
		left->children = $4;
		InlineItem *right = new InlineItem( InlineItem::Node );
		right->children = $6;

		/* Append the child lists. */
		$$->children = new InlineList;
		$$->children->append( left );
		$$->children->append( right );
	};

/* Turn off whitspace collecting when scanning inline blocks. */
SetNoWs: { inlineWhitespace = false; };

/* Turn on whitespace collecting when scanning inline blocks. */
SetWs: { inlineWhitespace = true; };

InlineExpr:
	InlineExpr InlineExprItem {
		$1->append( $2 );
		$$ = $1;
	} |
	/* Empty */ {
		/* Init the list used for this expr. */
		$$ = new InlineList;
	};

InlineExprItem:
	InlineExprAny {
		/* Return a text segment. */
		$$ = new InlineItem( $1, InlineItem::Text );
	} |
	InlineExprSymbol {
		/* Return a text segment, must heap alloc the text. */
		$$ = new InlineItem( strdup($1), InlineItem::Text );
	} |
	InlineExprInterpret {
		/* Pass the inline item up. */
		$$ = $1;
	} |
	'(' InlineExprCommaSepExpr ')' {
		/* Return a list of items. */
		$$ = new InlineItem( InlineItem::Node );
		$$->children = new InlineList;
		$$->children->append( new InlineItem( strdup("("), InlineItem::Text ) );
		$$->children->append( *($2) );
		delete $2;
		$$->children->append( new InlineItem( strdup(")"), InlineItem::Text ) );
	};

InlineExprInterpret:
	KW_PChar {
		$$ = new InlineItem( InlineItem::PChar );
	} |
	KW_Char {
		$$ = new InlineItem( InlineItem::Char );
	} |
	KW_CurState {
		$$ = new InlineItem( InlineItem::Curs );
	} |
	KW_TargState {
		$$ = new InlineItem( InlineItem::Targs );
	} |
	KW_Buf {
		$$ = new InlineItem( InlineItem::Buf );
	} |
	KW_BLen {
		$$ = new InlineItem( InlineItem::BufLen );
	} |
	KW_Entry SetNoWs '(' StateRef ')' SetWs {
		$$ = new InlineItem( new NameRef(nameRef), InlineItem::Entry );
	};

/* Comma separated list fo inline expressions. Passes an InlineList up. */
InlineExprCommaSepExpr:
	InlineExprCommaSepExpr ',' InlineExpr {
		$1->append( new InlineItem( strdup(","), InlineItem::Text ) );
		$1->append( *($3) );
		delete $3;
		$$ = $1;
	} |
	InlineExpr {
		/* Pass the expression list up. */
		$$ = $1;
	};

InlineExprAny:
	IL_WhiteSpace | IL_Comment | IL_Literal | IL_Symbol |
	TK_UInt | TK_PInt | TK_NInt | TK_Hex | TK_Word;

/* Anything in a ExecValExpr that is not dynamically allocated. This includes
 * all special symbols caught in inline code except the semi. */
InlineExprSymbol:
	'[' { $$ = "["; } | ']' { $$ = "]"; } | 
	'*' { $$ = "*"; } | TK_NameSep { $$ = "::"; };


/* Parser for regular expression fsms. Any number of expression items which
 * generally gives a machine one character long or one character long stared. */
RegularExpr:
	RegularExpr RegularExprItem {
		// An optimization to lessen the tree size. If a non-starred char is directly
		// under the left side on the right and the right side is another non-starred
		// char then paste them together and return the left side. Otherwise
		// just put the two under a new reg exp node.
		if ( $2->type == ReItem::Data && !$2->star &&
			$1->type == RegExpr::RecurseItem &&
			$1->item->type == ReItem::Data && !$1->item->star )
		{
			// Append the right side to the right side of the left and toss 
			// the right side.
			$1->item->str += $2->str;
			delete $2;
			$$ = $1;
		}
		else {
			$$ = new RegExpr( $1, $2 );
		}
	} |
	/* Nothing */ {
		// Can't optimize the tree.
		$$ = new RegExpr();
	};

/* RegularExprItems can be a character spec with an optional staring of the char. */
RegularExprItem:
	RegularExprChar RE_Star {
		$1->star = true;
		$$ = $1;
	} |
	RegularExprChar {
		$$ = $1;
	};

/* A character spec can be a set of characters inside of square parenthesis,
 * a dot specifying any character or some explicitly stated character. */
RegularExprChar:
	RE_SqOpen RegularExprOrData RE_SqClose {
		$$ = new ReItem( InputLoc(@1), $2, ReItem::OrBlock );
	} |
	RE_SqOpenNeg RegularExprOrData RE_SqClose {
		$$ = new ReItem( InputLoc(@1), $2, ReItem::NegOrBlock );
	} |
	RE_Dot {
		$$ = new ReItem( InputLoc(@1), ReItem::Dot );
	} |
	RE_Char {
		$$ = new ReItem( InputLoc(@1), $1[0] );
	};

/* The data inside of a [] expression in a regular expression. Accepts any
 * number of characters or ranges. */
RegularExprOrData:
	RegularExprOrData RegularExprOrChar {
		// An optimization to lessen the tree size. If an or char is directly
		// under the left side on the right and the right side is another or
		// char then paste them together and return the left side. Otherwise
		// just put the two under a new or data node.
		if ( $2->type == ReOrItem::Data &&
				$1->type == ReOrBlock::RecurseItem &&
				$1->item->type == ReOrItem::Data )
		{
			// Append the right side to right side of the left and toss
			// the right side.
			$1->item->str += $2->str;
			delete $2;
			$$ = $1;
		}
		else {
			// Can't optimize, put the left and right under a new node.
			$$ = new ReOrBlock( $1, $2 );
		}
	} | 
	/* Nothing */ {
		$$ = new ReOrBlock();
	};


/* A single character inside of an or expression. Can either be a character
 * or a set of characters. */
RegularExprOrChar:
	RE_Char {
		$$ = new ReOrItem( InputLoc(@1), $1[0] );
	} |
	RE_Char RE_Dash RE_Char {
		$$ = new ReOrItem( InputLoc(@2), $1[0], $3[0] );
	};

RangeLit:
	TK_Literal {
		// Range literas must have only one char.
		if ( strlen($1) != 1 ) {
			// Recover by using the literal anyways.
			error(@1) << "literal used in range must be of length 1" << endl;
		}
		$$ = new Literal( InputLoc(@1), $1, Literal::LitString );
	} |
	AlphabetNum {
		// Create a new literal number.
		$$ = new Literal( InputLoc(@1), $1, Literal::Number );
	};

%%

/* Try to do a definition, common to assignment and instantiation. Warns about 
 * instances other than main not being implemented yet. */
void tryMachineDef( const BISON_YYLTYPE &loc, char *name, Join *join, bool isInstance )
{
	GraphDictEl *newEl = pd->graphDict.insert( name );
	if ( newEl != 0 ) {
		/* New element in the dict, all good. */
		newEl->value = new VarDef( name, join );
		newEl->isInstance = isInstance;
		newEl->loc = loc;

		/* It it is an instance, put on the instance list. */
		if ( isInstance )
			pd->instanceList.append( newEl );
	}
	else {
		// Recover by ignoring the duplicate.
		error(loc) << "fsm \"" << name << "\" previously defined" << endl;
	}
	pd->machineGiven = true;
}

void yyerror( char *err )
{
	error(yylloc.first_line, yylloc.first_column) << err << endl;
}
